/******************************************************************************
 * $Id: dbfopen.c,v 1.48 2003/03/10 14:51:27 warmerda Exp $
 *
 * Project:  Shapelib
 * Purpose:  Implementation of .dbf access API documented in dbf_api.html.
 * Author:   Frank Warmerdam, warmerdam@pobox.com
 *
 ******************************************************************************
 * Copyright (c) 1999, Frank Warmerdam
 *
 * This software is available under the following "MIT Style" license,
 * or at the option of the licensee under the LGPL (see LICENSE.LGPL).  This
 * option is discussed in more detail in shapelib.html.
 *
 * --
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 ******************************************************************************
 *
 * $Log: dbfopen.c,v $
 * Revision 1.48  2003/03/10 14:51:27  warmerda
 * DBFWrite* calls now return FALSE if they have to truncate
 *
*/

#include "xbase.h"

#include <math.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#define xbase_c
#define LUA_LIB

#include "agena.h"
#include "agnxlib.h"
#include "agenalib.h"
#include "agnhlps.h"
#include "agnt64.h"
#include "agncmpt.h"

#ifndef lua_boxpointer
#define lua_boxpointer(L, u) \
	(*(void **)(lua_newuserdata(L,  sizeof(void *))) = (u))
#endif

#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_XBASELIBNAME "xbase"
LUALIB_API int (luaopen_xbase) (lua_State *L);
#endif


#ifndef FALSE
#  define FALSE		0
#  define TRUE		1
#endif

static int	nStringFieldLen = 0;
static char * pszStringField = NULL;

/* forward declarations */
void aux_checkpositions (lua_State *L, DBFHandle hnd, int *record, int *field, const char *pn);
static int xbase_gc (lua_State *L);


DBFHandle *aux_gethandle (lua_State *L, int idx, const char *pn) {
  DBFHandle *hnd = (DBFHandle *)luaL_checkudata(L, idx, "xbase");
  if (*hnd == NULL)
    luaL_error(L, "Error in " LUA_QS ": invalid file handle.", pn);
  return hnd;
}

/************************************************************************/
/*                             SfRealloc()                              */
/*                                                                      */
/*      A realloc cover function that will access a NULL pointer as     */
/*      a valid input.                                                  */
/************************************************************************/

static void * SfRealloc ( void * pMem, int nNewSize) {
    if (pMem == NULL)
        return( (void *) malloc(nNewSize));
    else
        return( (void *) realloc(pMem,nNewSize));
}

/************************************************************************/
/*                           DBFWriteHeader()                           */
/*                                                                      */
/*      This is called to write out the file header, and field          */
/*      descriptions before writing any actual data records.  This      */
/*      also computes all the DBFDataSet field offset/size/decimals     */
/*      and so forth values.                                            */
/************************************************************************/

static void DBFWriteHeader (DBFHandle psDBF) {
  unsigned char abyHeader[XBASE_FLDHDR_SZ];
  int i, yy, mm, dd;
  struct TM *stm;
  Time64_T t = time(NULL);
  if (!psDBF->bNoHeader) return;
  psDBF->bNoHeader = FALSE;
  /* Initialize the file header information.	*/
  for (i=0; i < XBASE_FLDHDR_SZ; i++)
    abyHeader[i] = 0;
  stm = gmtime64(&t);
  if (stm == NULL) {  /* invalid date ? */
    yy = 0; mm = 5; dd = 8;
  } else {
    yy = stm->tm_year;
    mm = stm->tm_mon+1;
    dd = stm->tm_mday;
  }
  abyHeader[0] = 0x03;   /* memo field? - just copying 	*/
  abyHeader[1] = yy;	 /* YY */
  abyHeader[2] = mm;	 /* MM */
  abyHeader[3] = dd;	 /* DD */
  /* date updated on close, record count preset at zero */
  abyHeader[8] = psDBF->nHeaderLength % 256;
  abyHeader[9] = psDBF->nHeaderLength / 256;
  abyHeader[10] = psDBF->nRecordLength % 256;
  abyHeader[11] = psDBF->nRecordLength / 256;
  abyHeader[29] = psDBF->codepage;
  /* Write the initial 32 byte file header, and all the field descriptions. */
  fseek(psDBF->fp, 0, 0);
  fwrite(abyHeader, XBASE_FLDHDR_SZ, 1, psDBF->fp);
  fwrite(psDBF->pszHeader, XBASE_FLDHDR_SZ, psDBF->nFields, psDBF->fp);
  /* Write out the newline character if there is room for it. */
  if (psDBF->nHeaderLength > 32*psDBF->nFields + 32) {
    char cNewline;
    cNewline = 0x0d;
    fwrite(&cNewline, 1, 1, psDBF->fp);
  }
}

/************************************************************************/
/*                           DBFFlushRecord()                           */
/*                                                                      */
/*      Write out the current record if there is one.                   */
/************************************************************************/

/* modified for Agena 0.32.4 */

static int DBFFlushRecord (DBFHandle psDBF) {
  int nRecordOffset;
  int result = 0;
  if (psDBF->bCurrentRecordModified && psDBF->nCurrentRecord > -1) {
	psDBF->bCurrentRecordModified = FALSE;
	nRecordOffset = psDBF->nRecordLength * psDBF->nCurrentRecord
	                                             + psDBF->nHeaderLength;
	if (fseek(psDBF->fp, nRecordOffset, 0) == 0) result |= 1;
	if (fwrite(psDBF->pszCurrentRecord, psDBF->nRecordLength, 1, psDBF->fp) == 1)
	  result |= 2;
	if (fflush(psDBF->fp) == 0) result |= 4;
  }
  return (result == 7);
}


static int xbase_sync (lua_State *L) {
  int res;
  DBFHandle *hnd;
  hnd = aux_gethandle(L, 1, "xbase.sync");
  res = DBFFlushRecord(*hnd);
  if (res == 1)
    lua_pushtrue(L);
  else
    lua_pushfail(L);
  return 1;
}


/************************************************************************/
/*                              DBFOpen()                               */
/*                                                                      */
/*      Open a .dbf file.                                               */
/************************************************************************/

DBFHandle SHPAPI_CALL DBFOpen (const char * pszFilename, const char * pszAccess) {
  DBFHandle psDBF;
  unsigned char *pabyBuf;
  int nFields, nHeadLen, nRecLen, iField, i;
  char *pszBasename, *pszFullname;
  /* We only allow the access strings "rb" and "r+". */
  if (strcmp(pszAccess,"r") != 0 && strcmp(pszAccess,"r+") != 0
    && strcmp(pszAccess,"rb") != 0 && strcmp(pszAccess,"rb+") != 0
    && strcmp(pszAccess,"r+b") != 0) {
    return NULL;
  }
  if (strcmp(pszAccess,"r") == 0)
    pszAccess = "rb";
  if (strcmp(pszAccess,"r+") == 0)
    pszAccess = "rb+";
  /* Compute the base (layer) name. If there is any extension */
  /* on the passed in filename we will strip it off. */
  pszBasename = (char *)malloc(strlen(pszFilename)+5);
  strcpy( pszBasename, pszFilename);
  for (i = strlen(pszBasename)-1;
    i > 0 && pszBasename[i] != '.' && pszBasename[i] != '/'
	       && pszBasename[i] != '\\';  i--) {}
  if (pszBasename[i] == '.')
    pszBasename[i] = '\0';
  pszFullname = (char *)malloc(strlen(pszBasename) + 5);
  sprintf(pszFullname, "%s.dbf", pszBasename);
  psDBF = (DBFHandle) calloc(1, sizeof(DBFInfo));
  psDBF->fp = fopen(pszFullname, pszAccess);
  if (psDBF->fp == NULL) {
    sprintf(pszFullname, "%s.DBF", pszBasename);
    psDBF->fp = fopen(pszFullname, pszAccess);
  }
  free(pszBasename);
  psDBF->filename = (char *)malloc(strlen(pszFilename)+5);
  strcpy(psDBF->filename, pszFullname);
  free(pszFullname);
  if (psDBF->fp == NULL) {
    free(psDBF);
    return NULL;
  }
  psDBF->bNoHeader = FALSE;
  psDBF->nCurrentRecord = -1;
  psDBF->bCurrentRecordModified = FALSE;
  /* Read Table Header info */
  pabyBuf = (unsigned char *)malloc(500);
  if (fread(pabyBuf, 32, 1, psDBF->fp) != 1) {
    fclose(psDBF->fp);
    free(pabyBuf);
    free(psDBF);
    return NULL;
  }
  psDBF->nRecords =
    pabyBuf[4] + pabyBuf[5]*256 + pabyBuf[6]*256*256 + pabyBuf[7]*256*256*256;
  psDBF->nHeaderLength = nHeadLen = pabyBuf[8] + pabyBuf[9]*256;
  psDBF->nRecordLength = nRecLen = pabyBuf[10] + pabyBuf[11]*256;
  psDBF->nFields = nFields = (nHeadLen - 32) / 32;
  psDBF->pszCurrentRecord = (char *)malloc(nRecLen);
  psDBF->codepage = pabyBuf[29];
  psDBF->lastmodified = (pabyBuf[1]+1900)*10000 + pabyBuf[2]*100 + pabyBuf[3];  /* added 0.32.5 */
  /* Read in Field Definitions */
  pabyBuf = (unsigned char *)SfRealloc(pabyBuf, nHeadLen);
  psDBF->pszHeader = (char *)pabyBuf;
  fseek(psDBF->fp, 32, 0);
  if (fread(pabyBuf, nHeadLen-32, 1, psDBF->fp) != 1) {
    fclose(psDBF->fp);
    free(pabyBuf);
    free(psDBF);
    return NULL;
  }
  psDBF->panFieldOffset = (int *) malloc(sizeof(int) * nFields);
  psDBF->panFieldSize = (int *) malloc(sizeof(int) * nFields);
  psDBF->panFieldDecimals = (int *) malloc(sizeof(int) * nFields);
  psDBF->pachFieldType = (char *) malloc(sizeof(char) * nFields);
  for (iField = 0; iField < nFields; iField++) {
    unsigned char *pabyFInfo;
    pabyFInfo = pabyBuf+iField*32;
    if (pabyFInfo[11] == 'N' || pabyFInfo[11] == 'F') {
      psDBF->panFieldSize[iField] = pabyFInfo[16];
      psDBF->panFieldDecimals[iField] = pabyFInfo[17];
    } else {
      psDBF->panFieldSize[iField] = pabyFInfo[16] + pabyFInfo[17]*256;
      psDBF->panFieldDecimals[iField] = 0;
    }
    psDBF->pachFieldType[iField] = (char)pabyFInfo[11];
    if (iField == 0)
      psDBF->panFieldOffset[iField] = 1;
    else
      psDBF->panFieldOffset[iField] =
        psDBF->panFieldOffset[iField-1] + psDBF->panFieldSize[iField-1];
  }
  /* check for valid xBase file (0x0d flag at end of fields block), 0.32.4 */
  if (nHeadLen > 32 && pabyBuf[nHeadLen-33] != 0x0d ) {
    fprintf(stderr, "Error in `xbase.open`: this does not seem to be a known or valid xBase file.\n\n");
    fclose(psDBF->fp);
    free(pabyBuf);
    free(psDBF);
    return NULL;
  }
  return psDBF;
}


static int xbase_open (lua_State *L) {
  const char *fn, *attr;
  DBFHandle hnd;
  fn = luaL_checkstring(L, 1);
  attr = luaL_optstring(L, 2, "read");
  if (strcmp(attr, "write") == 0 || strcmp(attr, "append") == 0 || strcmp(attr, "r+") == 0
    || strcmp(attr, "a") == 0 || strcmp(attr, "w") == 0)
    attr = "rb+";
  else if (strcmp(attr, "read") == 0 || strcmp(attr, "r") == 0)
    attr = "rb";
  else
    luaL_error(L, "Error in " LUA_QS ": unknown option " LUA_QS ".", "xbase.open", attr);
  hnd = DBFOpen(fn, attr);
  if (hnd == NULL)
    luaL_error(L, "Error in " LUA_QS ": could not open " LUA_QS ".", "xbase.open", fn);
  else {
    lua_boxpointer(L, hnd);
    luaL_getmetatable(L, "xbase");
    lua_setmetatable(L, -2);
  }
  return 1;
}


static int xbase_readdbf (lua_State *L) {
  const char *fn, *pn, *result;
  DBFHandle hnd;
  int nrecords, nfields, i, j, morefields, *fieldinfo, failure;
  fn = luaL_checkstring(L, 1);
  pn = "xbase.readdbf";
  hnd = DBFOpen(fn, "rb");
  if (hnd == NULL)
    luaL_error(L, "Error in " LUA_QS ": could not open " LUA_QS ".", pn, fn);
  nrecords = DBFGetRecordCount(hnd);
  nfields = DBFGetFieldCount(hnd);
  morefields = nfields != 1;
  fieldinfo = malloc(nfields*sizeof(int));
  if (fieldinfo == NULL)
    luaL_error(L, "Error in " LUA_QS ": memory allocation error.", pn);
  if (nrecords == 0 || nfields == 0)
    luaL_error(L, "Error in " LUA_QS ": data base is empty.", pn);
  agn_createseq(L, nrecords);
  for (j=0; j < nfields; j++) {
    fieldinfo[j] = DBFGetFieldInfo(hnd, j, NULL, NULL, NULL);
  }
  for (i=0; i < nrecords; i++) {
    if (morefields) agn_createseq(L, nfields);
    for (j=0; j < nfields; j++) {
      switch (fieldinfo[j]) {
        case FTString: {
          result = DBFReadStringAttribute(hnd, i, j);
          if (result == NULL)
            luaL_error(L, "Error in " LUA_QS ": no string stored in data base.", pn);
          else
            lua_seqsetistring(L, -1, (morefields) ? j+1 : i+1, result);
          break;
        }
        case FTDouble: {
          lua_seqsetinumber(L, -1, (morefields) ? j+1 : i+1, DBFReadDoubleAttribute(hnd, i, j, &failure));
          if (failure)
            luaL_error(L, "Error in " LUA_QS ": could not successfully read number in data base.", pn);
          break;
        }
        case FTLogical: {
          result = DBFReadLogicalAttribute(hnd, i, j);
          if (result == NULL)
            luaL_error(L, "Error in " LUA_QS ": no logical value stored in data base.", pn);
          else
            lua_seqsetistring(L, -1, (morefields) ? j+1 : i+1, result);
          break;
        }
        default:
          luaL_error(L, "Error in " LUA_QS ": unknown type of data.", pn);
      }
    }
    if (morefields) lua_seqseti(L, -2, i+1);
  }
  free(fieldinfo);
  DBFClose(hnd);
  return 1;
}


/************************************************************************/
/*                              DBFClose()                              */
/************************************************************************/

int SHPAPI_CALL DBFClose (DBFHandle psDBF) {
  /* Write out header if not already written. */
  int result = 0;
  if (psDBF == NULL) return -1;
  if (psDBF->bNoHeader) DBFWriteHeader(psDBF);
  DBFFlushRecord(psDBF);
  /* Update last access date, and number of records if we have write access. */
  if (psDBF->bUpdated) {
    unsigned char abyFileHeader[32];
    Time64_T t = time(NULL);
    struct TM *stm;
    int yy, mm, dd;
    fseek(psDBF->fp, 0, 0);
    fread(abyFileHeader, 32, 1, psDBF->fp);
    stm = gmtime64(&t);
    if (stm == NULL) {  /* invalid date ? */
      yy = 0; mm = 5; dd = 8;
    } else {
      yy = stm->tm_year;
      mm = stm->tm_mon+1;
      dd = stm->tm_mday;
    }
    abyFileHeader[1] = yy;	 /* YY */
    abyFileHeader[2] = mm;	 /* MM */
    abyFileHeader[3] = dd;	 /* DD */
    abyFileHeader[4] = psDBF->nRecords % 256;
    abyFileHeader[5] = (psDBF->nRecords/256) % 256;
    abyFileHeader[6] = (psDBF->nRecords/(256*256)) % 256;
    abyFileHeader[7] = (psDBF->nRecords/(256*256*256)) % 256;
    fseek(psDBF->fp, 0, 0);
    fwrite(abyFileHeader, 32, 1, psDBF->fp);
  }
  /* Close, and free resources. */
  result = fclose(psDBF->fp);
  if (psDBF->panFieldOffset != NULL) {
    free(psDBF->panFieldOffset);
    free(psDBF->panFieldSize);
    free(psDBF->panFieldDecimals);
    free(psDBF->pachFieldType);
  }
  free(psDBF->pszHeader);
  free(psDBF->pszCurrentRecord);
  free(psDBF->filename);  /* 0.32.4 */
  free(psDBF);
  if (pszStringField != NULL) {
    free(pszStringField);
    pszStringField = NULL;
    nStringFieldLen = 0;
  }
  return result;
}


static int xbase_close (lua_State *L) {
  DBFHandle *hnd;
  int result;
  hnd = aux_gethandle(L, 1, "xbase.close");
  result = DBFClose(*hnd);
  /* ignore closed files */
  if (*hnd != NULL) {  /* this will work */
    *hnd = NULL;
    xbase_gc(L);
  }
  lua_pushboolean(L, result == 0);
  return 1;
}


/************************************************************************/
/*                             DBFCreate()                              */
/*                                                                      */
/*      Create a new .dbf file.                                         */
/************************************************************************/

DBFHandle SHPAPI_CALL DBFCreate (const char *pszFilename) {
  DBFHandle	psDBF;
  FILE *fp;
  char *pszFullname, *pszBasename;
  int i;
  /* Compute the base (layer) name. If there is any extension on the passed in filename we will strip it off. */
  pszBasename = (char *)malloc(strlen(pszFilename)+5);
  strcpy(pszBasename, pszFilename);
  for(i = strlen(pszBasename)-1;
	 i > 0 && pszBasename[i] != '.' && pszBasename[i] != '/'
	       && pszBasename[i] != '\\';
	 i--) {}
  if (pszBasename[i] == '.')
        pszBasename[i] = '\0';
  pszFullname = (char *)malloc(strlen(pszBasename) + 5);
  sprintf(pszFullname, "%s.dbf", pszBasename);
  free(pszBasename);
  /* Create the file. */
  fp = fopen(pszFullname, "wb");
  if (fp == NULL) {
    free(pszFullname);
    return NULL;
  }
  fputc(0, fp);
  fclose(fp);
  fp = fopen(pszFullname, "rb+");
  if (fp == NULL) {
    free(pszFullname);
    return NULL;
  }
  /* Create the info structure. */
  psDBF = (DBFHandle)malloc(sizeof(DBFInfo));
  psDBF->fp = fp;
  psDBF->nRecords = 0;
  psDBF->nFields = 0;
  psDBF->nRecordLength = 1;
  psDBF->nHeaderLength = 33;
  psDBF->panFieldOffset = NULL;
  psDBF->panFieldSize = NULL;
  psDBF->panFieldDecimals = NULL;
  psDBF->pachFieldType = NULL;
  psDBF->pszHeader = NULL;
  psDBF->nCurrentRecord = -1;
  psDBF->bCurrentRecordModified = FALSE;
  psDBF->pszCurrentRecord = NULL;
  psDBF->bNoHeader = TRUE;
  psDBF->filename = (char *)malloc(strlen(pszFullname));  /* 0.32.4 */
  psDBF->codepage = 0x00;
  strcpy(psDBF->filename, pszFullname);
  free(pszFullname);
  return psDBF;
}


static int xbase_new (lua_State *L) {
  const char *fn, *pn, *fieldname, *typename;
  int i, n, nargs, pop;
  DBFHandle hnd;
  fn = luaL_checkstring(L, 1);
  pn = "xbase.new";
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs < 2)
    luaL_error(L, "Error in " LUA_QS ": need at least one field.", pn, fn);
  hnd = DBFCreate(fn);
  if (hnd == NULL)
    luaL_error(L, "Error in " LUA_QS ": creating " LUA_QS " failed.", pn, fn);
  for (i=2; i <= nargs; i++) {
    pop = 3;
    if (!lua_ispair(L, i))
      luaL_error(L, "Error in " LUA_QS ": pair expected for argument #%d.", pn, i);
    agn_pairgeti(L, i, 1);  /* get lhs */
    if (lua_type(L, -1) != LUA_TSTRING)
      luaL_error(L, "Error in " LUA_QS ": left-hand side of pair must be a string.", pn);
    fieldname = agn_tostring(L, -1);
    if (strcmp(fieldname, "codepage") == 0) {
      int codepage;
      if (nargs == 2)
        luaL_error(L, "Error in " LUA_QS ": at least one field expected.", pn);
      agn_pairgeti(L, i, 2);  /* get rhs */
      if (lua_type(L, -1) != LUA_TNUMBER)
        luaL_error(L, "Error in " LUA_QS ": number for codepage expected.", pn);
      codepage = agn_tonumber(L, -1);
      if (codepage < 0 || codepage > 255)
        luaL_error(L, "Error in " LUA_QS ": codepage must be in [0, 255].", pn);
      hnd->codepage = codepage;
      agn_poptoptwo(L);      
      continue;
    }
    agn_pairgeti(L, i, 2);  /* get rhs */
    if (lua_ispair(L, -1)) {
      agn_pairgeti(L, -1, 1);  /* get lhs (2nd pair) */
      if (lua_type(L, -1) != LUA_TSTRING)
        luaL_error(L, "Error in " LUA_QS ": left-hand side of second pair must be a string.", pn);
      typename = agn_tostring(L, -1);
      agn_pairgeti(L, -2, 2);  /* get rhs (2nd pair) */
      if (lua_type(L, -1) != LUA_TNUMBER)
        luaL_error(L, "Error in " LUA_QS ": right-hand side of second pair must be a number.", pn);
      n = (int)agn_tonumber(L, -1);
      agn_poptop(L);  /* pop rhs of 2nd pair */
    } else if (lua_type(L, -1) == LUA_TSTRING) {
      typename = agn_tostring(L, -1);
      if (strcmp(typename, "number") == 0) n = 15;  /* scale = number of digits following the decimal point */
      else if (strcmp(typename, "string") == 0) n = 64;
      else n = 1;  /* for Logical value */
      pop = 2;
    } else {
      typename = NULL; n = 0; /* to avoid compiler warnings */
      luaL_error(L, "Error in " LUA_QS ": right-hand side of pair must be a string or a pair.", pn);
    }
    if (strcmp(typename, "string") == 0) {
      if (n < 1 || n > 254)
        luaL_error(L, "Error in " LUA_QS ": string size not in [1, 254].", pn);
      if (DBFAddField(hnd, fieldname, FTString, n, 0) == -1)
        luaL_error(L, "Error in " LUA_QS ": adding string field to " LUA_QS " failed.", pn, fn);
      lua_pop(L, pop);
    } else if (strcmp(typename, "number") == 0) {
      if (n < 0 || n > 15)
        luaL_error(L, "Error in " LUA_QS ": scale not in [0, 15].", pn);
      if (DBFAddField(hnd, fieldname, FTDouble, 19, n) == -1)
        luaL_error(L, "Error in " LUA_QS ": adding number field to " LUA_QS " failed.", pn, fn);
      lua_pop(L, pop);
    } else if (strcmp(typename, "boolean") == 0) {
      if (DBFAddField(hnd, fieldname, FTLogical, 1, 0) == -1)
        luaL_error(L, "Error in " LUA_QS ": adding number field to " LUA_QS " failed.", pn, fn);
      lua_pop(L, pop);
    } else
      luaL_error(L, "Error in " LUA_QS ": unknown field type " LUA_QS " to be added to " LUA_QS ".",
        pn, typename, fn);
  }
  DBFClose(hnd);
  lua_pushnil(L);
  return 1;
}

/************************************************************************/
/*                            DBFAddField()                             */
/*                                                                      */
/*      Add a field to a newly created .dbf file before any records     */
/*      are written.                                                    */
/************************************************************************/

int SHPAPI_CALL DBFAddField (DBFHandle psDBF, const char *pszFieldName,
            DBFFieldType eType, int nWidth, int nDecimals) {
  char *pszFInfo;
  int i;
  /* Do some checking to ensure we can add records to this file. */
  if (psDBF->nRecords > 0)
    return -1;
  if (!psDBF->bNoHeader)
    return -1;
  if (eType != FTDouble && nDecimals != 0)
    return -1;
  if (nWidth < 1)
    return -1;
  /* SfRealloc all the arrays larger to hold the additional field information. */
  psDBF->nFields++;
  psDBF->panFieldOffset = (int *)
    SfRealloc( psDBF->panFieldOffset, sizeof(int) * psDBF->nFields);
  psDBF->panFieldSize = (int *)
    SfRealloc( psDBF->panFieldSize, sizeof(int) * psDBF->nFields);
  psDBF->panFieldDecimals = (int *)
    SfRealloc( psDBF->panFieldDecimals, sizeof(int) * psDBF->nFields);
  psDBF->pachFieldType = (char *)
    SfRealloc( psDBF->pachFieldType, sizeof(char) * psDBF->nFields);
  /* Assign the new field information fields. */
  psDBF->panFieldOffset[psDBF->nFields-1] = psDBF->nRecordLength;
  psDBF->nRecordLength += nWidth;
  psDBF->panFieldSize[psDBF->nFields-1] = nWidth;
  psDBF->panFieldDecimals[psDBF->nFields-1] = nDecimals;
  if (eType == FTLogical)
    psDBF->pachFieldType[psDBF->nFields-1] = 'L';
  else if (eType == FTString)
    psDBF->pachFieldType[psDBF->nFields-1] = 'C';
  /* else if (eType == FTInteger)
    psDBF->pachFieldType[psDBF->nFields-1] = 'I'; */
  else
    psDBF->pachFieldType[psDBF->nFields-1] = 'N';
  /* Extend the required header information. */
  psDBF->nHeaderLength += 32;
  psDBF->bUpdated = FALSE;
  psDBF->pszHeader = (char *)SfRealloc(psDBF->pszHeader,psDBF->nFields*32);
  pszFInfo = psDBF->pszHeader + 32 * (psDBF->nFields-1);
  for (i = 0; i < 32; i++)
    pszFInfo[i] = '\0';
  if ((int) strlen(pszFieldName) < 10)
    strncpy(pszFInfo, pszFieldName, strlen(pszFieldName));
  else
    strncpy(pszFInfo, pszFieldName, 10);
  pszFInfo[11] = psDBF->pachFieldType[psDBF->nFields-1];
  if (eType == FTString) {
    pszFInfo[16] = nWidth % 256;
    pszFInfo[17] = nWidth / 256;
  } else {
    pszFInfo[16] = nWidth;
    pszFInfo[17] = nDecimals;
  }
  /* Make the current record buffer appropriately larger. */
  psDBF->pszCurrentRecord = (char *)SfRealloc(psDBF->pszCurrentRecord,
    psDBF->nRecordLength);
  return psDBF->nFields-1;
}

/************************************************************************/
/*                          DBFReadAttribute()                          */
/*                                                                      */
/*      Read one of the attribute fields of a record.                   */
/************************************************************************/

static void *DBFReadAttribute (DBFHandle psDBF, int hEntity, int iField, char chReqType) {
    int	nRecordOffset;
    unsigned char *pabyRec;
    void *pReturnField = NULL;
    static double dDoubleField;
    /* Verify selection. */
    if (hEntity < 0 || hEntity >= psDBF->nRecords)
        return NULL;
    if (iField < 0 || iField >= psDBF->nFields)
        return NULL;
    /*	Have we read the record ? */
    if (psDBF->nCurrentRecord != hEntity) {
	  DBFFlushRecord(psDBF);
	  nRecordOffset = psDBF->nRecordLength * hEntity + psDBF->nHeaderLength;
	  if (fseek(psDBF->fp, nRecordOffset, 0 ) != 0) {
            fprintf(stderr, "Error in " LUA_QS ": fseek(%d) failed on DBF file.\n",
                     "xbase", nRecordOffset);
            return NULL;
      }
	  if (fread(psDBF->pszCurrentRecord, psDBF->nRecordLength, 1, psDBF->fp ) != 1) {
            fprintf(stderr, "Error in " LUA_QS ": fread(%d) failed on DBF file.\n",
                     "xbase", psDBF->nRecordLength);
            return NULL;
      }
	   psDBF->nCurrentRecord = hEntity;
    }
    pabyRec = (unsigned char *) psDBF->pszCurrentRecord;
    /*	Ensure our field buffer is large enough to hold this buffer. */
    if (psDBF->panFieldSize[iField]+1 > nStringFieldLen) {
	  nStringFieldLen = psDBF->panFieldSize[iField]*2 + 10;
	  pszStringField = (char *)SfRealloc(pszStringField,nStringFieldLen);
    }
    /* Extract the requested field. */
    strncpy( pszStringField,
	     ((const char *)pabyRec) + psDBF->panFieldOffset[iField],
	     psDBF->panFieldSize[iField]);
    pszStringField[psDBF->panFieldSize[iField]] = '\0';
    pReturnField = pszStringField;
    /* Decode the field. */
    if (chReqType == 'N') {
        dDoubleField = atof(pszStringField);
        pReturnField = &dDoubleField;
    }
    /* Should we trim white space off the string attribute value ? */
#ifdef TRIM_DBF_WHITESPACE
    else {
        char	*pchSrc, *pchDst;
        pchDst = pchSrc = pszStringField;
        while( *pchSrc == ' ')
            pchSrc++;
        while( *pchSrc != '\0')
            *(pchDst++) = *(pchSrc++);
        *pchDst = '\0';
        while( pchDst != pszStringField && *(--pchDst) == ' ')
            *pchDst = '\0';
    }
#endif
    return pReturnField;
}


/************************************************************************/
/*                        DBFReadIntegerAttribute()                     */
/*                                                                      */
/*      Read an integer attribute.                                      */
/************************************************************************/

/* int32_t SHPAPI_CALL DBFReadIntegerAttribute (DBFHandle psDBF, int iRecord, int iField ) {
  int32_t *pdValue;
  pdValue = (int32_t *)DBFReadAttribute(psDBF, iRecord, iField, 'N');
  if (pdValue == NULL)
    return 0;
  else
    return((int32_t)*pdValue );
} */

/************************************************************************/
/*                        DBFReadDoubleAttribute()                      */
/*                                                                      */
/*      Read a double attribute.                                        */
/************************************************************************/

/* failure parameter added for Agena binding 0.32.4, 06.06.2010 */

double SHPAPI_CALL DBFReadDoubleAttribute (DBFHandle psDBF, int iRecord, int iField, int *failure) {
  double *pdValue;
  *failure = 0;
  pdValue = (double *)DBFReadAttribute(psDBF, iRecord, iField, 'N');
  if (pdValue == NULL) {
    *failure = 1;
    return 0.0;
  } else
    return *pdValue;
}

/************************************************************************/
/*                        DBFReadStringAttribute()                      */
/*                                                                      */
/*      Read a string attribute.                                        */
/************************************************************************/

const char SHPAPI_CALL1(*)
DBFReadStringAttribute (DBFHandle psDBF, int iRecord, int iField) {
  return((const char *) DBFReadAttribute( psDBF, iRecord, iField, 'C'));
}

/************************************************************************/
/*                        DBFReadLogicalAttribute()                     */
/*                                                                      */
/*      Read a logical attribute.                                       */
/************************************************************************/

const char SHPAPI_CALL1(*)
DBFReadLogicalAttribute (DBFHandle psDBF, int iRecord, int iField) {
  return( (const char *) DBFReadAttribute(psDBF, iRecord, iField, 'L'));
}


static int xbase_readvalue (lua_State *L) {
  DBFHandle *hnd;
  int record, field, maxrecords, maxfields;
  const char *pn;
  pn = "xbase.readvalue";
  record = agnL_checkinteger(L, 2);
  field = agnL_checkinteger(L, 3);
  hnd = aux_gethandle(L, 1, pn);
  maxrecords = DBFGetRecordCount(*hnd);
  maxfields = DBFGetFieldCount(*hnd);
  if (maxrecords == 0 || maxfields == 0)
    luaL_error(L, "Error in " LUA_QS ": data base is empty.", pn);
  if (record < 1 || record > maxrecords)
    luaL_error(L, "Error in " LUA_QS ": record does not exist.", pn);
  if (field < 1 || field > maxfields)
    luaL_error(L, "Error in " LUA_QS ": field does not exist.", pn);
  field--; record--;
  if (DBFGetFieldInfo(*hnd, field, NULL, NULL, NULL) == FTString) {
    const char *result;
    result = DBFReadStringAttribute(*hnd, record, field);
    if (result == NULL)
      luaL_error(L, "Error in " LUA_QS ": no string stored in data base.", pn);
    else
      lua_pushstring(L, result);
  }
  else if (DBFGetFieldInfo(*hnd, field, NULL, NULL, NULL) == FTDouble) {
    lua_Number result;
    int failure;
    result = DBFReadDoubleAttribute(*hnd, record, field, &failure);
    if (failure)
      luaL_error(L, "Error in " LUA_QS ": could not read number in data base.", pn);
    else
      lua_pushnumber(L, result);
  }
  /* else if (DBFGetFieldInfo(*hnd, field, NULL, NULL, NULL) == FTInteger) {
    int32_t result;
    result = DBFReadIntegerAttribute(*hnd, record, field);
    #if BYTE_ORDER == BIG_ENDIAN
    tools_swapint32_t(&result);
    #endif
    lua_pushnumber(L, result);
  } */
  else if (DBFGetFieldInfo(*hnd, field, NULL, NULL, NULL) == FTLogical) {
    const char *result;
    result = DBFReadLogicalAttribute(*hnd, record, field);
    if (result == NULL)
      luaL_error(L, "Error in " LUA_QS ": no logical value stored in data base.", pn);
    else {
      switch (*result) {
        case 'Y': case 'y': case 'T': case 't':
          lua_pushtrue(L);
          break;
        case 'N': case 'n': case 'F': case 'f':
          lua_pushfalse(L);
          break;
        default:
          lua_pushfail(L);
      }
    }
  } else
    luaL_error(L, "Error in " LUA_QS ": unknown type of data.", pn);
  return 1;
}

/************************************************************************/
/*                         DBFIsAttributeNULL()                         */
/*                                                                      */
/*      Return TRUE if value for field is NULL.                         */
/*                                                                      */
/*      Contributed by Jim Matthews.                                    */
/************************************************************************/

int SHPAPI_CALL DBFIsAttributeNULL (DBFHandle psDBF, int iRecord, int iField) {
  const char	*pszValue;
  pszValue = DBFReadStringAttribute(psDBF, iRecord, iField);
  switch(psDBF->pachFieldType[iField]) {
    case 'N':
    case 'F':
      /* NULL numeric fields have value "****************" */
      return pszValue[0] == '*';
    case 'D':
      /* NULL date fields have value "00000000" */
      return strncmp(pszValue, "00000000", 8) == 0;
    case 'L':
      /* NULL boolean fields have value "?" */
      return pszValue[0] == '?';
    default:
      /* empty string fields are considered NULL */
      return strlen(pszValue) == 0;
  }
}


static int xbase_isVoid (lua_State *L) {
  DBFHandle *hnd;
  int record, field, maxrecords, maxfields;
  const char *pn;
  pn = "xbase.isVoid";
  record = agnL_checkinteger(L, 2);
  field = agnL_checkinteger(L, 3);
  hnd = aux_gethandle(L, 1, pn);
  maxrecords = DBFGetRecordCount(*hnd);
  maxfields = DBFGetFieldCount(*hnd);
  if (maxrecords == 0 || maxfields == 0)
    luaL_error(L, "Error in " LUA_QS ": data base is empty.", pn);
  if (record < 1 || record > maxrecords)
    luaL_error(L, "Error in " LUA_QS ": record does not exist.", pn);
  if (field < 1 || field > maxfields)
    luaL_error(L, "Error in " LUA_QS ": field does not exist.", pn);
  lua_pushboolean(L, DBFIsAttributeNULL(*hnd, --record, --field));
  return 1;
}

/************************************************************************/
/*                          DBFGetFieldCount()                          */
/*                                                                      */
/*      Return the number of fields in this table.                      */
/************************************************************************/

int SHPAPI_CALL DBFGetFieldCount (DBFHandle psDBF) {
  return( psDBF->nFields);
}

/************************************************************************/
/*                       DBFGetNativeFieldType()                        */
/*                                                                      */
/*      Return the DBase field type for the specified field.            */
/*                                                                      */
/*      Value can be one of: 'C' (String), 'D' (Date), 'F' (Float),     */
/*                           'N' (Numeric, with or without decimal),    */
/*                           'L' (Logical),                             */
/*                           'M' (Memo: 10 digits .DBT block ptr)       */
/************************************************************************/

char SHPAPI_CALL DBFGetNativeFieldType (DBFHandle psDBF, int iField) {
  if (iField >=0 && iField < psDBF->nFields)
    return psDBF->pachFieldType[iField];
  return  ' ';
}

/************************************************************************/
/*                         DBFGetRecordCount()                          */
/*                                                                      */
/*      Return the number of records in this table.                     */
/************************************************************************/

int SHPAPI_CALL DBFGetRecordCount (DBFHandle psDBF) {
  return( psDBF->nRecords);
}

/* assumes that table is on the top of the stack */

static void setintegerfield (lua_State *L, const char *key, int value) {
  lua_pushinteger(L, value);
  lua_setfield(L, -2, key);
}

/* assumes that table is on the top of the stack */

static void setstringfield (lua_State *L, const char *key, const char *value) {
  lua_pushstring(L, value);
  lua_setfield(L, -2, key);
}

/*
Code pages supported by dBASE III

from: http://www.clicketyclick.dk/databases/xbase/format/dbf.html#DBF_NOTE_5_TARGET

(Foxpro) Code pages: These values follow the DOS / Windows Code Page values.
Value 	Description 	Code page
01h 	DOS USA	code page 437
02h 	DOS Multilingual	code page 850
03h 	Windows ANSI	code page 1252
04h 	Standard Macintosh
64h 	EE MS-DOS	code page 852
65h 	Nordic MS-DOS	code page 865
66h 	Russian MS-DOS	code page 866
67h 	Icelandic MS-DOS
68h 	Kamenicky (Czech) MS-DOS
69h 	Mazovia (Polish) MS-DOS
6Ah 	Greek MS-DOS (437G)
6Bh 	Turkish MS-DOS
96h 	Russian Macintosh
97h 	Eastern European Macintosh
98h 	Greek Macintosh
C8h 	Windows EE	code page 1250
C9h 	Russian Windows
CAh 	Turkish Windows
CBh 	Greek Windows
*/


static int xbase_attrib (lua_State *L) {
  DBFHandle *hnd;
  DBFInfo dbf;
  int i, nfields;
  char szTitle[12];
  unsigned char s[2];
  hnd = aux_gethandle(L, 1, "xbase.attrib");
  dbf = **hnd;
  lua_createtable(L, 0, 3);
  setstringfield(L, "filename", dbf.filename);
  nfields = DBFGetFieldCount(*hnd);
  setintegerfield(L, "fields", nfields);
  setintegerfield(L, "records", DBFGetRecordCount(*hnd));
  setintegerfield(L, "headerlength", dbf.nHeaderLength);
  setintegerfield(L, "recordlength", dbf.nRecordLength);
  setintegerfield(L, "lastmodified", dbf.lastmodified);
  setintegerfield(L, "codepage", dbf.codepage);
  lua_pushstring(L, "fieldinfo");
  lua_newtable(L);
  lua_settable(L, -3);
  for (i=0; i < nfields; i++) {
    DBFFieldType eType;
    const char *pszTypeName = NULL;
    int nWidth = 0, nDecimals = 0;
    eType = DBFGetFieldInfo(*hnd, i, szTitle, &nWidth, &nDecimals);
    if (eType == FTString)
      pszTypeName = "string";
    else if (eType == FTInteger || eType == FTDouble)
      pszTypeName = "number";
    else if (eType == FTLogical)
      pszTypeName = "boolean";
    else if (eType == FTInvalid)
      pszTypeName = "unknown";
    lua_getfield(L, -1, "fieldinfo");
    lua_pushinteger(L, i+1);
    lua_createtable(L, 0, 4);
    setstringfield(L, "type", pszTypeName);
    s[0] = DBFGetNativeFieldType(*hnd, i);
    s[1] = '\0';  /* better sure than sorry */
    setstringfield(L, "nativetype", (const char *)s);
    setstringfield(L, "title", szTitle);
    setintegerfield(L, "width", nWidth);
    setintegerfield(L, "scale", nDecimals);
    lua_settable(L, -3);
    agn_poptop(L);
  }
  return 1;
}


/************************************************************************/
/*                          DBFGetFieldInfo()                           */
/*                                                                      */
/*      Return any requested information about the field.               */
/************************************************************************/

DBFFieldType SHPAPI_CALL
DBFGetFieldInfo (DBFHandle psDBF, int iField, char * pszFieldName,
                 int * pnWidth, int * pnDecimals) {
    if (iField < 0 || iField >= psDBF->nFields)
        return( FTInvalid);
    if (pnWidth != NULL)
        *pnWidth = psDBF->panFieldSize[iField];
    if (pnDecimals != NULL)
        *pnDecimals = psDBF->panFieldDecimals[iField];
    if (pszFieldName != NULL) {
	  int	i;
	  strncpy( pszFieldName, (char *) psDBF->pszHeader+iField*32, 11);
	  pszFieldName[11] = '\0';
	  for (i=10; i > 0 && pszFieldName[i] == ' '; i--)
	    pszFieldName[i] = '\0';
    }
    if (psDBF->pachFieldType[iField] == 'L')
	  return FTLogical;
    else if (psDBF->pachFieldType[iField] == 'N'
             || psDBF->pachFieldType[iField] == 'F'
             || psDBF->pachFieldType[iField] == 'D') {
	  if (psDBF->panFieldDecimals[iField] > 0)
	    return FTDouble;
	  else
	    return FTInteger;
    } else {
	  return FTString;
    }
}

/************************************************************************/
/*                         DBFWriteAttribute()                          */
/*									                                    */
/*	Write an attribute record to the file.				                */
/************************************************************************/

static int DBFWriteAttribute (DBFHandle psDBF, int hEntity, int iField, void * pValue) {
  int	nRecordOffset, i, j, nRetResult = TRUE;
  unsigned char *pabyRec;
  char szSField[400], szFormat[20];
  /* Is this a valid record ? */
  if (hEntity < 0 || hEntity > psDBF->nRecords || hEntity == INT_MAX)  /* INT_MAX check added 0.32.4 */
    return FALSE;
  if (psDBF->bNoHeader)
    DBFWriteHeader(psDBF);
  /* Is this a brand new record ? */
  if (hEntity == psDBF->nRecords) {
    DBFFlushRecord(psDBF);
    psDBF->nRecords++;
    for (i = 0; i < psDBF->nRecordLength; i++)
      psDBF->pszCurrentRecord[i] = ' ';
    psDBF->nCurrentRecord = hEntity;
  }
  /* Is this an existing record, but different than the last one we accessed ? */
  if (psDBF->nCurrentRecord != hEntity) {
    DBFFlushRecord(psDBF);
    nRecordOffset = psDBF->nRecordLength * hEntity + psDBF->nHeaderLength;
    fseek(psDBF->fp, nRecordOffset, 0);
    fread(psDBF->pszCurrentRecord, psDBF->nRecordLength, 1, psDBF->fp);
    psDBF->nCurrentRecord = hEntity;
  }
  pabyRec = (unsigned char *)psDBF->pszCurrentRecord;
  psDBF->bCurrentRecordModified = TRUE;
  psDBF->bUpdated = TRUE;
  /* Translate NULL value to valid DBF file representation. Contributed by Jim Matthews. */
  if (pValue == NULL) {
    switch(psDBF->pachFieldType[iField]) {
      case 'N':
      case 'F':
	  /* NULL numeric fields have value "****************" */
        memset((char *)(pabyRec+psDBF->panFieldOffset[iField]), '*',
                    psDBF->panFieldSize[iField]);
        break;
      case 'D':
	    /* NULL date fields have value "00000000" */
        memset((char *)(pabyRec+psDBF->panFieldOffset[iField]), '0',
                    psDBF->panFieldSize[iField]);
        break;
      case 'L':
	    /* NULL boolean fields have value "?" */
        memset( (char *)(pabyRec+psDBF->panFieldOffset[iField]), '?',
                    psDBF->panFieldSize[iField]);
        break;
      default:
        /* empty string fields are considered NULL */
        memset( (char *)(pabyRec+psDBF->panFieldOffset[iField]), '\0',
                    psDBF->panFieldSize[iField]);
        break;
    }
    return TRUE;
  }
  /* Assign all the record fields. */
  switch (psDBF->pachFieldType[iField]) {
    case 'D':
    case 'N':
    case 'F':
	  if (psDBF->panFieldDecimals[iField] == 0) {
        int	nWidth = psDBF->panFieldSize[iField];
        if (sizeof(szSField)-2 < nWidth)
          nWidth = sizeof(szSField)-2;
	    sprintf(szFormat, "%%%dd", nWidth);
	    sprintf(szSField, szFormat, (int) *((double *) pValue));
	    if ((int)strlen(szSField) > psDBF->panFieldSize[iField]) {
	      szSField[psDBF->panFieldSize[iField]] = '\0';
          nRetResult = FALSE;
        }
	    strncpy((char *)(pabyRec+psDBF->panFieldOffset[iField]),
		  szSField, strlen(szSField));
	  } else {
        int nWidth = psDBF->panFieldSize[iField];
        if (sizeof(szSField)-2 < nWidth)
          nWidth = sizeof(szSField)-2;
	    sprintf(szFormat, "%%%d.%df",
          nWidth, psDBF->panFieldDecimals[iField]);
	    sprintf(szSField, szFormat, *((double *)pValue));
	    if ((int) strlen(szSField) > psDBF->panFieldSize[iField]) {
	      szSField[psDBF->panFieldSize[iField]] = '\0';
                nRetResult = FALSE;
        }
	    strncpy((char *)(pabyRec+psDBF->panFieldOffset[iField]),
		  szSField, strlen(szSField));
	  }
	  break;
    case 'L':
      if (psDBF->panFieldSize[iField] >= 1  &&
        (*(char*)pValue == 'F' || *(char*)pValue == 'T'))
         *(pabyRec+psDBF->panFieldOffset[iField]) = *(char*)pValue;
      break;
    default:
      if ((int) strlen((char *)pValue) > psDBF->panFieldSize[iField]) {
        j = psDBF->panFieldSize[iField];
        nRetResult = FALSE;
      } else {
        memset(pabyRec+psDBF->panFieldOffset[iField], ' ',
                    psDBF->panFieldSize[iField]);
        j = strlen((char *) pValue);
      }
      strncpy((char *)(pabyRec+psDBF->panFieldOffset[iField]), (char *)pValue, j);
      break;
  }
  return nRetResult;
}


/* helper function for write functions */
void aux_checkpositions (lua_State *L, DBFHandle hnd, int *record, int *field, const char *pn) {
  int actrecords, actfields;
  actrecords = DBFGetRecordCount(hnd);
  actfields = DBFGetFieldCount(hnd);
  *record = agnL_checkinteger(L, 2) - 1;
  if (actfields == 0)
    luaL_error(L, "Error in " LUA_QS ": data base contains no fields.", pn);
  if (*record < 0 || *record > actrecords)
    luaL_error(L, "Error in " LUA_QS ": invalid record given, must be in [1, %d].", pn, actrecords+1);
  *field = agnL_checkinteger(L, 3) - 1;
  if (*field < 0 || *field >= actfields)
    luaL_error(L, "Error in " LUA_QS ": invalid field given, must be in [1, %d].", pn, actfields);
}

/************************************************************************/
/*                      DBFWriteDoubleAttribute()                       */
/*                                                                      */
/*      Write a double attribute.                                       */
/************************************************************************/

int SHPAPI_CALL
DBFWriteDoubleAttribute (DBFHandle psDBF, int iRecord, int iField, double dValue) {
  return DBFWriteAttribute(psDBF, iRecord, iField, (void *) &dValue);
}


static int xbase_writenumber (lua_State *L) {
  DBFHandle *hnd;
  int record, field;
  const char *pn;
  lua_Number val;
  pn = "xbase.writenumber";
  hnd = aux_gethandle(L, 1, pn);
  aux_checkpositions(L, *hnd, &record, &field, pn);
  val = agnL_checknumber(L, 4);
  if (DBFGetFieldInfo(*hnd, field, NULL, NULL, NULL) == FTDouble)
    lua_pushboolean(L, DBFWriteDoubleAttribute(*hnd, record, field, val));
  else
    luaL_error(L, "Error in " LUA_QS ": number expected.", pn);
  return 1;
}


/************************************************************************/
/*                      DBFWriteIntegerAttribute()                      */
/*                                                                      */
/*      Write an integer attribute.                                     */
/************************************************************************/

/* int SHPAPI_CALL DBFWriteIntegerAttribute (DBFHandle psDBF, int iRecord, int iField, int32_t nValue) {
  //double dValue = nValue;
  int32_t dValue = nValue;
  return DBFWriteAttribute(psDBF, iRecord, iField, (void *)&dValue);
} */


/* static int xbase_writeinteger (lua_State *L) {
  DBFHandle *hnd;
  int record, field;
  const char *pn;
  int32_t val;
  pn = "xbase.writeinteger";
  hnd = aux_gethandle(L, 1, pn);
  aux_checkpositions(L, *hnd, &record, &field, pn);
  val = (int32_t)agnL_checkinteger(L, 4);
  #if BYTE_ORDER == BIG_ENDIAN
  tools_swapint32_t(&val);
  #endif
  if (DBFGetFieldInfo(*hnd, field, NULL, NULL, NULL) == FTInteger)
    lua_pushboolean(L, DBFWriteIntegerAttribute(*hnd, record, field, val));
  else
    luaL_error(L, "Error in " LUA_QS ": expected a number.", pn);
  return 1;
} */

/************************************************************************/
/*                      DBFWriteStringAttribute()                       */
/*                                                                      */
/*      Write a string attribute.                                       */
/************************************************************************/

int SHPAPI_CALL DBFWriteStringAttribute (DBFHandle psDBF, int iRecord, int iField,
    const char *pszValue) {
  return DBFWriteAttribute(psDBF, iRecord, iField, (void *)pszValue);
}


static int xbase_writestring (lua_State *L) {
  DBFHandle *hnd;
  int record, field;
  const char *val, *pn;
  pn = "xbase.writestring";
  hnd = aux_gethandle(L, 1, pn);
  aux_checkpositions(L, *hnd, &record, &field, pn);
  val = luaL_checkstring(L, 4);
  if (DBFGetFieldInfo(*hnd, field, NULL, NULL, NULL ) == FTString)
    lua_pushboolean(L, DBFWriteStringAttribute(*hnd, record, field, val));
  else
    luaL_error(L, "Error in " LUA_QS ": string value expected.", pn);
  return 1;
}


/************************************************************************/
/*                      DBFWriteNULLAttribute()                         */
/*                                                                      */
/*      Write a string attribute.                                       */
/************************************************************************/

static int xbase_purge (lua_State *L) {
  DBFHandle *hnd;
  int record, field;
  const char *pn;
  pn = "xbase.purge";
  hnd = aux_gethandle(L, 1, pn);
  aux_checkpositions(L, *hnd, &record, &field, pn);
  lua_pushboolean(L,
    DBFWriteAttribute(*hnd, record, field, NULL));
  return 1;
}


/************************************************************************/
/*                      DBFWriteLogicalAttribute()                      */
/*                                                                      */
/*      Write a logical attribute.                                      */
/************************************************************************/

int SHPAPI_CALL DBFWriteLogicalAttribute (DBFHandle psDBF, int iRecord, int iField, const char lValue) {
  return DBFWriteAttribute(psDBF, iRecord, iField, (void *) (&lValue));
}


static int xbase_writeboolean (lua_State *L) {
  DBFHandle *hnd;
  int record, field, val;
  const char *pn;
  pn = "xbase.writeboolean";
  hnd = aux_gethandle(L, 1, pn);
  aux_checkpositions(L, *hnd, &record, &field, pn);
  val = (lua_isboolean(L, 4)) ? lua_toboolean(L, 4) : -1;
  if (DBFGetFieldInfo(*hnd, field, NULL, NULL, NULL) == FTLogical) {
    switch (val) {
      case 0:
        lua_pushboolean(L, DBFWriteLogicalAttribute(*hnd, record, field, 'F'));
        break;
      case 1:
        lua_pushboolean(L, DBFWriteLogicalAttribute(*hnd, record, field, 'T'));
        break;
      default:
        luaL_error(L, "Error in " LUA_QS ": true or false as fourth argument expected.", pn);
    }
  } else
    luaL_error(L, "Error in " LUA_QS ": boolean value expected.", pn);
  return 1;
}


static int xbase_lock (lua_State *L) {
  size_t nargs;
  int hnd;
  DBFHandle *ud;
  DBFInfo dbf;
  const char *pn = "xbase.lock";
  off64_t start, size;
  hnd = 0;
  ud = aux_gethandle(L, 1, pn);
  dbf = **ud;
  if (ud == NULL)
    luaL_error(L, "Error in " LUA_QS ": invalid file handle.", pn);
  hnd = fileno(dbf.fp);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1) {  /* lock entire file (in Windows lock 2^63 bytes only) */
    start = 0;
    size = 0;
  } else {
    /* lock from current file position */
    start = my_fpos(hnd);
    size = agnL_checknumber(L, 2);
    if (size < 0) luaL_error(L, "Error in " LUA_QS ": must lock at least one byte.", pn);
  }
  lua_pushboolean(L, my_lock(hnd, start, size) == 0);
  return 1;
}


static int xbase_unlock (lua_State *L) {
  size_t nargs;
  int hnd;
  DBFHandle *ud;
  DBFInfo dbf;
  const char *pn = "xbase.unlock";
  off64_t start, size;
  hnd = 0;
  ud = aux_gethandle(L, 1, pn);
  dbf = **ud;
  if (ud == NULL)
    luaL_error(L, "Error in " LUA_QS ": invalid file handle.", pn);
  hnd = fileno(dbf.fp);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1) {  /* lock entire file (in Windows lock 2^63 bytes only) */
    start = 0;
    size = 0;
  } else {
    /* lock from current file position */
    start = my_fpos(hnd);
    size = agnL_checknumber(L, 2);
    if (size < 0) luaL_error(L, "Error in " LUA_QS ": must lock at least one byte.", pn);
  }
  lua_pushboolean(L, my_unlock(hnd, start, size) == 0);
  return 1;
}


static int xbase_field (lua_State *L) {
  DBFHandle *hnd;
  int nrecords, nfields, fieldno, fieldinfo, i, restype, failure;
  lua_Number val;
  const char *pn, *result, *structure;
  pn = "xbase.field";
  hnd = aux_gethandle(L, 1, pn);
  fieldno = agnL_checkinteger(L, 2);
  structure = luaL_optstring(L, 3, "sequence");
  nrecords = DBFGetRecordCount(*hnd);
  nfields = DBFGetFieldCount(*hnd);
  if (fieldno < 1 || fieldno > nfields)
    luaL_error(L, "Error in " LUA_QS ": requested field out of range.", pn);
  if (nrecords == 0 || nfields == 0)
    luaL_error(L, "Error in " LUA_QS ": data base is empty.", pn);
  if (strcmp(structure, "sequence") == 0)
    restype = 1;
  else if (strcmp(structure, "set") == 0)
    restype = 0;
  else {
    restype = -1;
    luaL_error(L, "Error in " LUA_QS ": argument #3 invalid.", pn);
  }
  fieldno--;
  fieldinfo = DBFGetFieldInfo(*hnd, fieldno, NULL, NULL, NULL);
  if (restype)
    agn_createseq(L, nrecords);
  else
    agn_createset(L, nrecords);
  switch (fieldinfo) {
    case FTString: {
      for (i=0; i < nrecords; i++) {
        result = DBFReadStringAttribute(*hnd, i, fieldno);
        if (result == NULL)
          luaL_error(L, "Error in " LUA_QS ": no string stored in data base.", pn);
        else if (restype) {
          lua_seqsetistring(L, -1, i+1, result);
        }
        else {
          lua_sinsertstring(L, -1, result);
        }
      }
      break;
    }
    case FTDouble: {
      for (i=0; i < nrecords; i++) {
        val = DBFReadDoubleAttribute(*hnd, i, fieldno, &failure);
        if (failure)
          luaL_error(L, "Error in " LUA_QS ": could not successfully read number in data base.", pn);
        else if (restype) {
          lua_seqsetinumber(L, -1, i+1, val);
        }
        else {
          lua_sinsertnumber(L, -1, val);
        }
      }
      break;
    }
    case FTLogical: {
      for (i=0; i < nrecords; i++) {
        result = DBFReadLogicalAttribute(*hnd, i, fieldno);
        if (result == NULL)
          luaL_error(L, "Error in " LUA_QS ": no logical value stored in data base.", pn);
        else if (restype) {
          lua_seqsetistring(L, -1, i+1, result);
        }
        else {
          lua_sinsertstring(L, -1, result);
        }
      }
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": unknown type of data.", pn);
  }
  return 1;
}


static int xbase_record (lua_State *L) {
  DBFHandle *hnd;
  int nrecords, nfields, recordno, *fieldinfo, i, failure;
  const char *pn, *result;
  pn = "xbase.record";
  hnd = aux_gethandle(L, 1, pn);
  recordno = agnL_checkinteger(L, 2);
  nrecords = DBFGetRecordCount(*hnd);
  nfields = DBFGetFieldCount(*hnd);
  fieldinfo = malloc(nfields*sizeof(int));
  if (fieldinfo == NULL)
    luaL_error(L, "Error in " LUA_QS ": memory allocation error.", pn);
  if (recordno < 1 || recordno > nrecords)
    luaL_error(L, "Error in " LUA_QS ": requested field out of range.", pn);
  if (nrecords == 0 || nfields == 0)
    luaL_error(L, "Error in " LUA_QS ": data base is empty.", pn);
  recordno--;
  agn_createseq(L, 1);
  for (i=0; i < nfields; i++) {
    fieldinfo[i] = DBFGetFieldInfo(*hnd, i, NULL, NULL, NULL);
  }
  for (i=0; i < nfields; i++) {
    switch (fieldinfo[i]) {
      case FTString: {
        result = DBFReadStringAttribute(*hnd, recordno, i);
        if (result == NULL)
          luaL_error(L, "Error in " LUA_QS ": no string stored in data base.", pn);
        else
          lua_seqsetistring(L, -1, i+1, result);
        break;
      }
      case FTDouble: {
        lua_seqsetinumber(L, -1, i+1, DBFReadDoubleAttribute(*hnd, recordno, i, &failure));
        if (failure)
          luaL_error(L, "Error in " LUA_QS ": could not successfully read number in data base.", pn);
        break;
      }
      case FTLogical: {
        result = DBFReadLogicalAttribute(*hnd, recordno, i);
        if (result == NULL)
          luaL_error(L, "Error in " LUA_QS ": no logical value stored in data base.", pn);
        else
          lua_seqsetistring(L, -1, i+1, result);
        break;
      }
      default:
        luaL_error(L, "Error in " LUA_QS ": unknown type of data.", pn);
    }
  }
  free(fieldinfo);
  return 1;
}


static int xbase_filepos (lua_State *L) {
  off64_t result;
  DBFHandle *ud;
  DBFInfo dbf;
  const char *pn = "xbase.filepos";
  ud = aux_gethandle(L, 1, pn);
  dbf = **ud;
  if (ud == NULL)
    luaL_error(L, "Error in " LUA_QS ": invalid file handle.", pn);
  result = ftello64(dbf.fp);
  if (result == -1)
    luaL_error(L, "Error in " LUA_QS ": IO failure.", pn);
  lua_pushnumber(L, result);
  return 1;
}


static const luaL_Reg xbase[] = {
  {"attrib", xbase_attrib},              /* 05.06.2010 */
  {"close", xbase_close},                /* 05.06.2010 */
  {"field", xbase_field},                /* 13.06.2010 */
  {"filepos", xbase_filepos},            /* 13.06.2010 */
  {"isVoid", xbase_isVoid},              /* 13.06.2010 */
  {"lock", xbase_lock},                  /* 12.06.2010 */
  {"new", xbase_new},                    /* 05.06.2010 */
  {"open", xbase_open},                  /* 05.06.2010 */
  {"purge", xbase_purge},                /* 05.06.2010 */
  {"readdbf", xbase_readdbf},            /* 05.06.2010 */
  {"readvalue", xbase_readvalue},        /* 05.06.2010 */
  {"record", xbase_record},              /* 13.06.2010 */
  {"sync", xbase_sync},                  /* 05.06.2010 */
  {"unlock", xbase_unlock},              /* 12.06.2010 */
  {"writeboolean", xbase_writeboolean},  /* 05.06.2010 */
  /* {"writeinteger", xbase_writeinteger}, */ /* 05.06.2010 */
  {"writenumber", xbase_writenumber},    /* 05.06.2010 */
  {"writestring", xbase_writestring},    /* 05.06.2010 */
  {NULL, NULL}
};


static void createmetatable (lua_State *L) {
  lua_createtable(L, 0, 1);  /* create metatable for strings */
  lua_pushliteral(L, "");  /* dummy string */
  lua_pushvalue(L, -2);
  lua_setmetatable(L, -2);  /* set string metatable */
  agn_poptop(L);  /* pop dummy string */
  lua_pushvalue(L, -2);  /* string library... */
  lua_setfield(L, -2, "__index");  /* ...is the __index metamethod */
  agn_poptop(L);  /* pop metatable */
}


static int xbase_tostring (lua_State *L) {
  DBFHandle *hnd;
  hnd = aux_gethandle(L, 1, "__tostring");
  lua_pushfstring(L, "xbase(%p)", lua_topointer(L, 1));
  return 1;
}

static int xbase_gc (lua_State *L) {
  (void)L;
  return 0;
}

static const struct luaL_Reg xbase_lib [] = {
  {"__tostring", xbase_tostring},
  {"__gc", xbase_gc},
  {NULL, NULL}
};


/*
** Open xbase library
*/
LUALIB_API int luaopen_xbase (lua_State *L) {
  luaL_newmetatable(L, "xbase");
  luaL_register(L, NULL, xbase_lib);
  luaL_register(L, AGENA_XBASELIBNAME, xbase);
  lua_newtable(L);
  lua_setfield(L, -2, "openfiles");  /* table for information on all open files */
  createmetatable(L);
  return 1;
}


