/*
** $Id: utils.c, v 1.0.1 by Alexander Walz - Initiated: October 17, 2006
   Revision 08.08.2009 22:13:15
** See Copyright Notice in agena.h
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* instead of time.h: */
#include "agnt64.h"

/* the following package ini declarations must be included after `#include <` and before `include #` ! */

#define utils_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"

#include "utils.h"  /* typedefs and checkarray macro */

#define uchar(c)        ((unsigned char)(c))

static int utils_singlesubs (lua_State *L) {  /* Agena 1.5.0 */
   size_t i, l;
   unsigned char token;
   luaL_Buffer b;
   const char *s = luaL_checklstring(L, 1, &l);
   luaL_buffinit(L, &b);
   if (lua_type(L, 2) == LUA_TSEQ) {
      for (i=0; i < l; i++) {
         lua_seqgeti(L, 2, uchar(s[i])+1);  /* push substitution on stack */
         token = uchar(luaL_checknumber(L, -1));
         luaL_addchar(&b, token);
         agn_poptop(L);
      }
   } else
      luaL_error(L, "Error in " LUA_QS ": sequence expected, got %s.", "utils.singlesubs",
                 lua_typename(L, lua_type(L, 2)));
   luaL_pushresult(&b);
   return 1;
}

/*
----------------------------------------------------------------------------
  Internal utilities
----------------------------------------------------------------------------
*/

static void setfield (lua_State *L, const char *key, int value) {
  lua_pushinteger(L, value);
  lua_setfield(L, -2, key);
}


static void setboolfield (lua_State *L, const char *key, int value) {
  if (value < 0)  /* undefined? */
    return;  /* does not set field */
  lua_pushboolean(L, value);
  lua_setfield(L, -2, key);
}

/*
----------------------------------------------------------------------------
  Other functions
----------------------------------------------------------------------------
*/

static int utils_calendar (lua_State *L) {
  Time64_T seconds = luaL_opt(L, (time_t)agn_checknumber, 1, time(NULL));
  struct TM *stm;
  lua_createtable(L, 0, 9);  /* 9 = number of fields */
  if (seconds < 0) {  /* 0.31.3 patch */
    lua_pushfail(L);
    return 1;
  }
  stm = localtime64(&seconds);
  if (stm != NULL) {
    setfield(L, "sec", stm->tm_sec);
    setfield(L, "min", stm->tm_min);
    setfield(L, "hour", stm->tm_hour);
    setfield(L, "day", stm->tm_mday);
    setfield(L, "month", stm->tm_mon+1);
    setfield(L, "year", stm->tm_year+1900);
    setfield(L, "wday", stm->tm_wday+1);
    setfield(L, "yday", stm->tm_yday+1);
    setboolfield(L, "DST", stm->tm_isdst);
  } else
    lua_pushfail(L);
  return 1;
}


/* ******************************************************************
LICENCE:        Copyright (c) 2001 Bob Trower, Trantor Standard Systems Inc.

                Permission is hereby granted, free of charge, to any person
                obtaining a copy of this software and associated
                documentation files (the "Software"), to deal in the
                Software without restriction, including without limitation
                the rights to use, copy, modify, merge, publish, distribute,
                sublicense, and/or sell copies of the Software, and to
                permit persons to whom the Software is furnished to do so,
                subject to the following conditions:

                The above copyright notice and this permission notice shall
                be included in all copies or substantial portions of the
                Software.

                THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
                KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
                WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
                PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
                OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
                OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
                OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
                SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

VERSION HISTORY:
                Bob Trower 08/04/01 -- Create Version 0.00.00B

******************************************************************** */

/*
** Translation Table as described in RFC1113
*/
static const char cb64[]="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

/*
** Translation Table to decode (created by author)
*/
static const char cd64[]="|$$$}rstuvwxyz{$$$$$$$>?@ABCDEFGHIJKLMNOPQRSTUVW$$$$$$XYZ[\\]^_`abcdefghijklmnopq";

/*
** encodeblock
**
** encode 3 8-bit binary bytes as 4 '6-bit' characters
*/
void encodeblock( unsigned char in[3], unsigned char out[4], int len ) {
  out[0] = cb64[ in[0] >> 2 ];
  out[1] = cb64[ ((in[0] & 0x03) << 4) | ((in[1] & 0xf0) >> 4) ];
  out[2] = (unsigned char) (len > 1 ? cb64[ ((in[1] & 0x0f) << 2) | ((in[2] & 0xc0) >> 6) ] : '=');
  out[3] = (unsigned char) (len > 2 ? cb64[ in[2] & 0x3f ] : '=');
}


/*
** encode
**
** base64 encode a stream adding padding and line breaks as per spec.
*/

static int utils_encodeb64 (lua_State *L) {
  luaL_Buffer b;
  unsigned char in[3], out[4];
  size_t linesize;
  int i, len, blocksout = 0;
  const char *str = agn_checkstring(L, 1);
  linesize = agnL_optnumber(L, 2, 72);
  luaL_buffinit(L, &b);
  while (*str) {
    len = 0;
    for (i = 0; i < 3; i++) {
      in[i] = (unsigned char) *str;
      if (*str)
        len++;
      else {
        in[i] = 0;
        break;
      }
      str++;
    }
    if (len) {
      encodeblock(in, out, len);
      for (i = 0; i < 4; i++) {
        luaL_addchar(&b, out[i]);
      }
      blocksout++;
    }
    if (blocksout >= (linesize/4) || *str == '\0') {
      if (blocksout) {
        /* luaL_addchar(&b, '\r'); */  /* Linux base64 only works in ignore mode when encountering CRs */
        luaL_addchar(&b, '\n');
      }
      blocksout = 0;
    }
  }
  luaL_pushresult(&b);
  return 1;
}

/*
** decodeblock
**
** decode 4 '6-bit' characters into 3 8-bit binary bytes
*/
void decodeblock( unsigned char in[4], unsigned char out[3] )
{
    out[ 0 ] = (unsigned char ) (in[0] << 2 | in[1] >> 4);
    out[ 1 ] = (unsigned char ) (in[1] << 4 | in[2] >> 2);
    out[ 2 ] = (unsigned char ) (((in[2] << 6) & 0xc0) | in[3]);
}

/*
** decode
**
** decode a base64 encoded stream discarding padding, line breaks and noise
*/
static int utils_decodeb64 (lua_State *L) {
  luaL_Buffer b;
  unsigned char in[4], out[3], v;
  int i, len;
  const char *str = agn_checkstring(L, 1);
  luaL_buffinit(L, &b);
  while (*str) {
    for (len=0, i = 0; i < 4 && *str; i++) {
      v = 0;
      while (*str && v == 0) {
        v = (unsigned char) *str++;
        v = (unsigned char) ((v < 43 || v > 122) ? 0 : cd64[ v - 43 ]);
        if (v) {
          v = (unsigned char) ((v == '$') ? 0 : v - 61);
        }
      }
      if (*str) {
        len++;
        if (v) {
          in[i] = (unsigned char)(v - 1);
        }
      }
      else {
        in[i] = 0;
        break;
      }
    }
    if (len) {
      decodeblock(in, out);
      for (i = 0; i < len - 1; i++) {
        luaL_addchar(&b, out[i]);
      }
    }
  }
  luaL_pushresult(&b);
  return 1;
}


static int utils_checkdate (lua_State *L) {
  int year, month, day, hour, minute, second;
  size_t nops;
  year = month = day = hour = minute = second = 0;  /* to prevent compiler warnings */
  switch (lua_type(L, 1)) {
    case LUA_TTABLE: {  /* 1.12.3 */
      nops = agn_size(L, 1);
      if (nops < 3)
        luaL_error(L, "Error in " LUA_QS ": table must contain at least three values.", "utils.checkdate");
      year = agn_getinumber(L, 1, 1);
      month = agn_getinumber(L, 1, 2);
      day = agn_getinumber(L, 1, 3);
      hour = (nops > 3) ? agn_getinumber(L, 1, 4) : 0;
      minute = (nops > 4) ? agn_getinumber(L, 1, 5) : 0;
      second = (nops > 5) ? agn_getinumber(L, 1, 6) : 0;
      break;
    }
    case LUA_TSEQ: {  /* 1.12.3 */
      nops = agn_seqsize(L, 1);
      if (nops < 3)
        luaL_error(L, "Error in " LUA_QS ": sequence must contain at least three values.", "utils.checkdate");
      year = lua_seqgetinumber(L, 1, 1);
      month = lua_seqgetinumber(L, 1, 2);
      day = lua_seqgetinumber(L, 1, 3);
      hour = (nops > 3) ? lua_seqgetinumber(L, 1, 4) : 0;
      minute = (nops > 4) ? lua_seqgetinumber(L, 1, 5) : 0;
      second = (nops > 5) ? lua_seqgetinumber(L, 1, 6) : 0;
      break;    
    }
    case LUA_TNUMBER: {
      nops = lua_gettop(L);
      if (nops < 3)
        luaL_error(L, "Error in " LUA_QS ": expected at leat three arguments of type number.", "utils.checkdate");
      year = agn_checknumber(L, 1);
      month = agn_checknumber(L, 2);
      day = agn_checknumber(L, 3);
      hour = (nops > 3) ? agn_checknumber(L, 4) : 0;
      minute = (nops > 4) ? agn_checknumber(L, 5) : 0;
      second = (nops > 5) ? agn_checknumber(L, 6) : 0;
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": expected a table, sequence or at least three numbers.", "utils.checkdate");
  }
  agn_pushboolean(L, tools_checkdatetime(year, month, day, hour, minute, second));
  return 1;
}


static const luaL_Reg utils[] = {
  {"calendar", utils_calendar},           /* added on April 07, 2007 */
  {"checkdate", utils_checkdate},         /* added on Januray 31, 2013 */
  {"singlesubs", utils_singlesubs},       /* added on September 07, 2011 - added on August 05, 2007 */
  {"decodeb64", utils_decodeb64},         /* added on May 27, 2012 */
  {"encodeb64", utils_encodeb64},         /* added on May 27, 2012 */
  {NULL, NULL}
};


static void createmetatable (lua_State *L) {
  lua_createtable(L, 0, 1);  /* create metatable for strings */
  lua_pushliteral(L, "");  /* dummy string */
  lua_pushvalue(L, -2);
  lua_setmetatable(L, -2);  /* set string metatable */
  agn_poptop(L);  /* pop dummy string */
  lua_pushvalue(L, -2);  /* string library... */
  lua_setfield(L, -2, "__index");  /* ...is the __index metamethod */
  agn_poptop(L);  /* pop metatable */
}


/*
** Open utils library
*/
LUALIB_API int luaopen_utils (lua_State *L) {
  luaL_register(L, AGENA_UTILSLIBNAME, utils);
  createmetatable(L);
  return 1;
}

/* ====================================================================== */

