/*
** $Id: lstatlib.c, v 1.0.1 by Alexander Walz - initiated June 19, 2007
** Statistics library
** See Copyright Notice in agena.h
*/


#include <stdlib.h>
#include <math.h>
#include <string.h>

/* the following package ini declarations must be included after `#include <` and before `include #` ! */

#define stats_c
#define LUA_LIB

#include "agena.h"
#include "agnxlib.h"
#include "agenalib.h"
#include "agncmpt.h"  /* for TRUNC */
#include "agnhlps.h"  /* for tools_intpow, isnan, PI2, quicksort, k-th smallest */

#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_STATSLIBNAME "stats"
LUALIB_API int (luaopen_stats) (lua_State *L);
#endif


/* creates an array a of type lua_Number with size n, FREE it ! */
#define createarray(a, n, procname) { \
  (a) = malloc((n)*sizeof(lua_Number)); \
  if ((a) == NULL) \
    luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", (procname)); \
}

/* median of all values in an _already sorted table array or sequence_, June 18, 2007; tweaked Jan 13, 2008;
   faster than an Agena implementation; patched 0.25.4, August 01, 2009; patched 0.26.1, 11.08.2009;
   extended 1.6.0, April 31, 2012, extended 1.8.2, October 07, 2012; rewritten December 30, 2012 */

static int stats_median (lua_State *L) {
  int type;
  size_t n, i, ii;
  lua_Number *a, x2;
  luaL_checkany(L, 1);
  ii = 0;
  a = NULL;  /* to prevent compiler warnings */
  type = lua_type(L, 1);
  switch (type) {
    case LUA_TTABLE: {
      n = agn_size(L, 1);
      if (n == 0) {
        lua_pushfail(L);
        return 1;
      }
      createarray(a, n, "stats.median");
      for (i=1; i <= n; i++) {
        x2 = agn_getinumber(L, 1, i);
        if (!tools_isnan(x2)) a[++ii-1] = x2;
      }
      break;
    }
    case LUA_TSEQ: {
      n = agn_seqsize(L, 1);
      if (n == 0) {
        lua_pushfail(L);
        return 1;
      }
      createarray(a, n, "stats.median");
      for (i=1; i <= n; i++) {
        x2 = lua_seqgetinumber(L, 1, i);
        if (!tools_isnan(x2)) a[++ii-1] = x2;
      }
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": table or sequence expected, got " LUA_QS ".", "stats.median",
        lua_typename(L, lua_type(L, 1)));
  }
  if (ii == 0)
    lua_pushfail(L);
  else
    lua_pushnumber(L, ii&1 ? tools_kth_smallest(a, ii, ii/2) :
      (tools_kth_smallest(a, ii, ii/2-1) + tools_kth_smallest(a, ii, ii/2))/2
    );
  xfree(a);
  return 1;
}


static int stats_mad (lua_State *L) {
  int type;
  size_t n, i, ii;
  lua_Number *a, *b, x2, median;
  luaL_checkany(L, 1);
  ii = 0;
  a = b = NULL;  /* to prevent compiler warnings */
  type = lua_type(L, 1);
  switch (type) {
    case LUA_TTABLE: {
      n = agn_size(L, 1);
      if (n == 0) {
        lua_pushfail(L);
        return 1;
      }
      createarray(a, n, "stats.mad");
      for (i=1; i <= n; i++) {
        x2 = agn_getinumber(L, 1, i);
        if (!tools_isnan(x2)) a[++ii-1] = x2;
      }
      break;
    }
    case LUA_TSEQ: {
      n = agn_seqsize(L, 1);
      if (n == 0) {
        lua_pushfail(L);
        return 1;
      }
      createarray(a, n, "stats.mad");
      for (i=1; i <= n; i++) {
        x2 = lua_seqgetinumber(L, 1, i);
        if (!tools_isnan(x2)) a[++ii-1] = x2;
      }
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": table or sequence expected, got " LUA_QS ".", "stats.mad",
        lua_typename(L, lua_type(L, 1)));
  }
  if (ii == 0)
    lua_pushfail(L);
  else {
    createarray(b, n, "stats.mad");
    median = ii&1 ? tools_kth_smallest(a, ii, ii/2) :
      (tools_kth_smallest(a, ii, ii/2-1) + tools_kth_smallest(a, ii, ii/2))/2;
    for (i=0; i < ii; i++)
      b[i] = fabs(a[i] - median);
    lua_pushnumber(L, ii&1 ? tools_kth_smallest(b, ii, ii/2) :
      (tools_kth_smallest(b, ii, ii/2-1) + tools_kth_smallest(b, ii, ii/2))/2
    );
    xfree(b);
  }
  xfree(a);
  return 1;
}


/* sorts numbers in ascending order */

static int stats_sorted (lua_State *L) {
  int type, newmethod;
  size_t n, i, ii;
  lua_Number *a, x2;
  luaL_checkany(L, 1);
  newmethod = agnL_optboolean(L, 2, 0);  /* use recursive Quicksort algorithm by default */
  ii = 0;
  a = NULL;  /* to prevent compiler warnings */
  type = lua_type(L, 1);
  switch (type) {
    case LUA_TTABLE: {
      n = agn_size(L, 1);
      if (n == 0) {
        lua_pushfail(L);
        return 1;
      }
      createarray(a, n, "stats.sorted");
      for (i=1; i <= n; i++) {
        x2 = agn_getinumber(L, 1, i);
        if (!tools_isnan(x2)) a[++ii-1] = x2;
      }
      if (ii == 0)
        lua_pushfail(L);
      else {
        if (newmethod) {  /* 1.11.6, use pixel sort function: this _can_ be 15 % faster on older systems than the
          recursive quicksort implementation for lists of numbers in random order. Only if the list is already
          sorted in descending order, pixel sort is much slower. */
          if (pixel_qsort(a, ii) == 0) {
            xfree(a);  /* 1.12.4 */
            luaL_error(L, "Error in " LUA_QS ": internal memory allocation error (stack too large).", "stats.sorted");
          }
        } else  /* use recursive quicksort implementation */
          tools_dquicksort(a, 0, ii-1);
        lua_createtable(L, ii, 0);
        for (i=0; i < ii; i++) {
          lua_rawsetinumber(L, -1, i+1, a[i]);
        }
      }
      break;
    }
    case LUA_TSEQ: {
      n = agn_seqsize(L, 1);
      if (n == 0) {
        lua_pushfail(L);
        return 1;
      }
      createarray(a, n, "stats.sorted");
      for (i=1; i <= n; i++) {
        x2 = lua_seqgetinumber(L, 1, i);
        if (!tools_isnan(x2)) a[++ii-1] = x2;
      }
      if (ii == 0)
        lua_pushfail(L);
      else {
        if (newmethod) {
          if (pixel_qsort(a, ii) == 0) {
            xfree(a);
            luaL_error(L, "Error in " LUA_QS ": internal memory allocation failed, stack too large.", "stats.sorted");
          }
        } else  /* use recursive quicksort implementation */
          tools_dquicksort(a, 0, ii-1);
        agn_createseq(L, ii);
        for (i=0; i < ii; i++) {
          lua_seqsetinumber(L, -1, i+1, a[i]);
        }
      }
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": table or sequence expected, got " LUA_QS ".", "stats.sorted",
        lua_typename(L, lua_type(L, 1)));
  }
  xfree(a);
  return 1;
}


/*
   stats.smallest

   Returns the k-th smallest element in the numeric table or sequence a. If k is not given, it is set to 1.
*/

static int stats_smallest (lua_State *L) {
  int type;
  size_t n, i, ii;
  lua_Number *a, x2, k;
  luaL_checkany(L, 1);
  ii = 0;
  a = NULL;  /* to prevent compiler warnings */
  type = lua_type(L, 1);
  k = luaL_optnumber(L, 2, 1);
  switch (type) {
    case LUA_TTABLE: {
      n = agn_size(L, 1);
      if (k < 1 || k > n)
        luaL_error(L, "Error in " LUA_QS ": second argument too small or too large.", "stats.smallest");
      if (n == 0) {
        lua_pushfail(L);
        return 1;
      }
      createarray(a, n, "stats.smallest");
      for (i=1; i <= n; i++) {
        x2 = agn_getinumber(L, 1, i);
        if (!tools_isnan(x2)) a[++ii-1] = x2;
      }
      if (ii == 0)
        lua_pushfail(L);
      else
        lua_pushnumber(L, tools_kth_smallest (a, ii, k-1));
      break;
    }
    case LUA_TSEQ: {
      n = agn_seqsize(L, 1);
      if (k < 1 || k > n)
        luaL_error(L, "Error in " LUA_QS ": second argument too small or too large.", "stats.smallest");
      if (n == 0) {
        lua_pushfail(L);
        return 1;
      }
      createarray(a, n, "stats.smallest");
      for (i=1; i <= n; i++) {
        x2 = lua_seqgetinumber(L, 1, i);
        if (!tools_isnan(x2)) a[++ii-1] = x2;
      }
      if (ii == 0)
        lua_pushfail(L);
      else
        lua_pushnumber(L, tools_kth_smallest (a, ii, k-1));
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": table or sequence expected, got " LUA_QS ".", "stats.smallest",
        lua_typename(L, lua_type(L, 1)));
  }
  xfree(a);
  return 1;
}


/* tbl_minmax: returns the smallest and largest value in a list. If the option
   'sorted' is given, the list is not traversed, instead the first and the last entry
   in the list is returned.
   If the list is empty or has only one element, fail is returned. June 20, 2007;
   modified August 01, 2009, 0.25.4; modified 0.26.1, 11.08.2009, extended 1.3.3, 30.01.2011;
   patched 1.6.0, April 01, 2012 */

static int stats_minmax (lua_State *L) {
  int i, n, type;
  lua_Number min, max, number;
  luaL_checkany(L, 1);
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  const char *option = luaL_optstring(L, 2, "default");
  n = agn_nops(L, 1);
  if (n < 2) {
    lua_pushfail(L);
    return 1;
  }
  switch (type) {
    case LUA_TTABLE: {
      lua_newtable(L);
      if (strcmp(option, "sorted") == 0) {
        lua_pushinteger(L, 1);
        lua_rawgeti(L, 1, 1);
        lua_rawset(L, -3);
        lua_pushinteger(L, 2);
        lua_rawgeti(L, 1, n);
        lua_rawset(L, -3);
        return 1;
      }
      lua_pushnil(L);
      max = -HUGE_VAL; min = HUGE_VAL;  /* Agena 1.6.0 patch */
      while (lua_next(L, 1) != 0) {
        if (agn_isnumber(L, -1)) {
          number = agn_tonumber(L, -1);
          if (number < min)
            min = number;
          if (number > max)
            max = number;
        }
        agn_poptop(L);  /* pop value */
      }
      lua_rawsetinumber(L, -1, 1, min);
      lua_rawsetinumber(L, -1, 2, max);
      break;
    }
    case LUA_TSEQ: {
      agn_createseq(L, 2);
      if (strcmp(option, "sorted") == 0) {
        lua_seqgeti(L, 1, 1);
        lua_seqseti(L, -2, 1);
        lua_seqgeti(L, 1, n);
        lua_seqseti(L, -2, 2);
        return 1;
      }
      min = lua_seqgetinumber(L, 1, 1);
      max = min;
      for (i=2; i<=n; i++) {
        number = lua_seqgetinumber(L, 1, i);
        if (number < min)
          min = number;
        if (number > max)
          max = number;
      }
      lua_seqsetinumber(L, -1, 1, min);
      lua_seqsetinumber(L, -1, 2, max);
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  return 1;
}


/* IOS: Index On Stability
   operates on sorted and unsorted list but with different results ! Sum up absolute differences
   between neighbouring points. This indicator is quite useful to find out how stable or volatile
   an observation is.

   This C implementation is at least 3 to five times faster than the almost equivalent versions:

   stats.ios := proc(x::table) is
      local n, result := size(x), 0;    # n: size of table, result: intermediate results
      for i from 2 to n do
         inc result, abs(x[i]-x[i-1])   # add absolute distances of neighboring values
      od;
      return result/(n-1)               # divide by number of entries - 1 and return result
   end;

   or for short:

   stats.ios := << x::table -> sadd(stats.deltalist(x, true))/(size x-1) >>
*/

static int stats_ios (lua_State *L) {
  size_t i, n;
  int type;
  lua_Number result, x1, x2;
  luaL_checkany(L, 1);
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  n = agn_nops(L, 1);
  if (n < 2) {
    lua_pushfail(L);
    return 1;
  }
  result = 0;
  switch (type) {
    case LUA_TTABLE: {
      x1 = agn_getinumber(L, 1, 1);
      for (i=2; i<=n; i++) {
        x2 = agn_getinumber(L, 1, i);
        result += fabs(x2-x1);
        x1 = x2;
      }
      lua_pushnumber(L, result/((lua_Number)n-1));
      break;
    }
    case LUA_TSEQ: {
      x1 = lua_seqgetinumber(L, 1, 1);
      for (i=2; i<=n; i++) {
        x2 = lua_seqgetinumber(L, 1, i);
        result += fabs(x2-x1);
        x1 = x2;
      }
      lua_pushnumber(L, result/((lua_Number)n-1));
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  return 1;
}


/* Divides each element in an observation by its size and sums up the quotients to finally return the arithmetic mean.
   By dividing each element before summation, the function avoids arithmetic overflows and also uses a modified
   Kahan algorithm developed by Kazufumi Ozawa published in his paper `Analysis and Improvement of Kahan's Summation
   Algorithm` to prevent round-off errors during summation. Thus the function is more robust but also significantly
   slower than `stats.mean`.

   The function also accepts structures including the value `undefined`. In this case, all `undefined's` are ignored, so
   that the function can be used with incomplete observations. This feature has been inspired by TI-Nspire(tm) CX CAS.

   Initiated: Agena 1.7.7, August 26, 2012; extended Agena 1.8.2, 05.10.2012 */

static lua_Number kahanmean (lua_Number *a, size_t ii) {
  size_t i;
  volatile lua_Number q, s, sold, u, v, w, x, t;
  s = q = 0;  /* 1.12.6 fix */
  for (i=0; i < ii; i++) {
    /* Kahan summation to 1) prevent overflows with too large sums and 2) round-off errors */
    x = a[i]/ii;
    v = x - q;
    sold = s;
    s = s + v;  /* s contains the mean of the observation */
    if (fabs(x) < fabs(q)) {
      t = x;
      x = -q;
      q = t;
    }
    u = (v - x) + q;
    if (fabs(sold) < fabs(v)) {
      t = sold;
      sold = v;
      v = t;
    }
    w = (s - sold) - v;
    q = u + w;
    /* s contains the mean */
  }
  return s;
}


static int stats_amean (lua_State *L) {
  size_t i, ii, n;
  int type;
  lua_Number x2, s, *a;
  luaL_checkany(L, 1);
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  n = agn_nops(L, 1);
  if (n < 1) {
    lua_pushfail(L);
    return 1;
  }
  createarray(a, n, "stats.amean");
  ii = 0;
  switch (type) {
    case LUA_TTABLE: {
      for (i=1; i <= n; i++) {
        x2 = agn_getinumber(L, 1, i);
        if (!tools_isnan(x2)) a[++ii-1] = x2;
      }
      break;
    }
    case LUA_TSEQ: {
      for (i=1; i <= n; i++) {
        x2 = lua_seqgetinumber(L, 1, i);
        if (!tools_isnan(x2)) a[++ii-1] = x2;
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  s = kahanmean(a, ii);
  xfree(a);
  lua_pushnumber(L, s);
  return 1;
}


/* sum, Agena 1.8.2, 05.10.2012; extended 1.8.3, 08.10.2012

   Computes the sums of all numbers in a table or sequence. Contrary to the `sadd` operator, it prevents round-off
   errors during summation. Inspired by the TI-Nspire(tm) CX CAS.

   The function also accepts structures including the value `undefined`. In this case, all `undefined's` are simply
   ignored and skipped in the computation of the sums. Thus the function can be used with incomplete observations.
*/

static int stats_sum (lua_State *L) {
  size_t i, ii, j, n, nargs;
  int objtype, objpos;
  lua_Number x2, *a;
  volatile lua_Number q, s, sold, u, v, w, x, t;
  luaL_checkany(L, 1);
  nargs = lua_gettop(L);
  if (nargs < 1)
    luaL_error(L, "Error in " LUA_QS ": expected one or more arguments, got none.", "stats.sum");
  objpos = (nargs == 1) ? 1 : 2;
  objtype = lua_type(L, objpos);
  luaL_typecheck(L, objtype == LUA_TTABLE || objtype == LUA_TSEQ, objpos, "table or sequence expected", objtype);
  if (nargs != 1) {
    int firstarg;
    firstarg = lua_type(L, 1);
    luaL_typecheck(L, firstarg == LUA_TFUNCTION, 1, "procedure expected", firstarg);
  }
  n = agn_nops(L, objpos);  /* number of elements in structure */
  if (n < 1) {
    lua_pushfail(L);
    return 1;
  }
  createarray(a, n, "stats.sum");
  q = ii = 0;
  s = AGN_NAN;
  switch (objtype) {
    case LUA_TTABLE: {
      if (nargs == 1) {
        for (i=1; i <= n; i++) {
          x2 = agn_getinumber(L, 1, i);
          if (!tools_isnan(x2)) a[++ii-1] = x2;
        }
      } else {  /* procedure passed */
        for (i=1; i <= n; i++) {
          lua_pushvalue(L, 1);  /* push function */
          x2 = agn_getinumber(L, 2, i);
          lua_pushnumber(L, x2);
          for (j=3; j <= nargs; j++) {
            lua_pushvalue(L, j);
          }
          lua_call(L, nargs - 1, 1);  /* call function with (nargs -1 ) arguments and one result, 1.8.3 */
          if (agn_istrue(L, -1)) { if (!tools_isnan(x2)) a[++ii-1] = x2; }
          agn_poptop(L);
        }
      }
      break;
    }
    case LUA_TSEQ: {
      if (nargs == 1) {
        for (i=1; i <= n; i++) {
          x2 = lua_seqgetinumber(L, 1, i);
          if (!tools_isnan(x2)) a[++ii-1] = x2;
        }
      } else {
        for (i=1; i <= n; i++) {
          lua_pushvalue(L, 1);  /* push function */
          x2 = lua_seqgetinumber(L, 2, i);
          lua_pushnumber(L, x2);
          for (j=3; j <= nargs; j++) {
            lua_pushvalue(L, j);
          }
          lua_call(L, nargs - 1, 1);  /* call function with (nargs -1 ) arguments and one result, 1.8.3 */
          if (agn_istrue(L, -1)) { if (!tools_isnan(x2)) a[++ii-1] = x2; }
          agn_poptop(L);
        }
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  if (ii != 0) {
    s = 0;
    for (i=0; i < ii; i++) {
      /* Kahan summation to prevent round-off errors */
      x = a[i];
      v = x - q;
      sold = s;
      s = s + v;
      if (fabs(x) < fabs(q)) {
        t = x;
        x = -q;
        q = t;
      }
      u = (v - x) + q;
      if (fabs(sold) < fabs(v)) {
        t = sold;
        sold = v;
        v = t;
      }
      w = (s - sold) - v;
      q = u + w;
      /* s contains the sum of the observation, ignoring undefined's */
    }
  }
  xfree(a);
  lua_pushnumber(L, s);
  return 1;
}


/* Cumulative sum, Agena 1.7.9a, 11.09.2012, extended Agena 1.8.2, 05.10.2012

   Uses a modified Kahan algorithm developed by Kazufumi Ozawa published in his paper `Analysis and Improvement of
   Kahan's Summation Algorithm` to prevent round-off errors during summation. Inspired by the cumulativeSum function
   available on the TI-Nspire(tm) CX CAS.

   The function also accepts structures including the value `undefined`. In this case, all `undefined's` are simply
   included in the resulting structure and are ignored in the computation of the sums. Thus the function can be used
   with incomplete observations.
*/

static int stats_cumsum (lua_State *L) {
  size_t i, n;
  int type;
  volatile lua_Number q, s, sold, u, v, w, x, t;
  luaL_checkany(L, 1);
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  n = agn_nops(L, 1);
  if (n < 1) {
    lua_pushfail(L);
    return 1;
  }
  s = q = 0;
  switch (type) {
    case LUA_TTABLE: {
      lua_createtable(L, n, 0);
      for (i=1; i <= n; i++) {
        x = agn_getinumber(L, 1, i);
        if (tools_isnan(x)) {
          lua_rawsetinumber(L, -1, i, x);
          continue;
        }
        /* Kahan summation to 1) prevent overflows with too large sums and 2) round-off errors */
        v = x - q;
        sold = s;
        s = s + v;
        if (fabs(x) < fabs(q)) {
          t = x;
          x = -q;
          q = t;
        }
        u = (v - x) + q;
        if (fabs(sold) < fabs(v)) {
          t = sold;
          sold = v;
          v = t;
        }
        w = (s - sold) - v;
        q = u + w;
        lua_rawsetinumber(L, -1, i, s);
        /* s contains the cumulative sum */
      }
      break;
    }
    case LUA_TSEQ: {
      agn_createseq(L, n);
      for (i=1; i <= n; i++) {
        x = lua_seqgetinumber(L, 1, i);
        if (tools_isnan(x)) {
          lua_seqsetinumber(L, -1, i, x);
          continue;
        }
        /* Kahan summation to 1) prevent overflows with too large sums and 2) round-off errors */
        v = x - q;
        sold = s;
        s = s + v;  /* s contains the mean of the observation */
        if (fabs(x) < fabs(q)) {
          t = x;
          x = -q;
          q = t;
        }
        u = (v - x) + q;
        if (fabs(sold) < fabs(v)) {
          t = sold;
          sold = v;
          v = t;
        }
        w = (s - sold) - v;
        q = u + w;
        lua_seqsetinumber(L, -1, i, s);
        /* s contains the cumulative sum */
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  return 1;
}


/* checks whether all numbers in a table or sequence obj are stored in ascending order. If a value in obj is
   not a number, it is ignored. If obj is a table, you have to make sure that it does not contain holes.
   If it contains holes, apply tables.entries on obj. See also: sort. March 31, 2012, extended November 17, 2012 */

static int stats_issorted (lua_State *L) {
  size_t i, j, n, nargs;
  int type, isf;
  lua_Number x1, x2;
  luaL_checkany(L, 1);
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  if ((isf = !lua_isnoneornil(L, 2)))  /* is there a 2nd argument?  Agena 1.8.9 */
    luaL_checktype(L, 2, LUA_TFUNCTION);
  nargs = lua_gettop(L);
  n = agn_nops(L, 1);
  if (n < 2) {
    lua_pushfail(L);
    return 1;
  }
  switch (type) {
    case LUA_TTABLE: {
      x1 = agn_getinumber(L, 1, 1);
      if (isf) {  /* function passed as second argument ?, Agena 1.8.9 */
        for (i=2; i <= n; i++) {
          lua_pushvalue(L, 2);  /* push function */
          lua_pushnumber(L, x1);
          lua_rawgeti(L, 1, i);
          x2 = lua_tonumber(L, -1);
          for (j=3; j <= nargs; j++) lua_pushvalue(L, j);
          lua_call(L, nargs, 1);
          if (lua_isboolean(L, -1) && agn_isfalse(L, -1))
            return 1;
          agn_poptop(L);  /* pop result of lua_call */
          x1 = x2;
        }
        lua_pushtrue(L);
      } else {
        for (i=2; i <= n; i++) {
          x2 = agn_getinumber(L, 1, i);
          if (x1 > x2) {
            lua_pushfalse(L);
            return 1;
          }
          x1 = x2;
        }
        lua_pushtrue(L);
      }
      break;
    }
    case LUA_TSEQ: {
      x1 = lua_seqgetinumber(L, 1, 1);
      if (isf) {  /* function passed as second argument ?, Agena 1.8.9 */
        for (i=2; i <= n; i++) {
          lua_pushvalue(L, 2);  /* push function */
          lua_pushnumber(L, x1);
          lua_seqrawgeti(L, 1, i);
          x2 = lua_tonumber(L, -1);
          for (j=3; j <= nargs; j++) lua_pushvalue(L, j);
          lua_call(L, nargs, 1);
          if (lua_isboolean(L, -1) && agn_isfalse(L, -1))
            return 1;
          agn_poptop(L);  /* pop result of lua_call */
          x1 = x2;
        }
        lua_pushtrue(L);
      } else {
        for (i=2; i <= n; i++) {
          x2 = lua_seqgetinumber(L, 1, i);
          if (x1 > x2) {
            lua_pushfalse(L);
            return 1;
          }
          x1 = x2;
        }
        lua_pushtrue(L);
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  return 1;
}


/* `stats.deltalist` returns a structure of the deltas of neighbouring elements. It is an extended version of
   the TI-Nspire(tm) CX CAS version of `deltaList`. If obj is a table, you have to make sure that it does not
   contain holes. If it contains holes, apply tables.entries on obj. 12.03.2012 */

static int stats_deltalist (lua_State *L) {
  size_t i, n;
  int type, applyabs;
  lua_Number x1, x2;
  luaL_checkany(L, 1);
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  n = agn_nops(L, 1);
  applyabs = lua_toboolean(L, 2);  /* compute absolute differences ? */
  if (n < 2) {
    lua_pushfail(L);
    return 1;
  }
  switch (type) {
    case LUA_TTABLE: {
      lua_createtable(L, n-1, 0);  /* Agena 1.7.10 */
      x1 = agn_getinumber(L, 1, 1);
      if (applyabs) {
        for (i=2; i<=n; i++) {
          x2 = agn_getinumber(L, 1, i);
          lua_rawsetinumber(L, -1, i-1, fabs(x2-x1));
          x1 = x2;
        }
      } else {
        for (i=2; i<=n; i++) {
          x2 = agn_getinumber(L, 1, i);
          lua_rawsetinumber(L, -1, i-1, x2-x1);
          x1 = x2;
        }
      }
      break;
    }
    case LUA_TSEQ: {
      agn_createseq(L, n-1);  /* Agena 1.7.10 */
      x1 = lua_seqgetinumber(L, 1, 1);
      if (applyabs) {
        for (i=2; i<=n; i++) {
          x2 = lua_seqgetinumber(L, 1, i);
          lua_seqsetinumber(L, -1, i-1, fabs(x2-x1));
          x1 = x2;
        }
      } else {
        for (i=2; i<=n; i++) {
          x2 = lua_seqgetinumber(L, 1, i);
          lua_seqsetinumber(L, -1, i-1, x2-x1);
          x1 = x2;
        }
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  return 1;
}


/* Absolute deviation of all the values in a structure (the mean of the equally likely absolute deviations
   from the mean); absolute deviation is more robust as it is less sensitive to outliers. Also called `mean
   deviation`. */

static int stats_ad (lua_State *L) {
  int type;
  size_t i, n;
  lua_Number result, m;
  luaL_checkany(L, 1);
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  n = agn_nops(L, 1);
  if (n < 2) {
    lua_pushfail(L);
    return 1;
  }
  m = result = 0;
  switch (type) {
    case LUA_TTABLE: {
      /* compute the mean */
      lua_pushnil(L);
      while (lua_next(L, 1)) {
        m += agn_checknumber(L, -1);
        agn_poptop(L);  /* pop the number */
      }
      m = m/(lua_Number)n;
      /* compute the absolute deviation */
      lua_pushnil(L);
      while (lua_next(L, 1)) {
        result += fabs(agn_checknumber(L, -1) - m);
        agn_poptop(L);  /* pop the number */
      }
      lua_pushnumber(L, result/n);
      break;
    }
    case LUA_TSEQ: {
      /* compute the mean */
      for (i=1; i<=n; i++) {
        m += lua_seqgetinumber(L, 1, i);
      }
      m = m/(lua_Number)n;
      /* compute the absolute deviation */
      for (i=1; i<=n; i++) {
        result += fabs(lua_seqgetinumber(L, 1, i) - m);
      }
      lua_pushnumber(L, result/n);
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  return 1;
}


/* stats.tovals: convert strings in a table or sequence to numbers. If a string cannot be converted,
   it is returned unchanged. Equivalent to
      stats.tovals := << (o) -> map( << (x) -> tonumber(x) >>, o) >>,
   but up to 40 % faster. */

static int tonumber (lua_State *L, int idx) {
  lua_Number n;
  int exception;
  luaL_checkany(L, idx);
  if (agn_isstring(L, idx) || agn_isnumber(L, idx)) {
    n = agn_tonumberx(L, idx, &exception);
    if (exception) { /* conversion failed ? Check whether string contains a complex value */
      int cexception;
#ifndef PROPCMPLX
      agn_Complex c;
      c = agn_tocomplexx(L, idx, &cexception);
#else
      lua_Number c[2];
      agn_tocomplexx(L, idx, &cexception, c);
#endif
      if (cexception == 0)
#ifndef PROPCMPLX
        agn_createcomplex(L, c);
#else
        agn_createcomplex(L, c[0], c[1]);
#endif
      else
        lua_pushvalue(L, idx);
    }
    else
      lua_pushnumber(L, n);
  } else if (lua_type(L, idx) == LUA_TCOMPLEX) {
    lua_pushvalue(L, idx);
  } else {
    lua_pushfail(L);
  }
  return 1;
}

static int stats_tovals (lua_State *L) {
  luaL_checkany(L, 1);
  switch (lua_type(L, 1)) {
    case LUA_TTABLE: {
      /* very rough guess whether table is an array or dictionary */
      size_t n = lua_objlen(L, 1);
      if (n == 0)  /* assume table is a dictionary */
        lua_createtable(L, 0, agn_size(L, 1));
      else
        lua_createtable(L, n, 0);  /* lua_objlen not always returns correct results ! */
      lua_pushnil(L);
      while (lua_next(L, 1)) {  /* push the table key and the table value */
        tonumber(L, -1);
        lua_remove(L, -2);
        lua_rawset2(L, -3);  /* store value in a table, leave key on stack */
      }
      break;
    }  /* end of case LUA_TTABLE */
    case LUA_TSET: {
      agn_createset(L, agn_ssize(L, 1));
      lua_pushnil(L);
      while (lua_usnext(L, 1)) {  /* push item twice */
        tonumber(L, -1);
        lua_remove(L, -2);
        lua_sinsert(L, -3);       /* store result to new set */
      }
      break;
    }  /* end of case LUA_TSET */
    case LUA_TSEQ: {
      size_t i, nops;
      nops = agn_seqsize(L, 1);
      agn_createseq(L, nops);
      for (i=0; i < nops; i++) {
        lua_seqgeti(L, 1, i+1);   /* push item */
        tonumber(L, -1);
        lua_remove(L, -2);
        lua_seqinsert(L, -2);     /* store result to new sequence */
      }
      break;
    }  /* end of case LUA_TSEQ */
    default:
      luaL_error(L, "Error in " LUA_QS ": table, set, or sequence expected, got %s.", "stats.tovals",
        lua_typename(L, lua_type(L, 1)));
  }  /* end of switch */
  return 1;
}


/* internal procedure used by stats.sumdata and sum.moment */

static lua_Number sumdata (lua_State *L, size_t n, int type) {
  lua_Number x, d, p, r;
  int ispint;
  p = luaL_optnumber(L, 2, 1);  /* the moment (power p) */
  x = luaL_optnumber(L, 3, 0);  /* origin, quantity about which the moment is computed (optional value to be subtracted from x[i]) */
  ispint = ISINT(p);            /* p is an integer ? */
  r = 0;
  switch (type) {
    case LUA_TTABLE: {
      lua_pushnil(L);
      if (ispint) {
        while (lua_next(L, 1)) {
          d = agn_checknumber(L, -1) - x;
          r += tools_intpow(d, p);
          agn_poptop(L);  /* pop the number */
        }
      } else {
        while (lua_next(L, 1)) {
          d = agn_checknumber(L, -1) - x;
          r += pow(d, p);
          agn_poptop(L);  /* pop the number */
        }
      }
      break;
    }
    case LUA_TSEQ: {
      size_t i;
      if (ispint) {
        for (i=1; i<=n; i++) {
          d = lua_seqgetinumber(L, 1, i) - x;
          r += tools_intpow(d, p);
        }
      } else {
        for (i=1; i<=n; i++) {
          d = lua_seqgetinumber(L, 1, i) - x;
          r += pow(d, p);
        }
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  return r;
}


/* stats.moment: computes the various moments p of the given data x about any origin xm for a full population.
   It is equivalent to: sum((x[i]-xm)^p, i=1 .. n)/n.

   If only the table or sequence x is given, the moment p defaults to 1, and the origin xm defaults to 0. If
   given, the moment p and the origin xm must be numbers. */

static int stats_moment (lua_State *L) {
  int type;
  lua_Number n, result;
  n = agn_nops(L, 1);
  if (n == 0) {
    lua_pushfail(L);
    return 1;
  }
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  result = sumdata(L, (size_t)n, type);
  lua_pushnumber(L, result/n);
  return 1;
}


/* stats.sumdata: sums up all the powers p of the given table or sequence x of n elements about the origin xm.
   It is equivalent to: sum((x[i]-xm)^p, i=1 .. n). It is related to stats.moment.

   If only the structure x is given, the power p defaults to 1, and xm defaults to 0. If given, the moment p
   and the origin xm must be numbers. */

static int stats_sumdata (lua_State *L) {
  int type;
  lua_Number n;
  n = agn_nops(L, 1);
  if (n == 0) {
    lua_pushfail(L);
    return 1;
  }
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  lua_pushnumber(L, sumdata(L, n, type));
  return 1;
}


/*

Credit: Agena's `stats.scale` is a port of the ALGOL 60 function REASCL, being part of the NUMAL package,
originally published by The Stichting Centrum Wiskunde & Informatica, Amsterdam, The Netherlands.

The Stichting Centrum Wiskunde & Informatica (Stichting CWI) (legal successor of Stichting Mathematisch
Centrum) at Amsterdam has granted permission to Paul McJones to attach the integral NUMAL library manual
to his software preservation project web page.

( URL: http://www.softwarepreservation.org/projects/ALGOL/applications/ )

It may be freely used. It may be copied provided that the name NUMAL and the attribution to the Stichting
CWI are retained.

Original ALGOL 60 credits to REASCL:

AUTHORS  : T.J. DEKKER, W. HOFFMANN.
CONTRIBUTORS: W. HOFFMANN, S.P.N. VAN KAMPEN.
INSTITUTE: MATHEMATICAL CENTRE.
RECEIVED: 731030.

BRIEF DESCRIPTION:

The procedure REASCL (scale) normalises the the numbers in a table or sequence in such a way that
an element of maximum absolute value equals 1. The normalised numbers are returned in a new
table or sequence, where the type of return is defined by the type of the input.

RUNNING TIME: PROPORTIONAL TO N.

LANGUAGE:   ALGOL 60.

METHOD AND PERFORMANCE: SEE REF [1].

REFERENCES:
     [1].T.J. DEKKER AND W. HOFFMANN.
         ALGOL 60 PROCEDURES IN NUMERICAL ALGEBRA, PART 2.
         MC TRACT 23, 1968, MATH. CENTR., AMSTERDAM.

SOURCE TEXT(S):

 CODE 34183;
     COMMENT MCA 2413;
     PROCEDURE REASCL(A, N, N1, N2); VALUE N, N1, N2;
     INTEGER N, N1, N2; ARRAY A;
     BEGIN INTEGER I, J; REAL S;
         FOR J:= N1 STEP 1 UNTIL N2 DO
         BEGIN S:= 0;
             FOR I:= 1 STEP 1 UNTIL N DO
             IF ABS(A[I,J]) > ABS(S) THEN S:= A[I,J];
             IF S ^= 0 THEN
             FOR I:= 1 STEP 1 UNTIL N DO A[I,J]:= A[I,J] / S
         END
     END REASCL;
         EOP

*/

static int stats_scale (lua_State *L) {
  int type;
  size_t n, c;
  lua_Number s, x;
  n = agn_nops(L, 1);
  if (n == 0) {
    lua_pushfail(L);
    return 1;
  }
  s = 0;
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  switch (type) {
    case LUA_TTABLE: {
      c = 0;
      lua_pushnil(L);
      while (lua_next(L, 1)) {
        c++;
        x = agn_checknumber(L, -1);
        if (fabs(x) > fabs(s)) s = x;
        agn_poptop(L);  /* pop the number */
      }
      if (s == 0) {
        lua_pushfail(L);
        return 1;
      }
      lua_createtable(L, c, 0);
      c = 0;
      lua_pushnil(L);
      while (lua_next(L, 1)) {
        c++;
        lua_rawsetinumber(L, -3, c, agn_tonumber(L, -1)/s);
        agn_poptop(L);  /* pop the number */
      }
      break;
    }
    case LUA_TSEQ: {
      size_t i, nops;
      nops = agn_seqsize(L, 1);
      for (i=1; i <= nops; i++) {
        x = lua_seqgetinumber(L, 1, i);
        if (fabs(x) > fabs(s)) s = x;
      }
      if (s == 0) {
        lua_pushfail(L);
        return 1;
      }
      agn_createseq(L, nops);
      for (i=1; i <= nops; i++) {
        lua_seqsetinumber(L, -1, i, lua_seqgetinumber(L, 1, i)/s);
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  return 1;
}


/* stats.colnorm: Returns the largest absolute value of the numbers in a table or sequence, and the original
   value with the largest absolute magnitude. If the structure consists entirely of one or more 'undefined's,
   then the function returns the value undefined twice. If the structure is empty, 'fail' is returned.

   This is an extended version of the TI-Nspire's colNorm funtion when passed a one-dimensional matrix. See also:
   `stats.scale`, `stats.rownorm`.

   Agena 1.8.2, 03.10.2012 */

static int stats_colnorm (lua_State *L) {
  int type, onlyundefs;
  size_t n;
  lua_Number s, x;
  n = agn_nops(L, 1);
  if (n == 0) {
    lua_pushfail(L);
    return 1;
  }
  s = 0;  /* largest magnitude */
  onlyundefs = 1;   /* structure contains `undefined` only */
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  switch (type) {
    case LUA_TTABLE: {
      lua_pushnil(L);
      while (lua_next(L, 1)) {
        x = agn_checknumber(L, -1);
        if (!tools_isnan(x)) {
          onlyundefs = 0;
          if (fabs(x) > fabs(s)) s = x;
        }
        agn_poptop(L);  /* pop the number */
      }
      break;
    }
    case LUA_TSEQ: {
      size_t i, nops;
      nops = agn_seqsize(L, 1);
      for (i=1; i <= nops; i++) {
        x = lua_seqgetinumber(L, 1, i);
        if (!tools_isnan(x)) {
          onlyundefs = 0;
          if (fabs(x) > fabs(s)) s = x;
        }
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  if (onlyundefs) {
    lua_pushundefined(L); lua_pushundefined(L);
  } else {
    lua_pushnumber(L, fabs(s)); lua_pushnumber(L, s);
  }
  return 2;
}


/* stats.rownorm: Returns the sum of the absolute values of the numbers in a table or sequence. If the structure
   consists entirely of one or more 'undefined's, then the function returns 'undefined'. If the structure is empty,
   'fail' is returned.

   This is a version of the TI-Nspire's rowNorm funtion when passed a one-dimensional matrix. See also:
   `stats.scale`, `stats.colnorm`.

   Agena 1.8.2, 04.10.2012 */

static int stats_rownorm (lua_State *L) {
  int type, onlyundefs;
  size_t n;
  lua_Number s, x;
  n = agn_nops(L, 1);
  if (n == 0) {
    lua_pushfail(L);
    return 1;
  }
  s = 0;  /* largest magnitude */
  onlyundefs = 1;   /* structure contains `undefined` only */
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  switch (type) {
    case LUA_TTABLE: {
      lua_pushnil(L);
      while (lua_next(L, 1)) {
        x = agn_checknumber(L, -1);
        if (!tools_isnan(x)) {
          onlyundefs = 0;
          s += fabs(x);
        }
        agn_poptop(L);  /* pop the number */
      }
      break;
    }
    case LUA_TSEQ: {
      size_t i, nops;
      nops = agn_seqsize(L, 1);
      for (i=1; i <= nops; i++) {
        x = lua_seqgetinumber(L, 1, i);
        if (!tools_isnan(x)) {
          onlyundefs = 0;
          s += fabs(x);
        }
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  if (onlyundefs)
    lua_pushundefined(L);
  else
    lua_pushnumber(L, s);
  return 1;
}


/* Computes the probability density function for the normal distribution at the numeric value x. The defaults are
   mu = 0, with standard deviation sigma = 1; 1.10.6 */

static int stats_pdf (lua_State *L) {
  lua_Number x, mu, si, p, q;
  x = agn_checknumber(L, 1);
  mu = agnL_optnumber(L, 2, 0);
  si = agnL_optnumber(L, 3, 1);
  if (si <= 0)
    luaL_error(L, "Error in " LUA_QS ": standard deviation must be positive.", "stats.pdf");
  p = 1/(si*sqrt(PI2));
  q = -(x - mu)*(x - mu)/(2*si*si);
  lua_pushnumber(L, p * pow(_E, q));
  return 1;
}


static int stats_ndf (lua_State *L) {
  lua_Number si;
  if (lua_isnoneornil(L, 1))
    si = 1;
  else
    si = agn_checknumber(L, 1);
  if (si <= 0)
    luaL_error(L, "Error in " LUA_QS ": argument must be positive.", "stats.ndf");
  lua_pushnumber(L, 1/(si*sqrt(PI2)));
  return 1;
}


static int stats_nde (lua_State *L) {
  lua_Number x, mu, si;
  x = agn_checknumber(L, 1);
  mu = agnL_optnumber(L, 2, 0);
  si = agnL_optnumber(L, 3, 1);
  if (si <= 0)
    luaL_error(L, "Error in " LUA_QS ": third argument must be positive.", "stats.ndf");
  lua_pushnumber(L, exp(-(x-mu)*(x-mu)/(2*si*si)));
  return 1;
}


static int checkstructuresxx (lua_State *L, int *type, size_t *n, int *k, int *p, int *scientific,
    int *left, int *right, int outofrange, const char *procname) {  /* 2.1 RC 1 */
  luaL_checkany(L, 1);
  *type = lua_type(L, 1);
  luaL_typecheck(L, *type == LUA_TTABLE || *type == LUA_TSEQ, 1, "table or sequence expected", *type);
  luaL_typecheck(L, lua_type(L, 2) == LUA_TNUMBER, 2, "posint expected", lua_type(L, 2));
  luaL_typecheck(L, lua_type(L, 3) == LUA_TNUMBER, 3, "posint expected", lua_type(L, 3));
  *n = agn_nops(L, 1);
  if (*n < 1) {
    lua_pushfail(L);
    return 1;
  }
  *k = lua_tonumber(L, 2);  /* index of sample point */
  if (*k < 1) luaL_error(L, "Error in " LUA_QS ": second argument must be a posint.", procname);
  else if (*k > *n) luaL_error(L, "Error in " LUA_QS ": second argument too large.", procname);
  *p = lua_tonumber(L, 3);  /* period */
  if (*p < 2) luaL_error(L, "Error in " LUA_QS ": third argument must be greater than 1.", procname);
  else if (*n < *p) luaL_error(L, "Error in " LUA_QS ": too few elements in structure.", procname);
  *scientific = agnL_optboolean(L, 4, 0);  /* sample point at centre ? */
  if (*scientific) {  /* k is in the center of a period, period must be odd */
    lua_Number d, id;
    d = *p/2.0;
    id = trunc(d);
    if (d == id) luaL_error(L, "Error in " LUA_QS ": third argument must be an odd number.", procname);
    *left = *k - id;
    *right = *k + id;
  } else {
    *left = *k - *p + 1;
    *right = *k;
  }
  if (outofrange && (*left < 1 || *right > *n)) {
    lua_pushundefined(L);
    return 1;
  }
  return 0;
}

static int fillarray (lua_State *L, int idx, int type, lua_Number *a, size_t *ii, int left, int right, int reverse) {
  int i;
  size_t iii;
  lua_Number x;
  iii = 0;
  if (reverse) {
    switch (type) {
      case LUA_TTABLE: {
        for (i=right; i >= left; i--) {
          x = agn_getinumber(L, idx, i);
          if (tools_isnan(x)) return 1;
          a[++iii-1] = x;
        }
        break;
      }
      case LUA_TSEQ: {
        for (i=right; i >= left; i--) {
          x = lua_seqgetinumber(L, idx, i);
          if (tools_isnan(x)) return 1;
          a[++iii-1] = x;
        }
        break;
      }
      default: lua_assert(0);  /* should not happen */
    }
  } else {
    switch (type) {
      case LUA_TTABLE: {
        for (i=left; i <= right; i++) {
          x = agn_getinumber(L, idx, i);
          if (tools_isnan(x)) return 1;
          a[++iii-1] = x;
        }
        break;
      }
      case LUA_TSEQ: {
        for (i=left; i <= right; i++) {
          x = lua_seqgetinumber(L, idx, i);
          if (tools_isnan(x)) return 1;
          a[++iii-1] = x;
        }
        break;
      }
      default: lua_assert(0);  /* should not happen */
    }
  }
  *ii = iii;
  return 0;
}


static int stats_sma (lua_State *L) {  /* based on stats.amean */
  size_t ii, n;
  int left, right, type, scientific, k, p;  /* 1.12.3 */
  lua_Number *a, s;
  type = k = p = n = scientific = 0;  /* avoid compiler warnings */
  if (checkstructuresxx(L, &type, &n, &k, &p, &scientific, &left, &right, 1, "stats.sma")) return 1;
  /* if (left < 1) luaL_error(L, "Error in " LUA_QS ": second argument is too small.", "stats.sma");
    else if (right > sizeo) luaL_error(L, "Error in " LUA_QS ": second argument is too large.", "stats.sma"); */
  createarray(a, p, "stats.sma");
  ii = 0;
  if (fillarray(L, 1, type, a, &ii, left, right, 0)) {
    xfree(a);  /* 1.12.4 */
    luaL_error(L, "Error in " LUA_QS ": structure includes `undefined`.", "stats.sma");
  }
  s = kahanmean(a, ii);
  xfree(a);
  lua_pushnumber(L, s);
  return 1;
}


static int stats_smm (lua_State *L) {  /* based on stats.sma */
  size_t ii, n;
  int left, right, type, scientific, k, p;  /* 1.12.3 */
  lua_Number *a;
  type = k = p = n = scientific = left = right = ii = 0;  /* avoid compiler warnings */
  if (checkstructuresxx(L, &type, &n, &k, &p, &scientific, &left, &right, 1, "stats.smm")) return 1;
  createarray(a, p, "stats.smm");
  if (fillarray(L, 1, type, a, &ii, left, right, 0)) {
    xfree(a);  /* 1.12.4 */
    luaL_error(L, "Error in " LUA_QS ": structure includes `undefined`.", "stats.smm");
  }
  tools_dquicksort(a, 0, ii-1);
  lua_pushnumber(L, ii&1 ? tools_kth_smallest(a, ii, ii/2) :
    (tools_kth_smallest(a, ii, ii/2-1) + tools_kth_smallest(a, ii, ii/2))/2);
  xfree(a);
  return 1;
}


static int gsmm_iterator (lua_State *L) {
  lua_Number *a;
  size_t p, n, ii;
  int k, structure, type;
  k = lua_tonumber(L, lua_upvalueindex(2));
  p = lua_tonumber(L, lua_upvalueindex(3));
  n = lua_tonumber(L, lua_upvalueindex(4));
  if (k < 1 || (k + p - 1 > n && k <= n)) {
    lua_pushnumber(L, k + 1);
    lua_replace(L, lua_upvalueindex(2));  /* update index */
    lua_pushundefined(L);
    return 1;
  } else if (k > n) {
    lua_pushnil(L);
    return 1;
  }
  createarray(a, p, "stats.gsmm");
  ii = 0;
  structure = lua_upvalueindex(1);
  type = lua_type(L, structure);
  if (fillarray(L, structure, type, a, &ii, k, k + p - 1, 0)) {
    xfree(a);  /* 1.12.4 */
    luaL_error(L, "Error in " LUA_QS ": structure includes `undefined`.", "stats.gsmm");
  }
  tools_dquicksort(a, 0, ii-1);
  lua_pushnumber(L, k + 1);
  lua_replace(L, lua_upvalueindex(2));  /* update index k */
  lua_pushnumber(L, ii&1 ? tools_kth_smallest(a, ii, ii/2) :
    (tools_kth_smallest(a, ii, ii/2-1) + tools_kth_smallest(a, ii, ii/2))/2);
  xfree(a);
  return 1;  /* return median */
}

static int stats_gsmm (lua_State *L) {
  int left, right, type, k, p, scientific;
  size_t n;
  type = k = p = n = scientific = left = right = 0;  /* avoid compiler warnings */
  if (checkstructuresxx(L, &type, &n, &k, &p, &scientific, &left, &right, 0, "stats.gsmm")) return 1;
  lua_pushvalue(L, 1);      /* push structure, converting each number in the structure to an upvalue would speed up the function
                               even further, but in this case the number of samples would be limited to the maximum stack size. */
  lua_pushnumber(L, left);  /* push first sample point */
  lua_pushnumber(L, p);     /* push period */
  lua_pushnumber(L, n);     /* push size of structure */
  lua_pushcclosure(L, &gsmm_iterator, 4);  /* converts the values on the stack into upvalues and pops these three values from the stack */
  return 1;
}


static int gsma_iterator (lua_State *L) {
  lua_Number *a, s;
  size_t p, n, ii;
  int k, structure, type;
  k = lua_tonumber(L, lua_upvalueindex(2));
  p = lua_tonumber(L, lua_upvalueindex(3));
  n = lua_tonumber(L, lua_upvalueindex(4));
  if (k < 1 || (k + p - 1 > n && k <= n)) {
    lua_pushnumber(L, k + 1);
    lua_replace(L, lua_upvalueindex(2));  /* update index */
    lua_pushundefined(L);
    return 1;
  } else if (k > n) {
    lua_pushnil(L);
    return 1;
  }
  createarray(a, p, "stats.gsma");
  ii = 0;
  structure = lua_upvalueindex(1);
  type = lua_type(L, structure);
  if (fillarray(L, structure, type, a, &ii, k, k + p - 1, 0)) {
    xfree(a);  /* 1.12.4 */
    luaL_error(L, "Error in " LUA_QS ": structure includes `undefined`.", "stats.gsma");
  }
  s = kahanmean(a, ii);
  lua_pushnumber(L, k + 1);
  lua_replace(L, lua_upvalueindex(2));  /* update index k */
  lua_pushnumber(L, s);
  xfree(a);
  return 1;  /* return moving mean */
}

static int stats_gsma (lua_State *L) {
  int left, right, type, scientific, k, p;
  size_t n;
  type = k = p = n = scientific = left = right = 0;  /* avoid compiler warnings */
  if (checkstructuresxx(L, &type, &n, &k, &p, &scientific, &left, &right, 0, "stats.gsma")) return 1;
  lua_pushvalue(L, 1);      /* push structure, converting each number in the structure to an upvalue would speed up the function
                               even further, but in this case the number of samples would be limited to the maximum stack size. */
  lua_pushnumber(L, left);  /* push first sample point */
  lua_pushnumber(L, p);     /* push period */
  lua_pushnumber(L, n);     /* push size of structure */
  lua_pushcclosure(L, &gsma_iterator, 4);  /* converts the values on the stack into upvalues and pops these three values from the stack */
  return 1;
}


/*
stats.ema (obj, k, alpha [, mode [, y0star]])

Computes the exponential moving average of a table or sequence obj up to and inclduing its k-th element.

The smoothing factor alpha is a rational number in the range [0, 1].

The function supports two algorithms: If mode is 1 (the default), then the algorithm

      r := alpha * obj[k];
      s := 1 - alpha;
      for i from k - 1 to 1 by -1 do
         r := r + alpha * s ^ i * obj[i];
      od;
      r := r + s ^ k * y0star;

is used to compute the result r. In mode 1, you can pass an explicit first estimate y0*, otherwise
the first value y0* is equal to the sample moving average of obj.

If mode is 2, then the formula

      r := obj[k];
      for i from k - 1 to 1 by -1 do
         r := r + alpha * (obj[i] - r)
      od;

is applied.

The result is a number.
*/

static int checkstructureema (lua_State *L, int *type, size_t *n, int *k, lua_Number *alpha, const char *procname) {  /* 2.1 RC 1
  nok: no `k` value given;
  !!! do not delete the nok parameter for the future implementation of stats.fema !!! */
  luaL_checkany(L, 1);
  *type = lua_type(L, 1);
  luaL_typecheck(L, *type == LUA_TTABLE || *type == LUA_TSEQ, 1, "table or sequence expected", *type);
  luaL_typecheck(L, lua_type(L, 2) == LUA_TNUMBER, 2, "posint expected", lua_type(L, 2));
  *n = agn_nops(L, 1);
  if (*n < 1) {
    lua_pushfail(L);
    return 1;
  }
  *k = lua_tonumber(L, 2);  /* index of sample point */
  if (*k < 1) luaL_error(L, "Error in " LUA_QS ": second argument must be a posint.", procname);
  else if (*k > *n) luaL_error(L, "Error in " LUA_QS ": second argument too large.", procname);
  *alpha = agn_checknumber(L, 3);  /* smoothing factor */
  if (*alpha < 0 || *alpha > 1)
    luaL_error(L, "Error in " LUA_QS ": smoothing factor must be in the range [0, 1].", procname);
  return 0;
}


static int aux_ema (lua_State *L, lua_Number *a,  size_t ii, lua_Number alpha, int mode, lua_Number y0star, lua_Number *r) {  /* 2.1 RC 1 */
  size_t i;
  switch (mode) {
    case 1: {  /* see http://de.wikipedia.org/wiki/Exponentielle_Gl%C3%A4ttung */
      lua_Number s;
      if (tools_isnan(y0star)) y0star = kahanmean(a, ii);
      *r = alpha * a[0];
      s = 1 - alpha;
      for (i=1; i < ii; i++) *r += alpha * tools_intpow(s, i) * a[i];
      *r += tools_intpow(s, ii) * y0star;
      break;
    }
    case 2: {  /* see: http://stackoverflow.com/questions/7947352/exponential-moving-average */
      *r = a[0];
      for (i=1; i < ii; i++) *r += alpha * (a[i] - *r);
      break;
    }
    default:
      return 1;
  }
  return 0;
}

static int stats_ema (lua_State *L) {  /* 2.1 RC 1 */
  size_t ii, n;
  int type, k, mode;
  lua_Number *a, alpha, r, y0star;
  ii = type = k = n = r = 0;  /* avoid compiler warnings */
  if (checkstructureema(L, &type, &n, &k, &alpha, "stats.ema")) return 1;
  mode = agnL_optinteger(L, 4, 1);
  y0star = agnL_optnumber(L, 5, AGN_NAN);
  createarray(a, k, "stats.ema");
  /* fill array in the opposite direction to allow hardware prefetching */
  if (fillarray(L, 1, type, a, &ii, 1, k, 1)) {
    xfree(a);
    luaL_error(L, "Error in " LUA_QS ": structure includes `undefined`.", "stats.ema");
  }
  if (aux_ema(L, a, ii, alpha, mode, y0star, &r)) {
    xfree(a);
    luaL_error(L, "Error in " LUA_QS ": unknown mode %d.", "stats.ema", mode);
  }
  xfree(a);
  lua_pushnumber(L, r);
  return 1;
}


static int gema_iterator (lua_State *L) {  /* 2.1 RC 1 */
  lua_Number *a;
  size_t n, ii;
  int k, structure, type, mode;
  lua_Number alpha, y0star, r;
  r = 0;
  if (lua_gettop(L) != 0)
    luaL_error(L, "Error in " LUA_QS ": iterator requires no arguments.", "stats.gema");
  k = lua_tonumber(L, lua_upvalueindex(2));
  n = lua_tonumber(L, lua_upvalueindex(3));
  alpha = lua_tonumber(L, lua_upvalueindex(4));
  mode = lua_tointeger(L, lua_upvalueindex(5));
  y0star = lua_tonumber(L, lua_upvalueindex(6));
  if (k < 1 || k > n) {
    lua_pushnil(L);
    return 1;
  }
  createarray(a, k, "stats.gema");
  ii = 0;
  structure = lua_upvalueindex(1);
  type = lua_type(L, structure);
  if (fillarray(L, structure, type, a, &ii, 1, k, 1)) {
    xfree(a);  /* 1.12.4 */
    luaL_error(L, "Error in " LUA_QS ": structure includes `undefined`.", "stats.gema");
  }
  if (aux_ema(L, a, ii, alpha, mode, y0star, &r)) {
    xfree(a);
    luaL_error(L, "Error in " LUA_QS ": unknown mode %d.", "stats.gema", mode);
  }
  lua_pushnumber(L, k + 1);
  lua_replace(L, lua_upvalueindex(2));  /* update index k */
  lua_pushnumber(L, r);
  xfree(a);
  return 1;  /* return median */
}

/*

stats.gema (obj, alpha [, mode [, y0*]])

Like stats.ema, but returns a function that, each time it is called, returns the exponential
moving average, starting with sample obj[1], and progressing with sample obj[2], obj[3], etc.
with subsequent calls. It return `null` if there are no more samples in obj. It is much faster
than `stats.ema` with large observations.

The smoothing factor alpha is a rational number in the range [0, 1].

The function supports two algorithms: If mode is 1 (the default), then the algorithm

   ...

is used to compute the result. In mode 1, you can pass an explicit first estimate y0*, otherwise
the first value y0* is equal to the sample moving average of obj.

If mode is 2, then the formula

   ...

is applied to the period.

The result is a number.
*/

static int stats_gema (lua_State *L) {  /* 2.1 RC 1 */
  int type, k, mode;
  lua_Number alpha, y0star;
  size_t n;
  type = k = n = 0;  /* avoid compiler warnings */
  if (checkstructureema(L, &type, &n, &k, &alpha, "stats.gema")) return 1;
  mode = agnL_optinteger(L, 4, 1);
  y0star = agnL_optnumber(L, 5, AGN_NAN);
  lua_pushvalue(L, 1);        /* push structure, converting each number in the structure to an upvalue would speed up the function
                                 even further, but in this case the number of samples would be limited to the maximum stack size. */
  lua_pushnumber(L, k);       /* push first sample point */
  lua_pushnumber(L, n);       /* push _total_ size of structure */
  lua_pushnumber(L, alpha);   /* push smoothing factor alpha */
  lua_pushnumber(L, mode);    /* push mode */
  lua_pushnumber(L, y0star);  /* push y0star */
  lua_pushcclosure(L, &gema_iterator, 6);  /* converts the values on the stack into upvalues and pops these three values from the stack */
  return 1;
}


/* returns all elements in a table or sequence obj from the p-th percentile rank up but not including
   the q-th percentile rank. p and q must be positive integers in the range (0 .. [100. If p and q
   are not given, p is set to 25, and q to 75. If q is not given, it is set to 100 - p. The type of
   return is determined by the type of obj. */

#define FLOOR(x) floor((x) + 0.5)

static int stats_prange (lua_State *L) {  /* 1.12.9 */
  size_t i, n, nargs, p, q, c, rankp, rankq;
  int type;
  lua_Number x1, x2, *a;
  luaL_checkany(L, 1);
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  n = agn_nops(L, 1);
  nargs = lua_gettop(L);
  if (n < 2) {
    lua_pushfail(L);
    return 1;
  }
  if (nargs > 1) {
    p = agn_checkinteger(L, 2);
    if (p < 0 || p > 99)
      luaL_error(L, "Error in " LUA_QS ": second argument must be in the range [0, 100).", "stats.prange");
  } else
    p = 25;
  if (nargs > 2) {
    q = agn_checkinteger(L, 3);
    if (q < 0 || q > 99)
      luaL_error(L, "Error in " LUA_QS ": third argument must be in the range [0, 100).", "stats.prange");
  } else
    q = 100 - p;
  if (p > q)
    luaL_error(L, "Error in " LUA_QS ": second argument must be less than third argument.", "stats.prange");
  /* create an array and fill it */
  createarray(a, n, "stats.prange");
  switch (type) {
    case LUA_TTABLE: {
      for (i=0; i < n; i++)
        a[i] = agn_getinumber(L, 1, i+1);
      break;
    }
    case LUA_TSEQ: {
      for (i=0; i < n; i++)
        a[i] = lua_seqgetinumber(L, 1, i+1);
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  x1 = a[0];
  for (i=1; i < n; i++) {
    x2 = a[i];
    if (x1 > x2) {  /* array unsorted ? -> sort it in ascending order */
      tools_dquicksort(a, 0, n);
      break;
    }
    x1 = x2;
  }
  rankp = FLOOR((lua_Number)p/100*n+0.5);
  rankq = FLOOR((lua_Number)q/100*n+0.5);
  c = 1;
  if (type == LUA_TTABLE) {
    lua_createtable(L, n*(q - p)/100+1, 0);
    for (i=rankp-1; i < rankq-1; i++)
      lua_rawsetinumber(L, -1, c++, a[i]);
  } else {
    agn_createseq(L, n*(q - p)/100+1);
    for (i=rankp-1; i < rankq-1; i++)
      lua_seqsetinumber(L, -1, c++, a[i]);
  }
  xfree(a);
  return 1;
}


static const luaL_Reg statslib[] = {
  {"amean", stats_amean},                   /* added on August 26, 2012 */
  {"ad", stats_ad},                         /* added on March 11, 2012 */
  {"colnorm", stats_colnorm},               /* added on October 03, 2012 */
  {"cumsum", stats_cumsum},                 /* added on September 10, 2012 */
  {"deltalist", stats_deltalist},           /* added on March 12, 2012 */
  {"ema", stats_ema},                       /* added on December 06, 2013 */
  {"gsma", stats_gsma},                     /* added on July 21, 2013 */
  {"gema", stats_gema},                     /* added on December 09, 2013 */
  {"gsmm", stats_gsmm},                     /* added on July 20, 2013 */
  {"ios", stats_ios},                       /* added on March 11, 2012 */
  {"issorted", stats_issorted},             /* added on March 31, 2012 */
  {"mad", stats_mad},                       /* added on September 11, 2013 */
  {"median", stats_median},                 /* added on June 19, 2007 */
  {"minmax", stats_minmax},                 /* added on June 20, 2007 */
  {"moment", stats_moment},                 /* added on March 11, 2012 */
  {"nde", stats_nde},                       /* added on April 14, 2013 */
  {"ndf", stats_ndf},                       /* added on April 13, 2013 */
  {"pdf", stats_pdf},                       /* added on April 11, 2013 */
  {"prange", stats_prange},                 /* added on November 04, 2013 */
  {"rownorm", stats_rownorm},               /* added on October 04, 2012 */
  {"scale", stats_scale},                   /* added on October 01, 2012 */
  {"sma", stats_sma},                       /* added on July 11, 2013 */
  {"smallest", stats_smallest},             /* added on December 30, 2012 */
  {"smm", stats_smm},                       /* added on July 15, 2013 */
  {"sum", stats_sum},                       /* added on October 05, 2012 */
  {"sorted", stats_sorted},                 /* added on October 07, 2012 */
  {"sumdata", stats_sumdata},               /* added on March 11, 2012 */
  {"tovals", stats_tovals},                 /* added on March 11, 2012 */
  {NULL, NULL}
};


static void createmetatable (lua_State *L) {
  lua_createtable(L, 0, 1);  /* create metatable for strings */
  lua_pushliteral(L, "");  /* dummy string */
  lua_pushvalue(L, -2);
  lua_setmetatable(L, -2);  /* set string metatable */
  agn_poptop(L);  /* pop dummy string */
  lua_pushvalue(L, -2);  /* string library... */
  lua_setfield(L, -2, "__index");  /* ...is the __index metamethod */
  agn_poptop(L);  /* pop metatable */
}


/*
** Open stats library
*/
LUALIB_API int luaopen_stats (lua_State *L) {
  luaL_register(L, AGENA_STATSLIBNAME, statslib);
  createmetatable(L);
  return 1;
}

/* ====================================================================== */

