/*
** $Id: skycrane.c, initiated January 31, 2013 $
** Skycrane utilities library
** See Copyright Notice in agena.h
*/


#include <stdlib.h>
#include <string.h>
#include <math.h>

#define skycrane_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "agnhlps.h"
#include "agncmpt.h"

#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_SKYCRANELIBNAME "skycrane"
LUALIB_API int (luaopen_skycrane) (lua_State *L);
#endif


static int skycrane_tocomma (lua_State *L) {
  char *str, *r;
  size_t l, i;
  if (lua_type(L, 1) == LUA_TNUMBER)
    str = (char *)lua_tostring(L, 1);
  else if (lua_type(L, 1) == LUA_TSTRING) {
    str = (char *)agn_tostring(L, 1);
    if (tools_isnumericstring(str) == 0)
      luaL_error(L, "Error in " LUA_QS ": expected a numeric string.", "skycrane.tocomma");
  } else {
    str = NULL;  /* to avoid compiler warning */
    luaL_error(L, "Error in " LUA_QS ": expected a number or numeric string.", "skycrane.tocomma");
  }
  l = strlen(str);
  r = (char *)agn_malloc(L, (l+1)*sizeof(char), "skycrane.tocomma", NULL);
  for (i=0; i < l; i++) {
    if (*str == '.') r[i] = ','; else r[i] = *str;
    str++;
  }
  r[l] = '\0';
  lua_pushstring(L, r);
  free(r);
  return 1;
}


static int skycrane_removedquotes (lua_State *L) {
  size_t l;
  const char *str;
  str = agn_checklstring(L, 1, &l);
  if (l > 1 && str[0] == '"' && str[l-1] == '"') {
    size_t i;
    char *r;
    r = (char *)agn_malloc(L, sizeof(char)*(l-1), "skycrane.removedquotes", NULL);
    for (i=0; i < l-2; i++) r[i] = str[i+1];
    r[l-2] = '\0';
    lua_pushstring(L, r);
    free(r);
  } else
    lua_pushvalue(L, 1);
  return 1;
}


static int sema_floatcounter (lua_State *L) {
  lua_Number idx, step, y, t, c;
  idx = lua_tonumber(L, lua_upvalueindex(1));
  step = lua_tonumber(L, lua_upvalueindex(2));
  y = lua_tonumber(L, lua_upvalueindex(3));
  t = lua_tonumber(L, lua_upvalueindex(4));
  c = lua_tonumber(L, lua_upvalueindex(5));
  y = step - c;
  t = idx + y;
  c = (t - idx) - y;
  idx = t;
  lua_pushnumber(L, idx);  /* new value */
  lua_pushvalue(L, -1);    /* put it on stack for later return */
  lua_replace(L, lua_upvalueindex(1));  /* update upvalue */
  lua_pushnumber(L, y);
  lua_replace(L, lua_upvalueindex(3));  /* update upvalue */
  lua_pushnumber(L, t);
  lua_replace(L, lua_upvalueindex(4));  /* update upvalue */
  lua_pushnumber(L, c);
  lua_replace(L, lua_upvalueindex(5));  /* update upvalue */
  return 1;  /* return new value */
}

static int sema_intcounter (lua_State *L) {
  int val = lua_tonumber(L, lua_upvalueindex(1));
  lua_pushnumber(L, ++val);  /* new value */
  lua_pushvalue(L, -1);  /* duplicate it */
  lua_replace(L, lua_upvalueindex(1));  /* update upvalue */
  return 1;  /* return new value */
}

static int sema_intstepcounter (lua_State *L) {
  int val = lua_tonumber(L, lua_upvalueindex(1)) + lua_tonumber(L, lua_upvalueindex(2));
  lua_pushnumber(L, val);  /* new value */
  lua_pushvalue(L, -1);  /* duplicate it */
  lua_replace(L, lua_upvalueindex(1));  /* update upvalue */
  return 1;  /* return new value */
}

static int skycrane_counter (lua_State *L) {
  lua_Number count, step;
  count = luaL_optnumber(L, 1, 0);
  step = luaL_optnumber(L, 2, 1);
  if (ISINT(count) && ISINT(step)) {  /* all integers ? */
    lua_pushnumber(L, count - step);
    if (step == 1)
      lua_pushcclosure(L, &sema_intcounter, 1);
    else {
      lua_pushnumber(L, step);
      lua_pushcclosure(L, &sema_intstepcounter, 2);
    }
  } else {  /* apply Kahan-Ozawa summation preventing round-off errors */
    lua_Number c, y, t, idx;
    idx = count - step;
    y = step;
    t = idx + y;
    c = (t - idx) - y;
    lua_pushnumber(L, idx);     /* 1 - idx */
    lua_pushnumber(L, step);    /* 2 - step */
    lua_pushnumber(L, y);       /* 3 - y */
    lua_pushnumber(L, t);       /* 4 - t */
    lua_pushnumber(L, c);       /* 5 - c */
    lua_pushcclosure(L, &sema_floatcounter, 5);  /* pops these five values from the stack */
  }
  return 1;
}


static int skycrane_enclose (lua_State *L) {
  const char *str, *enc;
  char *r;
  str = lua_tostring(L, 1);
  if (str == NULL)
    luaL_error(L, "Error in " LUA_QS ": expected a string or number for argument #1.", "skycrane.enclose");
  enc = luaL_optstring(L, 2, "\"");
  r = concat(enc, str, enc, NULL);
  if (r == NULL)
    luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "skycrane.enclose");
  lua_pushstring(L, r);
  xfree(r);
  return 1;
}


static int skycrane_trimpath (lua_State *L) {
  char *str;
  str = (char *)agn_checkstring(L, 1);
  charreplace(str, '\\', '/', 1);
  lua_pushstring(L, str);
  return 1;
}


static const luaL_Reg skycranelib[] = {
  {"counter", skycrane_counter},                /* added on May 26, 2013 */
  {"enclose", skycrane_enclose},                /* added on May 26, 2013 */
  {"removedquotes", skycrane_removedquotes},    /* added on February 01, 2013 */
  {"tocomma", skycrane_tocomma},                /* added on January 31, 2013 */
  {"trimpath", skycrane_trimpath},              /* added on February 01, 2013 */
  {NULL, NULL}
};


/*
** Open skycrane library
*/
LUALIB_API int luaopen_skycrane (lua_State *L) {
  luaL_register(L, AGENA_SKYCRANELIBNAME, skycranelib);
  return 1;
}

