/*
** $Id: net.c, by Alexander Walz - initiated April 05, 2012
** Network library vor IPv4 connections
** See Copyright Notice in agena.h
** largely based on procedures published in Juergen Wolf's book 'C von A bis Z', Galileo Computing, Bonn, 3rd edition 2009
*/

#if defined(_WIN32) || defined(__unix__) || defined(__linux__) || defined(__APPLE__)

#include <errno.h>
#include <fcntl.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef _WIN32
#define FD_SETSIZE 4096  /* must be put before the #include of Winsock */
#include <winsock2.h>
#include <io.h>
#else
/* header files for UNIX/Linux */
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <signal.h>  /* 1.6.4, to avoid crashes with closed connections */
#endif

/* the following package ini declarations must be included after `#include <` and before `include #` ! */

#include "agnhlps.h"
#include "agncmpt.h"  /* trunc */

#define net_c
#define LUA_LIB

#include "agena.h"
#include "agnxlib.h"
#include "agenalib.h"
#include "net.h"
#include "llimits.h"  /* for MAX_INT */


#if !(defined(LUA_DOS))
#define AGENA_NETLIBNAME "net"
LUALIB_API int (luaopen_net) (lua_State *L);
#endif


static int so_reuseaddr = 1;

#define AGN_NET_CONNECT "net.connect"
#define AGN_NET_ACCEPT "net.accept"


/* auxiliary functions to administer sockets */
struct tree *socketattribs;

/* initialise binary tree, returns tree or NULL in case of errors */
static TREE *treeinit (void) {
  TREE *baum = malloc(sizeof *baum);
  if (baum == NULL) {  /* no memory available */
    return NULL;
  }
  else {  /* initialise */
    baum->root = NULL;
    baum->size = 0;
    return baum;
  }
}


/* insert element into the tree */
static int treeinsert (TREE *baum, AGN_SOCKET idx, STATUS *data) {
  NODE *knoten, **neu;
  neu = (NODE **) &baum->root;
  knoten = (NODE *) baum->root;
  for (;;) {
    if (knoten == NULL) {  /* empty `slot` found */
      knoten = *neu = malloc(sizeof *knoten);
      if (knoten != NULL) {  /* insert data */
        knoten->index = idx;
        knoten->status = malloc(sizeof *data);
        memcpy(knoten->status, data, sizeof *data);
        knoten->left = knoten->right = NULL;
        baum->size++;
        return 1;  /* success */
      }
      else
        return 0;  /* memory allocation error */
    }
    else if (idx > knoten->index)  {  /* search right side */
      neu = &knoten->right;
      knoten = knoten->right;
    }
    else {  /* search left side */
      neu = &knoten->left;
      knoten = knoten->left;
    }
  }
}


/* search for a socket handle denoted by idx */
static STATUS *treesearch (const TREE *baum, AGN_SOCKET idx) {
  const NODE *knoten;
  knoten = (NODE*) baum->root;
  for (;;) {
    if (knoten == NULL) {
      return NULL;
    }
    if (idx == knoten->index) {
      return knoten->status;
    }
    else if(idx > knoten->index)
      knoten = knoten->right;
    else
      knoten = knoten->left;
  }
}


/* updates administrative socket data for the handle idx */
static int treeupdate (const TREE *baum, AGN_SOCKET idx, STATUS *s) {
  const NODE *knoten;
  knoten = (NODE*) baum->root;
  for (;;) {
    if (knoten == NULL) {
      return -1;
    }
    if (idx == knoten->index) {
      memcpy(knoten->status, s, sizeof(knoten));
      return 0;
    }
    else if(idx > knoten->index)
      knoten = knoten->right;
    else
      knoten = knoten->left;
  }
}

/* deletes all administrative data for the given socket handle from the tree */
static int treedelete (TREE *baum, AGN_SOCKET idx) {
  /* pointer_z is the entry to be deleted  */
  NODE **pointer_q, *pointer_z, *pointer_y, *pointer_x;
  pointer_q = (struct node **)&baum->root;
  pointer_z = (struct node *)baum->root;
  for (;;) {
    if (pointer_z == NULL)
      return 0;
    else if (idx == pointer_z->index)  /* found idx that is to be deleted */
      break;
    else if (idx > pointer_z->index) {
      pointer_q = &pointer_z->right;
      pointer_z = pointer_z->right;
    }
    else { /* element to be deleted is smaller */
      pointer_q = &pointer_z->left;
      /* search left branch */
      pointer_z = pointer_z->left;
    }
  }
  /* process entry to be deleted */
  if (pointer_z->right == NULL)  /* right branch has no nodes */
    *pointer_q = pointer_z->left;
  else {  /* pointer_z has a right branch, but no left one */
    pointer_y = pointer_z->right;
    if (pointer_y->left == NULL) {
      /* pointer_z->right has no left branch */
      pointer_y->left = pointer_z->left;
      *pointer_q = pointer_y;
    }
    else {  /* there is a left branch */
      pointer_x = pointer_y->left;
      while (pointer_x->left != NULL) {  /* search as long as there is no further left branch */
        pointer_y = pointer_x;
        pointer_x = pointer_y->left;
      }
      /* now reconnect nodes */
      pointer_y->left = pointer_x->right;
      pointer_x->left = pointer_z->left;
      pointer_x->right = pointer_z->right;
      *pointer_q = pointer_x;
    }
  }
  baum->size--;
  free(pointer_z->status);
  xfree(pointer_z);
  return 1;
}


/* some auxiliary functions */

/* Winsock and Winsock 2 error mapping, taken from:
   http://msdn.microsoft.com/en-us/library/windows/desktop/ms740668%28v=vs.85%29.aspx
   "Windows Sockets Error Codes"
*/
#ifdef _WIN32
static const char *wstrerror(int err) {
  switch (err) {
    case WSAEACCES: return "permission denied (10013)";
    case WSAEADDRINUSE: return "address already in use (10048)";
    case WSAEADDRNOTAVAIL: return "cannot assign requested address (10049)";
    case WSAEAFNOSUPPORT: return "address family not supported by protocol family (10047)";
    case WSAEALREADY: return "operation already in progress (10037)";
    case WSAEBADF: return "file handle is not valid (10009)";
    case WSAECANCELLED: return "call has been canceled (10103)";
    case WSAECONNABORTED: return "software caused connection abort (10053)";
    case WSAECONNREFUSED: return "connection refused (10061)";
    case WSAECONNRESET: return "connection reset by peer (10054)";
    case WSAEDESTADDRREQ: return "destination address required (10039)";
    case WSAEDISCON: return "graceful shutdown in progress (10101)";
    case WSAEDQUOT: return "disk quota exceeded (10069)";
    case WSAEFAULT: return "bad address (10014)";
    case WSAEHOSTDOWN: return "host is down (10064)";
    case WSAEHOSTUNREACH: return "no route to host (10065)";
    case WSAEINPROGRESS: return "operation now in progress (10036)";
    case WSAEINTR: return "interrupted function call (10004)";
    case WSAEINVAL: return "invalid argument (10022)";
    case WSAEINVALIDPROCTABLE: return "procedure call table is invalid (10104)";
    case WSAEINVALIDPROVIDER: return "service provider is invalid (10105)";
    case WSAEISCONN: return "socket is already connected (10056)";
    case WSAELOOP: return "cannot translate name (10062)";
    case WSAEMFILE: return "too many open files (10024)";
    case WSAEMSGSIZE: return "message too long (10040)";
    case WSAENAMETOOLONG: return "name too long (10063)";
    case WSAENETDOWN: return "network is down (10050)";
    case WSAENETRESET: return "network dropped connection on reset (10052)";
    case WSAENETUNREACH: return "network is unreachable (10051)";
    case WSAENOBUFS: return "no buffer space available (10055)";
    case WSAENOMORE: return "no more results (10102)";
    case WSAENOPROTOOPT: return "bad protocol option (10042)";
    case WSAENOTCONN: return "socket is not connected (10057)";
    case WSAENOTEMPTY: return "directory not empty (10066)";
    case WSAENOTSOCK: return "socket operation on nonsocket (10038)";
    case WSAEOPNOTSUPP: return "operation not supported (10045)";
    case WSAEPFNOSUPPORT: return "protocol family not supported (10046)";
    case WSAEPROCLIM: return "too many processes (10067)";
    case WSAEPROTONOSUPPORT: return "protocol not supported (10043)";
    case WSAEPROTOTYPE: return "protocol wrong type for socket (10041)";
    case WSAEPROVIDERFAILEDINIT: return "service provider failed to initialize (10106)";
    case WSAEREFUSED: return "database query was refused (10112)";
    case WSAEREMOTE: return "item is remote (10071)";
    case WSAESHUTDOWN: return "cannot send after socket shutdown (10058)";
    case WSAESOCKTNOSUPPORT: return "socket type not supported (10044)";
    case WSAESTALE: return "stale file handle reference (10070)";
    case WSAETIMEDOUT: return "connection timed out (10060)";
    case WSAETOOMANYREFS: return "too many references (10059)";
    case WSAEUSERS: return "user quota exceeded (10068)";
    case WSAEWOULDBLOCK: return "resource temporarily unavailable (10035)";
    case WSAHOST_NOT_FOUND: return "host not found (11001)";
    case WSANOTINITIALISED: return "successful WSAStartup not yet performed (10093)";
    case WSANO_DATA: return "valid name, no data record of requested type (11004)";
    case WSANO_RECOVERY: return "this is a nonrecoverable error (11003)";
    case WSASERVICE_NOT_FOUND: return "service not found (10108)";
    case WSASYSCALLFAILURE: return "system call failure (10107)";
    case WSASYSNOTREADY: return "network subsystem is unavailable (10091)";
    case WSATRY_AGAIN: return "nonauthoritative host not found (11002)";
    case WSATYPE_NOT_FOUND: return "class type not found (10109)";
    case WSAVERNOTSUPPORTED: return "winsock.dll version out of range (10092)";
    case WSA_E_CANCELLED: return "call was canceled (10111)";
    case WSA_E_NO_MORE: return "no more results (10110)";
    case WSA_INVALID_HANDLE: return "specified event object handle is invalid (6)";
    case WSA_INVALID_PARAMETER: return "one or more parameters are invalid (87)";
    case WSA_IO_INCOMPLETE: return "overlapped I/O event object not in signaledstate (996)";
    case WSA_IO_PENDING: return "overlapped operations will complete later (997)";
    case WSA_NOT_ENOUGH_MEMORY: return "insufficient memory available (8)";
    case WSA_OPERATION_ABORTED: return "overlapped operation aborted (995)";
    case WSA_QOS_ADMISSION_FAILURE: return "QoS admission error (11010)";
    case WSA_QOS_BAD_OBJECT: return "QoS bad object (11013)";
    case WSA_QOS_BAD_STYLE: return "QoS bad style (11012)";
    case WSA_QOS_EFILTERCOUNT: return "incorrect QoS filter count (11021)";
    case WSA_QOS_EFILTERSTYLE: return "invalid QoS filter style (11019)";
    case WSA_QOS_EFILTERTYPE: return "invalid QoS filter type (11020)";
    case WSA_QOS_EFLOWCOUNT: return "incorrect QoS flow count (11023)";
    case WSA_QOS_EFLOWDESC: return "invalid QoS flow descriptor (11026)";
    case WSA_QOS_EFLOWSPEC: return "QoS flowspec error (11017)";
    case WSA_QOS_EOBJLENGTH: return "invalid QoS object length (11022)";
    case WSA_QOS_EPOLICYOBJ: return "invalid QoS policy object (11025)";
    case WSA_QOS_EPROVSPECBUF: return "invalid QoS provider buffer (11018)";
    case WSA_QOS_EPSFILTERSPEC: return "invalid QoS provider-specific filterspec (11028)";
    case WSA_QOS_EPSFLOWSPEC: return "invalid QoS provider-specific flowspec (11027)";
    case WSA_QOS_ESDMODEOBJ: return "invalid QoS shape discard mode object (11029)";
    case WSA_QOS_ESERVICETYPE: return "QoS service type error (11016)";
    case WSA_QOS_ESHAPERATEOBJ: return "invalid QoS shaping rate object (11030)";
    case WSA_QOS_EUNKOWNPSOBJ: return "unrecognized QoS object (11024)";
    case WSA_QOS_GENERIC_ERROR: return "QoS generic error (11015)";
    case WSA_QOS_NO_RECEIVERS: return "QoS no receivers (11008)";
    case WSA_QOS_NO_SENDERS: return "no QoS senders (11007)";
    case WSA_QOS_POLICY_FAILURE: return "QoS policy failure (11011)";
    case WSA_QOS_RECEIVERS: return "QoS receivers (11005)";
    case WSA_QOS_REQUEST_CONFIRMED: return "QoS request confirmed (11009)";
    case WSA_QOS_SENDERS: return "QoS senders (11006)";
    case WSA_QOS_TRAFFIC_CTRL_ERROR: return "QoS traffic control error (11014)";
    default: return "unknown error";
  }
}
#endif


static void errornosock (lua_State *L, char *error_message, const char* procname) {  /* issue error message */
#ifdef _WIN32
  const char *message = wstrerror(WSAGetLastError());
  if (strcmp(message, "unknown error") == 0)
    luaL_error(L, "Error in " LUA_QS ": %s.", procname, message);
  else
    luaL_error(L, "Error in " LUA_QS ": %s (%d).", procname, error_message, WSAGetLastError());
#else
  luaL_error(L, "Error in " LUA_QS ": %s (%s).", procname, error_message, strerror(errno));
#endif
}


/* return `false' and error message */
#if defined( _WIN32)
#define agn_neterror(L) { \
  lua_pushfalse(L); \
  if (strcmp(wstrerror(WSAGetLastError()), "unknown error") != 0) \
    lua_pushstring(L, wstrerror(WSAGetLastError())); \
   else \
    lua_pushstring(L, "unknown error"); \
  return 2; \
}
#else
#define agn_neterror(L) {\
  lua_pushfalse(L); \
  lua_pushstring(L, strerror(errno)); \
  return 2; \
}
#endif


#define agn_neterror2(L,msg) { \
  lua_pushfalse(L); \
  lua_pushstring(L, msg); \
  return 2; \
}


#define agn_neterrorfail(L,msg) { \
  lua_pushfail(L); \
  lua_pushstring(L, msg); \
  return 2; \
}


/* Taken from LuaSocket 2.0.2 license Copyright 2004-2007 Diego Nehab, file inet.c */
#ifndef inet_aton
int inet_aton (const char *cp, struct in_addr *inp) {
  unsigned int a = 0, b = 0, c = 0, d = 0;
  int n = 0, r;
  unsigned long int addr = 0;
  r = sscanf(cp, "%u.%u.%u.%u%n", &a, &b, &c, &d, &n);
  if (r == 0 || n == 0) return 0;
  cp += n;
  if (*cp) return 0;
  if (a > 255 || b > 255 || c > 255 || d > 255) return 0;
  if (inp) {
    addr += a; addr <<= 8;
    addr += b; addr <<= 8;
    addr += c; addr <<= 8;
    addr += d;
    inp->s_addr = htonl(addr);
  }
  return 1;
}
#endif


/* converts a non-numeric (or numeric) IP address into a numeric IP address */
static char *tonumericip (const char *address) {  /* 1.6.4 */
  struct in_addr addr;
  if ((inet_aton(address, &addr)) != 0) {  /* valid address ? */
    return inet_ntoa(addr);
  } else {
    struct hostent *host_info;
    /* convert server name to IP address */
    host_info = gethostbyname(address);
    if (host_info == NULL) {
      return NULL;
    }
    /* IP address of server */
    return host_info->h_addr;
  }
}


static void checkport (lua_State *L, int port, const char *procname) {
  if (port < 0 || port > 65535)
    luaL_error(L, "Error in " LUA_QS ": port must be in range [0, 65535], got %d.", procname, port);
}


/* check black and whitelist for outgoing or incoming connections */
static int checklist (lua_State *L, AGN_SOCKET sock, const char *address, const char *list, const char *procname) {  /* 1.6.4 */
  if (agnL_gettablefield(L, "net", list, procname, 1) == LUA_TSET) {
    int isblacklist;
    char *ipaddress;
    isblacklist = (strcmp(list, "blacklist") == 0);
    ipaddress = tonumericip(address);
    if (ipaddress) {  /* IP address could be resolved ? */
      lua_pushstring(L, ipaddress);
      lua_srawget(L, -2);
    }
    if (ipaddress == NULL || agnL_checkboolean(L, -1) == isblacklist) {
      AGN_SOCKET oldsock;
      oldsock = sock;
      /* net.accept & net.connect: now close (new) socket */
      if (close_socket(sock) == 0) {  /* socket closing has been succesful ? */
        if (strcmp(procname, AGN_NET_CONNECT) == 0 && treedelete(socketattribs, sock) == 0)
          luaL_error(L, "Error in " LUA_QS ": could not delete data from socket administration table.", procname);
      } else {
        luaL_error(L, "Error in " LUA_QS ": could not close socket %d.", procname, sock);
      }
      sock = INVALID_SOCKET;
      if (ipaddress != NULL) {  /* IP address could be resolved ? */
        agn_poptoptwo(L);  /* pop true or false and list */
        luaL_error(L, "Error in " LUA_QS ": partner %sin %slist, closing socket %d.", procname,
          isblacklist ? "" : "not ", isblacklist ? "black" : "white", oldsock);
      } else {
        agn_poptop(L); /* pop black/whitelist */
        luaL_error(L, "Error in " LUA_QS ": could not resolve address %s for socket %d.", procname, address, sock);
      }
    } else
      agn_poptop(L);  /* pop true or false */
  }
  agn_poptop(L);  /* pop result of agnL_gettablefield */
  return 0;  /* does not leave anything on top of the stack */
}

/* with a valid socket handle, returns the family number of the protocol and the port number used
   from net.opensockets */
static STATUS *getsocketattribs (lua_State *L, AGN_SOCKET socket, const char *procname) {
  STATUS *s;
  s = treesearch(socketattribs, socket);
  if (s == NULL)
    luaL_error(L, "Error in " LUA_QS ": could not access socket status table.", procname);
  return s;
}


/* checks whether a socket handle is still valid by looking into net.opensockets or net.openserversockets
   (defined by argument *type) and either returns 1 or issues an error */
static int checksocket (lua_State *L, AGN_SOCKET socket, const char *procname) {
  if (treesearch(socketattribs, socket) == NULL)
    luaL_error(L, "Error in " LUA_QS ": invalid socket handle %d received.", procname, socket);
  return 1;
}


/* Blocking/non-blocking code taken from LuaSocket 2.0.2, Copyright 2004-2007 Diego Nehab, MIT licence;
   returns 0 on success and <> 0 on failure. */
#ifdef _WIN32
/* put socket into blocking mode */
static int setblocking (AGN_SOCKET s) {
  u_long argp = 0;
  return ioctlsocket(s, FIONBIO, &argp);
}

/* put socket into non-blocking mode */
static int setnonblocking (AGN_SOCKET s) {
  u_long argp = 1;
  return ioctlsocket(s, FIONBIO, &argp);
}
#else  /* UNIX */
/* put socket into blocking mode */
static int setblocking (AGN_SOCKET s) {
  int flags = fcntl(s, F_GETFL, 0);
  flags &= (~(O_NONBLOCK));
  return fcntl(s, F_SETFL, flags);
}

/* put socket into non-blocking mode */
static int setnonblocking (AGN_SOCKET s) {
  int flags = fcntl(s, F_GETFL, 0);
  flags |= O_NONBLOCK;
  return fcntl(s, F_SETFL, flags);
}
#endif


static AGN_SOCKET createsocket (lua_State *L, int family, int blocking, int *reusesuccess, const char *procname) {
  AGN_SOCKET sock;
  sock = socket(family, SOCK_STREAM, 0);
#ifndef _WIN32
  if (sock < 0) errornosock(L, "socket could not be created", procname);
#else
  if (sock == INVALID_SOCKET) errornosock(L, "socket could not be created", procname);
#endif
  if (!blocking) setnonblocking(sock);
#ifdef _WIN32
  *reusesuccess = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, "1", sizeof(so_reuseaddr));
#else
  *reusesuccess = setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &so_reuseaddr, sizeof(so_reuseaddr));
#endif
  return sock;
}


/* create address of server socket: family, IP address, port number */
static char *createaddress (lua_State *L, struct sockaddr_in *server,
  char *address, size_t port, AGN_FAMILY family, const char *procname) {
  struct in_addr addr;
  if (strcmp(address, "*") == 0) {
    server->sin_addr.s_addr = htonl(INADDR_ANY);  /* INADDR_ANY binds the socket to all local interfaces */
    server->sin_family = AF_UNSPEC;
  }
  else if ((inet_aton(address, &addr)) != 0) {  /* valid address ? */
    memcpy((char *)&server->sin_addr, &addr, sizeof(addr));
  } else {
    struct hostent *host_info;
    /* convert server name to IP address */
    host_info = gethostbyname(address);
    if (host_info == NULL) {
      errornosock(L, "unknown host", procname);
    }
    /* IP address of server */
    server->sin_addr = *(struct in_addr *) host_info->h_addr;
  }
  /* set network protocol */
  server->sin_family = family;
  /* port number */
  server->sin_port = htons(port);
  return inet_ntoa(server->sin_addr);
}


/*********************************************************************************************************************
*   Agena C library functions                                                                                        *
*********************************************************************************************************************/

/* Opens a (client) socket using the IPv4 protocol. If the optional first argument is set to false, the socket is set to non-blocking mode.

The return is the socket handle (a number), the default address 'localhost' and default port 1234, the protocol (a number) and a Boolean indicating whether the handle can be reused by the system after the socket has been closed.

The procedure is a binding to C's socket function. */
#define AGN_NET_OPEN "net.open"
static int net_open (lua_State *L) {
  int r;
  size_t port;
  char blocking;
  char *address;
  STATUS attrib;
  AGN_SOCKET sock;
  AGN_FAMILY family;
  address = ADDRESS;
  port = PORT;
  family = AF_INET;  /* by default use IPv4 protocol */
  blocking = agnL_optboolean(L, 1, 1);
  r = -1;
  /* create socket */
  sock = createsocket(L, family, blocking, &r, AGN_NET_OPEN);
  /* add new open socket in internal socket attribute tree */
  attrib.isserver = 0;
  attrib.address = address;
  attrib.port = port;
  attrib.family = family;
  attrib.blocking = blocking;
  attrib.connected = 0;
  attrib.shutdown = -MAX_INT;
  if (treeinsert(socketattribs, sock, &attrib) == 0)
    luaL_error(L, "Error in " LUA_QS ": could not assign socket to administration table.", AGN_NET_OPEN);
  lua_pushinteger(L, sock);    /* socket handle */
  lua_pushstring(L, address);  /* server address */
  lua_pushinteger(L, port);    /* server port */
  lua_pushinteger(L, family);  /* network protocol */
  lua_pushboolean(L, r == 0);  /* REUSE SOCKET option could be set ? */
  return 5;
}


/* Connects the client denoted by it socket handle (first argument, a number) to a server at the
   specified IP address (second argument, a string) and its port (third argument) so that data
   can be sent later. If address is missing, the address is set to 'localhost', if port is missing,
   port 1234 will be used.

   If the client socket is set to blocking mode, the function waits until
   the server responds; if the client socket is set to non-blocking mode, it immediately returns
   without waiting for a server response.

   The return is either true or an error is issued at failure.

   The procedure is a binding to C's connect function. */

/* for AGN_NET* definition see top of file */
static int net_connect (lua_State *L) {
  struct sockaddr_in server;
  char *address, *realaddress;
  size_t port;
  STATUS *s;
  AGN_SOCKET sock;
  sock = luaL_checkinteger(L, 1);
  checksocket(L, sock, AGN_NET_CONNECT);
  s = getsocketattribs(L, sock, AGN_NET_CONNECT);
  address = (char *)luaL_optstring(L, 2, ADDRESS);
  port = luaL_optnumber(L, 3, PORT);
  checkport(L, port, AGN_NET_CONNECT);
  /* create address of server socket: family, IP address, port number */
  realaddress = createaddress(L, &server, address, port, s->family, AGN_NET_CONNECT);
  checklist(L, sock, realaddress, "blacklist", AGN_NET_CONNECT);  /* check black and white lists, if they exist, 1.6.4 */
  checklist(L, sock, realaddress, "whitelist", AGN_NET_CONNECT);
  if (connect(sock, (struct sockaddr*)&server, sizeof(server)) < 0) {
    agn_neterror(L);
  }
  s->address = address; s->port = port; s->connected = 1; s->shutdown = MAX_INT;
  if (treeupdate(socketattribs, sock, s) != 0)
    luaL_error(L, "Error in " LUA_QS ": could not assign socket to administration table.", AGN_NET_CONNECT);
  lua_pushtrue(L);
  return 1;
}


/* Sets a socket to blocking or not blocking mode. The functions expects the socket handle (a number) as its
   first argument and the mode (a Boolean) as its second argument. If the second argument is true, the socket
   is set to blocking mode, else to non-blocking mode. The return is true on success and false otherwise. */
#define AGN_NET_BLOCK "net.block"
static int net_block (lua_State *L) {
  int r;
  AGN_SOCKET sock;
  STATUS *s;
  char blocking;
  sock = luaL_checkinteger(L, 1);
  checksocket(L, sock, AGN_NET_BLOCK);
  s = getsocketattribs(L, sock, AGN_NET_BLOCK);
  blocking = agnL_checkboolean(L, 2);
  if (blocking)
    r = setblocking(sock);
  else
    r = setnonblocking(sock);
  if (r == 0) {
    s->blocking = blocking;
    if (treeupdate(socketattribs, sock, s) != 0)
      luaL_error(L, "Error in " LUA_QS ": could not assign socket to administration table.", AGN_NET_BLOCK);
  }
  lua_pushboolean(L, r == 0);
  return 1;
}


/* Assigns an address specified to a given socket s and returns this address (a string) on success and issues
   an error otherwise. A port (a number) may be passed, as well, but you may need administrative rights.

   The procedure is a binding to C's bind function. */

#define AGN_NET_BIND "net.bind"
static int net_bind (lua_State *L) {
  struct sockaddr_in server;
  size_t port;
  char *address;
  AGN_SOCKET sock;
  STATUS *s;
  sock = luaL_checkinteger(L, 1);
  checksocket(L, sock, AGN_NET_BIND);
  s = getsocketattribs(L, sock, AGN_NET_BIND);
  address = (char *)luaL_optstring(L, 2, s->address);
  port = luaL_optinteger(L, 3, s->port);
  checkport(L, s->port, AGN_NET_BIND);
  createaddress(L, &server, address, port, s->family, AGN_NET_BIND);
  if (bind(sock, (struct sockaddr*)&server, sizeof(server)) < 0) { /*  bind socket to server:port */
    agn_neterror(L);
  }
  /* enter new open socket in internal socket attribute tree, set own address */
  address = inet_ntoa(server.sin_addr);
  s->address=address; s->port = port;
  if (treeupdate(socketattribs, sock, s) != 0)
    luaL_error(L, "Error in " LUA_QS ": could not assign socket to administration table.", AGN_NET_BIND);
  lua_pushstring(L, (const char *)address);
  lua_pushinteger(L, port);
  return 2;
}


/* Converts the given socket to a server socket, enabling it to accept connections. You may optionally pass an integer in the
   range [1, 1024] determining the length of the queue for pending connections. The return is either true or an
   error is issued if listening failed. You must first run this function before querying the input from the
   client with net.serve.

   The procedure is a binding to C's listen function. */
#define AGN_NET_LISTEN "net.listen"
static int net_listen (lua_State *L) {
  size_t queuelen;
  AGN_SOCKET sock;
  STATUS *s;
  sock = luaL_checkinteger(L, 1);
  checksocket(L, sock, AGN_NET_LISTEN);
  s = getsocketattribs(L, sock, AGN_NET_LISTEN);
  queuelen = luaL_optinteger(L, 2, 5);  /* length of the queue for pending connections */
  if (queuelen < 1 || queuelen > 1024)
    luaL_error(L, "Error in " LUA_QS ": queue length too small or large, must be in [1, 1024].", AGN_NET_LISTEN);
  if (listen(sock, queuelen) == -1) {
    agn_neterror(L);
  }
  s->isserver = 1;
  if (treeupdate(socketattribs, sock, s) != 0)
    luaL_error(L, "Error in " LUA_QS ": could not assign socket to administration table.", AGN_NET_LISTEN);
  lua_pushtrue(L);
  return 1;
}


/* Accepts a connection request from a client on the given server socket handle. If the server socket has been set to blocking
   mode, it waits until there is an incoming connection. Use net.block(<socket>, false) on the server socket to set it to
   non-blocking mode.

   The function returns a new socket handle for the data to be received lateron, and the address and port of the
   client socket. Please note that the new socket created by accept must be closed separately to avoid too many open
   sockets.

   The procedure is a binding to C's accept function. */

/* for AGN_NET* definition see top of file */
static int net_accept (lua_State *L) {
  struct sockaddr_in client;
  socklen_t len;
  STATUS attrib, *s;
  AGN_SOCKET sock, fd;
  const char *address;
  sock = luaL_checkinteger(L, 1);
  checksocket(L, sock, AGN_NET_ACCEPT);
  s = getsocketattribs(L, sock, AGN_NET_ACCEPT);
  len = (socklen_t)sizeof(client);
  fd = accept(sock, (struct sockaddr*)&client, &len);  /* cpu-friendly block as long as there is no connection */
#ifndef _WIN32
  if (fd < 0) {
    agn_neterror(L);
  }
#else
  if (fd == INVALID_SOCKET) {
    agn_neterror(L);
  }
#endif
  address = inet_ntoa(client.sin_addr);
  checklist(L, fd, address, "blacklist", AGN_NET_ACCEPT);  /* check black and white lists, if they exist, 1.6.4 */
  checklist(L, fd, address, "whitelist", AGN_NET_ACCEPT);
  attrib = *s; attrib.connected = 1; attrib.shutdown = MAX_INT;
  if (treeinsert(socketattribs, fd, &attrib) == 0)
    luaL_error(L, "Error in " LUA_QS ": could not assign socket to administration table.", AGN_NET_ACCEPT);
  lua_pushinteger(L, fd);                                /* return new socket handle */
  lua_pushstring(L, address);                            /* client address */
  lua_pushinteger(L, (unsigned)ntohs(client.sin_port));  /* client socket */
  return 3;
}


/* Allows a server socket to receive a string from a client. The function returns this string and its length (a number).

   The optional argument determines the maximum number of characters to be received.

   The procedure is a binding to C's recv function. */
#define AGN_NET_RECEIVE "net.receive"
static int net_receive (lua_State *L) {
  int recv_size, getall;
  int32_t n;
  lua_Number length, maxsize;
  char *buffer;
  AGN_SOCKET sock;
  agnL_gettablefield(L, "environ", "buffersize", AGN_NET_RECEIVE, 1);
  n = (lua_isnumber(L, -1)) ? agn_tonumber(L, -1) : LUAL_BUFFERSIZE;
  if (n < 1) n = LUAL_BUFFERSIZE;
  agn_poptop(L);  /* pop "buffersize" */
  sock = luaL_checkinteger(L, 1);
  checksocket(L, sock, AGN_NET_RECEIVE);
  getall = agnL_optboolean(L, 2, 0);
  maxsize = luaL_optnumber(L, 3, HUGE_VAL);
  buffer = (char *)agn_malloc(L, (n+1)*sizeof(char), AGN_NET_RECEIVE, NULL);  /* Agena 1.6.4 Valgrind; 1.9.1 */
  length = 0;
  lua_pushstring(L, "");
  while ((recv_size = recv(sock, buffer, n, 0)) != 0) {
    if (recv_size < 0) {
      xfree(buffer);
      agn_neterror2(L, "failure during receipt");  /* Agena 1.6.3 */
    }
    buffer[recv_size] = '\0';
    length += recv_size;
    if (length > maxsize) {
      xfree(buffer);
      agn_neterror2(L, "too many bytes received");
    }
    lua_pushstring(L, buffer);  /* lua_pushstring makes or reuses an internal copy of buffer, so buffer can be freed therafter */
    lua_concat(L, 2);
    if (!getall) break;
  }
  lua_pushinteger(L, length);
  xfree(buffer);
  return 2;
}


/* Sends a string (second argument) from the client denoted by its socket handle (first argument, a number) to
   a server.

   The return is the number of the characters actually sent. If the kernel decides not to send all the data in
   one chunk, the function might not send the complete string. If an optional third argument, true, is given,
   net.send, however, tries to make sure that the complete string has been sent when it returns.

   The procedure is an extended binding to C's send function. */
#define AGN_NET_SEND "net.send"
static int net_send (lua_State *L) {
  const char *str;
  char forceall;
  int len;
  AGN_SOCKET sock;
  STATUS *s;
  sock = luaL_checkinteger(L, 1);
  checksocket(L, sock, AGN_NET_SEND);
  str = luaL_checkstring(L, 2);
  len = strlen(str);
  forceall = agnL_optboolean(L, 3, 0);
  s = getsocketattribs(L, sock, AGN_NET_SEND);
  if (!s->connected) {  /* Linux would crash otherwise */
    agn_neterrorfail(L, "socket not connected");
  }
  /* send string including null terminator to server */
  if (!forceall) {
    int sendlen;
    sendlen = send(sock, str, len, 0);
    if (sendlen == -1) {
      s->connected = 0;
      if (treeupdate(socketattribs, sock, s) != 0)
        luaL_error(L, "Error in " LUA_QS ": could not change socket status in administration table.", AGN_NET_CONNECT);
      agn_neterrorfail(L, "socket not connected");
    }
    else if (sendlen != len) {
      lua_pushfalse(L);
      lua_pushstring(L, "transfer size mismatch");
      lua_pushnumber(L, sendlen);
      return 3;
    }
    len = sendlen;
  } else {  /* if less data has actually been sent, try sending the rest not yet transmitted,
    taken from `Beej's Guide to Network Programming, Using Internet Sockets`, Version 3.0.14,
    September 8, 2009, Chapter 7.3 */
    int total = 0;         /* how many bytes we've sent */
    int bytesleft = len;   /* how many we have left to send */
    int n;
    while (total < len) {
      n = send(sock, str+total, bytesleft, 0);
      if (n == -1) {
        agn_neterror2(L, "sending data failed");
      }
      total += n;
      bytesleft -= n;
    }
    len = total;  /* return number actually sent here */
  }
  lua_pushinteger(L, len);
  return 1;
}


/* returns the largest descriptor of all open sockets */
static AGN_SOCKET treesetfds (lua_State *L, NODE *k, fd_set *x, AGN_SOCKET *max, const char *procname) {
  AGN_SOCKET s;
  if (k == NULL) return *max;
  s = k->index;
  FD_SET(s, x);
  if (s == INVALID_SOCKET || *max < s)
    *max = s;
  treesetfds(L, k->left, x, max, procname);
  treesetfds(L, k->right, x, max, procname);
  return *max;
}


#define AGN_NET_SURVEY "net.survey"
static int net_survey (lua_State *L) {
  int i, cread, cwrite, cexc, istimegiven, result, mode, throw, offset, *field;
  size_t size;
  const char *option;
  AGN_SOCKET max;
  struct timeval tv;
  fd_set read_fds, write_fds, exc_fds;  /* file descriptor lists for select() */
  max = istimegiven = mode = offset = 0;
  size = 0;
  field = NULL;
  /* this following should detect listeners, as well */
  if (socketattribs->size > FD_SETSIZE)
    luaL_error(L, "Error in " LUA_QS ": too many open sockets, %d allowed.", AGN_NET_SURVEY, FD_SETSIZE);
  FD_ZERO(&read_fds);
  FD_ZERO(&write_fds);
  FD_ZERO(&exc_fds);
  if (lua_gettop(L) > 0 && lua_isseq(L, 1)) {
    AGN_SOCKET val;
    offset = 1;
    size = agn_seqsize(L, 1);
    field = agn_malloc(L, size * sizeof(AGN_SOCKET), AGN_NET_SURVEY, NULL);
    for (i = 0; i < size; i++) {
      val = (AGN_SOCKET)lua_seqgetinumber(L, 1, i+1);
      if (val == HUGE_VAL || val < 0 || val >= FD_SETSIZE) {
        xfree(field);
        luaL_error(L, "Error in " LUA_QS ": invalid socket handle encountered.", AGN_NET_SURVEY);
      }
      *field = val;
      field += sizeof(AGN_SOCKET);
      if (val > max) max = val;
    }
    field -= size * sizeof(AGN_SOCKET);  /* reset pointer to beginning of array */
  } else {  /* traverse socket tree to look for largest file descriptor, also assign fd_set */
    max = treesetfds(L, socketattribs->root, &read_fds, &max, AGN_NET_SURVEY);
  }
  write_fds = exc_fds = read_fds;  /* also assign sockets to be scanned to write an exception fd_sets */
  if (lua_gettop(L) > offset && lua_isnumber(L, offset+1)) {
    lua_Number d = agn_tonumber(L, offset+1);
    if (d <= 0) {
      if (offset) xfree(field);
      luaL_error(L, "Error in " LUA_QS ": timeout must be positive, received %f seconds.", AGN_NET_SURVEY, d);  /* 2.1 RC 1 fix */
    }
    if (d != HUGE_VAL) {  /* 1.8.16 */
      tv.tv_sec = (int)(d);
      tv.tv_usec = ((int)(d) - trunc(d))*100000;
      istimegiven = 1;
    }
  }
  option = luaL_optstring(L, offset+2, "all");
  if (strcmp(option, "all") == 0)
    mode = 8;
  else if (strcmp(option, "read") == 0)
    mode = 1;
  else if (strcmp(option, "write") == 0)
    mode = 2;
  else if (strcmp(option, "except") == 0)
    mode = 4;
  else {
    if (offset) xfree(field);
    luaL_error(L, "Error in " LUA_QS ": unknown option `%s`.", AGN_NET_SURVEY, option);
  }
  throw = agnL_optboolean(L, offset+3, 1);
  /*  select returns 0 at timeout, 1 if input available, -1 at error. */
  if (offset) {  /* set fd_sets, 1.8.16 */
    for (i=0; i < size; i++) {
      FD_SET(*field, &read_fds);
      FD_SET(*field, &write_fds);
      FD_SET(*field, &exc_fds);
      field += sizeof(AGN_SOCKET);
    }
    field -= size * sizeof(AGN_SOCKET);  /* reset pointer to beginning of array */
    xfree(field);
  }
  if ((result = select(max+1, &read_fds, &write_fds, &exc_fds, (istimegiven) ? &tv : NULL)) == -1 && throw) {
    agn_neterror(L);
  }
  cread = cwrite = cexc = 0;
  if (mode == 8) {  /* scan all socket types ?  16.01.2013 */
    for (i=0; i < 3; i++) agn_createseq(L, 0);
    for (i=0; i <= max; i++) {
      if (FD_ISSET(i, &read_fds)) {
        lua_seqsetinumber(L, -3, ++cread, i);
      }
      if (FD_ISSET(i, &write_fds)) {
        lua_seqsetinumber(L, -2, ++cwrite, i);
      }
      if (FD_ISSET(i, &exc_fds)) {
        lua_seqsetinumber(L, -1, ++cexc, i);
      }
    }
  } else {  /* scan individual socket types ?  16.01.2013 */
    agn_createseq(L, 0);
    for (i=0; i <= max; i++) {
      if ((mode & 1) && FD_ISSET(i, &read_fds)) {
        lua_seqsetinumber(L, -1, ++cread, i);
      }
      else if ((mode & 2) && FD_ISSET(i, &write_fds)) {
        lua_seqsetinumber(L, -1, ++cwrite, i);
      }
      else if (FD_ISSET(i, &exc_fds)) {
        lua_seqsetinumber(L, -1, ++cexc, i);
      }
    }
  }
  agn_pushboolean(L, result);  /* 16.01.2013, true = input is available, false = timeout, fail in case of an exception */
  return ((mode == 8) ? 4 : 2); /* 16.01.2013 */
}


/* Terminates the server or client denoted by its socket handle and returns true on success, and false otherwise.

   The procedure is a binding to C's close function (closesocket in Windows). */
#define AGN_NET_CLOSE "net.close"
static int net_close (lua_State *L) {
  size_t i, top;
  AGN_SOCKET sock;
  top = lua_gettop(L);
  for (i=1; i <= top; i++) {
    sock = luaL_checkinteger(L, i);
    checksocket(L, sock, AGN_NET_CLOSE);
    if (close_socket(sock) == 0) {  /* socket closing has been succesful ? */
      if (treedelete(socketattribs, sock) == 0)
        luaL_error(L, "Error in " LUA_QS ": could not delete data from socket administration table.", AGN_NET_CLOSE);
      sock = INVALID_SOCKET;
    } else {
      int flag = (i == top - 1);
      lua_pushfalse(L);
      lua_pushstring(L, "could not close socket ");
      lua_pushstring(L, agn_tostring(L, i));
      if (flag) lua_pushstring(L, ", aborting closing remaining ones");
      lua_concat(L, (flag) ? 3 : 2);
      return (flag) ? 3 : 2;
    }
  }
  lua_pushtrue(L);
  return 1;
}


#ifdef _WIN32
#define AGN_NET_OPENWINSOCK "net.openwinsock"
static int net_openwinsock (lua_State *L) {
  WORD wVersionRequested;
  WSADATA wsaData;
  /* initialise TCP for Windows (Winsock) */
  wVersionRequested = MAKEWORD(1, 1);
  if (WSAStartup (wVersionRequested, &wsaData) != 0) {
    if (lua_gettop(L) == 0)
      errornosock(L, "initialisation of Winsock failed.", AGN_NET_OPENWINSOCK);
    else {
      lua_pushfail(L);
      lua_pushstring(L, wstrerror(WSAGetLastError()));
      return 2;
    }
  }
  lua_pushtrue(L);
  return 1;
}

#define AGN_NET_CLOSEWINSOCK "net.closewinsock"
static int net_closewinsock (lua_State *L) {
  /* Cleanup Winsock */
  if (WSACleanup() != 0) {
    if (lua_gettop(L) == 0)
      errornosock(L, "Winsock cleanup failed", AGN_NET_CLOSEWINSOCK);
    else {
      lua_pushfail(L);
      lua_pushstring(L, wstrerror(WSAGetLastError()));
      return 2;
    }
  }
  lua_pushtrue(L);
  return 1;
}
#endif


/* returns the address and port of the server that the client is connected to. */
#define AGN_NET_REMOTEADDRESS "net.remoteaddress"
static int net_remoteaddress (lua_State *L) {
  struct sockaddr_in addr;
  socklen_t length;
  AGN_SOCKET sock;
  sock = luaL_checkinteger(L, 1);
  checksocket(L, sock, AGN_NET_REMOTEADDRESS);
  length = sizeof addr;
  if (getpeername(sock, (struct sockaddr*)&addr, &length) != 0) {
    agn_neterror2(L, "could not get address");
  }
  lua_pushstring(L, inet_ntoa(addr.sin_addr));
  lua_pushinteger(L, (unsigned)ntohs(addr.sin_port));
  return 2;
}


/* returns the address and port of the socket denoted by its handle. */
#define AGN_NET_ADDRESS "net.address"
static int net_address (lua_State *L) {
  struct sockaddr_in addr;
  socklen_t length;
  AGN_SOCKET sock;
  sock = luaL_checkinteger(L, 1);
  checksocket(L, sock, AGN_NET_ADDRESS);
  length = sizeof (addr);
  if (getsockname(sock, (struct sockaddr*)&addr, &length) != 0) {
    agn_neterror2(L, "could not get address");
  }
  lua_pushstring(L, inet_ntoa(addr.sin_addr));
  lua_pushinteger(L, (unsigned)ntohs(addr.sin_port));
  return 2;
}


/* returns the address and port of the socket denoted by its handle. */
#define AGN_NET_SHUTDOWN "net.shutdown"
static int net_shutdown (lua_State *L) {
  const char *mode;
  int r, m;
  AGN_SOCKET sock;
  STATUS *s;
  sock = luaL_checkinteger(L, 1);
  checksocket(L, sock, AGN_NET_SHUTDOWN);
  s = getsocketattribs(L, sock, AGN_NET_SHUTDOWN);
  if (!s->connected) {
    agn_neterrorfail(L, "socket not connected");
  }
  mode = luaL_checkstring(L, 2);
  if (strcmp(mode, "read") == 0) {
    r = shutdown(sock, SHUTDOWN_RD); m = SHUTDOWN_RD;
  } else if (strcmp(mode, "write") == 0) {
    r = shutdown(sock, SHUTDOWN_WR); m = SHUTDOWN_WR;
  } else if (strcmp(mode, "readwrite") == 0) {
    r = shutdown(sock, SHUTDOWN_RDWR); m = SHUTDOWN_RDWR;
  } else {
    m = r = -MAX_INT;
    luaL_error(L, "Error in " LUA_QS ": unknown shutdown mode `%s`." AGN_NET_SHUTDOWN, mode);
  }
  if (r == 0) {
    s->shutdown = m;
    if (treeupdate(socketattribs, sock, s) != 0) {
      luaL_error(L, "Error in " LUA_QS ": could not assign socket to administration table.", AGN_NET_SHUTDOWN);
    }
  }
  lua_pushboolean(L, r == 0);
  return 1;
}


static int isipv4 (const char *_str) {
  size_t ndots;
  const char *str;
  ndots = 0;
  str = _str;  /* leave argument untouched */
  while (*str) {
    if (*str == '.') ndots++;
    if (((*str < '0' || *str > '9') && *str != '.') || ndots > 3)
      return 0;
    str++;
  }
  return (ndots == 3);
}

/* returns a table containing information on a given host, slimmed 1.6.4, patched 1.6.5, extended 1.8.16 */

/* remark from http://msdn.microsoft.com/en-us/library/windows/desktop/ms738521%28v=vs.85%29.aspx:
   "An application should not try to release the memory used by the returned hostent structure. The application must
   never attempt to modify this structure or to free any of its components." */
#define AGN_NET_LOOKUP "net.lookup"
static int net_lookup (lua_State *L) {
  int x;
  const char *entry, *ip;
  struct hostent *lu;
  if (lua_gettop(L) == 0)
    ip = "localhost";
  else
    ip = agn_checkstring(L, 1);
  if (isipv4(ip)) {  /* numeric IP ? */
    unsigned int addr;
    addr = inet_addr(ip);
    lu = gethostbyaddr((char *)&addr, sizeof(unsigned int), AF_INET);
  } else
    lu = gethostbyname(ip);
  if (!lu) {  /* Report lookup failure */
    errornosock(L, "lookup failure", AGN_NET_LOOKUP);
  }
  lua_createtable(L, 0, 4);
  lua_rawsetstringstring(L, -1, "official", lu->h_name);
  lua_pushstring(L, "alias");
  lua_newtable(L);
  for (x=0; lu->h_aliases[x]; ++x) {
    entry = lu->h_aliases[x];
    lua_rawsetilstring(L, -1, x+1, entry, strlen(entry));
  }
  lua_rawset(L, -3);
  lua_rawsetstringstring(L, -1, "type", lu->h_addrtype == AF_INET ? "IPv4" : "unknown");
  if (lu->h_addrtype == AF_INET) {
    lua_pushstring(L, "networkaddress");
    lua_newtable(L);
    for (x=0; lu->h_addr_list[x]; ++x ) {
      entry = inet_ntoa( *(struct in_addr *) lu->h_addr_list[x]);
      lua_rawsetilstring(L, -1, x+1, entry, strlen(entry));
    }
    lua_rawset(L, -3);
/* what is Windows' equivalent to inet_lnaof, inet_makeaddr ? */
/*
#ifndef _WIN32
    lua_pushstring(L, "localaddress");
    lua_newtable(L);
    for (x=0; lu->h_addr_list[x]; ++x ) {
      entry = ntohl(inet_lnaof( *(struct in_addr *) lu->h_addr_list[x]);
      lua_rawsetilstring(L, -1, x+1, entry, strlen(entry));
    }
    lua_rawset(L, -3);
#endif
*/
  }
  return 1;
}


static void treetraverse (lua_State *L, NODE *k) {
  STATUS *s;
  if (k == NULL) return;
  s = k->status;
  lua_pushnumber(L, k->index);
  lua_createtable(L, 6, 0);
  lua_rawsetstringboolean(L, -1, "server", s->isserver);
  lua_rawsetstringstring(L, -1, "address", s->address);
  lua_rawsetstringnumber(L, -1, "port", s->port);
  lua_rawsetstringnumber(L, -1, "protocol", s->family);
  lua_rawsetstringboolean(L, -1, "blocking", s->blocking);
  lua_rawsetstringboolean(L, -1, "connected", s->connected);
  switch (s->shutdown) {
     case SHUTDOWN_RD: lua_rawsetstringstring(L, -1, "mode", "write"); break;
     case SHUTDOWN_WR: lua_rawsetstringstring(L, -1, "mode", "read"); break;
     case SHUTDOWN_RDWR: lua_rawsetstringstring(L, -1, "mode", "shutdown"); break;
     case -MAX_INT: lua_rawsetstringstring(L, -1, "mode", "none"); break;
     case MAX_INT: lua_rawsetstringstring(L, -1, "mode", "readwrite"); break;
     default: luaL_error(L, "Error: invalid shutdown mode.");
  }
  lua_rawset(L, -3);
  treetraverse(L, k->left);
  treetraverse(L, k->right);
}


/* Returns all open sockets along with their respective attributes. The return is a table with its keys
   the open sockets, and their entries tables containing information whether the socket is
   a client (false) or server socket (true), their own address (a string), their own port (a number),
   the protocol used (a number), and whether the socket works in blocking (true) or non-blocking mode
   (false), in this order. */
static int net_opensockets (lua_State *L) {
  lua_createtable(L, 0, socketattribs->size);
  treetraverse(L, socketattribs->root);
  return 1;
}


/* destructor for open sockets admin table, use only for cleanup, 1.6.6 */
static void nodepurge (TREE *baum, NODE *t, int flag) {
  if (t) {
    nodepurge(baum, t->left, 0);
    nodepurge(baum, t->right, 0);
    if (close_socket(t->index) == 0) {  /* socket closing has been succesful ? 1.8.16 */
      t->index = INVALID_SOCKET;
    } else {  /* When terminating a process/programme, Windows seems to call WSACleanUp before running cleanup, so explicitly
       closing a socket in this function will always be unsuccessful. */
#ifndef _WIN32
      if (flag) fprintf(stderr, "\nError in `net` package during clean-up:\n");
      fprintf(stderr, "Could not close socket %d.\n", t->index);
      fflush(stderr);
#endif
    }
    free(t->status);
    xfree(t);
  }
}

static void treepurge (TREE *baum) {
  nodepurge(baum, baum->root, 1);
  xfree(baum);
}

void cleanup (void) {  /* 1.6.5, clanup is not called when pressing CTRL+C */
  treepurge(socketattribs);
#ifdef _WIN32
  WSACleanup();
#endif
}


static const luaL_Reg netlib[] = {
  {"accept", net_accept},                 /* added on April 12, 2012 */
  {"address", net_address},               /* added on April 09, 2012 */
  {"bind", net_bind},                     /* added on April 12, 2012 */
  {"block", net_block},                   /* added on April 14, 2012 */
  {"close", net_close},                   /* added on April 05, 2012 */
  {"connect", net_connect},               /* added on April 05, 2012 */
  {"listen", net_listen},                 /* added on April 05, 2012 */
  {"lookup", net_lookup},                 /* added on April 05, 2012 */
  {"open", net_open},                     /* added on April 05, 2012 */
  {"opensockets", net_opensockets},       /* added on April 12, 2012 */
  {"receive", net_receive},               /* added on April 05, 2012 */
  {"remoteaddress", net_remoteaddress},   /* added on April 05, 2012 */
  {"send", net_send},                     /* added on April 05, 2012 */
  {"shutdown", net_shutdown},             /* added on May 07, 2012 */
  {"survey", net_survey},                 /* added on April 22, 2012 */
#ifdef _WIN32
  {"closewinsock", net_closewinsock},     /* added on April 17, 2012 */
  {"openwinsock", net_openwinsock},       /* added on April 17, 2012 */
#endif
  {NULL, NULL}
};


static void createmetatable (lua_State *L) {
  lua_createtable(L, 0, 1);  /* create metatable for strings */
  lua_pushliteral(L, "");  /* dummy string */
  lua_pushvalue(L, -2);
  lua_setmetatable(L, -2);  /* set string metatable */
  agn_poptop(L);  /* pop dummy string */
  lua_pushvalue(L, -2);  /* string library... */
  lua_setfield(L, -2, "__index");  /* ...is the __index metamethod */
  agn_poptop(L);  /* pop metatable */
}

static int nopened = 0;  /* Agena 1.7.9a */

/*
** Open net library
*/
LUALIB_API int luaopen_net (lua_State *L) {
  nopened++;  /* Agena 1.7.9a */
  if (nopened > 1)  /* avoid segmentation faults when readlib'ing the net library more than once, at exit/bye of Agena, 1.7.9a */
    return 0;  /* immediate exit, just not calling signal if nopened > 1 does not help, XXX could be improved. */
#ifdef _WIN32
  WORD wVersionRequested;
  WSADATA wsaData;
  /* initialise TCP for Windows (Winsock) */
  wVersionRequested = MAKEWORD(1, 1);
  if (WSAStartup(wVersionRequested, &wsaData) != 0) {
    int err = WSAGetLastError();
    luaL_error(L, "Error in " LUA_QS " package: initialisation of Winsock failed:\n(%d) %s", "net", err, wstrerror(err));
  }
#else
  /* 1.6.4, some of the functions like `net.send` would crash if a socket has been disconnected */
  signal(SIGPIPE, SIG_IGN);  /* in UNIX, ignore signals and process the return values of the respective C network functions;
    if not called, Agena might crash, e.g. if send() tries to write data to a socket that discontinued a connection. */
#endif
  socketattribs = treeinit();
  if (socketattribs == NULL)
    luaL_error(L, "Error when initialising `net` package: could not create socket administration table.");
  luaL_register(L, AGENA_NETLIBNAME, netlib);
  lua_createtable(L, 0, 2);
  lua_rawsetstringnumber(L, -1, "maxnsockets", FD_SETSIZE);  /* in Winsock2, FD_SETSIZE is not the number of
     _bits_ assigned to store the largest server socket handle, but the maximum number of open sockets;
     however, Windows does not query FD_SETSIZE when opening new sockets. XXX -> Solaris ??? */
  lua_createtable(L, 1, 0);
  lua_rawsetistring(L, -1, AF_INET, "IPv4");
  lua_setfield(L, -2, "protocols");
  lua_setfield(L, -2, "admin");  /* table for information on internal network administration */
  createmetatable(L);
  atexit(cleanup);  /* 1.6.5 */
  return 1;
}

/* ====================================================================== */

#endif /* UNIX, Apple, Windows */

