/*
** $Id: lvm.c,v 2.63 15.03.2009 12:40:39 roberto Exp $
** Lua virtual machine
** See Copyright Notice in agena.h
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef PROPCMPLX
#include <complex.h>
#endif

#define lvm_c
#define LUA_CORE

#include "agena.h"

#include "ldebug.h"
#include "ldo.h"
#include "lfunc.h"
#include "lgc.h"
#include "lobject.h"
#include "lopcodes.h"
#include "lstate.h"
#include "lstring.h"
#include "ltable.h"
#include "ltm.h"
#include "lvm.h"
#include "lcode.h"    /* for typedef enum UnOpr */
#include "agnhlps.h"
#include "lset.h"
#include "lseq.h"
#include "lpair.h"
#include "lcomplex.h"
#include "llimits.h"
#include "lstrlib.h"  /* for lmemfind and match */
#include "agncmpt.h"
#include "agnconf.h"  /* for _SUNSOLARIS definition */

/* limit for table tag-method chains (to avoid loops) */
#define MAXTAGLOOP   100

#define uchar(c)        ((unsigned char)(c))


/* substitution table: upper to lower case */

static unsigned char lowercase[256] = {
   0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,  /* 0 .. 7*/
   0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,  /* 8 .. 15 */
   0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,  /* 16 .. 23 */
   0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,  /* 24 .. 31 */
   0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,  /* 32 .. 39 */
   0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,  /* 40 .. 47 */
   0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,  /* 48 .. 55 */
   0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,  /* 56 .. 63 */
   0x40, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,  /* 64 .. 71 */
/* 064@  065A  066B  067C  068D  069E  070F  071G */
   0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,  /* 72 .. 79 */
/* 072H  073I  074J  075K  076L  077M  078N  079O */
   0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,  /* 80 .. 87 */
/* 080P  081Q  082R  083S  084T  085U  086V  087W */
   0x78, 0x79, 0x7a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,  /* 88 .. 95 */
/* 088X  089Y  090Z  091   092   093   094   095  */
   0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,  /* 96 .. 103 */
/* 096`  097a  098b  099c  100d  101e  102f  103g */
   0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,  /* 104 .. 111 */
/* 104h  105i  106j  107k  108l  109m  110n  111o */
   0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,  /* 112 .. 119 */
/* 112p  113q  114r  115s  116t  117u  118v  119w */
   0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,  /* 120 .. 127 */
/* 120x  121y  122z  123{  124|  125}  126~  127 */
   0x87, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,  /* 128 .. 135 */
/* 128  129  130  131  132  133  134  135 */
   0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x84, 0x86,  /* 136 .. 143 */
/* 136  137  138  139  140  141  142  143 */
   0x82, 0x91, 0x91, 0x93, 0x94, 0x95, 0x96, 0x97,  /* 144 .. 151 */
/* 144  145  146  147  148  149  150  151 */
   0x98, 0x94, 0x81, 0x9b, 0x9c, 0x9b, 0x9e, 0x9f,  /* 152 .. 159 */
/* 152  153  154  155  156  157  158  159 */
   0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa4, 0xa6, 0xa7,  /* 160 .. 167 */
/* 160  161  162  163  164  165  166  167 */
   0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,  /* 168 .. 175 */
/* 168  169  170  171  172  173  174  175 */
   0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xa0, 0x83, 0x85,  /* 176 .. 183 */
/* 176  177  178  179  180  181  182  183 */
   0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,  /* 184 .. 191 */
/* 184  185  186  187+  188+  189  190  191+ */
   0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc6,  /* 192 .. 199 */
/* 192+  193-  194-  195+  196-  197+  198  199 */
   0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,  /* 200 .. 207 */
/* 200+  201+  202-  203-  204  205-  206+  207 */
   0xd0, 0xd0, 0x88, 0x89, 0x8a, 0xd5, 0xa1, 0x8c,  /* 208 .. 215 */
/* 208  209  210  211  212  213i  214  215 */
   0x8b, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0x8d, 0xdf,  /* 216 .. 223 */
/* 216  217+  218+  219  220_  221  222  223 */
   0xa2, 0xe1, 0x93, 0x95, 0xe4, 0xe4, 0xe6, 0xe7,  /* 224 .. 231 */
/* 224  225  226  227  228  229  230  231 */
   0xe7, 0xa3, 0x96, 0x97, 0xec, 0xec, 0xee, 0xef,  /* 232 .. 239 */
/* 232  233  234  235  236  237  238  239 */
   0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,  /* 240 .. 247 */
/* 240  241  242=  243  244  245  246  247 */
   0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff   /* 248 .. 255 */
/* 248  249  250  251  252  253  254  255 */
};


/* lower to upper case */

static unsigned char uppercase[256] = {
   0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,  /* 0 .. 7*/
   0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,  /* 8 .. 15 */
   0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,  /* 16 .. 23 */
   0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,  /* 24 .. 31 */
   0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,  /* 32 .. 39 */
   0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,  /* 40 .. 47 */
   0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,  /* 48 .. 55 */
   0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,  /* 56 .. 63 */
   0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,  /* 64 .. 71 */
/* 064@  065A  066B  067C  068D  069E  070F  071G */
   0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,  /* 72 .. 79 */
/* 072H  073I  074J  075K  076L  077M  078N  079O */
   0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,  /* 80 .. 87 */
/* 080P  081Q  082R  083S  084T  085U  086V  087W */
   0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,  /* 88 .. 95 */
/* 088X  089Y  090Z  091   092   093   094   095  */
   0x60, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,  /* 96 .. 103 */
/* 096`  097a  098b  099c  100d  101e  102f  103g */
   0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,  /* 104 .. 111 */
/* 104h  105i  106j  107k  108l  109m  110n  111o */
   0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,  /* 112 .. 119 */
/* 112p  113q  114r  115s  116t  117u  118v  119w */
   0x58, 0x59, 0x5a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,  /* 120 .. 127 */
/* 120x  121y  122z  123{  124|  125}  126~  127 */
   0x80, 0x9a, 0x90, 0xb6, 0x8e, 0xb7, 0x8f, 0x80,  /* 128 .. 135, ok */
/* 128  129  130  131  132  133  134  135 */
   0xd2, 0xd3, 0xd4, 0xd8, 0xd7, 0xde, 0x8e, 0x8f,  /* 136 .. 143 ok */
/* 136  137  138  139  140  141  142  143 */
   0x90, 0x92, 0x92, 0xe2, 0x99, 0xe3, 0xea, 0xeb,  /* 144 .. 151 ok */
/* 144  145  146  147  148  149  150  151 */
   0x98, 0x99, 0x9a, 0x9d, 0x9c, 0x9d, 0x9e, 0x9f,  /* 152 .. 159 ok */
/* 152? 153  154  155  156  157  158  159 */
   0xb5, 0xd6, 0xe0, 0xe9, 0xa5, 0xa5, 0xa6, 0xa7,  /* 160 .. 167 ok */
/* 160  161  162  163  164  165  166  167 */
   0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,  /* 168 .. 175 */
/* 168  169  170  171  172  173  174  175 */
   0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,  /* 176 .. 183 */
/* 176  177  178  179  180  181  182  183 */
   0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,  /* 184 .. 191 */
/* 184  185  186  187+  188+  189  190  191+ */
   0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc7, 0xc7,  /* 192 .. 199 ok */
/* 192+  193-  194-  195+  196-  197+  198  199 */
   0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,  /* 200 .. 207 ok */
/* 200+  201+  202-  203-  204  205-  206+  207 */
   0xd1, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,  /* 208 .. 215 ok */
/* 208  209  210  211  212  213i  214  215 */
   0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,  /* 216 .. 223 ok */
/* 216  217+  218+  219  220_  221  222  223 */
   0xe0, 0xe1, 0xe2, 0xe3, 0xe5, 0xe5, 0xe6, 0xe8,  /* 224 .. 231 ok */
/* 224  225  226  227  228  229  230  231 */
   0xe8, 0xe9, 0xea, 0xeb, 0xed, 0xed, 0xee, 0xef,  /* 232 .. 239 ok */
/* 232  233  234  235  236  237  238  239 */
   0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,  /* 240 .. 247 */
/* 240  241  242=  243  244  245  246  247 */
   0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff   /* 248 .. 255 */
/* 248  249  250  251  252  253  254  255 */
};


#define checkstack(L, text, fn) { \
  if ((L->top - L->ci->top) > LUA_MINSTACK) \
    luaG_runerror(L, text LUA_QS ": stack overflow.", fn); \
}


const TValue *luaV_tonumber (const TValue *obj, TValue *n) {
  lua_Number num;
  if (ttisnumber(obj)) return obj;
  if (ttisstring(obj)) {
    if (luaO_str2d(svalue(obj), &num)) {
      setnvalue(n, num);
      return n;
    } else if (strcmp(svalue(obj), "undefined") == 0) {
      setnvalue(n, AGN_NAN);
      return n;
    } else if (strcmp(svalue(obj), "infinity") == 0) {
      setnvalue(n, HUGE_VAL);
      return n;
    } else if (strcmp(svalue(obj), "-infinity") == 0) {
      setnvalue(n, -HUGE_VAL);
      return n;
    }
  }
  return NULL;
}


const TValue *luaV_arithmoperand (const TValue *obj, TValue *n) {
  if (ttisnumber(obj) || ttiscomplex(obj))
    return obj;
  else
    return NULL;
}


const TValue *luaV_tocomplex (const TValue *obj, TValue *n) {  /* 0.26.1 */
#ifndef PROPCMPLX
  agn_Complex num;
#else
  lua_Number num[2];
#endif
  if (ttiscomplex(obj)) return obj;
  if (ttisstring(obj)) {
#ifndef PROPCMPLX
    if (luaO_str2c(svalue(obj), &num)) {
#else
    if (luaO_str2c(svalue(obj), num)) {
#endif
      setcvalue(n, num);
      return n;
    }
  }
  return NULL;
}


int luaV_tostring (lua_State *L, StkId obj) {
  if (!ttisnumber(obj)) {
    return 0;
  }
  else {
    char s[LUAI_MAXNUMBER2STR];
    lua_Number n = nvalue(obj);
    if (n == HUGE_VAL)            /* added 0.9.2 */
      strcpy(s, "infinity\0");
    else if ((-n) == HUGE_VAL)
      strcpy(s, "-infinity\0");
    else if (isnan(n) || isnan(-n))
      strcpy(s, "undefined\0");
    else {
      lua_number2str(s, n);
    }
    setsvalue2s(L, obj, luaS_new(L, s));
    return 1;
  }
}


static void traceexec (lua_State *L, const Instruction *pc) {
  lu_byte mask = L->hookmask;
  const Instruction *oldpc = L->savedpc;
  L->savedpc = pc;
  if ((mask & LUA_MASKCOUNT) && L->hookcount == 0) {  /* 5.1.3 patch */
    resethookcount(L);
    luaD_callhook(L, LUA_HOOKCOUNT, -1);
  }
  if (mask & LUA_MASKLINE) {
    Proto *p = ci_func(L->ci)->l.p;
    int npc = pcRel(pc, p);
    int newline = getLine(p, npc);
    /* call linehook when enter a new function, when jump back (loop),
       or when enter a new line */
    if (npc == 0 || pc <= oldpc || newline != getLine(p, pcRel(oldpc, p)))
      luaD_callhook(L, LUA_HOOKLINE, newline);
  }
}


static void callTMres (lua_State *L, StkId res, const TValue *f,
                        const TValue *p1, const TValue *p2) {
  ptrdiff_t result = savestack(L, res);
  setobj2s(L, L->top, f);  /* push function */
  setobj2s(L, L->top+1, p1);  /* 1st argument */
  setobj2s(L, L->top+2, p2);  /* 2nd argument */
  luaD_checkstack(L, 3);
  L->top += 3;
  luaD_call(L, L->top - 3, 1);
  res = restorestack(L, result);
  L->top--;
  setobjs2s(L, res, L->top);
}


static void callTM (lua_State *L, const TValue *f, const TValue *p1,
                    const TValue *p2, const TValue *p3) {
  setobj2s(L, L->top, f);  /* push function */
  setobj2s(L, L->top+1, p1);  /* 1st argument */
  setobj2s(L, L->top+2, p2);  /* 2nd argument */
  setobj2s(L, L->top+3, p3);  /* 3th argument */
  luaD_checkstack(L, 4);
  L->top += 4;
  luaD_call(L, L->top - 4, 0);
}


void luaV_gettable (lua_State *L, const TValue *t, TValue *key, StkId val) {
  int loop;
  for (loop = 0; loop < MAXTAGLOOP; loop++) {
    const TValue *tm;
    if (ttistable(t)) {  /* `t' is a table ? */
      Table *h = hvalue(t);
      const TValue *res = luaH_get(h, key); /* do a primitive get */
      if (ttisnotnil(res) ||  /* result is no nil? */
          (tm = fasttm(L, h->metatable, TM_INDEX)) == NULL) { /* or no TM? */
        setobj2s(L, val, res);
        return;
      }
    }
    else if (ttisuset(t)) {  /* `t' is a set ? */
      UltraSet *h = usvalue(t);
      if (agnUS_get(h, key)) {
        setbvalue(val, 1);
        return;
      }
      else if ((tm = fasttm(L, h->metatable, TM_INDEX)) == NULL) {
        setbvalue(val, 0);
        return;
      }
    }
    else if (ttisseq(t) && ttisnumber(key)) {
      const TValue *res;
      size_t pos;
      Seq *h;
      h = seqvalue(t);
      pos = posrelat(nvalue(key), h->size);  /* 1.11.8 */
      res = agnSeq_geti(h, pos);
      if (res == NULL) {
        if ((tm = fasttm(L, h->metatable, TM_INDEX)) == NULL) {
          luaG_runerror(L, "sequence index %d out of range.", (int)nvalue(key));
          return;
        }
      }
      else {
        setobj2s(L, val, res);
        return;
      }
    }
    else if (ttisstring(t)) {
      size_t len;
      int index;
      const char *str;
      if (!(ttisnumber(key)))  /* Agena 1.3.1 */
        luaG_runerror(L, "invalid string index, expected an integer.");  /* 1.8.0 */
      str = svalue(t);  /* 0.20.1 patch */
      len = strlen(str);
      if (len == 0) {
        setfailvalue(val);
        return;
      }
      index = nvalue(key);
      if (index < 0) index = len + index + 1;
      if (index < 1 || index > len)
        luaG_runerror(L, "index %d out range.", index);
      setsvalue(L, val, luaS_newlstr(L, str+index-1, 1));
      return;
    }
    else if (ttispair(t) && ttisnumber(key)) {
      const TValue *res;
      Pair *h;
      h = pairvalue(t);
      res = agnPair_geti(h, nvalue(key));
      if (res == NULL) {
        if ((tm = fasttm(L, h->metatable, TM_INDEX)) == NULL) {
          luaG_runerror(L, "index out of range.");
          return;
        }
      }
      else {
        setobj2s(L, val, res);
        return;
      }
    }
    /* else will try the tag method */
    else if (ttisnil(tm = luaT_gettmbyobj(L, t, TM_INDEX)))
      luaG_typeerror(L, t, "index");
    if (ttisfunction(tm)) {
      callTMres(L, val, tm, t, key);
      return;
    }
    t = tm;  /* else repeat with `tm' */
  }
  luaG_runerror(L, "loop in gettable.");
}


void luaV_settable (lua_State *L, const TValue *t, TValue *key, StkId val) {
  int loop;
  TValue temp;
  for (loop = 0; loop < MAXTAGLOOP; loop++) {
    const TValue *tm;
    if (ttistable(t)) {  /* `t' is a table ? */
      Table *h = hvalue(t);
      /* TValue *oldval = luaH_set(L, h, key);  do a primitive set */
      /* if (ttisnotnil(oldval) ||  result is no nil?, removed 0.14.0 */
      /*  (tm = fasttm(L, h->metatable, TM_WRITEINDEX)) == NULL) {  or no TM? */
      if ((tm = fasttm(L, h->metatable, TM_WRITEINDEX)) == NULL) {  /* no TM? */
        TValue *oldval = luaH_set(L, h, key);  /* do a primitive set */
        setobj2t(L, oldval, val);
        luaC_barriert(L, h, val);
        if (ttisnil(val)) h->hasnil = 1;
        return;
      }
    }
    /* inserting values to sets with the indexing method is not supported,
       use the insert statement instead. */
    else if (ttisseq(t) && ttisnumber(key)) {  /* `t' is a sequence ? */
      Seq *h = seqvalue(t);
      if ((tm = fasttm(L, h->metatable, TM_WRITEINDEX)) == NULL) {
        if (agnSeq_seti(L, h, nvalue(key), val) == 0) {  /* 0.28.1 */
          luaG_runerror(L, "index out of range.");
          break;
        }
        return;
      }
    }
    else if (ttispair(t) && ttisnumber(key)) {  /* `t' is a pair ? */
      Pair *h = pairvalue(t);
      if ((tm = fasttm(L, h->metatable, TM_WRITEINDEX)) == NULL) {
        if (agnPair_seti(L, h, nvalue(key), val) == 0) {
          luaG_runerror(L, "index out of range.");
          break;
        }
        return;
      }
    }
    /* else will try the tag method */
    else if (ttisnil(tm = luaT_gettmbyobj(L, t, TM_WRITEINDEX))) {
      if (!ttisuset(t))
        luaG_typeerror(L, t, "index");
      else if (!ttisnil(val))  /* 0.31.5 */
        luaG_runerror(L, "use " LUA_QS " to put elements into a set.", "insert");
      else
        luaG_runerror(L, "use " LUA_QS " to remove elements from a set.", "delete");
    }
    if (ttisfunction(tm)) {
      callTM(L, tm, t, key, val);
      return;
    }
    /* else repeat with `tm', 0.24.2 patch */
    setobj(L, &temp, tm);  /* avoid pointing inside table (may rehash) */
    t = &temp;
  }
  luaG_runerror(L, "loop in settable.");
}


/* forward declaration */
static int call_binTM (lua_State *L, const TValue *p1, const TValue *p2,
                       StkId res, TMS event);


lua_Number agenaV_bottomindex (lua_State *L, Table *h) {  /* 1.1.0, tuned 1.3.3 */
  lua_lock(L);
  const TValue *k;
  Node *gn;
  lua_Number key_val, min_key;
  int i;
  min_key = HUGE_VAL;
  for (i=0; i < h->sizearray; i++) {
    if (ttisnotnil(&h->array[i])) {  /* array part is filled */
      min_key = i+1;
      break;  /* now try the hash part */
    }
  }
  if (min_key != HUGE_VAL) i = h->sizearray;
  for (i -= h->sizearray; i < sizenode(h); i++) {
    gn = gnode(h, i);
    k = key2tval(gn);
    if (ttisnumber(k) && !ttisnil(gval(gn))) {  /* key is a number and is assigned a non-null value ... */
      key_val = nvalue(k);
      if (key_val == trunc(key_val) && key_val < min_key) {  /* and an integer: is there a smaller key in hash part ? */
        min_key = key_val;
      }
    }
  }
  lua_unlock(L);
  return min_key;
}


lua_Number agenaV_topindex (lua_State *L, Table *h) {  /* 1.1.0, tuned 1.3.3 */
  lua_lock(L);
  const TValue *k;
  Node *gn;
  lua_Number key_val, max_key;
  int i;
  max_key = -HUGE_VAL;
  for (i=h->sizearray-1; i > -1; i--) {
    if (ttisnotnil(&h->array[i])) {  /* array part is filled */
      max_key = i+1;
      break;  /* now try the hash part */
    }
  }
  i = h->sizearray;
  for (i -= h->sizearray; i < sizenode(h); i++) {
    gn = gnode(h, i);
    k = key2tval(gn);
    if (ttisnumber(k) && !ttisnil(gval(gn))) {  /* key is a number and is assigned a non-null value ... */
      key_val = nvalue(k);
      if (key_val == trunc(key_val) && key_val > max_key) {  /* and an integer: is there a smaller key in hash part ? */
        max_key = key_val;
      }
    }
  }
  lua_unlock(L);
  return max_key;
}


void agenaV_bottom (lua_State *L, const TValue *t, StkId val) {  /* 0.24.0 */
  lua_lock(L);
  if (ttisseq(t)) {
    Seq *h;
    h = seqvalue(t);
    if (h->size != 0) {
      setobj2s(L, val, agnSeq_geti(h, 1));
    } else
      setnilvalue(val);
  } else if (ttistable(t)) {  /* Agena 1.1.0 */
      Table *h;
      h = hvalue(t);
      setobj2s(L, val, luaH_getnum(h, agenaV_bottomindex(L, h)));
  } else {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": sequence or table expected, got %s.", "bottom",
      luaT_typenames[(int)ttype(t)]);
  }
  lua_unlock(L);
}


void agenaV_top (lua_State *L, const TValue *t, StkId val) {  /* 0.24.0 */
  lua_lock(L);
  if (ttisseq(t)) {
    Seq *h;
    h = seqvalue(t);
    if (h->size != 0) {
      setobj2s(L, val, agnSeq_geti(h, h->size));
    } else
      setnilvalue(val);
  } else if (ttistable(t)) {  /* Agena 1.1.0 */
      Table *h;
      h = hvalue(t);
      setobj2s(L, val, luaH_getnum(h, agenaV_topindex(L, h)));
  } else {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": sequence or table expected, got %s.", "top",
      luaT_typenames[(int)ttype(t)]);
  }
  lua_unlock(L);
}


/* used by `element IN table/set/sequence/string`
   iterates over an object and checks whether the given argument is an element (a value,
   not a key) in this object;
   October 14, 2007 / November 23, 2007; patched Dec 08, 2007; tweaked January 04, 2008;
   tuned 1.3.3 */

void agenaV_in (lua_State *L, const TValue *value, const TValue *what, StkId idx) {
  lua_lock(L);
  switch (ttype(what)) {
    case LUA_TTABLE: {
      Table *t;
      const TValue *tm, *val;
      int i;
      t = hvalue(what);
      if ((tm = fasttm(L, t->metatable, TM_IN)) == NULL) {  /* no metamethod ? */
        for (i=0; i < t->sizearray; i++) {
          val = &t->array[i];
          if (ttisnotnil(val) && equalobj(L, value, val)) {
            setbvalue(idx, 1);
            lua_unlock(L);
            return;
          }
        }
        for (i -= t->sizearray; i < sizenode(t); i++) {  /* added 0.8.0 */
          val = gval(gnode(t, i));
          if (ttisnotnil(val) && equalobj(L, value, val)) {
            setbvalue(idx, 1);
            lua_unlock(L);
            return;
          }
        }
        setbvalue(idx, 0);
      } else if (ttisfunction(tm))
        call_binTM(L, value, what, idx, TM_IN);
      break;
    }
    case LUA_TSTRING: {
      if (ttisstring(value)) {
        int pos;
        const char *str = svalue(what);
        const char *sub = svalue(value);
        pos = strstr(str, sub) - str + 1;
        if (pos < 1 || strlen(sub) == 0)
          setnilvalue(idx);
        else
          setnvalue(idx, pos);
      } else {
        lua_unlock(L);
        luaG_runerror(L, "in " LUA_QS ": wrong type of argument.", "in");
      }
      break;
    }
    case LUA_TSET: {
      const TValue *tm;
      UltraSet *s;
      s = usvalue(what);
      if ((tm = fasttm(L, s->metatable, TM_IN)) == NULL) { /* no metamethod ? */
        if (agnUS_get(usvalue(what), value)) {
          setbvalue(idx, 1);
          lua_unlock(L);
        } else {
          if (ttype(value) > LUA_TTHREAD) {  /* is left object a structure ? -> iterate set */
            int i;
            for (i=0; i < sizenode(s); i++) {
              if (ttisnotnil(key2tval(gnode(s, i)))) {
                if (equalobj(L, value, key2tval(gnode(s, i)))) {
                  setbvalue(idx, 1);
                  lua_unlock(L);
                  return;
                }
              }
            }  /* of for i */
          }
          setbvalue(idx, 0);
        }
      } else if (ttisfunction(tm))
        call_binTM(L, value, what, idx, TM_IN);
      break;
    }
    case LUA_TSEQ: {
      const TValue *tm;
      Seq *h = seqvalue(what);
      if ((tm = fasttm(L, h->metatable, TM_IN)) == NULL) {  /* no metamethod ? */
        setbvalue(idx, agnSeq_get(seqvalue(what), value));
      } else if (ttisfunction(tm))
        call_binTM(L, value, what, idx, TM_IN);
      break;
    }
    case LUA_TPAIR: {
      const TValue *tm;
      Pair *h = pairvalue(what);
      if ((tm = fasttm(L, h->metatable, TM_IN)) == NULL) {  /* no metamethod ? */
        if (ttisnumber(value) && ttisnumber(pairitem(h, 0)) && ttisnumber(pairitem(h, 1)) ) {
          lua_Number x = nvalue(value);
          setbvalue(idx, x >= nvalue(pairitem(h, 0)) && x <= nvalue(pairitem(h, 1)) );
        } else
          setfailvalue(idx);
      } else if (ttisfunction(tm))
        call_binTM(L, value, what, idx, TM_IN);
      break;
    }
    default: {
      lua_unlock(L);
      luaG_runerror(L, "in " LUA_QS ": wrong type of argument.", "in");
    }
  }
  lua_unlock(L);
}


void agenaV_atendof (lua_State *L, StkId idxa, StkId idxb, StkId idxc) {
  lua_lock(L);
  if (ttisstring(idxb) && ttisstring(idxc)) {
    size_t p_len, s_len;
    const char *s, *p, *olds;
    char *pos;
    s = svalue(idxc);
    p = svalue(idxb);
    s_len = strlen(s);
    p_len = strlen(p);
    if (p_len == 0 || s_len == 0 || s_len <= p_len) {
      /* bail out if string or pattern is empty or if pattern is longer or equal in size */
      setnilvalue(idxa);
    } else {
      olds = s;
      s = s + s_len - p_len;
      pos = strstr(s, p);
      if (pos == s) {
        setnvalue(idxa, pos - olds + 1);
      } else {
        setnilvalue(idxa);
      }
    }
    setnilvalue(idxb);
    setnilvalue(idxc);
  } else {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": strings expected.", "atendof");
  }
  lua_unlock(L);
}


/* Compare contents of two tables. October 16, 2007; patched Dec 26, 2007; extended Jan 12, 2008;
   patched October 26, 2008; tweaked January 04, 2008
   mode = 0: only check for a SUBSET b;
   mode = 1: check a = b and b = a
   mode = 2: check a SUBSET b and a <> b */

int agenaV_comptables (lua_State *L, Table *a, Table *b, int mode) {
  Table *t1, *t2, *swap;
  TValue *t1val, *t2val;
  int i, j, oldi, oldj, flag, pass;
  t1 = a; t2 = b;
  pass = 0;
  flag = 1;
label:
  L->top++;  /* leave this increment here, otherwise there will be crashes, eg. [[1]] = [[1, 2]] */
  checkstack(L, "in VM function ", "agenaV_comptables");
  api_check(L, L->top < L->ci->top);
  i = -1;
  oldi = i;
  j = -1;
  oldj = j;  /* save value of j for consecutive iterations of 2nd table */
  for (i++; i < t1->sizearray; i++) {
    if (ttisnotnil(&t1->array[i])) {
      flag = 0;
      t1val = &t1->array[i];
      for (j++; j < t2->sizearray; j++) {
        if (ttisnotnil(&t2->array[j])) {
          t2val = &t2->array[j];
          if (equalobj(L, t1val, t2val)) {
            flag = 1;
            j = t2->sizearray;  /* set j properly so that next loop starts with correct value */
            break;  /* quit this loop if element has been found */
          }
        }
      } /* of for j */
      for (j -= t2->sizearray; j < sizenode(t2); j++) {
        if (ttisnotnil(gval(gnode(t2, j)))) {
          t2val = gval(gnode(t2, j));
          if (equalobj(L, t1val, t2val)) {
            flag = 1;
            break;  /* quit this loop if element has been found */
          }
        }
      } /* of for j */
      j = oldj;  /* reset pointer for 2nd table */
      if (!flag) break;  /* if an element was _not_ found exit for i loop completely */
    } /* of if */
  } /* of for i*/
  i = oldi+1+t1->sizearray;  /* reset i to upper bound in case element was not found */
  for (i -= t1->sizearray; i < sizenode(t1); i++) {  /* added 0.8.0 */
    if (ttisnotnil(gval(gnode(t1, i)))) {
      flag = 0;
      t1val = gval(gnode(t1, i));
      for (j++; j < t2->sizearray; j++) {
        if (ttisnotnil(&t2->array[j])) {
          t2val = &t2->array[j];
          if (equalobj(L, t1val, t2val)) {
            flag = 1;
            j = t2->sizearray;  /* set j properly so that next loop starts with correct value */
            break;  /* quit inner loop if element has been found */
          }
        }
      } /* of for j */
      for (j -= t2->sizearray; j < sizenode(t2); j++) {
        if (ttisnotnil(gval(gnode(t2, j)))) {
          t2val = gval(gnode(t2, j));
          if (equalobj(L, t1val, t2val)) {
            flag = 1;
            break;  /* quit inner loop if element has been found */
          }
        }
      } /* of for j */
      j = oldj;  /* reset pointer for 2nd table */
      if (!flag) break;  /* if an element was _not_ found exit loop completely */
    } /* of if */
  } /* of for i*/
  pass++;
  L->top--;
  /* in equality operation (mode = 1), check 2nd table against 1st table;
     condition: all elements in 1st table were found in 2nd table */
  if (pass == 1) {
    if (mode > 0 && flag == 1) {
      swap = t2;
      t2 = t1;
      t1 = swap;
      goto label;
    }
  } else {
    if (mode == 2) return !flag;
  }
  return flag;
}


int agenaV_comptablesonebyone (lua_State *L, Table *t1, Table *t2) {
  int i, j, flag;
  flag = 1;
  for (i=0, j=0; i < t1->sizearray && j < t2->sizearray; i++, j++) {
    if (ttisnotnil(&t1->array[i]) && ttisnotnil(&t2->array[j])) {
      /* FIXME: equalobj compares tables the Cantor way */
      if (!equalobjonebyone(L, &t1->array[i], &t2->array[j])) {
        flag = 0;
        break;
      }
    }
    else if (ttisnotnil(&t1->array[i]) || ttisnotnil(&t2->array[j])) {  /* 0.31.4 patch */
      flag = 0;
      break;
    }
  }
  if (flag) {  /* see whether there are surplus values */
    for (; i < t1->sizearray; i++) {
      if (ttisnotnil(&t1->array[i])) {
        flag = 0;
        break;
      }
    }
    if (flag) {
      for (; j < t2->sizearray; j++) {
        if (ttisnotnil(&t2->array[j])) {
          flag = 0;
          break;
        }
      }
    }
  }
  if (flag) {  /* if array entries were all equal, compare hash part */
    Node *t1node, *t2node;
    TValue *t1key, *t2key;
    const TValue *t1val, *t2val;
    for (i -= t1->sizearray; i < sizenode(t1); i++) {
      t1node = gnode(t1, i);
      if (ttisnotnil(gval(t1node))) {
        t1key = key2tval(t1node);
        t1val = gval(t1node);
        t2val = luaH_get(t2, t1key);
        if (!equalobjonebyone(L, t1val, t2val)) {
          flag = 0;
          break;
        }
      }
    }
    if (flag) {
      for (j -= t2->sizearray; j < sizenode(t2); j++) {
        t2node = gnode(t2, j);
        if (ttisnotnil(gval(t2node))) {
          t2key = key2tval(t2node);
          t2val = gval(t2node);
          t1val = luaH_get(t1, t2key);
          if (!equalobjonebyone(L, t2val, t1val)) {
            flag = 0;
            break;
          }
        }
      }
    }
  }
  return flag;
}


/* Compare contents of two UltraSets; April 08, 2008
   mode = 0: only check for a SUBSET b;
   mode = 1: check a = b and b = a
   mode = 2: check a XSUBSET b and a <> b */

int agenaV_compusets (lua_State *L, UltraSet *a, UltraSet *b, int mode) {
  UltraSet *t1, *t2, *swap;
  TValue *val;
  int i, j, flag, pass;
  t1 = a; t2 = b;
  pass = 0;
  flag = 1;
label:
  for (i=0; i < sizenode(t1); i++) {  /* added 0.8.0 */
    if (ttisnotnil(glkey(gnode(t1, i)))) {
      /* try a primitive comparison */
      val = key2tval(gnode(t1, i));
      if ((flag = agnUS_get(t2, val))) {
        continue;
      }
      if (ttype(val) > LUA_TTHREAD) {  /* is left object a structure ? -> iterate right set */
        for (j=0; j < sizenode(t2); j++) {
          if (ttisnotnil(key2tval(gnode(t2, j)))) {
            if (equalobj(L, val, key2tval(gnode(t2, j)))) {
              flag = 1;
              break;  /* quit this loop if element has been found */
            }
          }
        } /* of for j */
      }
      if (!flag) break;  /* if an element was _not_ found exit loop completely */
    } /* of if */
  } /* of for i*/
  pass++;
  /* in equality operation (mode = 1), check 2nd table against 1st table;
     condition: all elements in 1st table were found in 2nd table */
  if (pass == 1) {
    if (mode > 0 && flag == 1) {
      swap = t2;
      t2 = t1;
      t1 = swap;
      goto label;
    }
  } else {
    if (mode == 2) return !flag;
  }
  return flag;
}


/* Compare contents of two Sequences; June 10, 2008; tweaked January 04, 2008
   mode = 0: only check for a SUBSET b;
   mode = 1: check a = b and b = a
   mode = 2: check a XSUBSET b and a <> b */

int agenaV_compseqs (lua_State *L, Seq *a, Seq *b, int mode) {
  Seq *t1, *t2, *swap;
  int i, j, flag, pass;
  TValue *val;
  t1 = a; t2 = b;  /* seqs will be swapped later, so copy them */
  pass = 0;
  flag = 1;
label:
  checkstack(L, "in VM function ", "agenaV_compseqs");
  api_check(L, L->top < L->ci->top);
  for (i=0; i < t1->size; i++) {
    flag = 0;
    val = seqitem(t1, i);  /* faster than putting it on the stack */
    /* check whether value at L->top is in sequence t2 */
    for (j=0; j < t2->size; j++) {
      if (equalobj(L, val, seqitem(t2, j))) {
        flag = 1;
        break;
      }
    }
    if (!flag) break; /* if an element was _not_ found exit loop completely */
  } /* of for i*/
  pass++;
  /* in equality operation (mode = 1), check 2nd table against 1st table;
     condition: all elements in 1st table were found in 2nd table */
  if (pass == 1) {
    if (mode > 0 && flag == 1) {
      swap = t2;
      t2 = t1;
      t1 = swap;
      goto label;
    }
  } else {
    if (mode == 2) return !flag;
  }
  return flag;
}


/* Compare contents of two Sequences sequentially; May 28, 2009, 0.22.0 */

int agenaV_compseqsonebyone (lua_State *L, Seq *t1, Seq *t2) {
  int i, flag;
  flag = 1;
  checkstack(L, "in VM function ", "agenaV_compseqsonebyone");
  api_check(L, L->top < L->ci->top);
  if (t1->size != t2->size) return 0;
  for (i=0; i < t1->size; i++) {
    if (!equalobjonebyone(L, seqitem(t1, i), seqitem(t2, i))) {
      flag = 0;
      break;
    }
  }
  return flag;
}


int agenaV_comppairs (lua_State *L, Pair *a, Pair *b) {
  int i, flag;
  flag = 1;
  api_check(L, L->top < L->ci->top);
  checkstack(L, "in VM function ", "agenaV_comppairs");
  for (i=0; i < 2; i++) {
    const TValue *p = pairitem(a, i);
    const TValue *q = pairitem(b, i);
    if (!equalobj(L, p, q)) {
      flag = 0;
      break;
    }
  }
  return flag;
}


/* table (X)SUBSET operator; October 15, 2007; extended to Ultra Sets April 08, 2008 */

void agenaV_subset (lua_State *L, const TValue *tbl1, const TValue *tbl2, StkId idx, int mode) {
  if (ttistable(tbl1) && ttistable(tbl2)) {
    setbvalue(idx, agenaV_comptables(L, hvalue(tbl1), hvalue(tbl2), mode));
  } else if (ttisuset(tbl1) && ttisuset(tbl2)) {
    setbvalue(idx, agenaV_compusets(L, usvalue(tbl1), usvalue(tbl2), mode));
  } else if (ttisseq(tbl1) && ttisseq(tbl2)) {
    setbvalue(idx, agenaV_compseqs(L, seqvalue(tbl1), seqvalue(tbl2), mode));
  }
  else
    luaG_runerror(L, "in " LUA_QS ": tables, sets, or sequences expected.", (mode == 0) ? "subset" : "xsubset");
}


/* returns the number of entries in a table or dictionary; this function does not get
   confused by `holes` in a table but is slower since the entire table is traversed if has some;
   October 15, 2007; tuned on Dec 09, 2007 */

size_t agenaV_nops (lua_State *L, Table *t) {
  size_t i, c;
  i = c = 0; /* number of elements */
  /* iterate simple table of the form {v1, v2, ...}, not {1~v1, ...} ;
     if t is a dictionary, this loop is skipped */
  for (; i < t->sizearray; i++) {  /* 1.8.13 */
    if (ttisnotnil(&t->array[i])) c++;
  }
  /* iterate dictionary; if t is a simple table only one iteration is done */
  for (i -= t->sizearray; i < sizenode(t); i++) {
    if (ttisnotnil(gval(gnode(t, i)))) c++;
  }
  return c;
}


/* tuned 1.3.3 */

void agenaV_filled (lua_State *L, const TValue *tbl, StkId idx) {
  int i;
  if (ttistable(tbl)) {
    Table *t;
    t = hvalue(tbl);
    /* iterate simple table of the form {v1, v2, ...}, not {1~v1, ...} ;
       if t is a dictionary, this loop is also skipped */
    for (i=0; i < t->sizearray; i++) {
      if (ttisnotnil(&t->array[i])) {
        setbvalue(idx, 1);
        return;
      }
    }
    /* iterate dictionary; if t is a simple table only one iteration is done */
    for (i -= t->sizearray; i < sizenode(t); i++) {
      if (ttisnotnil(gval(gnode(t, i)))) { setbvalue(idx, 1); return; }
    }
    setbvalue(idx, 0);
  }
  else if (ttisuset(tbl)) {
    UltraSet *s = usvalue(tbl);
    setbvalue(idx, s->size != 0);
  }
  else if (ttisseq(tbl)) {
    Seq *s = seqvalue(tbl);
    setbvalue(idx, s->size != 0);
  }
  else
    luaG_runerror(L, "in " LUA_QS ": table, set, or sequence expected, got %s.", "filled",
      luaT_typenames[(int)ttype(tbl)]);
}


/* INSERT element INTO table statement; October 14, 2007;
   patched Nov 22, 2007 (crash if table was NULL); extended to support multiple returns on April 3, 2009 */

void agenaV_insert (lua_State *L, const TValue *t, StkId idx, int nargs) {
  int c;
  const TValue *tm;
  if (nargs == 0) {  /* are there multiple returns ? */
    nargs = cast_int(L->top - idx);
    L->top = L->ci->top;
  }
  if (ttistable(t)) {
    int n;
    Table *tbl = hvalue(t);
    if (fasttm(L, tbl->metatable, TM_WEAK) != NULL)
      luaG_runerror(L, LUA_QS " is not suited for weak tables.", "insert");  /* Agena 1.7.2 */
    n = luaH_getn(tbl)+1;
    if ((tm = fasttm(L, tbl->metatable, TM_WRITEINDEX)) == NULL) {
      for (c=0; c < nargs; c++) {
        setobj2t(L, luaH_setnum(L, tbl, n++), idx+c);
        luaC_barriert(L, tbl, idx+c);
        if (ttisnil(idx+c)) tbl->hasnil = 1;  /* Agena 1.7.2 */
      }
      return;
    }
    else {
      if (ttisfunction(tm)) {  /* better sure than sorry */
        for (c=0; c < nargs; c++)
          callTM(L, tm, t, idx+c, idx+c);
        return;
      }  /* else: do nothing */
    }
  } else if (ttisuset(t)) {
    UltraSet *s = usvalue(t);
    if ((tm = fasttm(L, s->metatable, TM_WRITEINDEX)) == NULL) {
      for (c=0; c < nargs; c++) {
        agnUS_set(L, s, idx+c);
        luaC_barrierset(L, s, idx+c);  /* Agena 1.2 fix */
      }
      return;
    }
    else {
      if (ttisfunction(tm)) {  /* better sure than sorry */
        for (c=0; c < nargs; c++)
          callTM(L, tm, t, idx+c, idx+c);
        return;
      }  /* else: do nothing */
    }
  } else if (ttisseq(t)) {
    Seq *s = seqvalue(t);
    if ((tm = fasttm(L, s->metatable, TM_WRITEINDEX)) == NULL) {
      for (c=0; c < nargs; c++) {
        if (agnSeq_set(L, seqvalue(t), idx+c) == NULL)
          luaG_runerror(L, "in " LUA_QS ": maximum sequence size exceeded.", "insert");
      }
      return;
    }
    else {
      if (ttisfunction(tm)) {  /* better sure than sorry */
        for (c=0; c < nargs; c++)
          callTM(L, tm, t, idx+c, idx+c);
        return;
      }  /* else: do nothing */
    }
  }
  else
    luaG_runerror(L, "in " LUA_QS ": table, set, or sequence expected, got %s.", "insert",
      luaT_typenames[(int)ttype(t)]);
}


/* DELETE element FROM table statement; October 14, 2007; patched Dec 08, 2007; tweaked January 07, 2009;
   extended to support multiple returns on April 3, 2009 */

void agenaV_delete (lua_State *L, const TValue *tbl, StkId val, int nargs) {
  if (nargs == 0) {  /* multiple arguments ? */
    nargs = cast_int(L->top - val);
  }
  if (ttistable(tbl)) {
    Table *t;
    TValue *value, *oldval;
    int i, j;
    t = hvalue(tbl);
    if (fasttm(L, t->metatable, TM_WEAK) != NULL)
      luaG_runerror(L, LUA_QS " is not suited for weak tables.", "delete");  /* Agena 1.7.2 */
    for (i=0; i < nargs; i++) {
      for (j=0; j < t->sizearray; j++) {
        if (ttisnotnil(&t->array[j])) {
          value = &t->array[j];
          if (equalobj(L, val+i, value)) {
            setnilvalue(L->top);
            setobj2t(L, luaH_setnum(L, t, cast_num(j+1)), L->top);
            luaC_barriert(L, t, L->top);
            t->hasnil = 1;
          }
        }
      }
      for (j -= t->sizearray; j < sizenode(t); j++) {  /* added 0.8.0 */
        if (ttisnotnil(gval(gnode(t, j)))) {
          L->top++;
          setobj2s(L, L->top, gval(gnode(t, j)));
          if (equalobj(L, val+i, L->top)) {
            setobj2s(L, L->top-1, key2tval(gnode(t, j)));
            oldval = luaH_set(L, t, L->top-1); /* do a primitive set */
            if (ttisnotnil(oldval)) {
              setnilvalue(L->top);
              setobj2t(L, oldval, L->top);
              luaC_barriert(L, t, L->top);
              t->hasnil = 1;
            }
          }
          L->top--;
        }
      }
    }
  } else if (ttisuset(tbl)) {
    int c;
    for (c=0; c < nargs; c++)
      agnUS_delete(L, usvalue(tbl), val+c);
  } else if (ttisseq(tbl)) {
    int c;
    Seq *seq = seqvalue(tbl);  /* 0.28.1 */
    for (c=0; c < nargs; c++)
      agnSeq_delete(L, seq, val+c);
  }
  else
    luaG_runerror(L, "in " LUA_QS ": table, set, or sequence expected, got %s.", "delete",
      luaT_typenames[(int)ttype(tbl)]);
}


/* compute union of two tables; used by UNION operator; October 16, 2007.
   All elements in both tables are copied into the new table, regardless
   whether they occur multiple times or not (i.a. [1] union [1] >> [1, 1]);
   tweaked 1.3.3 */

void agenaV_tableunion (lua_State *L, const TValue *tbl1, const TValue *tbl2, StkId idx) {
  if (ttistable(tbl1) && ttistable(tbl2)) {
    Table *t1, *t2, *newtable;
    TValue *val;
    int i;
    unsigned int c;
    c = 0;
    t1 = hvalue(tbl1);
    t2 = hvalue(tbl2);
    /* table for resulting union */
    newtable = luaH_new(L, t1->sizearray+t2->sizearray, 0);
    /* traverse first table */
    for (i=0; i < t1->sizearray; i++) {
      if (ttisnotnil(&t1->array[i])) {
        c++;
        val = &t1->array[i];
        setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), val);
        luaC_barriert(L, newtable, val);
      }
    }
    for (i -= t1->sizearray; i < sizenode(t1); i++) {  /* added 0.8.0 */
      if (ttisnotnil(gval(gnode(t1, i)))) {
        c++;
        val = gval(gnode(t1, i));
        setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), val);
        luaC_barriert(L, newtable, val);
      }
    }
    /* traverse second table */
    for (i=0; i < t2->sizearray; i++) {
      if (ttisnotnil(&t2->array[i])) {
        c++;
        val = &t2->array[i];
        setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), val);
        luaC_barriert(L, newtable, val);
      }
    }
    for (i -= t2->sizearray; i < sizenode(t2); i++) {  /* added 0.8.0 */
      if (ttisnotnil(gval(gnode(t2, i)))) {
        c++;
        val = gval(gnode(t2, i));
        setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), val);
        luaC_barriert(L, newtable, val);
      }
    }
    /* prepare table for return to Agena environment */
    sethvalue(L, idx, newtable);
  } else if (ttisuset(tbl1) && ttisuset(tbl2)) {
    UltraSet *t1, *t2, *newtable;
    TValue *v;
    LNode *val;
    int i;
    unsigned int c;
    c = 0;
    t1 = usvalue(tbl1);
    t2 = usvalue(tbl2);
    /* table for resulting union */
    newtable = agnUS_new(L, sizenode(t1) + sizenode(t2));
    /* traverse first set */
    for (i=0; i < sizenode(t1); i++) {  /* added 0.8.0 */
      val = gnode(t1, i);
      if (ttisnotnil(glkey(val))) {
        c++;
        v = key2tval(val);
        agnUS_set(L, newtable, v);
        luaC_barrierset(L, newtable, v);  /* Agena 1.2 fix */
      }
    }
    /* traverse second set */
    for (i=0; i < sizenode(t2); i++) {  /* added 0.8.0 */
      val = gnode(t2, i);
      if (ttisnotnil(glkey(val))) {
        c++;
        v = key2tval(val);
        agnUS_set(L, newtable, v);
        luaC_barrierset(L, newtable, v);  /* Agena 1.2 fix */
      }
    }
    /* prepare table for return to Agena environment */
    setusvalue(L, idx, newtable);
  } else if (ttisseq(tbl1) && ttisseq(tbl2)) {
    Seq *t1, *t2, *newtable;
    int i;
    unsigned int c;
    TValue *v;
    c = 0;
    t1 = seqvalue(tbl1);
    t2 = seqvalue(tbl2);
    /* stack must be increased to avoid `invalid key to next` errors */
    api_check(L, L->top < L->ci->top);
    /* table for resulting union */
    newtable = agnSeq_new(L, t1->size + t2->size);
    /* traverse first table */
    for (i=0; i < t1->size; i++) {  /* added 0.8.0 */
      c++;
      v = seqitem(t1, i);
      agnSeq_seti(L, newtable, c, v);
      luaC_barrierseq(L, newtable, v);  /* Agena 1.2 fix */
    }
    /* traverse second table */
    for (i=0; i < t2->size; i++) {  /* added 0.8.0 */
      c++;
      v = seqitem(t2, i);
      agnSeq_seti(L, newtable, c, v);
      luaC_barrierseq(L, newtable, v);  /* Agena 1.2 fix */
    }
    /* prepare table for return to Agena environment */
    setseqvalue(L, idx, newtable);
  }
  else
    luaG_runerror(L, "in " LUA_QS ": both operands must either be tables, sets, or sequences.", "union");
}


/* Compute intersection or difference sets; October 16, 2007; patched October 26, 2008; tweaked January 07, 2009
   All elements in both tables are copied into the new table, regardless
   whether they occur multiple times or not (i.a. [1, 1, 2] minus [2] >> [1, 1])
      mode = 0: intersection,
      mode = 1: minus */

void agenaV_setops (lua_State *L, const TValue *tbl1, const TValue *tbl2, StkId idx, int mode) {
  if (ttistable(tbl1) && ttistable(tbl2)) {
    Table *t1, *t2, *newtable;
    TValue *t1val, *t2val;
    int i, j, oldi, oldj, flag;
    unsigned int c;
    c = 0;
    t1 = hvalue(tbl1);
    t2 = hvalue(tbl2);
    i = oldi = j = oldj = -1;  /* save value of j for consecutive interations of 2nd table */
    newtable = luaH_new(L, t1->sizearray + t2->sizearray, 0);
    flag = 0;
    for (i++; i < t1->sizearray; i++) {
      if (ttisnotnil(&t1->array[i])) {
        flag = 0;
        t1val = &t1->array[i];  /* value */
        /* increase top for value of 2nd table */
        for (j++; j < t2->sizearray; j++) {
          t2val = &t2->array[j];
          if (ttisnotnil(t2val)) {
            if (equalobj(L, t1val, t2val)) {
              if (!mode) {  /* intersection operation ? -> put value into newtable */
                c++;
                setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), t2val);
                luaC_barriert(L, newtable, t2val);
              }
              j = t2->sizearray;  /* set j properly so that next loop starts with correct value */
              flag = 1;
              break;  /* quit inner loop */
            }
          }
        }  /* of for j */
        for (j -= t2->sizearray; j < sizenode(t2); j++) {
          t2val = gval(gnode(t2, j));
          if (ttisnotnil(t2val)) {
            if (equalobj(L, t1val, t2val)) {
              if (!mode) {  /* intersection operation ? -> put value into newtable */
                c++;
                setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), t2val);
                luaC_barriert(L, newtable, t2val);
              }
              flag = 1;
              break;  /* quit inner loop */
            }
          }
        }  /* of for j */
        j = oldj;  /* reset pointer for 2nd table */
        if (mode && !flag) {   /* minus operation and element not found in 2nd table ? */
          c++;
          setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), t1val);
          luaC_barriert(L, newtable, t1val);
        }
      }  /* of if */
    }  /* of for i */
    i = oldi+1+t1->sizearray;  /* reset i to upper bound in case element was not found */
    for (i -= t1->sizearray; i < sizenode(t1); i++) {  /* added 0.8.0 */
      if (ttisnotnil(gval(gnode(t1, i)))) {
        flag = 0;
        t1val = gval(gnode(t1, i));  /* value */
        for (j++; j < t2->sizearray; j++) {
          t2val = &t2->array[j];
          if (ttisnotnil(t2val)) {
            if (equalobj(L, t1val, t2val)) {
              if (!mode) {  /* intersection operation ? -> put value into newtable */
                c++;
                setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), t2val);
                luaC_barriert(L, newtable, t2val);
              }
              flag = 1;
              j = t2->sizearray;  /* set j properly so that next loop starts with correct value */
              break;  /* quit inner loop */
            }
          }
        }  /* of for j */
        for (j -= t2->sizearray; j < sizenode(t2); j++) {
          t2val = gval(gnode(t2, j));
          if (ttisnotnil(t2val)) {
            if (equalobj(L, t1val, t2val)) {
              if (!mode) {  /* intersection operation ? -> put value into newtable */
                c++;
                setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), t2val);
                luaC_barriert(L, newtable, t2val);
              }
              flag = 1;
              break;  /* quit inner loop */
            }
          }
        }  /* of for j */
        j = oldj;  /* reset pointer for 2nd table */
        if (mode && !flag) {   /* minus operation and element not found in 2nd table ? */
          c++;
          setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), t1val);
          luaC_barriert(L, newtable, t1val);
        }
      }  /* of if */
    }  /* of for i */
    luaH_resizearray(L, newtable, c);
    sethvalue(L, idx, newtable);
  } else if (ttisuset(tbl1) && ttisuset(tbl2)) {
    UltraSet *t1, *t2, *newset;
    TValue *t1val, *t2val;
    int i, flag;
    unsigned int c;
    c = 0;
    t1 = usvalue(tbl1);
    t2 = usvalue(tbl2);
    newset = agnUS_new(L, sizenode(t1) + sizenode(t2) + 2);
    flag = 0;
    for (i=0; i < sizenode(t1); i++) {  /* added 0.8.0 */
      if (ttisnotnil(glkey(gnode(t1, i)))) {
        flag = 0;
        t1val = key2tval(gnode(t1, i));  /* value */
        if (agnUS_get(t2, t1val)) {
          if (!mode) {  /* intersection operation ? -> put value into newtable */
            c++;
            agnUS_set(L, newset, t1val);
            luaC_barrierset(L, newset, t1val);  /* Agena 1.2 fix */
          }
          flag = 1;
        } else if (ttype(t1val) > LUA_TTHREAD) {
          /* is left object a structure ? -> iterate right set */
          int j;
          for (j=0; j < sizenode(t2); j++) {
            if (ttisnotnil(key2tval(gnode(t2, j)))) {
              t2val = key2tval(gnode(t2, j));
              if (equalobj(L, t1val, t2val)) {
                flag = 1;
                if (!mode) {  /* intersection operation ? -> put value into newtable */
                  c++;
                  agnUS_set(L, newset, t2val);
                  luaC_barrierset(L, newset, t2val);  /* Agena 1.2 fix */
                }
                break;  /* quit this loop if element has been found */
              }
            }
          }  /* of for j */
        }
        if (mode && !flag) {   /* minus operation and element not found in 2nd table ? */
          c++;
          agnUS_set(L, newset, t1val);
          luaC_barrierset(L, newset, t1val);  /* Agena 1.2 fix */
        }
      }  /* of if */
    }  /* of for i */
    agnUS_resize(L, newset, c);  /* FIXED 0.28.1: resize UltraSet */
    setusvalue(L, idx, newset);
  } else if (ttisseq(tbl1) && ttisseq(tbl2)) {
    Seq *t1, *t2, *newseq;
    TValue *t1val;
    int i, j, flag;
    unsigned int c;
    c = 0;
    t1 = seqvalue(tbl1);
    t2 = seqvalue(tbl2);
    newseq = agnSeq_new(L, t1->size + t2->size);
    flag = 0;
    for (i=0; i < t1->size; i++) {
      flag = 0;
      t1val = seqitem(t1, i);  /* value */
      for (j=0; j < t2->size; j++) {
        if (equalobj(L, t1val, seqitem(t2, j))) {
          if (!mode) {  /* intersection operation ? -> put value into newtable */
            c++;
            agnSeq_seti(L, newseq, c, t1val);
            luaC_barrierseq(L, newseq, t1val);  /* Agena 1.2 fix */
          }
          flag = 1;
          break;
        }
      }
      if (mode && !flag) {   /* minus operation and element not found in 2nd seq ? */
        c++;
        agnSeq_seti(L, newseq, c, t1val);
        luaC_barrierseq(L, newseq, t1val);  /* Agena 1.2 fix */
      }
    }
    agnSeq_resize(L, newseq, c);  /* 0.28.1 */
    setseqvalue(L, idx, newseq);
  } else
    luaG_runerror(L, "in " LUA_QS ": both operands must either be tables, sets, or sequences.",
      ((mode=0) ? "intersect" : "minus"));
}

/* SPLIT operator, 0.6.0 & 0.11.2; patched 0.25.5, 05.08.2009 */

void agenaV_split (lua_State *L, const char *s, const char *sep, StkId idx) {
  Seq *newtable;
  const char *e;
  int c;
  size_t len;
  c = 0;
  len = strlen(sep);
  newtable = agnSeq_new(L, 10);
  if (len == 0 || strcmp(s, sep) == 0) {
    setnilvalue(L->top);
    goto end;
  }
  while ((e=strstr(s, sep)) != NULL) {  /* strtok is slower */
    c++;
    setsvalue2s(L, L->top, luaS_newlstr(L, s, e-s));
    agnSeq_seti(L, newtable, cast_num(c), L->top);
    luaC_barrierseq(L, newtable, L->top);
    s = e + len;
  }
  /* process last token */
  setsvalue2s(L, L->top, luaS_newlstr(L, s, strlen(s)));
  agnSeq_seti(L, newtable, cast_num(c+1), L->top);
end:
  luaC_barrierseq(L, newtable, L->top);
  setseqvalue(L, idx, newtable);
}


/* UNIQUE operator, 0.5.4; patched 0.8.0; tuned January 07, 2009 - 0.12.3 */

void agenaV_tunique (lua_State *L, const TValue *tbl, StkId idx) {
  Table *t, *newtable;
  TValue *t1val, *t2val;
  int i, j, flag;
  unsigned int c;
  t = hvalue(tbl);
  c = 0;
  newtable = luaH_new(L, t->sizearray, 0);
  for (i=0; i < t->sizearray; i++) {
    t1val = &t->array[i];
    if (ttisnotnil(t1val)) {
      flag = 1;
      /* increase top for value in 2nd table so that value of first table is overwritten */
      for (j=i+1; j < t->sizearray; j++) {
        t2val = &t->array[j];
        if (ttisnotnil(t2val)) {
          if (equalobj(L, t1val, t2val)) {
            flag = 0;
            break;  /* quit inner loop if element has been found */
          }
        }
      }  /* of for j */
      if (flag) {
        c++;
        setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), t1val);
      }  /* if an element was _not_ found put it into new table */
    }  /* of if */
  }  /* of for i */
  for (i -= t->sizearray; i < sizenode(t); i++) {  /* added 0.8.0 */
    t1val = gval(gnode(t, i));
    if (ttisnotnil(t1val)) {
      flag = 1;
      for (j=i+1; j < sizenode(t); j++) {
        t2val = gval(gnode(t, j));
        if (!ttisnil(t2val)) {
          if (equalobj(L, t1val, t2val)) {
            flag = 0;
            break;
          }
        }
      }
      if (flag) {
        c++;
        setobj2t(L, luaH_setnum(L, newtable, cast_num(c)), t1val);
      }
    }
  }
  if (c > 0) luaH_resizearray(L, newtable, c);
  sethvalue(L, idx, newtable);
}


/* UNIQUE operator for sequences, 0.11.2; patched 0.12.0; tweaked January 10, 2009 */

void agenaV_sequnique (lua_State *L, const TValue *tbl, StkId idx) {
  Seq *t, *newseq;
  TValue *t1val, *t2val;
  int i, j, flag;
  unsigned int c;
  t = seqvalue(tbl);
  api_check(L, L->top < L->ci->top);
  c = 0;
  newseq = agnSeq_new(L, t->size);
  for (i=0; i < t->size; i++) {
    flag = 1;
    t1val = seqitem(t, i);  /* value */
    for (j=i+1; j < t->size; j++) {
      t2val = seqitem(t, j);
      if (equalobj(L, t1val, t2val)) {
        flag = 0;
        break;  /* quit inner loop if element has been found */
      }
    }
    if (flag) {
      c++;
      agnSeq_seti(L, newseq, c, t1val);
      luaC_barrierseq(L, newseq, t1val);  /* Agena 1.2 fix */
    }  /* if an element was _not_ found put it into new sequence */
  }
  if (c > 0) agnSeq_resize(L, newseq, c);  /* 0.28.1 */
  setseqvalue(L, idx, newseq);
}


/* COPY operator, 0.9.0; extended 1.1.0 to treat references accordingly; patched 1.7.1; extended 1.8.9 */

void agenaV_copy (lua_State *L, const TValue *tbl, StkId idx, Table *seen) {
  if (ttistable(tbl)) {
    Table *t, *dest;
    TValue *oldval;
    int i;
    t = hvalue(tbl);
    /* Agena 1.1.0: check whether table has already been traversed. If so, just return its reference, otherwise initialise seen table */
    if (seen == NULL) {
      seen = luaH_new(L, 0, 0);
      luaC_checkGC(L);  /* Agena 1.8.10 patch */
    } else {
      const TValue *dest;
      dest = luaH_get(seen, tbl);  /* seen[src], if assigned copy the reference only, not the structure itself */
      if (ttisnotnil(dest)) {
        setobj2s(L, L->top, dest);
        return;
      }
    }
    /* register new table in seen, ala seen[src] := dest */
    dest = luaH_new(L, t->sizearray, sizenode(t));
    sethvalue(L, L->top, dest);  /* 1.7.1 patch */
    TValue *old = luaH_set(L, seen, tbl);  /* do a primitive set, key = original table */
    setobj2t(L, old, L->top);  /* set new table dest to seen as its value */
    luaC_barriert(L, seen, L->top);
    for (i=0; i < t->sizearray; i++) {
      if (ttisnotnil(&t->array[i])) {
        setobj2s(L, L->top, &t->array[i]);  /* value */
        if (ttistable(L->top) || ttisuset(L->top) || ttisseq(L->top) || ttispair(L->top)) {  /* make a deep copy */
          checkstack(L, "in ", "copy");  /* prevent crashes with deeply nested tables */
          agenaV_copy(L, &t->array[i], L->top, seen);  /* copy value and put it at L->top */
        }
        setobj2t(L, luaH_setnum(L, dest, cast_num(i+1)), L->top);
        luaC_barriert(L, dest, L->top);
      }  /* of if */
    }  /* of for i */
    for (i -= t->sizearray; i < sizenode(t); i++) {
      if (ttisnotnil(gval(gnode(t, i)))) {
        setobj2s(L, L->top, key2tval(gnode(t, i)));  /* push key */
        if (ttistable(L->top) || ttisuset(L->top) || ttisseq(L->top) || ttispair(L->top)) {  /* make a deep copy */
          checkstack(L, "in ", "copy");  /* prevent crashes with deeply nested tables */
          agenaV_copy(L, key2tval(gnode(t, i)), L->top, seen);  /* copy value and put it at L->top */
        }
        L->top++;
        setobj2s(L, L->top, gval(gnode(t, i)));  /* push value */
        if (ttistable(L->top) || ttisuset(L->top) || ttisseq(L->top) || ttispair(L->top)) {  /* make a deep copy */
          checkstack(L, "in ", "copy");  /* prevent crashes with deeply nested tables */
          agenaV_copy(L, gval(gnode(t, i)), L->top, seen);  /* copy value and put it at L->top */
        }
        oldval = luaH_set(L, dest, L->top-1);
        setobj2t(L, oldval, L->top);
        luaC_barriert(L, dest, L->top);
        L->top--;
      }
    }
    /* add pointer to metatable of original table */
    dest->metatable = t->metatable;
    /* copy its type */
    dest->type = t->type;
    /* return new table */
    sethvalue(L, idx, dest);
  } else if (ttisuset(tbl)) {
    UltraSet *t, *dest;
    int i;
    t = usvalue(tbl);
    /* Agena 1.1.0: check whether set has already been traversed. If so, just return its reference, otherwise initialise seen table */
    if (seen == NULL) {
      seen = luaH_new(L, 0, 0);
      luaC_checkGC(L);  /* Agena 1.8.10 patch */
    } else {
      const TValue *dest;
      dest = luaH_get(seen, tbl);  /* seen[src], if assigned copy the reference only, not the structure itself */
      if (ttisnotnil(dest)) {
        setobj2s(L, L->top, dest);
        return;
      }
    }
    /* register new set in seen, ala seen[src] := dest */
    dest = agnUS_new(L, sizenode(t));
    setusvalue(L, L->top, dest);  /* 1.7.1 patch */
    TValue *old = luaH_set(L, seen, tbl);  /* do a primitive set */
    setobj2t(L, old, L->top);
    luaC_barriert(L, seen, L->top);
    for (i=0; i < sizenode(t); i++) {
      if (!ttisnil(glkey(glnode(t, i)))) {
        setobj2s(L, L->top, key2tval(glnode(t, i)));
        if (ttisuset(L->top) || ttistable(L->top) || ttisseq(L->top) || ttispair(L->top)) {  /* make a deep copy */
          checkstack(L, "in ", "copy");  /* prevent crashes with deeply nested sets */
          agenaV_copy(L, key2tval(glnode(t, i)), L->top, seen);
        }
        agnUS_set(L, dest, L->top);
        luaC_barrierset(L, dest, L->top);  /* Agena 1.2 fix */
      }
    }
    /* copy its type, Agena 1.0.5 */
    dest->type = t->type;
    /* add pointer to metatable of original set */
    dest->metatable = t->metatable;
    /* return new set */
    setusvalue(L, idx, dest);
  } else if (ttisseq(tbl)) {
    Seq *s, *dest;
    int i;
    api_check(L, L->top < L->ci->top);
    s = seqvalue(tbl);
    /* Agena 1.1.0: check whether set has already been traversed. If so, just return its reference, otherwise initialise seen table */
    if (seen == NULL) {
      seen = luaH_new(L, 0, 0);
      luaC_checkGC(L);  /* Agena 1.8.10 patch */
    } else {
      const TValue *dest;
      dest = luaH_get(seen, tbl);  /* seen[src], if assigned copy the reference only, not the structure itself */
      if (ttisnotnil(dest)) {
        setobj2s(L, L->top, dest);
        return;
      }
    }
    /* register new sequence in seen, ala seen[src] := dest */
    dest = agnSeq_new(L, s->maxsize);
    setseqvalue(L, L->top, dest);  /* 1.7.1 patch */
    TValue *old = luaH_set(L, seen, tbl);  /* do a primitive set, key = original table */
    setobj2t(L, old, L->top);  /* set new table dest to seen as its value */
    luaC_barriert(L, seen, L->top);
    for (i=0; i < s->size; i++) {
      setobj2s(L, L->top, agnSeq_geti(s, i+1));
      if (ttisseq(L->top) || ttistable(L->top) || ttisuset(L->top) || ttispair(L->top)) {  /* make a deep copy */
        checkstack(L, "in ", "copy");  /* prevent crashes with deeply nested sequences */
        agenaV_copy(L, agnSeq_geti(s, i+1), L->top, seen);
      }
      agnSeq_set(L, dest, L->top);
      luaC_barrierseq(L, dest, L->top);  /* XXX, I am too afraid to comment this out, there would only be an increase in speed of 1 % */
    }
    /* copy its type */
    dest->type = s->type;
    /* add pointer to metatable of original sequence */
    dest->metatable = s->metatable;
    /* return new sequence */
    setseqvalue(L, idx, dest);
  }
  else if (ttispair(tbl)) {  /* added with 1.8.9 */
    Pair *s, *dest;
    int i;
    api_check(L, L->top < L->ci->top);
    s = pairvalue(tbl);
    /* Agena 1.1.0: check whether set has already been traversed. If so, just return its reference, otherwise initialise seen table */
    if (seen == NULL) {
      seen = luaH_new(L, 0, 0);
      luaC_checkGC(L);  /* Agena 1.8.10 patch */
    } else {
      const TValue *dest;
      dest = luaH_get(seen, tbl);  /* seen[src], if assigned copy the reference only, not the structure itself */
      if (ttisnotnil(dest)) {
        setobj2s(L, L->top, dest);
        return;
      }
    }
    /* register new pair in seen, ala seen[src] := dest */
    dest = agnPair_new(L);
    setpairvalue(L, L->top, dest);  /* 1.7.1 patch */
    TValue *old = luaH_set(L, seen, tbl);  /* do a primitive set, key = original table */
    setobj2t(L, old, L->top);  /* set new table dest to seen as its value */
    luaC_barriert(L, seen, L->top);
    for (i=0; i < 2; i++) {
      setobj2s(L, L->top, agnPair_geti(s, i+1));
      if (ttisseq(L->top) || ttistable(L->top) || ttisuset(L->top) || ttispair(L->top)) {  /* make a deep copy */
        checkstack(L, "in ", "copy");  /* prevent crashes with deeply nested pairs */
        agenaV_copy(L, agnPair_geti(s, i+1), L->top, seen);
      }
      agnPair_seti(L, dest, i+1, L->top);
      luaC_barrierpair(L, dest, L->top);  /* XXX, I am too afraid to comment this out, there would only be an increase in speed of 1 % */
    }
    /* copy its type */
    dest->type = s->type;
    /* add pointer to metatable of original pair */
    dest->metatable = s->metatable;
    /* return new pair */
    setpairvalue(L, idx, dest);
  }
  else
    luaG_runerror(L, "in " LUA_QS ": table, set, sequence, or pair expected, got %s.", "copy", luaT_typenames[(int)ttype(tbl)]);
}


/* SADD operator for tables, 0.9.0, added 10.01.2008 */

void agenaV_sadd (lua_State *L, const TValue *tbl, StkId idx) {
  Table *t;
  TValue *key, *val;
  const TValue *tm;
  size_t i, c;  /* Agena 1.6.0 */
  lua_Number sum = 0;
  t = hvalue(tbl);
  if ((tm = fasttm(L, t->metatable, TM_SADD)) == NULL) {  /* no metamethod ? */
    c = 0;
    for (i=0; i < t->sizearray; i++) {
      val = &t->array[i];
      if (ttisnumber(val)) {
        sum += nvalue(val);
        c++;
      }
    }
    for (i -= t->sizearray; i < sizenode(t); i++) {
      key = key2tval(gnode(t, i));
      val = gval(gnode(t, i));
      if (ttisnumber(key) && ttisnumber(val)) {
        sum += nvalue(val);
        c++;
      }
    }
    if (c != 0) {
      setnvalue(idx, sum);
    } else {
      setnilvalue(idx);
    }
  } else if (ttisfunction(tm))
    call_binTM(L, tbl, tbl, idx, TM_SADD);
}


/* SADD operator for sequences, 0.11.2, added 09.06.2008 */

void agenaV_seqsadd (lua_State *L, const TValue *tbl, StkId idx) {
  Seq *t;
  TValue *val;
  const TValue *tm;
  size_t i;  /* Agena 1.6.0 */
  lua_Number sum = 0;
  t = seqvalue(tbl);
  api_check(L, L->top < L->ci->top);
  if ((tm = fasttm(L, t->metatable, TM_SADD)) == NULL) {  /* no metamethod ? */
    for (i=0; i < t->size; i++) {
      val = seqitem(t, i);
      if (ttisnumber(val)) {
        sum += nvalue(val);
      }
    }
    if (i != 0) {
      setnvalue(idx, sum);
    } else {
      setnilvalue(idx);
    }
  } else if (ttisfunction(tm))
    call_binTM(L, tbl, tbl, idx, TM_SADD);
}


/* QSADD operator for tables, 0.9.1, added 13.01.2008 */

void agenaV_qsadd (lua_State *L, const TValue *tbl, StkId idx) {
  Table *t;
  TValue *key, *val;
  const TValue *tm;
  size_t i, c;  /* Agena 1.6.0 */
  lua_Number sum, num;
  sum = 0;
  t = hvalue(tbl);
  if ((tm = fasttm(L, t->metatable, TM_QSADD)) == NULL) {  /* no metamethod ? */
    c = 0;
    for (i=0; i < t->sizearray; i++) {
      val = &t->array[i];
      if (ttisnumber(val)) {
        num = nvalue(val);
        sum += num*num;
        c++;
      }
    }
    for (i -= t->sizearray; i < sizenode(t); i++) {
      key = key2tval(gnode(t, i));
      val = gval(gnode(t, i));
      if (ttisnumber(key) && ttisnumber(val)) {
        num = nvalue(val);
        sum += num*num;
        c++;
      }
    }
    if (c != 0) {
      setnvalue(idx, sum);
    } else {
      setnilvalue(idx);
    }
  } else if (ttisfunction(tm))
    call_binTM(L, tbl, tbl, idx, TM_QSADD);
}


/* QSADD operator for sequences, 0.11.2, added 09.06.2008 */

void agenaV_seqqsadd (lua_State *L, const TValue *tbl, StkId idx) {
  Seq *t;
  TValue *val;
  const TValue *tm;
  size_t i;  /* Agena 1.6.0 */
  lua_Number sum, num;
  sum = 0;
  t = seqvalue(tbl);
  api_check(L, L->top < L->ci->top);
  if ((tm = fasttm(L, t->metatable, TM_QSADD)) == NULL) {  /* no metamethod ? */
    for (i=0; i < t->size; i++) {
      val = seqitem(t, i);
      if (ttisnumber(val)) {
        num = nvalue(val);
        sum += num*num;
      }
    }
    if (i != 0) {
      setnvalue(idx, sum);
    } else {
      setnilvalue(idx);
    }
  } else if (ttisfunction(tm))
    call_binTM(L, tbl, tbl, idx, TM_QSADD);
}


void agnV_trim (lua_State *L, const TValue *b, StkId idx) {
  size_t i, len;
  int c, begin;
  char *r;
  const char *s = svalue(b);
  len = tsvalue(b)->len;
  r = (char *)malloc(sizeof(char)*(len+1));
  if (r == 0)
    luaG_runerror(L, "memory allocation error in " LUA_QS ".", "trim");  /* Agena 1.0.4 */
  if (len == 0) {
    setsvalue(L, idx, luaS_newlstr(L, "", 0));
    xfree(r);  /* Agena 1.0.4/1.10.4 */
    return;
  }
  c = 0;
  for (i=0; i<len; i++) {
    if (!(uchar(s[i]) == ' ' && uchar(s[i+1]) == ' ')) {
      r[c] = uchar(s[i]);
      c++;
    }
  }
  r[c] = '\0';
  begin = 0;
  while(r[begin] == ' ') begin++;
  c--;
  while (r[c] == ' ') c--;
  if (c < 0) c = 0;
  setsvalue(L, idx, luaS_newlstr(L, r+begin, c-begin+1));
  xfree(r);  /* 1.10.4 */
}


void luaV_substring (lua_State *L, StkId string, StkId inds, StkId idx) {  /* Agena 1.2 */
  size_t len;
  int start, stop;
  const char *str;
  start = 0;  /* to prevent compiler warnings */
  str = svalue(string);
  len = stop = tsvalue(string)->len;  /* setting of stop to prevent compiler warnings */
  if (len == 0) {
    setfailvalue(idx);
    return;
  }
  if (ttisnumber(inds) && ttisnumber(inds+1)) {
    start = nvalue(inds);
    stop = nvalue(inds+1);
  }
  else
    luaG_runerror(L, "in %s: invalid arguments.", "substring op");
  if (start < 0) start = len + start + 1;
  if (stop < 0) stop = len + stop + 1;
  if (start < 1 || start > len || stop < 1 || stop > len)
    luaG_runerror(L, "in %s: index out of range.", "substring op");
  if (start > stop)
    luaG_runerror(L, "in %s: left index (%d) > right index (%d).", "substring op", start, stop);
  /* delete indices, keep substring */
  setnilvalue(inds);
  setnilvalue(inds+1);
  setsvalue(L, idx, luaS_newlstr(L, str+start-1, stop-start+1));
  /* setting L->top to func confuses the stack */
}


void luaV_tablesublist (lua_State *L, StkId t, StkId inds, StkId idx) {  /* Agena 1.2 */
  int i, start, stop;
  Table *h, *newtable;
  const TValue *k, *v;
  Node *gn;
  lua_Number n;
  lua_lock(L);
  start = stop = 0;  /* to prevent compiler warnings */
  h = hvalue(t);
  if (ttisnumber(inds) && ttisnumber(inds+1)) {
    start = nvalue(inds)-1;
    stop = nvalue(inds+1)-1;
    if (stop < start)
      luaG_runerror(L, "in %s: lower bound (%d) > upper bound (%d).", "table indexing", start+1, stop+1);
  }
  else
    luaG_runerror(L, "in %s: invalid arguments.", "table indexing");
  newtable = luaH_new(L, stop-start+1, 0);
  i = -1;
  if (start > i && start < h->sizearray) i = start-1;
  stop = (stop < h->sizearray) ? stop + 1 : h->sizearray;
  for (i++; i < stop; i++) {
    v = &h->array[i];
    if (ttisnotnil(v)) {  /* array element is filled */
      setobj2t(L, luaH_setnum(L, newtable, cast_num(i+1)), v);
      luaC_barriert(L, newtable, v);
    }
  }
  i = h->sizearray;
  for (i -= h->sizearray; i < sizenode(h); i++) {
    gn = gnode(h, i);
    k = key2tval(gn);
    if (ttisnumber(k) && !ttisnil(gval(gn))) {  /* key is a number and is assigned a non-null value ... */
      n = nvalue(k);
      if (trunc(n) == n && n >= start && n <= stop) {  /* and an integer: is there a smaller key in hash part ? */
        v = gval(gn);
        setobj2t(L, luaH_setnum(L, newtable, cast_num(n)), v);
        luaC_barriert(L, newtable, v);
      }
    }
  }
  /* delete indices, keep substring */
  setnilvalue(inds);
  setnilvalue(inds+1);
  /* return sublist */
  sethvalue(L, idx, newtable);
  /* setting L->top to func confuses the stack */
  lua_unlock(L);
}


void luaV_seqsublist (lua_State *L, StkId t, StkId inds, StkId idx) {  /* Agena 1.2 */
  int i, start, stop;
  size_t c, length;
  Seq *h, *newtable;
  const TValue *v;
  lua_lock(L);
  start = stop = 0;  /* to prevent compiler warnings */
  h = seqvalue(t);
  length = h->size;
  if (ttisnumber(inds) && ttisnumber(inds+1)) {
    start = posrelat(nvalue(inds), length);
    stop = posrelat(nvalue(inds+1), length);
    if (start > stop || (start < 1 || stop > length))  /* 1.11.8 */
      luaG_runerror(L, "sequence index out of range.");
  }
  else
    luaG_runerror(L, "in %s: invalid arguments.", "sequence indexing");  /* 1.11.8 */
  newtable = agnSeq_new(L, stop-start+1);
  i = 1; c = 0;
  if (start > 1 && start <= length) i = start;
  stop = (stop < length) ? stop : h->size;
  for (; i <= stop; i++) {
    c++;
    v = seqitem(h, i-1);
    agnSeq_seti(L, newtable, c, v);
    luaC_barrierseq(L, newtable, v);
  }
  /* delete indices, keep substring */
  setnilvalue(inds);
  setnilvalue(inds+1);
  /* return sublist */
  setseqvalue(L, idx, newtable);
  /* setting L->top to func confuses the stack */
  lua_unlock(L);
}


char *bufinit (lua_State *L, size_t initsize, size_t *buffersize, size_t *stringsize) {
  char *buffer;
  buffer = (char *)malloc((initsize+1)*sizeof(char));
  if (buffer == NULL)
    luaG_runerror(L, "buffer allocation error.");
  *buffersize = initsize + 1;
  *stringsize = 0;
  *buffer = '\0';  /* necessary, otherwise there would be stray characters in the string */
  return buffer;
}


char *appendstr (lua_State *L, char *buffer, const char *string, size_t n, size_t *buffersize, size_t *stringsize) {
  char *oldpos;
  char *temp;
  if (n == 0) return buffer;
  if (*stringsize + n + 1 > *buffersize) {  /* Agena 1.0.6 fix: +1 for trailing '\0' that is inserted later, otherwise Agena might crash */
    *buffersize = (*buffersize)*2 + n*sizeof(char);  /* Agena 1.0.6 change to speed up computation */
    temp = (char *)realloc(buffer, *buffersize);
    if (temp == NULL)  /* Agena 1.0.6 security fix */
      luaG_runerror(L, "buffer re-allocation error.");
    else
      buffer = temp;
  }
  oldpos = buffer;
  buffer += *stringsize;
  memcpy(buffer, string, n);  /* much faster than strncpy, as fast as for (i=0; i<n; i++) *buffer++ = *string++ */
  buffer += n;
  *buffer = '\0';
  *stringsize += n;
  return oldpos;
}


/* REPLACE operator, 0.10.0, 29.03.2008; extended 0.12.2, October 19/25, 2008; tweaked January 10, 2009;
   extended 0.26.0, 05.08.2009; patched 1.0.3, 11.11.2010 */

void agenaV_replace (lua_State *L, StkId idx, int nargs) {
  int init;
  size_t l1, l2, l3;
  const char *src, *p, *repl, *s2;
  lua_lock(L);
  if (!ttisstring(idx))
    luaG_runerror(L, "in " LUA_QS ": string expected for argument #1, got %s.",
      "replace", luaT_typenames[(int)ttype(idx)]);
  if (nargs == 3) {
    /* three strings passed */
    src = svalue(idx); l1 = tsvalue(idx)->len;
    if (ttisstring(idx+1) && ttisstring(idx+2)) {
      p = svalue(idx+1); l2 = tsvalue(idx+1)->len;
      if (l2 == 0) {
        setfailvalue(idx);
        return;
      }
      init = 0;  /* cursor */
      s2 = lmemfind(src+init, l1-init, p, l2);
      if (s2) {
        char *buf;
        size_t bufsize, strsize;
        bufsize = 0; strsize = 0;
        buf = bufinit(L, l1, &bufsize, &strsize);
        repl = svalue(idx+2); l3 = tsvalue(idx+2)->len;
        while (1) {
          if (init) /* do not conduct very first search again */
            s2 = lmemfind(src+init, l1-init, p, l2);
          if (s2) {
            buf = appendstr(L, buf, src+init, s2-(src+init), &bufsize, &strsize);
            buf = appendstr(L, buf, repl, l3, &bufsize, &strsize);
            init += (s2-(src+init)) + l2;
          } else {
            buf = appendstr(L, buf, src+init, l1-init, &bufsize, &strsize);
            break;
          }
        }
        setsvalue2s(L, idx, luaS_newlstr(L, buf, strsize));
        xfree(buf);  /* 1.10.4 */
      }
      setnilvalue(idx+1); setnilvalue(idx+2);
      lua_unlock(L);
      /* do not change L->top here since this will confuse the stack */
    } else if (ttisnumber(idx+1) && ttisstring(idx+2)) {  /* inspired by the REXX function `changestr` */
      size_t pos;
      char *buf;
      size_t bufsize, strsize;
      bufsize = 0; strsize = 0;
      buf = bufinit(L, l1, &bufsize, &strsize);
      pos = posrelat(nvalue(idx+1), l1);  /* 2nd argument is the position */
      if (pos < 1 || pos > l1)
        luaG_runerror(L, "in " LUA_QL("replace") ": index out of range.");
      repl = svalue(idx+2); l3 = tsvalue(idx+2)->len;
      buf = appendstr(L, buf, src, pos-1, &bufsize, &strsize);  /* push left part of original string */
      buf = appendstr(L, buf, repl, l3, &bufsize, &strsize);  /* push new string */
      buf = appendstr(L, buf, src+pos, strlen(src+pos), &bufsize, &strsize);  /* push rest of original string */
      setsvalue2s(L, idx, luaS_newlstr(L, buf, strsize));
      xfree(buf);  /* 1.10.4 */
      setnilvalue(idx+1); setnilvalue(idx+2);
      lua_unlock(L);
    }
    else
      luaG_runerror(L, "in " LUA_QS ": wrong kind of argument.", "replace");
  } else if (nargs == 2 && ttistable(idx+1)) {
    /* string and table passed */
    Table *t;
    Pair *nt;
    TValue *val, *lval, *rval;
    int j;
    t = hvalue(idx+1);
    for (j=0; j < t->sizearray; j++) {  /* iterate the substitution list */
      /* FIXME: handle dictionaries, as well. */
      if (ttisnotnil(&t->array[j])) {
        src = svalue(idx); l1 = tsvalue(idx)->len;
        val = &t->array[j];  /* push pair on stack */
        if (!ttispair(val))
          luaG_runerror(L, "in " LUA_QS ": pair expected, got %s.",
            "replace", luaT_typenames[(int)ttype(val)]);
        else {
          nt = pairvalue(val);
          /* get pattern */
          lval = pairitem(nt, 0);
          if (!ttisstring(lval))
            luaG_runerror(L, "in " LUA_QS ": pair of strings expected.", "replace");
          p = svalue(lval); l2 = tsvalue(lval)->len;
          if (l2 == 0) {
            setfailvalue(idx);
            return;
          }
          init = 0;  /* cursor for every pass */
          s2 = lmemfind(src+init, l1-init, p, l2);
          if (s2) {  /* only replace if the pattern has been found */
            char *buf;
            size_t bufsize, strsize;
            bufsize = 0; strsize = 0;
            /* get replacement */
            rval = pairitem(nt, 1);
            if (!ttisstring(rval))
              luaG_runerror(L, "in " LUA_QS ": pair of strings expected.", "replace");
            repl = svalue(rval); l3 = tsvalue(rval)->len;
            /* now substitute */
            buf = bufinit(L, l1, &bufsize, &strsize);
            while (1) {
              if (init)  /* do not conduct very first search again */
                s2 = lmemfind(src+init, l1-init, p, l2);
              if (s2) {
                buf = appendstr(L, buf, src+init, s2-(src+init), &bufsize, &strsize);
                buf = appendstr(L, buf, repl, l3, &bufsize, &strsize);
                init += (s2-(src+init)) + l2;
              } else {
                buf = appendstr(L, buf, src+init, l1-init, &bufsize, &strsize);
                break;
              }
            } /* of while */
            /* reset stack top */
            setsvalue2s(L, idx, luaS_newlstr(L, buf, strsize));
            xfree(buf);  /* 1.10.4 */
          }
        } /* of !ttispair */
      } /* of ttisnotnil */
    } /* of for j */
    setnilvalue(idx+1);  /* delete table */
    lua_unlock(L);
    /* do not change L->top here since this will confuse the stack */
  } else if (nargs == 2 && ttisseq(idx+1)) {
    /* string and sequence passed */
    Seq *t;
    Pair *nt;
    TValue *val, *lval, *rval;
    int j;
    t = seqvalue(idx+1);
    api_check(L, L->top < L->ci->top);
    for (j=0; j < t->size; j++) {  /* iterate the substitution list */
      src = svalue(idx); l1 = tsvalue(idx)->len;
      val = seqitem(t, j);  /* push pair */
      if (!ttispair(val))
        luaG_runerror(L, "in " LUA_QS ": pair expected, got %s.",
          "replace", luaT_typenames[(int)ttype(val)]);
      else {
        nt = pairvalue(val);
        /* get pattern */
        lval = pairitem(nt, 0);
        if (!ttisstring(lval))
          luaG_runerror(L, "in " LUA_QS ": pair of strings expected.", "replace");
        p = svalue(lval); l2 = tsvalue(lval)->len;
        if (l2 == 0) {
          setfailvalue(idx);
          return;
        }
        init = 0;  /* cursor for each pass */
        s2 = lmemfind(src+init, l1-init, p, l2);
        if (s2) {  /* only replace if the pattern has been found */
          char *buf;
          size_t bufsize, strsize;
          bufsize = 0; strsize = 0;
          /* get replacement */
          rval = pairitem(nt, 1);
          if (!ttisstring(rval))
            luaG_runerror(L, "in " LUA_QS ": pair of strings expected.", "replace");
          repl = svalue(rval); l3 = tsvalue(rval)->len;
          /* now substitute */
          buf = bufinit(L, l1, &bufsize, &strsize);
          while (1) {
            if (init)  /* do not conduct very first search again */
              s2 = lmemfind(src+init, l1-init, p, l2);
            if (s2) {
              buf = appendstr(L, buf, src+init, s2-(src+init), &bufsize, &strsize);
              buf = appendstr(L, buf, repl, l3, &bufsize, &strsize);
              init += (s2-(src+init)) + l2;
            } else {
              buf = appendstr(L, buf, src+init, l1-init, &bufsize, &strsize);
              break;
            }
          } /* of while */
          /* reset stack top */
          setsvalue2s(L, idx, luaS_newlstr(L, buf, strsize));
          xfree(buf);  /* 1.10.4 */
        }
      } /* of !ttispair */
    } /* of for j */
    setnilvalue(idx+1);  /* delete sequence */
    lua_unlock(L);
    /* do not change L->top here since this will confuse the stack */
  }
  else if (nargs == 2)
    luaG_runerror(L, "in " LUA_QS ": string and a table or sequence of pairs expected.", "replace");
  else
    luaG_runerror(L, "in " LUA_QS ": two or three arguments expected.", "replace");
}


/* JOIN operator, 0.11.0, patched 1.0.5, changed 1.1.0 */

void agenaV_join_aux (lua_State *L, StkId idx, int nargs, int size, int *start, int *stop) {
  if (nargs > 2 && ttisnumber(idx+2)) {
    *start = posrelat(nvalue(idx+2), size) - 1;
    if (*start < 0 || *start >= size)
      luaG_runerror(L, "in " LUA_QS ": index out of range.", "join");
  } else
    *start = 0;
  if (nargs > 3 && ttisnumber(idx+3)) {
    *stop = posrelat(nvalue(idx+3), size);
    if (*stop > size)
      luaG_runerror(L, "in " LUA_QS ": index out of range.", "join");
    if (*stop <= *start)
      luaG_runerror(L, "in " LUA_QS ": last index < first index.", "join");
  } else
    *stop = size;
}


void agenaV_join (lua_State *L, StkId idx, int nargs) {
  lua_lock(L);
  char *buf;
  const char *delim;
  int i, start, stop;
  size_t delimlen, bufsize, strsize, c;
  c = 0;
  bufsize = 0; strsize = 0;
  if (nargs > 1 && ttisstring(idx+1)) {
    delim = svalue(idx+1);
    delimlen = tsvalue(idx+1)->len;
  } else {
    delim = NULL;
    delimlen = 0;
  }
  if (ttistable(idx)) {
    Table *t;
    TValue *val;
    t = hvalue(idx);
    /* determine overall size of resulting string at once */
    for (i=0; i < t->sizearray; i++) {
      if (ttisnotnil(&t->array[i])) {
        val = &t->array[i];  /* value */
        if (ttisstring(val)) {
          c++;
          bufsize += tsvalue(val)->len;
        }
      }  /* of if */
    }  /* of for i */
    /* values in dictionaries are joined in random order, so this may be quite useless: */
    for (i -= t->sizearray; i < sizenode(t); i++) {
      if (ttisnotnil(gval(gnode(t, i)))) {
        val = gval(gnode(t, i));
        if (ttisstring(val)) {
          c++;
          bufsize += tsvalue(val)->len;
        }
      }
    }
    buf = bufinit(L, (bufsize+1) + (c-1)*delimlen, &bufsize, &strsize);  /* Agena 1.6.0 */
    /* now concat the strings */
    agenaV_join_aux(L, idx, nargs, t->sizearray, &start, &stop);
	for (i=start; i < stop; i++) {
      if (ttisnotnil(&t->array[i])) {
        val = &t->array[i];  /* value */
        if (ttisstring(val)) {
          buf = appendstr(L, buf, svalue(val), tsvalue(val)->len, &bufsize, &strsize);
          if (delim) {
            buf = appendstr(L, buf, delim, delimlen, &bufsize, &strsize);
          }
        } else
          luaG_runerror(L, "in " LUA_QS ": strings expected in structure, got %s.", "join",
            luaT_typenames[(int)ttype(val)]);
      }  /* of if */
    }  /* of for i */
    if (stop != t->sizearray) i = t->sizearray;
    /* values in dictionaries are joined in random order, so this may be quite useless: */
    for (i -= t->sizearray; i < sizenode(t); i++) {
      if (ttisnotnil(gval(gnode(t, i)))) {
        val = gval(gnode(t, i));
        if (ttisstring(val)) {
          buf = appendstr(L, buf, svalue(val), tsvalue(val)->len, &bufsize, &strsize);
          if (delim)
            buf = appendstr(L, buf, delim, delimlen, &bufsize, &strsize);
        } else
          luaG_runerror(L, "in " LUA_QS ": strings expected in structure, got %s.", "join",
            luaT_typenames[(int)ttype(val)]);
      }
    }
	/* Since the new string now terminates with the delimitor, remove the delimitor.
	   Agena 1.6.0 patch: if table contains no or empty strings do not try to delete the missing trailing delim. */
    if (strsize != 0 && delim) {  /* Agena 1.6.0 patch */
      char *oldpos = buf;
      buf += strsize-delimlen;
      *buf = '\0';
      buf = oldpos;
      strsize -= delimlen;
    }
  } else if (ttisseq(idx)) {
    const TValue *val;
    Seq *s;
    api_check(L, L->top < L->ci->top);
    s = seqvalue(idx);
    /* determine overall size of resulting string at once */
    for (i=0; i < s->size; i++) {
      val = agnSeq_geti(s, i+1);
      if (ttisstring(val)) {
        c++;
        bufsize += tsvalue(val)->len;
      }
    }
    buf = bufinit(L, (bufsize+1) + (c-1)*delimlen, &bufsize, &strsize);
    /* now concat the strings */
    agenaV_join_aux(L, idx, nargs, s->size, &start, &stop);
    for (i=start; i < stop; i++) {
      val = agnSeq_geti(s, i+1);
      if (ttisstring(val)) {
        buf = appendstr(L, buf, svalue(val), tsvalue(val)->len, &bufsize, &strsize);
      } else
        luaG_runerror(L, "in " LUA_QS ": strings expected in structure, got %s.", "join",
          luaT_typenames[(int)ttype(val)]);
      if (delim && i < stop - 1)
        buf = appendstr(L, buf, delim, delimlen, &bufsize, &strsize);
    }
  }
  else {
    luaG_runerror(L, "in " LUA_QS ": table or sequence expected, got %s.", "join",
      luaT_typenames[(int)ttype(idx)]);
    return;
  }
  /* return joined string */
  setsvalue2s(L, idx, luaS_newlstr(L, buf, strsize));
  xfree(buf);  /* 1.10.4 */
  lua_unlock(L);
}


/* XOR operator, 0.27.0, 23.08.2009, extended 0.27.1, 05.09.2009, patched 1.3.1, 09.01.2011, changed 1.6.3, 04.06.2012 */

void agenaV_xor (lua_State *L, StkId idxa, StkId idxb, StkId idxc) {
  lua_lock(L);
  if (ttisboolean(idxb) && ttisboolean(idxc)) {
    setbvalue(idxa, (l_istrue(idxb) && l_isfalseorfail(idxc)) || (l_isfalseorfail(idxb) && l_istrue(idxc)));
  } else if (ttisnil(idxb) && ttisboolean(idxc)) {
    setbvalue(idxa, l_istrue(idxc));
  } else if (ttisboolean(idxb) && ttisnil(idxc)) {
    setbvalue(idxa, l_istrue(idxb));
  } else if (ttisnil(idxb) && ttisnil(idxc)) {
    setbvalue(idxa, 0);
  } else if ((ttisnil(idxb) || l_isfalseorfail(idxb)) && !ttisnil(idxc)) {
    setobjs2s(L, idxa, idxc);
  } else if (!ttisnil(idxb) && (ttisnil(idxc) || l_isfalseorfail(idxc))) {
    /* do nothing, leave value at idxa at its place */
    setobjs2s(L, idxa, idxb);
  } else if ((l_istrue(idxb) && !ttisnil(idxc)) || (l_istrue(idxc) && !ttisnil(idxb))) {
    setbvalue(idxa, 0);
  } else if ((!ttisnil(idxb)) && (!ttisnil(idxc))) {
    setbvalue(idxa, 0);
  } else {
    setfailvalue(idxa);
  }
  setnilvalue(idxb);  /* delete second operand */
  setnilvalue(idxc);  /* delete third operand, Agena 1.6.3 */
  lua_unlock(L);
  /* do not change L->top here since this will confuse the stack */
}


/* BAND operator, 0.27.0, 26.08.2009, patched 1.3.1, 09.01.2011, changed 1.6.3, 04.06.2012 */

void agenaV_band (lua_State *L, StkId idxa, StkId idxb, StkId idxc) {
  lua_lock(L);
  if (ttisnumber(idxb) && ttisnumber(idxc)) {
    if (L->settings & 1) {
      setnvalue(idxa, (LUA_INTEGER)nvalue(idxb) & (LUA_INTEGER)nvalue(idxc)); }
    else {
      setnvalue(idxa, (LUAI_UINT32)nvalue(idxb) & (LUAI_UINT32)nvalue(idxc)); }
  } else {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": numbers expected.", "&&");
  }
  setnilvalue(idxb);  /* delete second operand */
  setnilvalue(idxc);  /* delete third operand, Agena 1.6.3 */
  lua_unlock(L);
  /* do not change L->top here since this will confuse the stack */
}


/* BOR operator, 0.27.0, 26.08.2009, patched 1.3.1, 09.01.2011, changed 1.6.3, 04.06.2012 */

void agenaV_bor (lua_State *L, StkId idxa, StkId idxb, StkId idxc) {
  lua_lock(L);
  if (ttisnumber(idxb) && ttisnumber(idxc)) {
    if (L->settings & 1) {
      setnvalue(idxa, (LUA_INTEGER)nvalue(idxb) | (LUA_INTEGER)nvalue(idxc)); }
    else {
      setnvalue(idxa, (LUAI_UINT32)nvalue(idxb) | (LUAI_UINT32)nvalue(idxc)); }
  } else {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": numbers expected.", "||");
  }
  setnilvalue(idxb);  /* delete second operand */
  setnilvalue(idxc);  /* delete third operand, Agena 1.6.3 */
  lua_unlock(L);
  /* do not change L->top here since this will confuse the stack */
}


/* BXOR operator, 0.27.0, 26.08.2009, patched 1.3.1, 09.01.2011, changed 1.6.3, 04.06.2012 */

void agenaV_bxor (lua_State *L, StkId idxa, StkId idxb, StkId idxc) {
  lua_lock(L);
  if (ttisnumber(idxb) && ttisnumber(idxc)) {
    if (L->settings & 1) {
      setnvalue(idxa, (LUA_INTEGER)nvalue(idxb) ^ (LUA_INTEGER)nvalue(idxc)); }
    else {
      setnvalue(idxa, (LUAI_UINT32)nvalue(idxb) ^ (LUAI_UINT32)nvalue(idxc)); }
  } else {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": numbers expected.", "^^");
  }
  setnilvalue(idxb);  /* delete second operand */
  setnilvalue(idxc);  /* delete third operand, Agena 1.6.3 */
  lua_unlock(L);
  /* do not change L->top here since this will confuse the stack */
}


/* PERCENT operator, 1.10.6, 10.04.2013 */

void agenaV_percent (lua_State *L, StkId idxa, StkId idxb, StkId idxc) {
  lua_lock(L);
  if (ttisnumber(idxb) && ttisnumber(idxc)) {
    setnvalue(idxa, luai_numpercent(nvalue(idxb), nvalue(idxc)));
  } else {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": numbers expected.", "*%");
  }
  setnilvalue(idxb);  /* delete second operand */
  setnilvalue(idxc);  /* delete third operand */
  lua_unlock(L);
  /* do not change L->top here since this will confuse the stack */
}


/* PERCENT operator, 1.11.4, 13.05.2013 */

void agenaV_percentratio (lua_State *L, StkId idxa, StkId idxb, StkId idxc) {
  lua_lock(L);
  if (ttisnumber(idxb) && ttisnumber(idxc)) {
    setnvalue(idxa, luai_numpercentratio(nvalue(idxb), nvalue(idxc)));
  } else {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": numbers expected.", "/%");
  }
  setnilvalue(idxb);  /* delete second operand */
  setnilvalue(idxc);  /* delete third operand */
  lua_unlock(L);
  /* do not change L->top here since this will confuse the stack */
}


/* PERCENTADD operator, 1.11.3, 09.05.2013 */

void agenaV_percentadd (lua_State *L, StkId idxa, StkId idxb, StkId idxc) {
  lua_lock(L);
  if (ttisnumber(idxb) && ttisnumber(idxc)) {
    setnvalue(idxa, luai_numpercentadd(nvalue(idxb), nvalue(idxc)));
  } else {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": numbers expected.", "+%");
  }
  setnilvalue(idxb);  /* delete second operand */
  setnilvalue(idxc);  /* delete third operand */
  lua_unlock(L);
  /* do not change L->top here since this will confuse the stack */
}


/* PERCENTSUB operator, 1.11.3, 09.05.2013 */

void agenaV_percentsub (lua_State *L, StkId idxa, StkId idxb, StkId idxc) {
  lua_lock(L);
  if (ttisnumber(idxb) && ttisnumber(idxc)) {
    setnvalue(idxa, luai_numpercentsub(nvalue(idxb), nvalue(idxc)));
  } else {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": numbers expected.", "-%");
  }
  setnilvalue(idxb);  /* delete second operand */
  setnilvalue(idxc);  /* delete third operand */
  lua_unlock(L);
  /* do not change L->top here since this will confuse the stack */
}


/* INSTR operator, 0.27.3, 19.10.2009; extended 1.1.0, 09.12.2010, extended 1.3.2, 13.01.2011;
   patched 1.3.3, 06.02.2011; extended 1.10.0, 07.03.2013 */

int agenaV_setpairints (lua_State *L, StkId idx, size_t a, size_t b) {  /* 1.10.0, patched 1.10.4 */
  Pair *p = agnPair_new(L);
  setnvalue(L->top, a);
  agnPair_seti(L, p, 1, L->top);
  luaC_barrierpair(L, p, L->top);
  setnvalue(L->top, b);
  agnPair_seti(L, p, 2, L->top);
  luaC_barrierpair(L, p, L->top);
  setpairvalue(L, idx, p);
  setnilvalue(L->top);  /* do NOT prematurely call luaC_checkGC to avoid memory leaks ! */
  return 1;  /* perform garbage collection later in `protected` VM */
}

int agenaV_instr (lua_State *L, StkId idx, int nargs) {
  int i;
  int dogc;
  lua_lock(L);
  dogc = 0;  /* tell VM tofinally conduct GC if a pair has been pushed (0 = no, 1 = yes) */
  if (nargs < 2)
    luaG_runerror(L, "in " LUA_QS ": at least two arguments expected, got %d.", "instr", nargs);
  if (ttisstring(idx) && ttisstring(idx+1)) {  /* 1.1.0: taken from the former string.seek function */
    const char *s, *p;
    int israw, isreverse, numbergiven, borders;
    size_t s_len;
    ptrdiff_t init;
    israw = 0; isreverse = 0; init = 1; numbergiven = 0; borders = 0;
    for (i=2; i < nargs; i++) {
      if (ttisnumber(idx+i)) { init = nvalue(idx+i); numbergiven = 1; continue; }
      if (ttisboolean(idx+i)) { israw = 1; continue; }
      if (ttisstring(idx+i)) {  /* 1.10.0 */
        if (strcmp(svalue(idx+i), "reverse") == 0) { isreverse = 1; continue; }
        if (strcmp(svalue(idx+i), "borders") == 0) borders = 1;
      }
    }
    s = svalue(idx);
    p = svalue(idx+1);
    s_len = tsvalue(idx)->len;
    if (!isreverse && israw) {  /* search from left to right */
      ptrdiff_t pos;
      init = posrelat(init, s_len) - 1;
      if (init < 0) init = 0;
      if ((size_t)init >= s_len) {
        for (i=0; i < nargs; i++) setnilvalue(idx+i);  /* push null and delete all arguments */
        return dogc;
      }
      if (tsvalue(idx+1)->len == 0) {  /* bail out if pattern is empty */
        for (i=0; i < nargs; i++) setnilvalue(idx+i);  /* push null and delete all arguments */
        return dogc;
      }
      pos = strstr(s+init, p) - s + 1;
      if (pos < 1) {
        setnilvalue(idx);
      } else {
        if (borders) { /* 1.10.0 */
          dogc = agenaV_setpairints(L, idx, pos, pos + tsvalue(idx+1)->len - 1);
        } else
          setnvalue(idx, pos);
      }
    } else if (isreverse) {  /* search from right to left */
      size_t ss_len, sp_len, i;
      char *ss, *sp;
      ptrdiff_t pos;
      init = posrelat(-(numbergiven ? init : -1), s_len) - 1;
      if (init < 0) init = 0;
      if ((size_t)init >= s_len) {
        for (i=0; i < nargs; i++) setnilvalue(idx+i);  /* push null and delete all arguments */
        return dogc;
      }
      ss_len = s_len;
      sp_len = tsvalue(idx+1)->len;
      if (ss_len == 0 || sp_len == 0) {  /* bail out if string or pattern is empty */
        for (i=0; i < nargs; i++) setnilvalue(idx+i);  /* push null and delete all arguments */
        return dogc;
      }
      ss = (char *)malloc(sizeof(char)*(ss_len+1));  /* Agena 1.0.4 */
      if (ss == NULL) luaG_runerror(L, "in " LUA_QS ": memory allocation error.", "instr");
      sp = (char *)malloc(sizeof(char)*(sp_len+1));  /* Agena 1.0.4 */
      if (sp == NULL) luaG_runerror(L, "in " LUA_QS ": memory allocation error.", "instr");
      /* reverse string content */
      for (i = ss_len-1; *s != '\0'; s++, i--) ss[i] = uchar(*s);
      ss[ss_len] = '\0';
      for (i = sp_len-1; *p != '\0'; p++, i--) sp[i] = uchar(*p);
      sp[sp_len] = '\0';
      pos = strstr(ss+init, sp) - ss + 1;
      if (pos < 1)  /* nothing has been found */
        setnilvalue(idx);  /* push null and delete all arguments */
      else {
        if (borders) {  /* 1.10.0 */
          size_t start = ss_len-pos-sp_len+2;
          dogc = agenaV_setpairints(L, idx, start, start + sp_len - 1);
        } else
          setnvalue(idx, ss_len-pos-sp_len+2);
      }
      xfree(ss); xfree(sp);  /* Agena 1.0.4/1.10.4 */
    } else {
      int anchor;
      const char *s1, *res;
      MatchState ms;
      init = posrelat(init, s_len) - 1;
      if (init < 0) init = 0;
      if ((size_t)init >= s_len) {
        for (i=0; i < nargs; i++) setnilvalue(idx+i);  /* push null and delete all arguments */
        return dogc;
      }
      /* checking with strpbrk for special characters gives no gain in speed */
      anchor = (*p == '^') ? (p++, 1) : 0;
      s1 = s+init;
      ms.L = L;
      ms.src_init = s;
      ms.src_end = s+strlen(s);
      do {
        ms.level = 0;
        if ((res=match(&ms, s1, p, 0)) != NULL) {
          if (borders) {  /* 1.10.0 */
            dogc = agenaV_setpairints(L, idx, s1-s+1, res-s);
          } else
            setnvalue(idx, s1-s+1);   /* start */
          /* setnvalue(idx+1, res-s);  end */
          break;
        }
      } while (s1++ < ms.src_end && !anchor);
      if (res == NULL) { setnilvalue(idx); }  /* 1.3.3 fix */
    }
  } else {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": strings expected.", "instr");
  }
  for (i=1; i < nargs; i++) setnilvalue(idx+i);  /* pattern not found - delete all arguments, set nulls instead */
  lua_unlock(L);
  /* do not change L->top here since this will confuse the stack */
  return dogc;  /* 1 = pushed a pair, conduct gc, 0 = pushed no pair, no gc */
}


/* VALUES operator, 1.3.0, 02.01.2011 */

void agenaV_values (lua_State *L, StkId idx, int nargs) {
  int i, index;
  lua_lock(L);
  if (nargs == 0) {  /* are there multiple returns ? */
    nargs = cast_int(L->top - idx);
    L->top = L->ci->top;
  } else  /* when called, nargs is number of arguments + 1, so change it to actual number of arguments */
    nargs -= 1;
  if (nargs < 2) {
    lua_unlock(L);
    luaG_runerror(L, "in " LUA_QS ": at least two arguments expected, got %d.", "values", nargs);
  }
  switch (ttype(idx)) {
    case LUA_TSEQ: {
      Seq *s, *newseq;
      TValue *v;
      int c;
      c = 0;
      s = seqvalue(idx);
      /* stack must be increased to avoid `invalid key to next` errors */
      api_check(L, L->top < L->ci->top);
      newseq = agnSeq_new(L, nargs - 1);
      /* traverse sequence (first arg) */
      for (i=1; i < nargs; i++) {
        if (ttisnumber(idx+i)) {
          index = posrelat((int)nvalue(idx+i), s->size) - 1;
          if (index < 0 || index >= s->size) {
            lua_unlock(L);
            luaG_runerror(L, "in " LUA_QS ": index %d out of range.", "values", index+1, nargs);
          }
          c++;
          v = seqitem(s, index);
          agnSeq_seti(L, newseq, c, v);
          luaC_barrierseq(L, newseq, v);
        }
      }
      /* prepare sequence for return to Agena environment */
      setseqvalue(L, idx, newseq);
      break;
    }
    case LUA_TTABLE: {
      Table *h, *newtable;
      h = hvalue(idx);
      /* stack must be increased to avoid `invalid key to next` errors */
      api_check(L, L->top < L->ci->top);
      newtable = luaH_new(L, nargs - 1, nargs - 1);
      /* traverse first table */
      for (i=1; i < nargs; i++) {
        const TValue *k = luaH_get(h, idx+i);  /* do a primitive get */
        TValue *oldval = luaH_set(L, newtable, idx+i);  /* do a primitive set */
        setobj2t(L, oldval, k);
        luaC_barriert(L, newtable, k);
      }
      /* prepare table for return to Agena environment */
      sethvalue(L, idx, newtable);
      break;
    }
    default: {
      lua_unlock(L);
      luaG_runerror(L, "in " LUA_QS ": table or sequence expected as first argument, got %s.", "values",
        luaT_typenames[(int)ttype(idx)]);
      }
  }
  for (i=1; i < nargs; i++) setnilvalue(idx+i);  /* delete second to n-th argument */
  lua_unlock(L);
  /* do not change L->top here since this will confuse the stack */
}


void agenaV_log (lua_State *L, StkId ra, StkId rb) {
  if (ttisnumber(ra) && ttisnumber(rb)) {
    lua_Number x, b;
    x = nvalue(ra); b = nvalue(rb);
    if (x == 1 && b < 0) {
      setnvalue(ra, 0);
    }
    else if (x == b && (x < 0 || x > 1)) {
      setnvalue(ra, 1);
    }
    else if (x <= 0 || b <= 0 || b == 1) {
      setnvalue(ra, AGN_NAN);
    }
    else {
      setnvalue(ra, log(x)/log(b));
    }
  }
  else if (ttiscomplex(ra) && ttisnumber(rb)) {
#ifndef PROPCMPLX
    agn_Complex z, b;
    z = cvalue(ra);
    b = nvalue(rb)+I*0;
    if (z == 0+0*I || b == 0+0*I || b == 1+0*I)
      setcvalue(ra, AGN_NAN)
    else
      setcvalue(ra, clog(z)/clog(b));
#else
/* > Re(t); 1;
     /  2     2\   /  2     2\
   ln\a0  + a1 / ln\b0  + b1 / + 4 argument(a0 + I a1) argument(b0 + I b1)
   -----------------------------------------------------------------------
                                2
                     /  2     2\                         2
                   ln\b0  + b1 /  + 4 argument(b0 + I b1)
> Im(t); 1;
     /  /  2     2\                                             /  2     2\\
   2 \ln\a0  + a1 / argument(b0 + I b1) - argument(a0 + I a1) ln\b0  + b1 //
 - -------------------------------------------------------------------------
                                 2
                      /  2     2\                         2
                    ln\b0  + b1 /  + 4 argument(b0 + I b1)
*/
    lua_Number a[2], b[2], z[2];
    a[0] = complexreal(ra); a[1] = compleximag(ra);
    b[0] = nvalue(rb); b[1] = 0;
    if ((a[0] == 0 && a[1] == 0) || b[0] == 0 || b[0] == 1) {
      z[0] = AGN_NAN; z[1] = AGN_NAN;
      setcvalue(ra, z);
    }
    else {
      lua_Number _a, _b, ta, tb, td, la, lb;
      _a = a[0]*a[0]+a[1]*a[1]; _b = b[0]*b[0]+b[1]*b[1];
      ta = atan2(a[1], a[0]); tb = atan2(b[1], b[0]);
      la = log(_a); lb = log(_b); td = lb*lb+4*tb*tb;
      z[0] = (la*lb+4*ta*tb) / td; z[1] = -2*(la*tb-ta*lb) / td;
      setcvalue(ra, z);
    }
#endif
  }
  else if (ttisnumber(ra) && ttiscomplex(rb)) {
#ifndef PROPCMPLX
    agn_Complex z, b;
    z = nvalue(ra)+I*0;
    b = cvalue(rb);
    if (z == 0+0*I || b == 0+0*I || b == 1+0*I)
      setcvalue(ra, AGN_NAN)
    else
      setcvalue(ra, clog(z)/clog(b));
#else
    lua_Number a[2], b[2], z[2];
    a[0] = nvalue(ra); a[1] = 0;
    b[0] = complexreal(rb); b[1] = compleximag(rb);
    if ((a[0] == 0 && a[1] == 0) || b[0] == 0 || b[0] == 1) {
      z[0] = AGN_NAN; z[1] = AGN_NAN;
      setcvalue(ra, z);
    }
    else {
      lua_Number _a, _b, ta, tb, td, la, lb;
      _a = a[0]*a[0]+a[1]*a[1];
      _b = b[0]*b[0]+b[1]*b[1];
      ta = atan2(a[1], a[0]);
      tb = atan2(b[1], b[0]);
      la = log(_a); lb = log(_b);
      td = lb*lb+4*tb*tb;
      z[0] = (la*lb+4*ta*tb) / td;
      z[1] = -2*(la*tb-ta*lb) / td;
      setcvalue(ra, z);
    }
#endif
  }
  else if (ttiscomplex(ra) && ttiscomplex(rb)) {
#ifndef PROPCMPLX
    agn_Complex z, b;
    z = cvalue(ra);
    b = cvalue(rb);
    if (z == 0+0*I || b == 0+0*I || b == 1+0*I)
      setcvalue(ra, AGN_NAN)
    else
      setcvalue(ra, clog(z)/clog(b));
#else
    lua_Number a[2], b[2], z[2];
    a[0] = complexreal(ra); a[1] = compleximag(ra);
    b[0] = complexreal(rb); b[1] = compleximag(rb);
    if ((a[0] == 0 && a[1] == 0) || b[0] == 0 || b[0] == 1) {
      z[0] = AGN_NAN; z[1] = AGN_NAN;
      setcvalue(ra, z);
    }
    else {
      lua_Number _a, _b, ta, tb, td, la, lb;
      _a = a[0]*a[0]+a[1]*a[1];
      _b = b[0]*b[0]+b[1]*b[1];
      ta = atan2(a[1], a[0]);
      tb = atan2(b[1], b[0]);
      la = log(_a); lb = log(_b);
      td = lb*lb+4*tb*tb;
      z[0] = (la*lb+4*ta*tb) / td;
      z[1] = -2*(la*tb-ta*lb) / td;
      setcvalue(ra, z);
    }
#endif
  }
  else {
    luaG_runerror(L, "in " LUA_QS ": two numbers or complex numbers expected.", "log");
  }
}


/* directly including the lpair.c code into this function is not much faster */
void agenaV_pair (lua_State *L, const TValue *tbl1, const TValue *tbl2, StkId idx) {
  Pair *eq;
  lua_lock(L);
  eq = agnPair_create(L, tbl1, tbl2);
  setpairvalue(L, idx, eq);
  lua_unlock(L);
}


static int call_binTM (lua_State *L, const TValue *p1, const TValue *p2,
                       StkId res, TMS event) {
  const TValue *tm = luaT_gettmbyobj(L, p1, event);  /* try first operand */
  if (ttisnil(tm))
    tm = luaT_gettmbyobj(L, p2, event);  /* try second operand */
  if (ttisnil(tm)) return 0;  /* Lua 5.1.2 patch */
  callTMres(L, res, tm, p1, p2);
  return 1;
}


static const TValue *get_compTM (lua_State *L, Table *mt1, Table *mt2,
                                  TMS event) {
  const TValue *tm1 = fasttm(L, mt1, event);
  const TValue *tm2;
  if (tm1 == NULL) return NULL;  /* no metamethod */
  if (mt1 == mt2) return tm1;  /* same metatables => same metamethods */
  tm2 = fasttm(L, mt2, event);
  if (tm2 == NULL) return NULL;  /* no metamethod */
  if (luaO_rawequalObj(tm1, tm2))  /* same metamethods? */
    return tm1;
  return NULL;
}


static int call_orderTM (lua_State *L, const TValue *p1, const TValue *p2,
                         TMS event) {
  const TValue *tm1 = luaT_gettmbyobj(L, p1, event);
  const TValue *tm2;
  if (ttisnil(tm1)) return -1;  /* no metamethod? */
  tm2 = luaT_gettmbyobj(L, p2, event);
  if (!luaO_rawequalObj(tm1, tm2))  /* different metamethods? */
    return -1;
  callTMres(L, L->top, tm1, p1, p2);
  return !l_isfalse(L->top);
}


static int l_strcmp (const TString *ls, const TString *rs) {
  const char *l = getstr(ls);
  size_t ll = ls->tsv.len;
  const char *r = getstr(rs);
  size_t lr = rs->tsv.len;
  for (;;) {
    int temp = strcoll(l, r);
    if (temp != 0) return temp;
    else {  /* strings are equal up to a `\0' */
      size_t len = strlen(l);  /* index of first `\0' in both strings */
      if (len == lr)  /* r is finished? */
        return (len == ll) ? 0 : 1;
      else if (len == ll)  /* l is finished? */
        return -1;  /* l is smaller than r (because r is not finished) */
      /* both strings longer than `len'; go on comparing (after the `\0') */
      len++;
      l += len; ll -= len; r += len; lr -= len;
    }
  }
}


int luaV_lessthan (lua_State *L, const TValue *l, const TValue *r) {
  int res;
  if (ttype(l) != ttype(r))
    return luaG_ordererror(L, l, r);
  else if (ttisnumber(l))
    return luai_numlt(nvalue(l), nvalue(r));
  else if (ttisstring(l))
    return l_strcmp(rawtsvalue(l), rawtsvalue(r)) < 0;
  else if ((res = call_orderTM(L, l, r, TM_LT)) != -1)
    return res;
  return luaG_ordererror(L, l, r);
}


static int lessequal (lua_State *L, const TValue *l, const TValue *r) {
  int res;
  if (ttype(l) != ttype(r))
    return luaG_ordererror(L, l, r);
  else if (ttisnumber(l))
    return luai_numle(nvalue(l), nvalue(r));
  else if (ttisstring(l))
    return l_strcmp(rawtsvalue(l), rawtsvalue(r)) <= 0;
  else if ((res = call_orderTM(L, l, r, TM_LE)) != -1)  /* first try `le' */
    return res;
  else if ((res = call_orderTM(L, r, l, TM_LT)) != -1)  /* else try `lt' */
    return !res;
  return luaG_ordererror(L, l, r);
}


int luaV_equalval (lua_State *L, const TValue *t1, const TValue *t2) {
  const TValue *tm;
  lua_assert(ttype(t1) == ttype(t2));
  switch (ttype(t1)) {
    case LUA_TNIL: return 1;
    case LUA_TNUMBER: return luai_numeq(nvalue(t1), nvalue(t2));
#ifndef PROPCMPLX
    case LUA_TCOMPLEX: return agnc_eq(cvalue(t1), cvalue(t2));
#else
    case LUA_TCOMPLEX: return agnc_eq(complexreal(t1), compleximag(t1), complexreal(t2), compleximag(t2));
#endif
    case LUA_TBOOLEAN: return bvalue(t1) == bvalue(t2);  /* true must be 1 !! */
    case LUA_TLIGHTUSERDATA: return pvalue(t1) == pvalue(t2);
    case LUA_TUSERDATA: {
      if (uvalue(t1) == uvalue(t2)) return 1;
      tm = get_compTM(L, uvalue(t1)->metatable, uvalue(t2)->metatable,
                         TM_EQ);
      break;  /* will try TM */
    }
    case LUA_TTABLE: {
      if (hvalue(t1) == hvalue(t2)) return 1;
      /* compare contents of two tables for equality, added 0.6.0 */
      if (agenaV_comptables(L, hvalue(t1), hvalue(t2), 1)) return 1;
      tm = get_compTM(L, hvalue(t1)->metatable, hvalue(t2)->metatable, TM_EQ);
      break;  /* will try TM */
    }
    case LUA_TSET: {
      if (usvalue(t1) == usvalue(t2)) return 1;
      /* compare contents of two tables for equality, added 0.6.0 */
      if (agenaV_compusets(L, usvalue(t1), usvalue(t2), 1)) return 1;
      tm = get_compTM(L, usvalue(t1)->metatable, usvalue(t2)->metatable, TM_EQ);
      break;  /* will try TM */
    }
    case LUA_TSEQ: {
      if (seqvalue(t1) == seqvalue(t2)) return 1;
      if (agenaV_compseqs(L, seqvalue(t1), seqvalue(t2), 1)) return 1;
      tm = get_compTM(L, seqvalue(t1)->metatable, seqvalue(t2)->metatable, TM_EQ);
      break;
    }
    case LUA_TPAIR: {
      if (pairvalue(t1) == pairvalue(t2)) return 1;
      if (agenaV_comppairs(L, pairvalue(t1), pairvalue(t2))) return 1;
      tm = get_compTM(L, pairvalue(t1)->metatable, pairvalue(t2)->metatable, TM_EQ);
      break;
    }
    case LUA_TFAIL: return 1;
    default: return gcvalue(t1) == gcvalue(t2);
  }
  if (tm == NULL) return 0;  /* no TM? */
  callTMres(L, L->top, tm, t1, t2);  /* call TM */
  return !l_isfalseorfail(L->top);
}


int luaV_equalvalonebyone (lua_State *L, const TValue *t1, const TValue *t2) {
  const TValue *tm;
  lua_assert(ttype(t1) == ttype(t2));
  switch (ttype(t1)) {
    case LUA_TTABLE: {
      if (hvalue(t1) == hvalue(t2)) return 1;
      /* compare contents of two tables for equality, added 0.6.0 */
      if (agenaV_comptablesonebyone(L, hvalue(t1), hvalue(t2))) return 1;
      tm = get_compTM(L, hvalue(t1)->metatable, hvalue(t2)->metatable, TM_EEQ);
      break;  /* will try TM */
    }
    case LUA_TSEQ: {
      if (seqvalue(t1) == seqvalue(t2)) return 1;
      if (agenaV_compseqsonebyone(L, seqvalue(t1), seqvalue(t2))) return 1;
      tm = get_compTM(L, seqvalue(t1)->metatable, seqvalue(t2)->metatable, TM_EEQ);
      break;
    }
    default: return luaV_equalval(L, t1, t2);
  }
  if (tm == NULL) return 0;  /* no TM? */
  callTMres(L, L->top, tm, t1, t2);  /* call TM */
  return !l_isfalseorfail(L->top);
}


int luaV_equalncomplex (lua_State *L, const TValue *t1, const TValue *t2) {
  if (ttisnumber(t1) && ttiscomplex(t2)) {
#ifndef PROPCMPLX
    return agnc_eq(nvalue(t1), cvalue(t2));
#else
    return agnc_eq(nvalue(t1), 0, complexreal(t2), compleximag(t2));
#endif
  }
  else if (ttiscomplex(t1) && ttisnumber(t2)) {
#ifndef PROPCMPLX
    return agnc_eq(cvalue(t1), nvalue(t2));
#else
    return agnc_eq(complexreal(t1), compleximag(t1), nvalue(t2), 0);
#endif
  }
  return 0;
}


void luaV_concat (lua_State *L, int total, int last) {
  do {
    StkId top = L->base + last + 1;
    int n = 2;  /* number of elements handled in this pass (at least 2) */
    if (!(ttisstring(top-2) || ttisnumber(top-2)) || !tostring(L, top-1)) {  /* Lua 5.1.2 patch */
      if (!call_binTM(L, top-2, top-1, top-2, TM_CONCAT)) {
        luaG_concaterror(L, top-2, top-1);
      }
    } else if (tsvalue(top-1)->len == 0) { /* second op is empty?   Lua 5.1.2 patch */
      (void)tostring(L, top - 2);  /* result is first op (as string) */
    }
    else {
      /* at least two string values; get as many as possible */
      size_t tl = tsvalue(top-1)->len;
      char *buffer;
      int i;
      /* collect total length */
      for (n = 1; n < total && tostring(L, top-n-1); n++) {
        size_t l = tsvalue(top-n-1)->len;
        if (l >= MAX_SIZET - tl) luaG_runerror(L, "string length overflow.");
        tl += l;
      }
      buffer = luaZ_openspace(L, &G(L)->buff, tl);
      tl = 0;
      for (i=n; i>0; i--) {  /* concat all strings */
        size_t l = tsvalue(top-i)->len;
        memcpy(buffer+tl, svalue(top-i), l);
        tl += l;
      }
      setsvalue2s(L, top-n, luaS_newlstr(L, buffer, tl));
    }
    total -= n-1;  /* got `n' strings to create 1 new */
    last -= n-1;
  } while (total > 1);  /* repeat until only 1 result left */
}


#define Protect(x)   { L->savedpc = pc; {x;}; base = L->base; }

static void Arith (lua_State *L, StkId ra, const TValue *rb,
                   const TValue *rc, TMS op) {  /* changed 0.5.3 */
  if (ttisnumber(rb) && ttisnumber(rc)) {
    lua_Number nb = nvalue(rb), nc = nvalue(rc);
    switch (op) {
      case TM_ADD: setnvalue(ra, luai_numadd(nb, nc)); break;
      case TM_SUB: setnvalue(ra, luai_numsub(nb, nc)); break;
      case TM_MUL: setnvalue(ra, luai_nummul(nb, nc)); break;
      case TM_DIV: setnvalue(ra, luai_numdiv(nb, nc)); break;
      case TM_MOD: setnvalue(ra, luai_nummod(nb, nc)); break;
      case TM_POW: setnvalue(ra, luai_numpow(nb, nc)); break;
      case TM_UNM: setnvalue(ra, luai_numunm(nb)); break;
      case TM_INTDIV: setnvalue(ra, luai_numintdiv(nb, nc)); break;  /* added 0.5.4 */
      default:  /* 0.26.1 patch */
        if (!call_binTM(L, rb, rc, ra, op))
          luaG_aritherror(L, rb, rc); break;
    }
  }
#ifndef PROPCMPLX
  else if (ttiscomplex(rb) && ttiscomplex(rc)) {
    agn_Complex nb, nc;
    nb = cvalue(rb); nc = cvalue(rc);
    switch (op) {
      case TM_ADD: setcvalue(ra, agnc_add(nb, nc)); break;
      case TM_SUB: setcvalue(ra, agnc_sub(nb, nc)); break;
      case TM_MUL: setcvalue(ra, agnc_mul(nb, nc)); break;
      case TM_DIV: setcvalue(ra, agnc_div(nb, nc)); break;
      case TM_POW: setcvalue(ra, agnc_pow(nb, nc)); break;
      case TM_UNM: setcvalue(ra, agnc_unm(nb)); break;
      default:  /* 0.26.1 patch */
        if (!call_binTM(L, rb, rc, ra, op))
          luaG_aritherror(L, rb, rc); break;
    }
  }
  else if (ttiscomplex(rb) && ttisnumber(rc)) {
    agn_Complex nb, nc;
    nb = cvalue(rb); nc = nvalue(rc);
    switch (op) {
      case TM_ADD: setcvalue(ra, agnc_add(nb, nc)); break;
      case TM_SUB: setcvalue(ra, agnc_sub(nb, nc)); break;
      case TM_MUL: setcvalue(ra, agnc_mul(nb, nc)); break;
      case TM_DIV: setcvalue(ra, agnc_div(nb, nc)); break;
      case TM_POW: setcvalue(ra, agnc_pow(nb, nc)); break;
      case TM_UNM: setcvalue(ra, agnc_unm(nb)); break;
      default:  /* 0.26.1 patch */
        if (!call_binTM(L, rb, rc, ra, op))
          luaG_aritherror(L, rb, rc); break;
    }
  }
  else if (ttisnumber(rb) && ttiscomplex(rc)) {
    agn_Complex nb, nc;
    nb = nvalue(rb); nc = cvalue(rc);
    switch (op) {
      case TM_ADD: setcvalue(ra, agnc_add(nb, nc)); break;
      case TM_SUB: setcvalue(ra, agnc_sub(nb, nc)); break;
      case TM_MUL: setcvalue(ra, agnc_mul(nb, nc)); break;
      case TM_DIV: setcvalue(ra, agnc_div(nb, nc)); break;
      case TM_POW: setcvalue(ra, agnc_pow(nb, nc)); break;
      case TM_UNM: setcvalue(ra, agnc_unm(nb)); break;
      default:  /* 0.26.1 patch */
        if (!call_binTM(L, rb, rc, ra, op))
          luaG_aritherror(L, rb, rc); break;
    }
  }
#else
  else if (ttiscomplex(rb) && ttiscomplex(rc)) {
    lua_Number nb1, nb2, nc1, nc2, z[2];
    nb1 = complexreal(rb); nb2 = compleximag(rb);
    nc1 = complexreal(rc); nc2 = compleximag(rc);
    switch (op) {
      case TM_ADD: agnc_add(z, nb1, nb2, nc1, nc2); setcvalue(ra, z); break;
      case TM_SUB: agnc_sub(z, nb1, nb2, nc1, nc2); setcvalue(ra, z); break;
      case TM_MUL: agnc_mul(z, nb1, nb2, nc1, nc2); setcvalue(ra, z); break;
      case TM_DIV: agnc_div(z, nb1, nb2, nc1, nc2); setcvalue(ra, z); break;
      case TM_POW: agnc_pow(z, nb1, nb2, nc1, nc2); setcvalue(ra, z); break;
      case TM_UNM: agnc_unm(z, nb1, nb2); setcvalue(ra, z); break;
      default:
        if (!call_binTM(L, rb, rc, ra, op))
          luaG_aritherror(L, rb, rc); break;
    }
    luaC_checkGC(L);
  }
  else if (ttiscomplex(rb) && ttisnumber(rc)) {
    lua_Number nb1, nb2, nc, z[2];
    nb1 = complexreal(rb); nb2 = compleximag(rb);
    nc = nvalue(rc);
    switch (op) {
      case TM_ADD: agnc_add(z, nb1, nb2, nc, 0); setcvalue(ra, z); break;
      case TM_SUB: agnc_sub(z, nb1, nb2, nc, 0); setcvalue(ra, z); break;
      case TM_MUL: agnc_mul(z, nb1, nb2, nc, 0); setcvalue(ra, z); break;
      case TM_DIV: agnc_div(z, nb1, nb2, nc, 0); setcvalue(ra, z); break;
      case TM_POW: agnc_pow(z, nb1, nb2, nc, 0); setcvalue(ra, z); break;
      case TM_UNM: agnc_unm(z, nb1, nb2); setcvalue(ra, z); break;
      default:
        if (!call_binTM(L, rb, rc, ra, op))
          luaG_aritherror(L, rb, rc); break;
    }
    luaC_checkGC(L);
  }
  else if (ttisnumber(rb) && ttiscomplex(rc)) {
    lua_Number nb, nc1, nc2, z[2];
    nb = nvalue(rb); nc1 = complexreal(rc); nc2 = compleximag(rc);
    switch (op) {
      case TM_ADD: agnc_add(z, nb, 0, nc1, nc2); setcvalue(ra, z); break;
      case TM_SUB: agnc_sub(z, nb, 0, nc1, nc2); setcvalue(ra, z); break;
      case TM_MUL: agnc_mul(z, nb, 0, nc1, nc2); setcvalue(ra, z); break;
      case TM_DIV: agnc_div(z, nb, 0, nc1, nc2); setcvalue(ra, z); break;
      case TM_POW: agnc_pow(z, nb, 0, nc1, nc2); setcvalue(ra, z); break;
      case TM_UNM: agnc_unm(z, nb, 0); setcvalue(ra, z); break;
      default:
        if (!call_binTM(L, rb, rc, ra, op))
          luaG_aritherror(L, rb, rc); break;
    }
    luaC_checkGC(L);
  }
#endif
  else if (!call_binTM(L, rb, rc, ra, op))
    luaG_aritherror(L, rb, rc);
}



/*
** some macros for common tasks in `luaV_execute'
*/

#define runtime_check(L, c)   { if (!(c)) break; }

#define RA(i)   (base+GETARG_A(i))
/* to be used after possible stack reallocation */
#define RB(i)   check_exp(getBMode(GET_OPCODE(i)) == OpArgR, base+GETARG_B(i))
#define RC(i)   check_exp(getCMode(GET_OPCODE(i)) == OpArgR, base+GETARG_C(i))
#define RKB(i)   check_exp(getBMode(GET_OPCODE(i)) == OpArgK, \
   ISK(GETARG_B(i)) ? k+INDEXK(GETARG_B(i)) : base+GETARG_B(i))
#define RKC(i)   check_exp(getCMode(GET_OPCODE(i)) == OpArgK, \
   ISK(GETARG_C(i)) ? k+INDEXK(GETARG_C(i)) : base+GETARG_C(i))
#define KBx(i)   check_exp(getBMode(GET_OPCODE(i)) == OpArgK, k+GETARG_Bx(i))

#ifndef PROPCMPLX
#define arith_op(L,op,opc,tm) { \
        TValue *rb = RKB(i); \
        TValue *rc = RKC(i); \
        if (ttisnumber(rb) && ttisnumber(rc)) { \
          lua_Number nb = nvalue(rb), nc = nvalue(rc); \
          setnvalue(ra, op(nb, nc)); \
        } \
        else if (ttiscomplex(rb) && ttiscomplex(rc)) { \
          agn_Complex nb = cvalue(rb), nc = cvalue(rc); \
          setcvalue(ra, opc(nb, nc)); \
        } \
        else if (ttiscomplex(rb) && ttisnumber(rc)) { \
          agn_Complex nb = cvalue(rb), nc = nvalue(rc); \
          setcvalue(ra, opc(nb, nc)); \
        } \
        else if (ttisnumber(rb) && ttiscomplex(rc)) { \
          agn_Complex nb = nvalue(rb), nc = cvalue(rc); \
          setcvalue(ra, opc(nb, nc)); \
        } \
        else \
          Protect(Arith(L, ra, rb, rc, tm)); \
      }
#else
#define arith_op(L,op,opc,tm) { \
        TValue *rb = RKB(i); \
        TValue *rc = RKC(i); \
        if (ttisnumber(rb) && ttisnumber(rc)) { \
          lua_Number nb = nvalue(rb), nc = nvalue(rc); \
          setnvalue(ra, op(nb, nc)); \
        } \
        else if (ttiscomplex(rb) && ttiscomplex(rc)) { \
          lua_Number z[2]; \
          lua_Number nb1 = complexreal(rb), nb2 = compleximag(rb), \
                     nc1 = complexreal(rc), nc2 = compleximag(rc); \
          opc(z, nb1, nb2, nc1, nc2); \
          setcvalue(ra, z); \
        } \
        else if (ttiscomplex(rb) && ttisnumber(rc)) { \
          lua_Number z[2]; \
          lua_Number nb1 = complexreal(rb), nb2 = compleximag(rb), nc = nvalue(rc); \
          opc(z, nb1, nb2, nc, 0); \
          setcvalue(ra, z); \
        } \
        else if (ttisnumber(rb) && ttiscomplex(rc)) { \
          lua_Number z[2]; \
          lua_Number nb = nvalue(rb), nc1 = complexreal(rc), nc2 = compleximag(rc); \
          opc(z, nb, 0, nc1, nc2); \
          setcvalue(ra, z); \
        } \
        else \
          Protect(Arith(L, ra, rb, rc, tm)); \
      }
#endif


#define arith_opNumber(op,tm) { \
        TValue *rb = RKB(i); \
        TValue *rc = RKC(i); \
        if (ttisnumber(rb) && ttisnumber(rc)) { \
          lua_Number nb = nvalue(rb), nc = nvalue(rc); \
          setnvalue(ra, op(nb, nc)); \
        } \
        else \
          Protect(Arith(L, ra, rb, rc, tm)); \
      }


#define arith_opfnNumber(op,tm) { \
        TValue *rb = RKB(i); \
        if (ttisnumber(rb)) { \
          lua_Number nb = nvalue(rb); \
          setnvalue(ra, op(nb)); \
        } \
        else \
          Protect(Arith(L, ra, rb, rb, tm)); \
      }

#ifndef PROPCMPLX
#define arith_opfn(L,op,opc,tm) { \
        TValue *rb = RKB(i); \
        if (ttisnumber(rb)) { \
          lua_Number nb = nvalue(rb); \
          setnvalue(ra, op(nb)); \
        } \
        else if (ttiscomplex(rb)) { \
          agn_Complex nc = cvalue(rb); \
          setcvalue(ra, opc(nc)); \
        } \
        else \
          Protect(Arith(L, ra, rb, rb, tm)); \
      }
#else
#define arith_opfn(L,op,opc,tm) { \
        TValue *rb = RKB(i); \
        if (ttisnumber(rb)) { \
          lua_Number nb = nvalue(rb); \
          setnvalue(ra, op(nb)); \
        } \
        else if (ttiscomplex(rb)) { \
          lua_Number z[2]; \
          lua_Number a, b; \
          a = complexreal(rb); \
          b = compleximag(rb); \
          opc(z, a, b); \
          setcvalue(ra, z); \
        } \
        else \
          Protect(Arith(L, ra, rb, rb, tm)); \
      }
#endif


#define dojump(L,pc,i)   {(pc) += (i); luai_threadyield(L);}


static void releasetry(lua_State *L) {  /* 2.1 RC 2, written by Hu Qiwei */
  struct lua_longjmp *pj = L->errorJmp;
  if (pj->type == JMPTYPE_TRY) {
    L->errfunc = pj->old_errfunc;
    L->errorJmp = pj->previous;
    luaM_free(L, pj);
  }
}


static void restoretry (lua_State *L, int seterr, int ra) {  /* 2.1 RC 2, written by Hu Qiwei */
  struct lua_longjmp *pj = L->errorJmp;

  StkId oldtop = restorestack(L, pj->old_top);
  luaF_close(L, oldtop);  /* close eventual pending closures */

  L->nCcalls = pj->oldnCcalls;
  L->ci = restoreci(L, pj->old_ci);
  L->base = L->ci->base;
  L->allowhook = pj->old_allowhooks;

  if (seterr)
    luaD_seterrorobj(L, pj->status, L->base + ra);
  L->top = oldtop;

  if (L->size_ci > LUAI_MAXCALLS) {  /* there was an overflow? */
    int inuse = cast_int(L->ci - L->base_ci);
    if (inuse + 1 < LUAI_MAXCALLS)  /* can `undo' overflow? */
      luaD_reallocCI(L, LUAI_MAXCALLS);
  }
  releasetry(L);
}


void luaV_execute (lua_State *L, int nexeccalls) {
  LClosure *cl;
  StkId base;
  TValue *k;
  const Instruction *pc;
  struct lua_longjmp *pj;  /* 2.1 RC 2, written by Hu Qiwei */
 reentry:  /* entry point */
  lua_assert(isLua(L->ci));
  pc = L->savedpc;
  cl = &clvalue(L->ci->func)->l;
  base = L->base;
  k = cl->p->k;
  /* main loop of interpreter */
  for (;;) {
    const Instruction i = *pc++;
    StkId ra;
    if ((L->hookmask & (LUA_MASKLINE | LUA_MASKCOUNT)) &&
        (--L->hookcount == 0 || L->hookmask & LUA_MASKLINE)) {
      traceexec(L, pc);
      if (L->status == LUA_YIELD) {  /* did hook yield? */
        L->savedpc = pc - 1;
        return;
      }
      base = L->base;
    }
    /* warning!! several calls may realloc the stack and invalidate `ra' */
    ra = RA(i);
    lua_assert(base == L->base && L->base == L->ci->base);
    lua_assert(base <= L->top && L->top <= L->stack + L->stacksize);
    lua_assert(L->top == L->ci->top || luaG_checkopenop(i));
    switch (GET_OPCODE(i)) {
      case OP_MOVE: {
        setobjs2s(L, ra, RB(i));
        continue;
      }
      case OP_LOADK: {
        setobj2s(L, ra, KBx(i));
        continue;
      }
      case OP_LOADBOOL: {
        int b = GETARG_B(i);  /* value b must be nonnegative, although declared signed */
        if (b < 2) {
          setbvalue(ra, b);
        }
        else {
          setfailvalue(ra);  /* 0.9.2 */
        }
        if (GETARG_C(i)) pc++;  /* skip next instruction (if C) */
        continue;
      }
      case OP_LOADNIL: {
        TValue *rb = RB(i);
        do {
          setnilvalue(rb--);
        } while (rb >= ra);
        continue;
      }
      case OP_GETUPVAL: {
        int b = GETARG_B(i);
        setobj2s(L, ra, cl->upvals[b]->v);
        continue;
      }
      case OP_GETGLOBAL: {
        TValue g;
        TValue *rb = KBx(i);
        sethvalue(L, &g, cl->env);
        lua_assert(ttisstring(rb));
        Protect(luaV_gettable(L, &g, rb, ra));
        continue;
      }
      case OP_GETTABLE: {
        Protect(luaV_gettable(L, RB(i), RKC(i), ra));
        continue;
      }
      case OP_IN: {  /* 0.7.1, November 23, 2007 */
        Protect(agenaV_in(L, RKB(i), RKC(i), ra));
        continue;
      }
      case OP_TSUBSET: {
        Protect(agenaV_subset(L, RKB(i), RKC(i), ra, 0));
        continue;
      }
      case OP_TXSUBSET: {  /* 0.9.1, January 12, 2008 */
        Protect(agenaV_subset(L, RKB(i), RKC(i), ra, 2));
        continue;
      }
      case OP_SETGLOBAL: {
        TValue g;
        sethvalue(L, &g, cl->env);
        lua_assert(ttisstring(KBx(i)));
        Protect(luaV_settable(L, &g, KBx(i), ra));
        continue;
      }
      case OP_SETUPVAL: {
        UpVal *uv = cl->upvals[GETARG_B(i)];
        setobj(L, uv->v, ra);
        luaC_barrier(L, uv, ra);
        continue;
      }
      case OP_SETTABLE: {
        Protect(luaV_settable(L, ra, RKB(i), RKC(i)));
        continue;
      }
      case OP_NEWTABLE: {
        int b = GETARG_B(i);
        int c = GETARG_C(i);
        sethvalue(L, ra, luaH_new(L, luaO_fb2int(b), luaO_fb2int(c)));
        Protect(luaC_checkGC(L));
        continue;
      }
      case OP_TINSERT: {
        Protect(agenaV_insert(L, ra, RKB(i), GETARG_C(i)); luaC_checkGC(L));
        continue;
      }
      case OP_TDELETE: {
        Protect(agenaV_delete(L, ra, RKB(i), GETARG_C(i)); luaC_checkGC(L));
        continue;
      }
      case OP_TUNION: {
        Protect(agenaV_tableunion(L, RKB(i), RKC(i), ra); luaC_checkGC(L));
        continue;
      }
      case OP_TMINUS: {
        Protect(agenaV_setops(L, RKB(i), RKC(i), ra, 1); luaC_checkGC(L));
        continue;
      }
      case OP_TINTERSECT: {
        Protect(agenaV_setops(L, RKB(i), RKC(i), ra, 0); luaC_checkGC(L));
        continue;
      }
      case OP_SELF: {
        StkId rb = RB(i);
        setobjs2s(L, ra+1, rb);
        Protect(luaV_gettable(L, rb, RKC(i), ra));
        continue;
      }
      case OP_ADD: {
        arith_op(L, luai_numadd, agnc_add, TM_ADD);
        continue;
      }
      case OP_SUB: {
        arith_op(L, luai_numsub, agnc_sub, TM_SUB);
        continue;
      }
      case OP_MUL: {
        arith_op(L, luai_nummul, agnc_mul, TM_MUL);
        continue;
      }
      case OP_DIV: {
        arith_op(L, luai_numdiv, agnc_div, TM_DIV);
        continue;
      }
      case OP_MOD: {
        arith_opNumber(luai_nummod, TM_MOD);
        continue;
      }
      case OP_POW: {
        arith_op(L, luai_numpow, agnc_pow, TM_POW);
        continue;
      }
      case OP_IPOW: {
        arith_op(L, luai_numipow, agnc_ipow, TM_IPOW);  /* 0.22.2 */
        continue;
      }
      case OP_UNM: {  /* unary minus */
        TValue *rb = RB(i);
        if (ttisnumber(rb)) {
          lua_Number nb = nvalue(rb);
          setnvalue(ra, luai_numunm(nb));
        }
        else if (ttiscomplex(rb)) {
#ifndef PROPCMPLX
          agn_Complex nb = cvalue(rb);
          setcvalue(ra, agnc_unm(nb));
#else
          lua_Number nb1 = complexreal(rb), nb2 = compleximag(rb);
          lua_Number z[2];
          agnc_unm(z, nb1, nb2);
          setcvalue(ra, z);
#endif
        }
        else {
          Protect(Arith(L, ra, rb, rb, TM_UNM));
        }
        continue;
      }
      case OP_NOT: {
        int res = l_isfalseorfail(RB(i));  /* Agena 1.4.0; next assignment may change this value */
        setbvalue(ra, res);
        continue;
      }
      case OP_LEN: {  /* `size' operator, extended to support TH_SIZE metamethod 1.9.1; */
        const TValue *rb = RB(i);
        switch (ttype(rb)) {
          case LUA_TTABLE: {
            const TValue *tm;
            Table *t;
            t = hvalue(rb);
            if ((tm = fasttm(L, t->metatable, TM_SIZE)) == NULL) { /* no metamethod ? */
              Protect(setnvalue(ra, cast_num(agenaV_nops(L, t))));
            } else if (ttisfunction(tm))
              call_binTM(L, rb, rb, ra, TM_SIZE);
            break;
          }
          case LUA_TSTRING: {
            setnvalue(ra, cast_num(tsvalue(rb)->len));
            break;
          }
          case LUA_TSET: {
            const TValue *tm;
            UltraSet *s;
            s = usvalue(rb);
            if ((tm = fasttm(L, s->metatable, TM_SIZE)) == NULL) { /* no metamethod ? */
              Protect(setnvalue(ra, cast_num(s->size)));
            } else if (ttisfunction(tm))
              call_binTM(L, rb, rb, ra, TM_SIZE);
            break;
          }
          case LUA_TSEQ: {
            const TValue *tm;
            Seq *s;
            s = seqvalue(rb);
            if ((tm = fasttm(L, s->metatable, TM_SIZE)) == NULL) { /* no metamethod ? */
              Protect(setnvalue(ra, cast_num(seqvalue(rb)->size)));
            } else if (ttisfunction(tm))
              call_binTM(L, rb, rb, ra, TM_SIZE);
            break;
          }
          case LUA_TPAIR: {
            const TValue *tm;
            Pair *p;
            p = pairvalue(rb);
            if ((tm = fasttm(L, p->metatable, TM_SIZE)) == NULL) { /* no metamethod ? */
              Protect(setnvalue(ra, 2));
            } else if (ttisfunction(tm))
              call_binTM(L, rb, rb, ra, TM_SIZE);
            break;
          }
          default: {
            luaG_runerror(L, "in " LUA_QS ": wrong type of argument.", "size");
          }
        }
        continue;
      }
      case OP_SPLIT: {  /* added 0.11.2 */
        /* get prefix, must use RB() and not RKB() to display line causing an error correctly */
        TValue *rb = RB(i);
        /* get postfix (the deliminator) */
        TValue *rc = RKC(i);
        if (ttype(rb) == LUA_TSTRING && ttype(rc) == LUA_TSTRING) {
           Protect(agenaV_split(L, svalue(rb), svalue(rc), ra); luaC_checkGC(L));
           /* you must call GC, otherwise split may consume large amounts of memory over time */
        } else {
           luaG_runerror(L, "in " LUA_QS ": two strings expected.", "split");
        }
        continue;
      }
      case OP_CONCAT: {
        int b = GETARG_B(i);
        int c = GETARG_C(i);
        Protect(luaV_concat(L, c-b+1, c); luaC_checkGC(L));
        setobjs2s(L, RA(i), base+b);
        continue;
      }
      case OP_JMP: {
        dojump(L, pc, GETARG_sBx(i));
        continue;
      }
      case OP_EQ: {
        TValue *rb = RKB(i);
        TValue *rc = RKC(i);
        Protect(
          if (equalobj(L, rb, rc) == GETARG_A(i)) {
            dojump(L, pc, GETARG_sBx(*pc));
          }
        )
        pc++;
        continue;
      }
      case OP_LT: {
        Protect(
          if (luaV_lessthan(L, RKB(i), RKC(i)) == GETARG_A(i))
            dojump(L, pc, GETARG_sBx(*pc));
        )
        pc++;
        continue;
      }
      case OP_LE: {
        Protect(
          if (lessequal(L, RKB(i), RKC(i)) == GETARG_A(i))
            dojump(L, pc, GETARG_sBx(*pc));
        )
        pc++;
        continue;
      }
      case OP_TEST: {
        if (l_isfalseorfail(ra) != GETARG_C(i))
          dojump(L, pc, GETARG_sBx(*pc));
        pc++;
        continue;
      }
      case OP_TESTSET: {
        TValue *rb = RB(i);
        if (l_isfalseorfail(rb) != GETARG_C(i)) {
          setobjs2s(L, ra, rb);
          dojump(L, pc, GETARG_sBx(*pc));
        }
        pc++;
        continue;
      }
      /* 0.9.2, March 09, 2008; patched 0.10.0, April 27, 2008 */
      case OP_DOTTED: {
        TValue *rb = RKB(i);
        if (ttisstring(rb)) {
          Protect(luaV_substring(L, rb, RKC(i), ra));
        } else if (ttistable(rb)) {
          Protect(luaV_tablesublist(L, rb, RKC(i), ra));
        } else if (ttisseq(rb)) {
          Protect(luaV_seqsublist(L, rb, RKC(i), ra));
        } else
          luaG_runerror(L, "in indexing operation: string, sequence, or table expected, got %s.",
            luaT_typenames[(int)ttype(rb)]);
        /* do not adjust L->top ! The stack will be confused otherwise */
        continue;
      }
      case OP_CALL: {
        int b = GETARG_B(i);
        int nresults = GETARG_C(i) - 1;
        if (b != 0) L->top = ra+b;  /* else previous instruction set top */
        L->savedpc = pc;
        switch (luaD_precall(L, ra, nresults)) {
          case PCRLUA: {
            nexeccalls++;
            goto reentry;  /* restart luaV_execute over new Lua function */
          }
          case PCRC: case PCRREMEMBER: {
            /* it was a C function (`precall' called it); adjust results */
            if (nresults >= 0) L->top = L->ci->top;
            base = L->base;
            continue;
          }
          default: {
            return;  /* yield */
          }
        }
      }
      /* for commands returning no values, added 0.6.0; mechanism changed in 0.9.1 (20 % speed gain) */
      case OP_CMD: {
        int nargs;
        StkId oldtop;
        nargs = GETARG_B(i);  /* number of arguments */
        oldtop = L->top;
        L->top = ra + nargs;
        switch (GETARG_C(i)) {  /* check instruction, changed 0.9.1 */
          case CMD_CASE: {  /* added 0.6.0 */
            int j, flag;
            flag = 1;
            Protect(
            TValue *left = ra;
            for (j=1; j < nargs; j++) {
              TValue *right = ra + j;
              if (equalobj(L, left, right) == 1) {
                flag = 0;
                break;
              }
            }
            if (flag)
              dojump(L, pc, GETARG_sBx(*pc));
            )
            pc++;
            break;
          }
          case CMD_POPTOP: {
            Protect(
            TValue *val = RA(i);
            if (ttisseq(val)) {
              Seq *h = seqvalue(val);
              if (h->size != 0) {
                setnilvalue(L->top);
                api_check(L, L->top < L->ci->top);
                agnSeq_seti(L, h, h->size, L->top);
                luaC_barrierseq(L, h, L->top);  /* Agena 1.2 fix */
              }
            } else if (ttistable(val)) {  /* Agena 1.1.0 */
              lua_Number b;
              Table *t = hvalue(val);
              b = agenaV_topindex(L, t);
              if (b != -HUGE_VAL) {  /* integer index found */
                setnilvalue(L->top);
                api_check(L, L->top < L->ci->top);
                setobj2t(L, luaH_setnum(L, t, cast_int(b)), L->top);  /* set it to null */
                luaC_barriert(L, t, L->top);
                t->hasnil = 1;
              }  /* else: do nothing */
            } else
              luaG_runerror(L, "in pop/top statement: sequence or table expected, got %s.",
                luaT_typenames[(int)ttype(val)]);
            );
            break;
          }
          case CMD_POPBOTTOM: {
            Protect(
            if (ttisseq(ra)) {
              Seq *h = seqvalue(ra);
              if (h->size != 0) {
                setnilvalue(L->top);
                api_check(L, L->top < L->ci->top);
                agnSeq_seti(L, h, 1, L->top);
                luaC_barrierseq(L, h, L->top);  /* Agena 1.2 fix */
              }
            } else if (ttistable(ra)) {  /* Agena 1.1.0 */
              lua_Number b;
              Table *t = hvalue(ra);
              b = agenaV_bottomindex(L, t);
              if (b != HUGE_VAL) {  /* integer index found */
                setnilvalue(L->top);
                api_check(L, L->top < L->ci->top);
                setobj2t(L, luaH_setnum(L, t, cast_int(b)), L->top);  /* set it to null */
                luaC_barriert(L, t, L->top);
                t->hasnil = 1;
              }  /* else: do nothing */
            } else
              luaG_runerror(L, "in pop/bottom statement: sequence or table expected, got %s.",
                luaT_typenames[(int)ttype(ra)]);
            );
            break;
          }
          case CMD_BYE: {
            if (!agn_getgui(L)) exit(0);
            break;
          }
          case CMD_CLS: {
            if (!agn_getgui(L))
              #if defined(_WIN32) || defined(LUA_DOS) || defined(__OS2__) || defined(__HAIKU__)
              if (system("cls") == -1)  /* 1.6.4 to prevent compiler warnings */
                luaG_runerror(L, "in `cls` statement: system call failed.");
              #elif defined(__unix__) || defined(__APPLE__)
              if (system("clear") == -1)
                luaG_runerror(L, "in `cls` statement: system call failed.");
            #endif
            break;
          }
          case CMD_GC: {  /* perform garbage collection for clear command, patched 1.6.0 Valgrind */
            /* put environ.gc function on stack */
            setobj2s(L, ra, luaH_getstr(hvalue(gt(L)), luaS_new(L, "environ")));
            if (ttistable(ra)) {
              setobj2s(L, ra, luaH_getstr(hvalue(ra), luaS_new(L, "gc")));
              if (!ttisfunction(ra))
                luaG_runerror(L, "in `clear` statement: `environ.gc` does not exist.");  /* 2.0.0 correction */
            } else
              luaG_runerror(L, "in `clear` statement: `environ` package table does not exist.");  /* 2.0.0 correction */
            /* do _not_ put null argument on stack */
            /* now conduct the garbage colection */
            Protect(luaD_call(L, ra, nargs));  /* 1.6.0. Valgrind, function, args and ignore result */
            /* delete result of gc() */
            break;
          }
          case CMD_IMPORT: {  /* call readlib from import statement 2.0.0 */
            /* ra currently is null, now overwrite it with `readlib`*/
            setobj2s(L, ra, luaH_getstr(hvalue(gt(L)), luaS_new(L, "readlib")));
            if (!ttisfunction(ra))
              luaG_runerror(L, "in `import` statement: `readlib` does not exist.");
            /* now call readlib */
            Protect(luaD_call(L, ra, 0));  /* call readlib with 0 results */
            break;
          }
          case CMD_ALIAS: {  /* call with from import statement 2.0.0 */
            /* ra currently is null, now overwrite it with `readlib`*/
            setobj2s(L, ra, luaH_getstr(hvalue(gt(L)), luaS_new(L, "with")));
            if (!ttisfunction(ra))
              luaG_runerror(L, "in `import` statement: `with` does not exist.");
            /* now call readlib */
            Protect(luaD_call(L, ra, 0));  /* call readlib with 0 results */
            break;
          }
          case CMD_RESTART: {  /* perform restart */
            /* put __RESTART function on stack */
            setobj2s(L, ra, luaH_getstr(hvalue(gt(L)), luaS_new(L, "environ")));
            if (ttistable(ra)) {
              setobj2s(L, ra, luaH_getstr(hvalue(ra), luaS_new(L, "__RESTART")));
              if (!ttisfunction(ra))
                luaG_runerror(L, "in `restart` statement: `environ.__RESTART` does not exist.");
            } else
              luaG_runerror(L, "in `restart` statement: `environ` package table does not exist.");
            /* put null argument on stack */
            setnilvalue(ra+1);  /* do not comment the following lines ! */
            luaD_checkstack(L, 2);  /* Agena 1.9.4 fix */
            L->top = ra+2;  /* func. + 1 arg */  /* Agena 1.9.4 fix */
            /* restart the environment */
            Protect(luaD_call(L, ra, 0));  /* function, args and 0 result */
            /* do not execute setnilvalue(ra) for it will corrupt the stack if restart is called after
               an error was issued !  Agena 1.0.5 fix */
            /* luaD_call will level the stack automatically, even in case of errors */
            break;  /* 0.22.3 fix */
          }
          default: {
            return;
          }
        }
        L->top = oldtop;
        continue;  /* leave continue here */
      }
      case OP_FN: {  /* for functions requiring exactly one or more arguments and returning one value;
        added 0.7.1, November 18, 2007 */
        switch(GETARG_C(i)) {
          case OPR_ASSIGNED: {
            switch (ttype(RB(i))) {
              case LUA_TNIL: case LUA_TNONE: case LUA_TFAIL: {
                setbvalue(ra, 0);
                break;
              }
              default: {
                setbvalue(ra, 1);
              }
            }
            continue;
          }
          case OPR_UNASSIGNED: {  /* added 0.7.1; tuned 0.10.0 */
            setbvalue(ra, ttisnil(RB(i)));
            continue;
          }
          case OPR_TYPE: {
            setsvalue(L, ra, luaS_new(L, luaT_typenames[RB(i)->tt]));
            continue;
          }
          case OPR_TYPEOF: {
            switch (RB(i)->tt) {
              case LUA_TFUNCTION: {
                Closure *c = clvalue(RB(i));
                if (c->l.type != NULL) {
                  setsvalue(L, ra, c->l.type);
                } else if (c->c.type != NULL) {
                  setsvalue(L, ra, c->c.type);
                } else
                  setsvalue(L, ra, luaS_new(L, luaT_typenames[RB(i)->tt]));
                break;
              }
              case LUA_TSEQ: {
                Seq *s = seqvalue(RB(i));
                if (s->type == NULL) {
                  setsvalue(L, ra, luaS_new(L, luaT_typenames[RB(i)->tt])); }
                else {
                  setsvalue(L, ra, s->type); }
                break;
              }
              case LUA_TPAIR: {
                Pair *s = pairvalue(RB(i));
                if (s->type == NULL) {
                  setsvalue(L, ra, luaS_new(L, luaT_typenames[RB(i)->tt])); }
                else {
                  setsvalue(L, ra, s->type); }
                break;
              }
              case LUA_TTABLE: {
                Table *s = hvalue(RB(i));
                if (s->type == NULL) {
                  setsvalue(L, ra, luaS_new(L, luaT_typenames[RB(i)->tt])); }
                else {
                  setsvalue(L, ra, s->type); }
                break;
              }
              case LUA_TSET: {
                UltraSet *s = usvalue(RB(i));
                if (s->type == NULL) {
                  setsvalue(L, ra, luaS_new(L, luaT_typenames[RB(i)->tt])); }
                else {
                  setsvalue(L, ra, s->type); }
                break;
              }
              default:
                setsvalue(L, ra, luaS_new(L, luaT_typenames[RB(i)->tt]));
            }
            continue;
          }
          case OPR_DIMTAB: {  /* 0.8.0 */
            TValue *rb = RB(i);
            if (ttisnumber(rb)) {
              Table *newtable;
              int nb = nvalue(rb);
              if (nb < 0) nb = 0;  /* 1.7.3 */
              newtable = luaH_new(L, nb, 0);
              newtable->hasnil = 1;  /* avoid troubles with binary searches of the `size` operator. 1.7.9a */
              sethvalue(L, ra, newtable);  /* 1.7.10 patch */
              Protect(luaC_checkGC(L));
            }
            else
              Protect(luaG_runerror(L, "in " LUA_QS " constructor, number expected, got %s.",
                 "table", luaT_typenames[(int)ttype(rb)]));
            continue;
          }
          case OPR_DIMDICT: {
            TValue *rb = RB(i);
            if (ttisnumber(rb)) {
              Table *newtable;
              int nb = nvalue(rb);
              if (nb < 0) nb = 0;  /* 1.7.3 */
              newtable = luaH_new(L, 0, (int)nb);
              newtable->hasnil = 1;  /* avoid troubles with binary searches of the `size` operator. 1.7.10 */
              sethvalue(L, ra, newtable);
              Protect(luaC_checkGC(L));
            }
            else
              Protect(luaG_runerror(L, "in " LUA_QS " constructor, number expected, got %s.",
                 "dict", luaT_typenames[(int)ttype(rb)]));
            continue;
          }
          case OPR_DIMSET: {  /* 0.10.0*/
            TValue *rb = RB(i);
            if (ttisnumber(rb)) {
              int nb = nvalue(rb);
              if (nb < 0) nb = 0;  /* 1.7.3 */
              setusvalue(L, ra, agnUS_new(L, (int)nb));
              Protect(luaC_checkGC(L));
            }
            else
              Protect(luaG_runerror(L, "in " LUA_QS " constructor, number expected, got %s.",
                 "set", luaT_typenames[(int)ttype(rb)]));
            continue;
          }
          case OPR_DIMSEQ: {  /* 0.11.0 */
            TValue *rb = RB(i);
            if (ttisnumber(rb)) {
              int nb = nvalue(rb);
              if (nb < 0) nb = 0;  /* 1.7.3 */
              setseqvalue(L, ra, agnSeq_new(L, (int)nb));
              Protect(luaC_checkGC(L));
            }
            else
              Protect(luaG_runerror(L, "in " LUA_QS " constructor, number expected, got %s.",
                "seq", luaT_typenames[(int)ttype(rb)]));
            continue;
          }
          case OPR_ABS: {
            TValue *rb = RB(i);
            switch (ttype(rb)) {
              case LUA_TNUMBER: {
                lua_Number nb = nvalue(rb);
                setnvalue(ra, luai_numabs(nb));
                break;
              }
              case LUA_TCOMPLEX: {
#ifndef PROPCMPLX
                agn_Complex nb = cvalue(rb);
                setnvalue(ra, agnc_abs(nb));
#else
                setnvalue(ra, agnc_abs(complexreal(rb), compleximag(rb)));
#endif
                break;
              }
              case LUA_TSTRING: {
                if (tsvalue(rb)->len == 1) {  /* output ASCII value for character */
                  const char *s = svalue(rb);
                  setnvalue(ra, cast_num(uchar(s[0])));
                } else {
                  setfailvalue(ra);
                }
                break;
              }
              case LUA_TBOOLEAN: {  /* convert true -> 1, false -> 0, fail -> -1, like in Algol 68 */
                lua_Number val = cast_num((rb)->value.b);
                if (val < 2) {
                  setnvalue(ra, val);
                }
                else {
                  setnvalue(ra, -1);
                }
                break;
              }
              case LUA_TNIL: {  /* convert true -> 1, false -> 0, like in Algol 68 */
                setnvalue(ra, -2);
                break;
              }
              default: {
                Protect(Arith(L, ra, rb, rb, TM_ABS));
              }
            }
            continue;
          }
          case OPR_LOWER: {  /* 0.8.0, Dec 07, 2007; extended 0.12.2, Oct 19, 2008 */
            TValue *rb = RB(i);
            if (ttisstring(rb)) {
              size_t i, len;
              const char *s = svalue(rb);
              len = tsvalue(rb)->len;
              Protect(
                char *str = (char *)malloc((len+1)*sizeof(char));
                if (str == 0)
                  luaG_runerror(L, "memory allocation error in " LUA_QS ".", "lower");  /* Agena 1.0.4 */
                for (i=0; i<len; i++) {
                  str[i] = lowercase[uchar(s[i])];
                }
                str[len] = '\0';
                setsvalue(L, ra, luaS_newlstr(L, str, len));
                xfree(str);  /* 1.10.4 */
              )
            }
            else {
              Protect(luaG_runerror(L, "in " LUA_QS ": string expected, got %s.",
                 "lower", luaT_typenames[(int)ttype(rb)]));
            }
            continue;
          }
          case OPR_UPPER: {  /* 0.8.0, Dec 07, 2007; extended 0.12.2, Oct 19, 2008 */
            TValue *rb = RB(i);
            if (ttisstring(rb)) {
              size_t i, len;
              const char *s = svalue(rb);
              char *str;
              Protect(
                len = tsvalue(rb)->len;
                str = (char *)malloc((len+1)*sizeof(char));  /* 1.9.1 */
                if (str == 0)
                  luaG_runerror(L, "memory allocation error in " LUA_QS ".", "upper");  /* Agena 1.0.4 */
                for (i=0; i<len; i++) {
                  str[i] = uppercase[uchar(s[i])];
                }
                str[len] = '\0';
                setsvalue(L, ra, luaS_newlstr(L, str, len));
                xfree(str);  /* 1.10.4 */
              );
            }
            else {
              Protect(luaG_runerror(L, "in " LUA_QS ": string expected, got %s.",
                 "upper", luaT_typenames[(int)ttype(rb)]));
            }
            continue;
          }
          case OPR_TRIM: {  /* 0.8.0, Dec 07, 2007 */
            TValue *rb = RB(i);
            if (ttisstring(rb)) {
              Protect(agnV_trim(L, rb, ra));
            }
            else {
              Protect(luaG_runerror(L, "in " LUA_QS ": string expected, got %s.",
                 "trim", luaT_typenames[(int)ttype(rb)]));
            }
            continue;
          }
          case OPR_CHAR: {  /* 0.8.0, Dec 09, 2007, patched 0.32.4, 06.06.2010 */
            TValue *rb = RB(i);
            if (ttisnumber(rb)) {
              Protect(
                char s[1];
                lua_Number nb = nvalue(rb);
                if (nb < 0 || nb > 255)
                  luaG_runerror(L, "in " LUA_QS ": invalid value.", "char");
                else {
                  s[0] = uchar(nb);  /* better sure than sorry, 1.5.0 */
                  setsvalue(L, ra, luaS_newlstr(L, s, 1));
                }
              );
            }
            else {
              Protect(luaG_runerror(L, "in " LUA_QS ": number expected, got %s.",
                 "char", luaT_typenames[(int)ttype(rb)]));
            }
            continue;
          }
          case OPR_JOIN: {  /* 0.11.0, May 17, 2008 */
            Protect(agenaV_join(L, ra, GETARG_B(i)); luaC_checkGC(L));
            continue;
          }
          case OPR_LEFT: {  /* 0.11.1, May 22, 2008 */
            TValue *rb = RB(i);
            if (ttype(rb) == LUA_TPAIR) {
              Protect(
                setobj2s(L, ra, pairitem(pairvalue(rb), 0));
              );
            } else {
              Protect(luaG_runerror(L, "in " LUA_QS ": pair expected, got %s.",
                "left", luaT_typenames[(int)ttype(rb)]));
            }
            continue;
          }
          case OPR_BOTTOM: {  /* 0.29.0, Nov 20, 2009 */
            TValue *rb = RB(i);
            Protect(
              agenaV_bottom(L, rb, ra); luaC_checkGC(L)
            );
            continue;
          }
          case OPR_RIGHT: {  /* 0.11.1, May 22, 2008 */
            TValue *rb = RB(i);
            if (ttype(rb) == LUA_TPAIR) {
              Protect(
                setobj2s(L, ra, pairitem(pairvalue(rb), 1));
              );
            } else {
              Protect(luaG_runerror(L, "in " LUA_QS ": pair expected, got %s.",
                 "right", luaT_typenames[(int)ttype(rb)]));
            }
            continue;
          }
          case OPR_TOP: {  /* 0.11.1, May 22, 2008 */
            TValue *rb = RB(i);
            Protect(
              agenaV_top(L, rb, ra); luaC_checkGC(L);
            );
            continue;
          }
          case OPR_REAL: {  /* 0.11.1, May 22, 2008 */
            TValue *rb = RB(i);
            if (ttiscomplex(rb)) {
              Protect(
                setnvalue(ra, complexreal(rb));
              );
            }
            else {
              Protect(luaG_runerror(L, "in " LUA_QS ": complex number expected, got %s.",
                 "real", luaT_typenames[(int)ttype(rb)]));
            }
            continue;
          }
          case OPR_IMAG: {  /* 0.11.1, May 22, 2008 */
            TValue *rb = RB(i);
            if (ttiscomplex(rb)) {
              Protect(
                setnvalue(ra, compleximag(rb));
              );
            }
            else {
              Protect(luaG_runerror(L, "in " LUA_QS ": complex number expected, got %s.",
                 "imag", luaT_typenames[(int)ttype(rb)]));
            }
            continue;
          }
          case OPR_ARCTAN: {
            arith_opfn(L, luai_numatan, agnc_atan, TM_ARCTAN);
            continue;
          }
          case OPR_ARCSIN: {  /* 0.27.0 */
            arith_opfn(L, luai_numasin, agnc_asin, TM_ARCSIN);
            continue;
          }
          case OPR_COS: {
            arith_opfn(L, luai_numcos, agnc_cos, TM_COS);
            continue;
          }
          case OPR_ENTIER: {
            arith_opfn(L, luai_numentier, agnc_entier, TM_ENTIER);
            continue;
          }
          case OPR_EVEN: {   /* 0.7.1, even numbers are beautiful */
            TValue *rb = RKB(i);
            if (ttisnumber(rb)) {
              lua_Number nb = nvalue(rb);
              setbvalue(ra, luai_numiseven(nb));
            }
            else if (ttiscomplex(rb)) {
              setbvalue(ra, 2);
            }
            else {
              Protect(Arith(L, ra, rb, rb, TM_EVEN));
            }
            continue;
          }
          case OPR_EXP: {
            arith_opfn(L, luai_numexp, agnc_exp, TM_EXP);
            continue;
          }
          case OPR_FINITE: {
            TValue *rb = RKB(i);
            if (ttisnumber(rb)) {
              lua_Number nb = nvalue(rb);
              setbvalue(ra, luai_numisfinite(nb));
            } else if (ttiscomplex(rb)) {
              setbvalue(ra, 2);
            } else {
              Protect(Arith(L, ra, rb, rb, TM_FINITE));
            }
            continue;
          }
          /* added 0.9.0; needed in factorial, binomial, and beta function,
             `Numerical Recipes in FORTRAN`, 2nd DE ed., p. 208f */
          case OPR_LNGAMMA: {
            arith_opfn(L, luai_numlngamma, agnc_lngamma, TM_LNGAMMA);  /* 0.28.1 */
            continue;
          }
          case OPR_INT: {
            arith_opfn(L, luai_numint, agnc_int, TM_INT);
            continue;
          }
          case OPR_LN: {
            arith_opfn(L, luai_numln, agnc_ln, TM_LN);
            continue;
          }
          case OPR_SIGN: {
            TValue *rb = RB(i);
            if (ttisnumber(rb)) {
              arith_opfnNumber(luai_numsign, TM_SIGN);
            }
            else if (ttiscomplex(rb)) {
#ifndef PROPCMPLX
              agn_Complex z = cvalue(rb);
              if (creal(z) > 0 || (creal(z) == 0 && cimag(z) > 0)) {
                setnvalue(ra, 1);
              }
              else if (creal(z) < 0 || (creal(z) == 0 && cimag(z) < 0)) {
                setnvalue(ra, -1);
              }
              else {
                setnvalue(ra, 0);
              }
#else
              setnvalue(ra, csgn(complexreal(rb), compleximag(rb)));
#endif
            }
            else if (!call_binTM(L, rb, rb, ra, TM_SIGN))
              luaG_typeerror(L, rb, "get sign of");
            continue;
          }
          case OPR_SIN: {
            arith_opfn(L, luai_numsin, agnc_sin, TM_SIN);
            continue;
          }
          case OPR_SQRT: {
            arith_opfn(L, luai_numsqrt, agnc_sqrt, TM_SQRT);
            continue;
          }
          case OPR_TAN: {
            arith_opfn(L, luai_numtan, agnc_tan, TM_TAN);
            continue;
          }
          case OPR_RECIP: {
            arith_opfn(L, luai_numrecip, agnc_recip, TM_RECIP);
            continue;
          }
          case OPR_TFILLED: {
            Protect(agenaV_filled(L, RKB(i), ra); luaC_checkGC(L));
            continue;
          }
          case OPR_TUNIQUE: {
            TValue *rb = RB(i);
            if (ttistable(rb)) {
              Protect(agenaV_tunique(L, RKB(i), ra); luaC_checkGC(L));
            } else if (ttisseq(rb)) {
              Protect(agenaV_sequnique(L, RKB(i), ra); luaC_checkGC(L));
            } else
              Protect(luaG_runerror(L, "in " LUA_QS ": table or sequence expected, got %s.",
                 "unique", luaT_typenames[(int)ttype(rb)]));
            continue;
          }
          case OPR_TCOPY: {
            Protect(agenaV_copy(L, RKB(i), ra, NULL); luaC_checkGC(L));
            continue;
          }
          case OPR_TSADD: {
            if (ttistable(RB(i))) {
              Protect(agenaV_sadd(L, RKB(i), ra); luaC_checkGC(L));
            }
            else if (ttisseq(RB(i))) {
              Protect(agenaV_seqsadd(L, RKB(i), ra); luaC_checkGC(L));
            }
            else
              Protect(luaG_runerror(L, "in " LUA_QS ": table or sequence expected, got %s.",
                 "sadd", luaT_typenames[(int)ttype(RB(i))]));
            continue;
          }
          case OPR_TQSADD: {
            if (ttistable(RB(i))) {
              Protect(agenaV_qsadd(L, RKB(i), ra); luaC_checkGC(L));
            }
            else if (ttisseq(RB(i))) {
              Protect(agenaV_seqqsadd(L, RKB(i), ra); luaC_checkGC(L));
            }
            else
              Protect(luaG_runerror(L, "in " LUA_QS ": table or sequence expected, got %s.",
                 "qsadd", luaT_typenames[(int)ttype(RB(i))]));
            continue;
          }
          case OPR_NARGS: {
            setnvalue(RA(i), cast_num(L->ci->nargs));
            continue;
          }
          case OPR_NEWUSET: {
            int b = GETARG_B(i);
            setusvalue(L, ra, agnUS_new(L, luaO_fb2int(b)));
            Protect(luaC_checkGC(L));
            continue;
          }
          case OPR_NEWSEQ: {
            int b = GETARG_B(i);
            setseqvalue(L, ra, agnSeq_new(L, luaO_fb2int(b)));
            Protect(luaC_checkGC(L));
            continue;
          }
          case OPR_REPLACE: {
            int nargs = GETARG_B(i);
            Protect(agenaV_replace(L, ra, nargs));
            continue;
          }
          case OPR_SINH: {  /* 0.23.0 */
            arith_opfn(L, luai_numsinh, agnc_sinh, TM_SINH);
            continue;
          }
          case OPR_COSH: {  /* 0.23.0 */
            arith_opfn(L, luai_numcosh, agnc_cosh, TM_COSH);
            continue;
          }
          case OPR_TANH: {  /* 0.23.0 */
            arith_opfn(L, luai_numtanh, agnc_tanh, TM_TANH);
            continue;
          }
          case OPR_ARCCOS: {  /* 0.27.0 */
            arith_opfn(L, luai_numacos, agnc_acos, TM_ARCCOS);
            continue;
          }
          case OPR_FLOAT: {  /* 0.23.0 */
            TValue *rb = RKB(i);
            if (ttisnumber(rb)) {
              lua_Number nb = nvalue(rb);
              setbvalue(ra, ISFLOAT(nb));  /* tuned 0.30.2 */
            }
            else {
              setbvalue(ra, 2);
            }
            continue;
          }
          case OPR_BNOT: {  /* 0.27.0 */
            TValue *rb = RKB(i);
            if (ttisnumber(rb)) {
              if (L->settings & 1) {
                LUA_INTEGER nb = nvalue(rb);
                setnvalue(ra, ~(nb));
              } else {
                LUAI_UINT32 nb = nvalue(rb);
                setnvalue(ra, ~(nb));
              }
            }
            else {
              Protect(luaG_runerror(L, "in " LUA_QS ": number expected, got %s.",
                "~~", luaT_typenames[(int)ttype(RB(i))]));
            }
            continue;
          }
          case OPR_INSTR: {  /* patched 0.30.3, patched 1.10.4 */
            Protect(
              if (agenaV_instr(L, ra, GETARG_B(i)) == 1) luaC_checkGC(L);
            );
            continue;
          }
          case OPR_VALUES: {  /* patched 0.30.3 */
            Protect(agenaV_values(L, ra, GETARG_B(i)));
            continue;
          }
          case OPR_GETHIGH: {  /* 0.27.0 */
            LUAI_INT32 hx;
            TValue *rb = RKB(i);
            if (ttisnumber(rb)) {
              GET_HI(hx, nvalue(rb));
              setnvalue(ra, hx);
            }
            else {
              Protect(luaG_runerror(L, "in " LUA_QS ": number expected, got %s.",
                "gethigh", luaT_typenames[(int)ttype(RB(i))]));
            }
            continue;
          }
          case OPR_GETLOW: {  /* 0.27.0 */
            LUAI_UINT32 lx;
            TValue *rb = RKB(i);
            if (ttisnumber(rb)) {
              GET_LO(lx, nvalue(rb));
              setnvalue(ra, lx);
            }
            else {
              Protect(luaG_runerror(L, "in " LUA_QS ": number expected, got %s.",
                "getlow", luaT_typenames[(int)ttype(RB(i))]));
            }
            continue;
          }
          case OPR_SETHIGH: {  /* 0.27.0 */
            lua_Number x;
            LUAI_UINT32 lx;
            int nargs = GETARG_B(i);
            lua_lock(L);
            if (nargs != 2)
              Protect(luaG_runerror(L, "in " LUA_QS ": two arguments expected, got %d.",
                "sethigh", nargs));
            if (ttisnumber(ra) && ttisnumber(ra+1)) {
              x = nvalue(ra);
              GET_LO(lx, x);
              /* using something like SET_HI setting only the higher byte does not work */
              INSERT_WORDS(x, nvalue(ra+1), lx);
              setnvalue(ra, x);
            }
            else {
              Protect(luaG_runerror(L, "in " LUA_QS ": two numbers expected.", "sethigh"));
            }
            setnilvalue(ra+1);  /* delete second argument */
            lua_unlock(L);
            continue;
          }
          case OPR_SETLOW: {  /* 0.27.0 */
            lua_Number x;
            LUAI_INT32 hx;
            int nargs = GETARG_B(i);
            lua_lock(L);
            if (nargs != 2)
              Protect(luaG_runerror(L, "in " LUA_QS ": two arguments expected, got %d.",
                "setlow", nargs));
            if (ttisnumber(ra) && ttisnumber(ra+1)) {
              x = nvalue(ra);
              GET_HI(hx, x);
              /* using something like SET_LO setting only the lower byte does not work */
              INSERT_WORDS(x, hx, nvalue(ra+1));
              setnvalue(ra, x);
            }
            else {
              Protect(luaG_runerror(L, "in " LUA_QS ": two numbers expected.", "setlow"));
            }
            setnilvalue(ra+1);  /* delete second argument */
            lua_unlock(L);
            continue;
          }
          case OPR_LOG: {  /* 1.9.3, February 19, 2013 */
            int nargs = GETARG_B(i);
            lua_lock(L);
            if (nargs != 2)
              Protect(luaG_runerror(L, "in " LUA_QS ": two arguments expected, got %d.",
                "log", nargs));
            agenaV_log(L, ra, ra+1);
            setnilvalue(ra+1);  /* delete second argument */
            lua_unlock(L);
            continue;
          }
          default:  /* this should not happen */
            luaG_runerror(L, "in Virtual Machine: invalid operator for OP_FN.");
        }
      }
      case OP_FNBIN: {  /* for binary operators returning one value; added 0.27.0, 23.08.2009 */
        if (!ttispair(RB(i)))
          luaG_runerror(L, "(OP_FNBIN:) expected a pair, got %s.", luaT_typenames[(int)ttype(RB(i))]);
        TValue *rb = pairitem(pairvalue(RB(i)), 0);  /* get lhs of the pair */
        TValue *rc = pairitem(pairvalue(RB(i)), 1);  /* get rhs of the pair */
        switch(GETARG_C(i)) {
          case OPR_TOFTYPE: {  /* added 1.3.0 */
            if (!ttisstring(rc)) {
              luaG_runerror(L, "right argument to " LUA_QS " must be a string, got %s.", "::",
                luaT_typenames[(int)ttype(rc)]);
            }
            switch (rb->tt) {
              case LUA_TFUNCTION: {
                Closure *c = clvalue(rb);
                if (c->l.type != NULL) {
                  setbvalue(ra, strcmp(getstr(c->l.type), svalue(rc)) == 0);
                } else if (c->c.type != NULL) {
                  setbvalue(ra, strcmp(getstr(c->c.type), svalue(rc)) == 0);
                } else
                  setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) == 0);
                break;
              }
              case LUA_TSEQ: {
                Seq *s = seqvalue(rb);
                if (s->type == NULL) {
                  setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) == 0); }
                else {
                  setbvalue(ra, strcmp(getstr(s->type), svalue(rc)) == 0); }
                break;
              }
              case LUA_TPAIR: {
                Pair *s = pairvalue(rb);
                if (s->type == NULL) {
                  setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) == 0); }
                else {
                  setbvalue(ra, strcmp(getstr(s->type), svalue(rc)) == 0); }
                break;
              }
              case LUA_TTABLE: {
                Table *s = hvalue(rb);
                if (s->type == NULL) {
                  setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) == 0); }
                else {
                  setbvalue(ra, strcmp(getstr(s->type), svalue(rc)) == 0); }
                break;
              }
              case LUA_TSET: {
                UltraSet *s = usvalue(rb);
                if (s->type == NULL) {
                  setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) == 0); }
                else {
                  setbvalue(ra, strcmp(getstr(s->type), svalue(rc)) == 0); }
                break;
              }
              default:
                setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) == 0);
            }
            continue;
          }
          case OPR_TNOTOFTYPE: {  /* added 1.3.0 */
            if (!ttisstring(rc)) {
              luaG_runerror(L, "right argument to " LUA_QS " must be a string, got %s.", ":-",
                luaT_typenames[(int)ttype(rc)]);
            }
            switch (rb->tt) {
              case LUA_TFUNCTION: {
                Closure *c = clvalue(rb);
                if (c->l.type != NULL) {
                  setbvalue(ra, strcmp(getstr(c->l.type), svalue(rc)) != 0);
                } else if (c->c.type != NULL) {
                  setbvalue(ra, strcmp(getstr(c->c.type), svalue(rc)) != 0);
                } else
                  setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) != 0);
                break;
              }
              case LUA_TSEQ: {
                Seq *s = seqvalue(rb);
                if (s->type == NULL) {
                  setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) != 0); }
                else {
                  setbvalue(ra, strcmp(getstr(s->type), svalue(rc)) != 0); }
                break;
              }
              case LUA_TPAIR: {
                Pair *s = pairvalue(rb);
                if (s->type == NULL) {
                  setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) != 0); }
                else {
                  setbvalue(ra, strcmp(getstr(s->type), svalue(rc)) != 0); }
                break;
              }
              case LUA_TTABLE: {
                Table *s = hvalue(rb);
                if (s->type == NULL) {
                  setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) != 0); }
                else {
                  setbvalue(ra, strcmp(getstr(s->type), svalue(rc)) != 0); }
                break;
              }
              case LUA_TSET: {
                UltraSet *s = usvalue(rb);
                if (s->type == NULL) {
                  setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) != 0); }
                else {
                  setbvalue(ra, strcmp(getstr(s->type), svalue(rc)) != 0); }
                break;
              }
              default:
                setbvalue(ra, strcmp(luaT_typenames[rb->tt], svalue(rc)) != 0);
            }
            continue;
          }
          case OPR_XOR: {  /* added 0.27.0 */
            Protect(agenaV_xor(L, ra, rb, rc));
            continue;
          }
          case OPR_ATENDOF: {  /* added 0.27.0 */
            Protect(agenaV_atendof(L, ra, rb, rc));
            continue;
          }
          case OPR_BAND: {  /* added 0.27.0 */
            Protect(agenaV_band(L, ra, rb, rc));
            continue;
          }
          case OPR_BOR: {  /* added 0.27.0 */
            Protect(agenaV_bor(L, ra, rb, rc));
            continue;
          }
          case OPR_BXOR: {  /* added 0.27.0 */
            Protect(agenaV_bxor(L, ra, rb, rc));
            continue;
          }
          case OPR_PERCENT: {  /* added 1.10.6 */
            Protect(agenaV_percent(L, ra, rb, rc));
            continue;
          }
          case OPR_PERCENTRATIO: {  /* added 1.11.4 */
            Protect(agenaV_percentratio(L, ra, rb, rc));
            continue;
          }
          case OPR_PERCENTADD: {  /* added 1.11.3 */
            Protect(agenaV_percentadd(L, ra, rb, rc));
            continue;
          }
          case OPR_PERCENTSUB: {  /* added 1.11.3 */
            Protect(agenaV_percentsub(L, ra, rb, rc));
            continue;
          }
          default:  /* this should not happen */
            luaG_runerror(L, "in Virtual Machine: invalid operator for OP_FNBIN.");
        }
      }
      case OP_PAIR: {
        Protect(agenaV_pair(L, RKB(i), RKC(i), ra); luaC_checkGC(L));
        continue;
      }
      case OP_COMPLEX: {
        TValue *rb = RKB(i); \
        TValue *rc = RKC(i); \
        if (ttisnumber(rb) && ttisnumber(rc)) {  /* 0.22.2 */
#ifndef PROPCMPLX
          setcvalue(ra, nvalue(rb)+I*nvalue(rc));
#else
          lua_Number z[2];
          z[0] = nvalue(rb);
          z[1] = nvalue(rc);
          setcvalue(ra, z);
#endif
        }
        else
          luaG_runerror(L, "number expected for " LUA_QS " operator.", "!");
        continue;
      }
      case OP_TEEQ: {  /* 0.22.0 */
        TValue *rb = RKB(i);
        TValue *rc = RKC(i);
        if (ttype(rb) == ttype(rc)) {
          setbvalue(ra, luaV_equalvalonebyone(L, rb, rc));
        } else {
          setbvalue(ra, 0);
        }
        continue;
      }
      case OP_TAILCALL: {
        int b = GETARG_B(i);
        if (b != 0) L->top = ra+b;  /* else previous instruction set top */
        L->savedpc = pc;
        lua_assert(GETARG_C(i) - 1 == LUA_MULTRET);
        switch (luaD_precall(L, ra, LUA_MULTRET)) {
          case PCRLUA: {
            int aux;
            /* tail call: put new frame in place of previous one */
            CallInfo *ci = L->ci - 1;  /* previous frame */
            StkId func = ci->func;
            StkId pfunc = (ci+1)->func;  /* previous function index */
            ci->nargs = (ci+1)->nargs;  /* nargs at current ci is incorrect, so get correct one, 0.13.1 patch 2 */
            if (L->openupval) luaF_close(L, ci->base);
            L->base = ci->base = ci->func + ((ci+1)->base - pfunc);
            for (aux = 0; pfunc+aux < L->top; aux++)  /* move frame down */
              setobjs2s(L, func+aux, pfunc+aux);
            ci->top = L->top = func+aux;  /* correct top */
            lua_assert(L->top == L->base + clvalue(func)->l.p->maxstacksize);
            ci->savedpc = L->savedpc;
            ci->tailcalls++;  /* one more call lost */
            L->ci--;  /* remove new frame */
            goto reentry;
          }
          case PCRC: case PCRREMEMBER: {  /* it was a C function (`precall' called it) */
            base = L->base;
            continue;
          }
          default: {
            return;  /* yield */
          }
        }
      }
      case OP_RETURN: {
        int b = GETARG_B(i);
        Proto *p = cl->p;
        if (b != 0) L->top = ra+b-1;
        if (L->openupval) luaF_close(L, base);
        if (p->is_rettypegiven) {  /* return type given ? Agena 1.3.0 */
          Protect(
          StkId slot;
          if (p->rettype != -1) {
            for (slot = ra; slot < L->top; slot++) {
              if (ttype(slot) != p->rettype) {
                luaG_runerror(L, LUA_QL("return") " value must be of type %s, got %s.",
                  luaT_typenames[p->rettype],
                  luaT_typenames[(int)ttype(slot)]);
              }
            }
          } else {  /* optional user-defined type specified */
            for (slot = ra; slot < L->top; slot++) {
              switch (ttype(slot)) {
                case LUA_TSEQ: {
                  Seq *s = seqvalue(slot);
                  if (s->type != NULL && strcmp(getstr(s->type), getstr(p->rettypets)) != 0)
                    luaG_runerror(L, LUA_QL("return") " value must be of type %s, got %s.",
                      getstr(p->rettypets), luaT_typenames[(int)ttype(slot)]);
                  break;
                }
                case LUA_TTABLE: {
                  Table *s = hvalue(slot);
                  if (s->type != NULL && strcmp(getstr(s->type), getstr(p->rettypets)) != 0)
                    luaG_runerror(L, LUA_QL("return") " value must be of type %s, got %s.",
                      getstr(p->rettypets), luaT_typenames[(int)ttype(slot)]);
                  break;
                }
                case LUA_TSET: {
                  UltraSet *s = usvalue(slot);
                  if (s->type != NULL && strcmp(getstr(s->type), getstr(p->rettypets)) != 0)
                    luaG_runerror(L, LUA_QL("return") " value must be of type %s, got %s.",
                      getstr(p->rettypets), luaT_typenames[(int)ttype(slot)]);
                  break;
                }
                case LUA_TPAIR: {
                  Pair *s = pairvalue(slot);
                  if (s->type != NULL && strcmp(getstr(s->type), getstr(p->rettypets)) != 0)
                    luaG_runerror(L, LUA_QL("return") " value must be of type %s, got %s.",
                      getstr(p->rettypets), luaT_typenames[(int)ttype(slot)]);
                  break;
                }
                case LUA_TFUNCTION: {
                  Closure *c = clvalue(slot);
                  if ((c->l.type != NULL && strcmp(getstr(c->l.type), getstr(p->rettypets)) != 0) ||
                     (c->c.type != NULL && strcmp(getstr(c->c.type), getstr(p->rettypets)) != 0))
                      luaG_runerror(L, LUA_QL("return") " value must be of type %s, got %s.",
                      getstr(p->rettypets), luaT_typenames[(int)ttype(slot)]);
                  break;
                }
                default:
                  luaG_runerror(L, LUA_QL("return") " value must be of type %s.", getstr(p->rettypets));
              }
            }
          }
          )
        }
        L->savedpc = pc;
        if (cl->rtable != NULL && cl->updatertable) {  /* does remember table exist and
          there is NO read RETURN with an entry from the rtable ? Then enter args and results
          into it. */
          luaD_rtableentry(L, cl, base, ra, b+LUA_MULTRET);
        }
        b = luaD_poscall(L, ra);
        if (--nexeccalls == 0)  /* was previous function running `here'? */
          return;  /* no: return */
        else {  /* yes: continue its execution */
          if (b) L->top = L->ci->top;
          lua_assert(isLua(L->ci));
          lua_assert(GET_OPCODE(*((L->ci)->savedpc - 1)) == OP_CALL);
          goto reentry;
        }
      }
      case OP_FORLOOP: {  /* tweaked 1.4.0 */
        lua_Number idx, limit, step;
        limit = nvalue(ra+1);
        step = nvalue(ra+2);
        if (nvalue(ra+3) != HUGE_VAL) {  /* step size is nonintegral ? This check is around twice as fast as a plain (int)(step) != step query */
          volatile lua_Number c, y, t;
          idx = nvalue(ra);
          /* increment index: apply Kahan summation algorithm to avoid roundoff errors */
          y = step - nvalue(ra+3);  /* ra+3: correction value c */
          t = idx + y;
          c = (t - idx) - y;
          idx = t;
          if (luai_numlt(0, step) ? luai_numle(idx, limit)
                                  : luai_numle(limit, idx)) {
            dojump(L, pc, GETARG_sBx(i));  /* jump back */
            if (fabs(idx) < AGN_EPSILON) idx = 0;
            setnvalue(ra+3, c);  /* update internal roundoff prevention variable c */
            setnvalue(ra+4, idx);  /* update external index */
          }
        } else {  /* step size is an integer */
          idx = luai_numadd(nvalue(ra), step);  /* increment index */
          if (luai_numlt(0, step) ? luai_numle(idx, limit)
                                  : luai_numle(limit, idx)) {
            dojump(L, pc, GETARG_sBx(i));  /* jump back */
            setnvalue(ra+4, idx);  /* update external index */
          }
        }
        setnvalue(ra, idx);  /* update internal index */  /* 0.12.3 */
        continue;
      }
      case OP_FORPREP: {
        const TValue *init = ra;
        const TValue *plimit = ra+1;
        const TValue *pstep = ra+2;
        lua_Number step = nvalue(pstep);
        L->savedpc = pc;  /* next steps may throw errors */
        if (!ttisnumber(init) || nvalue(init) == HUGE_VAL)
          luaG_runerror(L, LUA_QL("for") " initial value must be a number, got %s.", luaT_typenames[(int)ttype(init)]);
        /* else if (!ttisnumber(plimit) || nvalue(plimit) == HUGE_VAL) */
        else if (!ttisnumber(plimit))  /* infinity is again allowed, 2.0.0 RC 2 */
          luaG_runerror(L, LUA_QL("for") " limit must be a number, got %s.", luaT_typenames[(int)ttype(plimit)]);
        else if (!ttisnumber(pstep) || nvalue(pstep) == HUGE_VAL)
          luaG_runerror(L, LUA_QL("for") " step must be a number, got %s.", luaT_typenames[(int)ttype(pstep)]);
        if (ISFLOAT(step)) {  /* 0.30.2, do not use (int)(step) != step since 1e300 also is an integer */
          volatile lua_Number c, y, t, idx;
          idx = nvalue(ra);
          /* apply Kahan summation algorithm to avoid roundoff errors, 0.30.0 */
          y = step - 0.0;
          t = idx - y;
          c = (t - idx) + y;
          idx = t;
          setnvalue(ra, idx);  /* update internal index */
          setnvalue(ra+3, c);  /* correction */
        } else
          setnvalue(ra, luai_numsub(nvalue(ra), step));
        dojump(L, pc, GETARG_sBx(i));
        continue;
      }
      case OP_TFORLOOP: {
        StkId cb = ra + 3;  /* call base */
        setobjs2s(L, cb+2, ra+2);
        setobjs2s(L, cb+1, ra+1);
        setobjs2s(L, cb, ra);
        L->top = cb+3;  /* func. + 2 args (state and index) */
        Protect(luaD_call(L, cb, GETARG_C(i)));
        L->top = L->ci->top;
        cb = RA(i) + 3;  /* previous call may change the stack */
        if (ttisnotnil(cb)) {  /* continue loop? */
          setobjs2s(L, cb-1, cb);  /* save control variable */
          dojump(L, pc, GETARG_sBx(*pc));  /* jump back */
        }
        pc++;
        continue;
      }
      case OP_TFORPREP: {  /* changed 0.5.0, taken from Lua 5.0.3 */
        if (ttistable(ra) || ttisuset(ra) || ttisseq(ra) || ttisstring(ra)) {
          /* this must be checked, else iterator functions get confused */
          setobjs2s(L, ra+1, ra);
          setobj2s(L, ra, luaH_getstr(hvalue(gt(L)), luaS_new(L, "next")));
        }
        else if (ttisnil(ra)) {  /* 0.25.2 */
          setseqvalue(L, ra+1, agnSeq_new(L, 0));  /* do nothing */
          Protect(luaC_checkGC(L));
          setobj2s(L, ra, luaH_getstr(hvalue(gt(L)), luaS_new(L, "next")));
        }
        else if (ttype(ra) != LUA_TFUNCTION)  /* 0.24.1 */
          luaG_runerror(L, "cannot iterate a %s value.",
            luaT_typenames[(int)ttype(ra)]);
        dojump(L, pc, GETARG_sBx(i));
        break;
      }
      case OP_SETLIST: {
        int n = GETARG_B(i);
        int c = GETARG_C(i);
        int last;
        if (n == 0) {
          n = cast_int(L->top - ra) - 1;
          L->top = L->ci->top;
        }
        if (c == 0) c = cast_int(*pc++);
        if (ttistable(ra)) {  /* 0.11.0 */
          Table *h;
          h = hvalue(ra);
          last = ((c-1)*LFIELDS_PER_FLUSH) + n;
          if (last > h->sizearray)  /* needs more space? */
            luaH_resizearray(L, h, last);  /* pre-alloc it at once */
          for (; n > 0; n--) {
            TValue *val = ra+n;
            setobj2t(L, luaH_setnum(L, h, last--), val);
            luaC_barriert(L, h, val);
            if (ttisnil(val)) h->hasnil = 1;
          }
        } else if (ttisuset(ra)) {  /* 0.12.2 */
          UltraSet *h;
          h = usvalue(ra);
          last = ((c-1)*LFIELDS_PER_FLUSH) + n;
          if (last > h->size)  /* needs more space? */
            agnUS_resize(L, h, last);  /* pre-alloc it at once */
          for (; n > 0; n--) {
            TValue *val = ra+n;
            agnUS_set(L, h, val);
            luaC_barrierset(L, h, val);  /* Agena 1.2 fix */
          }
        }
        else if (ttisseq(ra)) {
          Seq *h = seqvalue(ra);
          last = ((c-1)*LFIELDS_PER_FLUSH) + n;
          if (last > h->maxsize)  /* needs more space? */
            agnSeq_resize(L, h, last);  /* pre-alloc it at once */
          for (; n > 0; n--) {
            Protect(
              if (agnSeq_constructorseti(L, h, last--, ra+n))
                luaG_runerror(L, " in " LUA_QL("seq") ", null not allowed in sequences.");
            );
          }
        }
        else
          break;
        continue;
      }
      case OP_CLOSE: {
        luaF_close(L, ra);
        continue;
      }
      case OP_CLOSURE: {
        Proto *p;
        Closure *ncl;
        int nup, j;
        p = cl->p->p[GETARG_Bx(i)];
        nup = p->nups;
        ncl = luaF_newLclosure(L, nup, cl->env);
        ncl->l.p = p;
        for (j=0; j<nup; j++, pc++) {
          if (GET_OPCODE(*pc) == OP_GETUPVAL)
            ncl->l.upvals[j] = cl->upvals[GETARG_B(*pc)];
          else {
            lua_assert(GET_OPCODE(*pc) == OP_MOVE);
            ncl->l.upvals[j] = luaF_findupval(L, base + GETARG_B(*pc));
          }
        }
        setclvalue(L, ra, ncl);
        Protect(luaC_checkGC(L));
        continue;
      }
      case OP_INTDIV: {  /* added 0.5.4 */
        arith_opNumber(luai_numintdiv, TM_DIV);
        continue;
      }
      case OP_SHIFT: {  /* added 0.5.4, changed 0.27.0 */
        TValue *rb = RKB(i);
        TValue *rc = RKC(i);
        if (L->settings & 1) {
          LUA_INTEGER a, b;
          a = nvalue(rb);
          b = nvalue(rc);
          setnvalue(ra, b > 0 ? a << b : a >> -b);
        } else {
          LUAI_UINT32 a, b;
          a = nvalue(rb);
          b = nvalue(rc);
          setnvalue(ra, b > 0 ? a << b : a >> -b);
        }
        continue;
      }
      case OP_TRY: {  /* 2.1 RC 2, written by Hu Qiwei */
        int status;
        pj = cast(struct lua_longjmp *, luaM_malloc(L, sizeof(struct lua_longjmp)));
        pj->type = JMPTYPE_TRY;
        pj->status = 0;
        pj->pc = (Instruction *)pc + GETARG_sBx(i);
        pj->previous = L->errorJmp;

        pj->oldnCcalls = L->nCcalls;
        pj->old_ci = saveci(L, L->ci);
        pj->old_allowhooks = L->allowhook;
        pj->old_errfunc = L->errfunc;
        pj->old_nexeccalls = nexeccalls;
        pj->old_top = savestack(L, L->top);
        L->errorJmp = pj;
        L->errfunc = 0;

        status = setjmp(pj->b);
        if (status) {
          pc = L->errorJmp->pc;
          nexeccalls = L->errorJmp->old_nexeccalls;
          restoretry(L, GET_OPCODE(*pc) == OP_CATCH, GETARG_A(*pc));
          L->savedpc = pc;
          goto reentry;
        }
        continue;
      }
      case OP_ENDTRY: {  /* 2.1 RC 2, written by Hu Qiwei */
        releasetry(L);
        continue;
      }
      case OP_CATCH: {  /* 2.1 RC 2, written by Hu Qiwei */
        /* dummy opcode, do nothing here ! */
        continue;
      }
    }
  }
}


