/*
** $Id: lstrlib.c,v 1.132 2006/04/26 20:41:19 roberto Exp $
** Standard library for string operations and pattern-matching
** See Copyright Notice in agena.h
*/


#include <ctype.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <math.h>   /* HUGE_VAL */

#define lstrlib_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "agncmpt.h"
#include "agnhlps.h"
#include "ldebug.h"  /* for luaG_runerror */
#include "lstrlib.h"
#include "lundump.h"

/* macro to `unsign' a character */
#define uchar(c)        ((unsigned char)(c))

/* Table indicating whether a character is a diacritic or not. The diacritics recognized
   here are:

                             
                 
                       
                         
                 
                   
*/

static unsigned char diacritics[256] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 0 .. 7*/
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 8 .. 15 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 16 .. 23 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 24 .. 31 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 32 .. 39 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 40 .. 47 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 48 .. 55 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 56 .. 63 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 64 .. 71 */
/* 064@  065A  066B  067C  068D  069E  070F  071G */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 72 .. 79 */
/* 072H  073I  074J  075K  076L  077M  078N  079O */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 80 .. 87 */
/* 080P  081Q  082R  083S  084T  085U  086V  087W */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 88 .. 95 */
/* 088X  089Y  090Z  091   092   093   094   095  */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 96 .. 103 */
/* 096`  097a  098b  099c  100d  101e  102f  103g */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 104 .. 111 */
/* 104h  105i  106j  107k  108l  109m  110n  111o */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 112 .. 119 */
/* 112p  113q  114r  115s  116t  117u  118v  119w */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 120 .. 127 */
/* 120x  121y  122z  123{  124|  125}  126~  127 */
   0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,  /* 128 .. 135, ok */
/* 128  129  130  131  132  133  134  135 */
   0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,  /* 136 .. 143 ok */
/* 136  137  138  139  140  141  142  143 */
   0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,  /* 144 .. 151 ok */
/* 144  145  146  147  148  149  150  151 */
   0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x00, 0x00,  /* 152 .. 159 ok */
/* 152? 153  154  155  156  157  158  159 */
   0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00,  /* 160 .. 167 ok */
/* 160  161  162  163  164  165  166  167 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 168 .. 175 */
/* 168  169  170  171  172  173  174  175 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01,  /* 176 .. 183 */
/* 176  177  178  179  180  181  182  183 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 184 .. 191 */
/* 184  185  186  187+  188+  189  190  191+ */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,  /* 192 .. 199 ok */
/* 192+  193-  194-  195+  196-  197+  198  199 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 200 .. 207 ok */
/* 200+  201+  202-  203-  204  205-  206+  207 */
   0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,  /* 208 .. 215 ok */
/* 208  209  210  211  212  213i  214  215 */
   0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,  /* 216 .. 223 ok */
/* 216  217+  218+  219  220_  221  222  223 */
   0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x01,  /* 224 .. 231 ok */
/* 224  225  226  227  228  229  230  231 */
   0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00,  /* 232 .. 239 ok */
/* 232  233  234  235  236  237  238  239 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 240 .. 247 */
/* 240  241  242=  243  244  245  246  247 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   /* 248 .. 255 */
/* 248  249  250  251  252  253  254  255 */
};

static unsigned char lowerDiacritics[256] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 0 .. 7*/
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 8 .. 15 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 16 .. 23 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 24 .. 31 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 32 .. 39 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 40 .. 47 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 48 .. 55 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 56 .. 63 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 64 .. 71 */
/* 064@  065A  066B  067C  068D  069E  070F  071G */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 72 .. 79 */
/* 072H  073I  074J  075K  076L  077M  078N  079O */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 80 .. 87 */
/* 080P  081Q  082R  083S  084T  085U  086V  087W */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 88 .. 95 */
/* 088X  089Y  090Z  091   092   093   094   095  */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 96 .. 103 */
/* 096`  097a  098b  099c  100d  101e  102f  103g */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 104 .. 111 */
/* 104h  105i  106j  107k  108l  109m  110n  111o */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 112 .. 119 */
/* 112p  113q  114r  115s  116t  117u  118v  119w */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 120 .. 127 */
/* 120x  121y  122z  123{  124|  125}  126~  127 */
   0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,  /* 128 .. 135, ok */
/* 128  129  130  131  132  133  134  135 */
   0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00,  /* 136 .. 143 ok */
/* 136  137  138  139  140  141  142  143 */
   0x00, 0x01, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01,  /* 144 .. 151 ok */
/* 144  145  146  147  148  149  150  151 */
   0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,  /* 152 .. 159 ok */
/* 152? 153  154  155  156  157  158  159 */
   0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 0x00, 0x00,  /* 160 .. 167 ok */
/* 160  161  162  163  164  165  166  167 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 168 .. 175 */
/* 168  169  170  171  172  173  174  175 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 176 .. 183 */
/* 176  177  178  179  180  181  182  183 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 184 .. 191 */
/* 184  185  186  187+  188+  189  190  191+ */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,  /* 192 .. 199 ok */
/* 192+  193-  194-  195+  196-  197+  198  199 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 200 .. 207 ok */
/* 200+  201+  202-  203-  204  205-  206+  207 */
   0x01, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,  /* 208 .. 215 ok */
/* 208  209  210  211  212  213i  214  215 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 216 .. 223 ok */
/* 216  217+  218+  219  220_  221  222  223 */
   0x00, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01,  /* 224 .. 231 ok */
/* 224  225  226  227  228  229  230  231 */
   0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,  /* 232 .. 239 ok */
/* 232  233  234  235  236  237  238  239 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 240 .. 247 */
/* 240  241  242=  243  244  245  246  247 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   /* 248 .. 255 */
/* 248  249  250  251  252  253  254  255 */
};

static unsigned char upperDiacritics[256] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 0 .. 7*/
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 8 .. 15 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 16 .. 23 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 24 .. 31 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 32 .. 39 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 40 .. 47 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 48 .. 55 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 56 .. 63 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 64 .. 71 */
/* 064@  065A  066B  067C  068D  069E  070F  071G */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 72 .. 79 */
/* 072H  073I  074J  075K  076L  077M  078N  079O */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 80 .. 87 */
/* 080P  081Q  082R  083S  084T  085U  086V  087W */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 88 .. 95 */
/* 088X  089Y  090Z  091   092   093   094   095  */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 96 .. 103 */
/* 096`  097a  098b  099c  100d  101e  102f  103g */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 104 .. 111 */
/* 104h  105i  106j  107k  108l  109m  110n  111o */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 112 .. 119 */
/* 112p  113q  114r  115s  116t  117u  118v  119w */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 120 .. 127 */
/* 120x  121y  122z  123{  124|  125}  126~  127 */
   0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 128 .. 135, ok */
/* 128  129  130  131  132  133  134  135 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01,  /* 136 .. 143 ok */
/* 136  137  138  139  140  141  142  143 */
   0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 144 .. 151 ok */
/* 144  145  146  147  148  149  150  151 */
   0x00, 0x01, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00,  /* 152 .. 159 ok */
/* 152? 153  154  155  156  157  158  159 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,  /* 160 .. 167 ok */
/* 160  161  162  163  164  165  166  167 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,  /* 168 .. 175 */
/* 168  169  170  171  172  173  174  175 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x01,  /* 176 .. 183 */
/* 176  177  178  179  180  181  182  183 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 184 .. 191 */
/* 184  185  186  187+  188+  189  190  191+ */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,  /* 192 .. 199 ok */
/* 192+  193-  194-  195+  196-  197+  198  199 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 200 .. 207 ok */
/* 200+  201+  202-  203-  204  205-  206+  207 */
   0x00, 0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x01,  /* 208 .. 215 ok */
/* 208  209  210  211  212  213i  214  215 */
   0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00,  /* 216 .. 223 ok */
/* 216  217+  218+  219  220_  221  222  223 */
   0x01, 0x00, 0x01, 0x01, 0x00, 0x01, 0x00, 0x00,  /* 224 .. 231 ok */
/* 224  225  226  227  228  229  230  231 */
   0x01, 0x01, 0x01, 0x01, 0x00, 0x01, 0x00, 0x00,  /* 232 .. 239 ok */
/* 232  233  234  235  236  237  238  239 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* 240 .. 247 */
/* 240  241  242=  243  244  245  246  247 */
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,   /* 248 .. 255 */
/* 248  249  250  251  252  253  254  255 */
};


ptrdiff_t posrelat (ptrdiff_t pos, size_t len) {
  /* relative string position: negative means back from end */
  /* return (pos>=0) ? pos : (ptrdiff_t)len+pos+1; */
  /* Lua 5.1.3 patch 9 */
  if (pos < 0) pos += (ptrdiff_t)len + 1;
  return (pos >= 0) ? pos : 0;
}


static int str_reverse (lua_State *L) {
  size_t l;
  luaL_Buffer b;
  const char *s = luaL_checklstring(L, 1, &l);
  luaL_buffinit(L, &b);
  while (l--) luaL_addchar(&b, s[l]);
  luaL_pushresult(&b);
  return 1;
}


static int str_repeat (lua_State *L) {
  size_t l;
  luaL_Buffer b;
  const char *s = luaL_checklstring(L, 1, &l);
  int n = agnL_checkint(L, 2);
  luaL_buffinit(L, &b);
  while (n-- > 0)
    luaL_addlstring(&b, s, l);
  luaL_pushresult(&b);
  return 1;
}


static int str_tochars (lua_State *L) {
  int n = lua_gettop(L);  /* number of arguments */
  int i;
  luaL_Buffer b;
  luaL_buffinit(L, &b);
  for (i=1; i<=n; i++) {
    int c = agnL_checkint(L, i);
    luaL_argcheck(L, uchar(c) == c, i, "invalid value");
    luaL_addchar(&b, uchar(c));
  }
  luaL_pushresult(&b);
  return 1;
}


static int str_tobytes (lua_State *L) {
  size_t i, l;
  const char *str = luaL_checklstring(L, 1, &l);
  agn_createseq(L, l);
  for (i=0; i < l; i++) {
    lua_seqsetinumber(L, -1, i+1, cast_num(uchar(str[i])));
  }
  return 1;
}


static int writer (lua_State *L, const void* b, size_t size, void* B) {  /* reintroduced 1.6.0 */
  (void)L;
  luaL_addlstring((luaL_Buffer*) B, (const char *)b, size);
  return 0;
}


static int str_dump (lua_State *L) {  /* reintroduced 1.6.0 */
  luaL_Buffer b;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  if (lua_iscfunction(L, 1))
    luaL_error(L, "Error in `strings.dump`: cannot dump C library functions.");  /* 1.6.1 */
  lua_settop(L, 1);
  luaL_buffinit(L,&b);
  if (lua_dump(L, writer, &b) != 0)
    luaL_error(L, "Error in `strings.dump`: unable to dump given function.");  /* 1.6.1 */
  luaL_pushresult(&b);
  return 1;
}


/*
** {======================================================
** PATTERN MATCHING
** =======================================================
*/


#define CAP_UNFINISHED   (-1)
#define CAP_POSITION   (-2)
#define L_ESC      '%'


static int check_capture (MatchState *ms, int l) {
  l -= '1';
  if (l < 0 || l >= ms->level || ms->capture[l].len == CAP_UNFINISHED)
    return luaL_error(ms->L, "invalid capture index.");
  return l;
}


static int capture_to_close (MatchState *ms) {
  int level = ms->level;
  for (level--; level>=0; level--)
    if (ms->capture[level].len == CAP_UNFINISHED) return level;
  return luaL_error(ms->L, "invalid pattern capture.");
}


const char *classend (MatchState *ms, const char *p) {
  switch (*p++) {
    case L_ESC: {
      if (*p == '\0')
        luaL_error(ms->L, "malformed pattern (ends with " LUA_QL("%%") ").");
      return p+1;
    }
    case '[': {
      if (*p == '^') p++;
      do {  /* look for a `]' */
        if (*p == '\0')
          luaL_error(ms->L, "malformed pattern (missing " LUA_QL("]") ").");
        if (*(p++) == L_ESC && *p != '\0')
          p++;  /* skip escapes (e.g. `%]') */
      } while (*p != ']');
      return p+1;
    }
    default: {
      return p;
    }
  }
}


static int isvowel (int c) {  /* 1.10.0 */
  switch (tolower(c)) {
    case 'a': case 'e': case 'i': case 'o': case 'u': case 'y': return 1;
    default: return 0;
  }
}

static int isconsonant (int c) {  /* 1.10.0 */
  int x = tolower(c);
  return ((
    (x > 'a' && x < 'e') || (x > 'e' && x < 'i') ||
    (x > 'i' && x < 'o') || (x > 'o' && x < 'u') ||
    (x > 'u' && x < 'y') || (x == 'z')) ? 1 : 0);
}

static int match_class (int c, int cl) {
  int res;
  switch (tolower(cl)) {
    case 'a' : res = isalpha(c); break;
    case 'c' : res = iscntrl(c); break;
    case 'd' : res = isdigit(c); break;
    case 'k' : res = isconsonant(c); break;
    case 'l' : res = islower(c); break;
    case 'p' : res = ispunct(c); break;
    case 's' : res = isspace(c); break;
    case 'u' : res = isupper(c); break;
    case 'v' : res = isvowel(c); break;
    case 'w' : res = isalnum(c); break;
    case 'x' : res = isxdigit(c); break;
    case 'z' : res = (c == 0); break;
    default: return (cl == c);
  }
  return (islower(cl) ? res : !res);
}


int matchbracketclass (int c, const char *p, const char *ec) {
  int sig = 1;
  if (*(p+1) == '^') {
    sig = 0;
    p++;  /* skip the `^' */
  }
  while (++p < ec) {
    if (*p == L_ESC) {
      p++;
      if (match_class(c, uchar(*p)))
        return sig;
    }
    else if ((*(p+1) == '-') && (p+2 < ec)) {
      p+=2;
      if (uchar(*(p-2)) <= c && c <= uchar(*p))
        return sig;
    }
    else if (uchar(*p) == c) return sig;
  }
  return !sig;
}


int singlematch (int c, const char *p, const char *ep) {
  switch (*p) {
    case '.': return 1;  /* matches any char */
    case L_ESC: return match_class(c, uchar(*(p+1)));
    case '[': return matchbracketclass(c, p, ep-1);
    default:  return (uchar(*p) == c);
  }
}


const char *matchbalance (MatchState *ms, const char *s,
                                   const char *p) {
  if (*p == 0 || *(p+1) == 0)
    luaL_error(ms->L, "unbalanced pattern.");
  if (*s != *p) return NULL;
  else {
    int b = *p;
    int e = *(p+1);
    int cont = 1;
    while (++s < ms->src_end) {
      if (*s == e) {
        if (--cont == 0) return s+1;
      }
      else if (*s == b) cont++;
    }
  }
  return NULL;  /* string ends out of balance */
}


const char *max_expand (MatchState *ms, const char *s,
                                 const char *p, const char *ep, int mode) {
  ptrdiff_t i = 0;  /* counts maximum expand for item */
  while ((s+i)<ms->src_end && singlematch(uchar(*(s+i)), p, ep))
    i++;
  /* keeps trying to match with the maximum repetitions */
  while (i>=0) {
    const char *res = match(ms, (s+i), ep+1, mode);
    if (res) return res;
    i--;  /* else didn't match; reduce 1 repetition to try again */
  }
  return NULL;
}


const char *min_expand (MatchState *ms, const char *s,
                                 const char *p, const char *ep, int mode) {
  for (;;) {
    const char *res = match(ms, s, ep+1, mode);
    if (res != NULL)
      return res;
    else if (s<ms->src_end && singlematch(uchar(*s), p, ep))
      s++;  /* try with one more repetition */
    else return NULL;
  }
}


const char *start_capture (MatchState *ms, const char *s,
                                    const char *p, int what, int mode) {
  const char *res;
  int level = ms->level;
  if (level >= LUA_MAXCAPTURES) luaL_error(ms->L, "too many captures.");
  ms->capture[level].init = s;
  ms->capture[level].len = what;
  ms->level = level+1;
  if ((res=match(ms, s, p, mode)) == NULL)  /* match failed? */
    ms->level--;  /* undo capture */
  return res;
}


const char *end_capture (MatchState *ms, const char *s,
                                  const char *p, int mode) {
  int l = capture_to_close(ms);
  const char *res;
  ms->capture[l].len = s - ms->capture[l].init;  /* close capture */
  if ((res = match(ms, s, p, mode)) == NULL)  /* match failed? */
    ms->capture[l].len = CAP_UNFINISHED;  /* undo capture */
  return res;
}


const char *match_capture (MatchState *ms, const char *s, int l) {
  size_t len;
  l = check_capture(ms, l);
  len = ms->capture[l].len;
  if ((size_t)(ms->src_end-s) >= len &&
      memcmp(ms->capture[l].init, s, len) == 0)
    return s+len;
  else return NULL;
}
/* mode = 1: call from Lua API, mode = 0: call from VM (for properly issueing errors from either API or VM) */

const char *match (MatchState *ms, const char *s, const char *p, int mode) {
  init: /* using goto's to optimize tail recursion */
  switch (*p) {
    case '(': {  /* start capture */
      if (*(p+1) == ')')  /* position capture? */
        return start_capture(ms, s, p+2, CAP_POSITION, mode);
      else
        return start_capture(ms, s, p+1, CAP_UNFINISHED, mode);
    }
    case ')': {  /* end capture */
      return end_capture(ms, s, p+1, mode);
    }
    case L_ESC: {
      switch (*(p+1)) {
        case 'b': {  /* balanced string? */
          s = matchbalance(ms, s, p+2);
          if (s == NULL) return NULL;
          p+=4; goto init;  /* else return match(ms, s, p+4); */
        }
        case 'f': {  /* frontier? */
          const char *ep; char previous;
          p += 2;
          if (*p != '[') {
            if (mode)
              luaL_error(ms->L, "missing " LUA_QL("[") " after "
                               LUA_QL("%%f") " in pattern.");
            else
              luaG_runerror(ms->L, "missing " LUA_QL("[") " after "
                               LUA_QL("%%f") " in pattern.");
          }
          ep = classend(ms, p);  /* points to what is next */
          previous = (s == ms->src_init) ? '\0' : *(s-1);
          if (matchbracketclass(uchar(previous), p, ep-1) ||
             !matchbracketclass(uchar(*s), p, ep-1)) return NULL;
          p=ep; goto init;  /* else return match(ms, s, ep); */
        }
        default: {
          if (isdigit(uchar(*(p+1)))) {  /* capture results (%0-%9)? */
            s = match_capture(ms, s, uchar(*(p+1)));
            if (s == NULL) return NULL;
            p+=2; goto init;  /* else return match(ms, s, p+2) */
          }
          goto dflt;  /* case default */
        }
      }
    }
    case '\0': {  /* end of pattern */
      return s;  /* match succeeded */
    }
    case '$': {
      if (*(p+1) == '\0')  /* is the `$' the last char in pattern? */
        return (s == ms->src_end) ? s : NULL;  /* check end of string */
      else goto dflt;
    }
    default: dflt: {  /* it is a pattern item */
      const char *ep = classend(ms, p);  /* points to what is next */
      int m = s<ms->src_end && singlematch(uchar(*s), p, ep);
      switch (*ep) {
        case '?': {  /* optional */
          const char *res;
          if (m && ((res=match(ms, s+1, ep+1, mode)) != NULL))
            return res;
          p=ep+1; goto init;  /* else return match(ms, s, ep+1); */
        }
        case '*': {  /* 0 or more repetitions */
          return max_expand(ms, s, p, ep, mode);
        }
        case '+': {  /* 1 or more repetitions */
          return (m ? max_expand(ms, s+1, p, ep, mode) : NULL);
        }
        case '-': {  /* 0 or more repetitions (minimum) */
          return min_expand(ms, s, p, ep, mode);
        }
        default: {
          if (!m) return NULL;
          s++; p=ep; goto init;  /* else return match(ms, s+1, ep); */
        }
      }
    }
  }
}


const char *lmemfind (const char *s1, size_t l1,
                               const char *s2, size_t l2) {
  if (l2 == 0) return s1;  /* empty strings are everywhere */
  else if (l2 > l1) return NULL;  /* avoids a negative `l1' */
  else {
    const char *init;  /* to search for a `*s2' inside `s1' */
    l2--;  /* 1st char will be checked by `memchr' */
    l1 = l1-l2;  /* `s2' cannot be found after that */
    while (l1 > 0 && (init = (const char *)memchr(s1, *s2, l1)) != NULL) {
      init++;   /* 1st char is already checked */
      if (memcmp(init, s2+1, l2) == 0)
        return init-1;
      else {  /* correct `l1' and `s1' to try again */
        l1 -= init-s1;
        s1 = init;
      }
    }
    return NULL;  /* not found */
  }
}


static void push_onecapture (MatchState *ms, int i, const char *s,
                                                    const char *e) {
  if (i >= ms->level) {
    if (i == 0)  /* ms->level == 0, too */
      lua_pushlstring(ms->L, s, e - s);  /* add whole match */
    else
      luaL_error(ms->L, "invalid capture index.");
  }
  else {
    ptrdiff_t l = ms->capture[i].len;
    if (l == CAP_UNFINISHED) luaL_error(ms->L, "unfinished capture.");
    if (l == CAP_POSITION)
      lua_pushinteger(ms->L, ms->capture[i].init - ms->src_init + 1);
    else
      lua_pushlstring(ms->L, ms->capture[i].init, l);
  }
}


static int push_captures (MatchState *ms, const char *s, const char *e) {
  int i;
  int nlevels = (ms->level == 0 && s) ? 1 : ms->level;
  luaL_checkstack(ms->L, nlevels, "too many captures");
  for (i = 0; i < nlevels; i++)
    push_onecapture(ms, i, s, e);
  return nlevels;  /* number of strings pushed */
}


static int str_find_aux (lua_State *L, int find) {
  size_t l1, l2;
  const char *s = agn_checklstring(L, 1, &l1);
  const char *p = agn_checklstring(L, 2, &l2);
  ptrdiff_t init = posrelat(agnL_optinteger(L, 3, 1), l1) - 1;
  if (init < 0) init = 0;
  else if ((size_t)(init) > l1) init = (ptrdiff_t)l1;
  if (find && (lua_toboolean(L, 4) ||  /* explicit request? */
      strpbrk(p, SPECIALS) == NULL)) {  /* or no special characters? */
    /* do a plain search */
    const char *s2 = lmemfind(s+init, l1-init, p, l2);
    if (s2) {
      lua_pushinteger(L, s2-s+1);
      lua_pushinteger(L, s2-s+l2);
      return 2;
    }
  }
  else {
    MatchState ms;
    int anchor = (*p == '^') ? (p++, 1) : 0;
    const char *s1=s+init;
    ms.L = L;
    ms.src_init = s;
    ms.src_end = s+l1;
    do {
      const char *res;
      ms.level = 0;
      if ((res=match(&ms, s1, p, 1)) != NULL) {
        if (find) {
          lua_pushinteger(L, s1-s+1);  /* start */
          lua_pushinteger(L, res-s);   /* end */
          return push_captures(&ms, NULL, 0) + 2;
        }
        else
          return push_captures(&ms, s1, res);
      }
    } while (s1++ < ms.src_end && !anchor);
  }
  lua_pushnil(L);  /* not found */
  return 1;
}


static int str_find (lua_State *L) {
  return str_find_aux(L, 1);
}


static int str_match (lua_State *L) {
  return str_find_aux(L, 0);
}


static int gmatch_aux (lua_State *L) {
  MatchState ms;
  size_t ls;
  const char *s = lua_tolstring(L, lua_upvalueindex(1), &ls);
  const char *p = lua_tostring(L, lua_upvalueindex(2));
  const char *src;
  ms.L = L;
  ms.src_init = s;
  ms.src_end = s+ls;
  for (src = s + (size_t)lua_tointeger(L, lua_upvalueindex(3));
       src <= ms.src_end;
       src++) {
    const char *e;
    ms.level = 0;
    if ((e = match(&ms, src, p, 1)) != NULL) {
      lua_Integer newstart = e-s;
      if (e == src) newstart++;  /* empty match? go at least one position */
      lua_pushinteger(L, newstart);
      lua_replace(L, lua_upvalueindex(3));
      return push_captures(&ms, src, e);
    }
  }
  return 0;  /* not found */
}


static int gmatch (lua_State *L) {
  luaL_checkstring(L, 1);
  luaL_checkstring(L, 2);
  lua_settop(L, 2);
  lua_pushinteger(L, 0);
  lua_pushcclosure(L, gmatch_aux, 3);
  return 1;
}


static int gfind_nodef (lua_State *L) {
  return luaL_error(L, LUA_QL("string.gfind") " was renamed to "
                       LUA_QL("string.gmatch") ".");
}


static void add_s (MatchState *ms, luaL_Buffer *b, const char *s,
                                                   const char *e) {
  size_t l, i;
  const char *news = lua_tolstring(ms->L, 3, &l);
  for (i = 0; i < l; i++) {
    if (news[i] != L_ESC)
      luaL_addchar(b, news[i]);
    else {
      i++;  /* skip ESC */
      if (!isdigit(uchar(news[i])))
        luaL_addchar(b, news[i]);
      else if (news[i] == '0')
          luaL_addlstring(b, s, e - s);
      else {
        push_onecapture(ms, news[i] - '1', s, e);
        luaL_addvalue(b);  /* add capture to accumulated result */
      }
    }
  }
}


static void add_value (MatchState *ms, luaL_Buffer *b, const char *s,
                                                       const char *e) {
  lua_State *L = ms->L;
  switch (lua_type(L, 3)) {
    case LUA_TNUMBER:
    case LUA_TSTRING: {
      add_s(ms, b, s, e);
      return;
    }
    case LUA_TFUNCTION: {
      int n;
      lua_pushvalue(L, 3);
      n = push_captures(ms, s, e);
      lua_call(L, n, 1);
      break;
    }
    case LUA_TTABLE: {
      push_onecapture(ms, 0, s, e);
      lua_gettable(L, 3);
      break;
    }
  }
  if (!lua_toboolean(L, -1)) {  /* nil or false? */
    agn_poptop(L);
    lua_pushlstring(L, s, e - s);  /* keep original text */
  }
  else if (!agn_isstring(L, -1))
    luaL_error(L, "invalid replacement value (a %s).", luaL_typename(L, -1));
  luaL_addvalue(b);  /* add result to accumulator */
}


static int str_gsub (lua_State *L) {
  size_t srcl;
  const char *src = luaL_checklstring(L, 1, &srcl);
  const char *p = luaL_checkstring(L, 2);
  int tr = lua_type(L, 3);  /* 5.1.3 patch */
  int max_s = agnL_optinteger(L, 4, srcl+1);
  int anchor = (*p == '^') ? (p++, 1) : 0;
  int n = 0;
  MatchState ms;
  luaL_Buffer b;
  luaL_argcheck(L, tr == LUA_TNUMBER || tr == LUA_TSTRING ||  /* 5.1.2 patch */
                   tr == LUA_TFUNCTION || tr == LUA_TTABLE, 3,
                      "string, procedure, or table expected");
  luaL_buffinit(L, &b);
  ms.L = L;
  ms.src_init = src;
  ms.src_end = src+srcl;
  while (n < max_s) {
    const char *e;
    ms.level = 0;
    e = match(&ms, src, p, 1);
    if (e) {
      n++;
      add_value(&ms, &b, src, e);
    }
    if (e && e>src) /* non empty match? */
      src = e;  /* skip it */
    else if (src < ms.src_end)
      luaL_addchar(&b, *src++);
    else break;
    if (anchor) break;
  }
  luaL_addlstring(&b, src, ms.src_end-src);
  luaL_pushresult(&b);
  lua_pushinteger(L, n);  /* number of substitutions */
  return 2;
}

/* }====================================================== */


/* maximum size of each formatted item (> len(format('%99.99f', -1e308))) */
#define MAX_ITEM   512
/* valid flags in a format specification */
#define FLAGS   "-+ #0"
/*
** maximum size of each format specification (such as '%-099.99d')
** (+10 accounts for %99.99x plus margin of error)
*/
#define MAX_FORMAT   (sizeof(FLAGS) + sizeof(LUA_INTFRMLEN) + 10)


static void addquoted (lua_State *L, luaL_Buffer *b, int arg) {
  size_t l;
  const char *s = luaL_checklstring(L, arg, &l);
  /* luaL_addchar(b, '"');  0.9.2 */
  while (l--) {
    switch (*s) {
      case '"': case '\\': case '\n': {
        luaL_addchar(b, '\\');
        luaL_addchar(b, *s);
        break;
      }
      case '\r': {
        luaL_addlstring(b, "\\r", 2);
        break;
      }
      case '\0': {
        luaL_addlstring(b, "\\000", 4);
        break;
      }
      default: {
        luaL_addchar(b, *s);
        break;
      }
    }
    s++;
  }
  /* luaL_addchar(b, '"'); 0.9.2 */
}

static const char *scanformat (lua_State *L, const char *strfrmt, char *form) {
  const char *p = strfrmt;
  while (*p != '\0' && strchr(FLAGS, *p) != NULL) p++;  /* skip flags - Lua 5.1.2 patch */
  if ((size_t)(p - strfrmt) >= sizeof(FLAGS))
    luaL_error(L, "invalid format (repeated flags).");
  if (isdigit(uchar(*p))) p++;  /* skip width */
  if (isdigit(uchar(*p))) p++;  /* (2 digits at most) */
  if (*p == '.') {
    p++;
    if (isdigit(uchar(*p))) p++;  /* skip precision */
    if (isdigit(uchar(*p))) p++;  /* (2 digits at most) */
  }
  if (isdigit(uchar(*p)))
    luaL_error(L, "invalid format (width or precision too long).");
  *(form++) = '%';
  strncpy(form, strfrmt, p - strfrmt + 1);
  form += p - strfrmt + 1;
  *form = '\0';
  return p;
}


static void addintlen (char *form) {
  size_t l = strlen(form);
  char spec = form[l - 1];
  strcpy(form + l - 1, LUA_INTFRMLEN);
  form[l + sizeof(LUA_INTFRMLEN) - 2] = spec;
  form[l + sizeof(LUA_INTFRMLEN) - 1] = '\0';
}


static int str_format (lua_State *L) {
  int top = lua_gettop(L);
  int arg = 1;
  size_t sfl;
  const char *strfrmt = luaL_checklstring(L, arg, &sfl);
  const char *strfrmt_end = strfrmt+sfl;
  luaL_Buffer b;
  luaL_buffinit(L, &b);
  while (strfrmt < strfrmt_end) {
    if (*strfrmt != L_ESC)
      luaL_addchar(&b, *strfrmt++);
    else if (*++strfrmt == L_ESC)
      luaL_addchar(&b, *strfrmt++);  /* %% */
    else { /* format item */
      char form[MAX_FORMAT];  /* to store the format (`%...') */
      char buff[MAX_ITEM];  /* to store the formatted item */
      if (++arg > top)  /* Lua 5.1.4 patch 7 */
        luaL_argerror(L, arg, "no value");
      strfrmt = scanformat(L, strfrmt, form);
      switch (*strfrmt++) {
        case 'c': {
          sprintf(buff, form, (int)agn_checknumber(L, arg));
          break;
        }
        case 'd':  case 'i': {
          addintlen(form);
          sprintf(buff, form, (LUA_INTFRM_T)agn_checknumber(L, arg));
          break;
        }
        case 'o':  case 'u':  case 'x':  case 'X': {
          addintlen(form);
          sprintf(buff, form, (unsigned LUA_INTFRM_T)agn_checknumber(L, arg));
          break;
        }
        case 'e':  case 'E': case 'f':
        case 'g': case 'G': {
          sprintf(buff, form, (double)agn_checknumber(L, arg));
          break;
        }
        case 'q': {
          addquoted(L, &b, arg);
          continue;  /* skip the 'addsize' at the end */
        }
        case 's': {
          size_t l;
          const char *s = luaL_checklstring(L, arg, &l);
          if (!strchr(form, '.') && l >= 100) {
            /* no precision and string is too long to be formatted;
               keep original string */
            lua_pushvalue(L, arg);
            luaL_addvalue(&b);
            continue;  /* skip the `addsize' at the end */
          }
          else {
            sprintf(buff, form, s);
            break;
          }
        }
        default: {  /* also treat cases `pnLlh' */
          return luaL_error(L, "invalid option " LUA_QL("%%%c") " to "
                               LUA_QL("format") ".", *(strfrmt - 1));
        }
      }
      luaL_addlstring(&b, buff, strlen(buff));
    }
  }
  luaL_pushresult(&b);
  return 1;
}


/*************************************************************************/
/* functions added 0.5.3 and later                                       */
/*************************************************************************/

/* 0.20.0, April 10, 2009; extended January 23, 2010, 0.30.4; modified September 10, 2011, 1.5.0;
   tuned and Valgrind-fixed 1.6.0 */
static int str_isabbrev (lua_State *L) {
  size_t p_len, s_len;
  int pos;
  MatchState ms;
  const char *s, *s1, *p;
  const char *res;
  s = agn_checklstring(L, 1, &s_len);
  p = agn_checklstring(L, 2, &p_len);
  if (p_len == 0 || s_len == 0) {
    /* bail out if string or pattern is empty or if pattern is longer or equal in size */
    lua_pushfalse(L);
    return 1;
  }
  if (lua_toboolean(L, 3) ||  /* explicit request? */
      strpbrk(p, SPECIALS) == NULL) {  /* or no special characters? Agena 1.3.2 */
    pos = strncmp(s, p, p_len);
    lua_pushboolean(L, pos == 0 && p_len < s_len);
    return 1;
  }
  p++;
  s1 = s;
  ms.L = L;
  ms.src_init = s;
  ms.src_end = s + s_len;
  ms.level = 0;
  if ((res = match(&ms, s1, p, 1)) != NULL)
    lua_pushtrue(L);
  else
    lua_pushfalse(L);
  return 1;
}


/* modified September 10, 2011, 1.5.0; tuned and Valgrind-fixed 1.6.0 */

static int str_isending (lua_State *L) {
  size_t p_len, s_len;
  MatchState ms;
  const char *s, *s1, *p;
  s = agn_checklstring(L, 1, &s_len);
  p = agn_checklstring(L, 2, &p_len);
  if (p_len == 0 || s_len == 0) {
    /* bail out if string or pattern is empty or if pattern is longer or equal in size */
    lua_pushfalse(L);
    return 1;
  }
  if (lua_toboolean(L, 3) ||  /* explicit request? */
      strpbrk(p, SPECIALS) == NULL) {  /* or no special characters?  Agena 1.3.2 */
    s = s + s_len - p_len;
    lua_pushboolean(L, strstr(s, p) == s && p_len < s_len);  /* Agena 1.6.0 */
    return 1;
  }
  s1 = s;
  ms.L = L;
  ms.src_init = s;
  ms.src_end = s + s_len;
  do {
    const char *res;
    ms.level = 0;
    if ((res = match(&ms, s1, p, 1)) != NULL) {
      lua_pushtrue(L);
      return 1;
    }
  } while (s1++ < ms.src_end);
  lua_pushfalse(L);
  return 1;
}


static int str_ismagic (lua_State *L) {  /* October 12, 2006; tuned December 16, 2007; changed November 01, 2008,
  optimised January 15, 2011 */
  unsigned char token;
  const char *s = agn_checkstring(L, 1);
  for (; *s != '\0'; s++) {
    token = uchar(*s);
    /* `if (ispunct(token) && !diacritics[token])` is slower */
    if ((token < 'a' || token > 'z') && (token < 'A' || token > 'Z') && !diacritics[token]) {
      lua_pushtrue(L);
      return 1;
    }
  }
  lua_pushfalse(L);
  return 1;
}


static int str_islatin (lua_State *L) {  /* October 12, 2006; tuned December 16, 2007 */
  unsigned char token;
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    token = uchar(*s);
    if ((token < 'a' || token > 'z') && (token < 'A' || token > 'Z')) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_islatinnumeric (lua_State *L) {  /* based upon isAlphaNumeric; June 13, 2009 */
  unsigned char token;
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    token = uchar(*s);
    if ((token < 'a' || token > 'z') && (token < 'A' || token > 'Z') && ((token < '0' || token > '9'))) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isnumber (lua_State *L) {  /* April 22, 2007; tuned December 16, 2007 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if (uchar(*s) < '0' || uchar(*s) > '9') {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isfloat (lua_State *L) {  /* January 23, 2010, 0.30.4; tweaked January 15, 2011, 1.3.2 */
  size_t l;
  const char *s = agn_checklstring(L, 1, &l);
  struct lconv *cv = localeconv();
  char decpoint = (cv ? cv->decimal_point[0] : '.');
  int flag = 1;
  for ( ; *s != '\0'; s++) {
    if (uchar(*s) < '0' || uchar(*s) > '9') {
      if (uchar(*s) == decpoint && flag && !(l == 1))
        flag = 0;
      else {
        lua_pushfalse(L);
        return 1;
      }
    }
  }
  lua_pushboolean(L, flag == 0);
  return 1;
}


static int str_isnumeric (lua_State *L) {  /* 26.08.2012, Agena 1.7.7 */
  agn_pushboolean(L, tools_isnumericstring(agn_checkstring(L, 1)));
  return 1;
}


static int str_iscenumeric (lua_State *L) {  /* 26.08.2012, Agena 1.7.7 */
  size_t l;
  const char *s = agn_checklstring(L, 1, &l);
  int flag = 1;
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if (uchar(*s) < '0' || uchar(*s) > '9') {
      if (uchar(*s) == ',' && flag && !(l == 1))
        flag = 0;
      else {
        lua_pushfalse(L);
        return 1;
      }
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isnumberspace (lua_State *L) {  /* June 29, 2007; tuned December 16, 2007 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ;*s != '\0'; s++) {
    if ((uchar(*s) < '0' || uchar(*s) > '9') && uchar(*s) != ' ') {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isspace (lua_State *L) {  /* Agena 1.8.0, September 25, 2012 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ;*s != '\0'; s++) {
    if (uchar(*s) != ' ') {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isisospace (lua_State *L) {  /* 31.12.2012 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if (!isISOspace(*s)) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isblank (lua_State *L) {  /* Agena 1.8.0, September 25, 2012 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if (!(uchar(*s) == ' ' || uchar(*s) == '\t')) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isalpha (lua_State *L) {  /* October 12, 2006, extended May 17, 2007; tuned December 16, 2007;
  changed November 01, 2008 */
  unsigned char token;
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    token = uchar(*s);
    if ((token < 'a' || token > 'z') && (token < 'A' || token > 'Z') && !diacritics[token] ) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


/* checks whether the string s consists entirely of following special characters and returnd `true or `false`:

a special character is:
32 <white space>  033 !   034 "   035 #   036 $   037 %   038 &
039 '             040 (   041 )   042 *   043 +   044 ,   045 -
046 .             047 /   091 [   092 \   093 ]   094 ^   095 _
096 `             123 {   124 |   125 }   126 ~   127 
058 :             059 ;   060 <   061 =   062 >   063 ?   064 @
021, 245         168    173 
*/

static int str_isspec (lua_State *L ) {  /* 1.10.6, 11.04.2013 */
  unsigned char token;
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    token = uchar(*s);
    if (!((token > 31 && token < 48) || (token > 57 && token < 65) || (token > 90 && token < 97) ||
      (token > 122 && token < 128) || (token == 21) || (token == 245) || (token == 168) || (token == 173))) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isalphaspec (lua_State *L) {  /* 1.10.6, 11.04.2013 */
  unsigned char token;
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    token = uchar(*s);
    if ((token < 'a' || token > 'z') && (token < 'A' || token > 'Z') && !diacritics[token] &&
      (!((token > 31 && token < 48) || (token > 57 && token < 65) || (token > 90 && token < 97)
      || (token > 122 && token < 128) || (token == 21) || (token == 245) || (token == 168) || (token == 173)))) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isisoalpha (lua_State *L) {  /* December 31, 2012 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if (!isISOalpha(*s)) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isalphaspace (lua_State *L) {  /* June 28, 2007; tuned December 16, 2007 */
  unsigned char token;
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    token = uchar(*s);
    if ((token < 'a' || token > 'z') && (token < 'A' || token > 'Z') && !diacritics[token] && (token != ' ')) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_islowerlatin (lua_State *L) {  /* April 30, 2007; tuned December 16, 2007 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if (uchar(*s) < 'a' || uchar(*s) > 'z') {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isloweralpha (lua_State *L) {  /* November 09, 2008 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if ((uchar(*s) < 'a' || uchar(*s) > 'z') && !lowerDiacritics[uchar(*s)]) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isisolower (lua_State *L) {  /* December 31, 2012 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if (!isISOlower(*s)) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isupperlatin (lua_State *L) {  /* 0.21.0, April 19, 2009 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if (uchar(*s) < 'A' || uchar(*s) > 'Z') {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isupperalpha (lua_State *L) {  /* 0.21.0, April 19, 2009 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if ((uchar(*s) < 'A' || uchar(*s) > 'Z') && !upperDiacritics[uchar(*s)]) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isisoupper (lua_State *L) {  /* 31.12.2012 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if (!isISOupper(*s)) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isalphanumeric (lua_State *L) {  /* October 12, 2006; extended May 17, 2007; tuned December 16, 2007 */
  unsigned char token;
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    token = uchar(*s);
    if ((token < 'a' || token > 'z') && (token < 'A' || token > 'Z') && !diacritics[token] &&
      ((token < '0' || token > '9'))) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_isisoprint (lua_State *L) {  /* 31.12.2012 */
  const char *s = agn_checkstring(L, 1);
  if (*s == '\0') {
    lua_pushfalse(L);
    return 1;
  }
  for ( ; *s != '\0'; s++) {
    if (!isISOprint(*s)) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int str_rtrim (lua_State *L) {
  size_t l;
  const char *s = agn_checklstring(L, 1, &l);
  const char *d = luaL_optstring(L, 2, " ");
  while (l > 0 && uchar(s[l-1]) == *d) l--;
  lua_pushlstring(L, s, l);
  return 1;
}


static int str_ltrim (lua_State *L) {
  const char *s = agn_checkstring(L, 1);
  const char *d = luaL_optstring(L, 2, " ");
  /* 0.13.4 patch; do not optimize `while (*s == ' ') s++` to `while (*s++ == ' ')` ! */
  while (*s == *d) s++;
  lua_pushstring(L, s);
  return 1;
}


/* 1.11.2, delete leading and trailing white spaces or the given given leading o trailing character */
static int str_lrtrim (lua_State *L) {
  const char *s, *d;
  size_t l;
  s = agn_checklstring(L, 1, &l);
  d = luaL_optstring(L, 2, " ");
  while (*s == *d) {s++; l--;};  /* remove leading spaces */
  while (l > 0 && uchar(s[l-1]) == *d) l--;   /* remove trailing spaces */
  lua_pushlstring(L, s, l);
  return 1;
}


static int str_isolower (lua_State *L) {  /* 31.12.2012 */
  luaL_Buffer b;
  const char *s = luaL_checkstring(L, 1);
  luaL_buffinit(L, &b);
  while (*s != '\0') { luaL_addchar(&b, toISOlower(*s)); s++; };
  luaL_pushresult(&b);
  return 1;
}


static int str_isoupper (lua_State *L) {  /* 31.12.2012 */
  luaL_Buffer b;
  const char *s = luaL_checkstring(L, 1);
  luaL_buffinit(L, &b);
  while (*s != '\0') { luaL_addchar(&b, toISOupper(*s)); s++; }
  luaL_pushresult(&b);
  return 1;
}


/* Applies a function f to the ASCII value of each character in string str and returns a new string. The function
   must return an integer in the range 0 .. 255, otherwise an error is issued. */

static int str_transform (lua_State *L) {  /* 31.12.2012 */
  luaL_Buffer b;
  const char *s;
  lua_Number x;
  int firstarg;
  firstarg = lua_type(L, 1);
  luaL_typecheck(L, firstarg == LUA_TFUNCTION, 1, "procedure expected", firstarg);
  s = luaL_checkstring(L, 2);
  luaL_buffinit(L, &b);
  while (*s != '\0') {
    lua_pushvalue(L, 1);  /* push function */
    lua_pushinteger(L, *s);
    lua_call(L, 1, 1);
    if (!agn_isnumber(L, -1))
      luaL_error(L, "Error in " LUA_QS ": function must return a number, got %s.", "strings.transform",
        lua_typename(L, lua_type(L, -1)));
    x = agn_tonumber(L, -1);
    agn_poptop(L);
    if (x < 0 || x > 255 || ISFLOAT(x))
      luaL_error(L, "Error in " LUA_QS ": function must return an integer in 0 .. 255.", "strings.transform");
    luaL_addchar(&b, x);
    s++;
  }
  luaL_pushresult(&b);
  return 1;
}


static int str_hits (lua_State *L) {
  size_t l1, l2;
  const char *s1;
  const char *src = agn_checklstring(L, 1, &l1);
  const char *p = agn_checklstring(L, 2, &l2);
  size_t n = 0;
  int init = 0;
  if (l2 == 0) {  /* else infinite loop */
    lua_pushnumber(L, 0);
    return 1;
  }
  if (lua_toboolean(L, 3) ||  /* Agena 1.3.2: explicit request? */
      strpbrk(p, SPECIALS) == NULL) {  /* or no special characters? */
    while (1) {
      s1 = lmemfind(src+init, l1-init, p, l2);
      if (s1) {
        init += (s1-(src+init)) + l2;
        n++;
      } else
        break;
    }
  } else {  /* Agena 1.3.2 */
    int anchor;
    MatchState ms;
    anchor = (*p == '^') ? (p++, 1) : 0;
    s1 = src;
    ms.L = L;
    ms.src_init = src;
    ms.src_end = src + l1;
    do {
      const char *res;
      ms.level = 0;
      if ((res=match(&ms, s1, p, 1)) != NULL) n++;
    } while (s1++ < ms.src_end && !anchor);
  }
  lua_pushnumber(L, (lua_Number)n);
  return 1;
}


/* 0.14.0 as of April 08, 2009, inspired by the REXX function `words`; the Regina REXX interpreter
   features a very elegant and shorter C implementation, but it is not faster. */
static int str_words (lua_State *L) {
  size_t i, len;
  int c, flag;
  const char *s = agn_checklstring(L, 1, &len);
  const char *d = luaL_optstring(L, 2, " ");
  flag = lua_toboolean(L, 3);  /* changed 1.5.1: explicit request to ignore succeeding delimitors ? */
  i = c = 0;
  while (flag && *s == *d) { s++; i++; }  /* skip leading delimitors */
  if ((i == len-1 && i != 0) || len == 0)  /* string consists entirely of delimitors or is empty ? */
    c = -1;  /* do not count anything */
  else {
    for (; *s != '\0'; i++, s++) {
      if (*s == *d) {
        if (flag && *(s+1) != *d) { c++; continue; };
        if (!flag) c++;
      }
    }
    s--;  /* step back one character */
    /* if it ends with a delimitor, decrement c because it was already counted in the loop above */
    if (flag && *s == *d) c--;
  }
  lua_pushnumber(L, c+1);
  return 1;
}


/* 0.20.0, inspired by the REXX function `delstr`; patched 1.10.0 */
static int str_remove (lua_State *L) {
  size_t pos, origlen;
  long int offset, nchars;
  const char *orig;
  orig = agn_checklstring(L, 1, &origlen);
  pos = posrelat(agnL_checkint(L, 2), origlen) - 1;  /* 2nd argument is the position */
  nchars = agnL_optinteger(L, 3, 1);  /* number of characters to be deleted, Agena 1.10.0 */
  if (nchars < 1)
    luaL_error(L, "in " LUA_QL("strings.remove") ": third argument must be positive.");
  if (pos < 0 || pos >= origlen)
    luaL_error(L, "in " LUA_QL("strings.remove") ": index %d out of range.", pos + 1);
  offset = pos + nchars;
  if (offset > origlen) offset = origlen;  /* avoid invalid accesses beyond the lenght of the string */
  /* directly writing the string is slower, e.g. using concat and substr(l) ! */
  lua_pushlstring(L, orig, pos);  /* push left part of original string */
  lua_pushstring(L, orig+offset);  /* push rest of original string */
  lua_concat(L, 2);
  return 1;
}


/* 1.2.0, inspired by the REXX function `delstr` (i.e. its counterpart `delstr`) */
static int str_include (lua_State *L) {
  size_t pos, origlen, newstrlen;
  const char *orig, *newstr;
  orig = agn_checklstring(L, 1, &origlen);
  pos = posrelat(agnL_checkint(L, 2), origlen);  /* 2nd argument is the position */
  newstr = agn_checklstring(L, 3, &newstrlen);
  if (pos < 1 || pos > origlen + 1)  /* 1.5.0 patch */
    luaL_error(L, "in " LUA_QL("strings.include") ": index %d out of range.", pos);
  /* directly writing the string is slower, e.g. using concat and substr(l) ! */
  lua_pushlstring(L, orig, pos-1);  /* push left part of original string */
  lua_pushlstring(L, newstr, newstrlen);
  lua_pushstring(L, orig+pos-1);  /* push rest of original string */
  lua_concat(L, 3);
  return 1;
}


/* 1.5.1, 14.11.2011, exract the given fields in the given order from a string;
   extended 2.0.0 RC4 to support sequences */
static int str_fields (lua_State *L) {
  const char *e, *s, *s1, *sep;
  int flag, isseq;
  size_t c, i, nargs, n, l1, l2, init;
  lua_Integer *inds, index;
  c = init = n = 0;  /* position of token in string */
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs < 2)
    luaL_error(L, "in " LUA_QL("strings.fields") ": expected at least two arguments.");
  s = agn_checklstring(L, 1, &l1);
  if ((flag = agn_isstring(L, nargs)))
    sep = agn_tostring(L, nargs);
  else
    sep = " ";
  if (nargs == 2 && flag)
    luaL_error(L, "in " LUA_QL("strings.fields") ": expected at least one index.");
  l2 = strlen(sep);
  isseq = lua_isseq(L, 2);  /* 2.0.0 RC4 */
  if (isseq) {
    nargs = agn_seqsize(L, 2);
    if (nargs == 0)
      luaL_error(L, "in " LUA_QL("strings.fields") ": expected at least one index in sequence.");
  } else
    nargs =  nargs - flag - 1;  /* if last argument is an optional delimiter, do not process it */
  agn_createseq(L, nargs);
  if (l2 == 0 || strcmp(s, sep) == 0) return 1;  /* delimter is the empty string or delimiter has not been found in string ?  */
  while (1) {  /* determine number of words */
    s1 = lmemfind(s+init, l1-init, sep, l2);
    if (s1) {
      init += (s1-(s+init)) + l2;
      n++;
    } else
      break;
  }
  n++;
  /* put all indices passed in the call into an array (this may not be an ellegant implementation but it saves memory) */
  inds = agn_malloc(L, nargs*sizeof(lua_Integer), "strings.fields", NULL);  /* 1.9.1 */
  if (isseq) {
    for (i=1; i <= nargs; i++) {
      index = lua_seqgetinumber(L, 2, i);
      if (index < 0) index += n + 1;
      if (index <= 0 || index > n)
        luaL_error(L, "in " LUA_QL("strings.fields") ": index %d out of range.", index);
      inds[i-1] = index;
      lua_pushstring(L, "");  /* fill sequence with empty strings for it might be later be filled in non-ascending field order */
      lua_seqseti(L, -2, i);
    }
  } else {
    for (i=1; i <= nargs; i++) {
      index = agn_checkinteger(L, i+1);
      if (index < 0) index += n + 1;
      if (index <= 0 || index > n)
        luaL_error(L, "in " LUA_QL("strings.fields") ": index %d out of range.", index);
      inds[i-1] = index;
      lua_pushstring(L, "");  /* fill sequence with empty strings for it might be later be filled in non-ascending field order */
      lua_seqseti(L, -2, i);
    }
  }
  /* collect all matching tokens into a sequence */
  while ((e=strstr(s, sep)) != NULL) {
    c++;
    for (i=0; i < nargs; i++) {
      if (inds[i] == c) {
        lua_pushlstring(L, s, e-s);
        lua_seqseti(L, -2, i+1);
      }
    }
    s = e + l2;
  }
  /* process last token */
  if (strlen(s) != 0) {  /* not the empty string ? */
    c++;
    for (i=0; i < nargs; i++) {
      if (inds[i] == c) {
        lua_seqsetistring(L, -1, i+1, s);
      }
    }
  }
  /* now return the respective fields */
  xfree(inds);
  return 1;
}



/*************************************************************************************************
  Implementation of Damerau-Levenshtein Distance, a blend of code written by
  Lorenzo Seidenari (sixmoney [guess-what] virgilio [guess-what] it) and Anders Sewerin Johansen
  Agena 1.6.0, May 18, 2012
**************************************************************************************************/

/* the minimum of three values */
static int minimum (int a, int b, int c) {
  int min = a;
  if (b < min) min = b;
  if (c < min) min = c;
  return min;
}


static int str_dleven (lua_State *L) {
  /* Step 1 */
  int k, i, j, cost, *d, distance;
  size_t n, m;
  const char *s = agn_checklstring(L, 1, &n);
  const char *t = agn_checklstring(L, 2, &m);
  distance = 0;
  if (n != 0 && m != 0) {
    m++;
    n++;
    d = agn_malloc(L, (sizeof(int))*(m*n), "strings.dleven", NULL);  /* 1.9.1 */
    /* Step 2 */
    for (k=0; k < n; k++) d[k]=k;
    for (k=0; k < m; k++) d[k*n]=k;
    /* Step 3 and 4 */
    for (i=1; i < n; i++) {
      for (j=1; j < m; j++) {
        /* Step 5 */
        if (s[i-1] == t[j-1])
          cost = 0;
        else
          cost = 1;
        /* Step 6 */
        distance = minimum(d[(j-1)*n+i] + 1, d[j*n+i-1] + 1, d[(j-1)*n+i-1] + cost);
        /* Step 6A: Cover transposition, in addition to deletion, insertion and substitution. This step is taken from:
           Berghel, Hal ; Roach, David : "An Extension of Ukkonen's Enhanced Dynamic Programming ASM Algorithm"
           (http://www.acm.org/~hlb/publications/asm/asm.html) */
        if (i > 2 && j > 2) {
          /* int trans = d[(i-2)*(j-2)] + 1; */
          int trans = d[(j-2)*n+(i-2)] + 1;
          if (s[i-2] != t[j-1]) trans++;
          if (s[i-1] != t[j-2]) trans++;
          if (distance > trans) distance = trans;
        }
        d[j*n+i] = distance;
      }
    }
    distance = d[n*m-1];
    xfree(d);
    lua_pushinteger(L, distance);
  }
  else {  /* infinity means that one or both strings are empty. */
    lua_pushnumber(L, HUGE_VAL);
  }
  return 1;
}

/* Creates a dynamically allocated copy of a string, changing the encoding from ISO-8859-15 to UTF-8. */

static int str_toutf8 (lua_State *L) {
  char *r;
  r = latin9_to_utf8(agn_checkstring(L, 1));
  if (r == NULL)
    luaL_error(L, "in " LUA_QL("strings.toutf8") ": memory allocation error.");
  lua_pushstring(L, (const char*)r);
  xfree(r);
  return 1;
}


/* Creates a dynamically allocated copy of a string, changing the encoding from UTF-8 to ISO-8859-1/15.
   Unsupported code points are ignored. */

static int str_tolatin (lua_State *L) {
  char *r;
  r = utf8_to_latin9(agn_checkstring(L, 1));
  if (r == NULL)
    luaL_error(L, "in " LUA_QL("strings.tolatin") ": memory allocation error.");
  lua_pushstring(L, (const char*)r);
  xfree(r);
  return 1;
}


/* detects that the given string is in UTF-8 encoding */

static int str_isutf8 (lua_State *L) {
  int flag;
  const char *str;
  str = agn_checkstring(L, 1);
  flag = 0;
  lua_pushboolean(L, is_utf8(str));
  while (*str != '\0') {
    if ((*str & 0xc0) == 0x80) { flag = 1; break; }
    str++;
  }
  lua_pushboolean(L, flag);
  return 2;
}


/* determines the size of an UTF-8 string */

static int str_utf8size (lua_State *L) {
  lua_pushnumber(L, size_utf8(agn_checkstring(L, 1)));
  return 1;
}


static int str_glob (lua_State *L) {
  const char *str, *pat;
  str = agn_checkstring(L, 1);
  pat = agn_checkstring(L, 2);
  agn_pushboolean(L, glob(pat, str));
  return 1;
}


/* substitution table: upper to lower case */

static unsigned char uppercase[256] = {
   0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,  /* 0 .. 7*/
   0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,  /* 8 .. 15 */
   0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,  /* 16 .. 23 */
   0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,  /* 24 .. 31 */
   0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,  /* 32 .. 39 */
   0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,  /* 40 .. 47 */
   0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,  /* 48 .. 55 */
   0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,  /* 56 .. 63 */
   0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,  /* 64 .. 71 */
/* 064@  065A  066B  067C  068D  069E  070F  071G */
   0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,  /* 72 .. 79 */
/* 072H  073I  074J  075K  076L  077M  078N  079O */
   0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,  /* 80 .. 87 */
/* 080P  081Q  082R  083S  084T  085U  086V  087W */
   0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,  /* 88 .. 95 */
/* 088X  089Y  090Z  091   092   093   094   095  */
   0x60, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,  /* 96 .. 103 */
/* 096`  097a  098b  099c  100d  101e  102f  103g */
   0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,  /* 104 .. 111 */
/* 104h  105i  106j  107k  108l  109m  110n  111o */
   0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,  /* 112 .. 119 */
/* 112p  113q  114r  115s  116t  117u  118v  119w */
   0x58, 0x59, 0x5a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,  /* 120 .. 127 */
/* 120x  121y  122z  123{  124|  125}  126~  127 */
   0x80, 0x9a, 0x90, 0xb6, 0x8e, 0xb7, 0x8f, 0x80,  /* 128 .. 135, ok */
/* 128  129  130  131  132  133  134  135 */
   0xd2, 0xd3, 0xd4, 0xd8, 0xd7, 0xde, 0x8e, 0x8f,  /* 136 .. 143 ok */
/* 136  137  138  139  140  141  142  143 */
   0x90, 0x92, 0x92, 0xe2, 0x99, 0xe3, 0xea, 0xeb,  /* 144 .. 151 ok */
/* 144  145  146  147  148  149  150  151 */
   0x98, 0x99, 0x9a, 0x9d, 0x9c, 0x9d, 0x9e, 0x9f,  /* 152 .. 159 ok */
/* 152? 153  154  155  156  157  158  159 */
   0xb5, 0xd6, 0xe0, 0xe9, 0xa5, 0xa5, 0xa6, 0xa7,  /* 160 .. 167 ok */
/* 160  161  162  163  164  165  166  167 */
   0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,  /* 168 .. 175 */
/* 168  169  170  171  172  173  174  175 */
   0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,  /* 176 .. 183 */
/* 176  177  178  179  180  181  182  183 */
   0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,  /* 184 .. 191 */
/* 184  185  186  187+  188+  189  190  191+ */
   0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc7, 0xc7,  /* 192 .. 199 ok */
/* 192+  193-  194-  195+  196-  197+  198  199 */
   0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,  /* 200 .. 207 ok */
/* 200+  201+  202-  203-  204  205-  206+  207 */
   0xd1, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,  /* 208 .. 215 ok */
/* 208  209  210  211  212  213i  214  215 */
   0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,  /* 216 .. 223 ok */
/* 216  217+  218+  219  220_  221  222  223 */
   0xe0, 0xe1, 0xe2, 0xe3, 0xe5, 0xe5, 0xe6, 0xe8,  /* 224 .. 231 ok */
/* 224  225  226  227  228  229  230  231 */
   0xe8, 0xe9, 0xea, 0xeb, 0xed, 0xed, 0xee, 0xef,  /* 232 .. 239 ok */
/* 232  233  234  235  236  237  238  239 */
   0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,  /* 240 .. 247 */
/* 240  241  242=  243  244  245  246  247 */
   0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff   /* 248 .. 255 */
/* 248  249  250  251  252  253  254  255 */
};


/* str_capitalise: The function is 30% faster than an Agena implementation. */

static int str_capitalise (lua_State *L) {  /* 1.10.0 */
  const char *str;
  size_t size;
  char r[2];
  str = agn_checklstring(L, 1, &size);
  if (size == 0) {
    lua_pushstring(L, "");
    return 1;
  }
  r[0] = uppercase[uchar(*str)]; r[1] = '\0';
  if (size == 1)
    lua_pushstring(L, r);
  else {
    str++;
    /* the following statements are 5 % faster than calling agnhlps.c/concat */
    lua_pushstring(L, r);
    lua_pushstring(L, str);
    lua_concat(L, 2);
  }
  return 1;
}


/* 1.10.2, just to play around, remains undocumented, call strings.hash(str, << x -> x shift 5 + x >>, numberofwords, 5381); */
static int str_hash (lua_State *L) {
  const char *str;
  int c;
  unsigned long size, h;
  str = agn_checkstring(L, 1);
  luaL_argcheck(L, 2, LUA_TFUNCTION, "procedure expected");
  size = agn_checknumber(L, 3);
  h = agnL_optnumber(L, 4, 5381);
  while ((c = uchar(*str++))) {
    lua_pushvalue(L, 2);
    lua_pushnumber(L, h);
    lua_call(L, 1, 1);
    if (!agn_isnumber(L, -1))
      luaL_error(L, "Error in " LUA_QS ": function return is not a number.", "strings.hash");
    h = agn_tonumber(L, -1) + c;
    agn_poptop(L);
  }
  lua_pushnumber(L, h);
  lua_pushnumber(L, h % size);
  return 2;
}


/* static int str_repl (lua_State *L) {
  size_t lp, lr;
  const char *str, *pattern, *replacement;
  char *result;
  str = agn_checkstring(L, 1);
  pattern = agn_checklstring(L, 2, &lp);
  if (lp == 0) {
    luaL_error(L, "Error in " LUA_QS ": pattern is the empty string.", "strings.repl");
  }
  replacement = agn_checklstring(L, 3, &lr);
  result = strreplace(str, pattern, replacement);
  lua_pushstring(L, result);
  xfree(result);
  return 1;
} */


static int str_separate (lua_State *L) {
  size_t c, l;
  const char *string, *delim;
  char *result, *str;
  string = agn_checkstring(L, 1);
  delim = agn_checklstring(L, 2, &l);
  if (l == 0) {
    lua_pushfail(L);
    return 1;
  }
  str = strdup(string);
  if (str == NULL) {
    luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "strings.separate");
  }
  result = strtok(str, delim);
  if (result == NULL) {
    lua_pushfail(L);
    goto bailout;
  }
  c = 1;
  agn_createseq(L, 10);
  lua_seqsetistring(L, -1, c, result);
  while ((result = strtok(NULL, delim)) != NULL)
    lua_seqsetistring(L, -1, ++c, result);

bailout:
  xfree(str);
  return 1;
}


static const luaL_Reg strlib[] = {
  {"capitalise", str_capitalise},         /* added on February 28, 2013 */
  {"dleven", str_dleven},                 /* added on October 10, 2011 */
  {"dump", str_dump},
  {"fields", str_fields},                 /* added on November 13, 2011 */
  {"find", str_find},
  {"format", str_format},
  {"gfind", gfind_nodef},
  {"glob", str_glob},                     /* added on February 20, 2013 */
  {"gmatch", gmatch},
  {"gsub", str_gsub},
  {"hash", str_hash},                     /* added March 20, 2013 */
  {"hits", str_hits},                     /* added on January 26, 2008 */
  {"include", str_include},               /* added on December 22, 2010 */
  {"isabbrev", str_isabbrev},             /* added on April 10, 2009 */
  {"isalpha", str_isalpha},               /* added on December 17, 2006 */
  {"isalphanumeric", str_isalphanumeric}, /* added on December 17, 2006 */
  {"isalphaspec", str_isalphaspec},       /* added on April 11, 2013 */
  {"isalphaspace", str_isalphaspace},     /* added on June 27, 2007 */
  {"isblank", str_isblank},               /* added on September 25, 2012 */
  {"iscenumeric", str_iscenumeric},       /* added on August 26, 2012 */
  {"isending", str_isending},             /* added on April 10, 2009 */
  {"isfloat", str_isfloat},               /* added on January 23, 2010 */
  {"isisoalpha", str_isisoalpha},         /* added on December 31, 2012 */
  {"isisolower", str_isisolower},         /* added on December 31, 2012 */
  {"isisoprint", str_isisoprint},         /* added on December 31, 2012 */
  {"isisospace", str_isisospace},         /* added on December 31, 2012 */
  {"isisoupper", str_isisoupper},         /* added on December 31, 2012 */
  {"islatin", str_islatin},               /* added on December 18, 2006 */
  {"islatinnumeric", str_islatinnumeric}, /* added on June 13, 2009 */
  {"isloweralpha", str_isloweralpha},     /* added on November 09, 2008 */
  {"islowerlatin", str_islowerlatin},     /* added on April 30, 2007 */
  {"isolower", str_isolower},             /* added on December 31, 2012 */
  {"isoupper", str_isoupper},             /* added on December 31, 2012 */
  {"ismagic", str_ismagic},               /* added on October 14/15, 2006 */
  {"isnumber", str_isnumber},             /* added on April 22, 2007 */
  {"isnumeric", str_isnumeric},           /* added on August 26, 2012 */
  {"isnumberspace", str_isnumberspace},   /* added on June 29, 2007 */
  {"isspace", str_isspace},               /* added on September 25, 2012 */
  {"isspec", str_isspec},                 /* added on April 11, 2013 */
  {"isupperalpha", str_isupperalpha},     /* added on April 19, 2009 */
  {"isupperlatin", str_isupperlatin},     /* added on April 19, 2009 */
  {"isutf8", str_isutf8},                 /* added on January 04, 2013 */
  {"lrtrim", str_lrtrim},                 /* added on April 30, 2013 */
  {"ltrim", str_ltrim},                   /* added on June 24, 2008 */
  {"match", str_match},
  {"remove", str_remove},                 /* added on April 10, 2009 */
  {"repeat", str_repeat},
  {"reverse", str_reverse},
  {"rtrim", str_rtrim},                   /* added on June 24, 2008 */
  {"separate", str_separate},             /* added on June 24, 2013 */
  {"tobytes", str_tobytes},
  {"tochars", str_tochars},
  {"tolatin", str_tolatin},               /* added on January 04, 2013 */
  {"toutf8", str_toutf8},                 /* added on January 04, 2013 */
  {"utf8size", str_utf8size},             /* added on January 04, 2013 */
  {"transform", str_transform},           /* added on January 03, 2013 */
  {"words", str_words},                   /* added on April 07, 2009 */
  {NULL, NULL}
};


static void createmetatable (lua_State *L) {
  lua_createtable(L, 0, 1);  /* create metatable for strings */
  lua_pushliteral(L, "");  /* dummy string */
  lua_pushvalue(L, -2);
  lua_setmetatable(L, -2);  /* set string metatable */
  agn_poptop(L);  /* pop dummy string */
  lua_pushvalue(L, -2);  /* string library... */
  lua_setfield(L, -2, "__index");  /* ...is the __index metamethod */
  agn_poptop(L);  /* pop metatable */
}


/*
** Open string library
*/
LUALIB_API int luaopen_string (lua_State *L) {
  luaL_register(L, LUA_STRLIBNAME, strlib);
  createmetatable(L);
  return 1;
}

