/*
** $Id: lseq.c v 0.1, based on ltable.c v2.32, 2006/01/18 11:49:02 roberto Exp $
** Agena Sequences
** See Copyright Notice in agena.h
*/


/*
** Implementation of Sequences
*/

#define lseq_c
#define LUA_CORE

#include <math.h>

#include "agena.h"

#include "ldebug.h"
#include "ldo.h"
#include "lgc.h"
#include "lmem.h"
#include "lobject.h"
#include "lstate.h"
#include "lseq.h"
#include "llimits.h"  /* for MAX_INT */


/*
** max size of seq is 2^MAXBITS
*/
#if LUAI_BITSINT > 26
#define MAXBITS      26
#else
#define MAXBITS      (LUAI_BITSINT-2)
#endif


/*
** }=============================================================
*/


static void setarrayvector (lua_State *L, Seq *t, int size) {
  int i;
  luaM_reallocvector(L, t->array, t->maxsize, size, TValue);
  for (i=t->maxsize; i<size; i++)
    setnilvalue(&t->array[i]);
  t->maxsize = size;
}


/* create a new sequence */

Seq *agnSeq_new (lua_State *L, int n) {
  Seq *t = luaM_new(L, Seq);
  luaC_link(L, obj2gco(t), LUA_TSEQ);
  t->flags = cast_byte(~0);
  /* temporary values (kept only if some malloc fails) */
  t->array = NULL;
  t->size = 0;
  t->maxsize = 0;
  t->metatable = NULL;
  t->type = NULL;
  setarrayvector(L, t, n<1 ? 1 : n);
  return t;
}

/* set a new value to the `end` of a sequence */

TValue *agnSeq_set (lua_State *L, Seq *t, const TValue *val) {
  TValue *n;
  if (t->maxsize < t->size+1) {  /* grow sequence */
    if (t->maxsize<<1 < MAX_INT) {  /* Agena 1.06 patch; avoid overflows; this check and the << op is 100 % faster than
      the computesizes routine for tables, which also quits with a memory error with the same number
      of entries (100'000'000) */
      agnSeq_resize(L, t, t->maxsize<<1);  /* x<<1 is double its current size */
    } else {
      return NULL;
    }
  }
  if (ttisnil(val))
    return (TValue *)val;  /* null cannot be added */
  n = seqitem(t, t->size);  /* get next reserved memory chunk */
  *n = *val;
  luaC_barrierseq(L, t, val);
  t->size++;
  return n;
}


static void purge (Seq *t, int index) {  /* 0.28.1 */
  size_t i;  /* Agena 1.1.0 */
  TValue *m;
  TValue *n = NULL;  /* to avoid compiler warnings */
  /* move items right of item to be deleted to the left */
  if (index < t->size) {  /* item to be deleted is not at the end -> move elements */
    for (i=index; i < t->size; i++) {
      m = seqitem(t, i-1);
      n = seqitem(t, i);
      *m = *n;
    }
  } else
    n = seqitem(t, index-1);
  setnilvalue(n);
  t->size--;
}


/* set a new value to a sequence at `index` i;
   returns:
       0:    index out of range
       1:    success
*/

int agnSeq_seti (lua_State *L, Seq *t, int index, const TValue *val) {
  t->flags = 0;
  /* sequence must be resized for it has become too small ? */
  if (index == t->size+1 && t->maxsize < t->size+1) {
    if (t->maxsize<<1 < MAX_INT) {  /* Agena 1.06 patch; avoid overflows; this check and the << op is 100 % faster than
      the computesizes routine for tables, which as agnSeq_seti also quits with a memory error with
      the same number of entries (100'000'000) */
      agnSeq_resize(L, t, t->maxsize<<1);  /* x<<1 is double its size */
    } else {
      luaG_runerror(L, "memory allocation error.");  /* 0.28.1 */
      return -1;
    }
  } else if ((index > t->size + 1) || index < 1) {
    return 0;  /* index out of range */
  }
  if (index > t->size) {  /* seq is empty or must be extended ?  -> add new value to seq */
    TValue *nnew;
    if (ttisnil(val)) return 1;  /* null cannot be added */
    nnew = seqitem(t, index-1);  /* get next reserved memory chunk */
    *nnew = *val;
    t->size++;
  } else {  /* replace value at existing index */
    if (!ttisnil(val)) {  /* replace old value with new one */
      TValue *n = seqitem(t, index-1);
      if (luaO_rawequalObj(n, val)) return 1;
      *n = *val;
    } else {  /* delete value/node */
      purge(t, index);
      return 1;
    }
  }
  luaC_barrierseq(L, t, val);
  return 1;
}


int agnSeq_constructorseti (lua_State *L, Seq *t, int index, const TValue *val) {
  TValue *nnew;
  t->flags = 0;
  if (index < 1 || index > t->maxsize || ttisnil(val)) return 1;  /* failure */
  nnew = seqitem(t, index-1);  /* get next reserved memory chunk */
  *nnew = *val;
  t->size++;
  luaC_barrierseq(L, t, val);
  return 0;
}


/* check whether a value exists in a sequence */

int agnSeq_get (Seq *t, const TValue *val) {
  TValue *n;
  int i;
  for (i=0; i < t->size; i++) {
    n = seqitem(t, i);
    if (luaO_rawequalObj(n, val))
      return 1;
  }
  return 0;
}


/* get the i-th value that has been added to a sequence */

const TValue *agnSeq_geti (Seq *t, size_t index) {
  if ((index > t->size) || index < 1)
    return NULL;  /* index out of range */
  else
    return seqitem(t, index-1);
}


const TValue *agnSeq_rawgeti (Seq *t, const TValue *ind) {
  int index;
  if (ttisnumber(ind))
    index = (int)nvalue(ind);
  else
    return NULL;
  if ((index > t->size) || index < 1)
    return NULL;  /* index out of range */
  return seqitem(t, index-1);
}


void agnSeq_resize (lua_State *L, Seq *t, int newsize) {
  int i;
  if (newsize == t->maxsize) return;  /* do nothing */
  luaM_reallocvector(L, t->array, t->maxsize, newsize, TValue);
  for (i=t->maxsize; i<newsize; i++)  /* if sequence grows, set new slots to null */
    setnilvalue(&t->array[i]);
  t->maxsize = newsize;
}


void agnSeq_delete (lua_State *L, Seq *t, const TValue *val) {
  TValue *n;
  int i;
  if (t->size == 0) return;
  for (i=0; i < t->size; i++) {
    n = seqitem(t, i);
    if (luaO_rawequalObj(n, val)) {
      purge(t, i+1);  /* 0.28.1 */
      agnSeq_delete(L, t, val);
      return;
    }
  }
}


/* free a sequence */

void agnSeq_free (lua_State *L, Seq *t) {
  luaM_freearray(L, t->array, t->maxsize, TValue);
  luaM_free(L, t);
}


int agnSeq_next (lua_State *L, Seq *t, StkId key) {  /* 0.24.1 */
  int i;
  if (ttisnil(key))  /* first iteration */
    i = 0;
  else if (ttisnumber(key))
    i = nvalue(key);  /* find original element */
  else {
    i = -1;  /* to avoid compiler warnings */
    luaG_runerror(L, "invalid key to " LUA_QL("next"));
  }
  if (i < t->size) {
    setnvalue(key, cast_num(i+1));
    setobj2s(L, key+1, seqitem(t, i));
    return 1;
  }
  return 0;  /* no more elements */
}

/* a version of agnSeq_next to iterate functions is not faster than doing it the primitive way */

