/*
** $Id: lparser.c,v 2.42 2006/06/05 15:57:59 roberto Exp $
** Lua/Agena Parser
** See Copyright Notice in agena.h
*/


#include <string.h>
#include <stdlib.h>  /* for `exit' and `system' */
#include <stdio.h>   /* for fputs, stdout, ... */

#define lparser_c
#define LUA_CORE

#include "agena.h"

#include "lcode.h"
#include "ldebug.h"
#include "ldo.h"
#include "lfunc.h"
#include "llex.h"
#include "lmem.h"
#include "lobject.h"
#include "lopcodes.h"
#include "lparser.h"
#include "lstate.h"
#include "lstring.h"
#include "ltable.h"

#include "agnhlps.h"


#define hasmultret(k)      ((k) == VCALL || (k) == VVARARG)

#define getlocvar(fs, i)   ((fs)->f->locvars[(fs)->actvar[i]])

#define luaY_checklimit(fs,v,l,m)   if ((v)>(l)) errorlimit(fs,l,m)


/*
** nodes for block list (list of active blocks)
*/
typedef struct BlockCnt {
  struct BlockCnt *previous;  /* chain */
  int breaklist;  /* list of jumps out of this loop */
  int continuelist;  /* list of jumps to the loop's test */
  int redolist;      /* for `redo` statement, Agena 2.1 RC 1 */
  int redojumpback;  /* for `redo` statement, Agena 2.1 RC 1 */
  lu_byte nactvar;  /* # active locals outside the breakable structure */
  lu_byte upval;  /* true if some variable in the block is an upvalue */
  lu_byte isbreakable;  /* 0: normal block, 1: loop, 2: try-catch, 2.1 RC 2 */
} BlockCnt;



/*
** prototypes for mostly recursive non-terminal functions
*/
static void chunk (LexState *ls);
static void defchunk (LexState *ls);  /* changed 0.5.2 */
static int cond (LexState *ls);       /* changed 0.5.2 */
static void expr (LexState *ls, expdesc *v);  /* changed 0.5.2 */
static BinOpr funcargexpr (LexState *ls, expdesc *v);  /* added 0.12.0 */
static void primaryexp (LexState *ls, expdesc *v);
static void simpleexp (LexState *ls, expdesc *v);  /* added 1.4 */

static void anchor_token (LexState *ls) {
  if (ls->t.token == TK_NAME || ls->t.token == TK_STRING) {
    TString *ts = ls->t.seminfo.ts;
    luaX_newstring(ls, getstr(ts), ts->tsv.len);
  }
}


static void error_expected (LexState *ls, int token) {
  luaX_syntaxerror(ls,
      luaO_pushfstring(ls->L, LUA_QS " expected", luaX_token2str(ls, token)));
}


static void error_expected2 (LexState *ls, int token1, int token2) {
  luaX_syntaxerror(ls,
      luaO_pushfstring(ls->L, LUA_QS " or " LUA_QS " expected",
        luaX_token2str(ls, token1),
        luaX_token2str(ls, token2)));
}


static void errorlimit (FuncState *fs, int limit, const char *what) {
  const char *msg = (fs->f->linedefined == 0) ?
    luaO_pushfstring(fs->L, "main procedure has more than %d %s", limit, what) :
    luaO_pushfstring(fs->L, "procedure at line %d has more than %d %s",
                            fs->f->linedefined, limit, what);
  luaX_lexerror(fs->ls, msg, 0);
}


static int testnext (LexState *ls, int c) {
  if (ls->t.token == c) {
    luaX_next(ls);
    return 1;
  }
  else return 0;
}


static void check (LexState *ls, int c) {
  if (ls->t.token != c)
    error_expected(ls, c);
}


static void checknext (LexState *ls, int c) {
  check(ls, c);
  luaX_next(ls);
}


#define check_condition(ls,c,msg)   { if (!(c)) luaX_syntaxerror(ls, msg); }

static void check_match (LexState *ls, int what, int who, int where) {
  if (!testnext(ls, what)) {
    if (where == ls->linenumber)
      error_expected(ls, what);
    else {
      luaX_syntaxerror(ls, luaO_pushfstring(ls->L,
             LUA_QS " expected (to close " LUA_QS " at line %d)",
              luaX_token2str(ls, what), luaX_token2str(ls, who), where));
    }
  }
}


static int check_match2 (LexState *ls, int what1, int what2, int who, int where) {  /* 2.0 RC 2 */
  int t1, t2;
  t1 = t2 = 0;
  if (!( (t1 = testnext(ls, what1)) || (t2 = testnext(ls, what2)) )) {
    if (where == ls->linenumber)
      error_expected2(ls, what1, what2);
    else {
      luaX_syntaxerror(ls, luaO_pushfstring(ls->L,
             LUA_QS " or " LUA_QS " expected (to close " LUA_QS " at line %d)",
             luaX_token2str(ls, what1), luaX_token2str(ls, what2),
             luaX_token2str(ls, who), where));
    }
  }
  return (t1) ? 0 : 1;
}


static TString *str_checkname (LexState *ls) {
  TString *ts;
  check(ls, TK_NAME);
  ts = ls->t.seminfo.ts;
  luaX_next(ls);
  return ts;
}


static void init_exp (expdesc *e, expkind k, int i) {
  e->f = e->t = NO_JUMP;
  e->k = k;
  e->u.s.info = i;
}


static void init_number (FuncState *fs, expdesc *e, lua_Number n) {
  e->f = e->t = NO_JUMP;
  e->k = VKNUM;
  e->u.s.info = 0;
  e->u.nval = (lua_Number)n;
  luaK_exp2anyreg(fs, e);
}


static void codestring (LexState *ls, expdesc *e, TString *s) {
  init_exp(e, VK, luaK_stringK(ls->fs, s));
}


static void checkname (LexState *ls, expdesc *e) {
  codestring(ls, e, str_checkname(ls));
}


static int registerlocalvar (LexState *ls, TString *varname) {
  FuncState *fs = ls->fs;
  Proto *f = fs->f;
  int oldsize = f->sizelocvars;
  luaM_growvector(ls->L, f->locvars, fs->nlocvars, f->sizelocvars,
                  LocVar, SHRT_MAX, "too many local variables");
  while (oldsize < f->sizelocvars) {
    f->locvars[oldsize].varname = NULL;
    f->locvars[oldsize].vartype = -1;  /* -1: no optional type specified, 0.20.0 */
    f->locvars[oldsize++].vartypets = NULL;
  }
  f->locvars[fs->nlocvars].varname = varname;
  luaC_objbarrier(ls->L, f, varname);
  if (testnext(ls, TK_OFTYPE)) {  /* new type checking method, 0.20.0 */
    if (ls->t.token >= TK_TBOOLEAN && ls->t.token <= TK_TSET) {
      f->locvars[fs->nlocvars].vartype = ls->t.token-TK_TBOOLEAN+1;
      f->is_typegiven = 1;
      luaX_next(ls);
    }
    else {  /* 0.28.0, user-defined type */
      TString *vartypets = str_checkname(ls);
      f->locvars[fs->nlocvars].vartypets = vartypets;
      luaC_objbarrier(ls->L, f, vartypets);
      f->is_typegiven = 1;
    }
  }
  else {
    f->locvars[fs->nlocvars].vartype = -1;  /* no optional type specified */
    f->locvars[fs->nlocvars].vartypets = NULL;
  }
  return fs->nlocvars++;
}


#define new_localvarliteral(ls,v,n) \
  new_localvar(ls, luaX_newstring(ls, "" v, (sizeof(v)/sizeof(char))-1), n)

static void new_localvar (LexState *ls, TString *name, int n) {
  FuncState *fs = ls->fs;
  luaY_checklimit(fs, fs->nactvar+n+1, LUAI_MAXVARS, "local variables");
  fs->actvar[fs->nactvar+n] = cast(unsigned short, registerlocalvar(ls, name));
}


static void adjustlocalvars (LexState *ls, int nvars) {
  FuncState *fs = ls->fs;
  fs->nactvar = cast_byte(fs->nactvar + nvars);
  for (; nvars; nvars--) {
    getlocvar(fs, fs->nactvar - nvars).startpc = fs->pc;
  }
}


static void removevars (LexState *ls, int tolevel) {
  FuncState *fs = ls->fs;
  while (fs->nactvar > tolevel)
    getlocvar(fs, --fs->nactvar).endpc = fs->pc;
}


static int indexupvalue (FuncState *fs, TString *name, expdesc *v) {
  int i;
  Proto *f = fs->f;
  int oldsize = f->sizeupvalues;
  for (i=0; i<f->nups; i++) {
    if (fs->upvalues[i].k == v->k && fs->upvalues[i].info == v->u.s.info) {
      lua_assert(f->upvalues[i] == name);
      return i;
    }
  }
  /* new one */
  luaY_checklimit(fs, f->nups + 1, LUAI_MAXUPVALUES, "upvalues");
  luaM_growvector(fs->L, f->upvalues, f->nups, f->sizeupvalues,
                  TString *, MAX_INT, "");
  while (oldsize < f->sizeupvalues) f->upvalues[oldsize++] = NULL;
  f->upvalues[f->nups] = name;
  luaC_objbarrier(fs->L, f, name);
  lua_assert(v->k == VLOCAL || v->k == VUPVAL);
  fs->upvalues[f->nups].k = cast_byte(v->k);
  fs->upvalues[f->nups].info = cast_byte(v->u.s.info);
  return f->nups++;
}


static int searchvar (FuncState *fs, TString *n) {
  int i;
  for (i=fs->nactvar-1; i >= 0; i--) {
    if (n == getlocvar(fs, i).varname)
      return i;
  }
  return -1;  /* not found */
}


static void markupval (FuncState *fs, int level) {
  BlockCnt *bl = fs->bl;
  while (bl && bl->nactvar > level) bl = bl->previous;
  if (bl) bl->upval = 1;
}


static int singlevaraux (FuncState *fs, TString *n, expdesc *var, int base) {
  if (fs == NULL) {  /* no more levels? */
    init_exp(var, VGLOBAL, NO_REG);  /* default is global variable */
    return VGLOBAL;
  }
  else {
    int v = searchvar(fs, n);  /* look up at current level */
    if (v >= 0) {
      init_exp(var, VLOCAL, v);
      if (!base)
        markupval(fs, v);  /* local will be used as an upval */
      return VLOCAL;
    }
    else {  /* not found at current level; try upper one */
      if (singlevaraux(fs->prev, n, var, 0) == VGLOBAL)
        return VGLOBAL;
      var->u.s.info = indexupvalue(fs, n, var);  /* else was LOCAL or UPVAL */
      var->k = VUPVAL;  /* upvalue in this level */
      return VUPVAL;
    }
  }
}


static void singlevar (LexState *ls, expdesc *var) {
  TString *varname = str_checkname(ls);
  FuncState *fs = ls->fs;
  if (singlevaraux(fs, varname, var, 1) == VGLOBAL)
    var->u.s.info = luaK_stringK(fs, varname);  /* info points to global name */
}


static TString *singlevarexpr (LexState *ls, expdesc *var) {
  TString *varname = str_checkname(ls);
  FuncState *fs = ls->fs;
  if (singlevaraux(fs, varname, var, 1) == VGLOBAL)
    var->u.s.info = luaK_stringK(fs, varname);  /* info points to global name */
  return varname;
}


static void adjust_assign (LexState *ls, int nvars, int nexps, expdesc *e) {
  FuncState *fs = ls->fs;
  int extra = nvars - nexps;
  if (hasmultret(e->k)) {
    extra++;  /* includes call itself */
    if (extra < 0) extra = 0;
    luaK_setreturns(fs, e, extra);  /* last exp. provides the difference */
    if (extra > 1) luaK_reserveregs(fs, extra-1);
  }
  else {
    if (e->k != VVOID) luaK_exp2nextreg(fs, e);  /* close last expression */
    if (extra > 0) {
      int reg = fs->freereg;
      luaK_reserveregs(fs, extra);
      luaK_nil(fs, reg, extra);
    }
  }
}


static void enterlevel (LexState *ls) {
  if (++ls->L->nCcalls > LUAI_MAXCCALLS)
   luaX_lexerror(ls, "chunk has too many syntax levels", 0);
}


#define leavelevel(ls)   ((ls)->L->nCcalls--)


static void enterblock (FuncState *fs, BlockCnt *bl, lu_byte isbreakable) {
  bl->breaklist = NO_JUMP;
  bl->continuelist = NO_JUMP;
  bl->redolist = NO_JUMP;  /* Agena 2.1 RC 1 */
  bl->redojumpback = NO_JUMP;
  bl->isbreakable = isbreakable;
  bl->nactvar = fs->nactvar;
  bl->upval = 0;
  bl->previous = fs->bl;
  fs->bl = bl;
  lua_assert(fs->freereg == fs->nactvar);
}


static void leaveblock (FuncState *fs) {
  BlockCnt *bl = fs->bl;
  fs->bl = bl->previous;
  removevars(fs->ls, bl->nactvar);
  if (bl->upval)
    luaK_codeABC(fs, OP_CLOSE, bl->nactvar, 0, 0);
  /* a block either controls scope or breaks (never both) */
  //lua_assert(!bl->isbreakable || !bl->upval);
  lua_assert(bl->isbreakable != 1 || !bl->upval);  /* 2.1 RC 2 */
  lua_assert(bl->nactvar == fs->nactvar);
  fs->freereg = fs->nactvar;  /* free registers */
  luaK_patchtohere(fs, bl->breaklist);
}


/* assigns the last iteration value of the numeric loop control variable to a local variable of the same
   name to the surrounding block */
static void leaveblocknumloop (FuncState *fs, TString *varname, expdesc *e) {  /* extended 1.4.0 */
  BlockCnt *bl = fs->bl;
  fs->bl = bl->previous;
  removevars(fs->ls, bl->nactvar);
  if (bl->upval)
    luaK_codeABC(fs, OP_CLOSE, bl->nactvar, 0, 0);
  /* a block either controls scope or breaks (never both) */
  lua_assert(!bl->isbreakable || !bl->upval);
  lua_assert(bl->nactvar == fs->nactvar);
  luaK_patchtohere(fs, bl->breaklist);
  fs->freereg = fs->nactvar+1;  /* free registers, leave last value of control variable on stack */
  /* bind last iteration value to expdesc val, no need to put it into a register for it is already in one */
  if (fs->prev == VVOID) {  /* no more levels ? */
    expdesc val;
    init_exp(&val, VNONRELOC, fs->freereg-1);  /* make it global */
    luaK_setoneret(fs, &val);
    /* store last iteration value to loop control variable */
    luaK_storevar(fs, e, &val);
  } else {  /* make it local */
    new_localvar(fs->ls, varname, 0);
    adjustlocalvars(fs->ls, 1);
    luaK_reserveregs(fs, 1);
  }
}


static void pushclosure (LexState *ls, FuncState *func, expdesc *v) {
  FuncState *fs = ls->fs;
  Proto *f = fs->f;
  int oldsize = f->sizep;
  int i;
  luaM_growvector(ls->L, f->p, fs->np, f->sizep, Proto *,
                  MAXARG_Bx, "constant table overflow");
  while (oldsize < f->sizep) f->p[oldsize++] = NULL;
  f->p[fs->np++] = func->f;
  luaC_objbarrier(ls->L, f, func->f);
  init_exp(v, VRELOCABLE, luaK_codeABx(fs, OP_CLOSURE, 0, fs->np-1));
  for (i=0; i<func->f->nups; i++) {
    OpCode o = (func->upvalues[i].k == VLOCAL) ? OP_MOVE : OP_GETUPVAL;
    luaK_codeABC(fs, o, 0, func->upvalues[i].info, 0);
  }
}


static void open_func (LexState *ls, FuncState *fs) {
  lua_State *L = ls->L;
  Proto *f = luaF_newproto(L);
  fs->f = f;
  fs->prev = ls->fs;  /* linked list of funcstates */
  fs->ls = ls;
  fs->L = L;
  ls->fs = fs;
  fs->pc = 0;
  fs->lasttarget = -1;
  fs->jpc = NO_JUMP;
  fs->freereg = 0;
  fs->nk = 0;
  fs->np = 0;
  fs->nlocvars = 0;
  fs->nactvar = 0;
  fs->bl = NULL;
  f->source = ls->source;
  f->maxstacksize = 2;  /* registers 0/1 are always valid */
  fs->h = luaH_new(L, 0, 0);
  /* anchor table of constants and prototype (to avoid being collected) */
  sethvalue2s(L, L->top, fs->h);
  incr_top(L);
  setptvalue2s(L, L->top, f);
  incr_top(L);
}


static void close_func (LexState *ls) {
  lua_State *L = ls->L;
  FuncState *fs = ls->fs;
  Proto *f = fs->f;
  removevars(ls, 0);
  luaK_ret(fs, 0, 0);  /* final return */
  luaM_reallocvector(L, f->code, f->sizecode, fs->pc, Instruction);
  f->sizecode = fs->pc;
  luaM_reallocvector(L, f->lineinfo, f->sizelineinfo, fs->pc, int);
  f->sizelineinfo = fs->pc;
  luaM_reallocvector(L, f->k, f->sizek, fs->nk, TValue);
  f->sizek = fs->nk;
  luaM_reallocvector(L, f->p, f->sizep, fs->np, Proto *);
  f->sizep = fs->np;
  luaM_reallocvector(L, f->locvars, f->sizelocvars, fs->nlocvars, LocVar);
  f->sizelocvars = fs->nlocvars;
  luaM_reallocvector(L, f->upvalues, f->sizeupvalues, f->nups, TString *);
  f->sizeupvalues = f->nups;
  lua_assert(luaG_checkcode(f));
  lua_assert(fs->bl == NULL);
  ls->fs = fs->prev;
  /* last token read was anchored in defunct function; must reanchor it */
  if (fs) anchor_token(ls);
  L->top -= 2;  /* remove table and prototype from the stack, Lua 5.1.4 patch 11, Agena 1.6.0 */
}


Proto *luaY_parser (lua_State *L, ZIO *z, Mbuffer *buff, const char *name) {
  struct LexState lexstate;
  struct FuncState funcstate;
  lexstate.buff = buff;
  luaX_setinput(L, &lexstate, z, luaS_new(L, name));
  open_func(&lexstate, &funcstate);
  funcstate.f->is_vararg = VARARG_ISVARARG;  /* main func. is always vararg */
  luaX_next(&lexstate);  /* read first token */
  chunk(&lexstate);
  check(&lexstate, TK_EOS);
  close_func(&lexstate);
  lua_assert(funcstate.prev == NULL);
  lua_assert(funcstate.f->nups == 0);
  lua_assert(lexstate.fs == NULL);
  return funcstate.f;
}



/*============================================================*/
/* GRAMMAR RULES */
/*============================================================*/


static void field (LexState *ls, expdesc *v) {
  /* field -> ['.' | '@'] NAME */
  FuncState *fs = ls->fs;
  expdesc key;
  luaK_exp2anyreg(fs, v);
  luaX_next(ls);  /* skip the dot or colon */
  checkname(ls, &key);
  luaK_indexed(fs, v, &key);
}


static int yindex (LexState *ls, expdesc *v, expdesc *key) {  /* 0.28.0, 1.2 */
  /* index -> '[' { expr | expr1 `to' expr2 } { `,' expr | expr1 `to' expr2 ... } ']' */
  FuncState *fs = ls->fs;
  luaX_next(ls);  /* skip the '[' */
  expr(ls, key);
  if (testnext(ls, TK_TO)) {  /* index range ? */
    v->u.s.aux = fs->freereg;
    luaK_exp2nextreg(fs, key);
    expr(ls, key);
    luaK_exp2nextreg(fs, key);
    checknext(ls, ']');
    return 0;
  } else {  /* conventional indices given */
    luaK_exp2val(fs, key);
    while (testnext(ls, ',')) {
      luaK_indexed(fs, v, key);
      luaK_exp2anyreg(fs, v);
      expr(ls, key);
      luaK_exp2val(fs, key);
    }
  }
  checknext(ls, ']');
  return 1;
}


/*
** {======================================================================
** Rules for Constructors
** =======================================================================
*/


struct ConsControl {
  expdesc v;  /* last list item read */
  expdesc *t;  /* table descriptor */
  int nh;  /* total number of `record' elements */
  int na;  /* total number of array elements */
  int tostore;  /* number of array elements pending to be stored */
};


static void item (LexState *ls, struct ConsControl *cc) {
  /* (recfield -> (NAME | STRING | NUMBER | exp1 ~ exp1) | listfield -> exp */
  FuncState *fs = ls->fs;
  int reg = ls->fs->freereg;
  expdesc val, v;
  int rkkey;
  expr(ls, &v);
  if (testnext(ls, TK_SEP)) {  /* recordfield ? */
    luaK_exp2val(ls->fs, &v);
    cc->nh++;
    rkkey = luaK_exp2RK(fs, &v);
    expr(ls, &val);
    luaK_codeABC(fs, OP_SETTABLE, cc->t->u.s.info, rkkey, luaK_exp2RK(fs, &val));
    fs->freereg = reg;  /* free registers */
  } else {  /* listfield */
    cc->v = v;
    luaY_checklimit(ls->fs, cc->na, MAX_INT, "items in a constructor");  /* Lua 5.1.2 patch */
    cc->na++;
    cc->tostore++;
  }
  return;
}


/* handles sequences and sets */
static void listfield (LexState *ls, struct ConsControl *cc) {
  expr(ls, &cc->v);
  luaY_checklimit(ls->fs, cc->na, MAX_INT, "items in a constructor");  /* Lua 5.1.2 patch */
  cc->na++;
  cc->tostore++;
}


static void closelistfield (FuncState *fs, struct ConsControl *cc) {
  if (cc->v.k == VVOID) return;  /* there is no list item */
  luaK_exp2nextreg(fs, &cc->v);
  cc->v.k = VVOID;
  if (cc->tostore == LFIELDS_PER_FLUSH) {
    luaK_setlist(fs, cc->t->u.s.info, cc->na, cc->tostore);  /* flush */
    cc->tostore = 0;  /* no more items pending */
  }
}


static void lastlistfield (FuncState *fs, struct ConsControl *cc) {
  if (cc->tostore == 0) return;
  if (hasmultret(cc->v.k)) {
    luaK_setmultret(fs, &cc->v);
    luaK_setlist(fs, cc->t->u.s.info, cc->na, LUA_MULTRET);
    cc->na--;  /* do not count last expression (unknown number of elements) */
  }
  else {
    if (cc->v.k != VVOID)
      luaK_exp2nextreg(fs, &cc->v);
    luaK_setlist(fs, cc->t->u.s.info, cc->na, cc->tostore);
  }
}


static void constructor (LexState *ls, expdesc *t) {
  /* table constructor -> ?? */
  FuncState *fs = ls->fs;
  int line = ls->linenumber;
  int pc = luaK_codeABC(fs, OP_NEWTABLE, 0, 0, 0);
  struct ConsControl cc;
  cc.na = cc.nh = cc.tostore = 0;
  cc.t = t;
  init_exp(t, VRELOCABLE, pc);
  init_exp(&cc.v, VVOID, 0);  /* no value (yet) */
  luaK_exp2nextreg(fs, t);  /* fix it at stack top (for gc) */
  checknext(ls, '[');
  do {
    lua_assert(cc.v.k == VVOID || cc.tostore > 0);
    if (ls->t.token == ']') break;
    closelistfield(fs, &cc);
    item(ls, &cc);
  } while (testnext(ls, ','));  /* || testnext(ls, ';')); changed, 0.5.2 */
  check_match(ls, ']', '[', line);
  lastlistfield(fs, &cc);
  SETARG_B(fs->f->code[pc], luaO_int2fb(cc.na)); /* set initial array size */
  SETARG_C(fs->f->code[pc], luaO_int2fb(cc.nh));  /* set initial table size */
}


static void setconstructor (LexState *ls, expdesc *t) {  /* 0.10.0 */
  /* set constructor -> ?? */
  FuncState *fs = ls->fs;
  int line = ls->linenumber;
  int pc = luaK_codeABC(fs, OP_FN, 0, 0, OPR_NEWUSET);
  struct ConsControl cc;
  cc.na = cc.nh = cc.tostore = 0;
  cc.t = t;
  init_exp(t, VRELOCABLE, pc);
  init_exp(&cc.v, VVOID, 0);  /* no value (yet) */
  luaK_exp2nextreg(fs, t);  /* fix it at stack top (for gc) */
  checknext(ls, '{');
  do {
    lua_assert(cc.v.k == VVOID || cc.tostore > 0);
    if (ls->t.token == '}') break;
    closelistfield(fs, &cc);
    listfield(ls, &cc);
  } while (testnext(ls, ','));
  check_match(ls, '}', '{', line);
  lastlistfield(fs, &cc);
  SETARG_B(fs->f->code[pc], luaO_int2fb(cc.na));  /* set initial set size */
}


static void seqconstructor (LexState *ls, expdesc *t) {  /* 0.11.0 */
  /* sequence constructor -> ?? */
  FuncState *fs = ls->fs;
  int line = ls->linenumber;
  int pc = luaK_codeABC(fs, OP_FN, 0, 0, OPR_NEWSEQ);
  struct ConsControl cc;
  cc.na = cc.nh = cc.tostore = 0;
  cc.t = t;
  init_exp(t, VRELOCABLE, pc);
  init_exp(&cc.v, VVOID, 0);  /* no value (yet) */
  luaK_exp2nextreg(fs, t);  /* fix it at stack top (for gc) */
  luaX_next(ls);
  checknext(ls, '(');
  do {
    lua_assert(cc.v.k == VVOID || cc.tostore > 0);
    if (ls->t.token == ')') break;
    closelistfield(fs, &cc);
    listfield(ls, &cc);
  } while (testnext(ls, ','));
  check_match(ls, ')', TK_SEQ, line);
  lastlistfield(fs, &cc);
  SETARG_B(fs->f->code[pc], luaO_int2fb(cc.na));  /* set array size */
}


/* }====================================================================== */

static void parlist (LexState *ls) {
  /* parlist -> [ param { `,' param } ] */
  FuncState *fs = ls->fs;
  Proto *f = fs->f;
  int nparams = 0;
  f->is_vararg = 0;
  if (ls->t.token != ')') {  /* is `parlist' not empty? */
    do {
      switch (ls->t.token) {
        case TK_NAME: {  /* param -> NAME */
          new_localvar(ls, str_checkname(ls), nparams++);
          break;
        }
        case TK_DOTS: {  /* param -> `?' */
          luaX_next(ls);
          /* use `varargs' as default name; because of collision between `?' and `varargs',
             varargs functionality has been removed */
          new_localvarliteral(ls, "varargs", nparams++);
          f->is_vararg = VARARG_HASARG | VARARG_NEEDSARG;
          f->is_vararg |= VARARG_ISVARARG;
          break;
        }
        default: luaX_syntaxerror(ls, "<name> or " LUA_QL("?") " expected");
      }
    } while (!f->is_vararg && testnext(ls, ','));
  }
  adjustlocalvars(ls, nparams);
  f->numparams = cast_byte(fs->nactvar - (f->is_vararg & VARARG_HASARG));
  luaK_reserveregs(fs, fs->nactvar);  /* reserve register for parameters */
}


static void parlistfn (LexState *ls, int bracketed) {
  /* parlist -> [ param { `,' param } ] */
  FuncState *fs = ls->fs;
  Proto *f = fs->f;
  int nparams = 0;
  f->is_vararg = 0;
  if ( (bracketed && ls->t.token != ')') || (!(bracketed) && ls->t.token != TK_ARROW) ) {  /* is `parlist' not empty?  Agena 1.2.1 fix */
    do {
      switch (ls->t.token) {
        case TK_NAME: {  /* param -> NAME */
          new_localvar(ls, str_checkname(ls), nparams++);
          break;
        }
        case TK_DOTS: {  /* param -> `?' */
          luaX_next(ls);
          /* use `varargs' as default name; because of collision between `?' and `varargs',
             varargs functionality has been removed */
          new_localvarliteral(ls, "varargs", nparams++);
          f->is_vararg = VARARG_HASARG | VARARG_NEEDSARG;
          f->is_vararg |= VARARG_ISVARARG;
          break;
        }
        default: luaX_syntaxerror(ls, "<name> or " LUA_QL("?") " expected");
      }
    } while (testnext(ls, ','));
  }
  adjustlocalvars(ls, nparams);
  f->numparams = cast_byte(fs->nactvar - (f->is_vararg & VARARG_HASARG));
  luaK_reserveregs(fs, fs->nactvar);  /* reserve register for parameters */
}


static void defbody (LexState *ls, expdesc *e, int line) {
  /* body ->  `<<' parlist `>>' `->' chunk */
  FuncState new_fs;
  int bracketed;
  open_func(ls, &new_fs);
  new_fs.f->linedefined = line;
  bracketed = testnext(ls, '(');
  parlistfn(ls, bracketed);
  if (bracketed)
    checknext(ls, ')');
  checknext(ls, TK_ARROW);
  defchunk(ls);
  new_fs.f->lastlinedefined = ls->linenumber;
  check_match(ls, TK_GT2, TK_LT2, line);
  close_func(ls);
  pushclosure(ls, &new_fs, e);
}


static void returntype (LexState *ls) {
  if (testnext(ls, TK_OFTYPE)) {  /* new return type checking method, 1.3.0 */
    Proto *f = ls->fs->f;
    if (ls->t.token >= TK_TBOOLEAN && ls->t.token <= TK_TSET) {
      f->rettype = ls->t.token-TK_TBOOLEAN+1;
      f->rettypets = NULL;
      luaX_next(ls);
    }
    else {  /* user-defined type */
      TString *returntypets = str_checkname(ls);
      f->rettypets = returntypets;
      luaC_objbarrier(ls->L, f, returntypets);
      f->rettype = -1;
    }
    f->is_rettypegiven = 1;
  }
}


static void body (LexState *ls, expdesc *e, int line) {
  /* body ->  `(' parlist `)' chunk END */
  FuncState new_fs;
  open_func(ls, &new_fs);
  new_fs.f->linedefined = line;
  checknext(ls, '(');
  parlist(ls);
  checknext(ls, ')');
  returntype(ls);
  checknext(ls, TK_IS);  /* Agena 1.5.0 */
  chunk(ls);
  new_fs.f->lastlinedefined = ls->linenumber;
  check_match(ls, TK_END, TK_PROC, line);  /* changed */
  close_func(ls);
  pushclosure(ls, &new_fs, e);
}


static int explist1 (LexState *ls, expdesc *v) {
  /* explist1 -> expr { `,' expr } */
  int n = 1;  /* at least one expression */
  expr(ls, v);
  while (testnext(ls, ',')) {
    luaK_exp2nextreg(ls->fs, v);
    expr(ls, v);
    n++;
  }
  return n;
}


static int funcargexplist (LexState *ls, expdesc *v) {
  /* explist1 -> expr { `,' expr } */
  int n = 1;  /* at least one expression */
  funcargexpr(ls, v);
  while (testnext(ls, ',')) {
    luaK_exp2nextreg(ls->fs, v);
    funcargexpr(ls, v);
    n++;
  }
  return n;
}


static void funcargs (LexState *ls, expdesc *f) {
  FuncState *fs = ls->fs;
  expdesc args;
  int base, nparams;
  int line = ls->linenumber;
  switch (ls->t.token) {
    case '(': {  /* funcargs -> `(' [ explist1 ] `)' */
      if (line != ls->lastline)
        luaX_syntaxerror(ls, "ambiguous syntax (procedure call x new statement)");
      luaX_next(ls);
      if (ls->t.token == ')') { /* arg list is empty? */
        args.k = VVOID;
      }
      else {
        funcargexplist(ls, &args);
        luaK_setmultret(fs, &args);
      }
      check_match(ls, ')', '(', line);
      break;
    }
    case TK_STRING: {  /* funcargs -> STRING */
      codestring(ls, &args, ls->t.seminfo.ts);
      luaX_next(ls);  /* must use `seminfo' before `next' */
      break;
    }
    default: {
      luaX_syntaxerror(ls, "procedure arguments expected");
      return;
    }
  }
  lua_assert(f->k == VNONRELOC);
  base = f->u.s.info;  /* base register for call */
  if (hasmultret(args.k))
    nparams = LUA_MULTRET;  /* open call */
  else {
    if (args.k != VVOID)
      luaK_exp2nextreg(fs, &args);  /* close last argument */
    nparams = fs->freereg - (base+1);
  }
  init_exp(f, VCALL, luaK_codeABC(fs, OP_CALL, base, nparams+1, 2));
  luaK_fixline(fs, line);
  fs->freereg = base+1;  /* call remove function and arguments and leaves
                            (unless changed) one result */
}


static void substringrange (LexState *ls, int base, expdesc *v) {
  FuncState *fs = ls->fs;
  int line = ls->linenumber;
  lua_assert(f->k == VNONRELOC);
  luaK_codeABC(fs, OP_DOTTED, base, v->u.s.info, v->u.s.aux);
  /* reset position to first value in register (the return) */
  luaK_fixline(fs, line);
  v->u.s.info = base;
  fs->freereg = base+1;
}


/* for JOIN, INSTR, SETLOW, SETHIGH, REPLACE operators, 1.1.0, December 09, 2010 */

static void multiargop (LexState *ls, expdesc *e, int line, int tkop, int vmop) {
  int nargs, base;
  FuncState *fs = ls->fs;
  nargs = 0;
  base = fs->freereg;
  init_exp(e, VRELOCABLE, base);
  switch (ls->t.token) {
    case '(': {  /* funcargs -> `(' [ explist1 ] `)' */
      if (line != ls->lastline)
        luaX_syntaxerror(ls, "ambiguous syntax (function call x new statement)");
      luaX_next(ls);
      nargs = explist1(ls, e);
      check_match(ls, ')', tkop, line);
      break;
    }
    default: {
      luaX_syntaxerror(ls, "arguments in parentheses expected");
      return;
    }
  }
  luaK_exp2nextreg(fs, e);  /* close last argument */
  luaK_codeABC(fs, OP_FN, base, nargs, vmop);
  luaK_fixline(fs, line);
  /* reset position to first value in register (the return) */
  e->u.s.info = base;
  fs->freereg -= nargs-1;
}


/* for VALUES operator, supports multiple returns, 1.9.3, February 18, 2013 */
static void multiargopmultrets (LexState *ls, expdesc *e, int line, int tkop, int vmop) {
  int nargs, base;
  FuncState *fs = ls->fs;
  nargs = 0;
  base = fs->freereg;
  init_exp(e, VRELOCABLE, base);
  switch (ls->t.token) {
    case '(': {  /* funcargs -> `(' [ explist1 ] `)' */
      if (line != ls->lastline)
        luaX_syntaxerror(ls, "ambiguous syntax (function call x new statement)");
      luaX_next(ls);
      nargs = explist1(ls, e);
      if (hasmultret(e->k)) {
        luaK_setmultret(fs, e);
        nargs = LUA_MULTRET;
      }
      check_match(ls, ')', tkop, line);
      break;
    }
    default: {
      luaX_syntaxerror(ls, "arguments in parentheses expected");
      return;
    }
  }
  if (e->k != VVOID) luaK_exp2nextreg(fs, e);  /* close last argument */
  luaK_codeABC(fs, OP_FN, base, nargs+1, vmop);
  luaK_fixline(fs, line);
  /* reset position to first value in register (the return) */
  e->u.s.info = base;
  fs->freereg = base + 1;
}


/*
** {======================================================================
** Expression parsing
** =======================================================================
*/

/* `..` conditional field operator, written by Sven Olsen and published in Lua Wiki/Power Patches */

static void conditional_field (LexState *ls, expdesc *v) {
  FuncState *fs = ls->fs;
  luaK_exp2nextreg(fs, v);
  luaK_codeABC(fs, OP_TEST, v->u.s.info, NO_REG, 0);
  {
    int old_free = fs->freereg;
    int vreg = v->u.s.info;
    int j = luaK_codeAsBx(fs, OP_JMP, 0, NO_JUMP);
    expdesc key;
    switch (ls->t.token) {
      case '[': {
        yindex(ls, v, &key);
        luaK_indexed(fs, v, &key);
        luaK_exp2nextreg(fs, v);
        break;
      }
      default: {
        checkname(ls, &key);
        luaK_indexed(fs, v, &key);
      }
    }
    luaK_exp2nextreg(fs, v);
    fs->freereg = old_free;
    /* I think this next check is unnecessary, as any complex key expressions should be courteous
       enough to leave the top of the stack where they found it. */
    if (v->u.s.info != vreg) {
      luaK_codeABC(fs, OP_MOVE, vreg, v->u.s.info, 0);
      v->u.s.info = vreg;
    }
    SETARG_sBx(fs->f->code[j], fs->pc - j - 1);
  }
}


static void prefixexp (LexState *ls, expdesc *v) {
  /* prefixexp -> NAME | '(' expr ')' */
  switch (ls->t.token) {
    case '(': {
      int line = ls->linenumber;
      luaX_next(ls);
      expr(ls, v);
      check_match(ls, ')', '(', line);
      luaK_dischargevars(ls->fs, v);
      return;
    }
    case TK_NAME: {
      singlevar(ls, v);
      return;
    }
    default: {
      luaX_syntaxerror(ls, "unexpected symbol");
      return;
    }
  }
}


static void primaryexp (LexState *ls, expdesc *v) {
  /* primaryexp ->
        prefixexp { `.' NAME | `[' exp `]' | `@' NAME funcargs | funcargs | `$' (...) } */
  FuncState *fs;
  int base;
  fs = ls->fs;
  /* base must be determined at this position so that substring and replace work properly at
     the interactive level with indexed names (why ???) */
  base = fs->freereg;
  prefixexp(ls, v);
  for (;;) {
    switch (ls->t.token) {
      case '.': {  /* field */
        field(ls, v);
        break;
      }
      case '[': {  /* `[' exp1 `]' */
        expdesc key;
        luaK_exp2anyreg(fs, v);
        if (yindex(ls, v, &key))  /* indices given ? 0.28.0, 1.2 */
          luaK_indexed(fs, v, &key);
        else  /* index range given, Agena 1.2 */
          substringrange(ls, base, v);
        break;
      }
      case '(': case TK_STRING: {  /* funcargs, changed 0.5.3 */
        luaK_exp2nextreg(fs, v);
        funcargs(ls, v);
        break;
      }
      case TK_OO: {  /* `@' NAME funcargs */
        expdesc key;
        luaX_next(ls);
        checkname(ls, &key);
        luaK_self(fs, v, &key);
        funcargs(ls, v);
        break;
      }
      case TK_DD: {  /* `..', 2.0.0 */
        luaX_next(ls);
        conditional_field(ls, v);
        break;
      }
      default: return;
    }
  }
}


/* for tablestat parsing: do not parse function arguments, 0.9.0 */
static void primaryexp2 (LexState *ls, expdesc *v) {
  /* primaryexp ->
        prefixexp { `.' NAME | `[' exp `]'} */
  FuncState *fs = ls->fs;
  prefixexp(ls, v);
  for (;;) {
    switch (ls->t.token) {
      case '.': {  /* field */
        field(ls, v);
        break;
      }
      case '[': {  /* `[' exp1 { `,' exp1 ... } `]' */
        expdesc key;
        luaK_exp2anyreg(fs, v);
        yindex(ls, v, &key);  /* 0.28.0 */
        luaK_indexed(fs, v, &key);
        break;
      }
      default: return;
    }
  }
}


static int check_isthen_block (LexState *ls, expdesc *e, int base) {
  /* test_isthen_block -> IS cond THEN block */
  int condexit;
  FuncState *fs = ls->fs;
  condexit = cond(ls);
  checknext(ls, TK_THEN);
  expr(ls, e);
  luaK_isoperation(fs, e, base);
  return condexit;
}


/* for IS operator, 0.8.0, December 20, 2007 */

static void isop (LexState *ls, expdesc *e, int line) {
  int flist, base, escapelist;
  /* cannot use e for preliminary result of comparison, for results get overwritten in procs */
  expdesc v;
  FuncState *fs = ls->fs;
  base = fs->freereg;
  escapelist = NO_JUMP;
  flist = check_isthen_block(ls, &v, base);
  checknext(ls, TK_ELSE);
  luaK_concat(fs, &escapelist, luaK_jump(fs));
  luaK_patchtohere(fs, flist);
  expr(ls, &v);
  luaK_isoperation(fs, &v, base);
  luaK_patchtohere(fs, escapelist);
  check_match(ls, TK_SI, TK_IS, line);
  /* create final result, assign preliminary result in v to e */
  init_exp(e, VRELOCABLE, v.u.s.info);
  /* fix it at top of stack; if not, results get overwritten in procs */
  luaK_exp2nextreg(fs, e);
}


static void simpleexp (LexState *ls, expdesc *v) {
  /* simpleexp -> NUMBER | STRING | null | true | false | fail | ... |
                  constructor | PROC body | primaryexp */
  switch (ls->t.token) {
    case TK_NUMBER: {
      init_exp(v, VKNUM, 0);
      v->u.nval = ls->t.seminfo.r;
      break;
    }
    case TK_STRING: {
      codestring(ls, v, ls->t.seminfo.ts);
      break;
    }
    case TK_NULL: {  /* changed 0.4.0 */
      init_exp(v, VNIL, 0);
      break;
    }
    case TK_TRUE: {
      init_exp(v, VTRUE, 0);
      break;
    }
    case TK_FALSE: {
      init_exp(v, VFALSE, 0);
      break;
    }
    case '[': {  /* table constructor */
      constructor(ls, v);
      return;
    }
    case '{': {  /* set constructor */
      setconstructor(ls, v);
      return;
    }
    case TK_SEQ: {  /* sequence constructor */
      seqconstructor(ls, v);
      return;
    }
    case TK_PROC: {  /* changed 0.4.0 */
      luaX_next(ls);
      body(ls, v, ls->linenumber);
      return;
    }
    case TK_LT2: {  /* added 0.5.2; revised 0.7.1 */
      luaX_next(ls);
      defbody(ls, v, ls->linenumber);
      return;
    }
    /* added 0.7.1, working since 0.8.0 */
    case TK_IS: {
      luaX_next(ls);
      isop(ls, v, ls->linenumber);
      return;
    }
    /* typenames; grep "(GREP_POINT) types;" if you want to add new ones, 0.8.0,
       type designators like `number`, `sequence`, `boolean`, etc. are processed here */
    case TK_TNUMBER: case TK_TSTRING: case TK_TTABLE: case TK_TPROCEDURE:
    case TK_TBOOLEAN: case TK_TTHREAD: case TK_TLIGHTUSERDATA: case TK_TUSERDATA:
    case TK_TSEQUENCE: case TK_TPAIR: case TK_TCOMPLEX: case TK_TSET: {
      codestring(ls, v, luaS_new(ls->L, luaX_tokens[ls->t.token-FIRST_RESERVED]));
      break;
    }
    case TK_NARGS: {
      FuncState *fs = ls->fs;
      init_exp(v, VRELOCABLE, luaK_codeABC(fs, OP_FN, 0, 1, OPR_NARGS));
      luaK_exp2nextreg(fs, v);
      break;
    }
    case TK_FAIL: {
      init_exp(v, VFAIL, 0);
      break;
    }
    case TK_INSTR: {  /* added 0.27.3 */
      luaX_next(ls);
      multiargop(ls, v, ls->linenumber, TK_INSTR, OPR_INSTR);  /* new Agena 1.1.0 */
      return;
    }
    case TK_REPLACE: {  /* added 0.10.0, changed 0.12.2 */
      luaX_next(ls);
      multiargop(ls, v, ls->linenumber, TK_REPLACE, OPR_REPLACE);  /* new Agena 1.1.0 */
      return;
    }
    case TK_JOIN: {  /* added 0.11.0; May 17, 2008 */
      luaX_next(ls);
      multiargop(ls, v, ls->linenumber, TK_JOIN, OPR_JOIN);  /* new Agena 1.1.0 */
      return;
    }
    case TK_VALUES: {  /* added 1.3.0 */
      luaX_next(ls);
      multiargopmultrets(ls, v, ls->linenumber, TK_VALUES, OPR_VALUES);  /* new Agena 1.3.0 */
      return;
    }
    case TK_LOG: {  /* added 1.9.3 */
      luaX_next(ls);
      multiargop(ls, v, ls->linenumber, TK_LOG, OPR_LOG);
      return;
    }
    case TK_SETHIGH: {  /* added 0.27.0 */
      luaX_next(ls);
      multiargop(ls, v, ls->linenumber, TK_SETHIGH, OPR_SETHIGH);  /* new Agena 1.1.0 */
      return;
    }
    case TK_SETLOW: {  /* added 0.27.0 */
      luaX_next(ls);
      multiargop(ls, v, ls->linenumber, TK_SETLOW, OPR_SETLOW);  /* new Agena 1.1.0 */
      return;
    }
    default: {
      primaryexp(ls, v);
      return;
    }
  }
  luaX_next(ls);
}


static UnOpr getunopr (int op) {
  switch (op) {
    case TK_NOT: return OPR_NOT;
    case '-': return OPR_MINUS;
    case TK_SIZE: return OPR_LEN;           /* added 0.6.0 */
    case TK_ABS: return OPR_ABS;            /* added 0.7.1; Nov 18, 2007 */
    case TK_ASSIGNED: return OPR_ASSIGNED;  /* added 0.6.0 */
    case TK_CHAR: return OPR_CHAR;          /* added 0.8.0; Dec 12, 2007 */
    case TK_COPY: return OPR_TCOPY;         /* added 0.9.0, Jan 06, 2008 */
    case TK_COS: return OPR_COS;            /* added 0.7.1; Nov 18, 2007 */
    case TK_COSH: return OPR_COSH;          /* added 0.23.0; Jun 14, 2009 */
    case TK_ENTIER: return OPR_ENTIER;      /* added 0.7.1; Nov 18, 2007 */
    case TK_EVEN: return OPR_EVEN;          /* added 0.7.1; Nov 22, 2007 */
    case TK_EXP: return OPR_EXP;            /* added 0.7.1; Nov 18, 2007 */
    case TK_FILLED: return OPR_TFILLED;     /* added 0.9.1; Jan 11, 2008 */
    case TK_FINITE: return OPR_FINITE;      /* added 0.9.1; Jan 26, 2008 */
    case TK_FLOAT: return OPR_FLOAT;        /* added 0.23.0; Jun 14, 2009 */
    case TK_LNGAMMA: return OPR_LNGAMMA;    /* added 0.9.0; Dec 22, 2007 */
    case TK_IMAG: return OPR_IMAG;          /* added 0.12.0; June 15, 2008 */
    case TK_INT: return OPR_INT;            /* added 0.7.1; Nov 18, 2007 */
    case TK_LEFT: return OPR_LEFT;          /* added 0.11.1; May 22, 2008 */
    case TK_BOTTOM: return OPR_BOTTOM;      /* added 0.29.0; Nov 20, 2009 */
    case TK_LN: return OPR_LN;              /* added 0.7.1; Nov 18, 2007 */
    case TK_LOWER: return OPR_LOWER;        /* added 0.8.0; Dec 07, 2007 */
    case TK_QSADD: return OPR_TQSADD;       /* added 0.9.1, Jan 13, 2008 */
    case TK_REAL: return OPR_REAL;          /* added 0.12.0; June 15, 2008 */
    case TK_RIGHT: return OPR_RIGHT;        /* added 0.11.1; May 22, 2007 */
    case TK_TOP: return OPR_TOP;            /* added 0.29.0; Nov 20, 2009 */
    case TK_SADD: return OPR_TSADD;         /* added 0.9.0, Jan 06, 2008 */
    case TK_SIGN: return OPR_SIGN;          /* added 0.7.1; Nov 18, 2007 */
    case TK_ARCTAN: return OPR_ARCTAN;      /* added 0.7.1; Nov 18, 2007 */
    case TK_ARCSIN: return OPR_ARCSIN;      /* added 0.27.0; Aug 23, 2009 */
    case TK_ARCCOS: return OPR_ARCCOS;      /* added 0.27.0; Aug 23, 2009 */
    case TK_SIN: return OPR_SIN;            /* added 0.7.1; Nov 18, 2007 */
    case TK_SINH: return OPR_SINH;          /* added 0.23.0; Jun 14, 2009 */
    case TK_SQRT: return OPR_SQRT;          /* added 0.7.1; Nov 18, 2007 */
    case TK_TAN: return OPR_TAN;            /* added 0.7.1; Nov 18, 2007 */
    case TK_TANH: return OPR_TANH;          /* added 0.23.0; Jun 14, 2009 */
    case TK_TRIM: return OPR_TRIM;          /* added 0.8.0; Dec 07, 2007 */
    case TK_TYPE: return OPR_TYPE;          /* added 0.6.0 */
    case TK_TYPEOF: return OPR_TYPEOF;      /* added 0.12.0; Jun 24, 2008 */
    case TK_UNASSIGNED: return OPR_UNASSIGNED;  /* added 0.11.2; June 13, 2008 */
    case TK_UNIQUE: return OPR_TUNIQUE;     /* added 0.6.0 */
    case TK_UPPER: return OPR_UPPER;        /* added 0.8.0; Dec 07, 2007 */
    case TK_BNOT: return OPR_BNOT;          /* added 0.27.0 */
    case TK_GETHIGH: return OPR_GETHIGH;    /* added 0.27.0 */
    case TK_GETLOW: return OPR_GETLOW;      /* added 0.27.0 */
    case TK_RECIP: return OPR_RECIP;        /* added 2.1 RC 1 */
    default: return OPR_NOUNOPR;
  }
}


static BinOpr getbinopr (int op) {
  switch (op) {
    case '+': return OPR_ADD;
    case '-': return OPR_SUB;
    case '*': return OPR_MUL;
    case '/': return OPR_DIV;
    case '%': return OPR_MOD;
    case '^': return OPR_POW;
    case '\\': return OPR_INTDIV;              /* added 0.5.4 */
    case TK_CONCAT: return OPR_CONCAT;
    case TK_SPLIT: return OPR_SPLIT;           /* added 0.6.0 */
    case TK_NEQ: return OPR_NE;                /* changed 0.4.0 */
    case TK_EQ: return OPR_EQ;
    case '<': return OPR_LT;
    case TK_LE: return OPR_LE;
    case '>': return OPR_GT;
    case TK_GE: return OPR_GE;
    case TK_AND: return OPR_AND;
    case TK_OR: return OPR_OR;
    case TK_IN: return OPR_IN;                 /* added 0.6.0 */
    case TK_SUBSET: return OPR_TSUBSET;        /* added 0.6.0 */
    case TK_XSUBSET: return OPR_TXSUBSET;      /* added 0.9.1 */
    case TK_UNION: return OPR_TUNION;          /* added 0.6.0 */
    case TK_MINUS: return OPR_TMINUS;          /* added 0.6.0 */
    case TK_INTERSECT: return OPR_TINTERSECT;  /* added 0.6.0 */
    case TK_IPOW: return OPR_IPOW;             /* added 0.9.2 */
    case TK_SHIFT: return OPR_SHIFT;           /* added 0.9.1 */
    case ':': return OPR_PAIR;                 /* added 0.11.1 */
    case '!': return OPR_COMPLEX;              /* added 0.12.0 */
    case TK_OFTYPE: return OPR_TOFTYPE;        /* added 1.3.0, Jan 01, 2011 */
    case TK_NOTOFTYPE: return OPR_TNOTOFTYPE;  /* added 1.3.0, Jan 01, 2011 */
    case TK_EEQ: return OPR_TEEQ;              /* added 0.22.0 */
    case TK_XOR: return OPR_XOR;               /* added 0.27.0 */
    case TK_ATENDOF: return OPR_ATENDOF;       /* added 0.27.0 */
    case TK_BAND: return OPR_BAND;             /* added 0.27.0 */
    case TK_BOR: return OPR_BOR;               /* added 0.27.0 */
    case TK_BXOR: return OPR_BXOR;             /* added 0.27.0 */
    case TK_PERCENT: return OPR_PERCENT;       /* added 1.10.6 */
    case TK_PERCENTRATIO: return OPR_PERCENTRATIO;  /* added 1.11.4 */
    case TK_PERCENTADD: return OPR_PERCENTADD; /* added 1.11.3 */
    case TK_PERCENTSUB: return OPR_PERCENTSUB; /* added 1.11.3 */
    default: return OPR_NOBINOPR;
  }
}


static const struct {
  lu_byte left;  /* left priority for each binary operator */
  lu_byte right; /* right priority */
} priority[] = {  /* ORDER OPR -> lcode.h */
   {6, 6}, {6, 6}, {7, 7}, {7, 7}, {7, 7},  /* `+', `-', `*', `/', `%' */
   {10, 9}, {10, 9},                /* `^`, `**`, both are right-associative, i.e. x^y^z = x^(y^z) */
   {7, 7}, {5, 4},                  /* `\` (added 0.5.4), `&` (concat, right associative) */
   {3, 3}, {3, 3},                  /* inequality  and equality */
   {3, 3}, {3, 3}, {3, 3}, {3, 3},  /* order */
   {2, 2}, {1, 1},                  /* logical (and/or), in this order */
   {4, 4}, {4, 4}, {4, 4},          /* in, subset, xsubset; added 0.7.1 & 0.5.4, 0.9.1 */
   {4, 4}, {4, 4}, {4, 4},          /* union, minus, intersect operators; added 0.6.0 */
   {6, 6}, {7, 7}, {5, 4}, {8, 8},  /* split, shift, pair constr, complex constr ! */
   {3, 3}, {1, 1}, {4, 4}, {7, 7}, {6, 6}, {6, 6},   /* `==` , xor, atendof, band, bor, bxor */
   {3, 3}, {3, 3}, {7, 7}, {7,7}, {7,7}, {7,7}   /* ::, :-, *%, /%, +% , -% */
};

#define UNARY_PRIORITY   8  /* priority for unary operators */

#define UNARYOP_PRIORITY 11  /* priority for new unary operators */

/*
** subexpr -> (simpleexp | unop subexpr) { binop subexpr }
** where `binop' is any binary operator with a priority higher than `limit'
*/
static BinOpr subexpr (LexState *ls, expdesc *v, unsigned int limit) {
  BinOpr op;
  UnOpr uop;
  enterlevel(ls);
  uop = getunopr(ls->t.token);
  if (uop != OPR_NOUNOPR) {  /* unary operator ? */
    luaX_next(ls);
    subexpr(ls, v, (uop < OPR_ABS) ? UNARY_PRIORITY : UNARYOP_PRIORITY);
    luaK_prefix(ls->fs, uop, v);
  }
  else simpleexp(ls, v);
  /* expand while operators have priorities higher than `limit' */
  op = getbinopr(ls->t.token);
  while (op != OPR_NOBINOPR && priority[op].left > limit) {
    expdesc v2;
    BinOpr nextop;
    luaX_next(ls);
    luaK_infix(ls->fs, op, v);
    /* read sub-expression with higher priority */
    nextop = subexpr(ls, &v2, priority[op].right);
    luaK_posfix(ls->fs, op, v, &v2);
    op = nextop;
  }
  leavelevel(ls);
  return op;  /* return first untreated operator */
}


static void expr (LexState *ls, expdesc *v) {
  subexpr(ls, v, 0);
}

/* evaluation of function arguments */

static BinOpr funcargexpr (LexState *ls, expdesc *v) {
  BinOpr op;
  UnOpr uop;
  int ispair;
  unsigned int limit;
  enterlevel(ls);
  uop = getunopr(ls->t.token);
  ispair = 0;
  limit = 0;
  if (uop != OPR_NOUNOPR) {  /* unary op ? */
    luaX_next(ls);
    subexpr(ls, v, (uop < OPR_ABS) ? UNARY_PRIORITY : UNARYOP_PRIORITY);
    luaK_prefix(ls->fs, uop, v);
  }
  else {
    if (ls->t.token == TK_NAME) {
      luaX_lookahead(ls);
      if (ls->lookahead.token == TK_EQ || ls->lookahead.token == TK_SEP) {  /* 0.32.0 */
        ispair = 1;
        checkname(ls, v);
      }
      else
        simpleexp(ls, v);
    }
    else
      simpleexp(ls, v);
  }
  /* expand while operators have priorities higher than `limit' */
  op = (ispair) ? OPR_PAIR : getbinopr(ls->t.token);
  while (op != OPR_NOBINOPR && priority[op].left > limit) {
    expdesc v2;
    BinOpr nextop;
    luaX_next(ls);
    luaK_infix(ls->fs, op, v);
    /* read sub-expression with higher priority */
    nextop = subexpr(ls, &v2, priority[op].right);
    luaK_posfix(ls->fs, op, v, &v2);
    op = nextop;
  }
  leavelevel(ls);
  return op;  /* return first untreated operator */
}


/* }==================================================================== */



/*
** {======================================================================
** Rules for Statements
** =======================================================================
*/


static int block_follow (int token) {
  switch (token) {
    case TK_END: case TK_ELSE: case TK_ELIF: case TK_FI: case TK_OD:
    case TK_OF: case TK_ESAC: case TK_GT2: case TK_AS: case TK_EPOCS:
    case TK_UNTIL:      /* added 2.0.0 RC 2 */
    case TK_ONSUCCESS:  /* added 2.0.0 RC 3 */
    case TK_CATCH:      /* added 2.1 RC 2 */
    case TK_YRT:        /* added 2.1 RC 2 */
    case TK_EOS:        /* changed 0.5.3 */
      return 1;
    default: return 0;
  }
}


static void block (LexState *ls) {
  /* block -> chunk */
  FuncState *fs = ls->fs;
  BlockCnt bl;
  enterblock(fs, &bl, 0);
  chunk(ls);
  lua_assert(bl.breaklist == NO_JUMP);
  leaveblock(fs);
}

/*
** structure to chain all variables in the left-hand side of an
** assignment
*/
struct LHS_assign {
  struct LHS_assign *prev;
  expdesc v;  /* variable (global, local, upvalue, or indexed) */
};


/*
** check whether, in an assignment to a local variable, the local variable
** is needed in a previous assignment  (to a table). If so, save original
** local value in a safe place and use this safe copy in the previous
** assignment.
*/
static void check_conflict (LexState *ls, struct LHS_assign *lh, expdesc *v) {
  FuncState *fs = ls->fs;
  int extra = fs->freereg;  /* eventual position to save local variable */
  int conflict = 0;
  for (; lh; lh = lh->prev) {
    if (lh->v.k == VINDEXED) {
      if (lh->v.u.s.info == v->u.s.info) {  /* conflict? */
        conflict = 1;
        lh->v.u.s.info = extra;  /* previous assignment will use safe copy */
      }
      if (lh->v.u.s.aux == v->u.s.info) {  /* conflict? */
        conflict = 1;
        lh->v.u.s.aux = extra;  /* previous assignment will use safe copy */
      }
    }
  }
  if (conflict) {
    luaK_codeABC(fs, OP_MOVE, fs->freereg, v->u.s.info, 0);  /* make copy */
    luaK_reserveregs(fs, 1);
  }
}


static void assignment (LexState *ls, struct LHS_assign *lh, int nvars) {
  expdesc e;
  check_condition(ls, VLOCAL <= lh->v.k && lh->v.k <= VINDEXED,
                      "syntax error in assignment statement");
  if (testnext(ls, ',')) {  /* assignment -> `,' primaryexp assignment */
    struct LHS_assign nv;
    nv.prev = lh;
    primaryexp(ls, &nv.v);
    if (nv.v.k == VLOCAL)
      check_conflict(ls, lh, &nv.v);
    luaY_checklimit(ls->fs, nvars, LUAI_MAXCCALLS - ls->L->nCcalls,  /* Lua 5.1.3 patch */
                    "variable names");
    assignment(ls, &nv, nvars+1);
  }
  else {  /* assignment -> `:=' explist1 */
    FuncState *fs = ls->fs;
    if (testnext(ls, TK_ASSIGN)) {
      /* default assignment */
      int nexps;
      nexps = explist1(ls, &e);
      if (nexps != nvars) {
        adjust_assign(ls, nvars, nexps, &e);
        if (nexps > nvars)
          ls->fs->freereg -= nexps - nvars;  /* remove extra values */
      }
      else {
        luaK_setoneret(fs, &e);  /* close last expression */
        luaK_storevar(fs, &lh->v, &e);
        return;  /* avoid default */
      }
    }
    /* assignment -> `->' expr;
       0.6.0, assign right-hand value to all left-hand values; modified 0.7.1 */
    else if (testnext(ls, TK_ARROW)) {
      int i;
      expr(ls, &e);  /* get right-hand value and put it in register */
      for (i=1; i < nvars; i++) {  /* copy it nvars - 1 times */
        luaK_exp2nextreg(fs, &e);  /* put e on stack */
        luaK_reserveregs(fs, 1);   /* register it */
      }
      luaK_setoneret(fs, &e);  /* close last expression */
      luaK_storevar(fs, &lh->v, &e);
      return;
    }
    else
      luaX_syntaxerror(ls, "unexpected symbol");
  }
  init_exp(&e, VNONRELOC, ls->fs->freereg-1);  /* default assignment */
  luaK_storevar(ls->fs, &lh->v, &e);
}


static int cond (LexState *ls) {
  /* cond -> exp */
  expdesc v;
  expr(ls, &v);  /* read condition */
  if (v.k == VNIL || v.k == VFAIL) v.k = VFALSE;  /* `falses' are all equal here */
  luaK_goiftrue(ls->fs, &v);
  return v.f;
}


static int condfalse (LexState *ls) {  /* added 0.5.3 */
  /* cond -> exp */
  expdesc v;
  expr(ls, &v);  /* read condition */
  if (v.k == VNIL || v.k == VFAIL) v.k = VFALSE;  /* `falses' are all equal here */
  luaK_goiffalse(ls->fs, &v);
  return v.t;
}


/* static void breakstat (LexState *ls) { changed */
static void break_or_continue (LexState *ls, const char *msg, size_t offset) {
  FuncState *fs = ls->fs;
  BlockCnt *bl = fs->bl;
  int upval = 0;
  while (bl && bl->isbreakable != 1) {  /* 2.1 RC 2 */
    if (bl->isbreakable == 2)
      luaK_codeABC(fs, OP_ENDTRY, 0, 0, 0);
    upval |= bl->upval;
    bl = bl->previous;
  }
  if (!bl)
    luaX_syntaxerror(ls, msg);
  if (upval)
    luaK_codeABC(fs, OP_CLOSE, bl->nactvar, 0, 0);
  luaK_concat(fs, (int*) (((char*) bl) + offset), luaK_jump(fs));
}

static void breakstat (LexState *ls) {   /* changed */
  break_or_continue(ls, "no loop to break", offsetof(BlockCnt, breaklist));
}

/* skip statement written by Wolfgang Oertl and posted to the Lua Mailing List on 2005-09-12 */
static void skipstat (LexState *ls) {  /* added */
  break_or_continue(ls, "no loop to skip", offsetof(BlockCnt, continuelist));
}


static void redostat (LexState *ls) {  /* Agena 2.1 RC 1 */
  FuncState *fs = ls->fs;
  BlockCnt *bl = fs->bl;
  int upval = 0;
  //while (bl && !bl->isbreakable) {
  while (bl && bl->isbreakable != 1) {  /* 2.1 RC 2 */
    if (bl->isbreakable == 2)
      luaK_codeABC(fs, OP_ENDTRY, 0, 0, 0);
    upval |= bl->upval;
    bl = bl->previous;
  }
  if (!bl || bl->redojumpback == NO_JUMP)
    luaX_syntaxerror(ls, "no `for` loop to redo");
  if (upval)
    luaK_codeABC(fs, OP_CLOSE, bl->nactvar, 0, 0);
  luaK_patchlist(ls->fs, luaK_jump(fs), bl->redojumpback);  /* and repeat iteration */
}


static void whilestat (LexState *ls, int line) {
  /* whilestat -> WHILE cond DO block OD */
  FuncState *fs = ls->fs;
  int whileinit;
  int condexit;
  BlockCnt bl;
  luaX_next(ls);  /* skip WHILE */
  whileinit = luaK_getlabel(fs);
  condexit = cond(ls);
  enterblock(fs, &bl, 1);
  checknext(ls, TK_DO);
  block(ls);
  luaK_patchlist(fs, luaK_jump(fs), whileinit);
  luaK_patchlist(fs, bl.continuelist, whileinit);  /* continue goes to start, too; changed */
  check_match(ls, TK_OD, TK_WHILE, line);  /* changed 0.4.0 */
  leaveblock(fs);
  luaK_patchtohere(fs, condexit);  /* false conditions finish the loop */
}


static void dostat (LexState *ls, int line) {
  /* repeatstat -> DO block { AS cond | UNTIL cond | OD } */
  int condexit, repeat_init;
  BlockCnt bl1, bl2;
  FuncState *fs = ls->fs;
  repeat_init = luaK_getlabel(fs);
  enterblock(fs, &bl1, 1);  /* loop block */
  enterblock(fs, &bl2, 0);  /* scope block */
  luaX_next(ls);  /* skip DO */
  chunk(ls);
  luaK_patchtohere(fs, bl1.continuelist);
  if (testnext(ls, TK_OD)) {
    expdesc v;
    init_exp(&v, VTRUE, 0);
    luaK_goiffalse(ls->fs, &v);
    condexit = v.t;
  } else {
    int r;
    r = check_match2(ls, TK_AS, TK_UNTIL, TK_DO, line);  /* 0.6.0 */
    condexit = (r) ? cond(ls) : condfalse(ls);  /* read condition (inside scope block) */
  }
  if (!bl2.upval) {  /* no upvalues? */
    leaveblock(fs);  /* finish scope */
    luaK_patchlist(ls->fs, condexit, repeat_init);  /* close the loop */
  }
  else {  /* complete semantics when there are upvalues */
    breakstat(ls);  /* if condition is false then break */
    luaK_patchtohere(ls->fs, condexit);  /* else... */
    leaveblock(fs);  /* finish scope... */
    luaK_patchlist(ls->fs, luaK_jump(fs), repeat_init);  /* and repeat */
  }
  leaveblock(fs);  /* finish loop */
}


static int exp1 (LexState *ls) {
  expdesc e;
  int k;
  expr(ls, &e);
  k = e.k;
  luaK_exp2nextreg(ls->fs, &e);
  return k;
}


static int fornumbody (LexState *ls, int base, int line) {
  /* forbody [WHILE cond] -> DO block
     WHILE part added 0.12.0, June 14, 2008, tweaked 1.4.0, 21.02.2011
     AS part added 2.0.0 RC 2, 14.11.2013 */
  BlockCnt bl;
  int prep, endfor, iswhile, isas, condexit;
  FuncState *fs = ls->fs;
  isas = 0;
  adjustlocalvars(ls, 4);  /* control variables */
  fs->bl->redojumpback = prep = luaK_codeAsBx(fs, OP_FORPREP, base, NO_JUMP);  /* 2.1 RC 1 */
  enterblock(fs, &bl, 0);  /* scope for declared variables */
  adjustlocalvars(ls, 1);
  luaK_reserveregs(fs, 1);
  if ((iswhile = testnext(ls, TK_WHILE)))  /* 0.12.0 */
    condexit = cond(ls);
  else
    condexit = 0;
  checknext(ls, TK_DO);
  block(ls);
  if (!iswhile) {
    if ((isas = testnext(ls, TK_AS)))  /* 2.0 RC 2 */
      condexit = cond(ls);
    else if ((isas = testnext(ls, TK_UNTIL)))
      condexit = condfalse(ls);
  }
  leaveblock(fs);  /* end of scope for declared variables */
  luaK_patchtohere(fs, prep);
  luaK_patchtohere(fs, bl.previous->continuelist);  /* skip, if any, jumps to here, changed */
  luaK_patchtohere(fs, bl.previous->redolist);
  endfor = luaK_codeAsBx(fs, OP_FORLOOP, base, NO_JUMP);
  if (iswhile || isas)
    luaK_patchtohere(fs, condexit);  /* false conditions finish the loop */
  luaK_fixline(fs, line);  /* pretend that `OP_FOR' starts the loop */
  luaK_patchlist(fs, endfor, prep + 1);
  return isas;  /* 1 if `as` token has been given to prevent the parser from looking for a closing `od` */
}


static int forinlistbody (LexState *ls, int base, int line, int nvars) {
  /* taken from Lua 5.0.3 to get rid of `for index in PAIRS(tablename) do`; NOT slower than Lua 5.1 !
     WHILE part added 0.12.0, June 14, 2008 */
  BlockCnt bl;
  FuncState *fs = ls->fs;
  int prep, iswhile, isas, whileinit, condexit;
  isas = 0;
  adjustlocalvars(ls, nvars);  /* scope for all variables including external list member */
  luaK_codeAsBx(fs, OP_TFORPREP, base, NO_JUMP);
  if ((iswhile = testnext(ls, TK_WHILE))) {  /* 0.12.0 */
    whileinit = luaK_getlabel(fs);
    condexit = cond(ls);
  } else
    whileinit = condexit = 0;
  checknext(ls, TK_DO);
  fs->bl->redojumpback = prep = (iswhile) ? whileinit : luaK_getlabel(fs);  /* extended 2.1 RC 1, get proper jump position */
  enterblock(fs, &bl, 0);  /* changed, loop block */
  block(ls);
  if (!iswhile) {
    if ((isas = testnext(ls, TK_AS)))  /* 2.0 RC 2 */
      condexit = cond(ls);
    else if ((isas = testnext(ls, TK_UNTIL)))
      condexit = condfalse(ls);
  }
  leaveblock(fs);
  luaK_patchtohere(fs, prep - 1);
  luaK_patchtohere(fs, bl.previous->continuelist);   /* skip, if any, jumps to here, changed */
  luaK_codeABC(fs, OP_TFORLOOP, base, 0, nvars - 3);  /* changed 0.5.0 */
  luaK_fixline(fs, line);  /* pretend that `OP_FOR' starts the loop */
  luaK_patchlist(fs, luaK_jump(fs), prep);  /* changed 0.5.0 */
  if (iswhile || isas)  /* 0.12.0 */
    luaK_patchtohere(fs, condexit);  /* false conditions finish the loop */
  return isas;
}


static int fornum (LexState *ls, TString *varname, int line) {  /* optimised 1.4.0, extended 2.0.0 RC 2, 14.11.2013 */
  /* fornum -> NAME [FROM exp1] [TO exp1] [BY exp1] forbody [WHILE cond] */
  FuncState *fs = ls->fs;
  int base = fs->freereg;
  new_localvarliteral(ls, "(for index)", 0);
  new_localvarliteral(ls, "(for limit)", 1);
  new_localvarliteral(ls, "(for step)", 2);
  /* 0.30.0, prevent roundoff errors using the Kahan algorithm */
  new_localvarliteral(ls, "(for roff)", 3);  /* correction variable */
  new_localvar(ls, varname, 4);
  if (testnext(ls, TK_FROM))  /* changed */
    exp1(ls);  /* initial value */
  else {  /* if `from' keyword is missing, assign loop start value to 1 */
    luaK_codeABx(fs, OP_LOADK, fs->freereg, luaK_numberK(fs, 1));
    luaK_reserveregs(fs, 1);  /* increment fs->freereg by one */
  }
  if (testnext(ls, TK_TO)) /* changed 0.4.0, 0.12.3, 0.31.0, 2.0.0 RC 2 */
    exp1(ls);  /* limit */
  else {  /* if `from' keyword is missing, assign loop start value to 1 */
    luaK_codeABx(fs, OP_LOADK, fs->freereg, luaK_numberK(fs, HUGE_VAL));
    luaK_reserveregs(fs, 1);  /* increment fs->freereg by one */
  }
  if (testnext(ls, TK_BY)) /* changed 0.4.0 */
    exp1(ls);  /* optional step */
  else {  /* default step = 1 */
    luaK_codeABx(fs, OP_LOADK, fs->freereg, luaK_numberK(fs, 1));
    luaK_reserveregs(fs, 1); /* increment fs->freereg by one */
  }
  luaK_codeABx(fs, OP_LOADK, fs->freereg, luaK_numberK(fs, HUGE_VAL));  /* 1.4.0 */
  luaK_reserveregs(fs, 1);  /* reserve register for flag */
  /* 0.30.0, set lvm variable c to prevent roundoff errors */
  return fornumbody(ls, base, line);  /* 1.4.0 */
}


static int forlist (LexState *ls, TString *indexname, int keysflag) {
  /* forlist -> NAME {,NAME} IN explist1 forbody */
  FuncState *fs = ls->fs;
  expdesc e;
  int nvars = 0;
  int line;
  int base = fs->freereg;
  /* create control variables */
  new_localvarliteral(ls, "(for generator)", nvars++);
  new_localvarliteral(ls, "(for state)", nvars++);
  new_localvarliteral(ls, "(for control)", nvars++);
  if (ls->t.token != ',' && !keysflag) {  /* iterate values only; changed 0.5.2 */
    new_localvarliteral(ls, "(_$_)", nvars++);  /* (_$_) holds the key if only the value variable has been passed */
  } else if (ls->t.token == ',' && keysflag) {
    luaX_syntaxerror(ls, "only one loop variable allowed for keys,");
  }
  /* create declared variables */
  new_localvar(ls, indexname, nvars++);
  while (testnext(ls, ','))
    new_localvar(ls, str_checkname(ls), nvars++);
  checknext(ls, TK_IN);
  line = ls->linenumber;
  adjust_assign(ls, nvars, explist1(ls, &e), &e);
  luaK_checkstack(fs, 3);  /* extra space to call generator */
  return forinlistbody(ls, base, line, nvars);
}


static void forstat (LexState *ls, int line) {
  /* forstat -> FOR (fornum | forlist) OD */
  FuncState *fs = ls->fs;
  TString *varname; //, *varname2;
  expdesc e;
  BlockCnt bl;
  int keysflag, numloop, isas;
  //varname2 = NULL;
  enterblock(fs, &bl, 1);  /* scope for loop and control variables */
  luaX_next(ls);  /* skip `for' */
  keysflag = (testnext(ls, TK_KEYS)) ? 1 : 0;  /* KEYS token ?, added 0.5.2 */
  /* first variable name; assigns name of loop control variable to expdesc e, as well */
  varname = singlevarexpr(ls, &e);
  numloop = 0;  /* 1 if for/to loop, 0 if not */
  isas = 0;
  switch (ls->t.token) {
    case TK_FROM: case TK_TO: case TK_DO: case TK_WHILE:
      numloop = 1;  /* 1 if for/to loop, 0 if not */
      isas = fornum(ls, varname, line);
      break;  /* changed 0.4.0, 0.12.3 */
    case ',': case TK_IN:
      isas = forlist(ls, varname, keysflag); //, &varname2);
      break;  /* changed 0.5.2 */
    default:
      luaX_syntaxerror(ls, LUA_QL("from") ", " LUA_QL("to") ", or " LUA_QL("in") " expected");
  }
  if (!isas)  /* 2.0 RC 2, 14.11.2013 */
    check_match(ls, TK_OD, TK_FOR, line); /* changed 0.4.0 */
  if (numloop)  /* loop scope (`break' jumps to this point) */
    /* for/to loops: export last value of iteration variable to surrounding block */
    leaveblocknumloop(fs, varname, &e);
  else
    leaveblock(fs);
}


/* `anonymous` loop, 0.6.0; October 19, 2007 */

static void tostat (LexState *ls, int line) {
  /* tostat -> TO (toexpr) DO ... OD */
  int base;
  FuncState *fs = ls->fs;
  BlockCnt bl;
  luaX_next(ls);  /* skip `to' */
  enterblock(fs, &bl, 1);  /* scope for loop and control variables */
  base = fs->freereg;
  new_localvarliteral(ls, "(for index)", 0);
  new_localvarliteral(ls, "(for limit)", 1);
  new_localvarliteral(ls, "(for step)", 2);
  new_localvarliteral(ls, "(for roff)", 3);
  new_localvarliteral(ls, "_$_", 4);
  luaK_codeABx(fs, OP_LOADK, fs->freereg, luaK_numberK(fs, 1));  /* from 1 */
  luaK_reserveregs(fs, 1);
  exp1(ls);  /* to limit */
  luaK_codeABx(fs, OP_LOADK, fs->freereg, luaK_numberK(fs, 1));  /* by 1 */
  luaK_reserveregs(fs, 1);  /* reserve register for step */
  /* do not use Kahan precision as step is assumed to be an integer, to be re-evaluated in lvm.c/OP_FORPREP */
  luaK_codeABx(fs, OP_LOADK, fs->freereg, luaK_numberK(fs, HUGE_VAL));
  luaK_reserveregs(fs, 1);
  fornumbody(ls, base, line);  /* 1.4.0 */
  check_match(ls, TK_OD, TK_TO, line); /* changed 0.4.0 */
  leaveblock(fs);  /* loop scope (`break' jumps to this point) */
}


static int test_then_block (LexState *ls) {
  /* test_then_block -> [IF | ELIF] cond THEN block */
  int condexit;
  luaX_next(ls);  /* skip IF or ELIF */
  condexit = cond(ls);
  checknext(ls, TK_THEN);
  block(ls);  /* `then' part */
  return condexit;
}


static void ifstat (LexState *ls, int line) {
  /* ifstat -> IF cond THEN block {ELIF cond THEN block} {ONSUCCESS block} [ELSE block] FI */
  FuncState *fs = ls->fs;
  int flist;
  int escapelist = NO_JUMP;
  flist = test_then_block(ls);  /* IF cond THEN block */
  while (ls->t.token == TK_ELIF) {  /* changed Oct. 12, 2006, changed 0.4.0 */
    luaK_concat(fs, &escapelist, luaK_jump(fs));  /* this seems to be a jump if all previous conditions failed */
    luaK_patchtohere(fs, flist);  /* dito */
    flist = test_then_block(ls);  /* ELIF cond THEN block */
  }
  if (ls->t.token == TK_ONSUCCESS) {  /* added 2.0.0 RC 3 */
    luaK_patchtohere(fs, escapelist);  /* dito */
    luaX_next(ls);  /* skip ONSUCCESS (after patch, for correct line info) */
    block(ls);  /* `onsuccess' part */
    escapelist = NO_JUMP;
  }
  if (ls->t.token == TK_ELSE) {
    luaK_concat(fs, &escapelist, luaK_jump(fs));  /* this seems to be a jump if all previous conditions failed */
    luaK_patchtohere(fs, flist);  /* dito */
    luaX_next(ls);  /* skip ELSE (after patch, for correct line info) */
    block(ls);  /* `else' part */
  }
  else {
    luaK_concat(fs, &escapelist, flist);
  }
  luaK_patchtohere(fs, escapelist);
  check_match(ls, TK_FI, TK_IF, line);  /* changed 0.4.0 */
}


/* Author of this Lua extension is Hu Qiwei. 

   Taken from the Lua User Group.
   
   Subject: Re: try..catch in Lua
   From: "Hu Qiwei" <huqiwei>
   Date: Tue, 29 Jan 2008 13:34:14 +0800 */

static void trystat (LexState *ls, int line) {  /* 2.1 RC 2 */
  /* trystat -> TRY block [CATCH err THEN block] END */
  FuncState *fs = ls->fs;
  BlockCnt bl;
  int escapelist = NO_JUMP;
  int jpc = fs->jpc;  /* save list of jumps to here */
  int pc;

  fs->jpc = NO_JUMP;
  luaX_next(ls);
  pc = luaK_codeAsBx(fs, OP_TRY, 0, NO_JUMP);
  luaK_concat(fs, &pc, jpc);  /* keep them on hold */

  enterblock(fs, &bl, 2);   /* try-catch block */
  block(ls);
  leaveblock(fs);

  if (ls->t.token == TK_CATCH) {
    TString *varname;
    int base;

    luaK_codeABC(fs, OP_ENDTRY, 0, 0, 0);
    luaK_concat(fs, &escapelist, luaK_jump(fs));
    luaK_patchtohere(fs, pc);

    /* local err */
    enterblock(fs, &bl, 0);
    luaX_next(ls);  /* skip `catch' */
    varname = str_checkname(ls);  /* error variable name */

    /* `then' */
    checknext(ls, TK_THEN);
    base = fs->freereg;
    new_localvar(ls, varname, 0);
    adjustlocalvars(ls, 1);  /* control variables */
    luaK_reserveregs(fs, 1);

    luaK_codeABC(fs, OP_CATCH, base, 0, 0);  /* OP_CATCH sets error object to local 'varname' */

    block(ls);
    leaveblock(fs);  /* catch scope (`break' jumps to this point) */
  }
  else {
    luaK_codeABC(fs, OP_ENDTRY, 0, 0, 0);
    luaK_concat(fs, &escapelist, pc);
  }

  luaK_patchtohere(fs, escapelist);

  check_match(ls, TK_YRT, TK_TRY, line);
}

/* written by Andreas Falkenhahn and posted to lua.bazar2.conectiva.com.br on 01 Sep 2004
   changed to process multiple values in 0.6.0, November 10, 2007; patched 0.29.1, December 13, 2009;
   patched 1.9.5, February 25, 2013 */

static void casestat (LexState *ls, expdesc *v) {
  FuncState *fs = ls->fs;
  expdesc v2;
  int nargs, base;
  /* do not set base to 0 for if case is used in (local) loops, it does not work properly */
  base = fs->freereg;
  luaK_exp2nextreg(fs, v);
  nargs = explist1(ls, &v2); /* put every value into a register */
  luaK_exp2nextreg(fs, &v2);
  codecompcase(fs, base, nargs+1, v);
  fs->freereg -= nargs+1;  /* free all registers */
  if (v->k == VNIL || v->k == VFAIL) v->k = VFALSE;
  luaK_goiftrue(fs, v);
  luaK_patchtohere(fs, v->t);
  checknext(ls, TK_THEN);
  block(ls);
}


/* written by Andreas Falkenhahn and posted to lua.bazar2.conectiva.com.br on 01 Sep 2004 */

static void selectstat (LexState *ls, int line) {
  /* selectstat -> CASE val1 OF val2 THEN block [OF valx THEN block [ELSE THEN block]] ESAC */
  FuncState *fs = ls->fs;
  BlockCnt bl;
  expdesc v;
  int escapelist = NO_JUMP;
  luaX_next(ls);  /* skip `case' */
  enterblock(fs, &bl, 0);  /* block to control variable scope */
  expr(ls, &v);
  luaK_dischargevars(fs, &v);  /* 1.9.5 fix to properly treat indexed values */
  checknext(ls, TK_OF);
  casestat(ls, &v);
  while (ls->t.token == TK_OF) {
    luaK_concat(fs, &escapelist, luaK_jump(fs));
    luaK_patchtohere(fs, v.f);
    luaX_next(ls);  /* skip OF */
    casestat(ls, &v);  /* OF val THEN block */
  }
  if (ls->t.token == TK_ONSUCCESS) {  /* added 2.0.0 RC 3 */
    luaK_patchtohere(fs, escapelist);  /* dito */
    luaX_next(ls);  /* skip ONSUCCESS (after patch, for correct line info) */
    block(ls);  /* `onsuccess' part */
    escapelist = NO_JUMP;
  }
  if (ls->t.token == TK_ELSE) {
    luaK_concat(fs, &escapelist, luaK_jump(fs));
    luaK_patchtohere(fs, v.f);
    luaX_next(ls);  /* skip ELSE */
    block(ls);
  } else
    luaK_concat(fs, &escapelist, v.f);
  luaK_patchtohere(fs, escapelist);
  check_match(ls, TK_ESAC, TK_CASE, line);
  leaveblock(fs);
}


static void localtablestat (LexState *ls, int nvars) {  /* added 0.8.0, December 20, 2007; extended Jan 01, 08 */
  /* CREATE LOCAL TABLE <var1[`(`expr`)`]> */
  FuncState *fs = ls->fs;
  expdesc t, e;
  struct LHS_assign v;
  int pc;
  new_localvar(ls, str_checkname(ls), nvars);
  init_exp(&v.v, VLOCAL, fs->freereg);
  luaK_reserveregs(fs, 1);
  adjustlocalvars(ls, 1);
  if (testnext(ls, '(')) {  /* set initial size for simple table (numeric keys) */
    expr(ls, &e);
    luaK_exp2anyreg(fs, &e);
    pc = codearithbypass(fs, 0, &e, OPR_DIMTAB);
    checknext(ls, ')');
  } else
    pc = luaK_codeABC(fs, OP_NEWTABLE, 0, 0, 0);
  init_exp(&t, VRELOCABLE, pc);
  luaK_exp2nextreg(fs, &t);  /* fix it at stack top (for gc) */
  luaK_setoneret(fs, &t);  /* close last expression */
  luaK_storevar(fs, &v.v, &t);
}


static void localdictstat (LexState *ls, int nvars) {  /* added 0.12.2 */
  /* CREATE LOCAL DICT <var1[`(`expr`)`]> */
  FuncState *fs = ls->fs;
  expdesc t, e;
  struct LHS_assign v;
  int pc;
  new_localvar(ls, str_checkname(ls), nvars);
  init_exp(&v.v, VLOCAL, fs->freereg);
  luaK_reserveregs(fs, 1);
  adjustlocalvars(ls, 1);
  if (testnext(ls, '(')) {  /* set initial size for simple table (numeric keys) */
    expr(ls, &e);
    luaK_exp2anyreg(fs, &e);
    pc = codearithbypass(fs, 0, &e, OPR_DIMDICT);
    checknext(ls, ')');
  } else
    pc = luaK_codeABC(fs, OP_NEWTABLE, 0, 0, 0);
  init_exp(&t, VRELOCABLE, pc);
  luaK_exp2nextreg(fs, &t);  /* fix it at stack top (for gc) */
  luaK_setoneret(fs, &t);  /* close last expression */
  luaK_storevar(fs, &v.v, &t);
}


static void localsetstat (LexState *ls, int nvars) {  /* added 0.10.0, April 17, 2008 */
  /* CREATE LOCAL SET <var1[`(`expr`)`]> */
  FuncState *fs = ls->fs;
  expdesc t, e;
  struct LHS_assign v;
  int pc;
  new_localvar(ls, str_checkname(ls), nvars);
  init_exp(&v.v, VLOCAL, fs->freereg);
  luaK_reserveregs(fs, 1);
  adjustlocalvars(ls, 1);
  if (testnext(ls, '(')) {  /* set initial size for set */
    expr(ls, &e);
    luaK_exp2anyreg(fs, &e);
    pc = codearithbypass(fs, 0, &e, OPR_DIMSET);
    checknext(ls, ')');
  } else {
    init_number(fs, &e, 0);
    pc = codearithbypass(fs, 0, &e, OPR_DIMSET);
  }
  init_exp(&t, VRELOCABLE, pc);
  luaK_exp2nextreg(fs, &t);  /* fix it at stack top (for gc) */
  luaK_setoneret(fs, &t);  /* close last expression */
  luaK_storevar(fs, &v.v, &t);
}


static void localseqstat (LexState *ls, int nvars) {  /* added 0.11.0, May 18, 2008; extended 0.12.0, June 30, 2008 */
  /* CREATE LOCAL SEQ <var1`(`expr`)`> */
  FuncState *fs = ls->fs;
  expdesc t, e;
  struct LHS_assign v;
  int pc;
  new_localvar(ls, str_checkname(ls), nvars);
  init_exp(&v.v, VLOCAL, fs->freereg);
  luaK_reserveregs(fs, 1);
  adjustlocalvars(ls, 1);
  if (testnext(ls, '(')) {
    expr(ls, &e);
    luaK_exp2anyreg(fs, &e);
    pc = codearithbypass(fs, 0, &e, OPR_DIMSEQ);
    checknext(ls, ')');
  } else {
    init_number(fs, &e, 0);
    pc = codearithbypass(fs, 0, &e, OPR_DIMSEQ);
  }
  init_exp(&t, VRELOCABLE, pc);
  luaK_exp2nextreg(fs, &t);  /* fix it at stack top (for gc) */
  luaK_setoneret(fs, &t);  /* close last expression */
  luaK_storevar(fs, &v.v, &t);
}


static void localenumstat (LexState *ls) {  /* added 0.9.0, December 22, 2007 */
  /* LOCAL ENUM <var1> [, var2, ...] [FROM expr] */
  expdesc start, step;
  int i;
  int nvars = 0;
  FuncState *fs = ls->fs;
  do {
    new_localvar(ls, str_checkname(ls), nvars++);
  } while (testnext(ls, ','));
  if (testnext(ls, TK_FROM)) {
    expr(ls, &start);
  }
  else {
    init_exp(&start, VKNUM, 0);
    start.u.nval = (lua_Number)1;
  }
  /* step size (forward counting) */
  init_exp(&step, VKNUM, 0);
  step.u.nval = (lua_Number)1;
  codearith(fs, OP_SUB, &start, &step);
  /* assign vars */
  for (i=0; i < nvars; i++) {
    codearith(fs, OP_ADD, &start, &step);
    luaK_exp2nextreg(fs, &start);  /* put start on stack */
    luaK_reserveregs(fs, 1);
  }
  adjust_assign(ls, nvars, nvars, &start);
  adjustlocalvars(ls, nvars);
}


#define nomix   luaX_syntaxerror(ls, "enums or structure declarations cannot be mixed with assignments,");

static void initlocals (LexState *ls, int nvars) {
  FuncState *fs = ls->fs;
  int reg = fs->freereg;
  luaK_reserveregs(fs, nvars);
  luaK_nil(fs, reg, nvars);
  adjustlocalvars(ls, nvars);
}

/* initiate non-structure vars that have already been passed before setting empty structure */
#define PrepareStructAssign(ls,nvars,isatom) \
      if (nvars != 0) {  \
        initlocals(ls, nvars);\
        nvars = 0;\
      }\
      isatom = 0;

/* extended to process mixed declarations, August 02, 2008/October 29, 2008 */
static void localstat (LexState *ls, int mode) {
  /* stat -> LOCAL NAME {`,' NAME} [{`:='|`->`} explist1] */
  int nvars, nexps, isatom, i;
  expdesc e;
  nvars = 0;
  isatom = 1;  /* flag to detect whether simple vars and structures have been mixed */
  nexps = 0;
  do {
    if (mode) {
      if (testnext(ls, TK_TTABLE)) {
        PrepareStructAssign(ls, nvars, isatom);
        localtablestat(ls, nvars);
      }
      else if (testnext(ls, TK_TSET)) {
        PrepareStructAssign(ls, nvars, isatom);
        localsetstat(ls, nvars);
      }
      else if (testnext(ls, TK_SEQ) || testnext(ls, TK_TSEQUENCE)) {  /* 1.11.8 */
        PrepareStructAssign(ls, nvars, isatom);
        localseqstat(ls, nvars);
      }
      else if (testnext(ls, TK_DICT)) {
        PrepareStructAssign(ls, nvars, isatom);
        localdictstat(ls, nvars);
      }
    } else {
      if (testnext(ls, TK_ENUM)) {  /* local enum vars */
        if (nvars != 0)
          luaX_syntaxerror(ls, "enums cannot be mixed with other declarations,");
        localenumstat(ls);
        isatom = 0;
        return;  /* no further declarations allowed in same statement */
      }
      else
        new_localvar(ls, str_checkname(ls), nvars++);
    }
  } while (testnext(ls, ','));
  if (testnext(ls, TK_ASSIGN)) {
    if (isatom)
      nexps = explist1(ls, &e);
    else
      nomix;
  }
  else if (testnext(ls, TK_ARROW)) {  /* multiple assignment statement; added 0.6.0 */
    if (isatom) {
      expr(ls, &e);
      /* i=1 instead of i=0: e was already put once into a register with the preceding
         expr(ls, &e) statement */
      for (i=1; i < nvars; i++) {
        luaK_exp2nextreg(ls->fs, &e); /* put e on stack */
        luaK_reserveregs(ls->fs, 1);  /* register it */
      }
      nexps = nvars;
    }
    else
      nomix;
  }
  else {
    e.k = VVOID;
    nexps = 0;
  }
  if (nvars != 0) {  /* only adjust* if there are simple variable declarations left */
    adjust_assign(ls, nvars, nexps, &e);
    adjustlocalvars(ls, nvars);
  }
}


static void exprstat (LexState *ls) {
  /* stat -> func | assignment */
  FuncState *fs = ls->fs;
  struct LHS_assign v;
  primaryexp(ls, &v.v);
  if (v.v.k == VCALL) { /* stat -> func */
    SETARG_C(getcode(fs, &v.v), 1);  /* call statement uses no results */
  }
  else {  /* stat -> assignment */
    v.prev = NULL;
    assignment(ls, &v, 1);
  }
}


static void globalstat (LexState *ls) {
  /* stat -> GLOBAL NAME {`,' NAME} */
  do {
    str_checkname(ls);
  } while (testnext(ls, ','));
}


/* forward declarations for global structure initialisation */
static void tablestat (LexState *ls);
static void setstat (LexState *ls);
static void seqstat (LexState *ls);
static void dictstat (LexState *ls);

static void createstat (LexState *ls) {
  /* stat -> CREATE [{table, set, seq}] NAME {`,' [{table, set, seq}] NAME} */
  do {
    if (testnext(ls, TK_LOCAL))
      localstat(ls, 1);
    else if (testnext(ls, TK_TTABLE))
      tablestat(ls);
    else if (testnext(ls, TK_TSET))
      setstat(ls);
    else if (testnext(ls, TK_SEQ) || testnext(ls, TK_TSEQUENCE))
      seqstat(ls);
    else if (testnext(ls, TK_DICT))
      dictstat(ls);
    else
      luaX_syntaxerror(ls, "keywords " LUA_QL("local") ", " LUA_QL("table") ", "
        LUA_QL("set") ", " LUA_QL("seq") ", or " LUA_QL("sequence") " expected");
  } while (testnext(ls, ','));
}


static void retstat (LexState *ls, int flag) {
  /* stat -> RETURN explist */
  FuncState *fs = ls->fs;
  BlockCnt *bl = fs->bl;  /* 2.1 RC 2 */
  expdesc e;
  int first, nret;  /* registers with returned values */
  first = nret = 0;
  if (flag)
    luaX_next(ls);  /* skip RETURN (when called by shortcut-function definitions) */
  if (flag && (block_follow(ls->t.token) || ls->t.token == ';'))
      first = nret = 0;  /* return no values */
  else if (!flag && block_follow(ls->t.token)) {
    first = nret = 0;
    luaX_syntaxerror(ls, "at least one return value expected");
  } else {
    nret = explist1(ls, &e);  /* optional return values */
    if (hasmultret(e.k)) {
      luaK_setmultret(fs, &e);
      if (e.k == VCALL && nret == 1) {  /* tail call? */
        SET_OPCODE(getcode(fs,&e), OP_TAILCALL);
        lua_assert(GETARG_A(getcode(fs,&e)) == fs->nactvar);
      }
      first = fs->nactvar;
      nret = LUA_MULTRET;  /* return all values */
    }
    else {
      if (nret == 1) { /* only one single value? */
        first = luaK_exp2anyreg(fs, &e);
      }
      else {
        luaK_exp2nextreg(fs, &e);  /* values must go to the `stack' */
        first = fs->nactvar;  /* return all `active' values */
        lua_assert(nret == fs->freereg - first);
      }
    }
  }
  /* before return, we should exit all try-catch blocks */
  while (bl) {
    if (bl->isbreakable == 2)
      luaK_codeABC(fs, OP_ENDTRY, 0, 0, 0);
    bl = bl->previous;
  }
  luaK_ret(fs, first, nret);
}


static void tablestat (LexState *ls) {  /* added 0.5.2, June 07, 2007; extended June 17, 2007,
  extended January 01, 2008; changed 0.12.1 */
  /* CREATE TABLE <var1> */
  FuncState *fs = ls->fs;
  expdesc t, e;
  struct LHS_assign v;
  int pc;
  primaryexp2(ls, &v.v);  /* no function call parsing */
  if (testnext(ls, '(')) {  /* set initial size for simple table with numeric keys */
    expr(ls, &e);
    luaK_exp2anyreg(fs, &e);
    pc = codearithbypass(fs, 0, &e, OPR_DIMTAB);
    checknext(ls, ')');
  } else
    pc = luaK_codeABC(fs, OP_NEWTABLE, 0, 0, 0);
  init_exp(&t, VRELOCABLE, pc);
  luaK_exp2nextreg(fs, &t);  /* fix it at stack top (for gc) */
  luaK_setoneret(fs, &t);  /* close last expression */
  luaK_storevar(fs, &v.v, &t);
}


static void dictstat (LexState *ls) {  /* added 0.12.2 */
  /* CREATE DICT <var1> */
  FuncState *fs = ls->fs;
  expdesc t, e;
  struct LHS_assign v;
  int pc;
  primaryexp2(ls, &v.v);  /* no function call parsing */
  if (testnext(ls, '(')) {  /* set initial size for simple table with numeric keys */
    expr(ls, &e);
    luaK_exp2anyreg(fs, &e);
    pc = codearithbypass(fs, 0, &e, OPR_DIMDICT);
    checknext(ls, ')');
  } else
    pc = luaK_codeABC(fs, OP_NEWTABLE, 0, 0, 0);
  init_exp(&t, VRELOCABLE, pc);
  luaK_exp2nextreg(fs, &t);  /* fix it at stack top (for gc) */
  luaK_setoneret(fs, &t);  /* close last expression */
  luaK_storevar(fs, &v.v, &t);
}


static void setstat (LexState *ls) {  /* added 0.10.0; April 17, 2008; changed 0.12.1 */
  /* CREATE SET <var1> */
  FuncState *fs = ls->fs;
  expdesc t, e;
  struct LHS_assign v;
  int pc;
  primaryexp2(ls, &v.v);  /* no function call parsing */
  if (testnext(ls, '(')) {  /* set initial size for simple table with numeric keys */
    expr(ls, &e);
    luaK_exp2anyreg(fs, &e);
    pc = codearithbypass(fs, 0, &e, OPR_DIMSET);
    checknext(ls, ')');
  } else {
    init_number(fs, &e, 0);
    pc = codearithbypass(fs, 0, &e, OPR_DIMSET);
  }
  init_exp(&t, VRELOCABLE, pc);
  luaK_exp2nextreg(fs, &t);  /* fix it at stack top (for gc) */
  luaK_setoneret(fs, &t);  /* close last expression */
  luaK_storevar(fs, &v.v, &t);
}


static void seqstat (LexState *ls) {  /* added 0.11.0; May 18, 2008; extended 0.12.0, June 30, 2008;
   changed 0.12.1 */
  /* CREATE SEQ <var1> */
  FuncState *fs = ls->fs;
  expdesc t, e;
  struct LHS_assign v;
  int pc;
  primaryexp2(ls, &v.v);  /* no function call parsing */
  if (testnext(ls, '(')) {
    expr(ls, &e);
    luaK_exp2anyreg(fs, &e);
    pc = codearithbypass(fs, 0, &e, OPR_DIMSEQ);
    checknext(ls, ')');
  } else {
    init_number(fs, &e, 0);
    pc = codearithbypass(fs, 0, &e, OPR_DIMSEQ);
  }
  init_exp(&t, VRELOCABLE, pc);
  luaK_exp2nextreg(fs, &t);  /* fix it at stack top (for gc) */
  luaK_setoneret(fs, &t);  /* close last expression */
  luaK_storevar(fs, &v.v, &t);
}


/* Used by INC and DEC statements
   added 0.5.2, June 08, 2007; extended November 04, 2007; patched May 10, 2010,
   September 17, 2010 */

static void changevalue (LexState *ls, int op) {
  FuncState *fs = ls->fs;
  expdesc v, e1, e2;
  unsigned int indices;
  testnext(ls, '(');
  indices = fs->freereg;
  primaryexp(ls, &v);
  e1 = v;
  if (v.k == VINDEXED)
    luaK_reserveregs(fs, fs->freereg - indices);  /* 1.0.1 patch: increase stack by the number of indices
       <- this call solved the problem with indexed values */
  if (testnext(ls, ',')) {
    luaK_exp2nextreg(fs, &e1);  /* <- this call solved the problem with indexed values, 0.32.0 */
    expr(ls, &e2);
  } else {  /* using special opcodes is not faster */
    init_exp(&e2, VKNUM, 0);
    e2.u.nval = (lua_Integer)1;
  }
  testnext(ls, ')');
  codearith(fs, (OpCode)op, &e1, &e2);
  luaK_setoneret(fs, &e1);  /* close last expression */
  luaK_storevar(fs, &v, &e1);
}


static void clearstat (LexState *ls) {  /* added 0.5.3, July 01, 2007; patched April 3, 2009 */
  FuncState *fs = ls->fs;
  expdesc e, v;
  int base;
  testnext(ls, '(');
  do {
    primaryexp(ls, &v);
    init_exp(&e, VNIL, 0);
    luaK_setoneret(fs, &e);
    luaK_storevar(fs, &v, &e);
  } while (testnext(ls, ','));
  testnext(ls, ')');  /* 0.13.3 */
  base = fs->freereg;
  luaK_codeABC(fs, OP_CMD, base, 0, CMD_GC);  /* perform garbage collection */
  fs->freereg = base;
}


static void insertstat (LexState *ls) {
  /* added 0.6.0, October 14, 2007; extended to process multiple values on Nov 24, 2007;
     extended to support multiple returns on April 03, 2009; patched May 09, 2010 to work
     with indexed names after the `into' token which it did not in very rare situations. */
  int nvals, base, fstemp;
  expdesc value, table;
  FuncState *fs = ls->fs;
  base = fs->freereg;
  /* 0.32.0 patch: indexed names confused the stack */
  fs->freereg += 2;  /* leave two slots for the structure (structure and possible index) into which
    the values shall be inserted */
  nvals = explist1(ls, &value);
  if (hasmultret(value.k)) {
    luaK_setmultret(fs, &value);
    nvals = 0;
  } else
    if (value.k != VVOID)
      luaK_exp2nextreg(fs, &value);  /* close last argument */
  checknext(ls, TK_INTO);
  fstemp = fs->freereg;
  fs->freereg = base;
  primaryexp(ls, &table);  /* insert the structure at the first position; in rare situations,
  expr does not work with indexed names, so it cannot be used. */
  luaK_exp2nextreg(fs, &table);
  fs->freereg = fstemp;
  luaK_codeABC(fs, OP_TINSERT, table.u.s.info, base+2, nvals);
  fs->freereg = base;
}


/* the reshuffle option was deleted since first deleting entries and then executing the reshuffle
   statement from the Agena environment is actually a little bit faster */
static void deletestat (LexState *ls) {
  /* added 0.6.0, October 14, 2007; extended to process multiple values on Nov 24, 2007;
     extended to support multiple returns on April 03, 2009; patched May 09, 2010 to work
     with indexed names after the `from' token which it did not in very rare situations. */
  int nvals, base, fstemp;
  expdesc value, table;
  FuncState *fs = ls->fs;
  base = fs->freereg;
  /* 0.32.0 patch: indexed names confused the stack */
  fs->freereg += 2;  /* leave two slots for the structure (structure and possible index) in which
    the values shall be inserted */
  nvals = explist1(ls, &value);
  if (hasmultret(value.k)) {
    luaK_setmultret(fs, &value);
    nvals = 0;
  } else if (value.k != VVOID)
    luaK_exp2nextreg(fs, &value);  /* close last argument */
  checknext(ls, TK_FROM);
  fstemp = fs->freereg;
  fs->freereg = base;
  primaryexp(ls, &table);  /* insert the structure at the first position */
  luaK_exp2nextreg(fs, &table);
  fs->freereg = fstemp;
  luaK_codeABC(fs, OP_TDELETE, table.u.s.info, base+2, nvals);
  fs->freereg = base;
}


static void popstat (LexState *ls) {
  /* added 0.24.0, June 20, 2009 */
  int base, islast;
  expdesc seq;
  FuncState *fs = ls->fs;
  islast = 0;
  base = fs->freereg;
  switch (ls->t.token) {
    case TK_TOP:
      islast = 1;
      luaX_next(ls);
      break;
    case TK_BOTTOM:
      islast = 0;
      luaX_next(ls);
      break;
    default:
      luaX_syntaxerror(ls, LUA_QL("bottom") " or " LUA_QL("top") " expected");
  }
  checknext(ls, TK_FROM);
  primaryexp(ls, &seq);
  luaK_exp2nextreg(fs, &seq);  /* 0.24.1 patch */
  luaK_codeABC(fs, OP_CMD, base, 1, islast ? CMD_POPTOP : CMD_POPBOTTOM);
  fs->freereg = base;
}


static void importstat (LexState *ls) {  /* 2.0.0, November 06 & 11, 2013 */
  int nvals, base, isalias;
  expdesc str;
  FuncState *fs = ls->fs;
  base = fs->freereg;
  nvals = 1;
  init_exp(&str, VNIL, 0);
  luaK_exp2nextreg(fs, &str);
  do {
    checkname(ls, &str);
    luaK_exp2nextreg(fs, &str);
    nvals++;
  } while (testnext(ls, ','));
  isalias = (ls->t.token == TK_ALIAS);
  if (isalias) {
    if (nvals > 2)
      luaX_syntaxerror(ls, "only one package name allowed if " LUA_QL("alias") " given");
    else {
      luaX_next(ls);
      if (ls->t.token == TK_NAME) {
        do {
          checkname(ls, &str);
          luaK_exp2nextreg(fs, &str);
          nvals++;
        } while (testnext(ls, ','));
      } else {  /* else call `with` with only the package name */
        init_exp(&str, VFALSE, 0);  /* do not print anything on screen */
        luaK_exp2nextreg(fs, &str);
        nvals++;
      }
      luaK_codeABC(fs, OP_CMD, base, nvals, CMD_ALIAS);
    }
  } else {
    init_exp(&str, VFALSE, 0);
    luaK_exp2nextreg(fs, &str);
    nvals++;
    luaK_codeABC(fs, OP_CMD, base, nvals, CMD_IMPORT);
  }
  fs->freereg = base;
}


void enumglobalstat (LexState *ls, struct LHS_assign *lh, int nvars) {
  if (testnext(ls, ',')) {
    struct LHS_assign v;
    v.prev = lh;
    simpleexp(ls, &v.v);
    luaY_checklimit(ls->fs, nvars, LUAI_MAXCCALLS - ls->L->nCcalls,
      "variable names");
    enumglobalstat(ls, &v, nvars+1);
  } else {
    expdesc start, step, offset;
    struct LHS_assign *varname;
    FuncState *fs = ls->fs;
    if (testnext(ls, TK_FROM)) {
      expr(ls, &start);
    } else {
      init_exp(&start, VKNUM, 0);
      start.u.nval = (lua_Number)1;
    }
    /* since loop count is backwards, increase start value by nvars */
    init_exp(&offset, VKNUM, 0);
    offset.u.nval = (lua_Number)(nvars);
    codearith(fs, OP_ADD, &start, &offset);
    luaK_exp2nextreg(fs, &start);  /* 0.32.1, this line solved the problem with indexed names following the `from' keyword */
    /* step size (backward counting) */
    init_exp(&step, VKNUM, 0);
    step.u.nval = (lua_Number)(-1);
    for (varname=lh; ; varname = varname->prev) {  /* traverse lh backwards ! */
      codearith(fs, OP_ADD, &start, &step);
      luaK_exp2nextreg(fs, &start);
      luaK_reserveregs(fs, 1);
      luaK_storevar(fs, &varname->v, &start);
      if (varname->prev == NULL) break;  /* cannot be included into the `to' clause of the for loop */
    }
  }
}


static void enumstat (LexState *ls) {
  struct LHS_assign v;
  /* global enum vars */
  primaryexp(ls, &v.v);
  v.prev = NULL;
  enumglobalstat(ls, &v, 1);
}


static void byestat (LexState *ls) {
  luaK_codeABC(ls->fs, OP_CMD, ls->fs->freereg, 0, CMD_BYE);
}


static void clsstat (LexState *ls) {
  luaK_codeABC(ls->fs, OP_CMD, ls->fs->freereg, 0, CMD_CLS);
}


static void restartstat (LexState *ls) {
  int base;
  FuncState *fs = ls->fs;
  base = fs->freereg;
  luaK_codeABC(fs, OP_CMD, base, 0, CMD_RESTART);
  fs->freereg = base;
}


static int statement (LexState *ls) {
  int line = ls->linenumber;  /* may be needed for error messages */
  switch (ls->t.token) {
    case TK_IF: {  /* stat -> ifstat */
      ifstat(ls, line);
      return 0;
    }
    case TK_WHILE: {  /* stat -> whilestat */
      whilestat(ls, line);
      return 0;
    }
    case TK_DO: {  /* stat -> dostat, changed 0.5.3 */
      dostat(ls, line);
      return 0;
    }
    case TK_FOR: {  /* stat -> forstat */
      forstat(ls, line);
      return 0;
    }
    case TK_LOCAL: {  /* stat -> localstat */
      luaX_next(ls);  /* skip LOCAL */
      localstat(ls, 0);
      return 0;
    }
    case TK_RET: {  /* changed 0.5.2, stat -> retstat */
      retstat(ls, 1);
      return 1;  /* must be last statement */
    }
    case TK_BREAK: {  /* stat -> breakstat */
      luaX_next(ls);  /* skip BREAK */
      breakstat(ls);
      return 1;  /* must be last statement */
    }
    case TK_SKIP: {  /* stat -> skipstat, changed  */
      luaX_next(ls);  /* skip SKIP */
      skipstat(ls);
      return 1;     /* must be last statement */
    }
    case TK_REDO: {  /* stat -> redostat, changed  */
      luaX_next(ls);  /* skip REDO */
      redostat(ls);
      return 1;     /* must be last statement */
    }
    case TK_CREATE: {  /* added 0.12.2, September 13, 2008 */
      luaX_next(ls);  /* skip SET */
      createstat(ls);
      return 0;
    }
    case TK_INC: {  /* added 0.5.2.*/
      luaX_next(ls);
      changevalue(ls, OP_ADD);
      return 0;
    }
    case TK_DEC: {  /* added 0.5.2.*/
      luaX_next(ls);
      changevalue(ls, OP_SUB);
      return 0;
    }
    case TK_CASE: {  /* added 0.5.2.*/
      selectstat(ls, line);
      return 0;
    }
    case TK_INSERT: {  /* added 0.6.0, stat -> putstat */
      luaX_next(ls);
      insertstat(ls);
      return 0;
    }
    case TK_DELETE: {  /* added 0.6.0, stat -> remstat */
      luaX_next(ls);
      deletestat(ls);
      return 0;
    }
    case TK_CLEAR: {  /* added 0.5.2.*/
      luaX_next(ls);
      clearstat(ls);
      return 0;
    }
    case TK_IMPORT: {  /* added 2.0.0 */
      luaX_next(ls);
      importstat(ls);
      return 0;
    }
    case TK_ENUM: {  /* added 0.6.0 */
      luaX_next(ls);
      enumstat(ls);
      return 0;
    }
    case TK_SCOPE: {  /* stat -> SCOPE block EPOCS */
      luaX_next(ls);  /* skip SCOPE */
      block(ls);
      check_match(ls, TK_EPOCS, TK_SCOPE, line);
      return 0;
    }
    case TK_POP: {
      luaX_next(ls);  /* skip POP */
      popstat(ls);
      return 0;
    }
    case TK_GLOBAL: {  /* stat -> localstat */
      luaX_next(ls);  /* skip GLOBAL */
      globalstat(ls);
      return 0;
    }
    case TK_TO: {  /* stat -> tostat */
      tostat(ls, line);
      return 0;
    }
    case TK_CLS: {  /* added 0.4.0 */
      clsstat(ls);
      luaX_next(ls);
      return 1;
    }
    case TK_BYE: {
      byestat(ls);
      luaX_next(ls);
      return 1;
    }
    case TK_RESTART: {
      restartstat(ls);
      luaX_next(ls);
      return 1;
    }
    case TK_TRY: {
      trystat(ls, line);
      return 0;
    }
    default: {
      exprstat(ls);
      return 0;  /* to avoid warnings */
    }
  }
}


/* evaluate short-cut function statement, added 0.5.2 */

static void defchunk (LexState *ls) {
  /* chunk -> { stat } */
  enterlevel(ls);
  retstat(ls, 0);
  lua_assert(ls->fs->f->maxstacksize >= ls->fs->freereg &&
             ls->fs->freereg >= ls->fs->nactvar);
  ls->fs->freereg = ls->fs->nactvar;  /* free registers */
  leavelevel(ls);
}


static void chunk (LexState *ls) {
  /* chunk -> { stat [`;'] } */
  int islast = 0;
  enterlevel(ls);
  while (!islast && !block_follow(ls->t.token)) {
    islast = statement(ls);
    testnext(ls, ';');
    lua_assert(ls->fs->f->maxstacksize >= ls->fs->freereg &&
               ls->fs->freereg >= ls->fs->nactvar);
    ls->fs->freereg = ls->fs->nactvar;  /* free registers */
  }
  leavelevel(ls);
}

/* }====================================================================== */

