/*
** $Id: loslib.c,v 1.19 2006/04/26 18:19:49 roberto Exp $
** Standard Operating System library
** See Copyright Notice in agena.h
*/

#include <errno.h>
#include <locale.h>
#include <stdlib.h>
#include <string.h>
#include <sys/timeb.h>  /* ftime, for milliseconds */

/* instead of time.h: */
#include "agnt64.h"

#include "sofa.h"

#include <sys/stat.h> /* for mkdir, stat */

/* for os.fstat to determine correct sizes of files > 4 GB */
#ifdef _WIN32
  #undef stat
  #define stat  _stati64
  #undef fstat
  #define fstat _fstati64
  #undef wstat
  #define wstat _wstati64
#elif defined (__SVR4) && defined (__sun)
  #undef stat
  #define stat stat64
  #undef lstat
  #define lstat lstat64
#elif defined(__unix__) && !defined(LUA_DOS)
  #undef _FILE_OFFSET_BITS
  #define _FILE_OFFSET_BITS	64
#endif


#if _WIN32
#include <windows.h>  /* for os_memstate */
#include <io.h>       /* for access */
#include <ctype.h>    /* for toupper */
#include <process.h>  /* for getpid & getppid */
#include <errno.h>     /* symlink */
#include <ole2.h>      /* symlink */
#include <shlobj.h>    /* symlink */
#endif

/* This OS2 header must be placed before including unistd, utsname, ncurses,
   otherwise GCC will not compile successfully. 0.13.3 */
#ifdef __OS2__
#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_NOPM
#define INCL_DOSPROCESS  /* for DosBeep */
#define INCL_DOSFILEMGR
#include <os2.h>     /* for memory status queries */
#endif

#if defined(__unix__) || defined(__OS2__) || defined(__APPLE__) || defined(__HAIKU__)
#include <unistd.h>       /* for access and sleep */
#include <sys/utsname.h>  /* for uname */
#endif

#if defined(__unix__) || defined(__APPLE__)
#include <signal.h>       /* for signal and alarm */
#endif

#ifdef __APPLE__
  #include <mach/mach.h>
  #include <sys/sysctl.h>
  #include <crt_externs.h>  /* for os.environ */
  #include <stdint.h>
  #include <sys/types.h>
  #include <sys/sysctl.h>
  #define environ (*_NSGetEnviron())  /* for os.environ */
#elif defined(__unix__) || defined(__OS2__)
  extern char **environ;  /* for os.environ */
#endif

#if (defined(__linux__))  /* for CD-ROM function(s) */
  #include <fcntl.h>
  #include <linux/cdrom.h>
  #include <sys/ioctl.h>
  #include <sys/types.h>
#endif

#if (defined(_WIN32) || defined(__unix__) || defined(__APPLE__) || defined(LUA_DOS))
#include <utime.h>  /* for utime() */
#endif

#include <dirent.h>  /* used by io_ls function */
#include <errno.h>   /* used by io_ls function */

#define loslib_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "agnconf.h"  /* for LUA_TMPNAMBUFSIZE */
#include "agnhlps.h"  /* do not change the order of this #include statement ! */

#ifdef __APPLE__
#include "agncmpt.h"
#endif


static int os_pushresult (lua_State *L, int i, const char *filename, const char *procname) {
  int en = errno;  /* calls to Lua API may change this value */
  if (i == -1) {  /* function failed ? */
    luaL_error(L, "Error in " LUA_QS " with " LUA_QS ": %s ", procname, filename, strerror(en));  /* Agena 1.2 */
    return 0;
  }
  else {
    lua_pushtrue(L);
    return 1;
  }
}


static void setboolean (lua_State *L, const char *key, unsigned int value) {
  lua_pushstring(L, key);
  lua_pushboolean(L, value);
  lua_settable(L, -3);
}


static void setstring (lua_State *L, const char *key, const char *value) {
  lua_pushstring(L, key);
  lua_pushstring(L, value);
  lua_rawset(L, -3);
}


static void setnumber (lua_State *L, const char *key, lua_Number value) {
  lua_pushstring(L, key);
  lua_pushnumber(L, value);
  lua_rawset(L, -3);
}


static int os_execute (lua_State *L) {
  if (system(NULL) == 0)  /* 1.9.1 */
    luaL_error(L, "Error in " LUA_QS ": command processor is not available.", "os.execute");
  lua_pushinteger(L, system(luaL_optstring(L, 1, NULL)));
  return 1;
}


/* even when deleting 100k or more files, accepting tables, sets, or sequences of file names did not speed up this function at
   least on a MacBook Pro manufactured in 2011 running Mac OS 10.7 */
static int os_remove (lua_State *L) {
  const char *filename = agn_checkstring(L, 1);
  if (access(filename, 00|04) != -1)  /* 0.26.0 patch */
    return os_pushresult(L, remove(filename), filename, "os.remove");  /* Agena 1.9.5 fix */
  else {
    luaL_error(L, "Error in " LUA_QS " with " LUA_QS ": file or directory does not exist.", "os.remove", filename);  /* Agena 1.2 */
    return 0;
  }
}


static int os_move (lua_State *L) {
  const char *fromname = agn_checkstring(L, 1);
  const char *toname = agn_checkstring(L, 2);
  if (access(fromname, 00|04) != -1)  /* 0.26.0 patch */
    return os_pushresult(L, rename(fromname, toname), fromname, "os.move");  /* Agena 1.9.5 */
  else {
    luaL_error(L, "Error in " LUA_QS " with " LUA_QS ": file or directory does not exist.", "os.move", fromname);  /* Agena 1.2 */
    return 0;
  }
}


static int os_tmpname (lua_State *L) {
  char buff[LUA_TMPNAMBUFSIZE];
  int err;
  lua_tmpnam(buff, err);
  if (err)
    return luaL_error(L, "Error in " LUA_QS ": unable to generate a unique filename.", "os.tmpname");
  lua_pushstring(L, buff);
  return 1;
}


static int os_getenv (lua_State *L) {
/* tested in Windows, Mac OS X, Solaris 10, Linux, OS/2, DOS */
#if defined(_WIN32)
  /* written by * Copyright 2007 Mark Edgar < medgar at gmail com >,
  taken from http://lua-ex-api.googlecode.com/svn/trunk/w32api/ex.c */
  const char *nam = luaL_checkstring(L, 1);
  char sval[256], *val;
  size_t len = GetEnvironmentVariable(nam, val = sval, sizeof sval);
  if (sizeof sval < len)
    len = GetEnvironmentVariable(nam, val = lua_newuserdata(L, len), len);
  if (len == 0)
    lua_pushnil(L);
  else
    lua_pushlstring(L, val, len);
#else
  lua_pushstring(L, getenv(luaL_checkstring(L, 1)));  /* if NULL push null */
#endif
  return 1;
}


/* code written by * Copyright 2007 Mark Edgar < medgar at gmail com >,
   taken from http://lua-ex-api.googlecode.com/svn/trunk/w32api/ex.c */
static int os_setenv (lua_State *L) {  /* Agena 1.0.2 */
/* tested in Windows, Mac OS X, Solaris 10, Linux, OS/2, DOS */
#if defined(_WIN32)
  int r;
  r = SetEnvironmentVariable(luaL_checkstring(L, 1), luaL_checkstring(L, 2));
  if (!r)
    luaL_error(L, "Error in " LUA_QS ": %s.", "os.setenv", GetLastError());
  else
    lua_pushtrue(L);
#else
  const char *nam, *val;
  int err = 0;
  nam = luaL_checkstring(L, 1);
  val = lua_tostring(L, 2);
  #ifdef LUA_DOS
  err = setenv(nam, val, 1);
  #elif !defined(__APPLE__)
  err = (val ? setenv(nam, val, 1) : unsetenv(nam));
  #else
  if (val == 0)
    unsetenv(nam);  /* unsetenv is of type void in Mac OS X */
  else
    err = setenv(nam, val, 1);
  #endif
  if (err == -1)
    luaL_error(L, "Error in " LUA_QS ": could not %s environment variable.", "os.setenv",
      val == 0 ? "unset" : "set");
  lua_pushtrue(L);
#endif
  return 1;
}


static int os_environ (lua_State *L) {  /* Agena 1.0.2,
  written by * Copyright 2007 Mark Edgar < medgar at gmail com >,
  taken from http://lua-ex-api.googlecode.com/svn/trunk/w32api/ex.c */

/* tested in Windows, Mac OS X, Solaris 10, Linux, OS/2, DOS */
#if defined(_WIN32)
  const char *nam, *val, *end;
  const char *envs = GetEnvironmentStrings();
  if (!envs) {
    luaL_error(L, "Error in " LUA_QS ": could not get environment.", "os.environ");
  }
  lua_newtable(L);
  for (nam = envs; *nam; nam = end + 1) {
    end = strchr(val = strchr(nam, '=') + 1, '\0');
    if (val - nam - 1 != 0) {  /* do not enter entry of the key is the empty string */
      lua_pushlstring(L, nam, val - nam - 1);
      lua_pushlstring(L, val, end - val);
      lua_settable(L, -3);
    }
  }
#elif defined(__unix__) || defined(__APPLE__) || defined(__OS2__)  /* works in DOS, too */
  const char *nam, *val, *end;
  const char **env;
  lua_newtable(L);
  for (env = (const char **)environ; (nam = *env); env++) {
    end = strchr(val = strchr(nam, '=') + 1, '\0');
    lua_pushlstring(L, nam, val - nam - 1);
    lua_pushlstring(L, val, end - val);
    lua_settable(L, -3);
  }
#else
  lua_pushfail(L);
#endif
  return 1;
}


/*
** {======================================================
** Time/Date operations
** { year=%Y, month=%m, day=%d, hour=%H, min=%M, sec=%S,
**   wday=%w+1, yday=%j, isdst=? }
** =======================================================
*/

static void setfield (lua_State *L, const char *key, int value) {
  lua_pushinteger(L, value);
  lua_setfield(L, -2, key);
}

static void setboolfield (lua_State *L, const char *key, int value) {
  if (value < 0)  /* undefined? */
    return;  /* does not set field */
  lua_pushboolean(L, value);
  lua_setfield(L, -2, key);
}

static int getboolfield (lua_State *L, const char *key) {
  int res;
  lua_getfield(L, -1, key);
  res = lua_isnil(L, -1) ? -1 : lua_toboolean(L, -1);
  agn_poptop(L);
  return res;
}


static int getfield (lua_State *L, const char *key, int d) {
  int res;
  lua_getfield(L, -1, key);
  if (agn_isnumber(L, -1))
    res = (int)lua_tointeger(L, -1);
  else {
    if (d < 0)
      return luaL_error(L, "field " LUA_QS " missing in date table.", key);
    res = d;
  }
  agn_poptop(L);
  return res;
}


static int os_date (lua_State *L) {
  struct TM *stm;
  struct timeb tp;
  int msecs, ye, mo, da, ho, mi, se;
  const char *s;
  Time64_T t = luaL_opt(L, (Time64_T)agnL_checknumber, 2, time(NULL));
  s = luaL_optstring(L, 1, "%c");
  if (*s == '!') {  /* UTC? */
    stm = gmtime64(&t);
    s++;  /* skip `!' */
  }
  else
    stm = localtime64(&t);
  msecs = -1;
  /* 1.11.7, get milliseconds */
  #ifdef _WIN32
  ftime(&tp);
  msecs = tp.millitm;
  #elif defined(__unix__) || defined(__APPLE__)
  if (ftime(&tp) == 0)  /* set milliseconds only when query has been successful */
    msecs = tp.millitm;
  #endif
  if (stm == NULL) { /* invalid date? */
    lua_pushnil(L);
    return 1;
  }
  ye = stm->tm_year+1900;
  mo = stm->tm_mon+1;
  da = stm->tm_mday;
  ho = stm->tm_hour;
  mi = stm->tm_min;
  se = stm->tm_sec;
  if (lua_gettop(L) == 0) {
    char *rstr, *year, *month, *day, *hour, *minute, *second, *msecond;
    year    = my_dtoa(ye);
    month   = my_dtoa(mo);
    day     = my_dtoa(da);
    hour    = my_dtoa(ho);
    minute  = my_dtoa(mi);
    second  = my_dtoa(se);
    msecond = my_dtoa(msecs);
    if (msecs != -1)
      rstr = concat(year, "/",
         (mo < 10 ? "0" : ""), month, "/",
         (da < 10 ? "0" : ""), day, " ",
         (ho < 10 ? "0" : ""), hour, ":",
         (mi < 10 ? "0" : ""), minute, ":",
         (se < 10 ? "0" : ""), second, ".",
         (msecs > 99 ? "" : (msecs > 9 ? "0" : "00")),
         msecond, NULL);
    else
      rstr = concat(year, "/",
         (mo < 10 ? "0" : ""), month, "/",
         (da < 10 ? "0" : ""), day, " ",
         (ho < 10 ? "0" : ""), hour, ":",
         (mi < 10 ? "0" : ""), minute, ":",
         (se < 10 ? "0" : ""), second, NULL);
    lua_pushstring(L, rstr);
    xfree(rstr); xfree(year); xfree(month); xfree(day); xfree(hour); xfree(minute); xfree(second); xfree(msecond);
  }
  else if (strcmp(s, "*t") == 0) {
    lua_createtable(L, 0, 10);  /* 9 = number of fields */
    setfield(L, "sec", se);
    setfield(L, "min", mi);
    setfield(L, "hour", ho);
    setfield(L, "day", da);
    setfield(L, "month", mo);
    setfield(L, "year", ye);
    setfield(L, "wday", stm->tm_wday+1);
    setfield(L, "yday", stm->tm_yday+1);
    setboolfield(L, "isdst", stm->tm_isdst);
    if (msecs != -1)
      setfield(L, "msec", msecs);  /* set milliseconds */
  }
  else {  /* Lua 5.1.2 patch, patched Agena 1.12.4 */
    char cc[3];
    luaL_Buffer b;
    cc[0] = '%'; cc[2] = '\0';
    luaL_buffinit(L, &b);
    for (; *s; s++) {
      if (*s != '%' || *(s + 1) == '\0')  /* no conversion specifier? */
        luaL_addchar(&b, *s);  /* add character to resulting string */
      else {
        size_t reslen;
        char buff[256];  /* should be big enough for any conversion result */
        cc[1] = *(++s);
        /* see: http://www.gnu.org/software/libc/manual/html_node/Formatting-Calendar-Time.html */
        reslen = strftime(buff, sizeof(buff), cc, stm);
        luaL_addlstring(&b, buff, reslen);
      }
    }
    luaL_pushresult(&b);
  }
  return 1;
}


static int os_time (lua_State *L) {
  Time64_T t;  /* 1.12.4 */
  int msecs, r;
  struct timeb tp;
  r = 1;
  if (lua_isnoneornil(L, 1))  /* called without args? */
    t = time(NULL);  /* get current time */
  else {
    struct TM ts;  /* 1.12.4 */
    switch (lua_type(L, 1)) {
      case LUA_TTABLE: {
        lua_settop(L, 1);  /* make sure table is at the top */
        ts.tm_sec = getfield(L, "sec", 0);
        ts.tm_min = getfield(L, "min", 0);
        ts.tm_hour = getfield(L, "hour", 12);
        ts.tm_mday = getfield(L, "day", -1);
        ts.tm_mon = getfield(L, "month", -1) - 1;
        ts.tm_year = getfield(L, "year", -1) - 1900;
        ts.tm_isdst = getboolfield(L, "isdst");
        t = mktime64(&ts);  /* Agena 1.8.0 */
        break;
      }
      case LUA_TSEQ: {
        size_t nops;
        lua_settop(L, 1);  /* make sure sequence is at the top */
        nops = agn_seqsize(L, 1);
        if (nops < 3)
          luaL_error(L, "Error in " LUA_QS ": expected a sequence of at least three numbers.", "os.time");
        ts.tm_year = lua_seqgetinumber(L, 1, 1) - 1900;
        ts.tm_mon = lua_seqgetinumber(L, 1, 2) - 1;
        ts.tm_mday = lua_seqgetinumber(L, 1, 3);
        ts.tm_hour = nops > 3 ? lua_seqgetinumber(L, 1, 4) : 0;
        ts.tm_min = nops > 4 ? lua_seqgetinumber(L, 1, 5) : 0;
        ts.tm_sec = nops > 5 ? lua_seqgetinumber(L, 1, 6) : 0;
        if (nops > 6) {
          lua_seqgeti(L, 1, 7);
          if (lua_isboolean(L, -1)) {
            ts.tm_isdst = lua_toboolean(L, -1) > 0;
            agn_poptop(L);
          } else {
            agn_poptop(L);
            luaL_error(L, "Error in " LUA_QS ": seventh element in sequence must be a boolean.", "os.time");
          }
        } else
          ts.tm_isdst = 0;
        t = mktime64(&ts);
        break;
      }
      default: {
        t = time(NULL);  /* to avoid compiler warnings */
        luaL_error(L, "Error in " LUA_QS ": first argument must either be a table or sequence.", "os.time");
      }
    }
  }
  msecs = -1;
  /* 1.11.7, get milliseconds */
  #ifdef _WIN32
  ftime(&tp);
  msecs = tp.millitm;
  #elif defined(__unix__) || defined(__APPLE__)
  if (ftime(&tp) == 0)  /* set milliseconds only when query has been successful */
    msecs = tp.millitm;
  #endif
  if (t == (time_t)(-1))
    lua_pushnil(L);
  else {
    lua_pushnumber(L, (lua_Number)t);
    if (msecs != -1) {
      lua_pushnumber(L, msecs);
      r = 2;
    }
  }
  return r;
}


static int os_difftime (lua_State *L) {
  lua_pushnumber(L, difftime((time_t)(agnL_checknumber(L, 1)),
                             (time_t)(agnL_optnumber(L, 2, 0))));
  return 1;
}

/* }====================================================== */


static int os_setlocale (lua_State *L) {
  static const int cat[] = {LC_ALL, LC_COLLATE, LC_CTYPE, LC_MONETARY,
                      LC_NUMERIC, LC_TIME};
  static const char *const catnames[] = {"all", "collate", "ctype", "monetary",
     "numeric", "time", NULL};
  const char *l = luaL_optstring(L, 1, NULL);
  int op = luaL_checkoption(L, 2, "all", catnames);
  lua_pushstring(L, setlocale(cat[op], l));
  return 1;
}


static int os_exit (lua_State *L) {
  exit(agnL_optinteger(L, 1, EXIT_SUCCESS));
  return 0;  /* to avoid warnings */
}


/* determine information on available memory on Windows platforms, 0.5.3, September 15, 2007;
   extended in 0.12.2 on September 29, 2008 to work in UNIX; extended to fully support OS/2 in 0.13.3;
   extended to support Mac OS X in 0.31.3 */

static int os_memstate (lua_State *L) {
#if defined(_WIN32) || (defined(__unix__) && !defined(LUA_DOS)) || defined(__OS2__) || defined(__APPLE__) || defined(__HAIKU__)
  long AvailPhys, TotalPhys;
  lua_Number div;
#if defined(_WIN32)
  MEMORYSTATUS memstat;
  long AvailVirtual, TotalVirtual, MemoryLoad;
  int Length;
#elif __OS2__
  APIRET result;
  ULONG memInfo[3];
#elif defined(__APPLE__)
  vm_size_t pagesize;
  vm_statistics_data_t vminfo;
  mach_port_t system;
  unsigned int c;
  int sysmask[2];
  size_t length;
  uint64_t memsize;
  lua_Number f;
#endif
  const char *option = luaL_optstring(L, 1, "bytes");
  if (strcmp(option, "bytes") == 0)
    div = 1;
  else if (strcmp(option, "kbytes") == 0)
    div = 1024;
  else if (strcmp(option, "mbytes") == 0)
    div = 1024*1024;
  else if (strcmp(option, "gbytes") == 0)
    div = 1024*1024*1024;
  else
    return luaL_error(L, "unknown option %s.", option);
#if defined(_WIN32)
  memstat.dwLength = sizeof(MEMORYSTATUS);
  GlobalMemoryStatus(&memstat);
  AvailPhys = (long)memstat.dwAvailPhys;
  TotalPhys = (long)memstat.dwTotalPhys;
  AvailVirtual = (long)memstat.dwAvailVirtual;
  TotalVirtual = (long)memstat.dwTotalVirtual;
  MemoryLoad = (long)memstat.dwMemoryLoad;
  Length = (int)memstat.dwLength;
  lua_newtable(L);
  setnumber(L, "freephysical", (lua_Number)(AvailPhys)/div);
  setnumber(L, "totalphysical", (lua_Number)(TotalPhys)/div);
  setnumber(L, "freevirtual", (lua_Number)(AvailVirtual)/div);
  setnumber(L, "totalvirtual", (lua_Number)(TotalVirtual)/div);  /* 0.26.0 patch */
#elif (defined(__unix__) && !defined(LUA_DOS)) || defined(__HAIKU__) /* for DJGPP */
  AvailPhys = sysconf(_SC_AVPHYS_PAGES) * sysconf(_SC_PAGESIZE);
  TotalPhys = sysconf(_SC_PHYS_PAGES) * sysconf(_SC_PAGESIZE);
  lua_newtable(L);
  setnumber(L, "freephysical", (lua_Number)(AvailPhys)/div);
  setnumber(L, "totalphysical", (lua_Number)(TotalPhys)/div);
#elif __OS2__
  result = DosQuerySysInfo(QSV_TOTPHYSMEM, QSV_TOTAVAILMEM,
      &memInfo[0], sizeof(memInfo));
  AvailPhys = 0;
  TotalPhys = 0;
    /* sysconf(_SC_AVPHYS_PAGES) * sysconf(_SC_PAGESIZE); does not work for
      the _SC* names are not defined in GCC unistd.h */
  if (result != 0) {
    lua_pushfail(L);
    return 1;
  }
  lua_newtable(L);
  setnumber(L, "totalphysical", (lua_Number)memInfo[0]/div);
  setnumber(L, "freevirtual", (lua_Number)memInfo[2]/div);
  setnumber(L, "resident", (lua_Number)memInfo[1]/div);
#elif defined(__APPLE__)
  sysmask[0] = CTL_HW;
  sysmask[1] = HW_MEMSIZE;
  length = sizeof(memsize);
  memsize = 0L;
  lua_newtable(L);
  c = HOST_VM_INFO_COUNT;
  system = mach_host_self();
  AvailPhys = AGN_NAN;  /* to prevent compiler warnings */
  TotalPhys = AGN_NAN;  /* to prevent compiler warnings */
  if (host_page_size(system, &pagesize) != KERN_SUCCESS) pagesize = 4096;
  f = pagesize/div;
  if (host_statistics(system, HOST_VM_INFO, (host_info_t)(&vminfo), &c) != KERN_SUCCESS) {
    setnumber(L, "freephysical", AvailPhys);
    setnumber(L, "active", AGN_NAN);
    setnumber(L, "inactive", AGN_NAN);
    setnumber(L, "speculative", AGN_NAN);
    setnumber(L, "wireddown", AGN_NAN);
    setnumber(L, "reactivated", AGN_NAN);
  }
  else {
    setnumber(L, "freephysical", ((vminfo.free_count - vminfo.speculative_count)) * f);
    setnumber(L, "active", vminfo.active_count * f);
    setnumber(L, "inactive", vminfo.inactive_count * f);
    setnumber(L, "speculative", vminfo.speculative_count * f);
    setnumber(L, "wireddown", vminfo.wire_count * f);
    setnumber(L, "reactivated", vminfo.reactivations * f);
  }
  if (sysctl(sysmask, 2, &memsize, &length, NULL, 0 ) == -1)
    setnumber(L, "totalphysical", TotalPhys);
  else
    setnumber(L, "totalphysical", memsize/div);
#endif
#else
  lua_pushfail(L);
#endif
  return 1;
}

/* get the current Windows login name, 0.5.3, September 15, 2007;
   extended 0.12.2, October 12, 2008; changed to support OS/2 0.13.3 */

static int os_login (lua_State *L) {
#if defined(_WIN32)
  char buffer[257];
  DWORD len = 256;
  if (GetUserName(buffer, &len)) {  /* Agena 1.6.0 */
    buffer[len] = '\0';
    lua_pushstring(L, buffer);
  } else
    lua_pushfail(L);
#elif defined(__unix__) || defined(__OS2__) || defined(__APPLE__) || defined(__HAIKU__)
  char *LoginName;
  LoginName = getlogin();
  if (LoginName == NULL)
    lua_pushfail(L);
  else
    lua_pushstring(L, LoginName);
#else
  lua_pushfail(L);
#endif
  return 1;
}


/* get the Windows computer name, 0.5.3, September 15, 2007;
   improved in 0.13.3 */

static int os_computername (lua_State *L) {
#if defined(_WIN32)
  char buffer[257];
  DWORD len = 256;
  if (GetComputerName(buffer, &len)) {  /* Agena 1.6.0 */
    buffer[len] = '\0';
    lua_pushstring(L, buffer);
  } else
    lua_pushfail(L);
#elif defined(__unix__) || defined(__APPLE__) || defined(__HAIKU__)
  /* does not work with OS/2: needs a file `TCPIP32` (as does bash). If not
     present, Agena does not start. */
  char buffer[256];
  if (gethostname(buffer, sizeof buffer) == 0)
    lua_pushstring(L, buffer);
  else
    lua_pushfail(L);
#else
  lua_pushfail(L);
#endif
  return 1;
}


/* return amount of free memory; extended 0.12.2, 28.09.2008; OS/2 addon
   August 07, 2009, OS X addon 0.31.3 March 28, 2010 */
static int os_freemem (lua_State *L) {
#if defined(_WIN32) || (defined(__unix__) && !defined(LUA_DOS)) || defined(__OS2__) || defined(__APPLE__) || defined(__HAIKU__)
  lua_Number div;
  const char *option = luaL_optstring(L, 1, "bytes");
#if defined(_WIN32)
  MEMORYSTATUS memstat;
#elif __OS2__
  APIRET result;
  ULONG memInfo[3];
#elif defined(__APPLE__)
  vm_size_t pagesize;
  vm_statistics_data_t vminfo;
  mach_port_t system;
  unsigned int c;
#endif
  if (strcmp(option, "bytes") == 0)
    div = 1;
  else if (strcmp(option, "kbytes") == 0)
    div = 1024;
  else if (strcmp(option, "mbytes") == 0)
    div = 1024*1024;
  else if (strcmp(option, "gbytes") == 0)
    div = 1024*1024*1024;
  else
    return luaL_error(L, "Error in " LUA_QS ": unknown option %s.", "os.freemem", option);
#if defined(_WIN32)
  memstat.dwLength = sizeof(MEMORYSTATUS);
  GlobalMemoryStatus(&memstat);
  lua_pushnumber(L, (lua_Number)memstat.dwAvailPhys/div);
#elif defined(__unix__) && !defined(LUA_DOS) || defined(__HAIKU__)  /* excludes DJGPP */
  lua_pushnumber(L, (lua_Number)(sysconf(_SC_AVPHYS_PAGES) * sysconf(_SC_PAGESIZE))/div);
#elif defined (__OS2__)
  result = DosQuerySysInfo(QSV_TOTPHYSMEM, QSV_TOTAVAILMEM,
      &memInfo[0], sizeof(memInfo));
  if (result != 0)  /* return free virtual memory */
    lua_pushfail(L);
  else
    lua_pushnumber(L, (lua_Number)(memInfo[2])/div);
#elif defined(__APPLE__)
  c = HOST_VM_INFO_COUNT;
  system = mach_host_self();
  if (host_page_size(mach_host_self(), &pagesize) != KERN_SUCCESS) pagesize = 4096;
  if (host_statistics(system, HOST_VM_INFO, (host_info_t)(&vminfo), &c) != KERN_SUCCESS)
    lua_pushfail(L);
  else
    lua_pushnumber(L, (uint64_t)(vminfo.free_count - vminfo.speculative_count) * pagesize/div);
#endif
#else
  lua_pushfail(L);
#endif
  return 1;
}


static int os_listcore (lua_State *L) {
  DIR *dir;
  struct dirent *entry;
  int i, isdir, isfile, islink, nops, push, hasfa;
  const char *path, *option, *fn, *pattern;
  char *tbf;  /* to be FREED after string has been pushed on the stack */
#ifndef _WIN32
  struct stat o;
  mode_t mode;
#else
  DWORD mode;
#endif
  isdir = isfile = islink = hasfa = 0;
  pattern = NULL;  /* 1.9.4 */
  if (lua_isnoneornil(L, 1)) {  /* no arguments ?  Agena 1.6.0 Valgrind */
    char *buffer;
    buffer = (char *)malloc((PATH_MAX+1)*sizeof(char));
    if (buffer == NULL)  /* Agena 1.6.0 Valgrind */
      luaL_error(L, "Error in " LUA_QS ": memory allocation error.", "os.listcore");
    if (tools_cwd(buffer) != 0)  /* tools_cwd frees buffer in case of errors */
      luaL_error(L, "Error in " LUA_QS ": memory allocation error.", "os.listcore");
    lua_pushstring(L, buffer);  /* create a duplicate */
    path = agn_checkstring(L, -1);
    agn_poptop(L);  /* pop buffer */
    xfree(buffer);  /* Agena 1.6.0 Valgrind */
  } else
    path = agn_checkstring(L, 1);
  path = (const char *)charreplace((char *)path, '\\', '/', 1);
  nops = lua_gettop(L);
  for (i=2; i <= nops; i++) {
    option = luaL_checkstring(L, i);
    if (strcmp(option, "files") == 0) { isfile = 1; hasfa = 1; }
    else if (strcmp(option, "dirs") == 0) { isdir = 1; hasfa = 1; }
    else if (strcmp(option, "links") == 0) { islink = 1; hasfa = 1; }
    else pattern = option;
  }
  dir = opendir(path);
  if (dir == NULL) {
    lua_pushnil(L);
    lua_pushstring(L, strerror(errno));
    return 2;
  }
  lua_newtable(L);
  i = 1;
  while ((entry = readdir(dir)) != NULL) {
    push = nops <= 1 || pattern != NULL;
    fn = entry->d_name;
    if (strcmp(fn, ".") == 0 || strcmp(fn, "..") == 0 || (pattern != NULL && glob(pattern, fn) == 0)) continue;  /* 1.9.4 */
    if (hasfa) {  /* do not create `//...`, 0.31.4, tuned 1.9.5 */
      lua_pushstring(L, (strcmp(path, "/") == 0) ? (tbf = concat("/", fn, NULL)) : (tbf = concat(path, "/", fn, NULL)));  /* free memory, 1.10.4 */
      xfree(tbf);
#ifdef _WIN32
      mode = GetFileAttributes(agn_tostring(L, -1));
      if (mode != INVALID_FILE_ATTRIBUTES)
        push = ((mode & FILE_ATTRIBUTE_DIRECTORY) && isdir) ||
               (!(mode & FILE_ATTRIBUTE_DIRECTORY) && isfile) ||
               ((mode & FILE_ATTRIBUTE_REPARSE_POINT) && islink);
      agn_poptop(L);
    }
#else
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__) || defined(__HAIKU__)
      if (lstat(agn_tostring(L, -1), &o) == 0) {  /* 1.12.1 */
#else
      if (stat(agn_tostring(L, -1), &o) == 0) {
#endif
        mode = o.st_mode;
        push = (
               (S_ISDIR(mode) && isdir)
               || (S_ISREG(mode) && isfile)
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__) || defined(__HAIKU__)
               || (S_ISLNK(mode) && islink)
#endif
             );
      }
      agn_poptop(L);
    }
#endif
    if (push) {
      lua_rawsetistring(L, -1, i++, fn);  /* 1.9.5 */
    }
  }
  closedir(dir);
  return 1;
}


#ifdef _WIN32
int issymlink (const char *fn) {
  DWORD mode = GetFileAttributes(fn);
  return (mode & FILE_ATTRIBUTE_REPARSE_POINT) == 1024;
}
#endif


static int os_fstat (lua_State *L) {  /* 0.12.2, October 11, 2008; modified 0.26.0, August 06, 2009; patched 0.31.3 */
  struct stat entry;
  const char *fn;
  mode_t mode;
  struct TM *stm;
  Time64_T seconds;
  char bits[18] = "----------:-----\0";
#ifdef _WIN32
  DWORD winmodet;
#elif defined(__OS2__)
  static FILESTATUS3 fstat, *fs;
  APIRET rc;
#endif
  fn = luaL_checkstring(L, 1);
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__) || defined(__HAIKU__)
  if (lstat(fn, &entry) == 0) {  /* 1.12.1 */
#else
  if (stat(fn, &entry) == 0) {
#endif
    lua_newtable(L);
    mode = entry.st_mode;
    lua_pushstring(L, "mode");
    if (S_ISDIR(mode)) {
      lua_pushstring(L, "DIR");
      bits[0] = 'd';
#if defined(_WIN32) || defined(__OS2__) || defined (LUA_DOS)
      bits[11] = 'd';
#endif
    }
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__) || defined(__HAIKU__)
    else if (S_ISLNK(mode)) {
      lua_pushstring(L, "LINK");
      bits[0] = 'l';
    }
#elif defined(_WIN32)
    else if (issymlink(fn)) {
      lua_pushstring(L, "LINK");
      bits[0] = 'l';
    }
#endif
    else if (S_ISREG(mode))
      lua_pushstring(L, "FILE");
    else if (S_ISCHR(mode))
      lua_pushstring(L, "CHARSPECFILE");
    else if (S_ISBLK(mode))
      lua_pushstring(L, "BLOCKSPECFILE");
    else
      lua_pushstring(L, "OTHER");
    lua_settable(L, -3);
    lua_pushstring(L, "length");
    lua_pushnumber(L, entry.st_size);
    lua_settable(L, -3);
    seconds = entry.st_mtime;
    lua_pushstring(L, "date");
    lua_createtable(L, 6, 0);  /* 6 = number of entries */
    stm = localtime64(&seconds);  /* 1.12.1 */
    if (stm != NULL) {  /* 0.31.3 patch */
      lua_rawsetinumber(L, -1, 1, stm->tm_year+1900);
      lua_rawsetinumber(L, -1, 2, stm->tm_mon+1);
      lua_rawsetinumber(L, -1, 3, stm->tm_mday);
      lua_rawsetinumber(L, -1, 4, stm->tm_hour);
      lua_rawsetinumber(L, -1, 5, stm->tm_min);
      lua_rawsetinumber(L, -1, 6, stm->tm_sec);
      lua_settable(L, -3);
    }
    /* 0.25.1 file attributes additions */
    /* determine owner rights */
    lua_pushstring(L, "user");
    lua_createtable(L, 0, 3);
    setboolean(L, "read", mode & S_IRUSR);
    setboolean(L, "write", mode & S_IWUSR);
    setboolean(L, "execute", mode & S_IXUSR);
    if (mode & S_IRUSR) bits[1] = 'r';
    if (mode & S_IWUSR) bits[2] = 'w';
    if (mode & S_IXUSR) bits[3] = 'x';
#ifdef __OS2__
    rc = DosQueryPathInfo((PCSZ)fn, FIL_STANDARD, &fstat, sizeof(fstat));
    if (rc == 0) {
      fs = &fstat;  /* do not change this, otherwise Agena will crash */
      unsigned int attribs = fs->attrFile;
      setboolean(L, "hidden", attribs & FILE_HIDDEN);
      setboolean(L, "readonly", attribs & FILE_READONLY);
      setboolean(L, "system", attribs & FILE_SYSTEM);
      setboolean(L, "archived", attribs & FILE_ARCHIVED);
      setboolean(L, "directory", attribs & FILE_DIRECTORY);
      if (attribs & FILE_DIRECTORY) bits[11] = 'd';
      if (attribs & FILE_READONLY) bits[12] = 'r';
      if (attribs & FILE_HIDDEN) bits[13] = 'h';
      if (attribs & FILE_ARCHIVED) bits[14] = 'a';
      if (attribs & FILE_SYSTEM) bits[15] = 's';
    }
#elif defined(_WIN32)
    winmodet = GetFileAttributes(fn);
    setboolean(L, "hidden", winmodet & FILE_ATTRIBUTE_HIDDEN);
    setboolean(L, "readonly", winmodet & FILE_ATTRIBUTE_READONLY);
    setboolean(L, "system", winmodet & FILE_ATTRIBUTE_SYSTEM);
    setboolean(L, "archived", winmodet & FILE_ATTRIBUTE_ARCHIVE);
    if (winmodet & FILE_ATTRIBUTE_READONLY) bits[12] = 'r';
    if (winmodet & FILE_ATTRIBUTE_HIDDEN) bits[13] = 'h';
    if (winmodet & FILE_ATTRIBUTE_ARCHIVE) bits[14] = 'a';
    if (winmodet & FILE_ATTRIBUTE_SYSTEM) bits[15] = 's';
#endif
    lua_settable(L, -3);
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__) || defined(__HAIKU__)
    /* determine group rights */
    lua_pushstring(L, "group");
    lua_createtable(L, 0, 3);
    setboolean(L, "read", mode & S_IRGRP);
    setboolean(L, "write", mode & S_IWGRP);
    setboolean(L, "execute", mode & S_IXGRP);
    lua_settable(L, -3);
    if (mode & S_IRGRP) bits[4] = 'r';
    if (mode & S_IWGRP) bits[5] = 'w';
    if (mode & S_IXGRP) bits[6] = 'x';
    /* determine other users' rights */
    lua_pushstring(L, "other");
    lua_createtable(L, 0, 3);
    setboolean(L, "read", mode & S_IROTH);
    setboolean(L, "write", mode & S_IWOTH);
    setboolean(L, "execute", mode & S_IXOTH);
    lua_settable(L, -3);
    if (mode & S_IROTH) bits[7] = 'r';
    if (mode & S_IWOTH) bits[8] = 'w';
    if (mode & S_IXOTH) bits[9] = 'x';
#endif
    /* 0.26.0, push file mode (mode_t) and permission `bits` */
    setnumber(L, "perms", mode);
    setstring(L, "bits", bits);
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__)
    setnumber(L, "blocks", entry.st_blocks);
    setnumber(L, "blocksize", entry.st_blksize);
    setnumber(L, "inode", entry.st_ino);
#endif
    setnumber(L, "device", entry.st_dev);
  } else
    lua_pushfail(L);  /* file does not exist */
  return 1;
}


static int os_exists (lua_State *L) {
  lua_pushboolean(L, (access(luaL_checkstring(L, 1), 00|04) != -1));  /* 00 = file exists, 04 = is readable */
  return 1;
}


/* The following routines were taken from the lposix.c file in the
*  POSIX library for Lua 5.0. Based on original by Claudio Terra for Lua 3.x.
*  Luiz Henrique de Figueiredo <lhf@tecgraf.puc-rio.br>
*  05 Nov 2003 22:09:10
*/

/** mkdir(path), modified for Agena; extended in 0.13.3 to support OS/2 */
static int os_mkdir (lua_State *L) {
#ifdef _WIN32
  const char *path = luaL_checkstring(L, 1);
  return os_pushresult(L, mkdir(path), path, "os.mkdir");
#elif defined(__unix__) || defined(__OS2__) || defined(__APPLE__) || defined(__HAIKU__)
  const char *path = luaL_checkstring(L, 1);
  return os_pushresult(L, mkdir(path, 0777), path, "os.mkdir");
#else
  lua_pushfail(L);
  return 1;
#endif
}


/* chdir(path), includes ex-os.curdir code:
   print working directory; July 07, 2008; modified 0.26.0, 04.08.2009 (PATH_MAX),
   extended Agena 1.1, tuned 1.6.0 */
static int os_chdir (lua_State *L) {
  if (agn_isstring(L, 1)) {
    const char *path = agn_tostring(L, 1);  /* Agena 1.6.0 */
    return os_pushresult(L, chdir(path), path, "os.chdir");
  } else if (lua_isnil(L, 1) || lua_gettop(L) == 0) {  /* Agena 1.1.0 */
    char *buffer = (char *)malloc(sizeof(char)*PATH_MAX);
    if (buffer == NULL)  /* Agena 1.0.4 */
      luaL_error(L, "Error in " LUA_QS ": memory allocation error.", "os.chdir");
    if (getcwd(buffer, PATH_MAX) == NULL) { /* 1.6.4 to avoid compiler warnings and unfreed memory */
      xfree(buffer);
      luaL_error(L, "Error in " LUA_QS ": internal error.", "os.chdir");  /* Agena 1.2.0 */
    }
    else {
      charreplace(buffer, '\\', '/', 1);
      lua_pushstring(L, buffer);
    }
    xfree(buffer);
  } else
    luaL_error(L, "Error in " LUA_QS ": expected a string or null.", "os.chdir");
  return 1;
}


/* rmdir(path) */
static int os_rmdir (lua_State *L) {
  const char *path = luaL_checkstring(L, 1);
  return os_pushresult(L, rmdir(path), path, "os.rmdir");
}


/* beep: sound the loudspeaker with a given frequency (first arg, a posint) and
   a duration (second arg, in seconds, a float), April 01, 2007 */

static int os_beep (lua_State *L) {
  int nargs;
#if defined(__unix__) || defined(__APPLE__) || defined(__HAIKU__)
  int i;
#endif
  lua_Number duration;
  nargs = lua_gettop(L);
  duration = 1;
#ifdef _WIN32
  if (nargs == 0) {
    putc('\007', stderr);
    lua_pushnil(L);
  } else {
    int tone;
    tone = agn_checknumber(L, 1);
    duration = agn_checknumber(L, 2);
    if (tone > 0 && duration > 0) {
      Beep(tone, duration*1000);
      lua_pushnil(L);
    } else
      lua_pushfail(L);
  }
#elif defined(__OS2__)
  if (nargs == 0) {
    putc('\007', stderr);
    lua_pushnil(L);
  } else {
    ULONG tone;
    tone = agn_checknumber(L, 1);
    duration = agn_checknumber(L, 2);
    if (tone > 0 && duration > 0) {
      if (DosBeep(tone, (ULONG)(duration*1000)))  /* invalid frequency */
        lua_pushfail(L);
      else
        lua_pushnil(L);
    } else
      lua_pushfail(L);
  }
#elif defined(__unix__) || defined(__APPLE__) || defined(__HAIKU__)
  duration = 1;
  if (nargs == 2) {
    duration = (int)agn_checknumber(L, 2);
  }
  for (i=0; i < duration; i++)
    putc('\007', stderr);
  lua_pushnil(L);
#else
  lua_pushfail(L);
#endif
  return 1;
}


/* wait: wait for x seconds; March 31, 2007 */

static int os_wait (lua_State *L) {
  lua_Number x = agn_checknumber(L, 1);
  if (x > 0) {
    #ifdef _WIN32
    Sleep(x*1000);
    lua_pushnil(L);
    #elif defined(__unix__) || defined(__OS2__) || defined(__APPLE__) || defined(__HAIKU__)
    sleep(x);
    lua_pushnil(L);
    #else
    lua_pushfail(L);
    #endif
  } else lua_pushfail(L);
  return 1;
}


/* retrieve information on your platform;
0.12.2, October 12, 2008; extended to correctly identify Vista, Windows 2003, and Windows 2007 Server
on April 08, 2009 - 0.13.4; updated 1.7.9, 07.09.2012. */

#ifdef _WIN32

typedef struct WinVer {
  int BuildNumber;
  int MajorVersion;
  int MinorVersion;
  int PlatformId;
  char ProductType;
} WinVer;


int getWindowsVersion (struct WinVer *winver) {
   int result, MajorVersion, MinorVersion, ProductType, PlatformId;
   OSVERSIONINFOEX version;
   version.dwOSVersionInfoSize = sizeof(version);
   GetVersionEx((OSVERSIONINFO *)&version);

   /* dwMajorVersion
   Major version number of the operating system. This member can be one of the following values. Value Meaning
   4 Windows NT 4.0, Windows Me, Windows 98, or Windows 95.
   5 Windows Server 2003, Windows XP, or Windows 2000
   6 Windows Vista, Windows Server 2008, Windows 7

   dwMinorVersion
   Minor version number of the operating system. This member can be one of the following values. Value Meaning
   0 Windows 2000, Windows NT 4.0, Windows 95, or Windows Vista
   1 Windows XP, Windows 2008 server
   2 Windows Server 2003, Windows 7
   10 Windows 98
   90 Windows Me

   dwPlatformId
   Operating system platform. This member can be one of the following values. Value Meaning
   VER_PLATFORM_WIN32_NT
   2 Windows Server 2003, Windows XP, Windows 2000, or Windows NT.
   VER_PLATFORM_WIN32_WINDOWS
   1 Windows Me, Windows 98, or Windows 95.
   VER_PLATFORM_WIN32s
   0 Win32s on Windows 3.1. */

   winver->BuildNumber = (int)version.dwBuildNumber;
   MajorVersion = winver->MajorVersion = (int)version.dwMajorVersion;
   MinorVersion = winver->MinorVersion = (int)version.dwMinorVersion;
   PlatformId = winver->PlatformId = (int)version.dwPlatformId;
   ProductType = winver->ProductType = (char)version.wProductType;
   if (MajorVersion == 4 && MinorVersion == 0 && PlatformId == 1)
      result = 2;  /* Win 95 */
   else if (MajorVersion == 4 && MinorVersion == 0 && PlatformId == 2)
      result = 3;  /* Win NT 4.0 */
   else if (MajorVersion == 4 && MinorVersion == 10)
      result = 4;  /* Win 98 */
   else if (MajorVersion == 4 && MinorVersion == 90)
      result = 5;  /* Win Me, return value changed 1.9.3 */
   else if (MajorVersion == 5 && MinorVersion == 0)
      result = 6;  /* Win 2000, return value changed 1.9.3  */
   else if (MajorVersion == 5 && MinorVersion == 1)
      result = 7;  /* Win XP */
   else if (MajorVersion == 6 && MinorVersion == 0 && ProductType == VER_NT_WORKSTATION)
      result = 8;  /* Win Vista */
   else if (MajorVersion == 6 && MinorVersion >= 0 && ProductType != VER_NT_WORKSTATION)
      result = 9;  /* Win Server 2008 */
   else if (MajorVersion == 5 && MinorVersion == 2)
      result = 10; /* Win 2003 */
   else if (MajorVersion == 6 && MinorVersion == 1 && ProductType == VER_NT_WORKSTATION)
      result = 11; /* Windows 7 */
   else if (MajorVersion == 6 && MinorVersion == 2 && ProductType == VER_NT_WORKSTATION)
      result = 12; /* Windows 8 */
   else if (MajorVersion == 6 && MinorVersion == 2 && ProductType != VER_NT_WORKSTATION)
      result = 13; /* Windows Server 2012 */
   else if (PlatformId == 0)
      result = 1;  /* Win32s under Windows 3.1 */
   else
      result = 0;
   return result;
}
#endif


/* Novell DOS 7 = 'IBMPcDos', '6', '00', 'pc'
   OS/2 Warp 4 Fixpack 15 = 'OS/2', '1', '2.45', 'i386'
   Windows = 'Windows', etc. */

static int os_system (lua_State *L) {
#ifdef _WIN32
   /* dwMajorVersion
   Major version number of the operating system. This member can be one of the following values. Value Meaning
   4 Windows NT 4.0, Windows Me, Windows 98, or Windows 95.
   5 Windows Server 2003, Windows XP, or Windows 2000
   6 Windows Vista, Windows Server 2008, Windows 7

   dwMinorVersion
   Minor version number of the operating system. This member can be one of the following values. Value Meaning
   0 Windows 2000, Windows NT 4.0, Windows 95, or Windows Vista
   1 Windows XP, Windows 2008 server
   2 Windows Server 2003, Windows 7
   10 Windows 98
   90 Windows Me

   dwPlatformId
   Operating system platform. This member can be one of the following values. Value Meaning
   VER_PLATFORM_WIN32_NT
   2 Windows Server 2003, Windows XP, Windows 2000, or Windows NT.
   VER_PLATFORM_WIN32_WINDOWS
   1 Windows Me, Windows 98, or Windows 95.
   VER_PLATFORM_WIN32s
   0 Win32s on Windows 3.1. */

  int winver;
  struct WinVer winversion;
  static const char *versions[] = {  /* extended 0.13.4, switched "ME" and "2000" in 1.9.3 */
    "unknown", "32s", "95", "NT 4.0", "98", "ME", "2000", "XP", "Vista", "Server 2008", "Server 2003", "7", "8", "Server 2012"
    /* if you add future Windows releases, extend getWindowsVersion above, as well */
  };
  winver = getWindowsVersion(&winversion);
  lua_newtable(L);
  lua_rawsetistring(L, -1, 1, "Windows");
  lua_rawsetistring(L, -1, 2, versions[winver]);
  lua_rawsetinumber(L, -1, 3, winversion.BuildNumber);
  lua_rawsetinumber(L, -1, 4, winversion.PlatformId);
  lua_rawsetinumber(L, -1, 5, winversion.MajorVersion);  /* added 0.13.4 */
  lua_rawsetinumber(L, -1, 6, winversion.MinorVersion);  /* added 0.13.4 */
  lua_rawsetinumber(L, -1, 7, winversion.ProductType);   /* added 0.13.4 */
#elif defined(__unix__) || defined(__OS2__) || defined(__APPLE__) || defined(__HAIKU__)
  /* applies to DJGPP compiled DOS code as well; on Apple returns Darwin
  version number */
  struct utsname info;
  if (uname(&info) != -1) {
    lua_newtable(L);
    lua_rawsetistring(L, -1, 1, info.sysname);
    lua_rawsetistring(L, -1, 2, info.release);
    lua_rawsetistring(L, -1, 3, info.version);
    lua_rawsetistring(L, -1, 4, info.machine);
   }
   else
     lua_pushfail(L);
#else
  lua_pushfail(L);
#endif
  return 1;
}


static int os_endian (lua_State *L) {
  int r = tools_endian();
  if (r == -1)
    lua_pushfail(L);
  else
    lua_pushinteger(L, r);
  return 1;
}


static int os_isANSI (lua_State *L) {  /* 0.28.2 */
#ifdef LUA_ANSI
  lua_pushtrue(L);
#else
  lua_pushfalse(L);
#endif
  return 1;
}


/* return status of battery on laptops, works in Windows 2000 and higher. 0.14.0, March 13, 2009 */
static int os_battery (lua_State *L) {
#ifdef _WIN32
  BYTE bACLineStatus, status, bBatteryLifePercent, bReserved1;
  DWORD bBatteryLifeTime, bBatteryFullLifeTime;
  SYSTEM_POWER_STATUS battstat;
  int BatteryCharging, winver;
  struct WinVer windowsver;
  static const char *ACstatus[] = {"off", "on", "unknown"};
  static const char *BatteryStatusMapping[] =
    {"medium", "high", "low", "critical", "charging", "no battery", "unknown"};
  winver = getWindowsVersion(&windowsver);
  if (winver < 2 || winver == 3) {  /* Windows NT 4.0 and earlier are not supported */
    lua_pushfail(L);
    return 1;
  }
  /* get battery information by calling the Win32 API */
  GetSystemPowerStatus(&battstat);
  bReserved1 = battstat.Reserved1;
  bACLineStatus = battstat.ACLineStatus;
  BatteryCharging = 0; /* not charging */
  if (bACLineStatus == 255)
    bACLineStatus = 2;
  switch (battstat.BatteryFlag) {
    case 0: /* >= 33 and <= 66 % */
      status = 0;
      break;
    case 1: /* high */
      status = 1;
      break;
    case 2: /* low */
      status = 2;
      break;
    case 4: /* critical */
      status = 3;
      break;
    case 8: case 9: case 10:  /* charging */
      status = battstat.BatteryFlag - 8;
      BatteryCharging = 1;
      break;
    case 12:  /* charging & critical */
      status = 3;
      BatteryCharging = 1;
    case 128: /* No system battery */
      status = 5;
      break;
    case 255: /* unknown status */
      status = 6;
      break;
    default:  /* got value 0 at around 45% on Acer, 10 right after reconnecting AC line  */
      status = 6;
      break;
  }
  bBatteryLifePercent = battstat.BatteryLifePercent;
  bBatteryLifeTime = battstat.BatteryLifeTime;
  bBatteryFullLifeTime = battstat.BatteryFullLifeTime;
  lua_createtable(L, 0, 0);
  setstring(L, "acline", ACstatus[bACLineStatus]);
  if (status == 4) {
    setboolean(L, "installed", 0);
  } else {
    setboolean(L, "installed", 1);
    setnumber(L, "life", bBatteryLifePercent);
    setstring(L, "status", BatteryStatusMapping[status]);
    setboolean(L, "charging", (BatteryCharging == 1));  /* 0.31.4 fix */
    setnumber(L, "flag", battstat.BatteryFlag);
    lua_pushstring(L, "lifetime");
    if (bBatteryLifeTime != -1)
      lua_pushnumber(L, bBatteryLifeTime);
    else
      lua_pushundefined(L);
    lua_settable(L, -3);
    lua_pushstring(L, "fulllifetime");
    if (bBatteryFullLifeTime != -1)
      lua_pushnumber(L, bBatteryFullLifeTime);
    else
      lua_pushundefined(L);
    lua_settable(L, -3);
  }
#elif __OS2__
#define IOCTL_POWER 0x000c
#define POWER_GETPOWERSTATUS 0x0060
  int result;
  HFILE pAPI;
  ULONG call;
  struct _Battery {
    USHORT usParamLen;
    USHORT usPowerFlags;
    UCHAR  ucACStatus;
    UCHAR  ucBatteryStatus;
    UCHAR  ucBatteryLife;
  } Battery;
  struct _Data {
    USHORT usReturn;
  } Data;
  ULONG sizeBattery, sizeData;
  result = DosOpen((PCSZ)"\\DEV\\APM$", &pAPI, &call, 0,
    FILE_NORMAL, FILE_OPEN, OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE,
    NULL);
  if (result != 0) {
    lua_pushfail(L);
    return 1;
  }
  sizeBattery = Battery.usParamLen = sizeof(Battery);
  sizeData = sizeof(Data);
  result = DosDevIOCtl(pAPI,
    IOCTL_POWER, POWER_GETPOWERSTATUS, (PVOID)&Battery, sizeof(Battery),
    &sizeBattery, (PVOID)&Data, sizeof(Data), &sizeData);
  if (result == ERROR_INVALID_PARAMETER) {
    lua_pushfail(L);
  } else {
    int acstatus, batterystatus;
    static const char *ACstatus[] = {"off", "on", "unknown", "invalid"};
    static const char *BatteryStatusMapping[] =
      {"high", "low", "critical", "charging", "unknown", "invalid"};
    lua_newtable(L);
    switch (Battery.ucACStatus) {
      case   0: acstatus = 0; break;
      case   1: acstatus = 1; break;
      case 255: acstatus = 2; break;
      default:  acstatus = 3; break;
    }
    switch (Battery.ucBatteryStatus) {
      case 0:   batterystatus = 0; break;
      case 1:   batterystatus = 1; break;
      case 2:   batterystatus = 2; break;
      case 3:   batterystatus = 3; break;
      case 255: batterystatus = 4; break;
      default:  batterystatus = 5; break;
    }
    setstring(L, "acline", ACstatus[acstatus]);
    setstring(L, "status", BatteryStatusMapping[batterystatus]);
    setnumber(L, "flags", (lua_Number)Battery.usPowerFlags);
    setboolean(L, "powermanagement", ((Battery.usPowerFlags & 0x0001) == 1));
    lua_pushstring(L, "life");
    if (Battery.ucACStatus != 1)  /* AC line is off */
      lua_pushnumber(L, (lua_Number)Battery.ucBatteryLife);
    else
      lua_pushundefined(L);
    lua_rawset(L, -3);
  }
#else
  lua_pushfail(L);
#endif
  return 1;
}


/* 0.25.0, July 19, 2009; extended 26.07.2009, 0.25.1; patched 0.26.0, August 06, 2009 */
static int os_fcopy (lua_State *L) {
  const char *fin, *fout;
  int in, out, bufsize, result, nr;
  size_t bytes;
  mode_t mode;
  struct stat entry;
  result = 0;
  fin = agn_checkstring(L, 1);
  fout = agn_checkstring(L, 2);
  if (strcmp(fin, fout) == 0) {  /* 1.8.0, prevent destroying the file to be copied if source = target */
    lua_pushfail(L);
    lua_pushstring(L, fin);
    return 2;
  }
  agnL_gettablefield(L, "environ", "buffersize", "os.fcopy", 1);  /* 0.32.0 & 1.6.4 */
  bufsize = ( lua_type(L, -1) == LUA_TNUMBER ) ? agn_tonumber(L, -1) : LUAL_BUFFERSIZE;
  if (bufsize < 1) bufsize = LUAL_BUFFERSIZE;
  agn_poptop(L);  /* pop "buffersize" */
  char buffer[bufsize];
  if ( (in = my_roopen(fin)) == -1) {
    lua_pushfail(L);
    lua_pushstring(L, fin);  /* 1.7.6 */
    return 2;
  }
  if ( (out = my_create(fout)) == -1) {
    close(in);
    lua_pushfail(L);
    lua_pushstring(L, fin);  /* 1.7.6 */
    return 2;
  }
  while ( (bytes = read(in, buffer, bufsize)) ) {
    if (bytes == -1) {  /* 1.6.4 */
      luaL_error(L, "Error in " LUA_QS ": could not read from file #%d.", "os.fcopy", in);
    }
    if (write(out, buffer, bytes) == -1) {  /* 1.6.4 to prevent compiler warnings */
      luaL_error(L, "Error in " LUA_QS ": could not write to file #%d.", "os.fcopy", out);
    }
  }
  close(in);
  close(out);
  nr = 1;
  if (bytes < 0) {
    lua_pushfail(L); lua_pushstring(L, fin); nr = 2;  /* 1.7.6 */
  } else {
    if (stat(fin, &entry) == 0) {
      /* set file permission attributes */
      mode = entry.st_mode;
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__) || defined(__HAIKU__)
      result = chmod(fout, (mode & S_IRUSR)|(mode & S_IWUSR)|(mode & S_IXUSR)|(mode & S_IRGRP)|(mode & S_IWGRP)|(mode & S_IXGRP)|(mode & S_IROTH)|(mode & S_IWOTH)|(mode & S_IXOTH));
#elif defined(_WIN32)
      /* added 0.26.0, 07.08.2009 */
      result = !(SetFileAttributes(fout, GetFileAttributes(fin)));
#elif defined(__OS2__)
      static FILESTATUS3 fstat, *fs;
      APIRET rc = DosQueryPathInfo((PCSZ)fin, FIL_STANDARD, &fstat, sizeof(fstat));
      if (rc == 0) {
        fs = &fstat;  /* do not change this, otherwise Agena will crash */
        DosSetPathInfo((PCSZ)fout, FIL_STANDARD, fs, sizeof(FILESTATUS3), 0);
      }
#else
      result = chmod(fout, (mode & S_IRUSR)|(mode & S_IWUSR)|(mode & S_IXUSR));
#endif
#if (defined(_WIN32) || defined(__unix__) || defined(__APPLE__) || defined(LUA_DOS))
      /* file date and time, Agena 1.8.0 */
      struct utimbuf times;
      times.modtime = times.actime = entry.st_mtime;
      result = result || utime(fout, &times);
#endif
      if (result == 0)
        lua_pushtrue(L);
      else {
        lua_pushfail(L); lua_pushstring(L, fin); nr = 2;  /* 1.7.6 */
      }
    }
    else {
      lua_pushfail(L); lua_pushstring(L, fin); nr = 2;  /* 1.7.6 */
    }
  }
  return nr;
}


static int os_drives (lua_State *L) {
#if defined(_WIN32) || defined(__OS2__)
  unsigned int i, c;
  int anydrivefound;
  char drive[1];
  #ifdef _WIN32
  long buff, mask;
  buff = GetLogicalDrives();
  #else
  ULONG buff, mask, curDrive;
  APIRET rc;
  rc = DosQueryCurrentDisk(&curDrive, &buff);
  if (rc != 0)
    luaL_error(L, "Error in " LUA_QS ": could not determine drives.", "os.drives");
  #endif
  anydrivefound = c = i = 0;
  agn_createseq(L, 26);
  while( i < 26 ) {
    mask = (1 << i);
    if( (buff & mask) > 0 ) {
      drive[0] = 65+i;
      lua_pushlstring(L, drive, 1);
      lua_seqseti(L, -2, ++c);
      /* prevent error messages if no media are inserted in floppy drives or CD-ROMs */
      anydrivefound = 1;
    }
    i++;
  }
  if (!anydrivefound) {
    agn_poptop(L);  /* delete sequence */
    lua_pushfail(L);
  }
#else
  lua_pushfail(L);
#endif
  return 1;
}


static int os_drivestat (lua_State *L) {
#ifdef _WIN32
  unsigned int drivetype;
  size_t l;
  long buff, mask;
  char drive[4];
  DWORD serial_number;
  DWORD max_component_length;
  DWORD file_system_flags;
  char name [256];
  char file_system_name[256];
  unsigned long sectors_per_cluster, bytes_per_sector, number_of_free_clusters, total_number_of_clusters;
  double freespace, totalspace;
  static const char *types[] = {
    "unknown", "no root dir", "Removable", "Fixed", "Remote", "CD-ROM", "RAMDISK"};
  const char *drivename = agn_checklstring(L, 1, &l);
  if (l < 1) {  /* empty string ? */
    lua_pushfail(L);
    return 1;
  }
  drive[0] = toupper(*drivename); drive[1] = ':'; drive[2] = '\\'; drive[3] = '\0';
  buff = GetLogicalDrives();
  mask = (1 << (drive[0]-65));
  if( (buff & mask) < 1 ) {  /* no valid drive ? */
    lua_pushfail(L);
    return 1;
  }
  /* set defaults to prevent invalid values when calling GetDiskFreeSpace */
  sectors_per_cluster = 0, bytes_per_sector = 0, number_of_free_clusters = 0, total_number_of_clusters = 0;
  /* set defaults to prevent invalid values when calling GetVolumeInformation  */
  strcpy(name, ""); strcpy(file_system_name, "");
  serial_number = 0; max_component_length = 0; file_system_flags = 0;
  drivetype = GetDriveType(drive);
  lua_createtable(L, 0, 5);
  GetVolumeInformation(drive, name,
    (sizeof (name)), &serial_number, &max_component_length, &file_system_flags,
    file_system_name, (sizeof (file_system_name)));
  if (name == NULL || strcmp(name, "") == 0)
    strcpy(name, "<none>");
  if (strlen(name) > 19) name[19] = '\0';
  if (file_system_name == NULL || strcmp(file_system_name, "") == 0)
    strcpy(file_system_name, "<unknown>");
  if (strlen(file_system_name) > 11) file_system_name[11] = '\0';
  /* prevent error messages if no media are inserted in floppy drives or CD-ROMs */
  SetErrorMode(0x0001);
  /* get info on free space on drive processed */
  GetDiskFreeSpace(drive, &sectors_per_cluster, &bytes_per_sector,
    &number_of_free_clusters, &total_number_of_clusters);
  SetErrorMode(0); /* set error mode back to normal (any exceptions printed again) */
  freespace = (double)number_of_free_clusters * (double)sectors_per_cluster * (double)bytes_per_sector;
  totalspace = (double)total_number_of_clusters * (double)sectors_per_cluster * (double)bytes_per_sector;
  setstring(L, "label", name);
  setstring(L, "drivetype", (drivetype < 7) ? types[drivetype] : "unknown");
  setstring(L, "filesystem",
    (strcmp(file_system_name, "") == 0) ? "<unknown>" : file_system_name);
  setnumber(L, "freesize", freespace);
  setnumber(L, "totalsize", totalspace);
#else
  lua_pushfail(L);
#endif
  return 1;
}


static mode_t newmode (mode_t oldmode, mode_t modeflag, int add) {
  mode_t modet = oldmode;
  if (add)
    modet |= modeflag;
  else if (modet & modeflag)
    modet -= modeflag;
  return modet;
}

#ifdef _WIN32
static DWORD winmode (DWORD oldmode, DWORD modeflag, int add) {
  DWORD winmodet = oldmode;
  if (add)
    winmodet |= modeflag;
  else if (winmodet & modeflag)
    winmodet -= modeflag;
  return winmodet;
}
#elif defined(__OS2__)
static unsigned int os2mode (unsigned int oldmode, unsigned int modeflag, int add) {
  unsigned int os2modet = oldmode;
  if (add)
    os2modet |= modeflag;
  else if (os2modet & modeflag)
    os2modet -= modeflag;
  return os2modet;
}
#endif


static Time64_T maketime (int year, int month, int day, int hour, int minute, int second) {  /* Agena 1.8.0 */
  struct TM time_str;
  if (tools_checkdatetime(year, month, day, hour, minute, second) == 0) return -2;  /* 1.9.1 */
  time_str.tm_year = year - 1900;
  time_str.tm_mon = month - 1;
  time_str.tm_mday = day;
  time_str.tm_hour = hour;
  time_str.tm_min = minute;
  time_str.tm_sec = second;
  time_str.tm_isdst = -1;
  return mktime64(&time_str);
}

static int os_fattrib (lua_State *L) {  /* 0.26.0 */
  const char *fn, *mode;
  mode_t modet;
  struct stat entry;
  int result, owner, add;
  size_t l, i;
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__) || defined(__HAIKU__)
  size_t j;
#elif defined(_WIN32)
  DWORD winmodet;
#elif defined(__OS2__)
  static FILESTATUS3 fstat, *fs;
  unsigned int os2modet;
  APIRET rc;
#endif
  fn = agn_checkstring(L, 1);
  if (lua_istable(L, 2)) {  /* change file access and modification date */
#if (defined(_WIN32) || defined(__unix__) || defined(__APPLE__) || defined(LUA_DOS))
    Time64_T t;
    struct utimbuf times;
    int year, month, day, hour, minute, second, nops;
    year = agn_getinumber(L, 2, 1);
    month = agn_getinumber(L, 2, 2);
    day = agn_getinumber(L, 2, 3);
    nops = agn_size(L, 2);
    hour = (nops > 3) ? agn_getinumber(L, 2, 4) : 0;
    minute = (nops > 4) ? agn_getinumber(L, 2, 5) : 0;
    second = (nops > 5) ? agn_getinumber(L, 2, 6) : 0;
    if (tools_checkdatetime(year, month, day, hour, minute, second) == 0)  /* 1.9.1 */
      luaL_error(L, "Error in " LUA_QS ": time component out of range.", "os.fattrib");
    if (year > 2037)  /* XXX */
      luaL_error(L, "Error in " LUA_QS ": year beyond 2037.", "os.fattrib");
    t = maketime(year, month, day, hour, minute, second);
    if (t != 0)
      luaL_error(L, "Error in " LUA_QS ": could not determine time.", "os.fattrib");
    times.modtime = times.actime = t;
    if (utime(fn, &times) == -1)
      lua_pushfail(L);
    else
      lua_pushtrue(L);
#else
    lua_pushfail(L);
#endif
    return 1;
  }
  modet = owner = add = 0;
#if defined(_WIN32)
  winmodet = GetFileAttributes(fn);
#elif defined(__OS2__)
  rc = DosQueryPathInfo((PCSZ)fn, FIL_STANDARD, &fstat, sizeof(fstat));
  if (rc != 0)
    luaL_error(L, "Error in " LUA_QS ": could not determine file attributes.", "os.fattrib");
  fs = &fstat;
  os2modet = fs->attrFile;
#endif
  if (stat(fn, &entry) == 0)
    modet = entry.st_mode;
  else
    luaL_error(L, "Error in " LUA_QS ": file does not exist.", "os.fattrib");
  mode = agn_checklstring(L, 2, &l);
  if (l < 3)
    luaL_error(L, "Error in " LUA_QS ": malformed settings string.", "os.fattrib");
#if !defined(__unix__) && !defined(__APPLE__) && !defined(__HAIKU__)
  mode_t modew[1] = {S_IWUSR};
  mode_t modex[1] = {S_IXUSR};
#else
#ifndef LUA_DOS
  mode_t moder[3] = {S_IRUSR, S_IRGRP, S_IROTH};
#else
  mode_t moder[1] = {S_IRUSR};
#endif
  mode_t modew[3] = {S_IWUSR, S_IWGRP, S_IWOTH};
  mode_t modex[3] = {S_IXUSR, S_IXGRP, S_IXOTH};
#endif
  switch (*mode++) {
    case 'u': case 'U': {
      owner = 0;  /* owner */
      break;
    }
    case 'g': case 'G': {
      owner = 1;  /* group */
      break;
    }
    case 'o': case 'O': {
      owner = 2;  /* other */
      break;
    }
    case 'a': case 'A': {
      owner = 3;  /* all */
      break;
    }
    default:
      luaL_error(L,
        "Error in " LUA_QS ": wrong user specification, must be `u` , `g`, `o`, or `a`.",
        "os.fattrib");
  }
#if ((!defined(__unix__) && !defined(__APPLE__) &&  !defined(__HAIKU__)) || defined(LUA_DOS))
  if (owner > 0) owner = 0;
#endif
  switch (*mode++) {
    case '+': {
      add = 1;  /* owner */
      break;
    }
    case '-': {
      add = 0;  /* group */
      break;
    }
    default:
      luaL_error(L,
        "Error in " LUA_QS ": wrong add/delete specification, must be `+` , or `-`.",
        "os.fattrib");
  }
  for (i=2; i < l; i++, mode++) {
    switch(*mode) {
#if defined(__unix__) || defined(__APPLE__) || defined(__HAIKU__)
      case 'r': case 'R': {
        if (owner < 3)
          modet = newmode(modet, moder[owner], add);
#ifndef LUA_DOS
        else
          for (j=0; j < 3; j++) {modet = newmode(modet, moder[j], add);}
#endif
        break;
      }
#endif
      case 'w': case 'W': {
        if (owner < 3)
          modet = newmode(modet, modew[owner], add);
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__) || defined(__HAIKU__)
        else
          for (j=0; j < 3; j++) {modet = newmode(modet, modew[j], add);}
#endif
        break;
      }
      case 'x': case 'X': {
        if (owner < 3)
          modet = newmode(modet, modex[owner], add);
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__) || defined(__HAIKU__)
        else
          for (j=0; j < 3; j++) {modet = newmode(modet, modex[j], add);}
#endif
        break;
      }
#ifdef _WIN32
      case 'h': case 'H': {
        winmodet = winmode(winmodet, FILE_ATTRIBUTE_HIDDEN, add);
        break;
      }
      case 'a': case 'A': {
        winmodet = winmode(winmodet, FILE_ATTRIBUTE_ARCHIVE, add);
        break;
      }
      case 's': case 'S': {
        winmodet = winmode(winmodet, FILE_ATTRIBUTE_SYSTEM, add);
        break;
      }
      case 'r': case 'R': {
        winmodet = winmode(winmodet, FILE_ATTRIBUTE_READONLY, add);
        break;
      }
#elif defined(__OS2__)
      case 'h': case 'H': {
        os2modet = os2mode(os2modet, FILE_HIDDEN, add);
        break;
      }
      case 'a': case 'A': {
        os2modet = os2mode(os2modet, FILE_ARCHIVED, add);
        break;
      }
      case 's': case 'S': {
        os2modet = os2mode(os2modet, FILE_SYSTEM, add);
        break;
      }
      case 'r': case 'R': {
        os2modet = os2mode(os2modet, FILE_READONLY, add);
        break;
      }
#endif
      default: lua_assert(0);
    }
  }
  result = chmod(fn, modet);
#ifdef _WIN32
  result = result || !(SetFileAttributes(fn, winmodet));
#elif defined(__OS2__)
  fs->attrFile = os2modet;
  rc = DosSetPathInfo((PCSZ)fn, FIL_STANDARD, fs, sizeof(FILESTATUS3), 0);
  result = result || rc;
#endif
  if (result == 0)
     lua_pushtrue(L);
  else
     lua_pushfail(L);
  return 1;
}


static int os_screensize (lua_State *L) {  /* 0.31.3 */
#ifdef _WIN32
  lua_pushnumber(L, GetSystemMetrics(SM_CXSCREEN));
  lua_pushnumber(L, GetSystemMetrics(SM_CYSCREEN));
  agn_createpair(L, -2, -1);
  lua_remove(L, -2);
  lua_remove(L, -2);
#else
  lua_pushfail(L);
#endif
  return 1;
}


static int os_mousebuttons (lua_State *L) {  /* 0.31.3 */
#ifdef _WIN32
  lua_pushnumber(L, GetSystemMetrics(SM_CMOUSEBUTTONS));
#else
  lua_pushfail(L);
#endif
  return 1;
}


/* os.datetosec, Agena 1.8.0, 21.09.2012

   receives a date of the form [year, month, date [, hour [, minute [, second]]], with a table of all values
   being integers, and transforms it to the number of seconds elapsed since the start of an `epoch`.
   The time zone acknowledged may depend on your operating system.

   See also: os.secstodate. */

static int os_datetosecs (lua_State *L) {  /* Agena 1.8.0, extended 1.12.3 */
  Time64_T t;
  int year, month, day, hour, minute, second;
  size_t nops;
  year = month = day = hour = minute = second = 0;  /* to prevent compiler warnings */
  switch (lua_type(L, 1)) {
    case LUA_TTABLE: {  /* 1.12.3 */
      nops = agn_size(L, 1);
      if (nops < 3)
        luaL_error(L, "Error in " LUA_QS ": table must contain at least three values.", "utils.checkdate");
      year = agn_getinumber(L, 1, 1);
      month = agn_getinumber(L, 1, 2);
      day = agn_getinumber(L, 1, 3);
      hour = (nops > 3) ? agn_getinumber(L, 1, 4) : 0;
      minute = (nops > 4) ? agn_getinumber(L, 1, 5) : 0;
      second = (nops > 5) ? agn_getinumber(L, 1, 6) : 0;
      break;
    }
    case LUA_TSEQ: {  /* 1.12.3 */
      nops = agn_seqsize(L, 1);
      if (nops < 3)
        luaL_error(L, "Error in " LUA_QS ": sequence must contain at least three values.", "utils.checkdate");
      year = lua_seqgetinumber(L, 1, 1);
      month = lua_seqgetinumber(L, 1, 2);
      day = lua_seqgetinumber(L, 1, 3);
      hour = (nops > 3) ? lua_seqgetinumber(L, 1, 4) : 0;
      minute = (nops > 4) ? lua_seqgetinumber(L, 1, 5) : 0;
      second = (nops > 5) ? lua_seqgetinumber(L, 1, 6) : 0;
      break;
    }
    case LUA_TNUMBER: {
      nops = lua_gettop(L);
      if (nops < 3)
        luaL_error(L, "Error in " LUA_QS ": expected at leat three arguments of type number.", "utils.checkdate");
      year = agn_checknumber(L, 1);
      month = agn_checknumber(L, 2);
      day = agn_checknumber(L, 3);
      hour = (nops > 3) ? agn_checknumber(L, 4) : 0;
      minute = (nops > 4) ? agn_checknumber(L, 5) : 0;
      second = (nops > 5) ? agn_checknumber(L, 6) : 0;
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": expected a table, sequence or at least three numbers.", "utils.checkdate");
  }
  if (tools_checkdatetime(year, month, day, hour, minute, second) == 0)  /* 1.9.1 */
    luaL_error(L, "Error in " LUA_QS ": time component out of range.", "os.datetosecs");
  t = maketime(year, month, day, hour, minute, second);
  if (t == -1)
    luaL_error(L, "Error in " LUA_QS ": could not determine time.", "os.datetosecs");
  else
    lua_pushnumber(L, t);
  return 1;
}


/* os.secstodate, Agena 1.8.0, 21.09.2012

   takes the number of seconds elapsed since the start of an epoch, in your local time zone, and returns a
   table of integers in the order: year, month, day, hour, minute, second. In case of an error, `fail` is
   returned.

   See also: os.datetosec. */

static int os_secstodate (lua_State *L) {  /* Agena 1.8.0 */
  Time64_T t;
  struct TM *stm;
  t = agn_checknumber(L, 1);
  stm = localtime64(&t);
  if (stm != NULL) {  /* 0.31.3 patch */
    lua_createtable(L, 6, 0);  /* 6 = number of entries */
    lua_rawsetinumber(L, -1, 1, stm->tm_year+1900);
    lua_rawsetinumber(L, -1, 2, stm->tm_mon+1);
    lua_rawsetinumber(L, -1, 3, stm->tm_mday);
    lua_rawsetinumber(L, -1, 4, stm->tm_hour);
    lua_rawsetinumber(L, -1, 5, stm->tm_min);
    lua_rawsetinumber(L, -1, 6, stm->tm_sec);
  } else
    lua_pushfail(L);
  return 1;
}


/* os.now, Agena 1.8.0, 21.09.2012

   returns rather low-level information on the current or given date and time in form of a dictionary: the `gmt` table
   represents the current date and time in GMT/UTC. The `localtime` table includes the same information for your local
   time zone. The `tz` entry represents the difference between your local time zone and GMT in minutes with daylight
   saving time cancelled out, and _east_ of Greenwich. The `seconds` entry is the number of seconds elapsed since
   some given start time (the `epoch`), which on most operating systems is January 01, 1970, 00:00:00.

   If no argument is passed, the function returns information on the current date and time. If a non-negative number
   is given which represents the amount of seconds elapsed since the start of the epoch, information on this date and time
   are determined (see os.datetosecs to convert a date to seconds).

   The `gmt` and `localtime` entries are of the same structure: it is a table of data of the following order:
   year, month, day, hour, minute, second, number of weekday (where 0 means Sunday, 1 is Monday, and so forth),
   the number of full days since the beginning of the year (in the range 0:365), whether daylight saving time
   is in effect at the time given (0: no, 1: yes), and the strings 'AM' or 'PM'.

   See also: os.datetosecs, os.secstodate, os.time.

*/

#define dia_ae "ae"
#define dia_ao "ao"
#define dia_CZ "cz"
#define dia_ue "ue"

/* New English calendar names */

static char *months[] = {
      "none", "January", "February", "March", "April", "May", "June", "July", "August",
      "September", "October", "November", "December" };

static char *weekdays[] = {"Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"};

static int os_now (lua_State *L) {  /* Agena 1.8.0, extended 1.8.2 */
  Time64_T tlct, tgmt, gmttime, lcttime, gmtday, lctday;
  struct TM *stmgmt, *stmlct;
  struct timeb tp;
  char ampm[3];
  double djm0, djm, djt;
  int msecs;
  if (lua_isnumber(L, 1)) {
    Time64_T t = agn_tonumber(L, 1);
    if (t < 0) luaL_error(L, "Error in " LUA_QS ": time delivered is negative.", "os.now");
    tlct = tgmt = t;
  } else {
    tlct = time(NULL);
    tgmt = time(NULL);
    if (tlct == -1 || tgmt == -1)
      luaL_error(L, "Error in " LUA_QS ": could not determine current time.", "os.now");
  }
  stmgmt = gmtime64(&tgmt);
  strftime(ampm, 3, "%p", stmgmt);
  ampm[2] = '\0';  /* better sure than sorry */
  gmttime = lcttime = gmtday = lctday = 0;
  lua_createtable(L, 0, 8);
  lua_pushstring(L, "gmt");
  if (stmgmt != NULL) {
    lua_createtable(L, 10, 0);
    lua_rawsetinumber(L, -1,  1, stmgmt->tm_year+1900);
    lua_rawsetinumber(L, -1,  2, stmgmt->tm_mon+1);
    lua_rawsetinumber(L, -1,  3, stmgmt->tm_mday);
    gmtday = stmgmt->tm_mday;
    lua_rawsetinumber(L, -1,  4, stmgmt->tm_hour);
    lua_rawsetinumber(L, -1,  5, stmgmt->tm_min);
    lua_rawsetinumber(L, -1,  6, stmgmt->tm_sec);
    lua_rawsetinumber(L, -1,  7, stmgmt->tm_wday);
    lua_rawsetinumber(L, -1,  8, stmgmt->tm_yday);
    lua_rawsetinumber(L, -1,  9, stmgmt->tm_isdst);
    lua_rawsetistring(L, -1, 10, ampm);
    lua_rawsetistring(L, -1, 11, months[stmgmt->tm_mon+1]);
    lua_rawsetistring(L, -1, 12, weekdays[stmgmt->tm_wday]);  /* Agena 1.8.2 fix */
    gmttime = stmgmt->tm_sec + stmgmt->tm_min*60 + stmgmt->tm_hour * 3600;
  } else {
    lua_pushfail(L);
  }
  lua_rawset(L, -3);
  stmlct = localtime64(&tlct);  /* put this _here_, otherwise the time would be overwritten */
  strftime(ampm, 3, "%p", stmlct);
  lua_pushstring(L, "localtime");
  if (stmgmt != NULL) {
    lua_createtable(L, 10, 0);
    lua_rawsetinumber(L, -1,  1, stmlct->tm_year+1900);
    lua_rawsetinumber(L, -1,  2, stmlct->tm_mon+1);
    lua_rawsetinumber(L, -1,  3, stmlct->tm_mday);
    lctday = stmlct->tm_mday;
    lua_rawsetinumber(L, -1,  4, stmlct->tm_hour);
    lua_rawsetinumber(L, -1,  5, stmlct->tm_min);
    lua_rawsetinumber(L, -1,  6, stmlct->tm_sec);
    lua_rawsetinumber(L, -1,  7, stmlct->tm_wday);
    lua_rawsetinumber(L, -1,  8, stmlct->tm_yday);
    lua_rawsetinumber(L, -1,  9, stmlct->tm_isdst);
    lua_rawsetistring(L, -1, 10, ampm);
    lua_rawsetistring(L, -1, 11, months[stmlct->tm_mon+1]);
    lua_rawsetistring(L, -1, 12, weekdays[stmlct->tm_wday]);  /* Agena 1.8.2 fix */
    lcttime = stmlct->tm_sec + stmlct->tm_min*60 + stmlct->tm_hour * 3600;
  } else {
    lua_pushfail(L);
  }
  lua_rawset(L, -3);
  lua_pushstring(L, "tz");
  if (lctday-gmtday == 0)
    lua_pushnumber(L, (lcttime - gmttime - stmlct->tm_isdst * 3600)/60);
  else
    lua_pushnumber(L, (lcttime - gmttime + 86400 - stmlct->tm_isdst * 3600)/60);
  lua_rawset(L, -3);
  lua_pushstring(L, "td");  /* Agena 1.8.2 */
  if (lctday-gmtday == 0)
    lua_pushnumber(L, (lcttime - gmttime)/60);
  else
    lua_pushnumber(L, (lcttime - gmttime + 86400)/60);
  lua_rawset(L, -3);
  /* set daylight saving time flag for local time zone, Agena 1.8.2 */
  lua_pushstring(L, "dst");
  lua_pushboolean(L, stmlct->tm_isdst);
  lua_rawset(L, -3);
  lua_pushstring(L, "seconds");
  lua_pushnumber(L, tgmt);  /* return time in seconds */
  lua_rawset(L, -3);
  msecs = -1;
  /* 1.11.7, get milliseconds */
  #ifdef _WIN32
  ftime(&tp);
  msecs = tp.millitm;
  #elif defined(__unix__) || defined(__APPLE__)
  if (ftime(&tp) == 0)  /* set milliseconds only when query has been successful */
    msecs = tp.millitm;
  #endif
  if (msecs != -1) {
    lua_pushstring(L, "mseconds");
    lua_pushnumber(L, msecs);
    lua_rawset(L, -3);
  }
  if (iauCal2jd(stmgmt->tm_year+1900, stmgmt->tm_mon+1, stmgmt->tm_mday, &djm0, &djm) >= 0 &&
      iauTf2d('+', stmgmt->tm_hour, stmgmt->tm_min, stmgmt->tm_sec, &djt) >= 0) {
    lua_pushstring(L, "jd");
    lua_pushnumber(L, djm0 + djm + djt);
    lua_rawset(L, -3);
  }
  return 1;
}


/* os.settime, Agena 1.8.0, 24.09.2012

   takes the number of seconds elapsed since the start of an epoch, in your local time zone, and sets the
   system clock accordingly. Agena must be run in root mode in order to change the system time. In case of
   an error, `fail` is returned. The function is only available in the Windows and UNIX versions of Agena (Solaris,
   Linux). */

static int os_settime (lua_State *L) {  /* Agena 1.8.0, 1.9.1. */
#if defined(__unix__) && !defined(LUA_DOS)
  int year, month, day, hour, minute, second, nops;
  time_t t;
#elif defined(_WIN32)
  WORD year, month, day, hour, minute, second;
  int t, nops;
  SYSTEMTIME lpSystemTime;
#else
  lua_pushfail(L);
#endif
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(_WIN32)
  /* common part */
  if (!lua_istable(L, 1))
    luaL_error(L, "Error in " LUA_QS ": expected a table of integers.", "os.settime");
  year = agn_getinumber(L, 1, 1);
  month = agn_getinumber(L, 1, 2);
  day = agn_getinumber(L, 1, 3);
  nops = agn_size(L, 1);
  hour = (nops > 3) ? agn_getinumber(L, 1, 4) : 0;
  minute = (nops > 4) ? agn_getinumber(L, 1, 5) : 0;
  second = (nops > 5) ? agn_getinumber(L, 1, 6) : 0;
  if (tools_checkdatetime(year, month, day, hour, minute, second) == 0)  /* 1.9.1 */
    luaL_error(L, "Error in " LUA_QS ": time component out of range.", "os.settime");
#ifndef _WIN32
  /* UNIX part */
  t = maketime(year, month, day, hour, minute, second);
  if (t != 0)
    luaL_error(L, "Error in " LUA_QS ": could not determine time.", "os.settime");
  if (stime(&t) == 0)
    lua_pushtrue(L);
  else
    lua_pushfail(L);
#else
  /* Windows part */
  lpSystemTime.wYear = year;
  lpSystemTime.wMonth = month;
  lpSystemTime.wDay = day;
  lpSystemTime.wHour = hour;
  lpSystemTime.wMinute = minute;
  lpSystemTime.wSecond = second;
  lpSystemTime.wMilliseconds = lpSystemTime.wDayOfWeek = 0;  /* defaults */
  t = SetLocalTime(&lpSystemTime);  /* returns BOOL */
  agn_pushboolean(L, t == 0 ? -1 : 1);
#endif
#endif
  return 1;
}


static int os_cpuload (lua_State *L) {  /* Agena 1.9.1. */
#if !(defined _WIN32) && !defined(__OS2__) && !defined (LUA_DOS) && !(defined (__SVR4) && defined (__sun))
  double loadavg[3];
  int i;
  if (getloadavg(loadavg, 3) == -1)
    luaL_error(L, "Error in " LUA_QS ": could not determine CPU load.", "os.cpuload");
  agn_createseq(L, 3);
  for (i=0; i < 3; i++)
    lua_seqsetinumber(L, -1, i+1, loadavg[i]);
#else
  lua_pushfail(L);
#endif
  return 1;
}


static int os_pid (lua_State *L) {  /* Agena 1.9.1. */
  luaL_checkstack(L, lua_gettop(L), "too many arguments");
  lua_pushnumber(L, getpid());
  return 1;
}


/* Taken from http://www.codeproject.com/Articles/7340/Get-the-Processor-Speed-in-two-simple-ways,
   originally written by Thomas Latuske in 2004; modified for Agena by alex walz */
#ifdef _WIN32
DWORD ProcSpeedRead() {
  DWORD BufSize, dwMHz;
  HKEY hKey;
  long keyError, valueError;
  BufSize = dwMHz = _MAX_PATH;
  keyError = RegOpenKeyEx(HKEY_LOCAL_MACHINE, "HARDWARE\\DESCRIPTION\\System\\CentralProcessor\\0", 0, KEY_READ, &hKey);
  if (keyError == ERROR_SUCCESS) {
    valueError = RegQueryValueEx(hKey, "~MHz", NULL, NULL, (LPBYTE)&dwMHz, &BufSize);  /* better sure than sorry */
    if (valueError != ERROR_SUCCESS) dwMHz = AGN_NAN;  /* info could not be read */
    RegCloseKey(hKey);
  } else
    dwMHz = AGN_NAN;  /* info could not be read */
  return dwMHz;
}

#define TOTALBYTES 1024  /* better be sure than sorry */
char *BrandRead() {
  DWORD BufSize;
  HKEY hKey;
  long keyError, valueError;
  char cpubrand[TOTALBYTES];  /* malloc will not work ! Brand info seems to be 48 bytes long, however, RegQueryValueEx
    returns the actual number of chars in the brand info. */
  cpubrand[0] = '\0';
  BufSize = sizeof(cpubrand);
  keyError = RegOpenKeyEx(HKEY_LOCAL_MACHINE, "HARDWARE\\DESCRIPTION\\System\\CentralProcessor\\0", 0, KEY_READ, &hKey);
  if (keyError == ERROR_SUCCESS) {
    valueError = RegQueryValueEx(hKey, "ProcessorNameString", NULL, NULL, (LPBYTE)cpubrand, &BufSize);  /* better sure than sorry */
    RegCloseKey(hKey);
    if (valueError == ERROR_SUCCESS && BufSize > 0 && BufSize <= TOTALBYTES && cpubrand != NULL) {
      cpubrand[BufSize-1] = '\0';
      return strdup(cpubrand);  /* FREE return after usage ! */
    }
  }
  return NULL;
}

char *VendorRead() {
  DWORD BufSize;
  HKEY hKey;
  long keyError, valueError;
  char vendor[TOTALBYTES];  /* malloc will not work ! Brand info seems to be 48 bytes long, however, RegQueryValueEx
    returns the actual number of chars in the brand info. */
  vendor[0] = '\0';
  BufSize = sizeof(vendor);
  keyError = RegOpenKeyEx(HKEY_LOCAL_MACHINE, "HARDWARE\\DESCRIPTION\\System\\CentralProcessor\\0", 0, KEY_READ, &hKey);
  if (keyError == ERROR_SUCCESS) {
    valueError = RegQueryValueEx(hKey, "VendorIdentifier", NULL, NULL, (LPBYTE)vendor, &BufSize);  /* better sure than sorry */
    RegCloseKey(hKey);
    if (valueError == ERROR_SUCCESS && BufSize > 0 && BufSize <= TOTALBYTES && vendor != NULL) {
      vendor[BufSize-1] = '\0';
      return strdup(vendor);  /* FREE return after usage ! */
    }
  }
  return NULL;
}
#endif

#ifdef __APPLE__

#define TOTALBYTES 1024
/* See: http://stackoverflow.com/questions/5782012/packagemaker-differentiate-between-ppc-and-intel */
const char *cputype[] = {
  "unknown", "unknown", "unknown", "unknown", "unknown", "unknown",  /* 0 to 5 */
  "MC680x0",  /* CPU_TYPE_MC680x0   (6) */
  "x86",      /* CPU_TYPE_X86       (7) */
  "unknown", "unknown",  /* 8 to 9 */
  "MC98000"   /* CPU_TYPE_MC98000   (10) */
  "HPPA",     /* CPU_TYPE_HPPA      (11) */
  "ARM",      /* CPU_TYPE_ARM       (12) */
  "MC88000",  /* CPU_TYPE_MC88000   (13) */
  "sparc",    /* CPU_TYPE_SPARC     (14) */
  "i860",     /* CPU_TYPE_I860      (15) */
  "unknown", "unknown",  /* 16 to 17 */
  "ppc"};      /* CPU_TYPE_POWERPC  (18) */
#endif

#if defined(__sparc) || defined(__powerpc__) || defined(LUA_DOS) || defined(__OS2__) || defined(__EMX__)
void otherarch (lua_State *L, const char *str) {  /* 1.9.5 */
  lua_createtable(L, 0, 2);
  lua_pushstring(L, "type");
  lua_pushstring(L, str);
  lua_rawset(L, -3);
  lua_pushstring(L, "bigendian");
  agn_pushboolean(L, tools_endian());
  lua_rawset(L, -3);
}
#endif

static int os_cpuinfo (lua_State *L) {  /* Agena 1.9.3 */
#ifdef _WIN32
  struct WinVer winversion;
  if (getWindowsVersion(&winversion) < 6) {  /*  */
    lua_pushfail(L);
    return 1;
  }
  SYSTEM_INFO sysinfo;
  GetSystemInfo(&sysinfo);
  char *brand, *vendor;
  brand = vendor = NULL;
  lua_createtable(L, 0, 8);
  lua_pushstring(L, "ncpu");
  lua_pushinteger(L, sysinfo.dwNumberOfProcessors);
  lua_rawset(L, -3);
  lua_pushstring(L, "type");
  switch (sysinfo.wProcessorArchitecture) {
    case PROCESSOR_ARCHITECTURE_AMD64: {  /* 9 */
      lua_pushstring(L, "x64");
      break;
    }
    case PROCESSOR_ARCHITECTURE_ARM: {  /* reserved */
      lua_pushstring(L, "ARM");
      break;
    }
    case PROCESSOR_ARCHITECTURE_IA64: {  /* 6 */
      lua_pushstring(L, "Itanium");
      break;
    }
    case PROCESSOR_ARCHITECTURE_INTEL: {  /* 0 */
      lua_pushstring(L, "x86");
      break;
    }
    case PROCESSOR_ARCHITECTURE_UNKNOWN: {   /* 0xffff */
      lua_pushstring(L, "unknown");
      break;
    }
    default: {
      lua_pushstring(L, "unknown");
    }
  }
  lua_rawset(L, -3);
  lua_pushstring(L, "level");
  lua_pushnumber(L, sysinfo.wProcessorLevel);
  lua_rawset(L, -3);
  lua_pushstring(L, "revision");
  lua_pushnumber(L, sysinfo.wProcessorRevision);
  lua_rawset(L, -3);
  lua_pushstring(L, "frequency");
  lua_pushnumber(L, ProcSpeedRead());
  lua_rawset(L, -3);
  brand = BrandRead();
  if (brand != NULL) {
    lua_pushstring(L, "brand");
    lua_pushstring(L, brand);
    lua_rawset(L, -3);
    xfree(brand);
  }
  vendor = VendorRead();
  if (vendor != NULL) {
    lua_pushstring(L, "vendor");
    lua_pushstring(L, vendor);
    lua_rawset(L, -3);
    xfree(vendor);
  }
  lua_pushstring(L, "bigendian");
  agn_pushboolean(L, tools_endian());
  lua_rawset(L, -3);
#elif __APPLE__
  uint64_t data;
  size_t size, brandlen, vendorlen;
  char brand[TOTALBYTES];
  char vendor[TOTALBYTES];
  brand[0] = '\0'; brandlen = TOTALBYTES;
  vendor[0] = '\0'; vendorlen = TOTALBYTES;
  data = 0;
  size = sizeof(data);
  lua_createtable(L, 0, 10);
  if (sysctlbyname("hw.cpufrequency", &data, &size, NULL, 0) >= 0) {
    lua_pushstring(L, "frequency");
    lua_pushnumber(L, data/1e6);
    lua_rawset(L, -3);
  }
  data = 0; size = sizeof(data);
  if (sysctlbyname("hw.ncpu", &data, &size, NULL, 0) >= 0) {
    lua_pushstring(L, "ncpu");
    lua_pushnumber(L, data);
    lua_rawset(L, -3);
  }
  data = 0; size = sizeof(data);
  if (sysctlbyname("hw.cputype", &data, &size, NULL, 0) >= 0) {
    lua_pushstring(L, "type");
    if (data >= 0 && data < 19) {
      uint64_t is64 = 0;
      if (sysctlbyname("hw.cpu64bit_capable", &is64, &size, NULL, 0) >= 0) {
        if (is64 && (data == 7)) lua_pushstring(L, "x64");
        else if (is64 && (data == 18)) lua_pushstring(L, "ppc64");
        else lua_pushstring(L, cputype[data]);
      } else
        lua_pushstring(L, cputype[data]);
    } else
      lua_pushstring(L, "unknown");
    lua_rawset(L, -3);
  }
  data = 0; size = sizeof(data);
  /* http://cr.yp.to/hardware/ppc.html:
     MacOS sysctl reports "hw.cputype: 18" for PowerPC, and in particular "hw.cpusubtype":
     1 for 601, 2 for 602, 3 or 4 or 5 for 603, 6 or 7 for 604, 8 for 620, 9 for 750, 10 for 7400, 11 for 7450, 100 for 970. */
  if (sysctlbyname("hw.cpusubtype", &data, &size, NULL, 0) >= 0) {
    lua_pushstring(L, "level");
    lua_pushnumber(L, data);
    lua_rawset(L, -3);
  }
  if (sysctlbyname("machdep.cpu.brand_string", &brand, &brandlen, NULL, 0) >= 0) {
    lua_pushstring(L, "brand");
    lua_pushstring(L, brand);
    lua_rawset(L, -3);
  }
  if (sysctlbyname("machdep.cpu.vendor", &vendor, &vendorlen, NULL, 0) >= 0) {
    lua_pushstring(L, "vendor");
    lua_pushstring(L, vendor);
    lua_rawset(L, -3);
  }
  data = 0; size = sizeof(data);
  if (sysctlbyname("machdep.cpu.model", &data, &size, NULL, 0) >= 0) {
    lua_pushstring(L, "model");
    lua_pushnumber(L, data);
    lua_rawset(L, -3);
  }
  data = 0; size = sizeof(data);
  if (sysctlbyname("machdep.cpu.stepping", &data, &size, NULL, 0) >= 0) {
    lua_pushstring(L, "stepping");
    lua_pushnumber(L, data);
    lua_rawset(L, -3);
  }
  lua_pushstring(L, "bigendian");
  agn_pushboolean(L, tools_endian());
  lua_rawset(L, -3);
#elif defined(__sparc)
  otherarch(L, "sparc");
#elif __powerpc__
  otherarch(L, "ppc");  /* 1.9.5 */
#elif defined(LUA_DOS) || defined(__OS2__) || defined(__EMX__)  /* 1.9.5 */
  otherarch(L, "x86");
#else
  lua_pushfail(L);
#endif
  return 1;
}


/*
* lalarm.c
* an alarm library for Lua 5.1 based on signal
* Luiz Henrique de Figueiredo <lhf@tecgraf.puc-rio.br>
* 03 May 2012 00:26:33
* This code is hereby placed in the public domain.

  The library [function] exports a single function, alarm([s, [f]]), which tells
  Lua to call f after s seconds have elapsed. This is done only once. If you want
  f to be called every s seconds, call alarm(s) inside f. Call alarm() without
  arguments or with s=0 to cancel any pending alarm.
*/

#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__)
#define NAME	"alarm handler"

static lua_State *LL = NULL;
static lua_Hook oldhook = NULL;
static int oldmask=0;
static int oldcount=0;

static void l_handler (lua_State *L, lua_Debug *ar) {
  (void)ar;
  L = LL;
  lua_sethook(L, oldhook, oldmask, oldcount);
  lua_getfield(L, LUA_REGISTRYINDEX, NAME);
  lua_call(L, 0, 0);
}

static void l_signal (int i) {	/* assert(i==SIGALRM); */
  signal(i, SIG_DFL);
  oldhook = lua_gethook(LL);
  oldmask = lua_gethookmask(LL);
  oldcount = lua_gethookcount(LL);
  lua_sethook(LL, l_handler, LUA_MASKCALL | LUA_MASKRET | LUA_MASKCOUNT, 1);
}

static int os_alarm (lua_State *L) {  /* alarm([secs, [func]]) */
  LL = L;
  switch (lua_gettop(L)) {
    case 0:
	  break;
    case 1:
	  lua_getfield(L, LUA_REGISTRYINDEX, NAME);
      if (lua_isnil(L, -1)) luaL_error(L, "no alarm handler set");
      break;
    default:
	  lua_settop(L, 2);
      luaL_checktype(L, 2, LUA_TFUNCTION);
	  lua_setfield(L, LUA_REGISTRYINDEX, NAME);
      break;
  }
  if (signal(SIGALRM, l_signal) == SIG_ERR)
    lua_pushnil(L);
  else
    lua_pushinteger(L, alarm(luaL_optinteger(L, 1, 0)));
  return 1;
}
#endif


/* CreateLink and ResolveLink have been taken from "Shell Links" in MSDN library, reproduced in parts
   by the dynamorio project; taken from sources available at http://code.google.com/p/dynamorio and
   largely modified for Agena */

#ifdef _WIN32
/* this modified version of CreateLink can also cope with file names with no absolute or relative paths. */
HRESULT CreateLink (LPCSTR lpszPathObj, LPSTR lpszPathLink, LPSTR lpszDesc) {
  int hres;
  IShellLink *psl = NULL;
  IPersistFile *ppf = NULL;
  WCHAR *wsz;
  char *ObjWorkingDir, *tmp, *cwdbuffer, *fullsourcepath;
  LPSTR link;
  cwdbuffer = fullsourcepath = NULL;
  hres = CoInitialize(NULL);
  if (hres != S_FALSE && hres != S_OK) goto error3;  /* CoUninitialize, checked */
  /* Get a pointer to the IShellLink interface */
  hres = CoCreateInstance(&CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER, &IID_IShellLink, (LPVOID *)&psl);
  if (!SUCCEEDED(hres)) goto error3;  /* CoUninitialize, checked */
  /* Set the path to the shortcut target, and add the description. */
  /* psl->lpVtbl->SetPath(psl, (LPCSTR)charreplace((char *)lpszPathObj, '/', '\\', 0)); */
  psl->lpVtbl->SetDescription(psl, lpszDesc);
  ObjWorkingDir = strdup(lpszPathObj);
  if (ObjWorkingDir != NULL) {
    charreplace(ObjWorkingDir, '/', '\\', 0);
    tmp = strrchr(ObjWorkingDir, '\\');
    if (tmp != NULL) {  /* source path given ? */
      tmp[0] = '\0';
      if (*ObjWorkingDir == '\0') goto error2;  /* leading slash ? */
      psl->lpVtbl->SetWorkingDirectory(psl, ObjWorkingDir);
      psl->lpVtbl->SetPath(psl, (LPCSTR)charreplace((char *)lpszPathObj, '/', '\\', 0));
    } else {  /* no source path given */
      cwdbuffer = (char *)malloc((PATH_MAX+1) * sizeof(char));
      if (tools_cwd(cwdbuffer) != 0) goto error2; /* frees cwdbuffer automatically; CoUninitialize & xfree(ObjWorkingDir) */
      psl->lpVtbl->SetWorkingDirectory(psl, cwdbuffer);
      fullsourcepath = concat(cwdbuffer, "\\", (char *)lpszPathObj);
      psl->lpVtbl->SetPath(psl, (LPCSTR)charreplace((char *)fullsourcepath, '/', '\\', 0));
    }
  } else goto error3;  /* strdup failed, no need to free ObjWorkingDir; conduct CoUninitialize */
  /* process link target now */
  charreplace(lpszPathLink, '/', '\\', 0);
  if (strrchr(lpszPathLink, '\\') == NULL) {  /* no path given ? */
    if (cwdbuffer == NULL) {  /* cwd not yet determined ? */
      cwdbuffer = (char *)malloc((PATH_MAX+1)*sizeof(char));
      if (tools_cwd(cwdbuffer) != 0) goto error2;  /* frees cwdbuffer automatically; CoUninitialize & xfree(ObjWorkingDir) */
      charreplace(cwdbuffer, '/', '\\', 0);
    }
    /* compose full link path, cwd at this point has already been determined */
    link = (LPSTR)concat(cwdbuffer, "\\", lpszPathLink, NULL);
    if (link == NULL) goto error1;  /* CoUninitialize & xfree(ObjWorkingDir) & xfree(cwdbuffer) */
  } else {  /* a path has been given, at this point CoUnIni, ObjWorkingDir, and possibly cwdbuffer */
    link = (LPSTR)concat(lpszPathLink, NULL);
    if (link == NULL) {
      if (cwdbuffer == NULL)
        goto error2;  /* xfree CoUnIni, and ObjWorkingDir */
      else
        goto error1;  /* xfree CoUnIni, cwdbuffer, and ObjWorkingDir */
    }
  }
  /* Query IShellLink for the IPersistFile interface for saving the shortcut in persistent storage. */
  /* at this point, CoIni, OWD, cwdbuffer, and link have been malloc'ed */
  hres = psl->lpVtbl->QueryInterface(psl, &IID_IPersistFile, (void **)&ppf);
  if (!SUCCEEDED(hres)) goto error0;
  /* Ensure that the string is ANSI, first determine the length of the resulting string wsz: */
  /* hres = MultiByteToWideChar(AreFileApisANSI() ? CP_ACP : CP_OEMCP, 0, lpszPathLink, -1, NULL, 0); */
  hres = MultiByteToWideChar(CP_UTF8, 0, link, -1, NULL, 0);
  if (!SUCCEEDED(hres)) goto error0;
  /* allocate memory for the buffer, plus 1 to be absolutely sure, malloc needs the number of characters,
     not the number of bytes */
  wsz = (WCHAR*)malloc(hres+1);
  if (wsz == NULL) goto error0;
  /* hres = MultiByteToWideChar(AreFileApisANSI() ? CP_ACP : CP_OEMCP, 0, lpszPathLink, -1, wsz, hres); */
  hres = MultiByteToWideChar(CP_UTF8, 0, link, -1, wsz, hres);
  if (!SUCCEEDED(hres)) goto error;
  /* Save the link by calling IPersistFile::Save. */
  hres = ppf->lpVtbl->Save(ppf, wsz, TRUE);
  ppf->lpVtbl->Release(ppf);
  psl->lpVtbl->Release(psl);
  CoUninitialize();
  xfree(ObjWorkingDir)
  xfree(link);
  xfree(cwdbuffer);
  xfree(wsz);
  return hres;

error:  /* fall through */
  xfree(wsz);

error0:
  xfree(link);

error1:
  xfree(cwdbuffer);

error2:
  xfree(ObjWorkingDir);
  if (fullsourcepath != NULL) xfree(fullsourcepath);

error3:
  CoUninitialize();
  return -1;
}

LUALIB_API int symlink (const char *src, const char *dest) {
  HRESULT res;
  res = CreateLink((LPCSTR)src, (LPSTR)dest, (LPSTR)src);
  return (int)res;
}

#define PATHLEN 4*(PATH_MAX-1)
/* ONLY call this function if the file lpszLinkFile really EXIST, otherwise the function would crash ! */
HRESULT ResolveLink(LPSTR lpszLinkFile, LPSTR lpszPath) {
  HRESULT hres;
  IShellLink* psl;
  IPersistFile* ppf;
  WCHAR *wsz;
  char szGotPath[PATHLEN];
  *lpszPath = 0;  /* assume failure */
  hres = CoInitialize(NULL);
  if (hres != S_FALSE && hres != S_OK) {
	CoUninitialize();
    return -1;
  }
  /* Get a pointer to the IShellLink interface. */
  hres = CoCreateInstance((REFCLSID)&CLSID_ShellLink, NULL, CLSCTX_INPROC_SERVER,
                          (REFIID)&IID_IShellLink, (LPVOID *)&psl);
  if (!SUCCEEDED(hres)) goto error2;
  hres = psl->lpVtbl->QueryInterface(psl, &IID_IPersistFile, (void **)&ppf);
  /* Ensure that the string is Unicode. */
  hres = MultiByteToWideChar(CP_UTF8, 0, lpszLinkFile, -1, NULL, 0);
  if (!SUCCEEDED (hres)) goto error2;
  wsz = (WCHAR*)malloc(hres+1);
  if (wsz == NULL) goto error2;
  hres = MultiByteToWideChar(CP_UTF8, 0, lpszLinkFile, -1, wsz, hres);
  if (!SUCCEEDED (hres)) goto error1;
  /* Load the shortcut. */
  hres = ppf->lpVtbl->Load(ppf, wsz, 0); /* STGM_READ); */
  if (!SUCCEEDED(hres)) goto error1;
  /* Resolve the link. */
  hres = psl->lpVtbl->Resolve(psl, NULL, 0); /* SLR_ANY_MATCH); */
  if (!SUCCEEDED(hres)) goto error1;
  /* Get the path to the link target. */
  hres = psl->lpVtbl->GetPath(psl, szGotPath, PATHLEN, NULL, SLGP_SHORTPATH);
  if (!SUCCEEDED(hres)) goto error1;
  lstrcpy(lpszPath, szGotPath);
  /* Release the pointer to the IPersistFile interface. */
  ppf->lpVtbl->Release(ppf);
  /* Release the pointer to the IShellLink interface. */
  psl->lpVtbl->Release(psl);
  CoUninitialize();
  xfree(wsz);
  return hres;

error1:
  xfree(wsz);
  /* fall through */

error2:
  CoUninitialize();
  return -1;
}

static int os_readlink (lua_State *L) {
  HRESULT res;
  char *buffer;
  const char *linkname;
  linkname = agn_checkstring(L, 1);
  if (access(linkname, 00|04) == -1) {
    lua_pushfail(L);
    return 1;
  }
  buffer = malloc(PATHLEN*sizeof(char));
  SetErrorMode(SEM_FAILCRITICALERRORS | SEM_NOOPENFILEERRORBOX);
  res = ResolveLink((char *)linkname, (LPSTR)buffer);
  SetErrorMode(0);
  if (res == -1)
    lua_pushfail(L);
  else
    lua_pushstring(L, buffer);
  xfree(buffer);
  return 1;
}
#endif


#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__) || defined(_WIN32)
static int os_symlink (lua_State *L) {
  int en, r;
  const char *oldname, *newname;
  oldname = agn_checkstring(L, 1);
  newname = agn_checkstring(L, 2);
  r = symlink(oldname, newname);
  en = errno;
  if (r == -1) {
    luaL_error(L, "Error in " LUA_QS ": %s.", "os.symlink", strerror(en));
  }
  lua_pushtrue(L);
  return 1;
}
#endif

#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__)
static int os_readlink (lua_State *L) {
  int bufsize, nchars, en;
  const char *filename;
  char *buffer;
  bufsize = 1128;
  buffer = NULL;
  filename = agn_checkstring(L, 1);
  if (access(filename, 00|04) == -1) {
    lua_pushfail(L);
    return 1;
  }
  while (1) {
    buffer = (char *)realloc(buffer, bufsize);
    if (buffer == NULL)
      luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "os.readlink");
    nchars = readlink(filename, buffer, bufsize);  /* with _no_ \0 included ! */
    en = errno;
    if (nchars < 0) {
      xfree(buffer);
      lua_pushfail(L);
      if (en == EINVAL)
        /* luaL_error(L, "Error in " LUA_QS ": %s is not a symbolic link.", "os.readlink", filename); */
        lua_pushstring(L, "file is not a symbolic link");
      else
        /* luaL_error(L, "Error in " LUA_QS ": %s.", "os.readlink", strerror(en)); */
        lua_pushstring(L, strerror(en));
      return 2;
    }
    if (nchars < bufsize) {
      lua_pushlstring(L, buffer, nchars);
      xfree(buffer);
      return 1;
    }
    bufsize *= 2;
  }
}
#endif


/*
#if __linux__
static int os_cdrom (lua_State *L) {
  int fh, mode;
  const char *devname, *option;
  devname = agn_checkstring(L, 1);
  option = agn_checkstring(L, 2);
  mode = -1;
  if (strcmp(option, "eject") == 0 || strcmp(option, "open") == 0)
    mode = CDROMEJECT;
  else if (strcmp(option, "close") == 0)
    mode = CDROMCLOSETRAY;
  else
    luaL_error(L, "Error in " LUA_QS " with %s: invalid option " LUA_QS ".", "os.cdrom", devname, option);
  fh = open(devname, O_RDONLY|O_NONBLOCK);
printf("%d %d\n", fh, mode);
  if (fh == -1)
    luaL_error(L, "Error in " LUA_QS " with %s: %s.", "os.cdrom", devname, strerror(errno));
  if (ioctl(fh, mode) == -1) {
    int en = errno;
    close(fh);
    if (en == EDRIVE_CANT_DO_THIS)
      luaL_error(L, "Error in " LUA_QS ": %s %s.", "os.cdrom", "not supported by drive", devname);
    else
      luaL_error(L, "Error in " LUA_QS ": %s %s.", "os.cdrom", strerror(en), devname);
  } else if (close(fh) == -1)
    luaL_error(L, "Error in " LUA_QS ": could not close device %s: %s.", "os.cdrom", devname, strerror(errno));
  lua_pushtrue(L);
  return 1;
}
#elif __APPLE__
  // drutil tray (open, close, eject)
static int os_cdrom (lua_State *L) {
  char *command;
  const char *option;
  option = agn_checkstring(L, 1);
  if (!(strcmp(option, "eject") == 0 || strcmp(option, "open") == 0 || strcmp(option, "close") == 0))
    luaL_error(L, "Error in " LUA_QS ": invalid option " LUA_QS ".", "os.cdrom", option);
  command = concat("drutil tray ", option, NULL);
  system(command);
  xfree(command);
  lua_pushtrue(L);
  return 1;
}
#endif
*/

static const luaL_Reg syslib[] = {
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__)
  {"alarm",     os_alarm},
#endif
#if (defined(__unix__) && !defined(LUA_DOS)) || defined(__APPLE__) || defined(_WIN32)
  {"symlink",   os_symlink},          /* 1.12.1, added on June 15, 2013 */
  {"readlink",  os_readlink},         /* 1.12.1, added on June 17, 2013 */
#endif
  {"battery",   os_battery},          /* 0.14.0, added on March 13, 2009 */
  {"beep",      os_beep},             /* added on March 31, 2007 */
  {"chdir",     os_chdir},            /* added July 07, 2008 */
  {"fattrib",   os_fattrib},          /* 0.26.0, added August 06, 2009 */
  {"computername", os_computername},  /* 0.5.3  Sept 15, 2007 */
  {"cpuinfo",   os_cpuinfo},          /* added on February 07, 2013 */
  {"cpuload",   os_cpuload},          /* added on January 31, 2013 */
  {"date",      os_date},
  {"datetosecs",   os_datetosecs},    /* added September 20, 2012 */
  {"difftime",  os_difftime},
  {"drives",    os_drives},           /* 0.26.0, 05.08.2009 */
  {"drivestat", os_drivestat},        /* 0.26.0, 05.08.2009 */
  {"endian",    os_endian},           /* 0.8.0, added on December 11, 2007 */
  {"environ",   os_environ},
  {"execute",   os_execute},          /* changed in 0.5.4 */
  {"exists",    os_exists},           /* added on April 09, 2007 */
  {"exit",      os_exit},
  {"fcopy",     os_fcopy},            /* added on July 19, 2009 */
  {"freemem",   os_freemem},          /* 0.5.3  Sept 15, 2007 */
  {"fstat",     os_fstat},            /* added on October 11, 2008 */
  {"getenv",    os_getenv},
  {"isANSI",    os_isANSI},           /* added 0.28.2, 17.11.2009 */
  {"listcore",  os_listcore},         /* added on August 01, 2007 */
  {"login",     os_login},            /* 0.5.3  Sept 15, 2007 */
  {"memstate",  os_memstate},         /* 0.5.3  Sept 15, 2007 */
  {"mkdir",     os_mkdir},            /* added July 07, 2008 */
  {"mousebuttons", os_mousebuttons},  /* added March 28, 2010 */
  {"move",      os_move},
  {"now",       os_now},              /* added September 20, 2012 */
  {"pid",       os_pid},              /* added February 02, 2012 */
  {"remove",    os_remove},
  {"rmdir",     os_rmdir},            /* added July 07, 2008 */
  {"screensize", os_screensize},      /* added March 28, 2010 */
  {"secstodate",   os_secstodate},    /* added September 20, 2012 */
  {"setenv",    os_setenv},           /* added October 24, 2010 */
  {"setlocale", os_setlocale},
  {"settime",   os_settime},          /* added September 24, 2012 */
  {"system",    os_system},           /* 0.12.2  October 12, 2008 */
  {"time",      os_time},
  {"tmpname",   os_tmpname},
  {"wait",      os_wait},             /* added on March 31, 2007 */
  {NULL, NULL}
};

/* }====================================================== */



LUALIB_API int luaopen_os (lua_State *L) {
  luaL_register(L, LUA_OSLIBNAME, syslib);
  return 1;
}

