/*
** $Id: lobject.c,v 2.22 2006/02/10 17:43:52 roberto Exp $
** Some generic functions over Lua & Agena objects
** See Copyright Notice in agena.h
*/

#include <ctype.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define lobject_c
#define LUA_CORE

#include "agena.h"

#include "ldo.h"
#include "lmem.h"
#include "lobject.h"
#include "lstate.h"
#include "lstring.h"
#include "lvm.h"



const TValue luaO_nilobject_ = {{NULL}, LUA_TNIL};

/*
** converts an integer to a "floating point byte", represented as
** (eeeeexxx), where the real value is (1xxx) * 2^(eeeee - 1) if
** eeeee != 0 and (xxx) otherwise.
*/
int luaO_int2fb (unsigned int x) {
  int e = 0;  /* expoent */
  while (x >= 16) {
    x = (x+1) >> 1;
    e++;
  }
  if (x < 8) return x;
  else return ((e+1) << 3) | (cast_int(x) - 8);
}


/* converts back */
int luaO_fb2int (int x) {
  int e = (x >> 3) & 31;
  if (e == 0) return x;
  else return ((x & 7)+8) << (e - 1);
}


int luaO_log2 (unsigned int x) {
  static const lu_byte log_2[256] = {
    0,1,2,2,3,3,3,3,4,4,4,4,4,4,4,4,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,5,
    6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,6,
    7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
    7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,
    8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
    8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
    8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,
    8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8,8
  };
  int l = -1;
  while (x >= 256) { l += 8; x >>= 8; }
  return l + log_2[x];

}


int luaO_rawequalObj (const TValue *t1, const TValue *t2) {
  if (ttype(t1) != ttype(t2)) return 0;
  else switch (ttype(t1)) {
    case LUA_TNIL:
      return 1;
    case LUA_TNUMBER:
      return luai_numeq(nvalue(t1), nvalue(t2));
    case LUA_TBOOLEAN:
      return bvalue(t1) == bvalue(t2);  /* boolean true must be 1 !! */
    case LUA_TLIGHTUSERDATA:
      return pvalue(t1) == pvalue(t2);
    case LUA_TFAIL:
      return 1;
    default:
      lua_assert(iscollectable(t1));
      return gcvalue(t1) == gcvalue(t2);
  }
}


int luaO_str2d (const char *s, lua_Number *result) {  /* modified 1.6.4 to avoid compiler warnings */
  char *endptr;
  *result = lua_str2number(s, &endptr);
  if (endptr == s) return 0;  /* conversion failed */
  switch (*endptr) {  /* patched 0.12.2 */
    case 'x':
    case 'X': /* maybe an hexadecimal constant? */
      *result = cast_num(strtoul(s, &endptr, 16));
      break;
    case 'b':
    case 'B': /* maybe a binary constant? 0.31.2 */
      if ((*(endptr+1) != '\0'))
        *result = cast_num(strtoul(endptr+1, &endptr, 2));
      break;
    case 'o':
    case 'O': /* maybe an octal constant? 0.31.2 */
      if ((*(endptr+1) != '\0'))
        *result = cast_num(strtoul(endptr+1, &endptr, 8));
      break;
    case 'k':
      *result *= 1000.0;
      ++endptr;
      break;
    case 'K':  /* kilobyte */
      *result *= 1024.0;
      ++endptr;
      break;
    case 'm':  /* million */
      *result *= 1000000.0;
      ++endptr;
      break;
    case 'M':  /* megabyte */
      *result *= 1048576.0;
      ++endptr;
      break;
    case 'g':  /* billion, 1.5.1 */
      *result *= 1000000000.0;
      ++endptr;
      break;
    case 'G':  /* gigabyte */
      *result *= 1073741824.0;
      ++endptr;
      break;
    case 't':  /* trillion, 1.5.1 */
      *result *= 1000000000000.0;
      ++endptr;
      break;
    case 'T':  /* terabyte, 1.5.1 */
      *result *= 1099511627776.0;
      ++endptr;
      break;
    /* overflow with Terabytes, so not queried here */
    case '%':  /* percent, 2.0.0 RC 2 */
      *result /= 100.0;
      ++endptr;
      break;
    case 'D':  /* dozen */
      *result *= 12;
      ++endptr;
      break;
  }
  if (*endptr == '\0') return 1;  /* most common case */
  while (isspace(cast(unsigned char, *endptr))) endptr++;
  if (*endptr != '\0') return 0;  /* invalid trailing characters? */
  return 1;
}


#ifndef PROPCMPLX
int luaO_str2c (const char *s, agn_Complex *result) {  /* 0.26.1 */
#else
int luaO_str2c (const char *s, lua_Number *result) {  /* 0.26.1 */
#endif
  char *endptr;
  lua_Number real, imag, sign;
  sign = 1;
  real = lua_str2number(s, &endptr);
  imag = 0;
  if (endptr == s) {  /* only `-I`, `+I` passed ? */
    while (isspace(cast(unsigned char, *endptr))) endptr++;
    if (*endptr == '+' || *endptr == '-') {
      if (*endptr == '-') sign = -1;
      ++endptr;
    }
    if (*endptr == 'I') {
#ifndef PROPCMPLX
      *result = 0 + sign*I;
#else
      result[0] = 0; result[1] = sign;
#endif
      ++endptr;
      goto end;
    } else {  /* conversion failed */
#ifndef PROPCMPLX
      *result = real;
#else
      result[0] = real; result[1] = real;
#endif
      return 0;
    }
  }
  /* as yet, a number has been recognized */
  while (isspace(cast(unsigned char, *endptr))) endptr++;
  if (*endptr == '*') {  /* only imaginary part passed ? */
    ++endptr;
    while (isspace(cast(unsigned char, *endptr))) endptr++;
    if (*endptr == 'I') {
#ifndef PROPCMPLX
      *result = 0 + real*I;
#else
      result[0] = 0; result[1] = real;
#endif
      ++endptr;
    } else {  /* conversion failed */
#ifndef PROPCMPLX
      *result = real;
#else
      result[0] = real; result[1] = real;
#endif
      return 0;
    }
  } else if (*endptr == '+' || *endptr == '-') {  /* real part passed, try imaginary part */
    const char *newpos;
    if (*endptr == '-') sign = -1;
    ++endptr;
    while (isspace(cast(unsigned char, *endptr))) endptr++;
    if (*endptr == 'I') {  /* only imaginary unit passed ? */
#ifndef PROPCMPLX
      *result = real + sign*I;
#else
      result[0] = real; result[1] = sign;
#endif
      ++endptr;
      goto end;
    }
    newpos = endptr;
    /* read number of imaginary part */
    imag = lua_str2number(newpos, &endptr);
    if (endptr == newpos) {  /* conversion failed */
#ifndef PROPCMPLX
      *result = imag;
#else
      result[0] = imag; result[1] = imag;
#endif
      return 0;
    }
    while (isspace(cast(unsigned char, *endptr))) endptr++;
    if (*endptr == '*') {
      ++endptr;
      while (isspace(cast(unsigned char, *endptr))) endptr++;
      if (*endptr == 'I') {
#ifndef PROPCMPLX
        *result = real + sign*imag*I;
#else
        result[0] = real; result[1] = sign*imag;
#endif
        ++endptr;
      } else {  /* conversion failed */
#ifndef PROPCMPLX
        *result = imag;
#else
        result[0] = imag; result[1] = imag;
#endif
        return 0;
      }
    } else {  /* conversion failed */
#ifndef PROPCMPLX
      *result = imag;
#else
      result[0] = imag; result[1] = imag;
#endif
      return 0;
    }
  }
end:
  if (*endptr == '\0') return 1;  /* most common case */
  while (isspace(cast(unsigned char, *endptr))) endptr++;
  if (*endptr != '\0') return 0;  /* invalid trailing characters? */
  return 1;
}


static void pushstr (lua_State *L, const char *str) {
  setsvalue2s(L, L->top, luaS_new(L, str));
  incr_top(L);
}


/* this function handles only `%d', `%c', %f, %p, and `%s' formats */
const char *luaO_pushvfstring (lua_State *L, const char *fmt, va_list argp) {
  int n = 1;
  pushstr(L, "");
  for (;;) {
    const char *e = strchr(fmt, '%');
    if (e == NULL) break;
    setsvalue2s(L, L->top, luaS_newlstr(L, fmt, e-fmt));
    incr_top(L);
    switch (*(e+1)) {
      case 's': {
        const char *s = va_arg(argp, char *);
        if (s == NULL) s = "(null)";
        pushstr(L, s);
        break;
      }
      case 'c': {
        char buff[2];
        buff[0] = cast(char, va_arg(argp, int));
        buff[1] = '\0';
        pushstr(L, buff);
        break;
      }
      case 'd': {
        setnvalue(L->top, cast_num(va_arg(argp, int)));
        incr_top(L);
        break;
      }
      case 'f': {
        setnvalue(L->top, cast_num(va_arg(argp, l_uacNumber)));
        incr_top(L);
        break;
      }
      case 'p': {
        char buff[4*sizeof(void *) + 8]; /* should be enough space for a `%p' */
        sprintf(buff, "%p", va_arg(argp, void *));
        pushstr(L, buff);
        break;
      }
      case '%': {
        pushstr(L, "%");
        break;
      } 
      default: {
        char buff[3];
        buff[0] = '%';
        buff[1] = *(e+1);
        buff[2] = '\0';
        pushstr(L, buff);
        break;
      }
    }
    n += 2;
    fmt = e+2;
  }
  pushstr(L, fmt);
  luaV_concat(L, n+1, cast_int(L->top - L->base) - 1);
  L->top -= n;
  return svalue(L->top - 1);
}


const char *luaO_pushfstring (lua_State *L, const char *fmt, ...) {
  const char *msg;
  va_list argp;
  va_start(argp, fmt);
  msg = luaO_pushvfstring(L, fmt, argp);
  va_end(argp);
  return msg;
}


void luaO_chunkid (char *out, const char *source, size_t bufflen) {
  if (*source == '=') {
    strncpy(out, source+1, bufflen);  /* remove first char */
    out[bufflen-1] = '\0';  /* ensures null termination */
  }
  else {  /* out = "source", or "...source" */
    if (*source == '@') {
      size_t l;
      source++;  /* skip the `@' */
      bufflen -= sizeof(" '...' ");
      l = strlen(source);
      strcpy(out, "");
      if (l > bufflen) {
        source += (l-bufflen);  /* get last part of file name */
        strcat(out, "...");
      }
      strcat(out, source);
    }
    else {  /* out = [string "string"] */
      size_t len = strcspn(source, "\n\r");  /* stop at first newline */
      bufflen -= sizeof(" [string \"...\"] ");
      if (len > bufflen) len = bufflen;
      strcpy(out, "[string \"");
      if (source[len] != '\0') {  /* must truncate? */
        strncat(out, source, len);
        strcat(out, "...");
      }
      else
        strcat(out, source);
      strcat(out, "\"]");
    }
  }
}


