/*
** $Id: lmathlib.c,v 1.67 2005/08/26 17:36:32 roberto Exp $
** Standard mathematical library
** See Copyright Notice in agena.h
*/


#include <stdlib.h>
#include <math.h>

#define lmathlib_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"

#include "agncmpt.h"  /* for trunc and isfinite, and FLT_EVAL_METHOD constant, and off64* types */
#include "lobject.h"

#undef PHI
#define PHI      ((1.0+sqrt(5.0))/2.0)


static int math_min (lua_State *L) {
  int n = lua_gettop(L);  /* number of arguments */
  lua_Number dmin = agn_checknumber(L, 1);
  int i;
  for (i=2; i<=n; i++) {
    lua_Number d = agn_checknumber(L, i);
    if (d < dmin)
      dmin = d;
  }
  lua_pushnumber(L, dmin);
  return 1;
}


static int math_max (lua_State *L) {
  int n = lua_gettop(L);  /* number of arguments */
  lua_Number dmax = agn_checknumber(L, 1);
  int i;
  for (i=2; i<=n; i++) {
    lua_Number d = agn_checknumber(L, i);
    if (d > dmax)
      dmax = d;
  }
  lua_pushnumber(L, dmax);
  return 1;
}


static LUAI_UINT32 m_w = 10;  /* must not be zero */
static LUAI_UINT32 m_z = 7;   /* must not be zero */

static int math_random (lua_State *L) {
  /* the `%' avoids the (rare) case of r==1, and is needed also because on some systems (SunOS!) `rand()' may return
     a value larger than RAND_MAX;
     RNG algorithm taken from http://en.wikipedia.org/wiki/Random_number_generation, written by George Marsaglia.
     0.32.2, 22.05.2010 */
  lua_Number r;
  m_z = 36969 * (m_z & 65535) + (m_z >> 16);
  m_w = 18000 * (m_w & 65535) + (m_w >> 16);
  r = (lua_Number)((m_z << 16) + m_w)/(lua_Number)(2*(double)LUAI_MAXINT32);  /* 32-bit result */
  switch (lua_gettop(L)) {  /* check number of arguments */
    case 0: {  /* no arguments */
      lua_pushnumber(L, r);  /* Number between 0 and 1 */
      break;
    }
    case 1: {  /* only upper limit */
      int u = agnL_checkint(L, 1);
      luaL_argcheck(L, 1<=u, 1, "interval is empty");
      lua_pushnumber(L, floor(r*u)+1);  /* int between 1 and `u' */
      break;
    }
    case 2: {  /* lower and upper limits */
      int l = agnL_checkint(L, 1);
      int u = agnL_checkint(L, 2);
      luaL_argcheck(L, l<=u, 2, "interval is empty");
      lua_pushnumber(L, floor(r*(u-l+1))+l);  /* int between `l' and `u' */
      break;
    }
    default: return luaL_error(L, "Error in " LUA_QS ": wrong number of arguments (0, 1, or 2 required).", "math.random");
  }
  return 1;
}


static int math_randomseed (lua_State *L) {  /* changed 0.32.2, 23.05.2010 */
  lua_Number w, z;
  w = agnL_checkint(L, 1);
  z = agnL_checkint(L, 2);
  if (w <= 0 || z <= 0)
    luaL_error(L, "Error in " LUA_QS ": both values must be positive. Seeds not changed.", "math.randomseed");
  m_w = (LUAI_UINT32)w;
  m_z = (LUAI_UINT32)z;
  lua_pushinteger(L, m_w);
  lua_pushinteger(L, m_z);
  return 2;
}



static int math_isprime (lua_State *L) {  /* patched 0.27.0; tuned 0.30.3; extended 1.0.6 */
  long long int x;
  lua_Number n;
  n = agnL_checknumber(L, 1);
  if (n > 9223372036854775807.0 || n < -9223372036854775808.0)  /* max/min of type LONG LONG INT */
    luaL_error(L, "Error in " LUA_QS ": number too big or too small.", "math.isprime");
  if (trunc(n) != n)
    luaL_error(L, "Error in " LUA_QS ": integer expected.", "math.isprime");
  x = (long long int)n;
  if (x < 4)
    lua_pushboolean(L, x > 1);
  else if (x % 2 == 0)  /* even number ? */
    lua_pushboolean(L, 0);
  else {
    long long int i;
    const long long int imax = (long long int)sqrt(x) + 1;
    for (i = 3; i <= imax; i += 2) {
      if (x % i == 0) {
        lua_pushfalse(L);
        return 1;
      }
    }
    lua_pushtrue(L);
  }
  return 1;
}


static int math_prevprime (lua_State *L) {  /* 0.33.2, fixed and extended 1.0.6 */
  long long int x;
  lua_Number n;
  n = agnL_checknumber(L, 1);
  if (n > 9223372036854775807.0 || n < -9223372036854775808.0)  /* max/min of type LONG LONG INT */
    luaL_error(L, "Error in " LUA_QS ": number too big or too small.", "math.prevprime");
  if (trunc(n) != n)
    luaL_error(L, "Error in " LUA_QS ": integer expected.", "math.prevprime");
  x = (long long int)n;
  if (x < 3)
    lua_pushfail(L);
  else if (x == 3)
    lua_pushinteger(L, 2);
  else if (x < 6)
    lua_pushinteger(L, 3);
  else {
    unsigned long long int n, b, i;
    b = (x % 2 == 0) ? x - 1 : x - 2;
    for (i=b; ; i = i - 2) {
      n = 3;
      while (n * n < i && i % n != 0)
        n += 2;
      if (i % n != 0) break;
    }
    lua_pushnumber(L, i);  /* Agena 1.0.6 */
  }
  return 1;
}


static int math_nextprime (lua_State *L) {  /* 0.33.2, extended 1.0.6 */
  long long int x;
  lua_Number n;
  n = agnL_checknumber(L, 1);
  if (n > 9223372036854775807.0 || n < -9223372036854775808.0)  /* max/min of type LONG LONG INT */
    luaL_error(L, "Error in " LUA_QS ": number too big or too small.", "math.nextprime");
  if (trunc(n) != n)
    luaL_error(L, "Error in " LUA_QS ": integer expected.", "math.nextprime");
  x = (long long int)n;
  if (x < 2)
    lua_pushinteger(L, 2);
  else if (x < 3)  /* patched Agena 1.0.6 */
    lua_pushinteger(L, 3);
  else {
    unsigned long long int n, b, i;
    b = (x % 2 == 0) ? x + 1 : x + 2;
    for (i=b; ; i = i + 2) {
      n = 3;
      while (n * n < i && i % n != 0)
        n += 2;
      if (i % n != 0) break;
    }
    lua_pushnumber(L, i);  /* Agena 1.0.6 */
  }
  return 1;
}


/* constant found in Cephes Math Library Release 2.8, June, 2000 by Stephen L. Moshier
   file sin.c (and fixed it) */
#define P64800   (4.8481368110953599358991410e-6)  /* = (Pi/180)/3600 */

/* converts degrees to radians */
static int math_toradians (lua_State *L) {
  lua_Number m, s;
  m = agnL_optnumber(L, 2, 0);
  s = agnL_optnumber(L, 3, 0);
  lua_pushnumber(L,
    ((agn_checknumber(L, 1)*60.0 + m)*60.0 + s)*P64800
  );
  return 1;
}


/* converts sexagesimal to decimal */
static int math_todecimal (lua_State *L) {
  lua_Number m, s;
  m = agnL_optnumber(L, 2, 0);
  s = agnL_optnumber(L, 3, 0);
  lua_pushnumber(L, agn_checknumber(L, 1) + m/60 + s/3600);
  return 1;
}


/* taken from: @(#)s_nextafter.c 5.1 93/09/24
 *
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 *
 * IEEE functions
 *   nextafter(x,y)
 *   return the next machine floating-point number of x in the direction toward y.
 *   Special cases:
 *
 * Modification taken from www.koders.com /..GNU/l/libc/libc/math/s_nextafter.c
 * because the original Sun implementation is non-ANSI C and does not work with GCC.
 */

double sun_nextafter (double x, double y) {
  LUAI_INT32 hx, hy, ix, iy;
  LUAI_UINT32 lx, ly;
  EXTRACT_WORDS(hx, lx, x);
  EXTRACT_WORDS(hy, ly, y);
  ix = hx & 0x7fffffff;      /* |x| */
  iy = hy & 0x7fffffff;      /* |y| */
  if (((ix >= 0x7ff00000) && ((ix - 0x7ff00000) | lx) != 0) ||   /* x is nan */
     ((iy >= 0x7ff00000) && ((iy - 0x7ff00000) | ly) != 0))     /* y is nan */
    return x+y;
  if (x == y) return y;      /* x = y, return y */
  if ((ix | lx) == 0) {         /* x == 0 */
    double u;
    INSERT_WORDS(x, hy & 0x80000000, 1);   /* return +-minsubnormal */
    u = math_opt_barrier(x);
    u = u*u;
    math_force_eval(u);      /* raise underflow flag */
    return x;
  }
  if (hx >= 0) {            /* x > 0 */
    if (hx > hy || ((hx == hy) && (lx > ly))) {   /* x > y, x -= ulp */
      if (lx == 0) hx -= 1;
      lx -= 1;
    } else {            /* x < y, x += ulp */
      lx += 1;
      if (lx == 0) hx += 1;
    }
  } else {            /* x < 0 */
    if (hy >= 0 || hx > hy || ((hx == hy) && (lx > ly))) {  /* x < y, x -= ulp */
      if (lx == 0) hx -= 1;
      lx -= 1;
    } else {            /* x > y, x += ulp */
      lx += 1;
      if (lx == 0) hx += 1;
    }
  }
  hy = hx & 0x7ff00000;
  if (hy >= 0x7ff00000) {
    x = x + x;   /* overflow  */
    if (FLT_EVAL_METHOD != 0 && FLT_EVAL_METHOD != 1)
      asm ("" : "+m"(x));
    return x;   /* overflow  */
  }
  if (hy < 0x00100000) {
    double u = x * x;         /* underflow */
    math_force_eval(u);      /* raise underflow flag */
  }
  INSERT_WORDS(x, hx, lx);
  return x;
}


static int math_nextafter (lua_State *L) {
  lua_Number x, y;
  x = agn_checknumber(L, 1);
  y = agn_checknumber(L, 2);
  lua_pushnumber(L, sun_nextafter(x, y));
  return 1;
}


lua_Number getnumber (lua_State *L, int idx, int n, const char *str) {
  lua_Number r;
  agn_pairgeti(L, idx, n);
  if (lua_type(L, -1) != LUA_TNUMBER) {
    agn_poptop(L);  /* clear stack, Agena 1.1.0 */
    luaL_error(L, "Error in " LUA_QS ": expected a pair of numbers as %s argument.", "math.norm", str);
  }
  r = agn_tonumber(L, -1);
  agn_poptop(L);
  return r;
}


static int math_norm (lua_State *L) {
  lua_Number x, a1, a2, b1, b2;
  int t3;
  x = agn_checknumber(L, 1);
  if (!lua_ispair(L, 2))
    luaL_error(L, "Error in " LUA_QS ": expected a pair of numbers as second argument.", "math.norm");
  a1 = getnumber(L, 2, 1, "second");
  a2 = getnumber(L, 2, 2, "second");
  if (x < a1 || x > a2)
    luaL_error(L, "Error in " LUA_QS ": first argument out of range.", "math.norm");
  t3 = lua_type(L, 3);
  if (t3 == LUA_TNONE) {
    b1 = 0;
    b2 = 1;
  }
  else if (t3 == LUA_TPAIR) {
    b1 = getnumber(L, 3, 1, "third");
    b2 = getnumber(L, 3, 2, "third");
  }
  else {
    b1 = 0; b2 = 0;  /* just to prevent compiler warnings */
    luaL_error(L, "Error in " LUA_QS ": expected a pair of numbers as third argument.", "math.norm");
  }
  lua_pushnumber(L, (x-a1) * (b2-b1)/(a2-a1));
  return 1;
}


static const luaL_Reg mathlib[] = {
  {"isprime", math_isprime},              /* added on August 19, 2007 */
  {"max", math_max},
  {"min", math_min},
  {"nextafter", math_nextafter},          /* added on August 17, 2009 */
  {"nextprime", math_nextprime},          /* added on June 24, 2010 */
  {"norm", math_norm},                    /* added on May 22, 2010 */
  {"prevprime", math_prevprime},          /* added on June 24, 2010 */
  {"random", math_random},
  {"randomseed", math_randomseed},
  {"todecimal", math_todecimal},          /* added August 25, 2013 */
  {"toradians", math_toradians},          /* added on January 21, 2008 */
  {NULL, NULL}
};


/*
** Open math library
*/
LUALIB_API int luaopen_math (lua_State *L) {
  luaL_register(L, LUA_MATHLIBNAME, mathlib);
  lua_pushnumber(L, PHI);      /* added 0.5.0 */
  lua_setfield(L, -2, "Phi");  /* added 0.5.0 */
  return 1;
}


