/*
** $Id: llex.c,v 2.20 2006/03/09 18:14:31 roberto Exp $
** Lexical Analyzer
** See Copyright Notice in agena.h
*/


#include <ctype.h>
#include <locale.h>
#include <string.h>

#define llex_c
#define LUA_CORE

#include "agena.h"

#include "ldo.h"
#include "llex.h"
#include "lobject.h"
#include "lparser.h"
#include "lstate.h"
#include "lstring.h"
#include "ltable.h"
#include "lzio.h"

#define next(ls) (ls->current = zgetc(ls->z))

#define currIsNewline(ls)   (ls->current == '\n' || ls->current == '\r')

/* ORDER RESERVED */
const char *const luaX_tokens [] = {
    "abs", "alias", "and", "arccos", "arcsin", "arctan", "as", "assigned", "atendof",
    "bottom", "break", "by", "bye",
    "case", "catch", "char", "clear", "cls", "copy", "cos", "cosh", "create", "dec", "delete", "dict",
    "do", "elif", "else", "end", "entier", "enum", "epocs", "esac", "even", "exp",
    "fail", "false", "fi", "filled", "finite", "float", "for", "from", "gethigh", "getlow", "global",
    "if", "imag", "import", "in", "inc", "insert", "instr", "int", "intersect", "into", "is", "join",
    "keys", "left", "ln", "lngamma", "local", "log", "lower", "minus", "nargs", "not", "null",
    "od", "of", "onsuccess", "or", "pop", "proc", "qsadd", "real", "recip", "redo", "replace", "restart", "return", "right",
    "sadd", "scope", "seq", "sethigh", "setlow", "shift", "si",
    "sign", "sin", "sinh", "size", "skip", "split", "sqrt", "subset",
    "tan", "tanh", "then", "to", "top", "trim", "true", "try", "type", "typeof",
    "unassigned", "union", "unique", "until", "upper", "values", "while", "xor", "xsubset", "yrt",
    "boolean", "number", "complex", "string", "procedure",
    "userdata", "lightuserdata", "thread", "table", "sequence", "pair", "set",
    ":=", "~", "&", "?", "=", ">=", "<=", "<>", "==", "**", "@", "::", ":-",
    "&&", "~~", "||", "^^", "*%", "/%", "+%", "-%", "->", "<<", ">>", "..",
    "<number>", "<name>", "<string>", "<eof>",
    NULL
};

#define save_and_next(ls) (save(ls, ls->current), next(ls))


static void save (LexState *ls, int c) {
  Mbuffer *b = ls->buff;
  if (b->n + 1 > b->buffsize) {
    size_t newsize;
    if (b->buffsize >= MAX_SIZET/2)
      luaX_lexerror(ls, "lexical element too long", 0);
    newsize = b->buffsize * 2;
    luaZ_resizebuffer(ls->L, b, newsize);
  }
  b->buffer[b->n++] = cast(char, c);
}


void luaX_init (lua_State *L) {
  int i;
  for (i=0; i<NUM_RESERVED; i++) {
    TString *ts = luaS_new(L, luaX_tokens[i]);
    luaS_fix(ts);  /* reserved words are never collected */
    lua_assert(strlen(luaX_tokens[i])+1 <= TOKEN_LEN);
    ts->tsv.reserved = cast_byte(i+1);  /* reserved word */
  }
}


#define MAXSRC          80


const char *luaX_token2str (LexState *ls, int token) {
  if (token < FIRST_RESERVED) {
    lua_assert(token == cast(unsigned char, token));
    return (iscntrl(token)) ? luaO_pushfstring(ls->L, "char(%d)", token) :
                              luaO_pushfstring(ls->L, "%c", token);
  }
  else
    return luaX_tokens[token-FIRST_RESERVED];
}


static const char *txtToken (LexState *ls, int token) {
  switch (token) {
    case TK_NAME:
    case TK_STRING:
    case TK_NUMBER:
      save(ls, '\0');
      return luaZ_buffer(ls->buff);
    default:
      return luaX_token2str(ls, token);
  }
}


void luaX_lexerror (LexState *ls, const char *msg, int token) {
  char buff[MAXSRC];
  luaO_chunkid(buff, getstr(ls->source), MAXSRC);
  /* msg = luaO_pushfstring(ls->L, "%s:%d: %s", buff, ls->linenumber, msg); */
  msg = strcmp(buff, "stdin") ?
     luaO_pushfstring(ls->L, "Error in %s at line %d:\n   %s", buff, ls->linenumber, msg) :
     luaO_pushfstring(ls->L, "Error at line %d: %s", ls->linenumber, msg);
  if (token)
    luaO_pushfstring(ls->L, "%s near " LUA_QS, msg, txtToken(ls, token));
  luaD_throw(ls->L, LUA_ERRSYNTAX);
}


void luaX_syntaxerror (LexState *ls, const char *msg) {
  luaX_lexerror(ls, msg, ls->t.token);
}


TString *luaX_newstring (LexState *ls, const char *str, size_t l) {
  lua_State *L = ls->L;
  TString *ts = luaS_newlstr(L, str, l);
  TValue *o = luaH_setstr(L, ls->fs->h, ts);  /* entry for `str' */
  if (ttisnil(o)) {
    setbvalue(o, 1);  /* make sure `str' will not be collected */
    luaC_checkGC(L);  /* Lua 5.1.4 patch 6 */
  }
  return ts;
}


static void inclinenumber (LexState *ls) {
  int old = ls->current;
  lua_assert(currIsNewline(ls));
  next(ls);  /* skip `\n' or `\r' */
  if (currIsNewline(ls) && ls->current != old)
    next(ls);  /* skip `\n\r' or `\r\n' */
  if (++ls->linenumber >= MAX_INT)
    luaX_syntaxerror(ls, "chunk has too many lines");
}


void luaX_setinput (lua_State *L, LexState *ls, ZIO *z, TString *source) {
  ls->decpoint = '.';
  ls->L = L;
  ls->lookahead.token = TK_EOS;  /* no look-ahead token */
  ls->z = z;
  ls->fs = NULL;
  ls->linenumber = 1;
  ls->lastline = 1;
  ls->source = source;
  luaZ_resizebuffer(ls->L, ls->buff, LUA_MINBUFFER);  /* initialize buffer */
  next(ls);  /* read first char */
}

/*
** =======================================================
** LEXICAL ANALYZER
** =======================================================
*/



static int check_next (LexState *ls, const char *set) {
  if (!strchr(set, ls->current))
    return 0;
  save_and_next(ls);
  return 1;
}


static void buffreplace (LexState *ls, char from, char to) {
  size_t n = luaZ_bufflen(ls->buff);
  char *p = luaZ_buffer(ls->buff);
  while (n--)
    if (p[n] == from) p[n] = to;
}


static void trydecpoint (LexState *ls, SemInfo *seminfo) {
  /* format error: try to update decimal point separator */
  struct lconv *cv = localeconv();
  char old = ls->decpoint;
  ls->decpoint = (cv ? cv->decimal_point[0] : '.');
  buffreplace(ls, old, ls->decpoint);  /* try updated decimal separator */
  if (!luaO_str2d(luaZ_buffer(ls->buff), &seminfo->r)) {
    /* format error with correct decimal point: no more options */
    buffreplace(ls, ls->decpoint, '.');  /* undo change (for error message) */
    luaX_lexerror(ls, "malformed number", TK_NUMBER);
  }
}


/* LUA_NUMBER */
static void read_numeral (LexState *ls, SemInfo *seminfo) {
  lua_assert(isdigit(ls->current));
  do {
    if (ls->current != '\'')  /* ignore single quotes in number; added 0.7.1 */
      save(ls, ls->current);
    next(ls);
    /* save_and_next(ls); */
  } while (isdigit(ls->current) || ls->current == '.' || ls->current == '\'');
  if (check_next(ls, "Ee"))  /* `E'? */
    check_next(ls, "+-");  /* optional exponent sign */
  while (isalnum(ls->current) || ls->current == '_')
    save_and_next(ls);
  save(ls, '\0');
  buffreplace(ls, '.', ls->decpoint);  /* follow locale for decimal point */
  if (!luaO_str2d(luaZ_buffer(ls->buff), &seminfo->r))  /* format error? */
    trydecpoint(ls, seminfo); /* try to update decimal point separator */
}


static void read_long_string (LexState *ls, SemInfo *seminfo, int sep) {  /* for comments and strings, optimised 1.6.7 */
  int quit = 0;
  save_and_next(ls);  /* skip 2nd `[' */
  if (currIsNewline(ls))  /* string starts with a newline? */
    inclinenumber(ls);  /* skip it */
  for (;;) {
    switch (ls->current) {
      case EOZ:
        luaX_lexerror(ls, (seminfo) ? "unfinished long string" :
                                   "unfinished long comment", TK_EOS);
        break;  /* to avoid warnings */
      case '/': {
        next(ls);
        if (ls->current == '#') {
          save_and_next(ls);  /* skip `#' */
          goto endloop;
        } else
          save(ls, '/');
        break;
      }
      case '\\': {
        int c;
        next(ls);  /* do not save the `\' */
        switch (ls->current) {
          case 'a': c = '\a'; break;
          case 'b': c = '\b'; break;
          case 'f': c = '\f'; break;
          case 'n': c = '\n'; break;
          case 'r': c = '\r'; break;
          case 't': c = '\t'; break;
          case 'v': c = '\v'; break;
          case '\n':  /* go through */
          case '\r': save(ls, '\n'); inclinenumber(ls); continue;
          case EOZ: continue;  /* will raise an error next loop */
          default: {
            if (!isdigit(ls->current))
              save_and_next(ls);  /* handles \\, \", \', and \? */
            else {  /* \xxx */
              int i = 0;
              c = 0;
              do {
                c = 10*c + (ls->current-'0');
                next(ls);
              } while (++i<3 && isdigit(ls->current));
              if (c > UCHAR_MAX)
                luaX_lexerror(ls, "escape sequence too large", TK_STRING);
              save(ls, c);
            }
            continue;
          }
        }
        save(ls, c);
        next(ls);
        continue;
      }
      case '\n':
      case '\r': {
        save(ls, '\n');
        inclinenumber(ls);
        if (!seminfo) luaZ_resetbuffer(ls->buff);  /* avoid wasting space */
        break;
      }
      case '"': case '\'': {
        if (seminfo) {  /* are we processing a string or a comment ? */
          save_and_next(ls);
          quit = 1; sep = 0;
          break;
        } /* else go through */
      }
      default: {
        if (seminfo) save_and_next(ls);
        else next(ls);
      }
    }
    if (quit) break;
  } endloop:
  if (seminfo) {
    seminfo->ts = luaX_newstring(ls, luaZ_buffer(ls->buff) + (1 + sep),  /* Valgrind, Agena 1.6.6 */
                                     luaZ_bufflen(ls->buff) - 2*(1 + sep));
  }
}


/* shortcut string definition, 0.8.0, December 18, 2007 */
static void read_shortstring (LexState *ls, SemInfo *seminfo) {
  save_and_next(ls);
  /* end parsing if one of the following tokens or if EOZ (end of stream) was found */
  while (strchr(AGN_SHORTSTRINGDELIM, ls->current) == NULL && (ls->current != EOZ)) {
    save_and_next(ls);
  }
  seminfo->ts = luaX_newstring(ls, luaZ_buffer(ls->buff) + 1,
                                   luaZ_bufflen(ls->buff) - 1);
}


static int llex (LexState *ls, SemInfo *seminfo) {
  luaZ_resetbuffer(ls->buff);
  for (;;) {
    switch (ls->current) {
      case '\n':
      case '\r': {
        inclinenumber(ls);
        continue;
      }
      case '#': {   /* changed */
        next(ls);
        if (ls->current == '/') {  /* long comment, changed, Valgrind 1.6.6 */
          luaZ_resetbuffer(ls->buff);  /* `skip_sep' may dirty the buffer */
          read_long_string(ls, NULL, 0);  /* long comment */
          luaZ_resetbuffer(ls->buff);
          continue;
        }
        /* else short comment */
        while (!currIsNewline(ls) && ls->current != EOZ)
          next(ls);
        continue;
      }
      case '=': {
        next(ls);
        if (ls->current == '=') { next(ls); return TK_EEQ; }
        else return TK_EQ;
      }
      case ':': {  /* changed */
        next(ls);
        if (ls->current == '=') { next(ls); return TK_ASSIGN; }
        if (ls->current == ':') { next(ls); return TK_OFTYPE; }
        if (ls->current == '-') { next(ls); return TK_NOTOFTYPE; }
        else return ':';
      }
      case '*': {  /* changed */
        next(ls);
        if (ls->current == '*') { next(ls); return TK_IPOW; }
        if (ls->current == '%') { next(ls); return TK_PERCENT; }
        else return '*';
      }
      case '-': {  /* changed */
        next(ls);
        if (ls->current == '%') { next(ls); return TK_PERCENTSUB; }
        if (ls->current == '>') { next(ls); return TK_ARROW; }
        else return '-';
      }
      case '<': {
        next(ls);
        /* changed Oct. 12, 2006 and June 17, 2007 */
        if (ls->current == '=') { next(ls); return TK_LE; }
        if (ls->current == '>') { next(ls); return TK_NEQ; } /* changed 0.4.0 */
        if (ls->current == '<') { next(ls); return TK_LT2; }  /* changed 0.5.2 */
        else return '<';
      }
      case '>': {
        next(ls);
        if (ls->current == '=') { next(ls); return TK_GE; }
        if (ls->current == '>') { next(ls); return TK_GT2; }
        else return '>';
      }
      case '`': {  /* added 0.8.0 */
        read_shortstring(ls, seminfo);
        return TK_STRING;
      }
      case '\'': case '\"': {  /* changed 0.25.5 */
        luaZ_resetbuffer(ls->buff);
        read_long_string(ls, seminfo, ls->current);
        return TK_STRING;
      }
      case '&': {    /* changed 0.4.0 */
        next(ls);
        if (ls->current == '&') { next(ls); return TK_BAND; }
        else return TK_CONCAT;
      }
      case '|': {    /* changed 0.4.0 */
        next(ls);
        if (ls->current == '|') { next(ls); return TK_BOR; }
        else return '|';
      }
      case '^': {    /* changed 0.4.0 */
        next(ls);
        if (ls->current == '^') { next(ls); return TK_BXOR; }
        else return '^';
      }
      case '/': {    /* changed 1.10.6 */
        next(ls);
        if (ls->current == '%') { next(ls); return TK_PERCENTRATIO; }
        else return '/';
      }
      case '+': {    /* changed 1.11.3 */
        next(ls);
        if (ls->current == '%') { next(ls); return TK_PERCENTADD; }
        else return '+';
      }
      case '~': {    /* changed 0.4.0 */
        next(ls);
        if (ls->current == '~') { next(ls); return TK_BNOT; }
        else return TK_SEP;
      }
      case '@': {    /* changed 0.4.0 */
        next(ls);
        return TK_OO;
      }
      case '.': {
        save_and_next(ls);
        if (ls->current == '.') { next(ls); return TK_DD; }
        if (!isdigit(ls->current)) return '.';
        else {
          read_numeral(ls, seminfo);
          return TK_NUMBER;
        }
      }
      case EOZ: {
        return TK_EOS;
      }
      case '?': {  /* changed 0.4.0 */
         next(ls);
         return TK_DOTS;
      }
      default: {
        if (isspace(ls->current)) {
          lua_assert(!currIsNewline(ls));
          next(ls);
          continue;
        }
        else if (isdigit(ls->current)) {
          read_numeral(ls, seminfo);
          return TK_NUMBER;
        }
        else if (isalpha(ls->current) || ls->current == '_' ) { /* || ls->current == '@' ) {  changed 0.4.1 */
          /* identifier or reserved word */
          TString *ts;
          do {
            save_and_next(ls);
          } while (isalnum(ls->current) || ls->current == '_');
          ts = luaX_newstring(ls, luaZ_buffer(ls->buff),
                                  luaZ_bufflen(ls->buff));
          if (ts->tsv.reserved > 0)  /* reserved word? */
            return ts->tsv.reserved - 1 + FIRST_RESERVED;
          else {
            seminfo->ts = ts;
            return TK_NAME;
          }
        }
        else {
          int c = ls->current;
          next(ls);
          return c;  /* single-char tokens (+ - / ...) */
        }
      }
    }
  }
}


void luaX_next (LexState *ls) {
  ls->lastline = ls->linenumber;
  if (ls->lookahead.token != TK_EOS) {  /* is there a look-ahead token? */
    ls->t = ls->lookahead;  /* use this one */
    ls->lookahead.token = TK_EOS;  /* and discharge it */
  }
  else {
    ls->t.token = llex(ls, &ls->t.seminfo);  /* read next token */
  }
}


void luaX_lookahead (LexState *ls) {
  lua_assert(ls->lookahead.token == TK_EOS);
  ls->lookahead.token = llex(ls, &ls->lookahead.seminfo);
}

