/*
** $Id: liolib.c,v 2.73 2006/05/08 20:14:16 roberto Exp $
** Standard I/O (and system) library
** See Copyright Notice in agena.h
*/


#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>  /* for ftruncate */

#if defined(_WIN32) || defined(__OS2__)
#include <conio.h>   /* getch; a UNIX version is included in agnhlps.c */
#include <windows.h>  /* for io_putclip */
#endif


#define liolib_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"

#include "agnhlps.h"  /* for getch() */

#define IO_INPUT   1
#define IO_OUTPUT  2

static const char *const fnames[] = {"input", "output"};

#define uchar(c)        ((unsigned char)(c))

/* mapping table to correctly read in diacritics from text files in Windows */

static unsigned char f2c[256] = {
  0, 1, 2, 3, 4, 5, 6, 7, 8, 9,
  10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
  20, 21, 22, 23, 24, 25, 26, 27, 28, 29,
  30, 31, 32, 33, 34, 35, 36, 37, 38, 39,
  40, 41, 42, 43, 44, 45, 46, 47, 48, 49,
  50, 51, 52, 53, 54, 55, 56, 57, 58, 59,
  60, 61, 62, 63, 64, 65, 66, 67, 68, 69,
  70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
  80, 81, 82, 83, 84, 85, 86, 87, 88, 89,
  90, 91, 92, 93, 94, 95, 96, 97, 98, 99,
  100, 101, 102, 103, 104, 105, 106, 107, 108, 109,  /* 100 .. 109 */
  110, 111, 112, 113, 114, 115, 116, 117, 118, 119,  /* 110 .. 119 */
  120, 121, 122, 123, 124, 125, 126, 127, 128, 129,  /* 120 .. 129 */
  130, 131, 132, 133, 134, 135, 136, 137, 138, 139,  /* 130 .. 139 */
  153, 141, 142, 143, 144, 145, 146, 147, 148, 149,  /* 140 .. 149 */
  150, 151, 152, 153, 154, 155, 148, 157, 158, 159,  /* 150 .. 159 */
  160, 161, 162, 163, 164, 165, 166, 245, 168, 169,  /* 160 .. 169 */
  170, 171, 172, 173, 174, 175, 176, 177, 178, 179,  /* 170 .. 179 */
  180, 181, 182, 183, 184, 185, 186, 187, 188, 189,  /* 180 .. 189 */
  190, 168, 183, 181, 182, 199, 142, 143, 146, 128,  /* 190 .. 199 */
  212, 144, 210, 211, 222, 214, 215, 216, 209, 165,  /* 200 .. 209 */
  227, 224, 226, 229, 153, 215, 157, 235, 233, 234,  /* 210 .. 219 */
  154, 237, 232, 225, 133, 160, 131, 198, 132, 134,  /* 220 .. 229 */
  145, 135, 138, 130, 136, 137, 141, 161, 140, 139,  /* 230 .. 239 */
  208, 164, 149, 162, 147, 228, 148, 247, 155, 151,  /* 240 .. 249 */
  163, 150, 129, 236, 231, 152                       /* 250 .. 255 */
};


static int pushresult (lua_State *L, int i, const char *filename) {
  int en = errno;  /* calls to Lua API may change this value */
  if (i) {
    lua_pushboolean(L, 1);
    return 1;
  }
  else {
    lua_pushnil(L);
    if (filename)
      lua_pushfstring(L, "%s: %s", filename, strerror(en));
    else
      lua_pushfstring(L, "%s", strerror(en));
    lua_pushinteger(L, en);
    return 3;
  }
}


static int issueioerror (lua_State *L, const char *filename, const char *fn) {
  int en = errno;  /* calls to Lua API may change this value */
  if (filename)
    luaL_error(L, "Error in " LUA_QS ": '%s': %s.", fn, filename, strerror(en));
  else
    luaL_error(L, "Error in " LUA_QS ": %s.", fn, strerror(en));
  return 2;
}


static void fileerror (lua_State *L, int arg, const char *filename) {
  lua_pushfstring(L, "%s: %s", filename, strerror(errno));
  luaL_argerror(L, arg, lua_tostring(L, -1));
}


#define topfile(L)   ((FILE **)luaL_checkudata(L, 1, LUA_FILEHANDLE))
#define topnfile(L, n)   ((FILE **)luaL_checkudata(L, n, LUA_FILEHANDLE))

static FILE *tofile (lua_State *L) {
  FILE **f = topfile(L);
  if (*f == NULL)
    luaL_error(L, "Error in " LUA_QS " package: attempt to use a closed file.", "io");
  return *f;
}



/*
** When creating file handles, always creates a `closed' file handle
** before opening the actual file; so, if there is a memory error, the
** file is not left opened.
*/
static FILE **newfile (lua_State *L) {
  FILE **pf = (FILE **)lua_newuserdata(L, sizeof(FILE *));
  *pf = NULL;  /* file handle is currently `closed' */
  luaL_getmetatable(L, LUA_FILEHANDLE);
  lua_setmetatable(L, -2);
  return pf;
}


/*
** this function has a separated environment, which defines the
** correct __close for 'popen' files
*/
static int io_pclose (lua_State *L) {
  FILE **p = topfile(L);
  int ok = lua_pclose(L, *p);
  *p = NULL;
  return pushresult(L, ok, NULL);
}


static int io_fclose (lua_State *L) {  /* 1.0.2, extended Agena 1.1.0, 1.6.0 */
  int i, nargs;
  nargs = lua_gettop(L) - 2;
  for (i=0; i < nargs; i++) {
    FILE **p = topnfile(L, i+1);
    int ok = (fclose(*p) == 0);
    if (ok) {
      /* 0.20.2, avoid segmentation faults; 1.6.4, it is not possible to issue an error, obviously io has its own stack ??? ! */
      if (agnL_gettablefield(L, "io", "openfiles", "io.close", 0) == LUA_TTABLE) {
        /* DO NOT use lua_istable since it checks a value on the stack and not the return value of agn_gettablefield (a nonneg int) */
        lua_pushfstring(L, "file(%p)", *p);
        lua_pushnil(L);
        lua_rawset(L, -3);
      }
      agn_poptop(L);  /* delete "openfiles" */
    }
    *p = NULL;
    pushresult(L, ok, NULL);
  }
  return nargs;
}


static int aux_close (lua_State *L) {
  lua_getfenv(L, 1);
  lua_getfield(L, -1, "__close");
  return (lua_tocfunction(L, -1))(L);
}


static int io_close (lua_State *L) {
  if (lua_isnone(L, 1))
    lua_rawgeti(L, LUA_ENVIRONINDEX, IO_OUTPUT);
  tofile(L);  /* make sure argument is a file */
  return aux_close(L);
}


static int io_gc (lua_State *L) {  /* 0.24.2 */
  FILE *f = *topfile(L);
  /* ignore closed files */
  if (f != NULL)
    aux_close(L);
  return 0;
}


static int io_tostring (lua_State *L) {  /* 0.24.2 */
  FILE *f = *topfile(L);
  if (f == NULL)
    lua_pushliteral(L, "file(closed)");
  else
    lua_pushfstring(L, "file(%p)", f);
  return 1;
}


static int io_open (lua_State *L) {
  const char *filename = luaL_checkstring(L, 1);
  const char *mode = luaL_optstring(L, 2, "r");
  FILE **pf = newfile(L);
  *pf = fopen(filename, mode);
  if (*pf == NULL) issueioerror(L, filename, "io.open");
  /* enter new open file to global io.openfiles table
     0.20.2, avoid segmentation faults; 1.6.4, it is not possible to issue an error, obviously io has its own stack ??? ! */
  if (agnL_gettablefield(L, "io", "openfiles", "io.open", 0) == LUA_TTABLE) {
    /* DO NOT use lua_istable since it checks a value on the stack and not the return value of agn_gettablefield (a nonneg int) */
    lua_pushfstring(L, "file(%p)", *pf);
    lua_createtable(L, 2, 0);
    lua_pushstring(L, filename);
    lua_rawseti(L, -2, 1);
    lua_pushstring(L, mode);
    lua_rawseti(L, -2, 2);
    lua_rawset(L, -3);
  }
  agn_poptop(L);  /* delete "openfiles" */
  return 1;
}


static int io_popen (lua_State *L) {
  const char *filename = luaL_checkstring(L, 1);
  const char *mode = luaL_optstring(L, 2, "r");
  FILE **pf = newfile(L);
  *pf = lua_popen(L, filename, mode);
  return (*pf == NULL) ? pushresult(L, 0, filename) : 1;
}


static int io_tmpfile (lua_State *L) {
  FILE **pf = newfile(L);
  *pf = tmpfile();
  return (*pf == NULL) ? pushresult(L, 0, NULL) : 1;
}


static FILE *getiofile (lua_State *L, int findex) {
  FILE *f;
  lua_rawgeti(L, LUA_ENVIRONINDEX, findex);
  f = *(FILE **)lua_touserdata(L, -1);
  if (f == NULL)
    luaL_error(L, "Error in " LUA_QS " package: standard %s file is closed.", "io", fnames[findex - 1]);
  return f;
}


static int g_iofile (lua_State *L, int f, const char *mode) {
  if (!lua_isnoneornil(L, 1)) {
    const char *filename = lua_tostring(L, 1);
    if (filename) {
      FILE **pf = newfile(L);
      *pf = fopen(filename, mode);
      if (*pf == NULL)
        fileerror(L, 1, filename);
    }
    else {
      tofile(L);  /* check that it's a valid file handle */
      lua_pushvalue(L, 1);
    }
    lua_rawseti(L, LUA_ENVIRONINDEX, f);
  }
  /* return current value */
  lua_rawgeti(L, LUA_ENVIRONINDEX, f);
  return 1;
}


static int io_input (lua_State *L) {
  return g_iofile(L, IO_INPUT, "r");
}


static int io_output (lua_State *L) {
  return g_iofile(L, IO_OUTPUT, "w");
}


static int io_auxreadline (lua_State *L);


static void aux_lines (lua_State *L, int idx, int toclose) {
  lua_pushvalue(L, idx);  /* push userdata object */
  lua_pushboolean(L, toclose);  /* close/not close file when finished */
  lua_pushcclosure(L, io_auxreadline, 2);
}


static int io_lines (lua_State *L) {
  if (lua_isnoneornil(L, 1)) {  /* no arguments? */
    /* will iterate over default input */
    lua_rawgeti(L, LUA_ENVIRONINDEX, IO_INPUT);
    tofile(L);  /* check that it's a valid file handle */
    aux_lines(L, 1, 0);
  }
  else if (lua_type(L, 1) == LUA_TSTRING) {
    const char *filename = agn_checkstring(L, 1);
    FILE **pf = newfile(L);
    *pf = fopen(filename, "r");
    if (*pf == NULL)
      fileerror(L, 1, filename);
    aux_lines(L, lua_gettop(L), 1);  /* the FILE userdata is at the top of the stack */
  }
  else {
    int result;
    FILE **f = (FILE**)luaL_getudata(L, 1, LUA_FILEHANDLE, &result);
    if (result) {  /* first argument is a file handle ? */
      if (*f == NULL) {  /* but a closed file ? */
        luaL_error(L, "Error in " LUA_QS ": attempt to use a closed file.", "io.lines");
      } else  /* read line */
        aux_lines(L, 1, 0);  /* but do not close it when done */
    }
    else
      luaL_error(L, "Error in " LUA_QS ": file handle expected.", "io.lines");
  }
  return 1;
}


/*
** {======================================================
** READ
** =======================================================
*/


static int read_number (lua_State *L, FILE *f) {
  lua_Number d;
  if (fscanf(f, LUA_NUMBER_SCAN, &d) == 1) {
    lua_pushnumber(L, d);
    return 1;
  }
  else {  /* Lua 5.1.4 patch 8 */
    lua_pushnil(L);  /* "result" to be removed */
    return 0;  /* read fails */
  }
}


static int test_eof (lua_State *L, FILE *f) {
  int c = getc(f);
  ungetc(c, f);
  lua_pushlstring(L, NULL, 0);
  return (c != EOF);
}


static int read_line (lua_State *L, FILE *f) {
  luaL_Buffer b;
  luaL_buffinit(L, &b);
  for (;;) {
    size_t l;
    char *p = luaL_prepbuffer(&b);
    if (fgets(p, LUAL_BUFFERSIZE, f) == NULL) {  /* eof? */
      luaL_pushresult(&b);  /* close buffer */
      return (lua_strlen(L, -1) > 0);  /* check whether read something */
    }
    l = strlen(p);
    if (l == 0 || p[l-1] != '\n')
      luaL_addsize(&b, l);
    else {
      if (l > 1 && p[l-2] == '\r')  /* Agena 1.2.1 patch; added in 0.5.4 for correctly reading DOS files on UNIX platforms */
        luaL_addsize(&b, l - 2);  /* do not include `eol' */
      else
        luaL_addsize(&b, l - 1);  /* do not include `eol' */
      luaL_pushresult(&b);  /* close buffer */
      return 1;  /* read at least an `eol' */
    }
  }
}


static int read_chars (lua_State *L, FILE *f, size_t n) {
  size_t rlen;  /* how much to read */
  size_t nr;  /* number of chars actually read */
  luaL_Buffer b;
  luaL_buffinit(L, &b);
  rlen = LUAL_BUFFERSIZE;  /* try to read that much each time */
  do {
    char *p = luaL_prepbuffer(&b);
    if (rlen > n) rlen = n;  /* cannot read more than asked */
    nr = fread(p, sizeof(char), rlen, f);
    luaL_addsize(&b, nr);
    n -= nr;  /* still have to read `n' chars */
  } while (n > 0 && nr == rlen);  /* until end of count or eof */
  luaL_pushresult(&b);  /* close buffer */
  return (n == 0 || lua_strlen(L, -1) > 0);
}


static int g_read (lua_State *L, FILE *f, int first) {
  int nargs, success, n;
  nargs = lua_gettop(L) - 1;
  clearerr(f);
  if (nargs == 0) {  /* no arguments? */
    success = read_line(L, f);
    n = first+1;  /* to return 1 result */
  }
  else {  /* ensure stack space for all results and for auxlib's buffer */
    luaL_checkstack(L, nargs+LUA_MINSTACK, "too many arguments");
    success = 1;
    for (n = first; nargs-- && success; n++) {
      if (lua_type(L, n) == LUA_TNUMBER) {
        size_t l = (size_t)lua_tointeger(L, n);
        success = (l == 0) ? test_eof(L, f) : read_chars(L, f, l);
      }
      else {
        const char *p = lua_tostring(L, n);
        luaL_argcheck(L, p && p[0] == '*', n, "invalid option");
        switch (p[1]) {
          case 'n':  /* number */
            success = read_number(L, f);
            break;
          case 'l':  /* line */
            success = read_line(L, f);
            break;
          case 'a':  /* file */
            read_chars(L, f, ~((size_t)0));  /* read MAX_SIZE_T chars */
            success = 1; /* always success */
            break;
          default:
            return luaL_argerror(L, n, "invalid format");
        }
      }
    }
  }
  if (ferror(f))
    return pushresult(L, 0, NULL);
  if (!success) {
    agn_poptop(L);  /* remove last result */
    lua_pushnil(L);  /* push nil instead */
  }
  return n - first;
}


static int io_read (lua_State *L) {
  if (lua_isnoneornil(L, 1)) {  /* no arguments? */
    /* will iterate over default input */
    if (agn_getgui(L))
      luaL_error(L, "Error in " LUA_QS ": keyboard input not supported by AgenaEdit.", "io.read");
    return g_read(L, getiofile(L, IO_INPUT), 1);
  }
  else {
    int result;
    FILE **pf = (FILE **)luaL_getudata(L, 1, LUA_FILEHANDLE, &result);
    if (result) {  /* first argument is a file handle ? */
      if (*pf == NULL) {  /* but a closed file ? */
        luaL_error(L, "Error in " LUA_QS " package: attempt to use a closed file.", "io");
      } else  /* read line */
        return g_read(L, *pf, 2);
    }
    else
      luaL_error(L, "Error in " LUA_QS " package: file handle expected.", "io");
  }
  return 1;
}


/* readfile: 1.10.2, 1 % faster on Windows than the Agena version, 8 % faster on Mac OS X;
   1.11.2 extended to return file contents without line breaks */
static int io_readfile (lua_State *L) {
  const char *filename, *pattern;
  int r, flag, removenls;
  FILE *pf;
  filename = agn_checkstring(L, 1);
  removenls = agnL_optboolean(L, 2, 0);  /* remove carriage returns and/or newlines */
  pattern = luaL_optstring(L, 3, NULL);
  flag = agnL_optboolean(L, 4, 1);  /* return only matching file contents, default = true */
  pf = fopen(filename, "rb");
  if (pf == NULL)
    luaL_error(L, "Error in " LUA_QS ": file %s could not be opened.", "io.readfile", filename);
  if (removenls) {
    int getout, strlength, linelength;
    char buffer[LUAL_BUFFERSIZE];
    char *line;
    luaL_Buffer b;
    luaL_buffinit(L, &b);
    r = getout = 0;
    while (fgets(buffer, sizeof(buffer), pf) != '\0') {  /* eof not yet reached ?  sizeof(buffer) is the length of the char array.*/
      strlength = linelength = strlen(buffer);
      line = (char *)malloc((strlength+1)*sizeof(char));
      if (line == NULL)
        luaL_error(L, "Error in " LUA_QS ": buffer allocation error.", "io.readfile");
      strcpy(line, buffer);
      while (strlength > 0 && uchar(*(buffer+strlength-1)) != '\n') {
        if (fgets(buffer, sizeof(buffer), pf) == '\0') {  /* eof reached ?  sizeof(buffer) is the length of the char array. */
          getout = 1;  /* but process last line in file like the other ones */
          line[linelength] = '\0';
          break;
        }
        strlength = strlen(buffer);
        linelength += strlength;
        line = (char *)realloc(line, (linelength+1)*sizeof(char));
        if (line == NULL)
          luaL_error(L, "Error in " LUA_QS ": buffer re-allocation error.", "io.readfile");
        strcat(line, buffer);
      }
      /* delete carriage return (in DOS files) */
      if (linelength > 1 && line[linelength-2] == '\r') {
        line[linelength-2] = line[linelength-1];
        linelength--;
      }
      /* delete newline */
      if (linelength > 0 && line[linelength-1] == '\n') {
        linelength--;
      }
      luaL_addlstring(&b, line, linelength);
      xfree(line);
      if (getout) break;
    }  /* while eof not yet reached */
    luaL_pushresult(&b);
    r = 1;
  } else {
    r = read_chars(L, pf, ~((size_t)0));  /* read MAX_SIZE_T chars */
  }
  if (pattern != NULL && r == 1) {  /* 1.11.6 */
    char *lookup;
    const char *str = lua_tostring(L, -1);
    if ((lookup = strstr(str, pattern)) == NULL) {  /* specific substring in file not found ? */
      if (flag == 1) {  /* return only matching file contents ? */
        agn_poptop(L);  /* pop string and return `null` */
        lua_pushnil(L);
      }  /* else leave file contents on stack */
    } else {  /* pattern found ? */
      if (flag == 1) {  /* return only matching file contents ? */
        lua_pushnumber(L, lookup - str + 1);  /* leave string on stack and also push position where the match starts */
        r = 2;
      } else {
        agn_poptop(L);  /* pop string and return `null` */
        lua_pushnil(L);
      }
    }
  }
  if (ferror(pf))
    luaL_error(L, "Error in " LUA_QS ": file %s could possibly not be read.", "io.readfile", filename);
  if (fclose(pf) != 0)
    luaL_error(L, "Error in " LUA_QS ": file %s could not be closed.", "io.readfile", filename);
  return r;
}


static int io_infile (lua_State *L) {  /* 1.11.6 */
  const char *filename, *pattern;
  char buffer[LUAL_BUFFERSIZE];
  char *line;
  int getout, strlength, linelength, found;
  size_t pl, maxlen;
  FILE *pf;
  filename = agn_checkstring(L, 1);
  pattern = agn_checklstring(L, 2, &pl);
  if (pl == 0)
    luaL_error(L, "Error in " LUA_QS ": pattern must not be the empty string.", "io.infile", filename);
  pf = fopen(filename, "rb");
  getout = found = 0;
  if (pf == NULL)
    luaL_error(L, "Error in " LUA_QS ": file %s could not be opened.", "io.infile", filename);
  maxlen = 0;
  line = NULL;  /* to avoid compiler warnings */
  while (fgets(buffer, sizeof(buffer), pf) != '\0') {  /* eof not yet reached ?  sizeof(buffer) is the length of the char array. */
    strlength = linelength = strlen(buffer);
    if (strlength < LUAL_BUFFERSIZE - 1) {  /* 1.12.5 patch */
      if (strstr(buffer, pattern) != NULL) {
        found = 1;
        break;
      } else
        continue;  /* read next line */
    }
    /* only build up `line` if line exceeds LUAL_BUFFERSIZE-1 chars */
    if (maxlen < linelength) {
      if (maxlen != 0) xfree(line);  /* line already used ? */
      line = (char *)malloc((strlength+1)*sizeof(char));  /* Agena 1.6.4 */
      if (line == NULL)  /* 0.27.2 */
        luaL_error(L, "Error in " LUA_QS ": buffer allocation error.", "io.infile");
      maxlen = linelength;
    }
    strcpy(line, buffer);
    while (strlength > 0 && uchar(*(buffer+strlength-1)) != '\n') {  /* 0.27.2, eol not yet reached ? -> expand buffer, 1.6.11 Valgrind */
      if (fgets(buffer, sizeof(buffer), pf) == '\0') {  /* eof reached ?  sizeof(buffer) is the length of the char array including the
        terminating \0 that is automatically added by fgets. */
        getout = 1;  /* but process last line in file like the other ones */
        line[linelength] = '\0';  /* 1.6.11 better sure than sorry */
        break;
      }
      strlength = strlen(buffer);
      linelength += strlength;
      if (maxlen < linelength) {
        line = (char *)realloc(line, (linelength+1)*sizeof(char));  /* 1.6.4 */
        if (line == NULL)  /* 0.27.2 */
          luaL_error(L, "Error in " LUA_QS ": buffer re-allocation error.", "io.infile");
        maxlen = linelength;
      }
      strcat(line, buffer);
    }
    if (strstr(line, pattern) != NULL) {
      getout = found = 1;
    }
    if (getout) break;  /* 0.27.2 */
  }
  if (maxlen != 0) xfree(line);
  if (ferror(pf))
    luaL_error(L, "Error in " LUA_QS ": file %s could possibly not be read.", "io.infile", filename);
  if (fclose(pf) != 0)
    luaL_error(L, "Error in " LUA_QS ": file %s could not be closed.", "io.infile", filename);
  lua_pushboolean(L, found);
  return 1;
}


static int io_auxreadline (lua_State *L) {
  FILE *f = *(FILE **)lua_touserdata(L, lua_upvalueindex(1));
  int sucess;
  if (f == NULL)  /* file is already closed? */
    luaL_error(L, "Error in " LUA_QS " package: file is already closed.", "io");
  sucess = read_line(L, f);
  if (ferror(f))
    return luaL_error(L, "%s.", strerror(errno));
  if (sucess) return 1;
  else {  /* EOF */
    if (lua_toboolean(L, lua_upvalueindex(2))) {  /* generator created file? */
      lua_settop(L, 0);
      lua_pushvalue(L, lua_upvalueindex(1));
      aux_close(L);  /* close it */
    }
    return 0;
  }
}

/* }====================================================== */


static int g_write (lua_State *L, FILE *f, int arg, int flag, char *delim) {
  int nargs, status;
  nargs = lua_gettop(L) - 1 - 1*(delim != NULL);  /* 0.30.4 */
  /* subtract 1 because if an option is given, it is still on the stack */
  status = 1;
  for (; nargs--; arg++) {  /* 0.26.0 simplification */
    size_t l;
    const char *s = luaL_checklstring(L, arg, &l);
    status = status && (fwrite(s, sizeof(char), l, f) == l);
    if (delim != NULL && nargs != 0) /* do not print delim after last argument */
      fwrite(delim, sizeof(char), strlen(delim), f);
  }
  if (flag) fwrite("\n", sizeof(char), 1, f);
  if (delim != NULL) xfree(delim);
  return pushresult(L, status, NULL);
}


/* extended November 22, 2008 - 0.12.2; patched 11.07.2009 (0.24.3), patched 31.01.2010 (0.30.4), patched
   14.11.2011 (1.5.1)
   newline = 0 -> no newline, 1 -> insert a newline */
static int io_writeaux (lua_State *L, int newline) {
  int result, nargs;
  char *delim;
  delim = NULL;  /* to avoid compiler warnings */
  nargs = lua_gettop(L);
  if (nargs > 0 && lua_ispair(L, nargs)) {  /* 0.24.3 patch */
    agn_pairgeti(L, nargs, 1);  /* get left value, set to stack index -2 */
    agn_pairgeti(L, nargs, 2);  /* get right value, set to stack index  -1 */
    if (strcmp("delim", agn_checkstring(L, -2)) != 0)
      luaL_error(L, "Error in " LUA_QS ": unknown option.", (newline == 0) ? "write" : "writeline");  /* 1.5.1 */
    else
      delim = strdup(agn_checkstring(L, -1));  /* free() see g_write; `Agena 1.0.4, 1.5.1 */
    agn_poptoptwo(L);  /* 0.30.4 */
  }
  FILE **f = (FILE **)luaL_getudata(L, 1, LUA_FILEHANDLE, &result);
  if (result) {  /* first argument is a file handle ? */
    if (*f == NULL) {  /* but a closed file ? */
      luaL_error(L, "Error in " LUA_QS ": attempt to use a closed file.", (newline == 0) ? "io.write" : "io.writeline");  /* 1.5.1 */
      return 0;
    } else  /* write to file */
      return g_write(L, *f, 2, newline, delim);
  }
  else
    return g_write(L, getiofile(L, IO_OUTPUT), 1, newline, delim);
}


static int io_write (lua_State *L) {
  io_writeaux(L, 0);
  return 1;
}


static int io_writeline (lua_State *L) {
  io_writeaux(L, 1);
  return 1;
}


static int io_seek (lua_State *L) {
  static const int mode[] = {SEEK_SET, SEEK_CUR, SEEK_END};
  static const char *const modenames[] = {"set", "cur", "end", NULL};
  FILE *f = tofile(L);
  int op = luaL_checkoption(L, 2, "cur", modenames);
  long offset = luaL_optlong(L, 3, 0);
  op = fseek(f, offset, mode[op]);
  if (op != 0)  /* 1.9.3 */
    return pushresult(L, 0, NULL);  /* error */
  else {
    lua_pushinteger(L, ftell(f));
    return 1;
  }
}


static int io_setvbuf (lua_State *L) {
  static const int mode[] = {_IONBF, _IOFBF, _IOLBF};
  static const char *const modenames[] = {"no", "full", "line", NULL};
  FILE *f = tofile(L);
  int op = luaL_checkoption(L, 2, NULL, modenames);
  lua_Integer sz = agnL_optinteger(L, 3, LUAL_BUFFERSIZE);
  int res = setvbuf(f, NULL, mode[op], sz);
  return pushresult(L, res == 0, NULL);
}


static int io_sync (lua_State *L) {
  if (lua_isnoneornil(L, 1))  /* no arguments? */
    return pushresult(L, fflush(getiofile(L, IO_OUTPUT)) == 0, NULL);
  else
    return pushresult(L, fflush(tofile(L)) == 0, NULL);
}

/* read in an entire file and return its line in a table; taken from own C code (clean.c);
   0.10.0, April 27, 2008; extended 0.12.2, November 08, 2008;
   patched 0.27.2, October 13, 2009 to fix the long-known `cannot read more than 2048
   chars per line` bug.
   extended 0.27.2, October 14, 2009 to accept file handles.
*/

static int io_readlines (lua_State *L) {
  int getout, strlength, linelength, hasoption, isfilename;
  size_t i;
  unsigned int c, convert;
  char buffer[LUAL_BUFFERSIZE];
  char *line;
  const char *skip;
  FILE *fp;
  skip = "";
  convert = hasoption = getout = c = isfilename = 0;
  for (i=2; i <= lua_gettop(L); i++) {
    if (agn_isstring(L, i)) {
      skip = agn_tostring(L, i);
      hasoption = 1;
    }
    else if (lua_isboolean(L, i)) {  /* Agena 1.6.0 */
      convert = lua_toboolean(L, i);
    }
  }
  if (agn_isstring(L, 1)) {  /* 0.27.2, 1.6.11 */
    const char *filename = agn_tostring(L, 1);
    fp = fopen(filename, "r");
    if (fp == NULL)
      fileerror(L, 1, filename);
    else
      isfilename = 1;
  }
  else {  /* 0.27.2 */
    fp = tofile(L);
    if (fp == NULL)  /* but a closed file ? */
      luaL_error(L, "in " LUA_QL("io.readlines") ", attempt to use a closed file.");
  }
  lua_newtable(L);
  while (fgets(buffer, sizeof(buffer), fp) != '\0') {  /* eof not yet reached ?  sizeof(buffer) is the length of the char array.*/
    strlength = linelength = strlen(buffer);
    line = (char *)malloc((strlength+1)*sizeof(char));  /* Agena 1.6.4 */
    if (line == NULL)  /* 0.27.2 */
      luaL_error(L, "Error in " LUA_QS ": buffer allocation error.", "io.readlines");
    strcpy(line, buffer);
    while (strlength > 0 && uchar(*(buffer+strlength-1)) != '\n') {  /* 0.27.2, eol not yet reached ? -> expand buffer, 1.6.11 Valgrind */
      if (fgets(buffer, sizeof(buffer), fp) == '\0') {  /* eof reached ?  sizeof(buffer) is the length of the char array. */
        getout = 1;  /* but process last line in file like the other ones */
        line[linelength] = '\0';  /* 1.6.11 better sure than sorry */
        break;
      }
      strlength = strlen(buffer);
      linelength += strlength;
      line = (char *)realloc(line, (linelength+1)*sizeof(char));  /* 1.6.4 */
      if (line == NULL)  /* 0.27.2 */
        luaL_error(L, "Error in " LUA_QS ": buffer re-allocation error.", "io.readlines");
      strcat(line, buffer);
    }
    /* A conversion is only needed in Windows; in your UNIX terminal, you may have to set the character set properly. */
    if (convert) {
      for (i=0; i < linelength; i++) {  /* 1.6.11 fix */
        line[i] = f2c[uchar(line[i])];
      }
    }
    /* delete \r in DOS files */
    if (linelength > 1 && line[linelength-2] == '\r') {  /* 1.6.11 Valgrind */
      line[linelength-2] = line[linelength-1];
      line[linelength-1] = '\0';
      linelength--;
    }
    /* delete newline */
    if (linelength > 0 && line[linelength-1] == '\n') {  /* 1.6.11 Valgrind */
      line[linelength-1] = '\0';
    }
    if (!hasoption || (strstr(line, skip) != line))
      /* if line begins with the optional second string skip it */
      lua_rawsetistring(L, -1, ++c, line);
    xfree(line);  /* Agena 1.1.0 */
    if (getout) break;  /* 0.27.2 */
  }
  if (isfilename) {
    if (fclose(fp) != 0)  /* 1.11.6 */
      luaL_error(L, "Error in " LUA_QS ": file could not be closed.", "io.readlines");
  }
  return 1;
}


static int io_nlines (lua_State *L) {
  int getout, strlength, isfilename;
#ifndef LUA_DOS
  uint32_t c;
#else
  unsigned long c;
#endif
  char buffer[LUAL_BUFFERSIZE];
  FILE *fp;
  getout = 0;
  c = 0;
  isfilename = 0;
  if (lua_type(L, 1) == LUA_TSTRING) {  /* 0.27.2 */
    const char *filename = agn_checkstring(L, 1);
    fp = fopen(filename, "r");
    if (fp == NULL)
      fileerror(L, 1, filename);
    else
      isfilename = 1;
  }
  else {  /* 0.27.2 */
    fp = tofile(L);
    if (fp == NULL)  /* but a closed file ? */
      luaL_error(L, "Error in " LUA_QL("io.nlines") ", attempt to use a closed file.");
  }
  while (1) {
    if (fgets(buffer, sizeof(buffer), fp) == '\0') break;  /* eof reached ? */
    c++;
    strlength = strlen(buffer);
    while (*(buffer+strlength-1) != '\n') {  /* 0.27.2, eol not yet reached ? */
      if (fgets(buffer, sizeof(buffer), fp) == '\0') {  /* eof reached ? */
        getout = 1;  /* but process last line in file like the other ones */
        break;
      }
      strlength = strlen(buffer);
    }
    if (getout) break;  /* 0.27.2 */
  }
  if (isfilename) {
    if (fclose(fp) != 0)  /* 1.11.6 */
      luaL_error(L, "Error in " LUA_QS ": file could not be closed.", "io.nlines");
  }
  lua_pushnumber(L, c);
  return 1;
}

/* Skips the given number of lines and sets the file position to the beginning of the line
   that follows the last line skipped.

   If f is a file name, then with each call to io.skiplines the search always starts at the
   very first line in the file. If you use a file handle, then lines can be skipped multiple
   times, always relative to the current file position.

   Ther second argument n may be any nonnegative number. If n is 0, then the function does
   nothing and does not change the file position.

   The function returns two values: the number of lines actually skipped and the number of
   characters skipped in this process, including newlines.

   Based on io_nlines; Agena 1.10.1, 12.03.2013 */

static int io_skiplines (lua_State *L) {
  int getout, strlength, isfilename;
#ifndef LUA_DOS
  uint32_t c;
  int32_t nlines, nchars;
#else
  unsigned long c;
  long long nlines, nchars;
#endif
  char buffer[LUAL_BUFFERSIZE];
  FILE *fp;
  getout = 0;
  c = 0;
  nchars = 0;
  isfilename = 0;
  if (lua_type(L, 1) == LUA_TSTRING) {
    const char *filename = agn_checkstring(L, 1);
    fp = fopen(filename, "r");
    if (fp == NULL)
      fileerror(L, 1, filename);
    else
      isfilename = 1;
  }
  else {  /* 0.27.2 */
    fp = tofile(L);
    if (fp == NULL)  /* but a closed file ? */
      luaL_error(L, "Error in " LUA_QL("io.skiplines") ", attempt to use a closed file.");
  }
  nlines = agnL_checkinteger(L, 2);
  if (nlines < 0)
    luaL_error(L, "Error in " LUA_QL("io.skiplines") ", second argument is negative.");
  else if (nlines == 0) {  /* do noting just return */
    if (isfilename) fclose(fp);
    lua_pushnumber(L, 0);
    lua_pushnumber(L, 0);
    return 2;
  }
  do {
    if (fgets(buffer, sizeof(buffer), fp) == '\0') break;  /* eof reached ? */
    c++;
    strlength = strlen(buffer);
    nchars += strlength;
    while (*(buffer+strlength-1) != '\n') {  /* eol not yet reached ? -> expand buffer */
      if (fgets(buffer, sizeof(buffer), fp) == '\0') {  /* eof reached ? */
        getout = 1;  /* but process last line in file like the other ones */
        break;
      }
      strlength = strlen(buffer);
      nchars += strlength;
    }
  } while (!(c == nlines || getout));
  if (isfilename) {
    if (fclose(fp) != 0)
      luaL_error(L, "Error in " LUA_QS ": file could not be closed.", "io.skiplines");
  }
  lua_pushnumber(L, c);
  lua_pushnumber(L, nchars);
  return 2;
}


/* get a single keystroke; June 29, 2007; does not work in Haiku */

#if defined(_WIN32) || defined(__unix__) || defined(__OS2__) || defined(__APPLE__)
static int io_getkey (lua_State *L) {
  if (agn_getgui(L))
    luaL_error(L, "Error in " LUA_QS ": keyboard input not supported by AgenaEdit.", "io.getkey");
  lua_pushinteger(L, getch());
  return 1;
}
#endif

static int io_anykey (lua_State *L) {
  if (agn_getgui(L))
    luaL_error(L, "Error in " LUA_QS ": keyboard input not supported by AgenaEdit.", "io.anykey");
#if defined(_WIN32)
  int i = kbhit();
  lua_pushboolean(L, i);
  if (i) { getch(); }  /* 0.31.4, `clear the buffer` */
#elif defined(__unix__) || defined(__APPLE__)
  int i;
  i = kbhit();
  if (i == -1)
    luaL_error(L, "Error in " LUA_QS ": encountered terminal IO failure.", "io.anykey");
  lua_pushboolean(L, i);
#else
  lua_pushfail(L);
#endif
  return 1;
}


void *io_gethandle (lua_State *L) {  /* 0.32.5 */
  void *ud;
  luaL_checkany(L, 1);
  ud = lua_touserdata(L, 1);
  if (ud == NULL || (*((FILE **)ud)) == NULL) return NULL;
  lua_getfield(L, LUA_REGISTRYINDEX, LUA_FILEHANDLE);
  if (!lua_getmetatable(L, 1)) {
    agn_poptop(L);  /* pop field */
    return NULL;
  }
  if (!lua_rawequal(L, -2, -1)) ud = NULL;
  agn_poptoptwo(L);  /* pop metatable and field */
  return ud;
}


/* determine whether the argument is a file descriptor pointing to an open file, 0.12.2,
   patched 0.32.5 for proper cleaning of stack before leaving */
static int io_isfdesc (lua_State *L) {
  void *ud = io_gethandle(L);
  lua_pushboolean(L, ud != NULL);
  return 1;
}


static int io_isopen (lua_State *L) {  /* 1.12.1 */
  void *ud = io_gethandle(L);
  if (ud == NULL) {
    lua_pushfalse(L);
  } else {
    FILE *f = *((FILE **)ud);
    lua_pushboolean(L, !(fseek(f, 0, SEEK_CUR) != 0 || ftell(f) == -1));
  }
  return 1;
}


static int io_lock (lua_State *L) {  /* 0.32.5 */
  size_t nargs;
  void *ud;
  int hnd;
  off64_t start, size;
  hnd = 0;
  ud = io_gethandle(L);
  if (ud == NULL)
    luaL_error(L, "Error in " LUA_QS ": invalid file handle.", "io.lock");
  hnd = fileno((*((FILE **)ud)));
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1) {  /* lock entire file (in Windows lock 2^63 bytes only) */
    start = 0;
    size = 0;
  } else {
    /* lock from current file position */
    start = my_fpos(hnd);
    size = agnL_checknumber(L, 2);
    if (size < 0) luaL_error(L, "Error in " LUA_QS ": must lock at least one byte.", "io.lock");
  }
  lua_pushboolean(L, my_lock(hnd, start, size) == 0);
  return 1;
}


static int io_unlock (lua_State *L) {  /* 0.32.5 */
  size_t nargs;
  void *ud;
  int hnd;
  off64_t start, size;
  hnd = 0;
  ud = io_gethandle(L);
  if (ud == NULL)
    luaL_error(L, "Error in " LUA_QS ": invalid file handle.", "io.unlock");
  hnd = fileno((*((FILE **)ud)));
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1) {  /* lock entire file (in Windows lock 2^63 bytes only) */
    start = 0;
    size = 0;
  } else {
    /* unlock from current file position */
    start = my_fpos(hnd);
    size = agnL_checknumber(L, 2);
    if (size < 0) luaL_error(L, "Error in " LUA_QS ": must unlock at least one byte.", "io.unlock");
  }
  lua_pushboolean(L, my_unlock(hnd, start, size) == 0);
  return 1;
}


static int io_fileno (lua_State *L) {  /* 1.9.3 */
  int filenr;
  FILE *f = tofile(L);
  filenr = fileno(f);
  if (filenr == -1)
    lua_pushfail(L);
  else
    lua_pushinteger(L, filenr);
  return 1;
}


/* creates a new file denoted by its first argument (a string) and writes all of the given strings or numbers
   starting with the second argument in binary mode to it. To write other values, use `tostring` or
   `strings.format`. After writing all data, the function automatically closes the new file.

   By default, no character is inserted between neighbouring strings. This may be changed by passing the
   option 'delim':<str> (i.e. a pair, e.g. 'delim':'|') as the last argument to the function with <str>
   being a string of any length.

   The function returns the total number of bytes written, and issues an error otherwise.
   */
static int io_writefile (lua_State *L) {  /* 1.11.3 */
  int nargs, status, arg;
  size_t bytes, sizedelim;
  FILE *pf;
  char *delim;
  const char *filename;
  bytes = 0;  /* chars to be written */
  sizedelim = 0;  /* size of the delimitor */
  status = 1;
  arg = 2;  /* start with writing argument #2 */
  filename = agn_checkstring(L, 1);
  delim = NULL;  /* to avoid compiler warnings */
  nargs = lua_gettop(L);
  if (nargs < 2) luaL_error(L, "Error in " LUA_QS ": must get at least two arguments.", "io.writefile");
  if (nargs > 2 && lua_ispair(L, nargs)) {
    agn_pairgeti(L, nargs, 1);  /* get left value, set to stack index -2 */
    agn_pairgeti(L, nargs, 2);  /* get right value, set to stack index  -1 */
    if (strcmp("delim", agn_checkstring(L, -2)) != 0)
      luaL_error(L, "Error in " LUA_QS ": unknown option.", "io.writefile");
    else {
      delim = strdup(agn_checkstring(L, -1));
      sizedelim = strlen(delim);
    }
    agn_poptoptwo(L);
  }
  pf = fopen(filename, "wb");
  if (pf == NULL)
    luaL_error(L, "Error in " LUA_QS ": file %s could not be opened.", "io.writefile", filename);
  clearerr(pf);
  nargs = nargs - 1 - (delim != NULL);  /* subtract 1 because if an option is given, it is still on the stack */
  for (; nargs--; arg++) {
    size_t l;
    const char *s = luaL_checklstring(L, arg, &l);
    status = status && (fwrite(s, sizeof(char), l, pf) == l);
    bytes += l*sizeof(char);
    if (delim != NULL && nargs != 0) {  /* do not print delim after last argument */
      fwrite(delim, sizeof(char), sizedelim, pf);
      bytes += sizedelim*sizeof(char);
    }
  }
  if (delim != NULL) xfree(delim);
  if (ferror(pf) || status == 0)
    luaL_error(L, "Error in " LUA_QS ": file %s could possibly not be written.", "io.writefile", filename);
  if (fclose(pf) != 0)
    luaL_error(L, "Error in " LUA_QS ": file %s could not be closed.", "io.writefile", filename);
  lua_pushnumber(L, bytes);
  return 1;
}


#ifdef _WIN32
/* 1.19.2: The following is based on code written by banders7, published at
   http://www.daniweb.com/software-development/c/code/217173/transfer-data-tofrom-the-windows-clipboard

   Copy input to Windows clipboard
   Data lines must be terminated by the CR LF pair (0xD,0xA)
   data in: line1CRLFline2CRLFline3CRLF --- Caller must format
   "this is a line\n" is not acceptable (alex: this does not seem to bother Windows 2000 SP4,
   "this is a line\r\n" is acceptable.
   If clipboard data shows square empty boxes at line ends in Windows,
   it is because lines are terminated by \n only. */

static int io_putclip (lua_State *L) {
  char far *buffer;
  char *toclipdata;
  size_t bytes;
  HGLOBAL clipbuffer;
  toclipdata = (char *)agn_checklstring(L, 1, &bytes);
  /* transfer to clipboard */
  OpenClipboard(NULL);
  EmptyClipboard();
  clipbuffer = GlobalAlloc(GMEM_DDESHARE, bytes+1);  /* plus trailing \0 */
  buffer = (char far*)GlobalLock(clipbuffer);
  if (buffer == NULL) {
    int en;
    en = GetLastError();
    GlobalFree(clipbuffer);
    lua_pushfail(L);
    lua_pushstring(L, strerror(en));
    return 2;
  }
  strcpy(buffer, toclipdata);
  GlobalUnlock(clipbuffer);
  SetClipboardData(CF_TEXT, clipbuffer);
  CloseClipboard();
  agn_pushboolean(L, bytes < 0 ? -1 : 1);  /* non-negative value is success */
  GlobalFree(clipbuffer);
  return 1;
}

/* Return pointer to clipboard data and set bytes returned value.
   If error occurs, set up error message, point to it, set bytes negative
   Whether successful or not, the caller SHOULD free the data */

static int io_getclip (lua_State *L) {
  int en;
  char *buffer;
  buffer = NULL;
  /* open the clipboard */
  if (OpenClipboard(NULL)) {
    HANDLE hData = GetClipboardData(CF_TEXT);
    char *buffer = (char*)GlobalLock(hData);
    en = GetLastError();
    GlobalUnlock(hData);
    CloseClipboard();
    if (buffer == NULL) {  /* return an error message */
      lua_pushfail(L);
      lua_pushstring(L, strerror(en));
      return 2;
    } else {  /* return pointer to retrieved data */
      char *data = (char *)malloc(strlen(buffer) + 1);
      if (data == NULL) {
        buffer = NULL;
        luaL_error(L, "Error in " LUA_QS ": memory allocation error.", "io.getclip");
      }
      strcpy(data, buffer);
      lua_pushstring(L, data);
      xfree(data);
      return 1;
    }
  } else {  /* return an open clipboard failed message */
    en = GetLastError();
    lua_pushfail(L);
    lua_pushstring(L, strerror(en));
    return 2;
  }
}
#endif


static FILE* io_aux_checkfile (lua_State *L, const char *procname) {  /* 2.0.0. RC 5 */
  FILE *f = tofile(L);
  if (f == NULL)  /* but a closed file ? */
    luaL_error(L, "Error in " LUA_QS ", attempt to use a closed file.", procname);
  if (fseek(f, 0, SEEK_CUR) != 0 || ftell(f) == -1)
    luaL_error(L, "Error in " LUA_QS ": could not determine length.", procname);
  return f;
}


/* Truncates an open file denoted by its handle fh at the current file position. The function returns true on
   success and issues an error otherwise. You may query and change the current file position by calling io.seek. */

static int io_truncate (lua_State *L) {  /* 2.0.0. RC 5 */
  int en;
  off_t pos, eof;
  FILE *f = io_aux_checkfile(L, "io.truncate");
  pos = ftello64(f);
  if (pos == -1)  /* there is no errno for ftell */
    luaL_error(L, "Error in " LUA_QS ": could not determine currrent file position.", "io.truncate");
  if (fseek(f, 0, SEEK_END) != 0)
    luaL_error(L, "Error in " LUA_QS ": could not determine length of file.", "io.truncate");
  eof = ftello64(f);
  if (eof == -1)  /* there is no errno for ftell */
    luaL_error(L, "Error in " LUA_QS ": could not determine length of file.", "io.truncate");
  if (eof == pos)
    lua_pushfalse(L);
  else {
    if (ftruncate(fileno(f), pos) != 0) {
      en = errno;
      luaL_error(L, "Error in " LUA_QS ": could not truncate file: %s.", "io.truncate", strerror(en));
    }
    else {
      if (fseek(f, pos, SEEK_SET) != 0)  /* reset file position */
        luaL_error(L, "Error in " LUA_QS ": could not reset file position.", "io.truncate");
      lua_pushtrue(L);
    }
  }
  return 1;
}


/* Returns the size of an open file denoted by its file handle fd and returns the number of bytes as a non-negative integer */

static int io_filesize (lua_State *L) {  /* 2.0.0. RC 5 */
  off_t pos, eof;
  FILE *f = io_aux_checkfile(L, "io.filesize");
  pos = ftello64(f);
  if (pos == -1)  /* there is no errno for ftell */
    luaL_error(L, "Error in " LUA_QS ": could not determine currrent file position.", "io.filesize");
  if (fseek(f, 0, SEEK_END) != 0)
    luaL_error(L, "Error in " LUA_QS ": could not determine length of file.", "io.filesize");
  eof = ftello64(f);
  if (eof == -1)  /* there is no errno for ftell */
    luaL_error(L, "Error in " LUA_QS ": could not determine length of file.", "io.filesize");
  if (fseek(f, pos, SEEK_SET) != 0)
    luaL_error(L, "Error in " LUA_QS ": could not reset file position.", "io.filesize");
  lua_pushnumber(L, eof);
  return 1;
}


/* Moves the current file position of the open file denoted by its handle fh either to the left or the right.
   If n is a positive integer, then the file position is moved n characters to the right, if it is a
   negative integer, it is moved n characters to the left. If n is zero, the position is not changed at all.
   The function returns true on success and false otherwise. */

static int io_move (lua_State *L) {  /* 2.0.0. RC 5 */
  FILE *f = io_aux_checkfile(L, "io.move");
  lua_pushboolean(L, fseek(f, agnL_checkinteger(L, 2), SEEK_CUR) == 0);
  return 1;
}


static const luaL_Reg flib[] = {
  {"__gc", io_gc},
  {"__tostring", io_tostring},
  {NULL, NULL}
};

static const luaL_Reg iolib[] = {
  {"anykey", io_anykey},
  {"close", io_close},
  {"fileno", io_fileno},         /* 1.9.3, February 19, 2013 */
  {"filesize", io_filesize},     /* 2.0.0, November 29, 2013 */
  #ifdef _WIN32
  {"getclip", io_getclip},       /* 1.12.9, November 01, 2013 */
  #endif
  {"input", io_input},
  {"infile", io_infile},         /* 1.11.6, June 02, 2013 */
  {"isfdesc", io_isfdesc},       /* 0.32.5, June 12, 2010 */
  {"isopen", io_isopen},         /* 1.12.1, June 14, 2013 */
  {"lines", io_lines},
  {"lock", io_lock},             /* 0.32.5, June 12, 2010 */
  {"move", io_move},             /* 2.0.0, November 29, 2013 */
  {"nlines", io_nlines},         /* 0.31.7, April 22, 2010 */
  {"open", io_open},
  {"output", io_output},
  {"popen", io_popen},
  #ifdef _WIN32
  {"putclip", io_putclip},       /* 1.12.9, November 01, 2013 */
  #endif
  {"read", io_read},
  {"readfile", io_readfile},     /* 1.10.2, March 19, 2013 */
  {"readlines", io_readlines},   /* 0.10.0, April 27, 2008 */
  {"seek", io_seek},
  {"setvbuf", io_setvbuf},
  {"skiplines", io_skiplines},   /* 1.10.1, March 12, 2013 */
  {"sync", io_sync},
  {"tmpfile", io_tmpfile},
  {"truncate", io_truncate},     /* 2.0.0 RC 5 */
  {"unlock", io_unlock},         /* 0.32.5, June 12, 2010 */
  {"write", io_write},
  {"writefile", io_writefile},   /* 1.11.3, May 08, 2013 */
  {"writeline", io_writeline},   /* May 08, 2007 - added 0.5.1 */
  #if defined(_WIN32) || defined(__unix__) || defined(__OS2__) || defined(__APPLE__)
  {"getkey", io_getkey},         /* added on June 29, 2007 */
  #endif
  {NULL, NULL}
};


static void createmeta (lua_State *L) {
  luaL_newmetatable(L, LUA_FILEHANDLE);  /* create metatable for file handles */
  lua_pushvalue(L, -1);  /* push metatable */
  lua_setfield(L, -2, "__index");  /* metatable.__index = metatable */
  luaL_register(L, NULL, flib);  /* file methods */
}


static void createstdfile (lua_State *L, FILE *f, int k, const char *fname) {
  *newfile(L) = f;
  if (k > 0) {
    lua_pushvalue(L, -1);
    lua_rawseti(L, LUA_ENVIRONINDEX, k);
  }
  lua_setfield(L, -2, fname);
}


LUALIB_API int luaopen_io (lua_State *L) {
  createmeta(L);
  /* create (private) environment (with fields IO_INPUT, IO_OUTPUT, __close) */
  lua_createtable(L, 2, 1);
  lua_replace(L, LUA_ENVIRONINDEX);
  /* open library */
  luaL_register(L, LUA_IOLIBNAME, iolib);
  /* table for information on all open files */
  lua_newtable(L);
  lua_setfield(L, -2, "openfiles");  /* table for information on all open files */
  /* create (and set) default files */
  createstdfile(L, stdin, IO_INPUT, "stdin");
  createstdfile(L, stdout, IO_OUTPUT, "stdout");
  createstdfile(L, stderr, 0, "stderr");
  /* create environment for 'popen' */
  lua_getfield(L, -1, "popen");
  lua_createtable(L, 0, 1);
  lua_pushcfunction(L, io_pclose);
  lua_setfield(L, -2, "__close");
  lua_setfenv(L, -2);
  agn_poptop(L);  /* pop 'popen' */
  /* set default close function */
  lua_pushcfunction(L, io_fclose);
  lua_setfield(L, LUA_ENVIRONINDEX, "__close");
  return 1;
}

