/*
** $Id: linalg.c, initiated September 04, 2008 $
** Linear Algebra library
** See Copyright Notice in agena.h
*/


#include <stdlib.h>
#include <math.h>
#include <string.h>

#define linalg_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "agncmpt.h"  /* for trunc in Linux */


#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_LINALGLIBNAME "linalg"
LUALIB_API int (luaopen_linalg) (lua_State *L);
#endif


/* checks whether the two arguments are vectors of the same dimension, modified Agena 1.4.3/1.5.0 */
#define checkvectors(L,a,b) { \
  if (!(agn_istableutype(L, (a), "vector") && agn_istableutype(L, (b), "vector")) ) \
    luaL_error(L, "two vectors expected."); \
  lua_getfield(L, (a), "dim"); \
  sizea = agn_checknumber(L, -1); \
  agn_poptop(L); \
  lua_getfield(L, (b), "dim"); \
  sizeb = agn_checknumber(L, -1); \
  agn_poptop(L); \
  if (sizea != sizeb) \
    luaL_error(L, "vectors of different size."); \
}


/* checks whether the argument is a vector, modified Agena 1.4.3/1.5.0 */
#define checkvector(L,a) { \
  if (!(agn_istableutype(L, (a), "vector")) ) \
    luaL_error(L, "vector expected, got %s.", lua_typename(L, lua_type((L), (a)))); \
  lua_getfield(L, (a), "dim"); \
  size = agn_checknumber(L, -1); \
  agn_poptop(L); \
}


/* Set dimensions of a matrix. The matrix must be at the top of the stack. The two remove
   statements at the end of the macro remove left and right operand nargs since agn_creatpair
   does not remove the two upper values. */
#define setmatrixdims(L, a, b) { \
  lua_pushstring(L, "dim"); \
  lua_pushnumber(L, nargs); \
  lua_pushnumber(L, nargs); \
  agn_createpair(L, -2, -1); \
  lua_remove(L, -2); lua_remove(L, -2); \
  lua_rawset(L, -3); \
}


/* Set dimensions of a vector. The vector must be at the top of the stack. */
#define setvectordim(L, a) { \
  lua_pushstring(L, "dim"); \
  lua_pushnumber(L, a); \
  lua_rawset(L, -3); \
}


/* Set metatable to a vector or matrix. o either is the string "vmt" or "mmt".
   The vector must be at the top of the stack. */
#define setmetatable(L, o) { \
  lua_getglobal(L, "linalg"); \
  lua_getfield(L, -1, o); \
  lua_setmetatable(L, -3); \
  agn_poptop(L); \
}

/* set vector attributes: user-defined type, metatable, and dimension to a vector
   which must be at the top of the stack */
#define setvattribs(L, a) { \
  lua_pushstring(L, "vector"); \
  agn_setutype(L, -2, -1); \
  agn_poptop(L); \
  setmetatable(L, "vmt"); \
  setvectordim(L, (a)); \
}


static int aux_getnumber (lua_State *L, int idx, int n) {
  lua_Number a;
  lua_pushnumber(L, n);
  lua_gettable(L, idx);
  a = agn_checknumber(L, -1);  /* Agena 1.4.3/1.5.0 */
  agn_poptop(L);
  return a;
}


/* Add two vectors. The result is a new vector. */
static int linalg_add (lua_State *L) {
  int i, sizea, sizeb;
  lua_Number a, b;
  checkvectors(L, 1, 2);
  lua_createtable(L, sizea, 1);
  /* now traverse vectors */
  for (i=1; i <= sizea; i++) {
    a = aux_getnumber(L, 1, i);
    b = aux_getnumber(L, 2, i);
    lua_rawsetinumber(L, -1, i, a+b);  /* store result to new sequence */
  }
  /* set attributes */
  setvattribs(L, sizea);
  return 1;
}


/* Subtract two vectors. The result is a new vector. */
static int linalg_sub (lua_State *L) {
  int i, sizea, sizeb;
  lua_Number a, b;
  checkvectors(L, 1, 2);
  lua_createtable(L, sizea, 1);
  /* now traverse vectors */
  for (i=1; i <= sizea; i++) {
    a = aux_getnumber(L, 1, i);
    b = aux_getnumber(L, 2, i);
    lua_rawsetinumber(L, -1, i, a-b);  /* store result to new sequence */
  }
  /* set attributes */
  setvattribs(L, sizea);
  return 1;
}


static int linalg_scalarmul (lua_State *L) {
  int i, size;
  lua_Number n, a;
  n = 0;  /* to avoid compiler warning */
  if (!(agn_istableutype(L, 2, "vector") && (lua_type(L, 1) == LUA_TNUMBER)))
    luaL_error(L, "Error in " LUA_QS ": number and vector expected, in this order.", "linalg.scalarmul");
  n = agn_tonumber(L, 1);
  lua_getfield(L, 2, "dim");
  size = agn_checknumber(L, -1);  /* Agena 1.4.3/1.5.0 */
  agn_poptop(L);
  lua_createtable(L, size, 1);
  /* now traverse vector */
  for (i=0; i < size; i++) {
    a = aux_getnumber(L, 2, i+1);
    lua_rawsetinumber(L, -1, i+1, n*a);  /* store result to new vector */
  }
  /* set attributes */
  setvattribs(L, size);
  return 1;
}


static int linalg_vector (lua_State *L) {
  int i, nops, type;
  nops = lua_gettop(L);
  if (nops == 0)
    luaL_error(L, "Error in " LUA_QS ": at least one argument expected.", "linalg.vector");
  luaL_checkstack(L, nops, "too many elements");
  type = lua_type(L, 1);
  if (nops == 2 && type == LUA_TNUMBER && lua_type(L, 2) == LUA_TTABLE) {  /* Maple-like syntax */
    int c, key;
    c = 0;
    nops = agnL_checkinteger(L, 1);
    lua_newtable(L);
    lua_pushnil(L);
    while (lua_next(L, 2) != 0) {
      c++;
      luaL_argcheck(L, lua_type(L, -1) == LUA_TNUMBER && lua_type(L, -2) == LUA_TNUMBER, c,
        "expected a number in " LUA_QL("linalg.vector"));
      key = (int)agn_tonumber(L, -2);
      if (key < 1 || key > nops)
        luaL_error(L, "Error in " LUA_QS ": table index out of range.", "linalg.vector");
      lua_rawset2(L, -3);  /* deletes only the value, but not the key */
    }
    if (c > nops)
      luaL_error(L, "Error in " LUA_QS ": more entries passed than given dimension.", "linalg.vector");
  } else if (type == LUA_TNUMBER) {  /* assume all arguments are numbers */
    lua_createtable(L, nops, 0);
    for (i=1; i <= nops; i++) {
      lua_rawsetinumber(L, -1, i, agn_checknumber(L, i));
    }
  } else if (type == LUA_TTABLE) {
    nops = luaL_getn(L, 1);
    if (nops == 0)
      luaL_error(L, "Error in " LUA_QS ": at least one table value expected.", "linalg.vector");
    lua_createtable(L, nops, 0);
    for (i=1; i <= nops; i++) {
      lua_rawgeti(L, 1, i);
      lua_rawsetinumber(L, -2, i, agn_checknumber(L, -1));
      agn_poptop(L);
    }
  } else if (type == LUA_TSEQ && agn_isutypeset(L, 1) == 0) {  /* a sequence and no user-defined type set ? */
    nops = agn_seqsize(L, 1);
    if (nops == 0)
      luaL_error(L, "Error in " LUA_QS ": at least one sequence value expected.", "linalg.vector");
    lua_createtable(L, nops, 0);
    for (i=1; i <= nops; i++) {
      lua_seqgeti(L, 1, i);
      lua_rawsetinumber(L, -2, i, agn_checknumber(L, -1));
      agn_poptop(L);
    }
  } else
    luaL_error(L, "Error in " LUA_QS ": numbers, a table, or sequence expected.", "linalg.vector");
  /* set vector attributes */
  setvattribs(L, nops);
  return 1;
}


static int linalg_zero (lua_State *L) {
  int i, nargs;
  nargs = agn_checknumber(L, 1);
  lua_createtable(L, nargs, 1);
  for (i=1; i <= nargs; i++)
    lua_rawsetinumber(L, -1, i, 0);
  /* set vector attributes */
  setvattribs(L, nargs);
  return 1;
}


static int linalg_identity (lua_State *L) {
  int i, nargs;
  nargs = agn_checknumber(L, 1);
  if (nargs < 1)
    luaL_error(L, "Error in " LUA_QS ": positive integer expected.", "linalg.identity");
  /* retrieve metamethods */
  lua_getglobal(L, "linalg");
  lua_getfield(L, -1, "vmt");
  lua_createtable(L, 1, 0);
  for (i=1; i <= nargs; i++) {
    lua_newtable(L);
    lua_pushnumber(L, i);
    lua_pushnumber(L, 1);
    lua_rawset(L, -3);
    /* set utype */
    lua_pushstring(L, "vector");
    agn_setutype(L, -2, -1);
    agn_poptop(L);
    /* set metatable */
    lua_pushvalue(L, -3);
    lua_setmetatable(L, -2);
    /* set dimension */
    setvectordim(L, nargs);
    /* insert vector into matrix and pop the vector */
    lua_rawseti(L, -2, i);
  }
  lua_pushstring(L, "matrix");
  agn_setutype(L, -2, -1);
  agn_poptop(L);
  lua_remove(L, -2);  /* drop linalg table */
  lua_remove(L, -2);  /* drop metatable */
  /* set metatable */
  setmetatable(L, "mmt");
  /* set matrix dimensions */
  setmatrixdims(L, nargs, nargs);
  return 1;
}


static int linalg_checkvector (lua_State *L) {
  int i, nargs, dim, olddim;
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments for " LUA_QL("linalg.checkvector"));
  olddim = 0;
  for (i=1; i <= nargs; i++) {
    if (!agn_istableutype(L, i, "vector")) {
      if (i > 1) lua_pop(L, i-1);  /* drop dimensions accumulated so far */
      luaL_error(L, "Error in " LUA_QS ": vector expected, got %s.", "linalg.checkvector", lua_typename(L, lua_type(L, i)));
    }
    lua_getfield(L, i, "dim");
    if (nargs != 1) {
      dim = agn_checknumber(L, -1);
      if (i == 1) olddim = dim;
      if (olddim == dim)
        olddim = dim;
      else {
        lua_pop(L, i);  /* drop dimensions accumulated so far */
        luaL_error(L, "Error in " LUA_QS ": vectors of different dimension." LUA_QS, "linalg.checkvector");
      }
    }
  }
  return nargs;  /* return all dimensions */
}


static int linalg_isvector (lua_State *L) {
  int i, nargs;
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments for " LUA_QL("linalg.isvector"));
  for (i=1; i <= nargs; i++) {
    if (agn_istableutype(L, i, "vector") == 0) {
      lua_pushboolean(L, 0);
      return 1;
    }
  }
  lua_pushboolean(L, 1);
  return 1;
}


static int linalg_checkmatrix (lua_State *L) {
  int i, nargs, retdims;
  retdims = 0;
  nargs = lua_gettop(L);
  if (nargs < 1)
    luaL_error(L, "Error in " LUA_QS ": got no argument.", "linalg.checkmatrix");
  luaL_checkstack(L, nargs, "too many arguments for " LUA_QL("linalg.checkmatrix"));
  if (lua_isboolean(L, nargs) && agn_istrue(L, nargs)) {  /* Agena 1.6.0 */  
    if (nargs > 1) nargs--; retdims = 1;
  }
  for (i=1; i <= nargs; i++) {
    if (!agn_istableutype(L, i, "matrix"))
      luaL_error(L, "Error in " LUA_QS ": matrix expected, got %s.", "linalg.checkmatrix", lua_typename(L, lua_type(L, i)));  /* Agena 1.8.1 */
    else if (retdims) {
      lua_getfield(L, i, "dim");
    }
  }
  return retdims * nargs;
}


static int linalg_ismatrix (lua_State *L) {
  int i, nargs;
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments " LUA_QL("linalg.ismatrix"));
  for (i=1; i <= nargs; i++) {
    if (agn_istableutype(L, i, "matrix") == 0) {
      lua_pushboolean(L, 0);
      return 1;
    }
  }
  lua_pushboolean(L, 1);
  return 1;
}


static int linalg_vzip (lua_State *L) {  /* extended 0.30.2 */
  int i, j, sizea, sizeb, nargs;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  checkvectors(L, 2, 3);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  lua_createtable(L, sizea, 0);
  /* now traverse vector */
  for (i=1; i <= sizea; i++) {
    lua_pushvalue(L, 1);      /* push function; FIXME: can be optimized */
    lua_pushnumber(L, i);
    lua_gettable(L, 2);
    lua_pushnumber(L, i);
    lua_gettable(L, 3);
    for (j=4; j<=nargs; j++) {
      lua_pushvalue(L, j);
    }
    lua_call(L, nargs-1, 1);  /* call function with nargs-1 arguments and one result */
    lua_rawseti(L, -2, i);    /* store result to new vector */
  }
  setvattribs(L, sizea);
  return 1;
}


static int linalg_vmap (lua_State *L) {
  int i, j, nargs, size;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments for " LUA_QL("linalg.vmap"));
  checkvector(L, 2);
  lua_createtable(L, size, 0);
  /* now traverse vector */
  for (i=1; i <= size; i++) {
    lua_pushvalue(L, 1);      /* push function; FIXME: can be optimized */
    lua_pushnumber(L, i);
    lua_gettable(L, 2);
    for (j=3; j<=nargs; j++)
      lua_pushvalue(L, j);
    lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument and one result */
    lua_rawseti(L, -2, i);    /* store result to new vector */
  }
  setvattribs(L, size);
  return 1;
}


static int linalg_setvelem (lua_State *L) {
  lua_Number dim, key;
  lua_getfield(L, 1, "dim");
  dim = agn_checknumber(L, -1);
  agn_poptop(L);
  lua_pushvalue(L, 2);  /* push key */
  key = agn_checknumber(L, -1);
  if (key < 1 || key > dim) {
    agn_poptop(L);  /* pop key */
    luaL_error(L, "Error in " LUA_QS ": index %d out of range 1:%d.", "linalg.setvelem", (int)key, (int)dim);
  }
  lua_pushvalue(L, 3);  /* push value */
  lua_rawset(L, 1);     /* now conduct assignment */
  return 0;
}


static const luaL_Reg linalglib[] = {
  {"add", linalg_add},                      /* added on September 04, 2008 */
  {"checkmatrix", linalg_checkmatrix},      /* added on September 06, 2008 */
  {"checkvector", linalg_checkvector},      /* added on September 06, 2008 */
  {"identity", linalg_identity},            /* added on September 06, 2008 */
  {"ismatrix", linalg_ismatrix},            /* added on September 06, 2008 */
  {"isvector", linalg_isvector},            /* added on September 06, 2008 */
  {"scalarmul", linalg_scalarmul},          /* added on September 04, 2008 */
  {"setvelem", linalg_setvelem},            /* added on December 20, 2008 */
  {"sub", linalg_sub},                      /* added on September 04, 2008 */
  {"vector", linalg_vector},                /* added on September 04, 2008 */
  {"vmap", linalg_vmap},                    /* added on December 19, 2008 */
  {"vzip", linalg_vzip},                    /* added on December 19, 2008 */
  {"zero", linalg_zero},                    /* added on September 06, 2008 */
  {NULL, NULL}
};


/*
** Open linalg library
*/
LUALIB_API int luaopen_linalg (lua_State *L) {
  luaL_register(L, AGENA_LINALGLIBNAME, linalglib);
  return 1;
}

