/*
** $Id: lenviron.c,v 1.00 17.12.2010 alex Exp $
** library to query the Agena environment
** See Copyright Notice in agena.h
*/


#include <stdlib.h>
#include <string.h>

#define lenviron_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"


static void getfunc (lua_State *L, int opt) {  /* Lua 5.1.2 patch */
  if (lua_isfunction(L, 1)) lua_pushvalue(L, 1);
  else {
    lua_Debug ar;
    int level = opt ? agnL_optinteger(L, 1, 1) : agnL_checkint(L, 1);  /* Lua 5.1.2 patch */
    luaL_argcheck(L, level >= 0, 1, "level must be non-negative");
    if (lua_getstack(L, level, &ar) == 0)
      luaL_argerror(L, 1, "invalid level");
    lua_getinfo(L, "f", &ar);
    if (lua_isnil(L, -1))
      luaL_error(L, "Error in " LUA_QS ": no function environment for tail call at level %d.",
                    "environ.getfunc", level);
  }
}


static int environ_getfenv (lua_State *L) {
  getfunc(L, 1); /* Lua 5.1.2 patch */
  if (lua_iscfunction(L, -1))  /* is a C function? */
    lua_pushvalue(L, LUA_GLOBALSINDEX);  /* return the thread's global env. */
  else
    lua_getfenv(L, -1);
  return 1;
}


static int environ_setfenv (lua_State *L) {
  luaL_checktype(L, 2, LUA_TTABLE);
  getfunc(L, 1);  /* Lua 5.1.2 patch */
  lua_pushvalue(L, 2);
  if (agn_isnumber(L, 1) && agn_tonumber(L, 1) == 0) {
    /* change environment of current thread */
    lua_pushthread(L);
    lua_insert(L, -2);
    lua_setfenv(L, -2);
    return 0;
  }
  else if (lua_iscfunction(L, -2) || lua_setfenv(L, -2) == 0)
    luaL_error(L,
          LUA_QL("environ.setfenv") " cannot change environment of given object.");
  return 1;
}


static int environ_userinfo (lua_State *L) {  /* 0.22.3, June 14, 2009, changed 1.6.4 */
  lua_Number a, b;
  int i, exception, type;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  a = agn_checknumber(L, 2);
  type = agnL_gettablefield(L, "environ", "infolevel", "environ.userinfo", 1);  /* 1.6.4 */
  if (type != LUA_TTABLE) {
    agn_poptop(L);  /* remove "infolevel" (or "environ" if table environ does not exist) */
    return 0;  /* infolevel not assigned or not a table */
  }
  lua_pushvalue(L, 1);  /* push function */
  lua_rawget(L, -2);  /* function is popped and the infolevel value is put on top of the stack */
  b = agn_tonumberx(L, -1, &exception);
  agn_poptoptwo(L);  /* pop value and "infolevel" */
  if (a <= b && !exception) {
    for (i=3; i <= lua_gettop(L); i++) {
      lua_pushvalue(L, i);
      agnL_printnonstruct(L, -1);
      agn_poptop(L);
    }
    fflush(stdout);
  }
  return 0;
}


static int environ_used (lua_State *L) {  /* 0.27.2 */
  static const char *const opts[] = {"bytes", "kbytes", "mbytes", "gbytes", NULL};
  int o = luaL_checkoption(L, 1, "kbytes", opts);
  switch (o) {
    case 0: /* bytes */
      lua_pushnumber(L, agn_usedbytes(L));
      break;
    case 1: /* kbytes */
      lua_pushnumber(L, agn_usedbytes(L)/(lua_Number)1024);
      break;
    case 2: /* mbytes */
      lua_pushnumber(L, agn_usedbytes(L)/(lua_Number)1048576);
      break;
    case 3: /* gbytes */
      lua_pushnumber(L, agn_usedbytes(L)/(lua_Number)1073741824);
      break;
    default:
      lua_pushfail(L); /* cannot happen */
  }
  return 1;
}


static int environ_collectgarbage (lua_State *L) {
  static const char *const opts[] = {"stop", "restart", "collect",
    "count", "step", "setpause", "setstepmul", NULL};
  static const int optsnum[] = {LUA_GCSTOP, LUA_GCRESTART, LUA_GCCOLLECT,
    LUA_GCCOUNT, LUA_GCSTEP, LUA_GCSETPAUSE, LUA_GCSETSTEPMUL};
  int o = luaL_checkoption(L, 1, "collect", opts);
  int ex = agnL_optinteger(L, 2, 0);
  int res = lua_gc(L, optsnum[o], ex);
  switch (optsnum[o]) {
    case LUA_GCCOUNT: {
      int b = lua_gc(L, LUA_GCCOUNTB, 0);
      lua_pushnumber(L, res + ((lua_Number)b/1024));
      return 1;
    }
    case LUA_GCSTEP: {
      lua_pushboolean(L, res);
      return 1;
    }
    default: {
      lua_pushnumber(L, res);
      return 1;
    }
  }
}


int environ_restart (lua_State *L) { /* added 0.4.0 */
  const char *path;
  int islibname, resetlibname;
  islibname = resetlibname = 0;
  resetlibname = agn_getlibnamereset(L); /* 0.26.0, also reset mainlibname and libname to its original value ? */
  path = NULL;  /* to prevent compiler warnings */
  agn_poptop(L);
  /* first, delete readlib'bed packages from package.loaded; 0.26.0 */
  lua_getglobal(L, "package");
  if (lua_istable(L, -1)) {
    lua_getfield(L, -1, "readlibbed");
    if (lua_isset(L, -1)) {
      lua_getfield(L, -2, "loaded");
      if (lua_istable(L, -1)) {
        lua_pushnil(L);
        while (lua_usnext(L, -3) != 0) {  /* delete readlibbed packages from package.loaded */
          lua_pushnil(L);
          lua_settable(L, -4);
        }
      }
      agn_poptop(L);  /* pop field `loaded' */
    }
    agn_poptop(L); /* delete value `readlibbed' */
  }
  agn_poptop(L);  /* delete `package' */
  /* second, delete all global vars except _origG and libname */
  lua_getfield(L, LUA_GLOBALSINDEX, "_G");  /* get _G on stack */
  if (!lua_istable(L, -1)) {  /* Agena 1.7.1 fix */
    fprintf(stderr, "Warning in " LUA_QS  ": " LUA_QS " is missing or not a table, no restart possible.\n", "restart", "_G");
    fflush(stderr);
    agn_poptop(L);
    return 0;
  }
  lua_pushnil(L);
  while (lua_next(L, -2) != 0) {
    islibname = (strcmp(lua_tostring(L, -2), "libname") == 0 || strcmp(lua_tostring(L, -2), "mainlibname") == 0);  /* 0.28.2 */
    if ((strcmp(lua_tostring(L, -2), "_origG") != 0 && !islibname) ||
        (resetlibname && islibname)) {
      lua_pushnil(L);
      /* delete key in Agena environment */
      lua_setfield(L, LUA_GLOBALSINDEX, lua_tostring(L, -3));
    }
    agn_poptop(L);  /* pop value */
  }
  agn_poptop(L); /* delete _G from stack */
  /* get _origG on stack */
  lua_getfield(L, LUA_GLOBALSINDEX, "_origG");
  if (!lua_istable(L, -1)) {
    fprintf(stderr, "Warning in " LUA_QS  ": " LUA_QS " is missing or not a table, no restart possible.\n", "restart", "_origG");
    fflush(stderr);
    agn_poptop(L);
    return 0;
  }
  /* set all values in _origG into environment */
  lua_pushnil(L);
  while (lua_next(L, -2) != 0) {
    lua_pushvalue(L, -2); /* key */
    lua_pushvalue(L, -2); /* value */
    lua_setfield(L, LUA_GLOBALSINDEX, lua_tostring(L, -2));  /* set key in Agena environment */
    agn_poptoptwo(L);  /* delete copied key and value */
  }
  agn_poptop(L);  /* pop _origG from stack, 0.26.0 patch */
  /* make Agena assign _G properly in the restarted environment. */
  lua_pushvalue(L, LUA_GLOBALSINDEX);
  lua_setglobal(L, "_G");
  /* reset package.readlibbed */
  lua_getglobal(L, "package");
  if (lua_istable(L, -1)) {
    agn_createset(L, 0);
    lua_setfield(L, -2, "readlibbed");
  }
  agn_poptop(L);  /* pop `package' */
  if (resetlibname) agnL_setLibname(L, 0, agn_getdebug(L));
  /* get libname */
  lua_getglobal(L, "libname");
  if (!lua_isstring(L, -1))  /* 1.7.4 */
    luaL_error(L, "Error in " LUA_QS ": " LUA_QS " could not be determined.", "restart", "libname");
  else
    path = lua_tostring(L, -1);
  if (path == NULL) {
    fprintf(stderr, "Warning in " LUA_QS ": " LUA_QS " could not be determined, initialisation\nunsuccessful.\n",
      "restart", "libname");
    fflush(stderr);
  }
  else
    agnL_initialise(L, 0, agn_getdebug(L));  /* 0.24.0 */
  agn_poptop(L);  /* drop libname */
  /* get mainlibname, 0.28.2 */
  lua_getglobal(L, "mainlibname");
  if (!lua_isstring(L, -1))  /* 1.7.4 */
    luaL_error(L, "Error in " LUA_QS ": " LUA_QS " could not be determined.", "restart", "mainlibname");  /* 1.9.4 fix */
  else
    path = lua_tostring(L, -1);
  if (path == NULL) {
    fprintf(stderr, "Warning in " LUA_QS ": " LUA_QS " could not be determined.\n", "restart", "mainlibname");
    fflush(stderr);
  }
  agn_poptop(L);  /* drop mainlibname */
  lua_gc(L, LUA_GCCOLLECT, 0);
  return 0;
}


static int environ_pointer (lua_State *L) {
  if (lua_topointer(L, 1) == NULL)
    lua_pushfail(L);
  else
    lua_pushfstring(L, "%p", lua_topointer(L, 1));
  return 1;
}


static void getobjutype (lua_State *L) {
  lua_pushstring(L, "utype");
  if (agn_getutype(L, 1)) {
    lua_rawset(L, -3);
  } else {
    lua_pushfail(L);
    lua_rawset(L, -3);
  }
}

static void isweak (lua_State *L) {
  if (lua_getmetatable(L, 1) != 0) {
    lua_pushstring(L, "weak");
    lua_getfield(L, -2, "__weak");
    if (lua_isstring(L, -1)) {
      lua_rawset(L, -4);
    } else {
      agn_poptop(L);
      lua_pushfail(L);
      lua_rawset(L, -4);
    }
    agn_poptop(L);
  } else {
    lua_pushstring(L, "weak");
    lua_pushfail(L);
    lua_rawset(L, -3);
  }
}

static int environ_attrib (lua_State *L) {  /* added 0.10.0 */
  switch (lua_type(L, 1)) {
    case LUA_TTABLE: {
      size_t a[6];
      lua_newtable(L);
      agn_tablestate(L, 1, a, 1);
      lua_rawsetstringnumber(L, -1, "array_assigned", a[0]);
      lua_rawsetstringnumber(L, -1, "hash_assigned", a[1]);
      lua_rawsetstringboolean(L, -1, "array_hasholes", a[2]);
      lua_rawsetstringboolean(L, -1, "nulls_assigned", a[5]);
      lua_rawsetstringnumber(L, -1, "array_allocated", a[3]);
      lua_rawsetstringnumber(L, -1, "hash_allocated", a[4]);
      lua_rawsetstringnumber(L, -1, "bytes", agn_getstructuresize(L, 1));
      getobjutype(L);
      isweak(L);
      break;
    }
    case LUA_TSET: {
      size_t a[2];
      lua_newtable(L);
      agn_sstate(L, 1, a);
      lua_rawsetstringnumber(L, -1, "hash_assigned", a[0]);
      lua_rawsetstringnumber(L, -1, "hash_allocated", a[1]);
      lua_rawsetstringnumber(L, -1, "bytes", agn_getstructuresize(L, 1));
      getobjutype(L);
      break;
    }
    case LUA_TSEQ: {
      size_t a[2];
      lua_newtable(L);
      agn_seqstate(L, 1, a);
      lua_rawsetstringnumber(L, -1, "size", a[0]);
      lua_rawsetstringnumber(L, -1, "maxsize", a[1]);
      lua_rawsetstringnumber(L, -1, "bytes", agn_getstructuresize(L, 1));
      getobjutype(L);
      isweak(L);
      break;
    }
    case LUA_TPAIR: {
      lua_newtable(L);
      lua_rawsetstringnumber(L, -1, "bytes", agn_getstructuresize(L, 1));
      getobjutype(L);
      break;
    }
    case LUA_TFUNCTION: {
      int a;
      lua_newtable(L);
      lua_pushstring(L, "rtableWritemode");
      /* returns the mode of a remember table:
         1 = true:  function has a remember table and RETURN statement can update the rtable,
         0 = false: has a remember table and RETURN statement canNOT update the rtable,
         2 = fail:  function has no remember table at all
         (-1:       object is not a function) */
      a = agn_getrtablewritemode(L, 1);
      if (a == 2)
        lua_pushfail(L);
      else
        lua_pushboolean(L, a);
      lua_rawset(L, -3);
      lua_rawsetstringboolean(L, -1, "C", agn_getfunctiontype(L, 1));
      lua_rawsetstringnumber(L, -1, "bytes", agn_getstructuresize(L, 1));
      getobjutype(L);
      /* true if `idx` is a C function, false if `idx` is an Agena function */
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": structure or procedure expected, got %s.", "environ.attrib",
        lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


#define unknown(L)  luaL_error(L, "Error in " LUA_QS ": unknown setting " LUA_QS ".", "environ.kernel", lua_tostring(L, -1));

void processoption (lua_State *L, void (*f)(lua_State *, int), int i, const char *option) {  /* 0.32.0 */
  agn_pairgeti(L, i, 2);  /* get right-hand side */
  if (lua_isboolean(L, -1)) {
    (*f)(L, lua_toboolean(L, -1));
    lua_remove(L, -2);  /* remove left-hand side, leave right-hand side (boolean) on stack */
  } else {
    int type = lua_type(L, -1);
    agn_poptop(L);  /* clear stack */
    luaL_error(L, "Error in " LUA_QS ": boolean for " LUA_QS " option expected, got %s.", "environ.kernel", option,
      lua_typename(L, type));
  }
}

static int environ_kernel (lua_State *L) {  /* 0.27.0 */
  int i, nargs;
  const char *setting;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": at least one argument expected.", "environ.kernel");
  } else {
    for (i=1; i <= nargs; i++) {
      if (lua_ispair(L, i)) {
        agn_pairgeti(L, i, 1);  /* get left-hand side */
        if (lua_type(L, -1) != LUA_TSTRING) {
          int type = lua_type(L, -1);
          agn_poptop(L);  /* clear stack */
          luaL_error(L, "Error in " LUA_QS ": string expected for left-hand side, got %s.", "environ.kernel",
            lua_typename(L, type));
        }
        setting = lua_tostring(L, -1);
        if (strcmp(setting, "signedbits") == 0) {
          processoption(L, agn_setbitwise, i, "signedbits");
        } else if (strcmp(setting, "emptyline") == 0) {
          processoption(L, agn_setemptyline, i, "emptyline");
        } else if (strcmp(setting, "libnamereset") == 0) {  /* 0.32.0 */
          processoption(L, agn_setlibnamereset, i, "libnamereset");
        } else if (strcmp(setting, "longtable") == 0) {  /* 0.32.0 */
          processoption(L, agn_setlongtable, i, "longtable");
        } else if (strcmp(setting, "debug") == 0) {  /* 0.32.2a */
          processoption(L, agn_setdebug, i, "debug");
        } else if (strcmp(setting, "gui") == 0) {  /* 0.32.2a */
          processoption(L, agn_setgui, i, "gui");
        } else if (strcmp(setting, "zeroedcomplex") == 0) {  /* 1.7.6 */
          processoption(L, agn_setzeroedcomplex, i, "zeroedcomplex");
        } else if (strcmp(setting, "promptnewline") == 0) {  /* 1.7.6 */
          processoption(L, agn_setpromptnewline, i, "promptnewline");
        } else if (strcmp(setting, "digits") == 0) {
          agn_pairgeti(L, i, 2);  /* get right-hand side */
          if (agn_isnumber(L, -1)) {
            lua_Number x = agn_tonumber(L, -1);
            agn_setdigits(L, x);
            lua_remove(L, -2);  /* remove left-hand side, leave right-hand side (number) on stack */
          } else {
            int type = lua_type(L, -1);
            agn_poptop(L);  /* clear stack */
            luaL_error(L, "Error in " LUA_QS ": number for `digits` option expected,\ngot %s.", "environ.kernel",
              lua_typename(L, type));
          }
        } else
          unknown(L);
      } else if (agn_isstring(L, i)) {
        setting = lua_tostring(L, i);
        if (strcmp(setting, "signedbits") == 0)
          lua_pushboolean(L, agn_getbitwise(L));
        else if (strcmp(setting, "emptyline") == 0)
          lua_pushboolean(L, agn_getemptyline(L));
        else if (strcmp(setting, "libnamereset") == 0) /* 0.32.0 */
          lua_pushboolean(L, agn_getlibnamereset(L));
        else if (strcmp(setting, "longtable") == 0) /* 0.32.0 */
          lua_pushboolean(L, agn_getlongtable(L));
        else if (strcmp(setting, "debug") == 0) /* 0.32.0 */
          lua_pushboolean(L, agn_getdebug(L));
        else if (strcmp(setting, "gui") == 0) /* 0.33.3 */
          lua_pushboolean(L, agn_getgui(L));
        else if (strcmp(setting, "zeroedcomplex") == 0) /* 1.7.6 */
          lua_pushboolean(L, agn_getzeroedcomplex(L));
        else if (strcmp(setting, "promptnewline") == 0) /* 1.7.6 */
          lua_pushboolean(L, agn_getpromptnewline(L));
        else if (strcmp(setting, "digits") == 0) {
          lua_Number n = agn_getdigits(L);
          if (n == 0) luaL_error(L, "Error in " LUA_QS ": internal error.", "environ.kernel");
          lua_pushinteger(L, n);
        } else
          unknown(L);
      } else
        luaL_error(L, "Error in " LUA_QS ": pair or string expected, got %s.", "environ.kernel",
          lua_typename(L, lua_type(L, i)));
    }
  }
  return nargs;
}


static const luaL_Reg environlib[] = {
  {"attrib", environ_attrib},                 /* added April 20, 2008 */
  {"gc", environ_collectgarbage},             /* formerly {"collectgarbage", ...},  changed 0.4.0 */
  {"getfenv", environ_getfenv},
  {"kernel", environ_kernel},                 /* added 27.08.2009 */
  {"pointer", environ_pointer},               /* added May 22, 2009 */
  {"setfenv", environ_setfenv},
  {"used", environ_used},
  {"userinfo", environ_userinfo},             /* added June 14, 2009 */
  {"__RESTART", environ_restart},             /* added December 16, 2006 */
  {NULL, NULL}
};


/*
** Open environ library
*/
LUALIB_API int luaopen_environ (lua_State *L) {
  luaL_register(L, AGENA_ENVIRONLIBNAME, environlib);
  /* new in 0.32.0 */
  lua_rawsetstringnumber(L, -1, "minlong", LUAI_MININT32);
  lua_rawsetstringnumber(L, -1, "maxlong", LUAI_MAXINT32);
  /* lua_rawsetstringnumber(L, -1, "unsignedmaxlong", (LUAI_UINT32));  XXX compiler error: 1.6.0 */
  lua_rawsetstringnumber(L, -1, "buffersize", LUAL_BUFFERSIZE);     /* 0.25.0 */
  lua_rawsetstringnumber(L, -1, "maxpathlength", PATH_MAX-1);       /* 1.6.0 */
  lua_rawsetstringstring(L, -1, "pathsep", LUA_PATHSEP);  /* 0.26.0 */
  /* set number of lines to be printed at once at table output */
  lua_rawsetstringnumber(L, -1, "more", 40);
  /* set protected names for `with` function */
  lua_pushstring(L, "withprotected");
  agn_createset(L, 2);
  lua_sinsertstring(L, -1, "next");
  lua_sinsertstring(L, -1, "print");
  lua_rawset(L, -3);
  return 1;
}


