/*
** $Id: ldo.c,v 2.38 2006/06/05 19:36:14 roberto Exp $
** Stack and Call structure of Lua/Agena
** See Copyright Notice in agena.h
*/


#include <setjmp.h>
#include <stdlib.h>
#include <string.h>

#define ldo_c
#define LUA_CORE

#include "agena.h"

#include "ldebug.h"
#include "ldo.h"
#include "lfunc.h"
#include "lgc.h"
#include "lmem.h"
#include "lobject.h"
#include "lopcodes.h"
#include "lparser.h"
#include "lstate.h"
#include "lstring.h"
#include "ltable.h"
#include "ltm.h"
#include "lundump.h"
#include "lvm.h"
#include "lzio.h"

#include <stdio.h>


/*
** {======================================================
** Error-recovery functions
** =======================================================
*/


/* chain list of long jump buffers */
//struct lua_longjmp {
//  struct lua_longjmp *previous;
//  luai_jmpbuf b;
//  volatile int status;  /* error code */
//};


void luaD_seterrorobj (lua_State *L, int errcode, StkId oldtop) {
  switch (errcode) {
    case LUA_ERRMEM: {
      setsvalue2s(L, oldtop, luaS_newliteral(L, MEMERRMSG));
      break;
    }
    case LUA_ERRERR: {
      setsvalue2s(L, oldtop, luaS_newliteral(L, "error in error handling"));
      break;
    }
    case LUA_ERRSYNTAX:
    case LUA_ERRRUN: {
      setobjs2s(L, oldtop, L->top - 1);  /* error message on current top */
      break;
    }
  }
  L->top = oldtop + 1;
}


static void restore_stack_limit (lua_State *L) {
  lua_assert(L->stack_last - L->stack == L->stacksize - EXTRA_STACK - 1);
  if (L->size_ci > LUAI_MAXCALLS) {  /* there was an overflow? */
    int inuse = cast_int(L->ci - L->base_ci);
    if (inuse + 1 < LUAI_MAXCALLS)  /* can `undo' overflow? */
      luaD_reallocCI(L, LUAI_MAXCALLS);
  }
}


static void resetstack (lua_State *L, int status) {
  L->ci = L->base_ci;
  L->base = L->ci->base;
  luaF_close(L, L->base);  /* close eventual pending closures */
  luaD_seterrorobj(L, status, L->base);
  L->nCcalls = L->baseCcalls;  /* 0.13.3 */
  L->allowhook = 1;
  restore_stack_limit(L);
  L->errfunc = 0;
  L->errorJmp = NULL;
}


void luaD_throw (lua_State *L, int errcode) {
  if (L->errorJmp) {
    L->errorJmp->status = errcode;
    LUAI_THROW(L, L->errorJmp);
  }
  else {
    L->status = cast_byte(errcode);
    if (G(L)->panic) {
      resetstack(L, errcode);
      lua_unlock(L);
      G(L)->panic(L);
    }
    exit(EXIT_FAILURE);
  }
}


int luaD_rawrunprotected (lua_State *L, Pfunc f, void *ud) {  /* changed Agena 2.1 RC 2, written by Hu Qiwei */
  struct lua_longjmp lj;
  lj.type = JMPTYPE_LONGJMP;
  lj.status = 0;
  lj.previous = L->errorJmp;  /* chain new error handler */
  L->errorJmp = &lj;
  LUAI_TRY(L, &lj,
    (*f)(L, ud);
  );
  L->errorJmp = lj.previous;  /* restore old error handler */
  return lj.status;
}

/* }====================================================== */

/* hashing routines for rtable */

#define hashpow2(n)      lmod((n), LUAI_MAXCALLS)

#define hashstr(str)     hashpow2((str)->tsv.hash)
#define hashboolean(p)   hashpow2(p)

#define hashmod(n)       ((n) % ((LUAI_MAXCALLS-1)|1))
#define hashpointer(p)   hashmod(IntPoint(p))

/* number of ints inside a lua_Number */
#define numints          cast_int(sizeof(lua_Number)/sizeof(int))


/* hash for lua_Numbers */
static int hashnum (lua_Number n) {
  unsigned int a[numints];
  int i;
  if (luai_numeq(n, 0))
    return 0;
  memcpy(a, &n, sizeof(a));
  for (i = 1; i < numints; i++) a[0] += a[i];
  return hashmod(a[0]);
}


LUAI_FUNC int hash (const TValue *key) {
  switch (ttype(key)) {
    case LUA_TNUMBER:
      return hashnum(nvalue(key));
    case LUA_TSTRING:
      return hashstr(rawtsvalue(key));
    case LUA_TBOOLEAN:
      return hashboolean(bvalue(key));
    case LUA_TLIGHTUSERDATA:
      return hashpointer(pvalue(key));
    default:
      return hashpointer(gcvalue(key));
  }
}

/* ************************************************************************* */


static void correctstack (lua_State *L, TValue *oldstack) {
  CallInfo *ci;
  GCObject *up;
  L->top = (L->top - oldstack) + L->stack;
  for (up = L->openupval; up != NULL; up = up->gch.next)
    gco2uv(up)->v = (gco2uv(up)->v - oldstack) + L->stack;
  for (ci = L->base_ci; ci <= L->ci; ci++) {
    ci->top = (ci->top - oldstack) + L->stack;
    ci->base = (ci->base - oldstack) + L->stack;
    ci->func = (ci->func - oldstack) + L->stack;
  }
  L->base = (L->base - oldstack) + L->stack;
}


void luaD_reallocstack (lua_State *L, int newsize) {
  TValue *oldstack = L->stack;
  int realsize = newsize + 1 + EXTRA_STACK;
  lua_assert(L->stack_last - L->stack == L->stacksize - EXTRA_STACK - 1);
  luaM_reallocvector(L, L->stack, L->stacksize, realsize, TValue);
  L->stacksize = realsize;
  L->stack_last = L->stack+newsize;
  correctstack(L, oldstack);
}


void luaD_reallocCI (lua_State *L, int newsize) {
  CallInfo *oldci = L->base_ci;
  luaM_reallocvector(L, L->base_ci, L->size_ci, newsize, CallInfo);
  L->size_ci = newsize;
  L->ci = (L->ci - oldci) + L->base_ci;
  L->end_ci = L->base_ci + L->size_ci - 1;
}


void luaD_growstack (lua_State *L, int n) {
  if (n <= L->stacksize)  /* double size is enough? */
    luaD_reallocstack(L, 2*L->stacksize);
  else
    luaD_reallocstack(L, L->stacksize + n);
}


static CallInfo *growCI (lua_State *L) {
  if (L->size_ci > LUAI_MAXCALLS)  /* overflow while handling overflow? */
    luaD_throw(L, LUA_ERRERR);
  else {
    luaD_reallocCI(L, 2*L->size_ci);
    if (L->size_ci > LUAI_MAXCALLS)
      luaG_runerror(L, "stack overflow");
  }
  return ++L->ci;
}


void luaD_callhook (lua_State *L, int event, int line) {
  lua_Hook hook = L->hook;
  if (hook && L->allowhook) {
    ptrdiff_t top = savestack(L, L->top);
    ptrdiff_t ci_top = savestack(L, L->ci->top);
    lua_Debug ar;
    ar.event = event;
    ar.currentline = line;
    if (event == LUA_HOOKTAILRET)
      ar.i_ci = 0;  /* tail call; no debug information about it */
    else
      ar.i_ci = cast_int(L->ci - L->base_ci);
    luaD_checkstack(L, LUA_MINSTACK);  /* ensure minimum stack size */
    L->ci->top = L->top + LUA_MINSTACK;
    lua_assert(L->ci->top <= L->stack_last);
    L->allowhook = 0;  /* cannot call hooks inside a hook */
    lua_unlock(L);
    (*hook)(L, &ar);
    lua_lock(L);
    lua_assert(!L->allowhook);
    L->allowhook = 1;
    L->ci->top = restorestack(L, ci_top);
    L->top = restorestack(L, top);
  }
}


static StkId adjust_varargs (lua_State *L, Proto *p, int actual) {
  int i;
  int nfixargs = p->numparams;
  Table *htab = NULL;
  StkId base, fixed;
  for (; actual < nfixargs; ++actual)
    setnilvalue(L->top++);
  if (p->is_vararg & VARARG_NEEDSARG) { /* compat. with old-style vararg? */
    int nvar = actual - nfixargs;  /* number of extra arguments */
    lua_assert(p->is_vararg & VARARG_HASARG);
    luaC_checkGC(L);
    htab = luaH_new(L, nvar, 1);  /* create `arg' table */
    for (i=0; i<nvar; i++)  /* put extra arguments into `arg' table */
      setobj2n(L, luaH_setnum(L, htab, i+1), L->top - nvar + i);
    /* store counter in field `n' */
  }
  /* move fixed parameters to final position */
  fixed = L->top - actual;  /* first fixed argument */
  base = L->top;  /* final position of first argument */
  for (i=0; i<nfixargs; i++) {
    setobjs2s(L, L->top++, fixed+i);
    setnilvalue(fixed+i);
  }
  /* add `arg' parameter */
  if (htab) {
    sethvalue(L, L->top++, htab);
    lua_assert(iswhite(obj2gco(htab)));
  }
  return base;
}


static StkId tryfuncTM (lua_State *L, StkId func) {
  const TValue *tm = luaT_gettmbyobj(L, func, TM_CALL);
  StkId p;
  ptrdiff_t funcr = savestack(L, func);
  if (!ttisfunction(tm))
    luaG_typeerror(L, func, "call");
  /* Open a hole inside the stack at `func' */
  for (p = L->top; p > func; p--) setobjs2s(L, p, p-1);
  incr_top(L);
  func = restorestack(L, funcr);  /* previous call may change stack */
  setobj2s(L, func, tm);  /* tag method is the new function to be called */
  return func;
}


/* 0.22.1: function to search a remember table, returns the index of the first result. */
static StkId searchrtable (lua_State *L, LClosure *cl, StkId func, int nargs, int *flag) {
  int i, j, k, res, paramc, hashval;
  Table *rk, *rt, *r;
  const TValue *entry;
  StkId rbase;
  Node *node;
  rbase = func;  /* to avoid warnings */
  rt = cl->rtable;
  *flag = res = 0;
  /* traverse remember table to check whether there is a result already */
  hashval = 0;
  for (i=0; i<nargs; i++)
    hashval += hash(func+1+i);  /* compute hash for all arguments passed */
  entry = luaH_getnum(rt, hashval);
  if (!ttisnil(entry)) {
    r = hvalue(entry);
    for (i = 0; i < sizenode(r); i++) {
      node = gnode(r, i);
      if (ttisnotnil(gval(node))) {  /* key (a table) found */
        rk = hvalue(key2tval(node));  /* assign key to Table object rk */
        /* prepare argument table traversal */
        setnilvalue(L->top);
        api_check(L, L->top < L->ci->top);
        j = findindex(L, rk, L->top);
        *flag = 1;  /* argument found in rtable ? */
        if (nargs > 0) {
          /* traverse each argument in key */
          paramc = 0;  /* number of arguments in rtable value table */
          for (j++; j < rk->sizearray; j++) {
            if (ttisnotnil(&rk->array[j])) {  /* check current argument */
              *flag = 1;
              paramc++;
              /* do not check values beyond func+paramc */
              if (paramc <= nargs && !equalobj(L, &rk->array[j], func+paramc)) {
                *flag = 0;
                break;
              }
            }
          }  /* of for */
          if (*flag && nargs == paramc) {
            /* set results in rtable to stack */
            Table *r;
            r = hvalue(gval(node));
            setnilvalue(L->top);
            api_check(L, L->top < L->ci->top);
            k = findindex(L, r, L->top);
            rbase = L->top;  /* index of first result */
            for (k++; k < r->sizearray; k++) {
              setobj2s(L, L->top++, &r->array[k]);
              luaD_checkstack(L, LUA_MINSTACK);  /* ensure minimum stack size */
            }
            break;  /* quit rtable iteration */
          }
          *flag = 0;
        }
      }
    }
  }
  return rbase;
}


#define inc_ci(L) \
  ((L->ci == L->end_ci) ? growCI(L) : \
   (condhardstacktests(luaD_reallocCI(L, L->size_ci)), ++L->ci))

/* fixed in Agena 1.4.2 */
#define checkutype(L) \
  if (s->type == NULL || strcmp(getstr(s->type), typets) != 0) \
    luaG_runerror(L, "type %s expected for argument #%d, got %s.", \
      typets, i+1, luaT_typenames[(int)ttype(base+i)]); \
  break;


int luaD_precall (lua_State *L, StkId func, int nresults) {
  LClosure *cl;
  StkId rbase;
  ptrdiff_t funcr;
  if (!ttisfunction(func)) { /* `func' is not a function? */
    func = tryfuncTM(L, func);  /* check the `function' tag method */
  }
  funcr = savestack(L, func);
  cl = &clvalue(func)->l;
  L->ci->savedpc = L->savedpc;
  rbase = NULL;
  if (!cl->isC) {  /* Agena function ?  Prepare its call */
    CallInfo *ci;
    StkId st, base, oldtop;
    Proto *p = cl->p;
    luaD_checkstack(L, p->maxstacksize);
    func = restorestack(L, funcr);
    oldtop = L->top;  /* for later determination of NARGS */
    if (!p->is_vararg) {  /* no varargs? */
      base = func + 1;
      if (L->top > base + p->numparams)
        L->top = base + p->numparams;
    }
    else {  /* vararg function */
      int nargs = cast_int(L->top - func) - 1;
      base = adjust_varargs(L, p, nargs);
      func = restorestack(L, funcr);  /* previous call may change the stack */
    }
    ci = inc_ci(L);  /* now `enter' new function */
    ci->func = func;
    L->base = ci->base = base;
    ci->top = L->base + p->maxstacksize;
    lua_assert(ci->top <= L->stack_last);
    L->savedpc = p->code;  /* starting point */
    ci->tailcalls = 0;
    ci->nresults = nresults;
    /* determine number of arguments for NARGS sysvar */
    ci->nargs = cast_byte(oldtop-func-1);
    if (p->is_typegiven) {  /* new type checking method, 0.20.0, modified 0.21.0 */
      int i, type;
      const char *typets;
      /* p->numparams must never be exceeded, which will be the case if surplus arguments are passed ! */
      for (i=0; i < p->numparams; i++) {
        /* first check the arguments actually passed */
        if ((type = p->locvars[i].vartype) != -1) {  /* optional basic type specified ? */
          if (i < cast_byte(oldtop-func-1)) {
            if (ttype(base+i) != type)
              luaG_runerror(L, "type %s expected for argument #%d: got %s.",
                luaT_typenames[(int)type], i+1, luaT_typenames[(int)ttype(base+i)]);
          } else
              luaG_runerror(L, "missing argument #%d (of type %s).", i+1,
                luaT_typenames[(int)type]);
        }
        else if (p->locvars[i].vartypets != NULL) {  /* optional user-defined type specified ? */
          typets = getstr(p->locvars[i].vartypets);
          if (i < cast_byte(oldtop-func-1)) {
            switch (ttype(base+i)) {
              case LUA_TSEQ: {
                Seq *s = seqvalue(base+i);
                checkutype(L);
              }
              case LUA_TTABLE: {
                Table *s = hvalue(base+i);
                checkutype(L);
              }
              case LUA_TSET: {
                UltraSet *s = usvalue(base+i);
                checkutype(L);
              }
              case LUA_TPAIR: {
                Pair *s = pairvalue(base+i);
                checkutype(L);
              }
              case LUA_TFUNCTION: {
                Closure *c = clvalue(base+i);
                if ((c->l.type != NULL && strcmp(getstr(c->l.type), typets) != 0) ||
                   (c->c.type != NULL && strcmp(getstr(c->c.type), typets) != 0))
                  luaG_runerror(L, "type %s expected for argument #%d, got %s.",
                    typets, i+1, luaT_typenames[(int)ttype(base+i)]);
                break;
              }
              default:  /* all other types */
                luaG_runerror(L, "type %s expected for argument #%d, got %s.",
                  typets, i+1, luaT_typenames[(int)ttype(base+i)]);
            }
          } else  /* Agena 1.4.2 */
              luaG_runerror(L, "missing argument #%d (of type %s).",
                i+1, typets);
        }
      }
    }
    for (st = L->top; st < ci->top; st++)
      setnilvalue(st);
    L->top = ci->top;
    if (L->hookmask & LUA_MASKCALL) {
      L->savedpc++;  /* hooks assume 'pc' is already incremented */
      luaD_callhook(L, LUA_HOOKCALL, -1);
      L->savedpc--;  /* correct 'pc' */
    }
    if (cl->rtable == NULL) {
      return PCRLUA;
    }
    else {  /* search remember table */
      int flag;
      rbase = searchrtable(L, cl, func, p->numparams, &flag);
      if (flag) {
        /* entry found and put on stack, do not execute function */
        luaD_poscall(L, rbase);
        return PCRREMEMBER;
      }
      else {  /* no entry found in rtable, execute body */
        return PCRLUA;
      }
    }
  }
  else {  /* if is a C function, call it */
    CallInfo *ci;
    StkId base;
    int n;
    luaD_checkstack(L, LUA_MINSTACK);  /* ensure minimum stack size */
    ci = inc_ci(L);  /* now `enter' new function */
    ci->func = restorestack(L, funcr);
    base = L->base = ci->base = ci->func + 1;
    ci->top = L->top + LUA_MINSTACK;
    lua_assert(ci->top <= L->stack_last);
    ci->nresults = nresults;
    if (cl->rtable != NULL) { /* search remember table, new 0.22.1 */
      int flag;
      rbase = searchrtable(L, cl, func, L->top-base, &flag);
      if (flag) {
        /* entry found and put on stack, do not execute C function */
        luaD_poscall(L, rbase);
        return PCRC;
      }
    }
    if (L->hookmask & LUA_MASKCALL)
      luaD_callhook(L, LUA_HOOKCALL, -1);
    lua_unlock(L);
    n = (*curr_func(L)->c.f)(L);  /* do the actual call */
    lua_lock(L);
    if (n < 0)  /* yielding? */
      return PCRYIELD;
    else {
      luaD_poscall(L, L->top - n);
      return PCRC;
    }
  }
}


static StkId callrethooks (lua_State *L, StkId firstResult) {
  ptrdiff_t fr = savestack(L, firstResult);  /* next call may change stack */
  luaD_callhook(L, LUA_HOOKRET, -1);
  if (f_isLua(L->ci)) {  /* Lua function? */
    while ((L->hookmask & LUA_MASKRET) && L->ci->tailcalls--) /* tail calls, 0.13.3 */
      luaD_callhook(L, LUA_HOOKTAILRET, -1);
  }
  return restorestack(L, fr);
}


int luaD_poscall (lua_State *L, StkId firstResult) {
  StkId res;
  int wanted, i;
  CallInfo *ci;
  if (L->hookmask & LUA_MASKRET)
    firstResult = callrethooks(L, firstResult);
  ci = L->ci--;
  res = ci->func;  /* res == final position of 1st result */
  wanted = ci->nresults;
  L->base = (ci - 1)->base;  /* restore base */
  L->savedpc = (ci - 1)->savedpc;  /* restore savedpc */
  /* move results to correct place */
  for (i = wanted; i != 0 && firstResult < L->top; i--)
    setobjs2s(L, res++, firstResult++);
  while (i-- > 0)
    setnilvalue(res++);
  L->top = res;
  return (wanted - LUA_MULTRET);  /* 0 iff wanted == LUA_MULTRET */
}


/*
** Call a function (C or Lua). The function to be called is at *func.
** The arguments are on the stack, right after the function.
** When returns, all the results are on the stack, starting at the original
** function position.
*/
void luaD_call (lua_State *L, StkId func, int nResults) {
  if (++L->nCcalls >= LUAI_MAXCCALLS) {
    if (L->nCcalls == LUAI_MAXCCALLS)
      luaG_runerror(L, "C stack overflow");
    else if (L->nCcalls >= (LUAI_MAXCCALLS + (LUAI_MAXCCALLS>>3)))
      luaD_throw(L, LUA_ERRERR);  /* error while handling stack error */
  }
  if (luaD_precall(L, func, nResults) == PCRLUA) { /* is a Lua function? */
    luaV_execute(L, 1);  /* call it */
  }
  L->nCcalls--;
  luaC_checkGC(L);
}


static void resume (lua_State *L, void *ud) {
  StkId firstArg = cast(StkId, ud);
  CallInfo *ci = L->ci;
  if (L->status == 0) {  /* start coroutine? */
    lua_assert(ci == L->base_ci && firstArg > L->base);
    if (luaD_precall(L, firstArg - 1, LUA_MULTRET) != PCRLUA)
      return;
  }
  else {  /* resuming from previous yield */
    lua_assert(L->status == LUA_YIELD);
    L->status = 0;
    if (!f_isLua(ci)) {  /* `common' yield? */
      /* finish interrupted execution of `OP_CALL' */
      lua_assert(GET_OPCODE(*((ci-1)->savedpc - 1)) == OP_CALL ||
                 GET_OPCODE(*((ci-1)->savedpc - 1)) == OP_TAILCALL);
      if (luaD_poscall(L, firstArg))  /* complete it... */
        L->top = L->ci->top;  /* and correct top if not multiple results */
    }
    else  /* yielded inside a hook: just continue its execution */
      L->base = L->ci->base;
  }
  luaV_execute(L, cast_int(L->ci - L->base_ci));
}


static int resume_error (lua_State *L, const char *msg) {
  L->top = L->ci->base;
  setsvalue2s(L, L->top, luaS_new(L, msg));
  incr_top(L);
  lua_unlock(L);
  return LUA_ERRRUN;
}


LUA_API int lua_resume (lua_State *L, int nargs) {  /* 0.13.3 */
  int status;
  lua_lock(L);
  if (L->status != LUA_YIELD && (L->status != 0 || L->ci != L->base_ci))
      return resume_error(L, "cannot resume non-suspended coroutine");
  if (L->nCcalls >= LUAI_MAXCCALLS)
    return resume_error(L, "C stack overflow");
  luai_userstateresume(L, nargs);
  lua_assert(L->errfunc == 0);
  L->baseCcalls = ++L->nCcalls;
  status = luaD_rawrunprotected(L, resume, L->top - nargs);
  if (status != 0) {  /* error? */
    L->status = cast_byte(status);  /* mark thread as `dead' */
    luaD_seterrorobj(L, status, L->top);
    L->ci->top = L->top;
  }
  else {
    lua_assert(L->nCcalls == L->baseCcalls);
    status = L->status;
  }
  --L->nCcalls;
  lua_unlock(L);
  return status;
}


LUA_API int lua_yield (lua_State *L, int nresults) {
  luai_userstateyield(L, nresults);
  lua_lock(L);
  if (L->nCcalls > L->baseCcalls)  /* 0.13.3 */
    luaG_runerror(L, "attempt to yield across metamethod/C-call boundary");
  L->base = L->top - nresults;  /* protect stack slots below */
  L->status = LUA_YIELD;
  lua_unlock(L);
  return -1;
}


int luaD_pcall (lua_State *L, Pfunc func, void *u,
                ptrdiff_t old_top, ptrdiff_t ef) {
  int status;
  unsigned short oldnCcalls = L->nCcalls;
  ptrdiff_t old_ci = saveci(L, L->ci);
  lu_byte old_allowhooks = L->allowhook;
  ptrdiff_t old_errfunc = L->errfunc;
  L->errfunc = ef;
  status = luaD_rawrunprotected(L, func, u);
  if (status != 0) {  /* an error occurred? */
    StkId oldtop = restorestack(L, old_top);
    luaF_close(L, oldtop);  /* close eventual pending closures */
    luaD_seterrorobj(L, status, oldtop);
    L->nCcalls = oldnCcalls;
    L->ci = restoreci(L, old_ci);
    L->base = L->ci->base;
    L->savedpc = L->ci->savedpc;
    L->allowhook = old_allowhooks;
    restore_stack_limit(L);
  }
  L->errfunc = old_errfunc;
  return status;
}



/*
** Execute a protected parser.
*/
struct SParser {  /* data to `f_parser' */
  ZIO *z;
  Mbuffer buff;  /* buffer to be used by the scanner */
  const char *name;
};

static void f_parser (lua_State *L, void *ud) {
  int i;
  Proto *tf;
  Closure *cl;
  struct SParser *p = cast(struct SParser *, ud);
  int c = luaZ_lookahead(p->z);
  luaC_checkGC(L);
  tf = ((c == LUA_SIGNATURE[0]) ? luaU_undump : luaY_parser)(L, p->z, &p->buff, p->name);
  cl = luaF_newLclosure(L, tf->nups, hvalue(gt(L)));
  cl->l.p = tf;
  for (i = 0; i < tf->nups; i++)  /* initialize eventual upvalues */
    cl->l.upvals[i] = luaF_newupval(L);
  setclvalue(L, L->top, cl);
  incr_top(L);
}


int luaD_protectedparser (lua_State *L, ZIO *z, const char *name) {
  struct SParser p;
  int status;
  p.z = z; p.name = name;
  luaZ_initbuffer(L, &p.buff);
  status = luaD_pcall(L, f_parser, &p, savestack(L, L->top), L->errfunc);
  luaZ_freebuffer(L, &p.buff);
  return status;
}


/* enters the results of a function into the remember table (if created); 0.9.2
   base: the first argument of the function */
void luaD_rtableentry (lua_State *L, LClosure *cl, StkId base, StkId firstResult, int nresults) {
  int i, nargs, hashval, type;
  Table *rkey, *rval, *rtab;
  StkId basepi;
  const TValue *entry;
  nargs = cl->p->numparams;
  /* create a table with the arguments, this table will be a key in the rtable */
  rkey = luaH_new(L, nargs, 0);
  hashval = 0;
  for (i=0; i < nargs; i++) {
    basepi = base+i;
    type = ttype(basepi);
    if (type > LUA_TSTRING || type < LUA_TBOOLEAN )
      /* 0.12.0: else out of memory errors */
      luaG_runerror(L, "only basic types allowed in remember tables, i.e. numbers,\n"
                       "strings, and booleans including fail. Do not pass any other type.");
    hashval += hash(basepi);
    setobjt2t(L, luaH_setnum(L, rkey, i+1), basepi);
    luaC_barriert(L, rkey, basepi);
  }
  sethvalue(L, L->top, rkey);
  /* table array for results */
  rval = luaH_new(L, nresults, 0);
  /* enter results into the table array */
  for (i=0; i < nresults; i++) {
    setobjt2t(L, luaH_setnum(L, rval, i+1), firstResult+i);
    luaC_barriert(L, rval, firstResult+i);
  }
  luaD_checkstack(L, LUA_MINSTACK);  /* ensure minimum stack size */
  L->top++;
  /* set table with results to stack top */
  sethvalue(L, L->top, rval);
  /* L->top-1: argument table
     L->top:   result table */
  entry = luaH_getnum(cl->rtable, hashval);
  if ttisnil(entry) {  /* no entry yet ? */
    /* create a table */
    rtab = luaH_new(L, 0, 1);
  } else {
    rtab = hvalue(entry);
  }
  /* enter arguments~results pair into table */
  setobjt2t(L, luaH_set(L, rtab, L->top-1), L->top);
  luaC_barriert(L, rtab, L->top);
  /* set table to stack top */
  sethvalue(L, L->top, rtab);
  /* enter key~value pair (arguments~results) into rtable */
  setobjt2t(L, luaH_setnum(L, cl->rtable, hashval), L->top);
  luaC_barriert(L, cl->rtable, L->top);
  L->top--;
}



