/** $Id: lbaselib.c,v 1.191 15.03.2009 12:38:02 roberto Exp $
** Basic library
   Revision 25.12.2009 07:53:37
** See Copyright Notice in agena.h
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifdef _WIN32
#include <conio.h>   /* getch */
#include <io.h>      /* access in MinGW */
#endif

#if defined(__unix__) || defined(__APPLE__) || defined(__HAIKU__)
#include <unistd.h>  /* access in UNIX */
#endif

#if defined(__OS2__)
#include <conio.h>   /* getch */
#include <unistd.h>  /* access */
#endif

#include <math.h>    /* fabs, isnan, fmod, HUGE_VAL */

#define lbaselib_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "loadlib.h"   /* for require subfunction fastload* */

#include "agnhlps.h"  /* for Pi and Exp */
#include "agncmpt.h"  /* for fmax */
#include "cephes.h"   /* for gamma, PIO2 */
#include "llimits.h"  /* for cast */

/*
** If your system does not support `stdout', you can just remove this function.
** If you need, you can define your own `print' function, following this
** model but changing `fputs' to put the strings at a proper place
** (a console window or a log file, for instance).
*/

/* this macro checks whether there is a user defined printing function for the requested object
   and if so calls this function; otherwise the macro is `skipped`. */

#define auxuserprint(L, fn) { \
  lua_getfield(L, -1, fn); \
  if (lua_isfunction(L, -1)) { \
    lua_pushvalue(L, -3); \
    lua_call(L, 1, 0); \
    agn_poptoptwo(L); \
    continue; \
  } else \
    agn_poptop(L); \
  break; \
}

static int luaB_print (lua_State *L) {
  int i, pprinter, nonewline, nargs, freedelim;
  char *delim = NULL;  /* to prevent compiler warnings, 1.12.9 */
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");  /* 1.10.5 */
  nonewline = 0;
  freedelim = 0;  /* 1.8.11 fix */
  if (lua_ispair(L, nargs)) {  /* 0.30.4 */
    agn_pairgeti(L, nargs, 1);  /* get left value, set to stack index -2 */
    agn_pairgeti(L, nargs, 2);  /* get right value, set to stack index  -1 */
    if (lua_type(L, -2) == LUA_TSTRING && strcmp("nonewline", agn_tostring(L, -2)) == 0) {
      nargs--;
      nonewline = lua_toboolean(L, -1);
    } else if (lua_type(L, -2) == LUA_TSTRING && strcmp("delim", agn_tostring(L, -2)) == 0) {
      nargs--;
      delim = strdup(agn_checkstring(L, -1));  /* Agena 1.0.4, 1.8.11 */
      freedelim = 1;
    }
    agn_poptoptwo(L);  /* 0.30.4 */
  }
  if (freedelim == 0) delim = "\t";  /* 1.8.11 fix */
  for (i=1; i <= nargs; i++) {
    if (i > 1) fputs(delim, stdout);
    if ((pprinter = luaL_callmeta(L, i, "__tostring"))) { /* is there a metamethod ?  Agena 1.3.2 fix */
      agn_poptop(L);  /* pop pretty printer */
    }
    lua_pushvalue(L, i);
    if (!pprinter) {  /* no __string metamethod ? */
      agnL_gettablefield(L, "environ", "aux", "print", 1);  /* Agena 1.7.0 */
      if (lua_type(L, -1) == LUA_TTABLE) { /* environ.aux is not a table ? */
        switch (lua_type(L, -2)) {
          case LUA_TTABLE:
            auxuserprint(L, "printtable");
          case LUA_TSET:
            auxuserprint(L, "printset");
          case LUA_TSEQ:
            auxuserprint(L, "printsequence");
          case LUA_TPAIR:
            auxuserprint(L, "printpair");
          case LUA_TCOMPLEX:
            auxuserprint(L, "printcomplex");
          case LUA_TFUNCTION:  /* 0.28.0 */
            auxuserprint(L, "printprocedure");
          default: break;
        }
      }
      agn_poptop(L);  /* remove environ.aux.<field> */
    }
    /* if no user-defined formatting function for the given structure could be found or if there
       is a `__string` metamethod for the object, then the next line will be executed. */
    agnL_printnonstruct(L, -1);  /* aux_printobject can only be called with a negative index */
    agn_poptop(L);
  }
  if (!nonewline) fputs("\n", stdout);
  fflush(stdout);
  if (freedelim) xfree(delim);  /* 1.8.11 fix */
  return 0;
}


static int luaB_mprint (lua_State *L) {
  int n = lua_gettop(L);  /* number of arguments */
  int i; int j;
  char space[2];  /* changed 0.5.3 */
  lua_getglobal(L, "tostring");
  strcpy(space, " \0");
  for (i=1; i<=n; i++) {
    const char *s;
    lua_pushvalue(L, -1);  /* function to be called */
    lua_pushvalue(L, i);   /* value to print */
    lua_call(L, 1, 1);
    s = lua_tostring(L, -1);  /* get result */
    if (s == NULL)
      return luaL_error(L, LUA_QL("tostring") " must return a string to "
                           LUA_QL("mprint") ".");
    if (n == 1) for (j=0; j<(80-strlen(s))/2; j++) fputs(space, stdout);
    if (i>1) fputs("\t", stdout);
    fputs(s, stdout);
    agn_poptop(L);  /* pop result */
  }
  return 0;
}


static int luaB_tonumber (lua_State *L) {
  int base = agnL_optinteger(L, 2, 10);
  if (base == 10) {  /* standard conversion */
    lua_Number n;
    int exception;
    luaL_checkany(L, 1);
    if (agn_isstring(L, 1) || agn_isnumber(L, 1)) {  /* changed 0.12.2 */
      n = agn_tonumberx(L, 1, &exception);
      if (exception) { /* conversion failed ? Check whether string contains a complex value */
        int cexception;
#ifndef PROPCMPLX
        agn_Complex c;
        c = agn_tocomplexx(L, 1, &cexception);
#else
        lua_Number c[2];
        agn_tocomplexx(L, 1, &cexception, c);
#endif
        if (cexception == 0)
#ifndef PROPCMPLX
          agn_createcomplex(L, c);
#else
          agn_createcomplex(L, c[0], c[1]);
#endif
        else
          lua_pushvalue(L, 1);
      }
      else
        lua_pushnumber(L, n);
      return 1;
    } else if (lua_type(L, 1) == LUA_TCOMPLEX) {
      lua_pushvalue(L, 1);
      return 1;
    }
  }
  else {
    const char *s1 = luaL_checkstring(L, 1);
    char *s2;
    unsigned long n;
    luaL_argcheck(L, 2 <= base && base <= 36, 2, "base out of range");
    n = strtoul(s1, &s2, base);
    if (s1 != s2) {  /* at least one valid digit? */
      while (isspace((unsigned char)(*s2))) s2++;  /* skip trailing spaces */
      if (*s2 == '\0') {  /* no invalid trailing characters? */
        lua_pushnumber(L, (lua_Number)n);
        return 1;
      }
    }
  }
  lua_pushfail(L);  /* else not a number */
  return 1;
}


static int luaB_error (lua_State *L) {
  int level = agnL_optinteger(L, 2, 1);
  lua_settop(L, 1);
  if (agn_isstring(L, 1) && level > 0) {  /* add extra information? */
    luaL_where(L, level);
    lua_pushvalue(L, 1);
    lua_concat(L, 2);
  }
  return lua_error(L);
}


static int luaB_getmetatable (lua_State *L) {
  luaL_checkany(L, 1);
  if (!lua_getmetatable(L, 1)) {
    lua_pushnil(L);
    return 1;  /* no metatable */
  }
  luaL_getmetafield(L, 1, "__metatable");
  return 1;  /* returns either __metatable field (if present) or metatable */
}


static int luaB_setmetatable (lua_State *L) {
  int s = lua_type(L, 1);
  int t = lua_type(L, 2);
  luaL_typecheck(L, s == LUA_TTABLE || s == LUA_TSEQ || s == LUA_TSET || s == LUA_TPAIR , 1,
                    "table, set, sequence, or pair expected", s);
  luaL_typecheck(L, t == LUA_TNIL || t == LUA_TTABLE, 2,
                    "null or table expected", t);
  if (luaL_getmetafield(L, 1, "__metatable"))
    luaL_error(L, "Error in " LUA_QS ": cannot change a protected metatable.", "setmetatable");
  lua_settop(L, 2);
  lua_setmetatable(L, 1);
  return 1;
}


static int luaB_rawequal (lua_State *L) {
  luaL_checkany(L, 1);
  luaL_checkany(L, 2);
  lua_pushboolean(L, lua_rawequal(L, 1, 2));
  return 1;
}


static int luaB_rawget (lua_State *L) {
  int s = lua_type(L, 1);
  luaL_checkany(L, 2);
  lua_settop(L, 2);
  switch (s) {
    case LUA_TTABLE:
      lua_rawget(L, 1);
      break;
    case LUA_TSET:
      lua_srawget(L, 1);
      break;
    case LUA_TSEQ:
      lua_seqrawget(L, 1);
      break;
    case LUA_TPAIR:
      agn_pairrawget(L, 1);
      break;
    default:
      luaL_error(L, "Error in " LUA_QS ": table, set, sequence, or pair expected, got %s.", "rawget",
        lua_typename(L, s));
  }
  return 1;
}


/* the switch/case version is a little bit slower */
static int luaB_rawset (lua_State *L) {
  int s = lua_type(L, 1);
  luaL_checkany(L, 2);
  /* luaL_checkany(L, narg): Checks whether the function has an argument of
     any type (including nil) at position narg. */
  if (s == LUA_TTABLE) {
    if (lua_gettop(L) == 2) {  /* 0.21.0, index missing ? */
      lua_pushinteger(L, agn_size(L, 1) + 1);  /* determine next free index */
      lua_insert(L, 2);  /* move it into arg position #2 */
    } else
      luaL_checkany(L, 3);
    lua_settop(L, 3);
    lua_rawset(L, 1);
  }
  else if (s == LUA_TSET) {
    lua_settop(L, 2);
    lua_srawset(L, 1);
  }
  else if (s == LUA_TSEQ) {
    if (lua_gettop(L) == 2) {  /* 0.21.0, index missing ? */
      lua_pushinteger(L, agn_seqsize(L, 1) + 1);  /* determine next free index */
      lua_insert(L, 2);  /* move it into arg position #2 */
    } else
      luaL_checkany(L, 3);
    lua_settop(L, 3);
    lua_seqset(L, 1);
  }
  else if (s == LUA_TPAIR) {
    luaL_checkany(L, 3);
    lua_settop(L, 3);
    agn_pairrawset(L, 1);
  }
  else
    luaL_error(L, "Error in " LUA_QS ": table, set, sequence, or pair expected, got %s.", "rawset",
      lua_typename(L, s));
  return 1;
}


static int luaB_next (lua_State *L) {  /* 0.10.0: extended to handle sets, as well */
  switch (lua_type(L, 1)) {
    case LUA_TTABLE: {
      lua_settop(L, 2);  /* create a 2nd argument if there isn't one */
      if (lua_next(L, 1))
        return 2;
      else {
        lua_pushnil(L);
        return 1;
      }
    }
    case LUA_TSET: {
      lua_settop(L, 2);  /* create a 2nd argument if there isn't one */
      if (lua_usnext(L, 1))
        return 2;
      else {
        lua_pushnil(L);
        return 1;
      }
    }
    case LUA_TSEQ: {
      lua_settop(L, 2);  /* create a 2nd argument if there isn't one */
      if (lua_seqnext(L, 1))
        return 2;
      else {
        lua_pushnil(L);
        return 1;
      }
    }
    case LUA_TSTRING: {
      lua_settop(L, 2);  /* create a 2nd argument if there isn't one */
      if (lua_strnext(L, 1))
        return 2;
      else {
        lua_pushnil(L);
        return 1;
      }
    }
    default: {
      luaL_error(L, "Error in " LUA_QS ": table, set, sequence, or string expected, got %s.", "next",
        lua_typename(L, lua_type(L, 1)));
    }
  }
  return 1;
}


int load_aux (lua_State *L, int status) {  /* Agena 1.6.1: global declaration */
  if (status == 0)  /* OK? */
    return 1;
  else {
    lua_pushnil(L);
    lua_insert(L, -2);  /* put before error message */
    return 2;  /* return nil plus error message */
  }
}


static int luaB_loadstring (lua_State *L) {
  size_t l;
  const char *s = luaL_checklstring(L, 1, &l);
  const char *chunkname = luaL_optstring(L, 2, s);
  return load_aux(L, luaL_loadbuffer(L, s, l, chunkname));
}


static int luaB_loadfile (lua_State *L) {
  const char *fname = luaL_optstring(L, 1, NULL);
  return load_aux(L, luaL_loadfile(L, fname));
}


/*
** Reader for generic `load' function: `lua_load' uses the
** stack for internal stuff, so the reader cannot change the
** stack top. Instead, it keeps its resulting string in a
** reserved slot inside the stack.
*/
static const char *generic_reader (lua_State *L, void *ud, size_t *size) {
  (void)ud;  /* to avoid warnings */
  luaL_checkstack(L, 2, "too many nested functions");
  lua_pushvalue(L, 1);  /* get function */
  lua_call(L, 0, 1);  /* call it */
  if (lua_isnil(L, -1)) {
    *size = 0;
    return NULL;
  }
  else if (agn_isstring(L, -1)) {
    lua_replace(L, 3);  /* save string in a reserved stack slot */
    return lua_tolstring(L, 3, size);
  }
  else luaL_error(L, "reader function must return a string.");
  return NULL;  /* to avoid warnings */
}


static int luaB_load (lua_State *L) {
  int status;
  const char *cname = luaL_optstring(L, 2, "=(load)");
  luaL_checktype(L, 1, LUA_TFUNCTION);
  lua_settop(L, 3);  /* function, eventual name, plus one reserved slot */
  status = lua_load(L, generic_reader, NULL, cname);
  return load_aux(L, status);
}


static int luaB_run (lua_State *L) {  /* rewritten 1.6.0 */
  int n, r;
  const char *fname = luaL_optstring(L, 1, NULL);
  char *fn;
  n = lua_gettop(L);
  if (access(fname, 00|04) == -1)
    fn = concat(fname, ".agn", NULL);
  else
    fn = concat(fname, NULL);
  if (fn == NULL) luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "run");
  r = luaL_loadfile(L, fn);
  xfree(fn);
  if (r != 0) lua_error(L);
  lua_call(L, 0, LUA_MULTRET);
  return lua_gettop(L) - n;
}


static int luaB_assume (lua_State *L) {
  luaL_checkany(L, 1);
  if (!lua_toboolean(L, 1))
    return luaL_error(L, "Error in `assume`: %s.", luaL_optstring(L, 2, "assumption failed"));  /* Agena 1.5.0 */
  return lua_gettop(L);
}


static int luaB_unpack (lua_State *L) {
  int i, e, n, o;
  o = lua_type(L, 1);
  luaL_typecheck(L, o == LUA_TTABLE || o == LUA_TSEQ, 1, "table or sequence expected", o);
  i = agnL_optinteger(L, 2, 1);
  e = luaL_opt(L, luaL_checkint, 3, (o == LUA_TTABLE) ? luaL_getn(L, 1) : agn_seqsize(L, 1));
    /* 0.14.0, do not use agn_nops for it also counts hash values */
  if (i > e) return 0;  /* empty range; Lua 5.1.3 patch 4 */
  n = e - i + 1;  /* number of elements */
  if (n <= 0 || !lua_checkstack(L, n))  /* n <= 0 means arith. overflow */
    return luaL_error(L, "Error in " LUA_QS ": too many results to unpack.", "unpack");
  switch (o) {
    case LUA_TTABLE:
      lua_rawgeti(L, 1, i);  /* push arg[i] (avoiding overflow problems) */
      while (i++ < e)  /* push arg[i + 1...e] */
        lua_rawgeti(L, 1, i);
      break;
    case LUA_TSEQ:
      lua_seqgeti(L, 1, i);  /* push arg[i] (avoiding overflow problems) */
      while (i++ < e)  /* push arg[i + 1...e] */
        lua_seqgeti(L, 1, i);
      break;
    default:
      lua_assert(0); break;  /* should not happen */
  }
  return n;
}


static int luaB_protect (lua_State *L) {  /* Agena 1.0.3 */
  int status;
  luaL_checkany(L, 1);
  /* delete value of lasterror */
  lua_pushnil(L);
  lua_setglobal(L, "lasterror");
  status = lua_pcall(L, lua_gettop(L) - 1, LUA_MULTRET, 0);
  if (status == 0) {  /* successful ? */
    return lua_gettop(L);  /* return all results */
  } else {
    lua_pushvalue(L, -1); /* Agena 1.4.3/1.5.0, duplicate error message */
    lua_setglobal(L, "lasterror");  /* set error message to lasterror, and pop it */
    return 1;  /* return error message */
  }
}


static int luaB_xpcall (lua_State *L) {
  int status;
  luaL_checkany(L, 2);
  lua_settop(L, 2);
  lua_insert(L, 1);  /* put error function under function to be called */
  status = lua_pcall(L, 0, LUA_MULTRET, 1);
  lua_pushboolean(L, (status == 0));
  lua_replace(L, 1);
  return lua_gettop(L);  /* return status + all results */
}


static int luaB_tostring (lua_State *L) {
  luaL_checkany(L, 1);
  if (luaL_callmeta(L, 1, "__tostring"))  /* is there a metafield? */
    return 1;  /* use its value */
  switch (lua_type(L, 1)) {
    case LUA_TNUMBER:
      lua_pushstring(L, lua_tostring(L, 1));
      break;
    case LUA_TSTRING:
      lua_pushvalue(L, 1);
      break;
    case LUA_TBOOLEAN:
      if (lua_toboolean(L, 1) == LUA_TFAIL)
        lua_pushstring(L, "fail");
      else
        lua_pushstring(L, (lua_toboolean(L, 1) ? "true" : "false"));
      break;
    case LUA_TNIL:
      lua_pushliteral(L, "null");  /* changed */
      break;
    case LUA_TCOMPLEX: {  /* 0.14.0 */
#ifndef PROPCMPLX
      agn_Complex z = agn_tocomplex(L, 1);
      lua_pushnumber(L, creal(z));
      lua_pushstring(L, lua_tostring(L, -1));
      lua_remove(L, -2);  /* delete creal(z) */
      lua_pushnumber(L, cimag(z));
#else
      lua_Number a = agn_complexreal(L, 1), b = agn_compleximag(L, 1);
      lua_pushnumber(L, a);
      lua_pushstring(L, lua_tostring(L, -1));
      lua_remove(L, -2);  /* delete a */
      lua_pushnumber(L, b);
#endif
      lua_pushstring(L, lua_tostring(L, -1));
      lua_remove(L, -2);  /* delete cimag(z) / a */
      return 2;
    }
    case LUA_TPAIR: {   /* 0.14.0 */
      lua_pushnumber(L, 1);  /* get left-hand side */
      agn_pairrawget(L, 1);  /* and put it onto the stack top, deleting 1 */
      lua_pushstring(L, lua_tostring(L, -1));
      lua_remove(L, -2);     /* remove left-hand side */
      lua_pushnumber(L, 2);  /* get right-hand side */
      agn_pairrawget(L, 1);  /* and put it onto the stack top, deleting 2 */
      lua_pushstring(L, lua_tostring(L, -1));
      lua_remove(L, -2);     /* remove right-hand side */
      return 2;
    }
    default:
      lua_pushfstring(L, "%s(%p)", luaL_typename(L, 1), lua_topointer(L, 1));
      break;
  }
  return 1;
}


static int luaB_newproxy (lua_State *L) {
  lua_settop(L, 1);
  lua_newuserdata(L, 0);  /* create proxy */
  if (lua_toboolean(L, 1) == 0)
    return 1;  /* no metatable */
  else if (lua_isboolean(L, 1)) {
    lua_newtable(L);  /* create a new metatable `m' ... */
    lua_pushvalue(L, -1);  /* ... and mark `m' as a valid metatable */
    lua_pushboolean(L, 1);
    lua_rawset(L, lua_upvalueindex(1));  /* weaktable[m] = true */
  }
  else {
    int validproxy = 0;  /* to check if weaktable[metatable(u)] == true */
    if (lua_getmetatable(L, 1)) {
      lua_rawget(L, lua_upvalueindex(1));
      validproxy = lua_toboolean(L, -1);
      agn_poptop(L);  /* remove value */
    }
    luaL_argcheck(L, validproxy, 1, "boolean or proxy expected");
    lua_getmetatable(L, 1);  /* metatable is valid; get it */
  }
  lua_setmetatable(L, 2);
  return 1;
}

/* ADDITIONS */


static int agn_time (lua_State *L) {
  lua_pushnumber(L, ((lua_Number)clock())/(lua_Number)CLOCKS_PER_SEC);
  return 1;
}

#define has_query(L, pvalue, pop) { \
  if (lua_type(L, -1) > LUA_TTHREAD) { \
    lua_pushvalue(L, pvalue); \
    agnB_has_aux(L); \
    if (agn_istrue(L, -1)) { \
      lua_pop(L, pop); \
      flag = 1; \
      break; \
    } \
  } \
  agn_poptop(L); }

static int agnB_has_aux (lua_State *L) {  /* Agena 1.6.13, 3 to 7 times faster than the former Agena implementation */
  int flag, isstruct;
  flag = isstruct = 0;
  if (lua_checkstack(L, 2) == 0)
    luaL_error(L, "Error in " LUA_QS ": stack cannot be extended.", "has");
  if (lua_equal(L, -1, -2)) {
    agn_poptoptwo(L);
    lua_pushtrue(L);
    return 1;
  }
  switch (lua_type(L, -2)) {
    case LUA_TTABLE: {
      lua_pushnil(L);
      while (lua_next(L, -3)) {
        if (lua_equal(L, -1, -3)) {
          agn_poptoptwo(L);  /* pop key and value */
          flag = 1;
          break;
        } else if (!agn_isnumber(L, -3)) {  /* x :- number and assigned(struct[x]) */
          lua_pushvalue(L, -3);  /* push second argument */
          lua_gettable(L, -5);   /* get entry */
          if (!lua_isnil(L, -1)) {
            lua_pop(L, 3);  /* remove entry, value, and key */
            flag = 1;
            break;
          } else {
            agn_poptop(L);  /* remove entry */
          }
        }
        if (!isstruct && lua_type(L, -1) > LUA_TTHREAD) isstruct = 1;  /* 1.6.14 */
        agn_poptop(L);  /* remove value */
      }
      break;
    }
    case LUA_TSET: {
      lua_pushvalue(L, -1);
      lua_srawget(L, -3);  /* pushes true or false on the stack */
      if (agn_istrue(L, -1)) {
        flag = 1;
      }
      agn_poptop(L);  /* pop true or false */
      if (!flag) isstruct = 1;  /* 1.6.14 */
      break;
    }
    case LUA_TSEQ: {
      int i;
      for (i=0; i < agn_seqsize(L, -2); i++) {
        lua_seqgeti(L, -2, i+1);
        if (lua_equal(L, -1, -2)) {
          agn_poptop(L);
          flag = 1;
          break;
        } else {
          if (!isstruct && lua_type(L, -1) > LUA_TTHREAD) isstruct = 1;  /* 1.6.14 */
          agn_poptop(L);
        }
      }
      break;
    }
    case LUA_TPAIR: {
      int i;
      for (i=1; i < 3; i++) {
        agn_pairgeti(L, -2, i);
        if (lua_equal(L, -1, -2)) flag = 1;
        agn_poptop(L);
        if (flag) break;
      }
      if (!flag) {
        agn_createseq(L, 2);
        for (i=1; i < 3; i++) {
          agn_pairgeti(L, -3, i);
          lua_seqseti(L, -2, i);
        }
        lua_remove(L, -3); lua_insert(L, -2);
        isstruct = 1;  /* 1.6.14 */
      }
      break;
    }
    case LUA_TFAIL: case LUA_TNIL: case LUA_TNONE: {  /* 1.6.14 */
      luaL_error(L, "Error in " LUA_QS ": type %s not accepted.", "has", lua_typename(L, lua_type(L, -2)));
    }
    default: {  /* = case LUA_TNUMBER: case LUA_TBOOLEAN: case LUA_TCOMPLEX: case LUA_TSTRING: case LUA_TFUNCTION: case LUA_TTHREAD:
                     case LUA_TUSERDATA: case LUA_TLIGHTUSERDATA */
      agn_poptoptwo(L);
      lua_pushfail(L);
      return 1;
    }
  }
  if (!flag && isstruct) {  /* 1.6.14 */
    switch (lua_type(L, -2)) {
      case LUA_TTABLE: {
        lua_pushnil(L);
        while (lua_next(L, -3)) {
          has_query(L, -3, 2);
        }
        break;
      }
      case LUA_TSET: {
        lua_pushnil(L);
        while (lua_usnext(L, -3)) {
          has_query(L, -3, 2);
        }
        break;
      }
      case LUA_TSEQ: {
        int i;
        for (i=0; i < agn_seqsize(L, -2); i++) {
          lua_seqgeti(L, -2, i+1);
          has_query(L, -2, 1);
        }
        break;
      }
      default: {
        luaL_error(L, "Error in " LUA_QS ": type %s not accepted.", "has", lua_typename(L, lua_type(L, -2)));
      }
    }
  }
  agn_poptoptwo(L);  /* remove first and second argument */
  lua_pushboolean(L, flag);
  return 1;
}


static int agnB_has (lua_State *L) {
  if (lua_gettop(L) != 2) {
    luaL_error(L, "Error in " LUA_QS ": requires two arguments.", "has");
  }
  lua_pushvalue(L, 1);
  lua_pushvalue(L, 2);
  agnB_has_aux(L);
  return 1;
}


#define recurse_query(L, pvalue, pop) { \
  if (lua_type(L, -1) > LUA_TTHREAD) { \
    lua_pushvalue(L, pvalue); \
    agnB_recurse_aux(L); \
    if (!lua_isboolean(L, -1)) { \
      luaL_error(L, "Error in " LUA_QS ": procedure must evaluate to a boolean, returned %s.", "recurse", lua_typename(L, lua_type(L, -1))); \
    } \
    if (agn_istrue(L, -1)) { \
      lua_pop(L, pop); \
      flag = 1; \
      break; \
    } \
  } \
  agn_poptop(L); }

static int agnB_recurse_aux (lua_State *L) {  /* Agena 1.6.15 */
  int flag, isstruct;
  isstruct = 0;
  flag = 0;
  if (lua_checkstack(L, 2) == 0)
    luaL_error(L, "Error in " LUA_QS ": stack cannot be extended.", "recurse");
  switch (lua_type(L, -2)) {
    case LUA_TTABLE: {
      lua_pushnil(L);
      while (lua_next(L, -3)) {
        lua_pushvalue(L, -3);  /* push function */
        lua_pushvalue(L, -2);  /* duplicate table value */
        lua_call(L, 1, 1);
        if (!lua_isboolean(L, -1)) {
          luaL_error(L, "Error in " LUA_QS ": procedure must evaluate to a boolean, returned %s.", "recurse", lua_typename(L, lua_type(L, -1)));
        }
        if (agn_istrue(L, -1)) {
          lua_pop(L, 3);  /* pop true, and value, and key */
          flag = 1;
          break;
        }
        agn_poptop(L);  /* remove false */
        if (!agn_isnumber(L, -2)) {  /* check key, key :- number and assigned(struct[key]) */
          lua_pushvalue(L, -3);  /* push function */
          lua_pushvalue(L, -3);  /* push key */
          if (!isstruct && lua_type(L, -1) > LUA_TTHREAD) isstruct = 1;
          lua_call(L, 1, 1);
          if (!lua_isboolean(L, -1)) {
            luaL_error(L, "Error in " LUA_QS ": procedure must evaluate to a boolean, returned %s.", "recurse", lua_typename(L, lua_type(L, -1)));
          }
          if (agn_istrue(L, -1)) {
            lua_pop(L, 3);  /* pop true, and value, and key */
            flag = 1;
            break;
          } else {
            agn_poptop(L);  /* pop false */
          }
        }
        if (!isstruct && lua_type(L, -1) > LUA_TTHREAD) isstruct = 1;
        agn_poptop(L);  /* remove value */
      }
      break;
    }
    case LUA_TSET: {
      lua_pushnil(L);
      while (lua_usnext(L, -3)) {
        lua_pushvalue(L, -3);  /* push function */
        lua_pushvalue(L, -2);  /* push value */
        lua_call(L, 1, 1);
        if (!lua_isboolean(L, -1)) {
          luaL_error(L, "Error in " LUA_QS ": procedure must evaluate to a boolean, returned %s.", "recurse", lua_typename(L, lua_type(L, -1)));
        }
        if (agn_istrue(L, -1)) {
          lua_pop(L, 3);  /* pop true, and value, and key */
          flag = 1;
          break;
        }
        agn_poptop(L);  /* remove false */
        if (!isstruct && lua_type(L, -1) > LUA_TTHREAD) isstruct = 1;
        agn_poptop(L);  /* remove value */
      }
      break;
    }
    case LUA_TSEQ: {
      int i;
      for (i=0; i < agn_seqsize(L, -2); i++) {
        lua_pushvalue(L, -1);  /* push function */
        lua_seqgeti(L, -3, i+1);  /* push value */
        if (!isstruct && lua_type(L, -1) > LUA_TTHREAD) isstruct = 1;
        lua_call(L, 1, 1);
        if (!lua_isboolean(L, -1)) {
          luaL_error(L, "Error in " LUA_QS ": procedure must evaluate to a boolean, returned %s.", "recurse", lua_typename(L, lua_type(L, -1)));
        }
        if (agn_istrue(L, -1)) {
          agn_poptop(L);  /* pop true */
          flag = 1;
          break;
        }
        agn_poptop(L);  /* pop false */
      }
      break;
    }
    case LUA_TPAIR: {  /* Agena 1.8.4 */
      int i;
      for (i=0; i < 2; i++) {
        lua_pushvalue(L, -1);  /* push function */
        agn_pairgeti(L, -3, i+1);  /* push value */
        if (!isstruct && lua_type(L, -1) > LUA_TTHREAD) isstruct = 1;
        lua_call(L, 1, 1);
        if (!lua_isboolean(L, -1)) {
          luaL_error(L, "Error in " LUA_QS ": procedure must evaluate to a boolean, returned %s.", "recurse", lua_typename(L, lua_type(L, -1)));
        }
        if (agn_istrue(L, -1)) {
          agn_poptop(L);  /* pop true */
          flag = 1;
          break;
        }
        agn_poptop(L);  /* pop false */
      }
      break;
    }
    case LUA_TNONE: {  /* Agena 1.8.4 */
      luaL_error(L, "Error in " LUA_QS ": type %s not accepted.", "recurse", lua_typename(L, LUA_TNONE));
      break;
    }
    default: {  /* = case LUA_TNUMBER: case LUA_TNIL case LUA_TBOOLEAN: case LUA_TCOMPLEX: case LUA_TSTRING: case LUA_TFUNCTION: case LUA_TTHREAD:
                     case LUA_TUSERDATA: case LUA_TLIGHTUSERDATA */
      agn_poptoptwo(L);
      lua_pushfail(L);
      return 1;
    }
  }
  if (!flag && isstruct) {
    switch (lua_type(L, -2)) {
      case LUA_TTABLE: {
        lua_pushnil(L);
        while (lua_next(L, -3)) {
          recurse_query(L, -3, 2);
        }
        break;
      }
      case LUA_TSET: {
        lua_pushnil(L);
        while (lua_usnext(L, -3)) {
          recurse_query(L, -3, 2);
        }
        break;
      }
      case LUA_TSEQ: {
        int i;
        for (i=0; i < agn_seqsize(L, -2); i++) {
          lua_seqgeti(L, -2, i+1);
          recurse_query(L, -2, 1);
        }
        break;
      }
      case LUA_TPAIR: {  /* Agena 1.8.4 */
        int i;
        for (i=0; i < 2; i++) {  /* Agena 1.8.5 fix */
          agn_pairgeti(L, -2, i+1);
          recurse_query(L, -2, 1);
        }
        break;
      }
      default: {
        luaL_error(L, "Error in " LUA_QS ": type %s not accepted.", "recurse", lua_typename(L, lua_type(L, -2)));
      }
    }
  }
  agn_poptoptwo(L);  /* remove first and second argument */
  lua_pushboolean(L, flag);
  return 1;
}


static int agnB_recurse (lua_State *L) {
  if (lua_gettop(L) != 2) {
    luaL_error(L, "Error in " LUA_QS ": requires two arguments.", "recurse");
  }
  luaL_checktype(L, 2, LUA_TFUNCTION);
  lua_pushvalue(L, 1);
  lua_pushvalue(L, 2);
  agnB_recurse_aux(L);
  return 1;
}


/* December 10, 2006; revised December 24, 2007; extended June 21, 2008, patched December 23, 2008;
   changed June 21, 2009; fixed 0.25.4, August 01, 2009; extended November 22, 2009; extended December 24, 2009
   Since values popped may be gc'ed, the strings put on the stack are only popped when not needed
   any longer. Do not put an agn_poptop() statement right after the str = agn_tostring(L, n)
   assignment !!! */

static int agn_readlib (lua_State *L) {
  char *fn;
  const char *procname;
  int nargs, i, clibsuccess, success, globalsuccess, quitaterror;
  quitaterror = 0;
  nargs = lua_gettop(L);
  procname =  "readlib";
  luaL_checkstack(L, nargs, "too many arguments");
  if (lua_isboolean(L, nargs)) {
    nargs--;
    if (agn_istrue(L, nargs))  /* 0.29.0 extension, tuned 1.6.0 */
      quitaterror = 1;
    else
      procname = "import";  /* 2.0.0 */
  }
  if (nargs == 0)
    luaL_error(L, "Error in " LUA_QS ": requires at least one argument, a string.", procname);
  globalsuccess = 1;
  /* try library names */
  for (i=1; i <= nargs; i++) {
    const char *apath, *path, *libname;
    char *buffer;
    int result;
    libname = agn_checkstring(L, i);  /* 2.0.0 change */
    success = 0;
    /* prepend current working directory to library name */
    buffer = (char *)malloc(sizeof(char)*(PATH_MAX+1));  /* Agena 1.5.0 */
    if (buffer == NULL)
      luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", procname);
    if (getcwd(buffer, PATH_MAX) == buffer) {  /* 1.6.11, to avoid compiler warnings */
      lua_pushstring(L, buffer);
      lua_pushstring(L, LUA_PATHSEP);
    }
    else
      luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", procname);
    /* add mainlibname, 0.29.3 */
    lua_getfield(L, LUA_GLOBALSINDEX, "mainlibname");
    if (!agn_isstring(L, -1)) {
      luaL_error(L, "Error in " LUA_QS ": " LUA_QS
        " is not a string, must be a path to the main\nAgena folder.", procname, "mainlibname");
    }
    lua_pushstring(L, LUA_PATHSEP);
    /* add libname since path gets modified by pushnexttemplate, reload libname */
    lua_getfield(L, LUA_GLOBALSINDEX, "libname");
    if (!agn_isstring(L, -1)) {  /* 0.20.2 */
      luaL_error(L, "Error in " LUA_QS ": " LUA_QS
        " is not a string, must be a path to the main\nAgena folder.", procname, "libname");
    }
    lua_concat(L, 5);
    path = agn_tostring(L, -1);
    if (quitaterror)
      fprintf(stderr, "Search paths for " LUA_QS ": " LUA_QS "\n", procname, path);
    /* iterate libname */
    while ((path = luaL_pushnexttemplate(L, path)) != NULL) {  /* pushes a path on the stack */
      clibsuccess = 0;
      apath = agn_tostring(L, -1);  /* assign string pushed by pushnexttemplate */
      #if !defined(__OS2__) && !defined(LUA_ANSI)  /* 0.27.1 patch: prevent that the DOS, OS/2 and strict-ANSI
      versions of Agena try to initialise external C libraries */
      /* find and read CLIB (DLLs, SOs) if present */
      fn = concat(apath, "/", libname, AGN_CLIB, NULL);  /* Agena 1.5.0, 1.6.0 */
      if (fn == NULL)
        luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", procname);
      fastloader_C(L, (const char*)fn, libname, &clibsuccess, quitaterror);  /* if found the package table is at the top of the stack */
      if (clibsuccess == -1) {  /* 0.29.0 */
        fprintf(stderr, "Warning in " LUA_QS ", package file " LUA_QS " corrupt or other error.\n", procname, fn);
        if (!agn_getdebug(L)) {  /* 1.12.9 */
          fprintf(stderr, "Try `environ.kernel(debug=true)` and retry loading the package to get further\n"
            "hints.\n");
        }
        fflush(stderr);
      }
      xfree(fn);  /* Agena 1.6.0 Valgrind, 1.7.5 */
      #endif
      /* find and read agn text file library if present */
      fn = concat(apath, "/", libname, ".agn", NULL);  /* Agena 1.5.0 */
      if (fn == NULL)
        luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", procname);
      result = luaL_loadfile(L, (const char *)fn);  /* pushes a chunk or an error string */
      xfree(fn);  /* Agena 1.6.0 Valgrind */
      if (result != LUA_ERRFILE) {  /* file found, run it */
        if (result != 0) lua_error(L);  /* but there is a syntax error in it */
        if (lua_pcall(L, 0, 0, 0) != 0) {  /* the loaded file must be run to get procs assigned; the chunk is
                                              popped, and no results are pushed on the stack */
          fprintf(stderr, "%s.\n", lua_tostring(L, -1));  /* print error message, 0.25.4 */
          fflush(stderr);
          agn_poptop(L);  /* drop error message */
        } else {
          /* now clean up */
          success = 1;
        }
      }
      else
        agn_poptop(L);  /* drop error code */
      agn_poptop(L);   /* drop string pushed by pushnexttemplate */
      if (clibsuccess == 1) {
        success = 1;
        /* if a package consists of both a C DLL and an Agena text file, then they
         must reside in the very same folder. */
      }
      if (success) break;  /* quit while loop */
    }  /* end of libname while iteration loop */
    agn_poptop(L);  /* drop libname */
    if (!success) {
      fprintf(stderr, "Warning in " LUA_QS ": package " LUA_QS " not found.\n", procname, libname);
      fflush(stderr);
      globalsuccess = 0;
    } else {
      /* register package in package.loaded and package.readlibbed, 0.26.0 */
      lua_getfield(L, LUA_REGISTRYINDEX, "_LOADED");  /* table is at stack index -1 */
      lua_getglobal(L, libname);  /* try to get value of `libname' from global environment */
      if (lua_isnil(L, -1)) {
        agn_poptop(L);  /* pop null */
        lua_pushtrue(L);  /* and push true instead */
      }
      lua_setfield(L, -2, libname);  /* set value of `libname' to package.loaded.libname and pop this value */
      agn_poptop(L);  /* pop package.loaded.libname */
      /* add entry to package.readlibbed */
      agn_setreadlibbed(L, libname);
    }
    xfree(buffer);
  }  /* of for in library names */
  if (globalsuccess)
    lua_pushtrue(L);
  else
    lua_pushfail(L);
  return 1;
}


/* patched 25.12.2009, 0.29.3 */
static int agnB_map (lua_State *L) {
  int j, nargs, nops;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  switch (lua_type(L, 2)) {
    case LUA_TTABLE: {
      /* very rough guess whether table is an array or dictionary */
      size_t n = lua_objlen(L, 2);
      if (n == 0)  /* assume table is a dictionary */
        lua_createtable(L, 0, agn_size(L, 2));
      else
        lua_createtable(L, n, 0);  /* lua_objlen not always returns correct results ! */
      lua_pushnil(L);
      while (agn_fnext(L, 2, 1, 0)) {
        for (j=3; j <= nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        lua_rawset2(L, -3);       /* store result in a table; rawset2 only deletes the value from the stack, the key is kept */
      }
      /* set type of new table to the one of the original table */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2))
        lua_setmetatable(L, -2);
      break;
    }  /* end of case LUA_TTABLE */
    case LUA_TSET: {
      agn_createset(L, agn_ssize(L, 2));
      lua_pushnil(L);
      while (lua_usnext(L, 2)) {
        lua_pushvalue(L, 1);      /* push function; FIXME: can be optimized */
        lua_pushvalue(L, -2);     /* push key (value) */
        for (j=3; j<=nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        lua_srawset(L, -4);       /* store result to new set */
        agn_poptop(L);
      }
      /* set type of new table to the one of the original table */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2))
        lua_setmetatable(L, -2);
      break;
    } /* end of case LUA_TSET */
    case LUA_TSEQ: {  /* 0.11.0 */
      int i;  /* index */
      nops = agn_seqsize(L, 2);
      agn_createseq(L, nops);
      /* set its type to the one of the original sequence */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2))
        lua_setmetatable(L, -2);
      /* now traverse array */
      for (i=0; i < nops; i++) {
        lua_pushvalue(L, 1);      /* push function */
        lua_seqgeti(L, 2, i+1);
        for (j=3; j<=nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        lua_seqinsert(L, -2);     /* store result to new sequence */
      }
      break;
    } /* end of case LUA_TSEQ */
    case LUA_TSTRING: {  /* 1.5.0 */
      size_t l;
      const char *str = agn_checklstring(L, 2, &l);
      agn_createseq(L, l);
      /* now traverse string */
      while (*str) {
        lua_pushvalue(L, 1);      /* push function */
        lua_pushlstring(L, str, 1);
        for (j=3; j<=nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        lua_seqinsert(L, -2);
        str++;
      }
      break;
    } /* end of case LUA_TSTRING */
    case LUA_TPAIR: {  /* 1.8.8 */
      int i;  /* index */
      /* traverse pair */
      for (i=1; i < 3; i++) {
        lua_pushvalue(L, 1);  /* push function */
        agn_pairgeti(L, 2, i);
        for (j=3; j<=nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
      }
      agn_createpair(L, -2, -1);
      lua_remove(L, -2); lua_remove(L, -2);  /* remove left and right value, agn_createpair does not pop these values */
      /* set its type to the one of the original sequence */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2))
        lua_setmetatable(L, -2);
      break;
    } /* end of case LUA_TPAIR */
    default:
      luaL_error(L, "Error in " LUA_QS ": table, set, sequence, pair, or string expected for argument #2,\n   got %s.", "map",
        lua_typename(L, lua_type(L, 2)));
  } /* end of switch */
  return 1;
}


static int agnB_subs (lua_State *L) {
  int i, n, nargs, nops;
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  for (i=1; i < nargs; i++)
    luaL_checktype(L, i, LUA_TPAIR);
  switch (lua_type(L, nargs)) {
    case LUA_TTABLE: {
      n = luaL_getn(L, nargs);
      if (n == 0)
        lua_createtable(L, 0, agn_size(L, nargs));
      else
        lua_createtable(L, n, 0);
      lua_pushnil(L);
      while (lua_next(L, nargs)) {
        for (i=1; i < nargs; i++) {
          agn_pairgeti(L, i, 1);
          if (lua_equal(L, -1, -2)) {
            agn_poptoptwo(L);  /* delete value and lhs */
            agn_pairgeti(L, i, 2);  /* push rhs */
          } else {
            agn_poptop(L);  /* delete lhs */
          }
        }
        lua_rawset2(L, -3); /* store result in a table */
      }
      /* set type of new table to the one of the original table */
      if (agn_getutype(L, nargs)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, nargs))
        lua_setmetatable(L, -2);
      break;
    }  /* end of case LUA_TTABLE */
    case LUA_TSET: {
      agn_createset(L, agn_ssize(L, nargs));
      lua_pushnil(L);
      while (lua_usnext(L, nargs)) {
        for (i=1; i < nargs; i++) {
          agn_pairgeti(L, i, 1);
          if (lua_equal(L, -1, -2)) {
            agn_poptoptwo(L);  /* delete value and lhs */
            agn_pairgeti(L, i, 2);  /* push rhs */
          } else {
            agn_poptop(L);  /* delete lhs */
          }
        }
        lua_srawset(L, -3); /* store value to set, leave index on stack */
      }
      /* set type of new table to the one of the original table */
      if (agn_getutype(L, nargs)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, nargs))
        lua_setmetatable(L, -2);
      break;
    } /* end of case LUA_TSET */
    case LUA_TSEQ: {
      int j;
      nops = agn_seqsize(L, nargs);
      agn_createseq(L, nops);
      /* set its type to the one of the original sequence */
      if (agn_getutype(L, nargs)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, nargs))
        lua_setmetatable(L, -2);
      /* now traverse array */
      for (i=0; i < nops; i++) {
        lua_seqgeti(L, nargs, i+1);
        for (j=1; j < nargs; j++) {
          agn_pairgeti(L, j, 1);
          if (lua_equal(L, -1, -2)) {
            agn_poptoptwo(L);  /* delete value and lhs */
            agn_pairgeti(L, j, 2);  /* push rhs */
          } else {
            agn_poptop(L);  /* delete lhs */
          }
        }
        lua_seqinsert(L, -2);  /* store result to new sequence */
      }
      break;
    } /* end of case LUA_TSEQ */
    default:
      luaL_error(L, "Error in " LUA_QS ": table, set, or sequence expected as last argument, got %s.", "subs",
        lua_typename(L, lua_type(L, nargs)));
  } /* end of switch */
  return 1;
}



/* 0.9.1; January 11, 2008; fixed June 30, 2008; fixed December 25, 2009 */
static int luaB_select (lua_State *L) {
  size_t j, nargs;
  int newarray;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  newarray = (nargs > 2) ? agn_paircheckbooloption(L, "select", nargs, "newarray") : -2;
  if (newarray != -2) nargs--;  /* `newarray` option found ? */
  switch(lua_type(L, 2)) {
    case LUA_TTABLE: {
      /* very rough guess whether table is an array or dictionary */
      size_t n, c;
      int asc;
      n = lua_objlen(L, 2);
      asc = newarray == 1;
      c = 1;
      if (n == 0)  /* assume table is a dictionary */
        agn_createtable(L, 0, agn_size(L, 2));
      else
        agn_createtable(L, n, 0);  /* lua_objlen not always returns correct results ! */
      lua_pushnil(L);
      while (agn_fnext(L, 2, 1, 1)) {
        for (j=3; j <= nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        if (agn_istrue(L, -1)) {
          agn_poptop(L);
          if (asc)
            lua_rawseti(L, -3, c++);
          else
            lua_rawset2(L, -3);  /* store value in a table, leave key on stack */
        } else
          agn_poptoptwo(L);  /* delete result and value, leave key on stack  */
      }
      /* set type of new table to the one of the original table */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2))
        lua_setmetatable(L, -2);
      break;
    }
    case LUA_TSET: {
      agn_createset(L, agn_ssize(L, 2));
      /* set its type to the one of the original array */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2))
        lua_setmetatable(L, -2);
      lua_pushnil(L);
      while (lua_usnext(L, 2)) {
        lua_pushvalue(L, 1);      /* push function; FIXME: can be optimized */
        lua_pushvalue(L, -2);     /* push key (value) */
        for (j=3; j<=nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        if (agn_istrue(L, -1)) {
          agn_poptop(L);
          lua_srawset(L, -3);    /* set key to new set */
        } else
          agn_poptoptwo(L);
      }
      break;
    }
    case LUA_TSEQ: {  /* 0.11.1 */
      int i, nops;
      nops = agn_seqsize(L, 2);
      agn_createseq(L, nops);
      /* set its type to the one of the original array */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2))
        lua_setmetatable(L, -2);
      /* now traverse sequence */
      for (i=0; i < nops; i++) {
        lua_pushvalue(L, 1);      /* push function */
        lua_seqgeti(L, 2, i+1);
        for (j=3; j <= nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        if (agn_istrue(L, -1)) {
          lua_seqgeti(L, 2, i+1);
          lua_seqinsert(L, -3);   /* store value to new sequence */
        }
        agn_poptop(L);
      }
      break;
    } /* end of case LUA_TSEQ */
    default:
      luaL_error(L, "Error in " LUA_QS ": table, set, or sequence expected for argument #2, got %s.", "select",
        lua_typename(L, lua_type(L, 2)));
  } /* end of switch */
  /* FIXME: Should resize the table to actual size */
  return 1;
}


/* Agena 1.8.9, November 09, 2012 */
static int luaB_selectremove (lua_State *L) {
  size_t j, nargs;
  int br, newarray;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  newarray = (nargs > 2) ? agn_paircheckbooloption(L, "selectremove", nargs, "newarray") : -2;
  if (newarray != -2) nargs--;  /* `newarray` option found ? */
  switch(lua_type(L, 2)) {
    case LUA_TTABLE: {
      /* very rough guess whether table is an array or dictionary */
      size_t n, tsize, c, d;
      int asc;
      asc = newarray == 1;
      c = d = 1;  /* c: index counter for first resulting table, d: same for second resulting table */
      n = lua_objlen(L, 2);
      tsize = agn_size(L, 2);
      if (n == 0) {  /* assume table is a dictionary */
        agn_createtable(L, 0, tsize); agn_createtable(L, 0, tsize);
      } else {
        agn_createtable(L, n, 0); agn_createtable(L, n, 0);  /* lua_objlen not always returns correct results ! */
      }
      lua_pushnil(L);
      while (agn_fnext(L, 2, 1, 1)) {
        for (j=3; j <= nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        br = agn_istrue(L, -1);
        agn_poptop(L);
        if (asc)
          lua_rawseti(L, -3-br, (br) ? c++ : d++);
        else
          lua_rawset2(L, -3-br);  /* store value in a table, leave key on stack */
      }
      /* set type of new table to the one of the original table */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_setutype(L, -3, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2)) {
        lua_pushvalue(L, -1);
        lua_setmetatable(L, -3);
        lua_setmetatable(L, -2);
      }
      break;
    }
    case LUA_TSET: {
      size_t ssize = agn_ssize(L, 2);
      agn_createset(L, ssize);
      agn_createset(L, ssize);
      /* set its type to the one of the original array */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_setutype(L, -3, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2)) {
        lua_pushvalue(L, -1);
        lua_setmetatable(L, -3);
        lua_setmetatable(L, -2);
      }
      lua_pushnil(L);
      while (lua_usnext(L, 2)) {
        lua_pushvalue(L, 1);      /* push function; FIXME: can be optimized */
        lua_pushvalue(L, -2);     /* push key (value) */
        for (j=3; j<=nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        br = agn_istrue(L, -1);
        agn_poptop(L);
        lua_srawset(L, -3-br);    /* set key to new set */
      }
      break;
    }
    case LUA_TSEQ: {  /* 0.11.1 */
      int i, nops;
      nops = agn_seqsize(L, 2);
      agn_createseq(L, nops);
      agn_createseq(L, nops);
      /* set its type to the one of the original array */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_setutype(L, -3, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2)) {
        lua_pushvalue(L, -1);
        lua_setmetatable(L, -3);
        lua_setmetatable(L, -2);
      }
      /* now traverse sequence */
      for (i=0; i < nops; i++) {
        lua_pushvalue(L, 1);      /* push function */
        lua_seqgeti(L, 2, i+1);
        for (j=3; j <= nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        br = agn_istrue(L, -1);
        lua_seqgeti(L, 2, i+1);
        lua_seqinsert(L, -3-br);   /* store value to new sequence */
        agn_poptop(L);
      }
      break;
    } /* end of case LUA_TSEQ */
    default:
      luaL_error(L, "Error in " LUA_QS ": table, set, or sequence expected for argument #2, got %s.", "selectremove",
        lua_typename(L, lua_type(L, 2)));
  } /* end of switch */
  /* FIXME: Should resize the table to actual size */
  return 2;
}


/* 0.30.3, 18.01.2010 */
static int agnB_countitems (lua_State *L) {
  int i, j, nargs, nops;
  size_t c = 0;
  luaL_checkany(L, 1);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (lua_type(L, 1) == LUA_TFUNCTION) {
    switch (lua_type(L, 2)) {
      case LUA_TTABLE: {
        lua_pushnil(L);
        while (agn_fnext(L, 2, 1, 0)) {  /* push the table key, the function, and the table value */
          for (j=3; j <= nargs; j++)
            lua_pushvalue(L, j);
          lua_call(L, nargs-1, 1);       /* call function with nargs-1 argument(s) and one result */
          if (agn_istrue(L, -1)) c++;
          agn_poptop(L);                 /* pop result */
        }
        break;
      }  /* end of case LUA_TTABLE */
      case LUA_TSET: {
        lua_pushnil(L);
        while (lua_usnext(L, 2)) {  /* push item twice */
          lua_pushvalue(L, 1);      /* push function; FIXME: can be optimized */
          lua_pushvalue(L, -2);     /* push key (value) */
          for (j=3; j<=nargs; j++)
            lua_pushvalue(L, j);
          lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
          if (agn_istrue(L, -1)) c++;  /* check result */
          agn_poptoptwo(L);         /* remove result and one item */
        }
        break;
      } /* end of case LUA_TSET */
      case LUA_TSEQ: {
        nops = agn_seqsize(L, 2);
        /* now traverse array */
        for (i=0; i < nops; i++) {
          lua_pushvalue(L, 1);      /* push function; FIXME: can be optimized */
          lua_seqgeti(L, 2, i+1);   /* push item */
          for (j=3; j<=nargs; j++)
            lua_pushvalue(L, j);
          lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
          if (agn_istrue(L, -1)) c++;
          agn_poptop(L);            /* remove result */
        }
        break;
      } /* end of case LUA_TSEQ */
      default:
        luaL_error(L, "Error in " LUA_QS ": table, set, or sequence expected as second argument, got %s.", "countitems",
          lua_typename(L, lua_type(L, 2)));
    } /* end of switch */
  } else {
    switch (lua_type(L, 2)) {
      case LUA_TTABLE: {
        lua_pushnil(L);
        while (lua_next(L, 2)) {  /* push the table key and the table value */
          if (lua_equal(L, -1, 1)) c++;
          agn_poptop(L);          /* pop value */
        }
        break;
      }  /* end of case LUA_TTABLE */
      case LUA_TSET: {
        lua_pushnil(L);
        while (lua_usnext(L, 2)) {  /* push item twice */
          if (lua_equal(L, -1, 1)) c++;
          agn_poptop(L);           /* remove one item */
        }
        break;
      } /* end of case LUA_TSET */
      case LUA_TSEQ: {
        nops = agn_seqsize(L, 2);
        for (i=0; i < nops; i++) {
          lua_seqgeti(L, 2, i+1);   /* push item */
          if (lua_equal(L, -1, 1)) c++;
          agn_poptop(L);            /* remove result */
        }
        break;
      } /* end of case LUA_TSEQ */
      default:
        luaL_error(L, "Error in " LUA_QS ": table, set, or sequence expected for argument #2, got %s.", "countitems",
          lua_typename(L, lua_type(L, 2)));
    } /* end of switch */
  }
  lua_pushinteger(L, c);
  return 1;
}


/* 0.10.0; March 30, 2008; extended with sequences on June 30, 2008; extended 0.28.0; fixed December 25, 2009 */
static int tbl_remove (lua_State *L) {
  size_t j, nargs;
  int newarray;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  newarray = (nargs > 2) ? agn_paircheckbooloption(L, "remove", nargs, "newarray") : -2;
  if (newarray != -2) nargs--;  /* `newarray` option found ? */
  switch (lua_type(L, 2)) {
    case LUA_TTABLE: {
      /* very rough guess whether table is an array or dictionary */
      size_t n, c;
      int asc;
      n = lua_objlen(L, 2);
      asc = newarray == 1;
      c = 1;
      if (n == 0)  /* assume table is a dictionary */
        agn_createtable(L, 0, agn_size(L, 2));
      else
        agn_createtable(L, n, 0);  /* lua_objlen not always returns correct results ! */
      lua_pushnil(L);
      while (agn_fnext(L, 2, 1, 1)) {
        for (j=3; j <= nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        if (agn_isfalse(L, -1)) {
          agn_poptop(L);
          if (asc)
            lua_rawseti(L, -3, c++);
          else
            lua_rawset2(L, -3);  /* store value in a table, leave key on stack */
        }
        else
          agn_poptoptwo(L);  /* delete result and value, leave key on stack  */
      }
      /* set its type to the one of the original array */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2))
        lua_setmetatable(L, -2);
      break;
    } /* end of case LUA_TTABLE */
    case LUA_TSET: {
      agn_createset(L, agn_ssize(L, 2));
      lua_pushnil(L);
      while (lua_usnext(L, 2)) {
        lua_pushvalue(L, 1);      /* push function */
        lua_pushvalue(L, -2);     /* push key (value) */
        for (j=3; j<=nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        if (agn_isfalse(L, -1)) {
          agn_poptop(L);
          lua_srawset(L, -3);     /* set key to new set */
        } else
          agn_poptoptwo(L);
      }
      /* set its type to the one of the original array */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2))
        lua_setmetatable(L, -2);
      break;
    } /* end of case LUA_TSET */
    case LUA_TSEQ: {  /* 0.12.0 */
      int i, nops;
      nops = agn_seqsize(L, 2);
      agn_createseq(L, nops);
      /* set its type to the one of the original array */
      if (agn_getutype(L, 2)) {
        agn_setutype(L, -2, -1);
        agn_poptop(L);  /* pop type (string) */
      }
      /* copy metatable */
      if (lua_getmetatable(L, 2))
        lua_setmetatable(L, -2);
      /* now traverse sequence */
      for (i=0; i < nops; i++) {
        lua_pushvalue(L, 1);      /* push function */
        lua_seqgeti(L, 2, i+1);
        for (j=3; j<=nargs; j++)
          lua_pushvalue(L, j);
        lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
        if (agn_isfalse(L, -1)) {
          lua_seqgeti(L, 2, i+1);
          lua_seqinsert(L, -3);   /* store value to new sequence */
        }
        agn_poptop(L);            /* delete result */
      }
      break;
    } /* end of case LUA_TSEQ */
    default:
      luaL_error(L, "Error in " LUA_QS ": table, set, or sequence expected for argument #2, got %s.", "remove",
        lua_typename(L, lua_type(L, 2)));
  }
  /* FIXME: Should resize the table to actual size */
  return 1;
}


/* convert a string to a table, needed by phonetiqs.config;
   extended to general function converting structures to tables 0.12.2, November 10, 2008 */

static int agnB_totable (lua_State *L) {
  size_t i, l;
  switch (lua_type(L, 1)) {
    case LUA_TSTRING: {
      const char *str = lua_tolstring(L, 1, &l);
      lua_createtable(L, l, 0);
      for (i=1; i<l+1; i++) {
        lua_rawsetilstring(L, -1, i, str++, 1);
      }
      break;
    }
    case LUA_TSET: {
      lua_createtable(L, agn_ssize(L, 1), 0);
      i = 0;
      lua_pushnil(L);
      while (lua_usnext(L, 1)) {
        i++;
        lua_rawseti(L, -3, i);
      }
      break;
    }
    case LUA_TSEQ: {
      size_t nops;
      nops = agn_seqsize(L, 1)+1;
      lua_createtable(L, nops-1, 0);
      for (i=1; i < nops; i++) {
        lua_seqgeti(L, 1, i);
        lua_rawseti(L, -2, i);
      }
      break;
    }
    default: luaL_error(L, "Error in " LUA_QS ": set, sequence, or string expected, got %s.", "totable",
      lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


static int agnB_toseq (lua_State *L) {
  size_t i, l;
  switch (lua_type(L, 1)) {
    case LUA_TSTRING: {
      const char *str = lua_tolstring(L, 1, &l);
      agn_createseq(L, l);
      for (i=1; i<l+1; i++) {
        lua_pushlstring(L, str++, 1);
        lua_seqseti(L, -2, i);
      }
      break;
    }
    case LUA_TTABLE: {
      i = 0;
      agn_createseq(L, agn_size(L, 1));
      lua_pushnil(L);
      while (lua_next(L, 1) != 0) {
        lua_seqseti(L, -3, ++i);
      }
      break;
    }
    case LUA_TSET: {
      i = 0;
      agn_createseq(L, agn_ssize(L, 1));
      lua_pushnil(L);
      while (lua_usnext(L, 1)) {
        lua_seqseti(L, -3, ++i);
      }
      break;
    }
    default: luaL_error(L, "Error in " LUA_QS ": table, set, or string expected, got %s.", "toseq",
      lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


static int agnB_toset (lua_State *L) {  /* 0.27.2 */
  size_t i, l;
  switch (lua_type(L, 1)) {
    case LUA_TSTRING: {
      const char *str = lua_tolstring(L, 1, &l);
      agn_createset(L, l);
      for (i=1; i<l+1; i++) {
        lua_sinsertlstring(L, -1, str++, 1);
      }
      break;
    }
    case LUA_TTABLE: {
      agn_createset(L, agn_size(L, 1));
      lua_pushnil(L);
      while (lua_next(L, 1) != 0) {
        lua_srawset(L, -3);
      }
      break;
    }
    case LUA_TSEQ: {
      size_t nops;
      nops = agn_seqsize(L, 1)+1;
      agn_createset(L, nops-1);
      for (i=1; i < nops; i++) {
        lua_seqgeti(L, 1, i);
        lua_srawset(L, -2);
      }
      break;
    }
    default: luaL_error(L, "Error in " LUA_QS ": table, sequence, or string expected, got %s.", "toset",
      lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


static int luaB_ops (lua_State *L) {
  int n, type;
  n = lua_gettop(L);
  type = lua_type(L, 1);
  if (type == LUA_TSTRING && *lua_tostring(L, 1) == '#') {
    lua_pushinteger(L, n - 1);
    return 1;
  }
  else if (type == LUA_TNUMBER) {
    int i = agnL_checkint(L, 1);
    if (i < 0) i = n + i;
    else if (i > n) i = n;
    luaL_argcheck(L, 1 <= i, 1, "index out of range");
    return n - i;
  } else if (type == LUA_TSEQ) {  /* Agena 1.2 */
    int c, i, j, nops;
    nops = agn_seqsize(L, 1);
    c = 0;
    for (j=0; j < nops; j++) {
      i = lua_seqgetinumber(L, 1, j+1);  /* get index position */
      if (i == HUGE_VAL) continue;  /* conversion failed ?  -> do nothing and ignore */
      if (i < 0) i = n + i;
      else if (i > n) i = n;
      if (i > 0 && i < n) { /* ignore invalid indices */
        /* check stack, there will be crashes in Solaris otherwise. */
        luaL_checkstack(L, ++c, "too many operands");
        lua_pushvalue(L, i+1);  /* push value on the stack */
      }
    }
    return c;
  } else
    luaL_error(L, "Error in " LUA_QS ": number, string, or sequence expected, got %s.", "ops",
      lua_typename(L, lua_type(L, 1)));
  return 0;  /* to prevent compiler warnings */
}


static int luaB_settype (lua_State *L) {  /* added 0.11.0, extended 0.12.0, 0.12.2, 0.14.0  */
  int i, s, t, nargs;
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs < 2)
    luaL_error(L, "Error in " LUA_QS ": at least two arguments expected, got %d.", "settype", nargs);
  t = lua_type(L, nargs);
  luaL_typecheck(L, t == LUA_TSTRING || t == LUA_TNIL, nargs, "string or null expected", t);
  for (i=1; i<nargs; i++) {
    s = lua_type(L, i);
    luaL_typecheck(L, s == LUA_TSEQ || s == LUA_TPAIR || s == LUA_TTABLE || s == LUA_TSET || s == LUA_TFUNCTION,
      i, "procedure, sequence, set, pair, or table expected", s);
    agn_setutype(L, i, nargs);
  }
  lua_pushnil(L);
  return 1;
}


static int luaB_gettype (lua_State *L) {  /* added 0.11.0, changed 0.12.2; extended 0.14.0 */
  int s = lua_type(L, 1);
  if (!(s == LUA_TSEQ || s == LUA_TPAIR || s == LUA_TTABLE || s == LUA_TFUNCTION || s == LUA_TSET) || !agn_getutype(L, 1))
    lua_pushnil(L);
  return 1;
}


/*
** {======================================================
** Quicksort
** (based on `Algorithms in MODULA-3', Robert Sedgewick;
**  Addison-Wesley, 1993.)
*/

#define luaBset2(L, i, j) { \
  lua_rawseti(L, 1, (i)); \
  lua_rawseti(L, 1, (j)); \
}


#define luaBsetseq2(L, i, j) { \
  lua_seqseti(L, 1, (i)); \
  lua_seqseti(L, 1, (j)); \
}


#define luaBset3(L, idx, i, j) { \
  lua_rawseti(L, (idx), (i)); \
  lua_rawseti(L, (idx+1), (j)); \
}


#define luaBsetseq3(L, idx, i, j) { \
  lua_seqseti(L, (idx), (i)); \
  lua_seqseti(L, (idx+1), (j)); \
}


int luaBsort_comp (lua_State *L, int a, int b) {
  if (!lua_isnil(L, 2)) {  /* function? */
    int res;
    lua_pushvalue(L, 2);
    lua_pushvalue(L, a-1);  /* -1 to compensate function */
    lua_pushvalue(L, b-2);  /* -2 to compensate function and `a' */
    lua_call(L, 2, 1);
    res = lua_toboolean(L, -1);
    agn_poptop(L);
    return res;
  }
  else  /* a < b? */
    return lua_lessthan(L, a, b);
}


static void auxsorttable (lua_State *L, int l, int u) {
  while (l < u) {  /* for tail recursion */
    int i, j;
    /* sort elements a[l], a[(l+u)/2] and a[u] */
    lua_rawgeti(L, 1, l);
    lua_rawgeti(L, 1, u);
    if (luaBsort_comp(L, -1, -2)) { /* a[u] < a[l]? */
      luaBset2(L, l, u);  /* swap a[l] - a[u] */
    } else
      agn_poptoptwo(L);
    if (u-l == 1) break;  /* only 2 elements */
    i = (l+u)/2;
    lua_rawgeti(L, 1, i);
    lua_rawgeti(L, 1, l);
    if (luaBsort_comp(L, -2, -1)) { /* a[i]<a[l]? */
      luaBset2(L, i, l);
    } else {
      agn_poptop(L);  /* remove a[l] */
      lua_rawgeti(L, 1, u);
      if (luaBsort_comp(L, -1, -2)) { /* a[u]<a[i]? */
        luaBset2(L, i, u);
      } else
        agn_poptoptwo(L);
    }
    if (u-l == 2) break;  /* only 3 elements */
    lua_rawgeti(L, 1, i);  /* Pivot */
    lua_pushvalue(L, -1);
    lua_rawgeti(L, 1, u-1);
    luaBset2(L, i, u-1);
    /* a[l] <= P == a[u-1] <= a[u], only need to sort from l+1 to u-2 */
    i = l; j = u-1;
    for (;;) {  /* invariant: a[l..i] <= P <= a[j..u] */
      /* repeat ++i until a[i] >= P */
      while (lua_rawgeti(L, 1, ++i), luaBsort_comp(L, -1, -2)) {
        if (i > u) luaL_error(L, "Error in " LUA_QS ": invalid order function for sorting.", "sort");
        agn_poptop(L);  /* remove a[i] */
      }
      /* repeat --j until a[j] <= P */
      while (lua_rawgeti(L, 1, --j), luaBsort_comp(L, -3, -1)) {
        if (j < l) luaL_error(L, "Error in " LUA_QS ": invalid order function for sorting.", "sort");
        agn_poptop(L);  /* remove a[j] */
      }
      if (j < i) {
        lua_pop(L, 3);  /* pop pivot, a[i], a[j] */
        break;
      }
      luaBset2(L, i, j);
    }
    lua_rawgeti(L, 1, u-1);
    lua_rawgeti(L, 1, i);
    luaBset2(L, u-1, i);  /* swap pivot (a[u-1]) with a[i] */
    /* a[l..i-1] <= a[i] == P <= a[i+1..u] */
    /* adjust so that smaller half is in [j..i] and larger one in [l..u] */
    if (i-l < u-i) {
      j = l; i = i-1; l = i+2;
    }
    else {
      j = i+1; i = u; u = j-2;
    }
    auxsorttable(L, j, i);  /* call recursively the smaller one */
  }  /* repeat the routine for the larger one */
}


static void auxsortseq (lua_State *L, int l, int u) {
  while (l < u) {  /* for tail recursion */
    int i, j;
    /* sort elements a[l], a[(l+u)/2] and a[u] */
    lua_seqgeti(L, 1, l);
    lua_seqgeti(L, 1, u);
    if (luaBsort_comp(L, -1, -2)) { /* a[u] < a[l]? */
      luaBsetseq2(L, l, u);  /* swap a[l] - a[u] */
    } else
      agn_poptoptwo(L);
    if (u-l == 1) break;  /* only 2 elements */
    i = (l+u)/2;
    lua_seqgeti(L, 1, i);
    lua_seqgeti(L, 1, l);
    if (luaBsort_comp(L, -2, -1)) { /* a[i]<a[l]? */
      luaBsetseq2(L, i, l);
    } else {
      agn_poptop(L);  /* remove a[l] */
      lua_seqgeti(L, 1, u);
      if (luaBsort_comp(L, -1, -2)) { /* a[u]<a[i]? */
        luaBsetseq2(L, i, u);
      } else
        agn_poptoptwo(L);
    }
    if (u-l == 2) break;  /* only 3 elements */
    lua_seqgeti(L, 1, i);  /* Pivot */
    lua_pushvalue(L, -1);
    lua_seqgeti(L, 1, u-1);
    luaBsetseq2(L, i, u-1);
    /* a[l] <= P == a[u-1] <= a[u], only need to sort from l+1 to u-2 */
    i = l; j = u-1;
    for (;;) {  /* invariant: a[l..i] <= P <= a[j..u] */
      /* repeat ++i until a[i] >= P */
      while (lua_seqgeti(L, 1, ++i), luaBsort_comp(L, -1, -2)) {
        if (i > u) luaL_error(L, "Error in " LUA_QS ": invalid order function for sorting.", "sort");
        agn_poptop(L);  /* remove a[i] */
      }
      /* repeat --j until a[j] <= P */
      while (lua_seqgeti(L, 1, --j), luaBsort_comp(L, -3, -1)) {
        if (j < l) luaL_error(L, "Error in " LUA_QS ": invalid order function for sorting.", "sort");
        agn_poptop(L);  /* remove a[j] */
      }
      if (j<i) {
        lua_pop(L, 3);  /* pop pivot, a[i], a[j] */
        break;
      }
      luaBsetseq2(L, i, j);
    }
    lua_seqgeti(L, 1, u-1);
    lua_seqgeti(L, 1, i);
    luaBsetseq2(L, u-1, i);  /* swap pivot (a[u-1]) with a[i] */
    /* a[l..i-1] <= a[i] == P <= a[i+1..u] */
    /* adjust so that smaller half is in [j..i] and larger one in [l..u] */
    if (i-l < u-i) {
      j = l; i = i-1; l = i+2;
    }
    else {
      j = i+1; i = u; u = j-2;
    }
    auxsortseq(L, j, i);  /* call recursively the smaller one */
  }  /* repeat the routine for the larger one */
}


static int luaB_sort (lua_State *L) {
  int n = agn_nops(L, 1);
  luaL_checkstack(L, 40, "");  /* assume array is smaller than 2^40 */
  if (!lua_isnoneornil(L, 2))  /* is there a 2nd argument? */
    luaL_checktype(L, 2, LUA_TFUNCTION);
  lua_settop(L, 2);  /* make sure there is two arguments */
  switch (lua_type(L, 1)) {
    case LUA_TTABLE:
      auxsorttable(L, 1, n);
      break;
    case LUA_TSEQ:
      auxsortseq(L, 1, n);
      break;
    default:
      luaL_error(L, "Error in " LUA_QS ": table or sequence expected as first argument, got %s.", "sort",
        lua_typename(L, lua_type(L, 1)));
  }
  return 0;
}


static void auxsortedtable (lua_State *L, int l, int u) {
  while (l < u) {  /* for tail recursion */
    int i, j;
    /* sort elements a[l], a[(l+u)/2] and a[u] */
    lua_rawgeti(L, -1, l);
    lua_rawgeti(L, -2, u);
    if (luaBsort_comp(L, -1, -2)) { /* a[u] < a[l]? */
      luaBset3(L, -3, l, u);  /* swap a[l] - a[u] */
    } else
      agn_poptoptwo(L);
    if (u-l == 1) break;  /* only 2 elements */
    i = (l+u)/2;
    lua_rawgeti(L, -1, i);
    lua_rawgeti(L, -2, l);
    if (luaBsort_comp(L, -2, -1)) { /* a[i]<a[l]? */
      luaBset3(L, -3, i, l);
    } else {
      agn_poptop(L);  /* remove a[l] */
      lua_rawgeti(L, -2, u);
      if (luaBsort_comp(L, -1, -2)) { /* a[u]<a[i]? */
        luaBset3(L, -3, i, u);
      } else
        agn_poptoptwo(L);
    }
    if (u-l == 2) break;  /* only 3 elements */
    lua_rawgeti(L, -1, i);  /* Pivot */
    lua_pushvalue(L, -1);
    lua_rawgeti(L, -3, u-1);
    luaBset3(L, -4, i, u-1);
    /* a[l] <= P == a[u-1] <= a[u], only need to sort from l+1 to u-2 */
    i = l; j = u-1;
    for (;;) {  /* invariant: a[l..i] <= P <= a[j..u] */
      /* repeat ++i until a[i] >= P */
      while (lua_rawgeti(L, -2, ++i), luaBsort_comp(L, -1, -2)) {
        if (i > u) luaL_error(L, "Error in " LUA_QS ": invalid order function for sorting.", "sorted");
        agn_poptop(L);  /* remove a[i] */
      }
      /* repeat --j until a[j] <= P */
      while (lua_rawgeti(L, -3, --j), luaBsort_comp(L, -3, -1)) {
        if (j < l) luaL_error(L, "Error in " LUA_QS ": invalid order function for sorting.", "sorted");
        agn_poptop(L);  /* remove a[j] */
      }
      if (j < i) {
        lua_pop(L, 3);  /* pop pivot, a[i], a[j] */
        break;
      }
      luaBset3(L, -4, i, j);
    }
    lua_rawgeti(L, -1, u-1);
    lua_rawgeti(L, -2, i);
    luaBset3(L, -3, u-1, i);  /* swap pivot (a[u-1]) with a[i] */
    /* a[l..i-1] <= a[i] == P <= a[i+1..u] */
    /* adjust so that smaller half is in [j..i] and larger one in [l..u] */
    if (i-l < u-i) {
      j = l; i = i-1; l = i+2;
    }
    else {
      j = i+1; i = u; u = j-2;
    }
    auxsortedtable(L, j, i);  /* call recursively the smaller one */
  }  /* repeat the routine for the larger one */
}


static void auxsortedseq (lua_State *L, int l, int u) {
  while (l < u) {  /* for tail recursion */
    int i, j;
    /* sort elements a[l], a[(l+u)/2] and a[u] */
    lua_seqgeti(L, -1, l);
    lua_seqgeti(L, -2, u);
    if (luaBsort_comp(L, -1, -2)) { /* a[u] < a[l]? */
      luaBsetseq3(L, -3, l, u);  /* swap a[l] - a[u] */
    } else
      agn_poptoptwo(L);
    if (u-l == 1) break;  /* only 2 elements */
    i = (l+u)/2;
    lua_seqgeti(L, -1, i);
    lua_seqgeti(L, -2, l);
    if (luaBsort_comp(L, -2, -1)) { /* a[i]<a[l]? */
      luaBsetseq3(L, -3, i, l);
    } else {
      agn_poptop(L);  /* remove a[l] */
      lua_seqgeti(L, -2, u);
      if (luaBsort_comp(L, -1, -2)) { /* a[u]<a[i]? */
        luaBsetseq3(L, -3, i, u);
      } else
        agn_poptoptwo(L);
    }
    if (u-l == 2) break;  /* only 3 elements */
    lua_seqgeti(L, -1, i);  /* Pivot */
    lua_pushvalue(L, -1);
    lua_seqgeti(L, -3, u-1);
    luaBsetseq3(L, -4, i, u-1);
    /* a[l] <= P == a[u-1] <= a[u], only need to sort from l+1 to u-2 */
    i = l; j = u-1;
    for (;;) {  /* invariant: a[l..i] <= P <= a[j..u] */
      /* repeat ++i until a[i] >= P */
      while (lua_seqgeti(L, -2, ++i), luaBsort_comp(L, -1, -2)) {
        if (i > u) luaL_error(L, "Error in " LUA_QS ": invalid order function for sorting.", "sorted");
        agn_poptop(L);  /* remove a[i] */
      }
      /* repeat --j until a[j] <= P */
      while (lua_seqgeti(L, -3, --j), luaBsort_comp(L, -3, -1)) {
        if (j < l) luaL_error(L, "Error in " LUA_QS ": invalid order function for sorting.", "sorted");
        agn_poptop(L);  /* remove a[j] */
      }
      if (j<i) {
        lua_pop(L, 3);  /* pop pivot, a[i], a[j] */
        break;
      }
      luaBsetseq3(L, -4, i, j);
    }
    lua_seqgeti(L, -1, u-1);
    lua_seqgeti(L, -2, i);
    luaBsetseq3(L, -3, u-1, i);  /* swap pivot (a[u-1]) with a[i] */
    /* a[l..i-1] <= a[i] == P <= a[i+1..u] */
    /* adjust so that smaller half is in [j..i] and larger one in [l..u] */
    if (i-l < u-i) {
      j = l; i = i-1; l = i+2;
    }
    else {
      j = i+1; i = u; u = j-2;
    }
    auxsortedseq(L, j, i);  /* call recursively the smaller one */
  }  /* repeat the routine for the larger one */
}


static int luaB_sorted (lua_State *L) {
  int n = agn_nops(L, 1);
  luaL_checkstack(L, 40, "");  /* assume array is smaller than 2^40 */
  if (!lua_isnoneornil(L, 2))  /* is there a 2nd argument? */
    luaL_checktype(L, 2, LUA_TFUNCTION);
  lua_settop(L, 2);  /* make sure there is two arguments */
  agn_copy(L, 1);
  switch (lua_type(L, 1)) {
    case LUA_TTABLE:
      auxsortedtable(L, 1, n);
      break;
    case LUA_TSEQ:
      auxsortedseq(L, 1, n);
      break;
    default:
      luaL_error(L, "Error in " LUA_QS ": table or sequence expected as first argument, got %s.", "sorted",
        lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


/* extended 0.29.3, December 25, 2009; extended 0.30.2, January 11, 2010 */
static int agnB_zip (lua_State *L) {
  int i, j, sizea, sizeb, nargs;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (lua_type(L, 2) == LUA_TSEQ && lua_type(L, 3) == LUA_TSEQ) {
    sizea = agn_seqsize(L, 2);
    sizeb = agn_seqsize(L, 3);
    if (sizea != sizeb)
      luaL_error(L, "Error in " LUA_QS ": sequences of different size.", "zip");
    agn_createseq(L, sizea);
    /* set its type to the one of the original sequence */
    if (agn_getutype(L, 2)) {
      agn_setutype(L, -2, -1);
      agn_poptop(L);  /* pop type (string) */
    } else if (agn_getutype(L, 3)) {
      agn_setutype(L, -2, -1);
      agn_poptop(L);  /* pop type (string) */
    }
    /* copy metatable */
    if (lua_getmetatable(L, 2))
      lua_setmetatable(L, -2);
    else if (lua_getmetatable(L, 3))
      lua_setmetatable(L, -2);
    /* now traverse sequences */
    for (i=0; i < sizea; i++) {
      lua_pushvalue(L, 1);      /* push function; FIXME: can be optimized */
      lua_seqgeti(L, 2, i+1);
      lua_seqgeti(L, 3, i+1);
      for (j=4; j <= nargs; j++) {
        lua_pushvalue(L, j);
      }
      lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
      lua_seqinsert(L, -2);     /* store result to new sequence */
    }
  }
  else if (lua_type(L, 2) == LUA_TTABLE && lua_type(L, 3) == LUA_TTABLE) {
    sizea = agn_size(L, 2);
    sizeb = agn_size(L, 3);
    if (sizea != sizeb)
      luaL_error(L, "Error in " LUA_QS ": tables of different size.", "zip");
    lua_createtable(L, sizea, 0);
    /* set its type to the one of the original table */
    if (agn_getutype(L, 2)) {
      agn_setutype(L, -2, -1);
      agn_poptop(L);  /* pop type (string) */
    } else if (agn_getutype(L, 3)) {
      agn_setutype(L, -2, -1);
      agn_poptop(L);  /* pop type (string) */
    }
    /* copy metatable */
    if (lua_getmetatable(L, 2))
      lua_setmetatable(L, -2);
    else if (lua_getmetatable(L, 3))
      lua_setmetatable(L, -2);
    /* now traverse tables, changed 0.29.3 */
    lua_pushnil(L);
    while (agn_fnext(L, 2, 1, 0) != 0) {
      lua_pushvalue(L, -3);     /* push key of left-hand table */
      lua_gettable(L, 3);       /* get value of right-hand table */
      if (lua_isnil(L, -1)) {
        const char *key = lua_tostring(L, -4);
        if (key != NULL)
          luaL_error(L, "Error in " LUA_QS ": key " LUA_QS " missing in second table.", "zip", key);
        else
          luaL_error(L, "Error in " LUA_QS ": key missing in second table.", "zip");
      }
      for (j=4; j <= nargs; j++) {
        lua_pushvalue(L, j);
      }
      lua_call(L, nargs-1, 1);  /* call function with nargs-1 argument(s) and one result */
      lua_rawset2(L, -3);       /* set result to new table, keep key on stack */
    }
  } else
    luaL_error(L, "Error in " LUA_QS ": expected either two sequences or two tables.", "zip");
  return 1;
}


/* luaB_min: return the smallest value in a list. If the option 'sorted' is given,
   the list is not traversed, instead the first entry in the list is returned.
   If the list is empty, NULL is returned. lua_getn must be called at first,
   otherwise lua_rawgeti would return 0 with empty lists.
   June 20, 2007 */

static int luaB_min (lua_State *L) {
  int i, n, type;
  lua_Number min;
  const char *option;
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  option = luaL_optstring(L, 2, "default");
  n = agn_nops(L, 1);
  min = 0;  /* to prevent compiler warnings */
  if (n == 0) {  /* 0.26.1 patch */
    lua_pushnil(L);
    return 1;
  }
  if (strcmp(option, "sorted") == 0) {
    lua_pushnumber(L, (type == LUA_TTABLE) ? agn_getinumber(L, 1, 1) : lua_seqgetinumber(L, 1, 1) );
    return 1;
  }
  switch (type) {
    case LUA_TTABLE: {
      min = agn_getinumber(L, 1, 1);
      for (i=2; i<=n; i++) {
        lua_Number v = agn_getinumber(L, 1, i);
        if (v < min)
          min = v;
      }
      break;
    }
    case LUA_TSEQ: {
      min = lua_seqgetinumber(L, 1, 1);
      for (i=2; i<=n; i++) {
        lua_Number v = lua_seqgetinumber(L, 1, i);
        if (v < min)
          min = v;
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  lua_pushnumber(L, min);
  return 1;
}


/* tbl_max: return the largest value in a list. If the option 'sorted' is given,
   the list is not traversed, instead the last entry in the list is returned.
   If the list is empty, NULL is returned. lua_getn must be called at first,
   otherwise lua_rawgeti would return 0 with empty lists.
   June 20, 2007; tuned December 23, 2007 */

static int luaB_max (lua_State *L) {
  int i, n, type;
  lua_Number max;
  const char *option;
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  option = luaL_optstring(L, 2, "default");
  n = agn_nops(L, 1);
  max = 0;  /* to prevent compiler warnings */
  if (n == 0) {  /* 0.26.1 patch */
    lua_pushnil(L);
    return 1;
  }
  if (strcmp(option, "sorted") == 0) {
    lua_pushnumber(L, (type == LUA_TTABLE) ? agn_getinumber(L, 1, n) : lua_seqgetinumber(L, 1, n) );
    return 1;
  }
  switch (type) {
    case LUA_TTABLE: {
      max = agn_getinumber(L, 1, 1);
      for (i=2; i<=n; i++) {
        lua_Number v = agn_getinumber(L, 1, i);
        if (v > max)
          max = v;
      }
      break;
    }
    case LUA_TSEQ: {
      max = lua_seqgetinumber(L, 1, 1);
      for (i=2; i<=n; i++) {
        lua_Number v = lua_seqgetinumber(L, 1, i);
        if (v > max)
          max = v;
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  lua_pushnumber(L, max);
  return 1;
}


/* creates a sequence with values a, a+step ..., b-step, b; based on calc.fseq
   uses a modified version of Kahan's summation algorithm to avoid roundoff errors
   0.30.0, December 30, 2009; merged with calc.fseq 1.3.3, February 01, 2011

   calc.fseq comments: creates a sequence with values f(a), ..., f(b); June 30, 2008;
   20 % faster than an Agena implementation; extended August 20, 2009, 0.27.0
   extended 0.30.0, December 29, 2009 with Kahan summation algorithm to avoid roundoff errors;
   extended 1.8.8, November 07, 2012, to also return non-numeric sequences */
static int luaB_nseq (lua_State *L) {
  /* use `volatile` so that the compiler does not render the Kahan code effectless */
  volatile lua_Number idx, step, c, y, t;
  lua_Number a, b, total;
  size_t counter, i, j, nargs;
  int offset;
  nargs = lua_gettop(L);
  counter = 0;
  offset = lua_isfunction(L, 1);
  a = agn_checknumber(L, 1+offset);
  b = agn_checknumber(L, 2+offset);
  c = 0;
  step = agnL_optnumber(L, 3+offset, 1);
  if (step <= 0)
    luaL_error(L, "Error in " LUA_QS ": step size must be positive.", "nseq");
  if (a > b)
    luaL_error(L, "Error in " LUA_QS ": start value greater than stop value.", "nseq");
  total = trunc(fabs(b-a)/step+1);  /* total number of iterations, do not use floor, (int), etc ! */
  agn_createseq(L, total);
  idx = a;
  /* total > counter: prevents that the last element is inserted even if a roundoff error occured. */
  if (offset) {  /* function passed ? */
    while (idx <= b || total > counter) {
      lua_pushvalue(L, 1);
      lua_pushnumber(L, (fabs(idx) < AGN_EPSILON) ? 0 : idx);  /* quite dirty hack to avoid roundoff errors with 0 */
      for (i = 4+offset, j = 0; i <= nargs; i++, j++) lua_pushvalue(L, i);  /* Agena 1.8.8 */
      lua_call(L, 1+j, 1);
      lua_seqseti(L, -2, ++counter);
      y = step - c;
      t = idx + y;
      c = (t - idx) - y;
      idx = t;
    }
  } else {
    while (idx <= b || total > counter) {
      if (fabs(idx) < AGN_EPSILON) idx = 0;
      lua_seqsetinumber(L, -1, ++counter, idx);
      y = step - c;
      t = idx + y;
      c = (t - idx) - y;
      idx = t;
    }
  }
  return 1;
}


static lua_Number getnumber (lua_State *L, int idx, int b, const char *fname) {
  lua_Number r;
  agn_pairgeti(L, idx, b);
  if (lua_type(L, -1) != LUA_TNUMBER)
    luaL_error(L, "Error in " LUA_QS ": pair of numbers expected, got %s.",
      fname, lua_typename(L, lua_type(L, -1)));
  r = agn_tonumber(L, -1);
  agn_poptop(L);
  return r;
}


static int createvector (lua_State *L, int idx, int flag) {
  int a, b, narray, nhash, type;
  type = lua_type(L, idx);
  luaL_typecheck(L, type == LUA_TPAIR, idx, "expected a pair of numbers", type);
  a = getnumber(L, idx, 1, "dimension");
  b = getnumber(L, idx, 2, "dimension");
  if (a > b)
    luaL_error(L, "Error in " LUA_QS ": left border greater than right border.", "dimension");
  narray = (b > 0) ? b : 0;
  nhash =  (a < 1) ? -a + 1 : 0;
  lua_createtable(L, narray, nhash);
  if (flag) {  /* initialiser given ? */
    int i;
    for (i=a; i <= b; i++) {
      lua_pushvalue(L, idx+1);
      if (lua_type(L, -1) >= LUA_TTABLE) {  /* Agena 1.3.2, generate genuine copies of the structure */
         agn_copy(L, -1);
         lua_remove(L, -2);
      }
      lua_rawseti(L, -2, i);
    }
  }
  return 1;
}


static int agnB_dimension (lua_State *L) {  /* 0.30.1, 01.01.2010 */
  size_t nargs;
  int flag = 0;
  nargs = lua_gettop(L);
  if (nargs < 1)
    luaL_error(L, "Error in " LUA_QS ": at least one argument expected.", "dimension");
  else if (nargs == 3 && lua_type(L, 3) == LUA_TPAIR)
    luaL_error(L, "Error in " LUA_QS ": third argument must not be a pair.", "dimension");
  else if (nargs > 3)
    luaL_error(L, "Error in " LUA_QS ": too many arguments.", "dimension");
  if (lua_type(L, nargs) != LUA_TPAIR) {  /* initialiser given ? */
    nargs--;
    flag = 1;
  }
  if (nargs == 1) {
    createvector(L, 1, flag);  /* create a vector */
  } else {  /* create a matrix */
    int i, a, b, narray, nhash, type;
    type = lua_type(L, 1);
    luaL_typecheck(L, type == LUA_TPAIR, 1, "expected a pair of numbers", type);
    a = getnumber(L, 1, 1, "dimension");
    b = getnumber(L, 1, 2, "dimension");
    narray = (b > 0) ? b : 0;
    nhash =  (a < 1) ? -a + 1 : 0;
    lua_createtable(L, narray, nhash);
    for (i=a; i <= b; i++) {  /* put row vectors into table */
      createvector(L, 2, flag);
      lua_rawseti(L, -2, i);
    }
  }
  return 1;
}


/* get an entry from a table or sequence without issuing an error if an index does not exist, 13.01.2010, 0.30.3;
   the Agena version is five times slower; patched 0.31.5 to work with sequences as designed */
static int agnB_getentry (lua_State *L) {
  int type, nargs, i;
  type = lua_type(L, 1);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs++, "too many arguments");
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  lua_pushvalue(L, 1);
  for (i=2; i < nargs && (type == LUA_TTABLE || type == LUA_TSEQ); i++) {
    lua_pushvalue(L, i);
    if (type == LUA_TTABLE)  /* separating the code for tables and for seqs does not produce faster results */
      lua_rawget(L, -2);
    else if (type == LUA_TSEQ)
      lua_seqrawget2(L, -2);
    type = lua_type(L, -1);
    lua_replace(L, -2);  /* remove old structure, replace it by new one */
  }
  if (nargs > i) {  /* return null where Agena would throw an error otherwise because of invalid indexing */
    agn_poptop(L);  /* remove last entry */
    lua_pushnil(L);  /* push null */
  }  /* otherwise return last entry */
  return 1;
}

#ifdef PROPCMPLX
void aux_arctan2 (lua_Number a, lua_Number b, lua_Number c, lua_Number d, lua_Number *real, lua_Number *imag) {
  lua_Number t1, t2, t3, t4, t6, t8, t10, t13, t15, t16, t17, t19,
    t20, t22, t24, t28, t30, t31, t32, t42, t44, t47, t49, t51, t54,
    t60, re, im;
  t1 = b/2+c/2;
  t2 = c*c;
  t3 = t2*t2;
  t4 = d*d;
  t6 = a*a;
  t8 = b*b;
  t10 = t4*t4;
  t13 = t6*t6;
  t15 = t8*t8;
  t16 = c*d;
  t17 = a*b;
  t19 = t3+2.0*t2*t4+2.0*t2*t6-2.0*t2*t8+t10-2.0*t4*t6+2.0*t4*t8+t13+2.0*t6*t8+t15+8.0*t16*t17;
  t20 = sqrt(t19);
  t22 = sqrt(2.0*t20+2.0*t2-2.0*t4+2.0*t6-2.0*t8);
  t24 = 1/t20;
  re = 2*t16+2*t17;
  im = -t2+t4-t6+t8;
  t28 = csgn(re, im);
  t30 = 2.0*t20-2.0*t2+2.0*t4-2.0*t6+2.0*t8;
  t31 = sqrt(t30);
  t32 = t31*t24;
  t42 = a-d;
  t44 = t28*t28;
  t47 = 1/(t20/2+t2/2-t4/2+t6/2-t8/2+t44*t30/4);
  t49 = b+c;
  t51 = t31*t47;
  t54 = pow(t42*t22*t47/2+t49*t28*t51/2, 2.0);
  t60 = pow(t49*t22*t47/2-t42*t28*t51/2, 2.0);
  *real = atan2(t1*t22*t24+(-a/2+d/2)*t28*t32, (a/2-d/2)*t22*t24+t1*t28*t32);
  *imag = -log(t54+t60)/2;
}
#endif

/* ANSI part completely rewritten 0.33.2 */
static int mathB_arctan2 (lua_State *L) {
  if (lua_type(L, 1) == LUA_TNUMBER && lua_type(L, 2) == LUA_TNUMBER)
    lua_pushnumber(L, atan2(agn_checknumber(L, 1), agn_checknumber(L, 2)));
  else if (lua_type(L, 1) == LUA_TCOMPLEX && lua_type(L, 2) == LUA_TCOMPLEX) {
#ifndef PROPCMPLX
    agn_Complex x, y;
    y = agn_tocomplex(L, 1);
    x = agn_tocomplex(L, 2);
    agn_createcomplex(L, -I*clog((x+I*y)/csqrt(x*x+y*y)));
#else
    lua_Number a, b, c, d, re, im;
    a = agn_complexreal(L, 1); b = agn_compleximag(L, 1);
    c = agn_complexreal(L, 2); d = agn_compleximag(L, 2);
    aux_arctan2(a, b, c, d, &re, &im);
    agn_createcomplex(L, re, im);
#endif
  }
  else if (lua_type(L, 1) == LUA_TCOMPLEX && lua_type(L, 2) == LUA_TNUMBER) {
#ifndef PROPCMPLX
    agn_Complex x, y;
    y = agn_tocomplex(L, 1);
    x = agn_tonumber(L, 2) + 0*I;  /* Agena 1.4.3/1.5.0 */
    agn_createcomplex(L, -I*clog((x+I*y)/csqrt(x*x+y*y)));
#else
    lua_Number a, b, c, d, re, im;
    a = agn_complexreal(L, 1); b = agn_compleximag(L, 1);
    c = agn_tonumber(L, 2); d = 0;  /* Agena 1.4.3/1.5.0 */
    aux_arctan2(a, b, c, d, &re, &im);
    agn_createcomplex(L, re, im);
#endif
  }
  else if (lua_type(L, 1) == LUA_TNUMBER && lua_type(L, 2) == LUA_TCOMPLEX) {
#ifndef PROPCMPLX
    agn_Complex x, y;
    y = agn_tonumber(L, 1);  /* Agena 1.4.3/1.5.0 */
    x = agn_tocomplex(L, 2);
    agn_createcomplex(L, -I*clog((x+I*y)/csqrt(x*x+y*y)));
#else
    lua_Number a, b, c, d, re, im;
    a = agn_tonumber(L, 1); b = 0;  /* Agena 1.4.3/1.5.0 */
    c = agn_complexreal(L, 2); d = agn_compleximag(L, 2);
    aux_arctan2(a, b, c, d, &re, &im);
    agn_createcomplex(L, re, im);
#endif
  }
  else {
    if (lua_gettop(L) != 2)
      luaL_error(L, "Error in " LUA_QS ": two arguments expected.", "arctan2");
    else
      luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "arctan2",
        lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


static int mathB_arccoth (lua_State *L) {
  switch (lua_type(L, 1)) {
    case LUA_TNUMBER: {
      lua_Number x, arg;
      x = agn_tonumber(L, 1);
      arg = (x+1.0)/(x-1.0);
      if (arg <= 1 || isnan(arg) || !isfinite(arg))  /* patch 0.26.1 */
        lua_pushundefined(L);
      else
        lua_pushnumber(L, 0.5*log(arg));
      break;
    }
    case LUA_TCOMPLEX: {
      lua_Number re, im, t1, t2, t3, t6;
#ifndef PROPCMPLX
      agn_Complex z = agn_tocomplex(L, 1);
      re = creal(z); im = cimag(z);
#else
      re = agn_complexreal(L, 1); im = agn_compleximag(L, 1);
#endif
      if (im == 0 && (re == 1 || re == -1))  /* patch 0.26.1 */
        lua_pushundefined(L);
      else {
        t1 = re+1;
        t2 = t1*t1;
        t3 = im*im;
        t6 = pow(re-1, 2);
	if (im != 0 || re <= 1)  /* 0.33.1 fix */
#ifndef PROPCMPLX
          agn_createcomplex(L, log((t2+t3)/(t6+t3))/4+I*(PI/2+atan2(im, t1)/2-atan2(-im, 1-re)/2));
#else
          agn_createcomplex(L, log((t2+t3)/(t6+t3))/4, PI/2+atan2(im, t1)/2-atan2(-im, 1-re)/2);
#endif
        else
#ifndef PROPCMPLX
          agn_createcomplex(L, log((t2+t3)/(t6+t3))/4 + I*0);
#else
          agn_createcomplex(L, log((t2+t3)/(t6+t3))/4, 0);
#endif
      }
      break;
    }
    default: luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "arccoth",
      lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


static int mathB_argument (lua_State *L) {
  switch (lua_type(L, 1)) {
    case LUA_TCOMPLEX:
#ifndef PROPCMPLX
      lua_pushnumber(L, carg(agn_tocomplex(L, 1))); break;
#else
      { lua_Number a = agn_complexreal(L, 1), b = agn_compleximag(L, 1);
      lua_pushnumber(L, atan2(b, a)); break; }
#endif
    case LUA_TNUMBER:
      lua_pushnumber(L, (agn_tonumber(L, 1) < 0) ? PI : 0); break;  /* 0.27.0 patch */
    default: luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "argument",
      lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


/* 9 % faster than the Agena implementation */

static int mathB_conjugate (lua_State *L) {
  switch (lua_type(L, 1)) {
    case LUA_TCOMPLEX:
#ifndef PROPCMPLX
      agn_createcomplex(L, conj(agn_tocomplex(L, 1))); break;
#else
      { lua_Number a = agn_complexreal(L, 1), b = agn_compleximag(L, 1);
      agn_createcomplex(L, a, -b); break; }
#endif
    case LUA_TNUMBER:
      lua_pushvalue(L, 1);
      break;
    default: luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "conjugate",
      lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


static int mathB_root (lua_State *L) {  /* equivalent to Maple's `surd` */
  lua_Number n;
  luaL_checkany(L, 2);
  n = agn_checknumber(L, 2);
  if (n == 0) { /* 0.33.2 */
    lua_pushundefined(L);
    return 1;
  } else if (ISFLOAT(n))
    luaL_error(L, "Error in " LUA_QS ": second argument must be an integer, got %f.", "root", n);
  switch (lua_type(L, 1)) {
    case LUA_TNUMBER: {
      lua_Number x = agn_tonumber(L, 1);
      if (x == 0 && n < 0)
        lua_pushundefined(L);
      else if (x >= 0)
        lua_pushnumber(L, pow(x, 1/n));
      else if (fmod(n, 2) == 0)  /* x < 0 and n is even ? */
        lua_pushundefined(L);
      else
        lua_pushnumber(L, -pow(-x, 1/n));
      break;
    }
    case LUA_TCOMPLEX: {  /* completely rewritten 0.33.2 */
      lua_Number a, b, k, t1, t2, t3, t7, t9, t11, t13, t14, t17, t18;
#ifndef PROPCMPLX
      agn_Complex x;
      x = agn_tocomplex(L, 1);
      a = creal(x); b = cimag(x);
#else
      a = agn_complexreal(L, 1); b = agn_compleximag(L, 1);
#endif
      k = rint(atan2(b, a)*(n-1)/(2*DP1));
      t1 = 1/n;
      t2 = a*a;
      t3 = b*b;
      t7 = exp(t1*log(t2+t3)/2);
      t9 = t1*atan2(b, a);
      t11 = t7*cos(t9);
      t13 = k*DP1*t1;  /* DP1 = Pi */
      t14 = cos(2.0*t13);
      t17 = t7*sin(t9);
      t18 = sin(2.0*t13);
      if (b == 0 && a < 0 && n > 0 && fmod(n, 2) == 0)
#ifndef PROPCMPLX
        agn_createcomplex(L, t11*t14-t17*t18 - I*(t17*t14+t11*t18));
      else
        agn_createcomplex(L, t11*t14-t17*t18 + I*(t17*t14+t11*t18));
#else
        agn_createcomplex(L, t11*t14-t17*t18, -(t17*t14+t11*t18));
      else
        agn_createcomplex(L, t11*t14-t17*t18, t17*t14+t11*t18);
#endif
      break;
    }
    default: luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "root",
      lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


/*
> restart;
> assume(n, posint);
> f := (a+I*b)^(1/n);

                                       / 1  \
                                       |----|
                                       \ n~ /
                         f := (a + I b)

> evalc(f);

            2    2
        ln(a  + b )      arctan(b, a)
exp(1/2 -----------) cos(------------)
            n~                n~

                     2    2
                 ln(a  + b )      arctan(b, a)
     + I exp(1/2 -----------) sin(------------)
                     n~                n~

> readlib(C);

                           proc()  ...  end

> C(evalc(f), optimized);
      t1 = 1/n;
      t2 = a*a;
      t3 = b*b;
      t7 = exp(t1*log(t2+t3)/2);
      t9 = t1*atan2(b,a);
      t14 = t7*cos(t9)+sqrt(-1.0)*t7*sin(t9);
*/

static int mathB_proot (lua_State *L) {  /* computer the principal root, equivalent to Maple's `root`, 1.12.7 */
  lua_Number n;
  luaL_checkany(L, 2);
  n = agn_checknumber(L, 2);
  if (ISFLOAT(n) || n < 1)
    luaL_error(L, "Error in " LUA_QS ": second argument must be a positive integer, got %f.", "proot", n);
  switch (lua_type(L, 1)) {
    case LUA_TNUMBER: {
      lua_Number x = agn_tonumber(L, 1);
      if (x >= 0)
        lua_pushnumber(L, pow(x, 1/n));
      else if (fmod(n, 2) == 0)  /* x < 0 and n is even ? */
        lua_pushundefined(L);
      else
        lua_pushnumber(L, -pow(-x, 1/n));
      break;
    }
    case LUA_TCOMPLEX: {
      lua_Number a, b, t1, t2, t3, t7, t9;
#ifndef PROPCMPLX
      agn_Complex x;
      x = agn_tocomplex(L, 1);
      a = creal(x); b = cimag(x);
#else
      a = agn_complexreal(L, 1); b = agn_compleximag(L, 1);
#endif
      t1 = 1/n;
      t2 = a*a;
      t3 = b*b;
      t7 = exp(t1*log(t2+t3)/2);
      t9 = t1*atan2(b,a);
#ifndef PROPCMPLX
      agn_createcomplex(L, t7*cos(t9) + I*t7*sin(t9));
#else
      agn_createcomplex(L, t7*cos(t9), t7*sin(t9));
#endif
      break;
    }
    default: luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "proot",
      lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


/*
> readlib(C);

> f := evalc((a+I*b)^(1/3));

       2    2 1/6
f := (a  + b )    cos(1/3 arctan(b, a))

           2    2 1/6
     + I (a  + b )    sin(1/3 arctan(b, a))

> C(f, optimized);
      t1 = a*a;
      t2 = b*b;
      t4 = pow(t1+t2,1.0/6.0);
      t5 = atan2(b,a);
      t10 = t4*cos(t5/3)+sqrt(-1.0)*t4*sin(t5/3);
*/

static int agnB_cbrt (lua_State *L) {  /* 1.12.7 */
  switch (lua_type(L, 1)) {
    case LUA_TNUMBER: {
      lua_Number x, r;
      x = agn_tonumber(L, 1);
      r = (x < 0) ? -pow(-x, 0.3333333333333333) : pow(x, 0.3333333333333333);
      lua_pushnumber(L, r);
      break;
    }
    case LUA_TCOMPLEX: {  /* completely rewritten 0.33.2 */
      lua_Number a, b, t1, t2, t4, t5;
#ifndef PROPCMPLX
      agn_Complex x;
      x = agn_tocomplex(L, 1);
      a = creal(x); b = cimag(x);
#else
      a = agn_complexreal(L, 1); b = agn_compleximag(L, 1);
#endif
      t1 = a*a;
      t2 = b*b;
      t4 = pow(t1+t2,1.0/6.0);
      t5 = atan2(b,a);
#ifndef PROPCMPLX
      agn_createcomplex(L, t4*cos(t5/3) + I*t4*sin(t5/3));
#else
      agn_createcomplex(L, t4*cos(t5/3), t4*sin(t5/3));
#endif
      break;
    }
    default: luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "cbrt",
      lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}

/* redefined 0.26.1, 11.08.2009 */

static int mathB_irem (lua_State *L) {
  lua_pushnumber(L, fmod(agn_checknumber(L, 1), agn_checknumber(L, 2)));
  return 1;
}


static int agnB_drem (lua_State *L) {  /* 2.0.0 R5 */
  lua_pushnumber(L, remainder(agn_checknumber(L, 1), agn_checknumber(L, 2)));
  return 1;
}


/* computes both the integer quotient and the integer remainder of the number a divided by the number b and returns them.
   If a or b are not integers, the function returns `undefined twice. 10.04.2013 */

static int mathB_iqr (lua_State *L) {  /* 1.10.6 */
  lua_Number a, b, sa, sb;
  a = agn_checknumber(L, 1);
  b = agn_checknumber(L, 2);
  if (ISFLOAT(a) || ISFLOAT(b)) {  /* behave like Maple's irem and iquo functions */
    lua_pushundefined(L);
    lua_pushundefined(L);
  } else {
    sa = (a == 0) ? 0 : ((a < 0) ? -1 : 1);
    sb = (b == 0) ? 0 : ((b < 0) ? -1 : 1);
    lua_pushnumber(L, (b != 0) ? sa*sb*floor(fabs(a/b)) : AGN_NAN);
    lua_pushnumber(L, fmod(a, b));
  }
  return 2;
}


static int mathB_modf (lua_State *L) {
  lua_Number ip;
  lua_Number fp = modf(agn_checknumber(L, 1), &ip);
  lua_pushnumber(L, ip);
  lua_pushnumber(L, fp);
  return 2;
}


static int mathB_frexp (lua_State *L) {
  int e;
  lua_pushnumber(L, frexp(agn_checknumber(L, 1), &e));
  lua_pushinteger(L, e);
  return 2;
}


static int mathB_ldexp (lua_State *L) {
  lua_pushnumber(L, ldexp(agn_checknumber(L, 1), agnL_checkint(L, 2)));
  return 1;
}


/* mathB_roundf: rounds real value x to the d-th digit; patched Dec 22, 2007 for negative values */

static int mathB_roundf (lua_State *L) {
  lua_Number x = agn_checknumber(L, 1);
  lua_Number d = agnL_optinteger(L, 2, 0);
  if (x < 0)
    lua_pushnumber(L, ceil(pow(10, d)*x-0.5) * pow(10, (-d)));
  else
    lua_pushnumber(L, floor(pow(10, d)*x+0.5) * pow(10, (-d)));
  return 1;
}


/* hypotenuse; June 19, 2007 */

static int mathB_hypot (lua_State *L) {
  lua_pushnumber(L, hypot(agn_checknumber(L, 1), agn_checknumber(L, 2)));
  return 1;
}


/* computes the binomial coefficient;
   added December 26, 2007; 0.9.0; changed April 20, 2008 - 0.10.0 */

static int mathB_binomial (lua_State *L) {
  lua_Number n, k;
  n = agn_checknumber(L, 1);
  k = agn_checknumber(L, 2);
  if (n < 0 || k < 0 || ISFLOAT(n) || ISFLOAT(k))
    lua_pushundefined(L);
  else if (n < k)  /* 0.28.3 patch */
    lua_pushnumber(L, 0);
  else
    lua_pushnumber(L, exp(lgamma(n+1) - lgamma(k+1) - lgamma(n-k+1)));
  return 1;
}


/* complementary error function for complex arguments
   written by Takuya Ooura in FORTRAN and available at http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html.
   See gamerf2a.f file.

   Copyright(C) 1996 Takuya OOURA.
   You may use, copy, modify this code for any purpose and without fee.
   You may distribute this ORIGINAL package. */

static double e_p[13] = {
  2.94608570191793668e-01,
  1.81694307871527086e-01,
  6.91087778921425355e-02,
  1.62114197106901582e-02,
  2.34533471539159422e-03,
  2.09259199579049675e-04,
  1.15149016557480535e-05,
  3.90779571296927748e-07,
  8.17898509247247602e-09,
  1.05575446466983499e-10,
  8.40470321453263734e-13,
  4.12646136715431977e-15,
  1.24947948599560084e-17
};

static double e_q[13] = {
  6.04152433382652546e-02,
  5.43737190044387291e-01,
  1.51038108345663136e+00,
  2.96034692357499747e+00,
  4.89363471039948562e+00,
  7.31024444393009580e+00,
  1.02101761241668280e+01,
  1.35934297511096823e+01,
  1.74600053247586586e+01,
  2.18099028451137569e+01,
  2.66431223121749773e+01,
  3.19596637259423197e+01,
  3.77595270864157841e+01
};

static double e_r[13] = {
  1.56478036351085356e-01,
  2.45771407110492625e-01,
  1.19035163906534275e-01,
  3.55561834455977740e-02,
  6.55014550718381002e-03,
  7.44188068433574137e-04,
  5.21447276257559040e-05,
  2.25337799750608244e-06,
  6.00556181041662576e-08,
  9.87118243564461826e-10,
  1.00064645539515792e-11,
  6.25587539334288736e-14,
  2.41207864479170276e-16
};

static double e_s[13] = {
  0,
  2.41660973353061018e-01,
  9.66643893412244073e-01,
  2.17494876017754917e+00,
  3.86657557364897629e+00,
  6.04152433382652546e+00,
  8.69979504071019666e+00,
  1.18413876942999899e+01,
  1.54663022945959052e+01,
  1.95745388415979425e+01,
  2.41660973353061018e+01,
  2.92409777757203832e+01,
  3.47991801628407866e+01
};

#ifndef PROPCMPLX
static agn_Complex aux_cerf (agn_Complex x) {
  agn_Complex y, z;
  double pv, ph;
  pv = 1.27813464856668857e+01;
  ph = 6.64067324283344283e+00;
  y = x*x;
  if ((fabs(creal(x)) + fabs(cimag(x))) < ph) {
    z = cexp(pv*x);
    if (creal(z) >= 0) {
      y = cexp(-y)*x*(e_p[12]/(y+e_q[12])+e_p[11]/(y+e_q[11])
          + e_p[10]/(y+e_q[10]) + e_p[9]/(y+e_q[9]) + e_p[8]/(y+e_q[8]) + e_p[7]/(y+e_q[7])
          + e_p[6]/(y+e_q[6]) + e_p[5]/(y+e_q[5]) + e_p[4]/(y+e_q[4]) + e_p[3]/(y+e_q[3])
          + e_p[2]/(y+e_q[2]) + e_p[1]/(y+e_q[1]) + e_p[0]/(y+e_q[0])) + 2/(1+z);
    } else {
      y = cexp(-y)*x*(e_r[12]/(y+e_s[12])+e_r[11]/(y+e_s[11])
          + e_r[10]/(y+e_s[10]) + e_r[9]/(y+e_s[9]) + e_r[8]/(y+e_s[8]) + e_r[7]/(y+e_s[7])
          + e_r[6]/(y+e_s[6]) + e_r[5]/(y+e_s[5]) + e_r[4]/(y+e_s[4]) + e_r[3]/(y+e_s[3])
          + e_r[2]/(y+e_s[2]) + e_r[1]/(y+e_s[1]) + e_r[0]/y) + 2/(1-z);
    }
  } else {
    y = cexp(-y)*x*(e_p[12]/(y+e_q[12])+e_p[11]/(y+e_q[11])
        + e_p[10]/(y+e_q[10]) + e_p[9]/(y+e_q[9]) + e_p[8]/(y+e_q[8]) + e_p[7]/(y+e_q[7])
        + e_p[6]/(y+e_q[6]) + e_p[5]/(y+e_q[5]) + e_p[4]/(y+e_q[4]) + e_p[3]/(y+e_q[3])
        + e_p[2]/(y+e_q[2]) + e_p[1]/(y+e_q[1]) + e_p[0]/(y+e_q[0]));
    if (creal(x) < 0) y = y + 2;
  }
  return y;
}
#else
static void aux_cerf (double re, double im, double *rre, double *rim) {
  lua_Number x[2], y[2], z[2], pv, ph, t1, t3;
  pv = 1.27813464856668857e+01;
  ph = 6.64067324283344283e+00;
  x[0] = re;
  x[1] = im;
  t1 = pow(x[0],2.0);
  t3 = pow(x[1],2.0);
  y[0] = t1 - t3;
  y[1] = +2.0*x[0]*x[1];
  if ((fabs(re) + fabs(im)) < ph) {
    double tr, ti;
    tr = exp(pv*x[0]);
    ti = pv*x[1];
    z[0] = tr*cos(ti);
    z[1] = tr*sin(ti);
    if (z[0] >= 0) {
      double t1, t3, t6, t8, t9, t11, t12, t14, t16, t18, t20, t22, t24, t26, t28,
             t30, t32, t34, t36, t38, t40, t42, t44, t46, t48, t50, t52, t54, t56,
             t58, t60, t62, t64, t66, t68, t70, t72, t74, t76, t78, t80, t82, t84,
             t86, t88, t92, t119, t121, t122, t123, t125;
      t1 = exp(-y[0]);
      t3 = t1*cos(y[1]);
      t6 = t1*sin(y[1]);
      t8 = t3*x[0]+t6*x[1];
      t9 = y[0]+e_q[12];
      t11 = t9*t9;
      t12 = pow(y[1],2.0);
      t14 = 1/(t11+t12);
      t16 = y[0]+e_q[11];
      t18 = t16*t16;
      t20 = 1/(t18+t12);
      t22 = y[0]+e_q[10];
      t24 = t22*t22;
      t26 = 1/(t24+t12);
      t28 = y[0]+e_q[9];
      t30 = t28*t28;
      t32 = 1/(t30+t12);
      t34 = y[0]+e_q[8];
      t36 = t34*t34;
      t38 = 1/(t36+t12);
      t40 = y[0]+e_q[7];
      t42 = t40*t40;
      t44 = 1/(t42+t12);
      t46 = y[0]+e_q[6];
      t48 = t46*t46;
      t50 = 1/(t48+t12);
      t52 = y[0]+e_q[5];
      t54 = t52*t52;
      t56 = 1/(t54+t12);
      t58 = y[0]+e_q[4];
      t60 = t58*t58;
      t62 = 1/(t60+t12);
      t64 = y[0]+e_q[3];
      t66 = t64*t64;
      t68 = 1/(t66+t12);
      t70 = y[0]+e_q[2];
      t72 = t70*t70;
      t74 = 1/(t72+t12);
      t76 = y[0]+e_q[1];
      t78 = t76*t76;
      t80 = 1/(t78+t12);
      t82 = y[0]+e_q[0];
      t84 = t82*t82;
      t86 = 1/(t84+t12);
      t88 = e_p[12]*t9*t14+e_p[11]*t16*t20+e_p[10]*t22*t26+e_p[9]*t28*t32+e_p[8]*t34*t38
             + e_p[7]*t40*t44+e_p[6]*t46*t50+e_p[5]*t52*t56+e_p[4]*t58*t62+e_p[3]*t64*t68
             + e_p[2]*t70*t74+e_p[1]*t76*t80+e_p[0]*t82*t86;
      t92 = -t6*x[0]+t3*x[1];
      t119 = -e_p[12]*y[1]*t14-e_p[11]*y[1]*t20-e_p[10]*y[1]*t26-e_p[9]*y[1]*t32
             -e_p[8]*y[1]*t38-e_p[7]*y[1]*t44-e_p[6]*y[1]*t50-e_p[5]*y[1]*t56
             -e_p[4]*y[1]*t62-e_p[3]*y[1]*t68-e_p[2]*y[1]*t74-e_p[1]*y[1]*t80-e_p[0]*y[1]*t86;
      t121 = 1.0+z[0];
      t122 = t121*t121;
      t123 = pow(z[1],2.0);
      t125 = 1/(t122+t123);
      y[0] = t8*t88-t92*t119+2.0*t121*t125;
      y[1] = t92*t88+t8*t119-2.0*z[1]*t125;
    } else {
      double t1, t3, t6, t8, t9, t11, t12, t14, t16, t18, t20, t22, t24, t26, t28,
             t30, t32, t34, t36, t38, t40, t42, t44, t46, t48, t50, t52, t54, t56,
             t58, t60, t62, t64, t66, t68, t70, t72, t74, t76, t78, t80, t83, t85,
             t87, t91, t118, t120, t121, t122, t124;
      t1 = exp(-y[0]);
      t3 = t1*cos(y[1]);
      t6 = t1*sin(y[1]);
      t8 = t3*x[0]+t6*x[1];
      t9 = y[0]+e_s[12];
      t11 = t9*t9;
      t12 = pow(y[1],2.0);
      t14 = 1/(t11+t12);
      t16 = y[0]+e_s[11];
      t18 = t16*t16;
      t20 = 1/(t18+t12);
      t22 = y[0]+e_s[10];
      t24 = t22*t22;
      t26 = 1/(t24+t12);
      t28 = y[0]+e_s[9];
      t30 = t28*t28;
      t32 = 1/(t30+t12);
      t34 = y[0]+e_s[8];
      t36 = t34*t34;
      t38 = 1/(t36+t12);
      t40 = y[0]+e_s[7];
      t42 = t40*t40;
      t44 = 1/(t42+t12);
      t46 = y[0]+e_s[6];
      t48 = t46*t46;
      t50 = 1/(t48+t12);
      t52 = y[0]+e_s[5];
      t54 = t52*t52;
      t56 = 1/(t54+t12);
      t58 = y[0]+e_s[4];
      t60 = t58*t58;
      t62 = 1/(t60+t12);
      t64 = y[0]+e_s[3];
      t66 = t64*t64;
      t68 = 1/(t66+t12);
      t70 = y[0]+e_s[2];
      t72 = t70*t70;
      t74 = 1/(t72+t12);
      t76 = y[0]+e_s[1];
      t78 = t76*t76;
      t80 = 1/(t78+t12);
      t83 = pow(y[0],2.0);
      t85 = 1/(t83+t12);
      t87 = e_r[12]*t9*t14+e_r[11]*t16*t20+e_r[10]*t22*t26+e_r[9]*t28*t32+e_r[8]*t34*t38
            + e_r[7]*t40*t44+e_r[6]*t46*t50+e_r[5]*t52*t56+e_r[4]*t58*t62+e_r[3]*t64*t68
            + e_r[2]*t70*t74+e_r[1]*t76*t80+e_r[0]*y[0]*t85;
      t91 = -t6*x[0]+t3*x[1];
      t118 = -e_r[12]*y[1]*t14-e_r[11]*y[1]*t20-e_r[10]*y[1]*t26-e_r[9]*y[1]*t32
             -e_r[8]*y[1]*t38-e_r[7]*y[1]*t44-e_r[6]*y[1]*t50-e_r[5]*y[1]*t56-e_r[4]*y[1]*t62
             -e_r[3]*y[1]*t68-e_r[2]*y[1]*t74-e_r[1]*y[1]*t80-e_r[0]*y[1]*t85;
      t120 = 1.0-z[0];
      t121 = t120*t120;
      t122 = pow(z[1],2.0);
      t124 = 1/(t121+t122);
      y[0] = t8*t87-t91*t118+2.0*t120*t124;
      y[1] = t91*t87+t8*t118+2.0*z[1]*t124;
    }
  } else {
    double t1, t3, t6, t8, t9, t11, t12, t14, t16, t18, t20, t22, t24, t26, t28, t30,
           t32, t34, t36, t38, t40, t42, t44, t46, t48, t50, t52, t54, t56, t58, t60,
           t62, t64, t66, t68, t70, t72, t74, t76, t78, t80, t82, t84, t86, t88, t92,
           t119;
    t1 = exp(-y[0]);
    t3 = t1*cos(y[1]);
    t6 = t1*sin(y[1]);
    t8 = t3*x[0]+t6*x[1];
    t9 = y[0]+e_q[12];
    t11 = t9*t9;
    t12 = pow(y[1],2.0);
    t14 = 1/(t11+t12);
    t16 = y[0]+e_q[11];
    t18 = t16*t16;
    t20 = 1/(t18+t12);
    t22 = y[0]+e_q[10];
    t24 = t22*t22;
    t26 = 1/(t24+t12);
    t28 = y[0]+e_q[9];
    t30 = t28*t28;
    t32 = 1/(t30+t12);
    t34 = y[0]+e_q[8];
    t36 = t34*t34;
    t38 = 1/(t36+t12);
    t40 = y[0]+e_q[7];
    t42 = t40*t40;
    t44 = 1/(t42+t12);
    t46 = y[0]+e_q[6];
    t48 = t46*t46;
    t50 = 1/(t48+t12);
    t52 = y[0]+e_q[5];
    t54 = t52*t52;
    t56 = 1/(t54+t12);
    t58 = y[0]+e_q[4];
    t60 = t58*t58;
    t62 = 1/(t60+t12);
    t64 = y[0]+e_q[3];
    t66 = t64*t64;
    t68 = 1/(t66+t12);
    t70 = y[0]+e_q[2];
    t72 = t70*t70;
    t74 = 1/(t72+t12);
    t76 = y[0]+e_q[1];
    t78 = t76*t76;
    t80 = 1/(t78+t12);
    t82 = y[0]+e_q[0];
    t84 = t82*t82;
    t86 = 1/(t84+t12);
    t88 = e_p[12]*t9*t14+e_p[11]*t16*t20+e_p[10]*t22*t26+e_p[9]*t28*t32+e_p[8]*t34*t38
          +e_p[7]*t40*t44+e_p[6]*t46*t50+e_p[5]*t52*t56+e_p[4]*t58*t62+e_p[3]*t64*t68
          +e_p[2]*t70*t74+e_p[1]*t76*t80+e_p[0]*t82*t86;
    t92 = -t6*x[0]+t3*x[1];
    t119 = -e_p[12]*y[1]*t14-e_p[11]*y[1]*t20-e_p[10]*y[1]*t26-e_p[9]*y[1]*t32
           -e_p[8]*y[1]*t38-e_p[7]*y[1]*t44-e_p[6]*y[1]*t50-e_p[5]*y[1]*t56-e_p[4]*y[1]*t62
           -e_p[3]*y[1]*t68-e_p[2]*y[1]*t74-e_p[1]*y[1]*t80-e_p[0]*y[1]*t86;
    y[0] = t8*t88-t92*t119;
    y[1] = t92*t88+t8*t119;
    if (re < 0) y[0] += 2;
  }
  *rre = y[0];
  *rim = y[1];
}
#endif


static int mathB_erf (lua_State *L) {
  switch (lua_type(L, 1)) {
    case LUA_TNUMBER: {
      lua_pushnumber(L, erf(agn_tonumber(L, 1)));
      break;
    }
#ifndef PROPCMPLX
    /* error function for complex arguments
       written by Takuya Ooura in FORTRAN and available at http://www.kurims.kyoto-u.ac.jp/~ooura/gamerf.html.
       See gamerf2a.f file.

       Copyright(C) 1996 Takuya OOURA.
       You may use, copy, modify this code for any purpose and without fee.
       You may distribute this ORIGINAL package. */

    case LUA_TCOMPLEX: {  /* 0.28.1 */
      agn_Complex x, y, r;
      double p0, p1, p2, p3, p4, p5, p6;
      x = agn_tocomplex(L, 1);
      p0 =  1.12837916709551257;
      p1 = -3.76126389031837525e-01;
      p2 =  1.12837916709551257e-01;
      p3 = -2.68661706451312518e-02;
      p4 =  5.22397762544218784e-03;
      p5 = -8.54832702345085283e-04;
      p6 = 1.20553329817896643e-04;
      if (fabs(creal(x)) + fabs(cimag(x)) > 0.125) {
        if (creal(x) >= 0)
          r = 1 - aux_cerf(x);
        else
          r = aux_cerf(-x) - 1;
      } else {
        y = x*x;
        r = ((((((p6*y+p5)*y+p4)*y+p3)*y+p2)*y+p1)*y+p0)*x;
      }
      agn_createcomplex(L, r);
      break;
    }
#else
    case LUA_TCOMPLEX: {  /* 0.32.4 */
      lua_Number x[2], y[2], r[2], p0, p1, p2, p3, p4, p5, p6, re, im;
      x[0] = agn_complexreal(L, 1);
      x[1] = agn_compleximag(L, 1);
      p0 =  1.12837916709551257;
      p1 = -3.76126389031837525e-01;
      p2 =  1.12837916709551257e-01;
      p3 = -2.68661706451312518e-02;
      p4 =  5.22397762544218784e-03;
      p5 = -8.54832702345085283e-04;
      p6 = 1.20553329817896643e-04;
      if (fabs(x[0]) + fabs(x[1]) > 0.125) {
        if (x[0] >= 0) {
          aux_cerf(x[0], x[1], &re, &im);
          r[0] = 1 - re;
          r[1] = -im;
        } else {
          aux_cerf(-x[0], -x[1], &re, &im);
          r[0] = re - 1;
          r[1] = im;
        }
      } else {
        double t1, t3, t2, t4, t6, t7, t9, t11, t12, t14, t15, t17, t19, t22, t24, t26, t27,
               t29, t31, t33, t35, t36, t38, t40, t41, t43, t44, t46, t47, t49, t51, t52, t54,
               t57, t59, t61, t64, t65, t67, t68, t83, t85, t86, t90, t98, t99, t101, t106;
        t1 = pow(x[0], 2.0);
        t3 = pow(x[1], 2.0);
        y[0] = t1 - t3;
        y[1] = +2.0*x[0]*x[1];
        t1 = pow(y[0],2.0);
        t2 = t1*t1;
        t3 = t2*y[0];
        t4 = p5*t3;
        t6 = pow(y[1],2.0);
        t7 = t6*t6;
        t9 = p6*t7*t6;
        t11 = t7*y[1];
        t12 = p5*t11;
        t14 = t6*y[1];
        t15 = p3*t14;
        t17 = p1*y[1];
        t19 = p4*t7;
        t22 = p6*t2*t1;
        t24 = p2*t1;
        t26 = t1*y[0];
        t27 = p3*t26;
        t29 = p1*y[0];
        t31 = p2*t6;
        t33 = p4*t2;
        t35 = p4*t1;
        t36 = t6*x[0];
        t38 = p6*t2;
        t40 = t4*x[0]-t9*x[0]-t12*x[1]+t15*x[1]-t17*x[1]+t19*x[0]+t22*x[0]+t24*x[0]+t27*x[0]+t29*x[0]
              -t31*x[0]+t33*x[0]-6.0*t35*t36-15.0*t38*t36;
        t41 = p5*t26;
        t43 = p2*y[1];
        t44 = y[0]*x[1];
        t46 = p4*t26;
        t47 = y[1]*x[1];
        t49 = p5*t2;
        t51 = p5*t1;
        t52 = t14*x[1];
        t54 = p6*y[0];
        t57 = p6*t26;
        t59 = p6*t3;
        t61 = p3*t1;
        t64 = p3*t6;
        t65 = y[0]*x[0];
        t67 = p5*y[0];
        t68 = t7*x[0];
        t83 = y[1]*x[0];
        t85 = t29*x[1]+t33*x[1]+t17*x[0]-t15*x[0]-t9*x[1]-t31*x[1]+t24*x[1]+t12*x[0]+t19*x[1]+t22*x[1]
              +t27*x[1]+t4*x[1]+p0*x[1]+5.0*t49*t83;
        t86 = t14*x[0];
        t90 = t6*x[1];
        t98 = p6*t1;
        t99 = t7*x[1];
        t101 = p4*y[0];
        t106 = -20.0*t57*t86-10.0*t51*t86+2.0*t43*t65-6.0*t35*t90+6.0*t54*t11*x[0]+3.0*t61*t83+4.0*t46*t83+6.0*t59*t83
               -15.0*t38*t90+15.0*t98*t99-4.0*t101*t86-10.0*t41*t90+5.0*t67*t99-3.0*t64*t44;
        r[0] = t40-10.0*t41*t36-2.0*t43*t44-4.0*t46*t47-5.0*t49*t47+10.0*t51*t52
               -6.0*t54*t11*x[1]+20.0*t57*t52-6.0*t59*t47-3.0*t61*t47+p0*x[0]
               -3.0*t64*t65+5.0*t67*t68+15.0*t98*t68+4.0*t101*t52;
        r[1] = t85+t106;
      }
      agn_createcomplex(L, r[0], r[1]);
      break;
    }
#endif
    default:
      luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "erf",
        lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


static int mathB_erfc (lua_State *L) {
  switch (lua_type(L, 1)) {
    case LUA_TNUMBER: {
      lua_pushnumber(L, erfc(agn_checknumber(L, 1)));
      break;
    }
#ifndef PROPCMPLX
    case LUA_TCOMPLEX: {  /* 0.32.1 */
      agn_createcomplex(L, aux_cerf(agn_tocomplex(L, 1)));
      break;
    }
#else
    case LUA_TCOMPLEX: {  /* 0.32.4 */
      lua_Number re, im;
      aux_cerf(agn_complexreal(L, 1), agn_compleximag(L, 1), &re, &im);
      agn_createcomplex(L, re, im);
      break;
    }
#endif
    default:
      luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "erfc",
        lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


#define MAXLOG	+170.0

/* return exp(x*x) if sign is nonnegative, and exp(-x*x) if sign is negative
   based on file `expx2.c` in the Cephes Math Library Release 2.9:  June, 2000
   Copyright 2000 by Stephen L. Moshier
   0.32.1, 15.05.2010 */

static int mathB_expx2 (lua_State *L) {
  double x, sign, u, u1, m, f;
  switch (lua_type(L, 1)) {
    case LUA_TNUMBER: {
      x = fabs(agn_tonumber(L, 1));
      sign = agnL_checknumber(L, 2);
      if (sign < 0) x = -x;
      /* Represent x as an exact multiple of M (= 128) plus a residual.
        M is a power of 2 chosen so that exp(m * m) does not overflow
        or underflow and so that |x - m| is small. */
      m = .0078125 * floor(128.0 * x + 0.5);
      f = x - m;
      /* x^2 = m^2 + 2mf + f^2 */
      u = m*m;
      u1 = 2*m*f + f*f;
      if (sign < 0) {
        u = -u;
        u1 = -u1;
      }
      if ((u+u1) > MAXLOG)
        lua_pushnumber(L, HUGE_VAL);
      else  /* u is exact, u1 is small.  */
        lua_pushnumber(L, exp(u) * exp(u1));
      break;
    }
    case LUA_TCOMPLEX: {
      lua_Number a, b, t1, t2, t4, t5;
#ifndef PROPCMPLX
      agn_Complex z = agn_tocomplex(L, 1);
      a = creal(z);
      b = cimag(z);
#else
      a = agn_complexreal(L, 1); b = agn_compleximag(L, 1);
#endif
      sign = agnL_checknumber(L, 2);
      t1 = a*a;
      t2 = b*b;
      t4 = (sign < 0) ? exp(-t1+t2) : exp(t1-t2);
      t5 = a*b;
#ifndef PROPCMPLX
      agn_createcomplex(L, t4*cos(2.0*t5) - ((sign < 0) ? 1 : -1)*I*t4*sin(2.0*t5));
#else
      agn_createcomplex(L, t4*cos(2.0*t5), -((sign < 0) ? 1 : -1)*t4*sin(2.0*t5));
#endif
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": number or complex number expected, got %s.", "expx2",
        lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


#ifndef PROPCMPLX
/*
   BesselJY complex aux functions
   taken from the FORTRAN mcjyna.for file written by Shanjie Zhang and Jianming Jin,
   Computation of Special Functions, Copyright 1996 by John Wiley & Sons, Inc.,
   by kind permission of Jianming Jin
   Modifications: calculation of the derivaties has been removed.
*/

lua_Number envj(lua_Number n, lua_Number x) {
  return 0.5*log10(6.28*n)-n*log10(1.36*x/n);
}

/*     ===================================================
       Function: msta1
       Purpose:  Determine the starting point for backward
                 recurrence such that the magnitude of
                 Jn(x) at that point is about 10^(-mp)
       Input :   x     --- Argument of Jn(x)
                 mp    --- Value of magnitude
       Output:   msta1 --- Starting point
       =================================================== */

int msta1 (lua_Number x, lua_Number mp) {
  lua_Number a0, f, f0, f1;
  int it, n0, n1, nn;
  a0 = fabs(x);
  n0 = trunc(1.1*a0)+1;
  f0 = envj(n0, a0)-mp;
  n1 = n0+5;
  f1 = envj(n1, a0)-mp;
  for(it=1; it < 21; it++) {
    nn = n1 - (n1-n0)/(1.0 - f0/f1);
    f = envj(nn, a0)-mp;
    if (fabs(nn-n1) < 1) break;
    n0 = n1;
    f0 = f1;
    n1 = nn;
    f1 = f;
  }
  return nn;
}

/*     ===================================================
       Function: msta2
       Purpose: Determine the starting point for backward
                recurrence such that all Jn(x) has mp
                significant digits
       Input :  x  --- Argument of Jn(x)
                n  --- Order of Jn(x)
                mp --- Significant digit
       Output:  msta2 --- Starting point
       =================================================== */

int msta2 (lua_Number x, lua_Number n, lua_Number mp) {
  lua_Number a0, hmp, ejn, obj, f, f0, f1;
  int it, n0, n1, nn;
  a0 = fabs(x);
  hmp = 0.5*mp;
  ejn = envj(n, a0);
  if (ejn <= hmp) {
    obj = mp;
    n0 = trunc(1.1*a0);
  } else {
    obj = hmp + ejn;
    n0 = n;
  }
  f0 = envj(n0, a0)-obj;
  n1 = n0 + 5;
  f1 = envj(n1, a0)-obj;
  for (it=1; it < 21; it++) {
    nn = n1-(n1-n0)/(1.0-f0/f1);
    f = envj(nn, a0)-obj;
    if (fabs(nn-n1) < 1) break;
    n0 = n1;
    f0 = f1;
    n1 = nn;
    f1 = f;
  }
  return nn + 10;
}

/*     ===========================================================
       Purpose: Compute complex Bessel functions J0(z), J1(z)
                Y0(z), Y1(z), and their derivatives
       Input :  z --- Complex argument
       Output:  cbj0 --- J0(z)
                cbj1 --- J1(z)
                cby0 --- Y0(z)
                cby1 --- Y1(z)
       =========================================================== */

void cjy01 (agn_Complex z, agn_Complex *cbj0, agn_Complex *cbj1,
            agn_Complex *cby0, agn_Complex *cby1) {
  lua_Number a0, k0, rp2, w0, w1;
  agn_Complex cr, ci, cp, cp0, cp1, cq0, cq1, cs, ct1, ct2, cu, z1, z2;
  int k;
  rp2 = 2.0/PI;
  ci = 0.0+1.0*I;
  a0 = cabs(z);
  z2 = z*z;
  z1 = z;
  if (a0 == 0.0) {
    *cbj0 = 1.0+0.0*I;
    *cbj1 = 0.0+0.0*I;
    *cby0 = -HUGE_VAL;
    *cby1 = -HUGE_VAL;
    return;
  }
  if (creal(z) < 0.0) z1 = -z;
  if (a0 <= 12.0) {
    *cbj0 = 1.0+0.0*I;
    cr = 1.0+0.0*I;
    for (k=1; k < 41; k++) {
      cr = -0.25*cr*z2/(k*k);
      *cbj0 = *cbj0+cr;
      if (cabs(cr/(*cbj0)) < 1.0e-15) break;
    }
    *cbj1 = 1.0+0.0*I;
    cr = 1.0+0.0*I;
    for (k=1; k < 41; k++) {
      cr = -0.25*cr*z2/(k*(k+1.0));
      *cbj1 = *cbj1+cr;
      if (cabs(cr/(*cbj1)) < 1.0e-15) break;
    }
    *cbj1 = 0.5*z1*(*cbj1);
    w0 = 0.0;
    cr = 1.0+0.0*I;
    cs = 0.0+0.0*I;
    for (k=1; k < 41; k++) {
      w0 = w0+1.0/k;
      cr = -0.25*cr/(k*k)*z2;
      cp = cr*w0;
      cs = cs+cp;
      if (cabs(cp/cs) < 1.0e-15) break;
    }
    *cby0 = rp2*(clog(z1/2.0)+EULERGAMMA)*(*cbj0)-rp2*cs;
    w1 = 0.0;
    cr = 1.0+0.0*I;
    cs = 1.0+0.0*I;
    for (k=1; k < 41; k++) {
      w1 = w1 + 1.0/k;
      cr = -0.25*cr/(k*(k+1))*z2;
      cp = cr*(2.0*w1 + 1.0/(k+1.0));
      cs = cs + cp;
      if (cabs(cp/cs) < 1.0e-15) break;
    }
    *cby1 = rp2*((clog(z1/2.0)+EULERGAMMA)*(*cbj1)-1.0/z1-.25*z1*cs);
  } else {
    lua_Number a[12] = {-.703125e-01,.112152099609375e+00,
       -.5725014209747314e+00,.6074042001273483e+01,
       -.1100171402692467e+03,.3038090510922384e+04,
       -.1188384262567832e+06,.6252951493434797e+07,
       -.4259392165047669e+09,.3646840080706556e+11,
       -.3833534661393944e+13,.4854014686852901e+15};
    lua_Number b[12] = {.732421875e-01,-.2271080017089844e+00,
        .1727727502584457e+01,-.2438052969955606e+02,
        .5513358961220206e+03,-.1825775547429318e+05,
        .8328593040162893e+06,-.5006958953198893e+08,
        .3836255180230433e+10,-.3649010818849833e+12,
        .4218971570284096e+14,-.5827244631566907e+16};
    lua_Number a1[12] = {.1171875e+00,-.144195556640625e+00,
        .6765925884246826e+00,-.6883914268109947e+01,
        .1215978918765359e+03,-.3302272294480852e+04,
        .1276412726461746e+06,-.6656367718817688e+07,
        .4502786003050393e+09,-.3833857520742790e+11,
        .4011838599133198e+13,-.5060568503314727e+15};
    lua_Number b1[12] = {-.1025390625e+00,.2775764465332031e+00,
        -.1993531733751297e+01,.2724882731126854e+02,
        -.6038440767050702e+03,.1971837591223663e+05,
        -.8902978767070678e+06,.5310411010968522e+08,
        -.4043620325107754e+10,.3827011346598605e+12,
        -.4406481417852278e+14,.6065091351222699e+16};
    k0 = 12;
    if (a0 >= 35.0) k0 = 10;
    if (a0 >= 50.0) k0 = 8;
    ct1 = z1-0.25*PI;
    cp0 = 1.0+0.0*I;
    for (k=0; k < k0; k++)
      cp0 = cp0 + a[k]*cpow(z1, (-2*(k+1)));
    cq0 = -0.125/z1;
    for (k=0; k < k0; k++)
      cq0 = cq0 + b[k]*cpow(z1, (-2*(k+1)-1));
    cu = csqrt(rp2/z1);
    *cbj0 = cu * (cp0*ccos(ct1) - cq0*csin(ct1));
    *cby0 = cu * (cp0*csin(ct1) + cq0*ccos(ct1));
    ct2 = z1-0.75*PI;
    cp1 = 1.0+0.0*I;
    for (k=0; k < k0; k++)
      cp1 = cp1 + a1[k]*cpow(z1, (-2*(k+1)));
    cq1 = 0.375/z1;
    for (k=0; k < k0; k++)
      cq1 = cq1 + b1[k]*cpow(z1, (-2*(k+1)-1));
    *cbj1 = cu*(cp1*ccos(ct2) - cq1*csin(ct2));
    *cby1 = cu*(cp1*csin(ct2) + cq1*ccos(ct2));
  }
  if (creal(z) < 0.0) {
    if (cimag(z) < 0.0) *cby0 = (*cby0) - 2.0*ci*(*cbj0);
    if (cimag(z) > 0.0) *cby0 = (*cby0) + 2.0*ci*(*cbj0);
    if (cimag(z) < 0.0) *cby1 = -((*cby1) - 2.0*ci*(*cbj1));
    if (cimag(z) > 0.0) *cby1 = -((*cby1) + 2.0*ci*(*cbj1));
    *cbj1 = -(*cbj1);
  }
}

/*     =======================================================
       Purpose: Compute Bessel functions Jn(z), Yn(z) and
                their derivatives for a complex argument
       Input :  z --- Complex argument of Jn(z) and Yn(z)
                n --- Order of Jn(z) and Yn(z)
       Output:  cbj(n) --- Jn(z)
                cby(n) --- Yn(z)
                nm --- Highest order computed
       Routines called:
            (1) CJY01 to calculate J0(z), J1(z), Y0(z), Y1(z)
            (2) msta1 and msta2 to calculate the starting
                point for backward recurrence
       =======================================================

  modified 0.32.2 to correct problems with out-of-range indices

*/

void cjyna (int n, agn_Complex z,
            agn_Complex *ccbj, agn_Complex *ccby) {
  agn_Complex cg0, cg1, cbj0, cbj1, cj0, cj1, cjk, cby0, cby1, cf, cf1, cf2, cs,
              cp11, cp12, cp21, cp22, cyk, cylk, cyl1, cyl2, ch0, ch1, ch2;
  agn_Complex cbj[n+2], cby[n+2];
  /* one field more than needed to prevent out-of-range index access */
  lua_Number a0, ya0, ya1, yak;
  int k, lb, lb0, m, nm;
  lb0 = 0;
  a0 = cabs(z);
  nm = n;
  cf = ch0 = 0+0*I;  /* to avoid inexplicable compiler warnings under OpenSUSE 10.3 */
  if (a0 < 1.0e-100) {
    for (k=0; k <= n; k++) {
      cbj[k] = 0.0 + 0.0*I;
      cby[k] = -HUGE_VAL;
    }
    cbj[0] = 1.0 + 0.0*I;
    goto l1;
  } else {  /* better sure than sorry, 0.32.2 */
    for (k=0; k < n+2; k++) {
      cbj[k] = AGN_NAN;
      cby[k] = AGN_NAN;
    }
  }
  cjy01(z, &cbj0, &cbj1, &cby0, &cby1);
  cbj[0] = cbj0;
  cby[0] = cby0;
  if (n != 0) {  /* 0.32.2 fix */
    cbj[1] = cbj1;
    cby[1] = cby1;
  }
  if (n <= 1) goto l1;
  if (n < trunc(0.25*a0)) {
    cj0 = cbj0;
    cj1 = cbj1;
    for (k=2; k <= n; k++) {
      cjk = 2.0*(k-1.0)/z*cj1 - cj0;
      cbj[k] = cjk;
      cj0 = cj1;
      cj1 = cjk;
    }
  } else {
    m = msta1(a0, 200);
    if (m < n)
      nm = m;
    else
      m = msta2(a0, n, 15);
    cf2 = 0.0 + 0.0*I;
    cf1 = 1.0e-100 + 0.0*I;
    for (k=m; k >= 0; k--) {
      cf = 2.0*(k+1.0)/z*cf1 - cf2;
      if (k <= nm) cbj[k] = cf;
      cf2 = cf1;
      cf1 = cf;
    }
    if (cabs(cbj0) > cabs(cbj1))
      cs = cbj0/cf;
    else
      cs = cbj1/cf2;
    for (k=0; k <= nm; k++) {
      cbj[k] = cs*cbj[k];
    }
  }
  ya0 = cabs(cby0);
  lb = 0;
  cg0 = cby0;
  cg1 = cby1;

  for (k=2; k <= nm; k++) {
    cyk = 2.0*(k-1.0)/z*cg1 - cg0;
    if (cabs(cyk) > 1.0e+290) continue;
    yak = cabs(cyk);
    ya1 = cabs(cg0);
    if ((yak < ya0) && (yak < ya1)) lb = k;
    cby[k] = cyk;
    cg0 = cg1;
    cg1 = cyk;
  }
  if ((lb <= 4) || (cimag(z) == 0.0)) goto l1;

  while (lb != lb0) {
    ch2 = 1.0+0.0*I;
    ch1 = 0.0+0.0*I;
    lb0 = lb;
    for (k=lb; k > 0; k--) {
      ch0 = 2.0*k/z*ch1-ch2;
      ch2 = ch1;
      ch1 = ch0;
    }
    cp12 = ch0;
    cp22 = ch2;
    ch2 = 0.0+0.0*I;
    ch1 = 1.0+0.0*I;
    for (k=lb; k > 0; k--) {
      ch0 = 2.0*k/z*ch1-ch2;
      ch2 = ch1;
      ch1 = ch0;
    }
    cp11 = ch0;
    cp21 = ch2;
    if (lb == nm) cbj[lb+1] = 2.0*lb/z*cbj[lb]-cbj[lb-1];
    if (cabs(cbj[0]) > cabs(cbj[1])) {
      cby[lb+1] = (cbj[lb+1]*cby0-2.0*cp11/(PI*z))/cbj[0];
      cby[lb] = (cbj[lb]*cby0+2.0*cp12/(PI*z))/cbj[0];
    } else {
      cby[lb+1] = (cbj[lb+1]*cby1-2.0*cp21/(PI*z))/cbj[1];
      cby[lb] = (cbj[lb]*cby1+2.0*cp22/(PI*z))/cbj[1];
    }
    cyl2 = cby[lb+1];
    cyl1 = cby[lb];
    for (k=lb-1; k >= 0; k--) {
      cylk = 2.0*(k+1.0)/z*cyl1-cyl2;
      cby[k] = cylk;
      cyl2 = cyl1;
      cyl1 = cylk;
    }
    cyl1 = cby[lb];
    cyl2 = cby[lb+1];
    for (k=lb+1; k <= nm; k++) {
      cylk = 2.0*k/z*cyl2-cyl1;
      cby[k+1] = cylk;
      cyl1 = cyl2;
      cyl2 = cylk;
    }
    for (k=2; k <= nm; k++) {
      if (cabs(cby[k]) < cabs(cby[k-1])) lb = k;
    }

  }

l1:
  *ccbj = cbj[n];
  *ccby = cby[n];
}
#endif


static int mathB_besselj (lua_State *L) {
  switch (lua_type(L, 2)) {
    case LUA_TNUMBER: {
      lua_pushnumber(L, jn(agn_checknumber(L, 1), agn_checknumber(L, 2)));
      break;
    }
#ifndef PROPCMPLX
    case LUA_TCOMPLEX: {
      agn_Complex cbj, cby;
      int n = agn_checknumber(L, 1);
      if (n < 0) luaL_error(L, "Error in " LUA_QS ": order must be nonnegative.", "besselj");
      cjyna(n, agn_tocomplex(L, 2), &cbj, &cby);
      agn_createcomplex(L, cbj);
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "besselj",
        lua_typename(L, lua_type(L, 2)));
#else
    default:
      luaL_error(L, "Error in " LUA_QS ": number expected, got %s.", "besselj",
        lua_typename(L, lua_type(L, 2)));
#endif
  }
  return 1;
}


static int mathB_bessely (lua_State *L) {
  switch (lua_type(L, 2)) {
    case LUA_TNUMBER: {
      lua_Number x = agn_checknumber(L, 2);
      if (x <= 0)  /* 0.32.2 */
        lua_pushundefined(L);
      else
        lua_pushnumber(L, yn(agn_checknumber(L, 1), x));
      break;
    }
#ifndef PROPCMPLX
    case LUA_TCOMPLEX: {
      agn_Complex cbj, cby;
      int n = agn_checknumber(L, 1);
      if (n < 0) luaL_error(L, "Error in " LUA_QS ": order must be nonnegative.", "bessely");
      cjyna(n, agn_tocomplex(L, 2), &cbj, &cby);
      agn_createcomplex(L, cby);
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "bessely",
        lua_typename(L, lua_type(L, 2)));
#else
    default:
      luaL_error(L, "Error in " LUA_QS ": number expected, got %s.", "bessely",
        lua_typename(L, lua_type(L, 2)));
#endif
  }
  return 1;
}


static int mathB_fma (lua_State *L) {  /* 0.29.0 */
#ifdef fma
  lua_pushnumber(L, fma(agn_checknumber(L, 1), agn_checknumber(L, 2), agn_checknumber(L, 3)));
#else
  /* missing in OpenSUSE; including <tgmath.h> does not work */
  lua_pushnumber(L, agn_checknumber(L, 1) * agn_checknumber(L, 2) + agn_checknumber(L, 3));
#endif
  return 1;
}


static int mathB_gamma (lua_State *L) {  /* 0.31.2, extended 0.33.2 */
  if (lua_type(L, 1) == LUA_TNUMBER)
    lua_pushnumber(L, cephes_gamma(agn_tonumber(L, 1)));
#ifndef PROPCMPLX
  else if (lua_type(L, 1) == LUA_TCOMPLEX)
    agn_createcomplex(L, cephes_cgamma(agn_tocomplex(L, 1)));
  else
    luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "gamma",
      lua_typename(L, lua_type(L, 1)));
#else
  else
    luaL_error(L, "Error in " LUA_QS ": number expected, got %s.", "gamma",
      lua_typename(L, lua_type(L, 1)));
#endif
  return 1;
}


/* simplified version of Donald Knuth's approximation algorithm published
   in `Seminumerical Algorithms` by checking whether the _relative_ error is
   bound to a given tolerance; patched 0.26.1, 10.08.2009; patched 0.26.4, 20.08.2009;
   extended 0.32.1, 17.05.2010 */
static int mathB_approx (lua_State *L) {
  lua_Number a1, a2, b1, b2, eps, dist;
  int ttype1, ttype2;
#ifndef PROPCMPLX
  agn_Complex ca, cb;
#endif
  if (lua_gettop(L) == 2) {
    lua_getglobal(L, "Eps");
    if (lua_type(L, -1) == LUA_TNUMBER) {
      eps = agn_tonumber(L, -1);
      agn_poptop(L);
    } else {
      eps = 0;  /* to avoid compiler warnings */
      agn_poptop(L);  /* pop invalid value */
      luaL_error(L,
        "Error in " LUA_QS ": no accuracy given and " LUA_QS " not defined or not a number.",
        "approx", "Eps");
    }
  } else
    eps = agn_checknumber(L, 3);
  ttype1 = lua_type(L, 1); ttype2 = lua_type(L, 2);
  if (ttype1 == LUA_TNUMBER && ttype2 == LUA_TNUMBER) {
    a1 = agn_tonumber(L, 1);
    b1 = agn_tonumber(L, 2);
    if ((a1 == b1) || (isnan(a1) && isnan(b1))) {  /* Agena 1.6.0 */
      lua_pushtrue(L);
    } else {
      dist = fabs(a1 - b1);
      lua_pushboolean(L, dist <= eps || dist <= (eps * fMax(fabs(a1), fabs(b1))));  /* 0.27.0 patch. Do NOT use fmax ! (see agncmpt.h) */
    }
    return 1;
  } else if (ttype1 == LUA_TCOMPLEX && ttype2 == LUA_TNUMBER) {
#ifndef PROPCMPLX
    ca = agn_tocomplex(L, 1);
    a1 = creal(ca); a2 = agn_tonumber(L, 2); b1 = cimag(ca); b2 = 0;
#else
    a1 = agn_complexreal(L, 1); b1 = agn_compleximag(L, 1);
    a2 = agn_tonumber(L, 2); b2 = 0;
#endif
  } else if (ttype1 == LUA_TNUMBER && ttype2 == LUA_TCOMPLEX) {
#ifndef PROPCMPLX
    cb = agn_tocomplex(L, 2);
    a1 = agn_tonumber(L, 1); a2 = creal(cb); b1 = 0; b2 = cimag(cb);
#else
    a1 = agn_tonumber(L, 1); b1 = 0;
    a2 = agn_complexreal(L, 2); b2 = agn_compleximag(L, 2);
#endif
  } else if (ttype1 == LUA_TCOMPLEX && ttype2 == LUA_TCOMPLEX) {
#ifndef PROPCMPLX
    ca = agn_tocomplex(L, 1);
    cb = agn_tocomplex(L, 2);
    a1 = creal(ca); a2 = creal(cb); b1 = cimag(ca); b2 = cimag(cb);
#else
    a1 = agn_complexreal(L, 1); b1 = agn_compleximag(L, 1);
    a2 = agn_complexreal(L, 2); b2 = agn_compleximag(L, 2);
#endif
  } else {
    a1 = b1 = a2 = b2 = 0;  /* just to avoid compiler warnings */
    luaL_error(L, "Error in " LUA_QS ": numbers or complex values expected.", "approx");
  }
  dist = sqrt(pow(a2-a1, 2) + pow(b2-b1, 2));
  lua_pushboolean(L,
    dist < eps || (dist <= (eps * fMax(fabs(a1), fabs(b1))) && dist <= (eps * fMax(fabs(a2), fabs(b2)))) || (isnan(a1) && isnan(a2)));  /* 1.9.3 */
  return 1;
}


static int agnB_put (lua_State *L) {  /* formerly tinsert */
  int nops = lua_gettop(L);
  switch(lua_type(L, 1)) {
    case LUA_TTABLE: {
      int e = luaL_getn(L, 1) + 1;  /* first empty element */
      int pos;  /* where to insert new element */
      switch (nops) {
        case 2: {  /* called with only 2 arguments */
          pos = e;  /* insert new element at the end */
          break;
        }
        case 3: {
          int i;
          pos = agnL_checkint(L, 2);  /* 2nd argument is the position */
          if (pos > e) e = pos;  /* `grow' array if necessary */
          for (i = e; i > pos; i--) {  /* move up elements */
            lua_rawgeti(L, 1, i-1);
            lua_rawseti(L, 1, i);  /* t[i] = t[i-1] */
          }
          break;
        }
        default: {
          return luaL_error(L, "wrong number of arguments to " LUA_QL("put") ", two or three required.");
        }
      }
      luaL_setn(L, 1, e);  /* new size */
      lua_rawseti(L, 1, pos);  /* t[pos] = v */
      break;
    }
    case LUA_TSEQ: {  /* 0.31.5 */
      int e = agn_seqsize(L, 1) + 1;  /* first empty element */
      int pos;  /* where to insert new element */
      switch (nops) {
        case 2: {  /* called with only 2 arguments */
          pos = e;  /* insert new element at the end */
          break;
        }
        case 3: {
          int i;
          pos = agnL_checkint(L, 2);  /* 2nd argument is the position */
          if (pos > e) e = pos;  /* `grow' array if necessary */
          for (i = e; i > pos; i--) {  /* move up elements */
            lua_seqgeti(L, 1, i-1);   /* get old value */
            lua_seqseti(L, 1, i);
          }
          break;
        }
        default: {
          return luaL_error(L, "Error in " LUA_QS ": wrong number of arguments, two or three required.", "put");
        }
      }
      lua_seqseti(L, 1, pos);
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": table or sequence expected as first argument, got %s.", "put",
        lua_typename(L, lua_type(L, 1)));
  }
  return 0;
}


static int agnB_purge (lua_State *L) {  /* based on tables.remove */
  switch(lua_type(L, 1)) {
    case LUA_TTABLE: {
      int e = luaL_getn(L, 1);
      int pos = agnL_optinteger(L, 2, e);
      if (!(1 <= pos && pos <= e))  /* 5.1.3 patch; position is outside bounds? */
        return 0;  /* nothing to remove */
      luaL_setn(L, 1, e - 1);  /* t.n = n-1 */
      lua_rawgeti(L, 1, pos);  /* result = t[pos] */
      for ( ;pos<e; pos++) {
        lua_rawgeti(L, 1, pos+1);
        lua_rawseti(L, 1, pos);  /* t[pos] = t[pos+1] */
      }
      lua_pushnil(L);
      lua_rawseti(L, 1, e);  /* t[e] = nil */
      return 1;
    }
    case LUA_TSEQ: {  /* 0.31.5 */
      int e = agn_seqsize(L, 1);
      int pos = agnL_optinteger(L, 2, e);
      if (!(1 <= pos && pos <= e))  /* 5.1.3 patch; position is outside bounds? */
        return 0;  /* nothing to remove */
      lua_seqgeti(L, 1, pos);  /* result = t[pos]; */
      lua_pushnil(L);
      lua_seqseti(L, 1, pos);  /* t[e] = null, moves all elements at the right to the left */
      return 1;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": table or sequence expected as first argument, got %s.", "purge",
        lua_typename(L, lua_type(L, 1)));
  }
  return 0;
}


#define isaux(L, type, procname) { \
  int nargs, i; \
  nargs = lua_gettop(L); \
  if (nargs == 0) { \
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", procname); \
  } \
  for (i=1; i <= nargs; i++) { \
    if (lua_type(L, i) != type) { \
      lua_pushfalse(L); \
      return 1; \
    } \
  } \
  lua_pushtrue(L); \
  return 1; \
}


static int agnB_isboolean (lua_State *L) {  /* extended 0.34.2 */
  isaux(L, LUA_TBOOLEAN, "isboolean");
}


static int agnB_isnumber (lua_State *L) {  /* extended 0.34.2 */
  isaux(L, LUA_TNUMBER, "isnumber");
}


static int agnB_iscomplex (lua_State *L) {  /* extended 0.34.2 */
  isaux(L, LUA_TCOMPLEX, "iscomplex");
}


static int agnB_isnumeric (lua_State *L) {  /* extended 0.34.2 */
  int nargs, i, t;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "isnumeric");
  }
  for (i=1; i <= nargs; i++) {
    t = lua_type(L, i);
    if (t != LUA_TNUMBER && t != LUA_TCOMPLEX) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int agnB_isstring (lua_State *L) {  /* extended 0.34.2 */
  isaux(L, LUA_TSTRING, "isstring");
}


static int agnB_istable (lua_State *L) {  /* extended 0.34.2 */
  isaux(L, LUA_TTABLE, "istable");
}


static int agnB_isset (lua_State *L) {  /* extended 0.34.2 */
  isaux(L, LUA_TSET, "isset");
}


static int agnB_isseq (lua_State *L) {  /* extended 0.34.2 */
  isaux(L, LUA_TSEQ, "isseq");
}


static int agnB_ispair (lua_State *L) {  /* extended 0.34.2 */
  isaux(L, LUA_TPAIR, "ispair");
}


static int agnB_isstructure (lua_State *L) {  /* extended 0.34.2 */
  int nargs, i;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "isstructure");
  }
  for (i=1; i <= nargs; i++) {
    if (lua_type(L, i) < LUA_TTABLE) {
      lua_pushfalse(L);
      return 1;
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int agnB_isint (lua_State *L) {  /* 1.6.0 */
  int nargs, i;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "isint");
  }
  for (i=1; i <= nargs; i++) {
    if (lua_type(L, i) != LUA_TNUMBER) {
      lua_pushfail(L);
      return 1;
    }
    else {
      lua_Number n;
      n = agn_tonumber(L, i);
      if (ISFLOAT(n)) {
        lua_pushfalse(L);
        return 1;
      }
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int agnB_isposint (lua_State *L) {  /* extended 0.34.2 */
  int nargs, i;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "isposint");
  }
  for (i=1; i <= nargs; i++) {
    if (lua_type(L, i) != LUA_TNUMBER) {
      lua_pushfail(L);
      return 1;
    }
    else {
      lua_Number n;
      n = agn_tonumber(L, i);
      if (ISFLOAT(n) || n < 1) {
        lua_pushfalse(L);
        return 1;
      }
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int agnB_notisposint (lua_State *L) {  /* 1.9.1, extended 1.11.3 */
  int nargs, i;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "notisposint");
  }
  for (i=1; i <= nargs; i++) {
    if (lua_type(L, i) != LUA_TNUMBER) {
      lua_pushfail(L);
      return 1;
    }
    else {
      lua_Number n;
      n = agn_tonumber(L, i);
      if (ISFLOAT(n) || n < 1) {
        lua_pushtrue(L);
        return 1;
      }
    }
  }
  lua_pushfalse(L);
  return 1;
}


static int agnB_ispositive (lua_State *L) {  /* 1.11.3 */
  int nargs, i;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "ispositive");
  }
  for (i=1; i <= nargs; i++) {
    if (lua_type(L, i) != LUA_TNUMBER) {
      lua_pushfail(L);
      return 1;
    }
    else {
      lua_Number n;
      n = agn_tonumber(L, i);
      if (n <= 0) {
        lua_pushfalse(L);
        return 1;
      }
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int agnB_isnegative (lua_State *L) {  /* 1.11.3 */
  int nargs, i;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "isnegative");
  }
  for (i=1; i <= nargs; i++) {
    if (lua_type(L, i) != LUA_TNUMBER) {
      lua_pushfail(L);
      return 1;
    }
    else {
      lua_Number n;
      n = agn_tonumber(L, i);
      if (n >= 0) {
        lua_pushfalse(L);
        return 1;
      }
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int agnB_isnonneg (lua_State *L) {  /* 1.11.3 */
  int nargs, i;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "isnonneg");
  }
  for (i=1; i <= nargs; i++) {
    if (lua_type(L, i) != LUA_TNUMBER) {
      lua_pushfail(L);
      return 1;
    }
    else {
      lua_Number n;
      n = agn_tonumber(L, i);
      if (n < 0) {
        lua_pushfalse(L);
        return 1;
      }
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int agnB_notisnegint (lua_State *L) {  /* 1.11.3 */
  int nargs, i;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "notisnegint");
  }
  for (i=1; i <= nargs; i++) {
    if (lua_type(L, i) != LUA_TNUMBER) {
      lua_pushfail(L);
      return 1;
    }
    else {
      lua_Number n;
      n = agn_tonumber(L, i);
      if (ISFLOAT(n) || n > -1) {
        lua_pushtrue(L);
        return 1;
      }
    }
  }
  lua_pushfalse(L);
  return 1;
}


static int agnB_isnonnegint (lua_State *L) {  /* extended 0.34.2 */
  int nargs, i;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "isnonnegint");
  }
  for (i=1; i <= nargs; i++) {
    if (lua_type(L, i) != LUA_TNUMBER) {
      lua_pushfail(L);
      return 1;
    }
    else {
      lua_Number n;
      n = agn_tonumber(L, i);
      if (ISFLOAT(n) || n < 0) {
        lua_pushfalse(L);
        return 1;
      }
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int agnB_isnonposint (lua_State *L) {  /* 1.1.0 */
  int nargs, i;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "isnonposint");
  }
  for (i=1; i <= nargs; i++) {
    if (lua_type(L, i) != LUA_TNUMBER) {
      lua_pushfail(L);
      return 1;
    }
    else {
      lua_Number n;
      n = agn_tonumber(L, i);
      if (ISFLOAT(n) || n > 0) {
        lua_pushfalse(L);
        return 1;
      }
    }
  }
  lua_pushtrue(L);
  return 1;
}


static int agnB_isnegint (lua_State *L) {  /* extended 0.34.2 */
  int nargs, i;
  nargs = lua_gettop(L);
  if (nargs == 0) {
    luaL_error(L, "Error in " LUA_QS ": need at least one argument.", "isnegint");
  }
  for (i=1; i <= nargs; i++) {
    if (lua_type(L, i) != LUA_TNUMBER) {
      lua_pushfail(L);
      return 1;
    }
    else {
      lua_Number n;
      n = agn_tonumber(L, i);
      if (ISFLOAT(n) || n > -1) {
        lua_pushfalse(L);
        return 1;
      }
    }
  }
  lua_pushtrue(L);
  return 1;
}

# define castx(x, isinteger) ((isinteger) ? (LUA_INTEGER)((x)) : (LUAI_UINT32)((x)))

static int agnB_setbit (lua_State *L) {  /* 1.6.5 */
  int flag;
  lu_byte isinteger;
  lua_Number x, pos;
  isinteger = agn_getbitwise(L);
  if (lua_gettop(L) != 3)
    luaL_error(L, "Error in " LUA_QS ": need three arguments.", "setbit");
  x = agn_checknumber(L, 1);
  if (ISFLOAT(x))
    luaL_error(L, "Error in " LUA_QS ": first argument must be an integer.", "setbit");
  pos = agn_checknumber(L, 2);
  if (pos < 1 || pos > 31)
    luaL_error(L, "Error in " LUA_QS ": second argument must be positive and less than 32.", "setbit");
  else
    pos = trunc(pos);
  if (lua_isboolean(L, 3))
    flag = lua_toboolean(L, 3);
  else if (lua_isnumber(L, 3)) {
    flag = lua_tointeger(L, 3);
    if (flag != 0 && flag != 1)
      luaL_error(L, "Error in " LUA_QS ": third argument must be either Boolean, or 0 or 1.", "setbit");
  } else {
    flag = 0;  /* tp prevent comnpiler warnings */
    luaL_error(L, "Error in " LUA_QS ": third argument must be either Boolean, or 0 or 1.", "setbit");
  }
  if (flag) {  /* t := (abs(x) ^^ ((-(abs(flag = true)) ^^ abs(x)) && 2**(pos-1)));
                  return t * is x <> 0 then sign(x) else 1 si */
    if (x >= 0)
      lua_pushinteger(L, castx(x, isinteger) | 1 << (castx(pos, isinteger) - 1));
    else
      lua_pushinteger(L, -(-castx(x, isinteger) | 1 << (castx(pos, isinteger) - 1)));
  } else {
    if (x >= 0)
      lua_pushinteger(L, castx(x, isinteger) & ~(1 << (castx(pos, isinteger) - 1)));
    else
      lua_pushinteger(L, -(-castx(x, isinteger) & ~(1 << (castx(pos, isinteger) - 1))));
  }
  return 1;
}


static int agnB_getbit (lua_State *L) {  /* 1.6.5 */
  lu_byte isinteger;
  lua_Number x, pos;
  isinteger = agn_getbitwise(L);
  if (lua_gettop(L) != 2)
    luaL_error(L, "Error in " LUA_QS ": need two arguments.", "getbit");
  x = agn_checknumber(L, 1);
  if (ISFLOAT(x))
    luaL_error(L, "Error in " LUA_QS ": first argument must be an integer.", "getbit");
  pos = agn_checknumber(L, 2);
  if (pos < 1 || pos > 31)
    luaL_error(L, "Error in " LUA_QS ": second argument must be positive and less than 32.", "getbit");
  else
    pos = trunc(pos);
  if (x >= 0) /*  x && (1 shift (pos-1)) <> 0 */
    lua_pushboolean(L, (castx(x, isinteger) & (1 << (castx(pos, isinteger) - 1))) != 0);
  else
    lua_pushboolean(L, (castx(-x, isinteger) & (1 << (castx(pos, isinteger) - 1))) != 0);
  return 1;
}


/* returns the indices for a given value; December 28, 2007; extended July 06, 2008 */

static int agnB_whereis (lua_State *L) {  /* C implementation: Agena 1.8.2, 06.10.2012 */
  size_t i, ii, type;
  ii = 0;
  type = lua_type(L, 1);
  luaL_typecheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence expected", type);
  switch (type) {
    case LUA_TTABLE: {
      lua_createtable(L, agn_size(L, 1), 0);
      lua_pushnil(L);
      while (lua_next(L, 1) != 0) {
        if (lua_equal(L, -1, 2)) {
          ii++;
          agn_poptop(L);
          lua_pushvalue(L, -1);  /* push key */
          lua_rawseti(L, -3, ii);  /* assign it to table and drop it */
        } else
          agn_poptop(L);
      }
      break;
    }
    case LUA_TSEQ: {
      size_t nops;
      nops = agn_seqsize(L, 1);
      agn_createseq(L, nops);
      for (i=1; i <= nops; i++) {
        lua_seqgeti(L, 1, i);
        if (lua_equal(L, -1, 2)) {
          ii++;
          lua_seqsetinumber(L, -2, ii, i);
        }
        agn_poptop(L);
      }
      break;
    }
    default: lua_assert(0);  /* should not happen */
  }
  return 1;
}


static int agnB_checkoptions (lua_State *L) {  /* Agena 1.8.6, 18.8.12 */
  size_t i, nargs;
  int vtype, ctype, throwunknown;
  const char *procname, *etype;
  throwunknown = 1;  /* 1.8.12 */
  nargs = lua_gettop(L);
  procname = agn_checkstring(L, 1);
  if (nargs < 3)
    luaL_error(L, "Error in " LUA_QS ": at least three arguments expected.", procname);
  vtype = lua_type(L, 2);
  luaL_argcheck(L, vtype == LUA_TTABLE, 2, "table of pairs expected");
  lua_createtable(L, 0, 0);
  lua_createtable(L, 0, nargs-2);  /* temporary table */
  for (i=3; i <= nargs; i++) {
    if (i == nargs && lua_isboolean(L, nargs)) {  /* 1.8.12 */
      if (agn_istrue(L, nargs))  /* 1.8.12 */
        throwunknown = 0;
    }
    else if (!lua_ispair(L, i))
      luaL_error(L, "Error in " LUA_QS ": pair expected, got %s.", procname, lua_typename(L, lua_type(L, i)));
    else { /* value is a pair */
      agn_pairgeti(L, i, 1);
      agn_pairgeti(L, i, 2);
      if (lua_type(L, -1) != LUA_TSTRING)
        luaL_error(L, "Error in " LUA_QS ": string expected for right operand, got %s.", procname, lua_typename(L, lua_type(L, -1)));
      else
        lua_settable(L, -3);  /* enter expected type into temporary table */
    }
  }
  lua_pushnil(L);
  while (lua_next(L, 2)) {
    if (lua_ispair(L, -1)) {
      agn_pairgeti(L, -1, 1);  /* push option name */
      if (!agn_isstring(L, -1))
        luaL_error(L, "Error in " LUA_QS ": left-hand side of option must be a string, got %s.", procname, lua_typename(L, lua_type(L, -1)));
      agn_pairgeti(L, -2, 2);  /* push value */
      lua_pushvalue(L, -2);    /* push option name again */
      lua_gettable(L, -6);     /* get expected type from temporary table */
      if (lua_isnil(L, -1)) {  /* 1.8.12 */
        if (throwunknown)
          luaL_error(L, "Error in " LUA_QS ": unknown option %s.", procname, agn_checkstring(L, -3));
        else {
          agn_poptop(L);  /* remove null */
          lua_settable(L, -6);  /* pops option name and value, put `unknown` option:value into result table */
          agn_poptop(L);  /* pop pair, leave key on stack */
          continue;
        }
      }
      ctype = lua_type(L, -2);  /* current type to be checked */
      etype = agn_checkstring(L, -1);  /* expected type */
      if (strcmp(etype, lua_typename(L, ctype)) != 0) {  /* type mismatch ? */
        if (ctype > LUA_TTHREAD || ctype == LUA_TFUNCTION) {  /* check for user-defined type */
          if (agn_getutype(L, -2) == 1) {  /* a user-defined type has been set ? */
            if (strcmp(agn_tostring(L, -1), etype) != 0) {  /* utype mismatch ? */
              luaL_error(L, "Error in " LUA_QS ": %s expected for %s option, got %s.", procname,
                etype, agn_checkstring(L, -4), agn_tostring(L, -1));
            }
            agn_poptop(L);  /* drop utype */
          }
          else
            luaL_error(L, "Error in " LUA_QS ": %s expected for %s option, got %s.", procname,
              etype, agn_checkstring(L, -3), agn_tostring(L, -1));
        } else
          luaL_error(L, "Error in " LUA_QS ": %s expected for %s option, got %s.", procname,
            agn_checkstring(L, -1),
            agn_checkstring(L, -3),
            lua_typename(L, lua_type(L, -2))
          );
      }
      agn_poptop(L);  /* remove expected type */
      lua_settable(L, -6);  /* set option and actual value to result table */
    }
    agn_poptop(L);  /* pop pair or other value */
  }
  agn_poptop(L);  /* pop temp table */
  return 1;
}


static int agnB_alternate (lua_State *L) {
  if (lua_gettop(L) != 2)
    luaL_error(L, "Error in " LUA_QS ": expected two arguments.", "alternate");
  lua_pushvalue(L, 2-lua_isnil(L, 2));
  return 1;
}


/* base-2 logarithm of a numeric or complex argument, 1.9.3 */
static int agnB_log2 (lua_State *L) {
  int ntype = lua_type(L, 1);
  switch (ntype) {
    case LUA_TNUMBER: {
      lua_Number x = agn_tonumber(L, 1);
      if (x > 0)
        lua_pushnumber(L, log2(x));
      else
        lua_pushundefined(L);
      break;
    }
    case LUA_TCOMPLEX: {
#ifndef PROPCMPLX
      agn_Complex z = agn_tocomplex(L, 1);
      if (z == 0+I*0)
        agn_createcomplex(L, AGN_NAN);
      else
        agn_createcomplex(L, clog(z)/log(2.0));
#else
      lua_Number a, b;
      a = agn_complexreal(L, 1);
      b = agn_compleximag(L, 1);
      if (a == 0 && b == 0)
        agn_createcomplex(L, AGN_NAN, 0);
      else
        agn_createcomplex(L, log(sqrt(a*a+b*b))/log(2.0), atan2(b, a)/log(2.0));
#endif
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "log2",
        lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


/* base-10 logarithm of a numeric or complex argument, 1.9.3 */
static int agnB_log10 (lua_State *L) {
  int ntype = lua_type(L, 1);
  switch (ntype) {
    case LUA_TNUMBER: {
      lua_Number x = agn_tonumber(L, 1);
      if (x > 0)
        lua_pushnumber(L, log10(x));
      else
        lua_pushundefined(L);
      break;
    }
    case LUA_TCOMPLEX: {
#ifndef PROPCMPLX
      agn_Complex z = agn_tocomplex(L, 1);
      if (z == 0+I*0)
        agn_createcomplex(L, AGN_NAN);
      else
        agn_createcomplex(L, clog(z)/log(10.0));  /* clog10 is a GNU extension, so do not use it */
#else
      lua_Number a, b;
      a = agn_complexreal(L, 1);
      b = agn_compleximag(L, 1);
      if (a == 0 && b == 0)
        agn_createcomplex(L, AGN_NAN, 0);
      else
        agn_createcomplex(L, log(sqrt(a*a+b*b))/log(10.0), atan2(b, a)/log(10.0));
#endif
      break;
    }
    default:
      luaL_error(L, "Error in " LUA_QS ": number or complex value expected, got %s.", "log10",
        lua_typename(L, lua_type(L, 1)));
  }
  return 1;
}


/* transforms the complex number z in Cartesian notation or the number z to polar co-ordinates. If z is a number and is zero,
   or if z is complex and its real and imaginary parts equal zero, it returns zero twice. April 16, 2013 */

static int mathB_polar (lua_State *L) {
  lua_Number x, y;
#ifndef PROPCMPLX
  agn_Complex z;
#endif
  x = 0; y = 0;  /* to avoid compiler warnings */
  switch (lua_type(L, 1)) {
    case LUA_TNUMBER:
      x = agn_tonumber(L, 1);
      y = 0;
      break;
    case LUA_TCOMPLEX:
#ifndef PROPCMPLX
      z = agn_tocomplex(L, 1);
      x = creal(z); y = cimag(z);
#else
      x = agn_complexreal(L, 1); y = agn_compleximag(L, 1);
#endif
      break;
    default:
      luaL_error(L, "Error in " LUA_QS ": number or complex number expected, got %s.", "polar",
        lua_typename(L, lua_type(L, 1)));
  }
  lua_pushnumber(L, hypot(x, y));
  lua_pushnumber(L, atan2(y, x));
  return 2;
}


static const luaL_Reg base_funcs[] = {
  {"alternate", agnB_alternate},           /* added January 24, 2013 */
  {"assume", luaB_assume},
  {"cbrt", agnB_cbrt},                     /* added August 25, 2013 */
  {"checkoptions", agnB_checkoptions},     /* added October 29, 2012 */
  {"countitems", agnB_countitems},         /* added January 18, 2010 */
  {"dimension", agnB_dimension},           /* added January 01, 2010 */
  {"drem", agnB_drem},                     /* added December 01, 2013 */  
  {"error", luaB_error},
  {"getbit", agnB_getbit},                 /* added June 10, 2012 */
  {"getentry", agnB_getentry},             /* added January 13, 2010 */
  {"getmetatable", luaB_getmetatable},
  {"gettype", luaB_gettype},               /* added May 05, 2008 */
  {"has", agnB_has},                       /* added June 30, 2012 */
  {"iqr", mathB_iqr},                      /* added April 10, 2013 */
  {"isboolean", agnB_isboolean},           /* added on June 24, 2010 */
  {"iscomplex", agnB_iscomplex},           /* added on June 24, 2010 */
  {"isint", agnB_isint},                   /* added on April 01, 2010 */
  {"isnegative", agnB_isnegative},         /* added May 12, 2013 */
  {"isnegint", agnB_isnegint},             /* added on June 24, 2010 */
  {"isnonneg", agnB_isnonneg},             /* added May 12, 2013 */
  {"isnonnegint", agnB_isnonnegint},       /* added on June 24, 2010 */
  {"isnonposint", agnB_isnonposint},       /* added December 13, 2010 */
  {"isnumber", agnB_isnumber},             /* added on June 24, 2010 */
  {"isnumeric", agnB_isnumeric},           /* added on June 24, 2010 */
  {"ispair", agnB_ispair},                 /* added on June 24, 2010 */
  {"isposint", agnB_isposint},             /* added on June 24, 2010 */
  {"ispositive", agnB_ispositive},         /* added May 12, 2013 */
  {"isstring", agnB_isstring},             /* added on June 24, 2010 */
  {"istable", agnB_istable},               /* added on June 24, 2010 */
  {"isset", agnB_isset},                   /* added on June 24, 2010 */
  {"isseq", agnB_isseq},                   /* added on June 24, 2010 */
  {"isstructure", agnB_isstructure},       /* added on June 24, 2010 */
  {"load", luaB_load},
  {"loadfile", luaB_loadfile},
  {"loadstring", luaB_loadstring},
  {"map", agnB_map},                       /* added April 07, 2007 */
  {"max", luaB_max},
  {"min", luaB_min},
  {"mprint", luaB_mprint},
  {"next", luaB_next},
  {"notisnegint", agnB_notisnegint},       /* added May 12, 2013 */
  {"notisposint", agnB_notisposint},       /* added February 04, 2013 */
  {"nseq", luaB_nseq},
  {"ops", luaB_ops},
  {"polar", mathB_polar},                  /* added April 16, 2013 */
  {"print", luaB_print},
  {"protect", luaB_protect},               /* added November 07, 2010 */
  {"purge", agnB_purge},
  {"put", agnB_put},
  {"rawequal", luaB_rawequal},
  {"rawget", luaB_rawget},
  {"rawset", luaB_rawset},
  {"readlib", agn_readlib},                /* added December 10, 2006 */
  {"recurse", agnB_recurse},               /* added July 08, 2012 */
  {"remove", tbl_remove},                  /* added March 30, 2008 */
  {"run", luaB_run},
  {"select", luaB_select},
  {"selectremove", luaB_selectremove},     /* November 09, 2012 */
  {"setbit", agnB_setbit},                 /* added June 10, 2012 */
  {"setmetatable", luaB_setmetatable},
  {"settype", luaB_settype},               /* added May 05, 2008 */
  {"sort", luaB_sort},
  {"sorted", luaB_sorted},                 /* added November 17, 2012 */
  {"subs", agnB_subs},                     /* added November 01, 2009 */
  {"time", agn_time},
  {"tonumber", luaB_tonumber},
  {"toseq", agnB_toseq},                   /* added June 14, 2008 */
  {"toset", agnB_toset},                   /* added October 13, 2009 */
  {"tostring", luaB_tostring},
  {"totable", agnB_totable},               /* added June 08, 2007 */
  {"unpack", luaB_unpack},
  {"whereis", agnB_whereis},               /* added October 06, 2012 */
  {"xpcall", luaB_xpcall},
  {"zip", agnB_zip},                       /* added September 04, 2008 */
  /* mathematical functions */
  {"approx", mathB_approx},
  {"arccoth", mathB_arccoth},              /* added on June 29, 2008 */
  {"arctan2", mathB_arctan2},
  {"argument", mathB_argument},            /* added on June 29, 2008 */
  {"besselj", mathB_besselj},              /* added on August 22, 2009 */
  {"bessely", mathB_bessely},              /* added on August 22, 2009 */
  {"binomial", mathB_binomial},            /* added on December 26, 2007 */
  {"conjugate", mathB_conjugate},          /* added on June 29, 2008 */
  {"erf", mathB_erf},                      /* added on August 22, 2009 */
  {"erfc", mathB_erfc},                    /* added on August 22, 2009 */
  {"expx2", mathB_expx2},                  /* added May 15, 2010 */
  {"fma", mathB_fma},                      /* added on November 23, 2009 */
  {"frexp", mathB_frexp},
  {"gamma", mathB_gamma},                  /* added June 21, 2010 */
  {"hypot", mathB_hypot},                  /* added on June 19, 2007 */
  {"irem", mathB_irem},                    /* added on June 19, 2007 */
  {"ldexp", mathB_ldexp},
  {"log2", agnB_log2},                     /* added on February 16, 2013 */
  {"log10", agnB_log10},                   /* added on February 18, 2013 */
  {"modf", mathB_modf},
  {"root", mathB_root},                    /* added on June 29, 2008 */
  {"proot", mathB_proot},                  /* added on August 28, 2013 */
  {"roundf", mathB_roundf},                /* added on December 03, 2006 */
  /* {"test", luaB_test}, */
  {NULL, NULL}
};


/*
** {======================================================
** Coroutine library
** =======================================================
*/

static int auxresume (lua_State *L, lua_State *co, int narg) {
  int status;
  if (!lua_checkstack(co, narg))
    luaL_error(L, "too many arguments to resume.");
  if (lua_status(co) == 0 && lua_gettop(co) == 0) {
    lua_pushliteral(L, "cannot resume dead coroutine");
    return -1;  /* error flag */
  }
  lua_xmove(L, co, narg);
  status = lua_resume(co, narg);
  if (status == 0 || status == LUA_YIELD) {
    int nres = lua_gettop(co);
    /* if (!lua_checkstack(L, nres)) 5.1.4 patch 2 */
    if (!lua_checkstack(L, nres + 1))
      luaL_error(L, "too many results to resume.");
    lua_xmove(co, L, nres);  /* move yielded values */
    return nres;
  }
  else {
    lua_xmove(co, L, 1);  /* move error message */
    return -1;  /* error flag */
  }
}


static int luaB_coresume (lua_State *L) {
  lua_State *co = lua_tothread(L, 1);
  int r;
  luaL_argcheck(L, co, 1, "coroutine expected");
  r = auxresume(L, co, lua_gettop(L) - 1);
  if (r < 0) {
    lua_pushfalse(L);
    lua_insert(L, -2);
    return 2;  /* return false + error message */
  }
  else {
    lua_pushtrue(L);
    lua_insert(L, -(r + 1));
    return r + 1;  /* return true + `resume' returns */
  }
}


static int luaB_auxwrap (lua_State *L) {
  lua_State *co = lua_tothread(L, lua_upvalueindex(1));
  int r = auxresume(L, co, lua_gettop(L));
  if (r < 0) {
    if (agn_isstring(L, -1)) {  /* error object is a string? */
      luaL_where(L, 1);  /* add extra info */
      lua_insert(L, -2);
      lua_concat(L, 2);
    }
    lua_error(L);  /* propagate error */
  }
  return r;
}


static int luaB_cocreate (lua_State *L) {
  lua_State *NL = lua_newthread(L);
  luaL_argcheck(L, lua_isfunction(L, 1) && !lua_iscfunction(L, 1), 1,
    "Agena function expected");
  lua_pushvalue(L, 1);  /* move function to top */
  lua_xmove(L, NL, 1);  /* move function from L to NL */
  return 1;
}


static int luaB_cowrap (lua_State *L) {
  luaB_cocreate(L);
  lua_pushcclosure(L, luaB_auxwrap, 1);
  return 1;
}


static int luaB_yield (lua_State *L) {
  return lua_yield(L, lua_gettop(L));
}


static int luaB_costatus (lua_State *L) {
  lua_State *co = lua_tothread(L, 1);
  luaL_argcheck(L, co, 1, "coroutine expected");
  if (L == co) lua_pushliteral(L, "running");
  else {
    switch (lua_status(co)) {
      case LUA_YIELD:
        lua_pushliteral(L, "suspended");
        break;
      case 0: {
        lua_Debug ar;
        if (lua_getstack(co, 0, &ar) > 0)  /* does it have frames? */
          lua_pushliteral(L, "normal");  /* it is running */
        else if (lua_gettop(co) == 0)
            lua_pushliteral(L, "dead");
        else
          lua_pushliteral(L, "suspended");  /* initial state */
        break;
      }
      default:  /* some error occured */
        lua_pushliteral(L, "dead");
        break;
    }
  }
  return 1;
}


static int luaB_corunning (lua_State *L) {
  if (lua_pushthread(L))
    return 0;  /* main thread is not a coroutine */
  else
    return 1;
}


static const luaL_Reg co_funcs[] = {
  {"setup", luaB_cocreate},
  {"resume", luaB_coresume},
  {"running", luaB_corunning},
  {"status", luaB_costatus},
  {"wrap", luaB_cowrap},
  {"yield", luaB_yield},
  {NULL, NULL}
};

/* }====================================================== */


static void base_open (lua_State *L) {
  /* set global _G */
  lua_pushvalue(L, LUA_GLOBALSINDEX);
  lua_setglobal(L, "_G");
  /* define Pi, 2*Pi, PI/2, and EXP */
  lua_pushnumber(L, PI);
  lua_setglobal(L, "Pi");
  lua_pushnumber(L, PI2);
  lua_setglobal(L, "Pi2");
  lua_pushnumber(L, PIO2);
  lua_setglobal(L, "PiO2");
  lua_pushnumber(L, PIO4);
  lua_setglobal(L, "PiO4");
  lua_pushnumber(L, _E);
  lua_setglobal(L, "Exp");
  lua_pushnumber(L, _E);
  lua_setglobal(L, "E");
  lua_pushnumber(L, EULERGAMMA);
  lua_setglobal(L, "EulerGamma");
  lua_pushnumber(L, AGN_EPSILON);
  lua_setglobal(L, "Eps");
  lua_pushnumber(L, HUGE_VAL);
  lua_setglobal(L, "infinity");
  lua_pushnumber(L, AGN_NAN);
  lua_setglobal(L, "undefined");
  lua_pushnumber(L, RADIANS_PER_DEGREE);
  lua_setglobal(L, "radians");
  lua_pushnumber(L, 1/RADIANS_PER_DEGREE);
  lua_setglobal(L, "degrees");
  /* open lib into global table */
  luaL_register(L, "_G", base_funcs);
  lua_pushliteral(L, AGENA_RELEASE);  /* added */
  lua_setglobal(L, "_RELEASE");  /* set global _RELEASE */
  /* `newproxy' needs a weaktable as upvalue */
  lua_createtable(L, 0, 1);  /* new table `w' */
  lua_pushvalue(L, -1);  /* `w' will be its own metatable */
  lua_setmetatable(L, -2);
  lua_pushliteral(L, "kv");
  lua_setfield(L, -2, "__weak");  /* metatable(w).__mode = "kv" */
  lua_pushcclosure(L, luaB_newproxy, 1);
  lua_setglobal(L, "newproxy");  /* set global `newproxy' */
}


LUALIB_API int luaopen_base (lua_State *L) {
  base_open(L);
  luaL_register(L, LUA_COLIBNAME, co_funcs);
  return 2;
}


