/*
** $Id: lapi.c,v 2.55 2006/06/07 12:37:17 roberto Exp $
** Lua API
** See Copyright Notice in agena.h
*/

#include <assert.h>
#include <math.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>  /* for isdigit */

#define lapi_c
#define LUA_CORE

#include "agena.h"

#include "agnxlib.h"
#include "lcomplex.h"
#include "ldebug.h"
#include "ldo.h"
#include "lfunc.h"
#include "lgc.h"
#include "lmem.h"
#include "lobject.h"
#include "lpair.h"
#include "lseq.h"
#include "lset.h"
#include "lstate.h"
#include "lstring.h"
#include "ltable.h"
#include "ltm.h"
#include "lundump.h"
#include "lvm.h"

const char lua_ident[] =
  "$Agena: " AGENA_RELEASE " " AGENA_COPYRIGHT " $\n"
  "$Authors: " LUA_AUTHORS " $\n"
  "$URL: agena.sourceforge.net $\n";

#define api_checknelems(L, n)   api_check(L, (n) <= (L->top - L->base))

#define api_checkvalidindex(L, i)   api_check(L, (i) != luaO_nilobject)

#define api_incr_top(L)   {api_check(L, L->top < L->ci->top); L->top++;}

#define api_incr_top_by_three(L)   {api_check(L, L->top + 1 < L->ci->top); L->top += 3;}

#define api_incr_top_by_two(L)   {api_check(L, L->top + 1 < L->ci->top); L->top += 2;}

/* static TValue *index2adr (lua_State *L, int idx) { */
TValue *index2adr (lua_State *L, int idx) {
  if (idx > 0) {
    TValue *o = L->base + (idx - 1);
    api_check(L, idx <= L->ci->top - L->base);
    if (o >= L->top) return cast(TValue *, luaO_nilobject);
    else return o;
  }
  else if (idx > LUA_REGISTRYINDEX) {
    api_check(L, idx != 0 && -idx <= L->top - L->base);
    return L->top + idx;
  }
  else switch (idx) {  /* pseudo-indices */
    case LUA_REGISTRYINDEX: return registry(L);
    case LUA_ENVIRONINDEX: {
      Closure *func = curr_func(L);
      sethvalue(L, &L->env, func->c.env);
      return &L->env;
    }
    case LUA_GLOBALSINDEX: return gt(L);
    default: {
      Closure *func = curr_func(L);
      idx = LUA_GLOBALSINDEX - idx;
      return (idx <= func->c.nupvalues)
                ? &func->c.upvalue[idx-1]
                : cast(TValue *, luaO_nilobject);
    }
  }
}


static Table *getcurrenv (lua_State *L) {
  if (L->ci == L->base_ci)  /* no enclosing function? */
    return hvalue(gt(L));  /* use global table as environment */
  else {
    Closure *func = curr_func(L);
    return func->c.env;
  }
}


void luaA_pushobject (lua_State *L, const TValue *o) {
  setobj2s(L, L->top, o);
  api_incr_top(L);
}


LUA_API int lua_checkstack (lua_State *L, int size) {
  /* 5.1.4 patch 3 */
  int res = 1;
  lua_lock(L);
  if (size > LUAI_MAXCSTACK || (L->top - L->base + size) > LUAI_MAXCSTACK)
    res = 0;  /* stack overflow */
  else if (size > 0) {
    luaD_checkstack(L, size);
    if (L->ci->top < L->top + size)
      L->ci->top = L->top + size;
  }
  lua_unlock(L);
  return res;
}


LUA_API void lua_xmove (lua_State *from, lua_State *to, int n) {
  int i;
  if (from == to) return;
  lua_lock(to);
  api_checknelems(from, n);
  api_check(from, G(from) == G(to));
  api_check(from, to->ci->top - to->top >= n);
  from->top -= n;
  for (i = 0; i < n; i++) {
    setobj2s(to, to->top++, from->top + i);
  }
  lua_unlock(to);
}


LUA_API lua_CFunction lua_atpanic (lua_State *L, lua_CFunction panicf) {
  lua_CFunction old;
  lua_lock(L);
  old = G(L)->panic;
  G(L)->panic = panicf;
  lua_unlock(L);
  return old;
}


LUA_API lua_State *lua_newthread (lua_State *L) {
  lua_State *L1;
  lua_lock(L);
  luaC_checkGC(L);
  L1 = luaE_newthread(L);
  setthvalue(L, L->top, L1);
  api_incr_top(L);
  lua_unlock(L);
  luai_userstatethread(L, L1);
  return L1;
}


/*
** basic stack manipulation
*/


LUA_API int lua_gettop (lua_State *L) {
  return cast_int(L->top - L->base);
}


LUA_API void lua_settop (lua_State *L, int idx) {
  lua_lock(L);
  if (idx >= 0) {
    api_check(L, idx <= L->stack_last - L->base);
    while (L->top < L->base + idx)
      setnilvalue(L->top++);
    L->top = L->base + idx;
  }
  else {
    api_check(L, -(idx+1) <= (L->top - L->base));
    L->top += idx+1;  /* `subtract' index (index is negative) */
  }
  lua_unlock(L);
}


LUA_API void lua_remove (lua_State *L, int idx) {
  StkId p;
  lua_lock(L);
  p = index2adr(L, idx);
  api_checkvalidindex(L, p);
  while (++p < L->top) setobjs2s(L, p-1, p);
  L->top--;
  lua_unlock(L);
}


LUA_API void lua_insert (lua_State *L, int idx) {
  StkId p;
  StkId q;
  lua_lock(L);
  p = index2adr(L, idx);
  api_checkvalidindex(L, p);
  for (q = L->top; q>p; q--) setobjs2s(L, q, q-1);
  setobjs2s(L, p, L->top);
  lua_unlock(L);
}


LUA_API void lua_replace (lua_State *L, int idx) {
  StkId o;
  lua_lock(L);
  /* explicit test for incompatible code */
  if (idx == LUA_ENVIRONINDEX && L->ci == L->base_ci)
    luaG_runerror(L, "no calling environment");
  api_checknelems(L, 1);
  o = index2adr(L, idx);
  api_checkvalidindex(L, o);
  if (idx == LUA_ENVIRONINDEX) {
    Closure *func = curr_func(L);
    api_check(L, ttistable(L->top - 1));
    func->c.env = hvalue(L->top - 1);
    luaC_barrier(L, func, L->top - 1);
  }
  else {
    setobj(L, o, L->top - 1);
    if (idx < LUA_GLOBALSINDEX)  /* function upvalue? */
      luaC_barrier(L, curr_func(L), L->top - 1);
  }
  L->top--;
  lua_unlock(L);
}


LUA_API void lua_pushvalue (lua_State *L, int idx) {
  lua_lock(L);
  setobj2s(L, L->top, index2adr(L, idx));
  api_incr_top(L);
  lua_unlock(L);
}



/*
** access functions (stack -> C)
*/


LUA_API int lua_type (lua_State *L, int idx) {
  StkId o = index2adr(L, idx);
  return (o == luaO_nilobject) ? LUA_TNONE : ttype(o);
}


LUA_API const char *lua_typename (lua_State *L, int t) {
  UNUSED(L);
  return (t == LUA_TNONE) ? "no value" : luaT_typenames[t];
}


LUA_API int lua_iscfunction (lua_State *L, int idx) {
  StkId o = index2adr(L, idx);
  return iscfunction(o);
}


LUA_API int lua_isnumber (lua_State *L, int idx) {
  TValue n;
  const TValue *o = index2adr(L, idx);
  return tonumber(o, &n);
}


LUA_API int lua_isstring (lua_State *L, int idx) {
  int t = lua_type(L, idx);
  return (t == LUA_TSTRING || t == LUA_TNUMBER);
}


LUA_API int agn_isnumber (lua_State *L, int idx) {  /* Agena 1.4.3/1.5.0 */
  return lua_type(L, idx) == LUA_TNUMBER;  /* changed 0.5.3 */
}


LUA_API int agn_isstring (lua_State *L, int idx) {  /* Agena 1.4.3/1.5.0 */
  return lua_type(L, idx) == LUA_TSTRING;  /* changed 0.5.3 */
}


LUA_API int lua_isuserdata (lua_State *L, int idx) {
  const TValue *o = index2adr(L, idx);
  return (ttisuserdata(o) || ttislightuserdata(o));
}


LUA_API int lua_rawequal (lua_State *L, int index1, int index2) {
  StkId o1 = index2adr(L, index1);
  StkId o2 = index2adr(L, index2);
  return (o1 == luaO_nilobject || o2 == luaO_nilobject) ? 0
         : luaO_rawequalObj(o1, o2);
}


LUA_API int lua_equal (lua_State *L, int index1, int index2) {
  StkId o1, o2;
  int i;
  lua_lock(L);  /* may call tag method */
  o1 = index2adr(L, index1);
  o2 = index2adr(L, index2);
  i = (o1 == luaO_nilobject || o2 == luaO_nilobject) ? 0 : equalobj(L, o1, o2);
  lua_unlock(L);
  return i;
}


LUA_API int lua_lessthan (lua_State *L, int index1, int index2) {
  StkId o1, o2;
  int i;
  lua_lock(L);  /* may call tag method */
  o1 = index2adr(L, index1);
  o2 = index2adr(L, index2);
  i = (o1 == luaO_nilobject || o2 == luaO_nilobject) ? 0
       : luaV_lessthan(L, o1, o2);
  lua_unlock(L);
  return i;
}


LUA_API lua_Number lua_tonumber (lua_State *L, int idx) {
  TValue n;
  const TValue *o = index2adr(L, idx);
  if (tonumber(o, &n))
    return nvalue(o);
  else
    return 0;
}


LUA_API lua_Number agn_tonumber (lua_State *L, int idx) {
  const TValue *o = index2adr(L, idx);
  return nvalue(o);
}


/* do not delete this function since it is absolutely needed by `toNumber` to convert strings to numbers
   (agn_tonumber does not do this) */
LUA_API lua_Number agn_tonumberx (lua_State *L, int idx, int *exception) {  /* added October 18, 2008 */
  TValue n;
  const TValue *o = index2adr(L, idx);
  if (tonumber(o, &n)) {
    *exception = 0;
    return nvalue(o);
  }
  else {  /* failure, could not convert value */
    *exception = 1;
    return 0;
  }
}

#ifndef PROPCMPLX
LUA_API agn_Complex agn_tocomplexx (lua_State *L, int idx, int *exception) {  /* added 10.08.2009, 0.26.1 */
  TValue n;
  const TValue *o = index2adr(L, idx);
  if (tocomplex(o, &n)) {
    *exception = 0;
    return cvalue(o);
  }
  else {  /* failure, could not convert value */
    *exception = 1;
    return 0;
  }
}

LUA_API agn_Complex agn_tocomplex (lua_State *L, int idx) {
  StkId o = index2adr(L, idx);
  return cvalue(o);
}

#else
LUA_API void agn_tocomplexx (lua_State *L, int idx, int *exception, lua_Number *a) {  /* added 10.08.2009, 0.26.1 */
  TValue n;
  const TValue *o = index2adr(L, idx);
  if (tocomplex(o, &n)) {
    *exception = 0;
    a[0] = cvalue(o)[0];
    a[1] = cvalue(o)[1];
  }
  else {  /* failure, could not convert value */
    a[0] = a[1] = 0;
    *exception = 1;
  }
}


LUA_API lua_Number agn_complexreal (lua_State *L, int idx) {
  StkId o = index2adr(L, idx);
  return cvalue(o)[0];
}

LUA_API lua_Number agn_compleximag (lua_State *L, int idx) {
  StkId o = index2adr(L, idx);
  return cvalue(o)[1];
}
#endif

LUA_API lua_Integer lua_tointeger (lua_State *L, int idx) {
  TValue n;
  const TValue *o = index2adr(L, idx);
  if (tonumber(o, &n)) {
    lua_Integer res;
    lua_Number num = nvalue(o);
    lua_number2integer(res, num);
    return res;
  }
  else
    return 0;
}


LUA_API int32_t lua_toint32_t (lua_State *L, int idx) {
  TValue n;
  const TValue *o = index2adr(L, idx);
  if (tonumber(o, &n)) {
    int32_t res;
    lua_Number num = nvalue(o);
    res = (int32_t)num;
    return res;
  }
  else
    return 0;
}


LUA_API off64_t lua_tooff64_t (lua_State *L, int idx) {
  TValue n;
  const TValue *o = index2adr(L, idx);
  if (tonumber(o, &n)) {
    off64_t res;
    lua_Number num = nvalue(o);
    res = (off64_t)num;
    return res;
  }
  else
    return 0;
}


LUA_API int lua_toboolean (lua_State *L, int idx) {
  const TValue *o = index2adr(L, idx);
  if l_isfail(o)
    return LUA_TFAIL;  /* = -1 */
  else
    return !l_isfalse(o);
}


LUA_API const char *lua_tolstring (lua_State *L, int idx, size_t *len) {
  StkId o = index2adr(L, idx);
  if (!ttisstring(o)) {
    lua_lock(L);  /* `luaV_tostring' may create a new string */
    if (!luaV_tostring(L, o)) {  /* conversion failed? */
      if (len != NULL) *len = 0;
      lua_unlock(L);
      return NULL;
    }
    luaC_checkGC(L);
    o = index2adr(L, idx);  /* previous call may reallocate the stack */
    lua_unlock(L);
  }
  if (len != NULL) *len = tsvalue(o)->len;
  return svalue(o);
}


LUA_API const char *agn_tostring (lua_State *L, int idx) {  /* 0.24.0 */
  StkId o = index2adr(L, idx);
  return svalue(o);
}


LUA_API size_t lua_objlen (lua_State *L, int idx) {
  StkId o = index2adr(L, idx);
  switch (ttype(o)) {
    case LUA_TSTRING: return tsvalue(o)->len;
    case LUA_TUSERDATA: return uvalue(o)->len;
    case LUA_TTABLE: return luaH_getn(hvalue(o));
    case LUA_TNUMBER: {
      size_t l;
      lua_lock(L);  /* `luaV_tostring' may create a new string */
      l = (luaV_tostring(L, o) ? tsvalue(o)->len : 0);
      lua_unlock(L);
      return l;
    }
    default: return 0;
  }
}


LUA_API lua_CFunction lua_tocfunction (lua_State *L, int idx) {
  StkId o = index2adr(L, idx);
  return (!iscfunction(o)) ? NULL : clvalue(o)->c.f;
}


LUA_API void *lua_touserdata (lua_State *L, int idx) {
  StkId o = index2adr(L, idx);
  switch (ttype(o)) {
    case LUA_TUSERDATA: return (rawuvalue(o) + 1);
    case LUA_TLIGHTUSERDATA: return pvalue(o);
    default: return NULL;
  }
}


LUA_API lua_State *lua_tothread (lua_State *L, int idx) {
  StkId o = index2adr(L, idx);
  return (!ttisthread(o)) ? NULL : thvalue(o);
}


LUA_API const void *lua_topointer (lua_State *L, int idx) {
  StkId o = index2adr(L, idx);
  switch (ttype(o)) {
    case LUA_TTABLE: return hvalue(o);
    case LUA_TFUNCTION: return clvalue(o);
    case LUA_TTHREAD: return thvalue(o);
    case LUA_TUSERDATA:
    case LUA_TLIGHTUSERDATA:
      return lua_touserdata(L, idx);
    case LUA_TSET: return usvalue(o);  /* added 0.10.0 */
    case LUA_TSEQ: return seqvalue(o);  /* added 0.11.0 */
    case LUA_TPAIR: return pairvalue(o);  /* added 0.11.1 */
#ifdef LUA_ANSI
    case LUA_TCOMPLEX: return cvalue(o);  /* added 0.12.0 */
#endif
    default: return NULL;
  }
}



/*
** push functions (C -> stack)
*/


LUA_API void lua_pushnil (lua_State *L) {
  lua_lock(L);
  setnilvalue(L->top);
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void lua_pushnumber (lua_State *L, lua_Number n) {
  lua_lock(L);
  setnvalue(L->top, n);
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void lua_pushundefined (lua_State *L) {
  lua_lock(L);
  setnvalue(L->top, AGN_NAN);
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void lua_pushinteger (lua_State *L, lua_Integer n) {
  lua_lock(L);
  setnvalue(L->top, cast_num(n));
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void lua_pushlstring (lua_State *L, const char *s, size_t len) {
  lua_lock(L);
  luaC_checkGC(L);
  setsvalue2s(L, L->top, luaS_newlstr(L, s, len));
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void lua_pushstring (lua_State *L, const char *s) {
  if (s == NULL)
    lua_pushnil(L);
  else
    lua_pushlstring(L, s, strlen(s));
}


LUA_API const char *lua_pushvfstring (lua_State *L, const char *fmt,
                                      va_list argp) {
  const char *ret;
  lua_lock(L);
  luaC_checkGC(L);
  ret = luaO_pushvfstring(L, fmt, argp);
  lua_unlock(L);
  return ret;
}


LUA_API const char *lua_pushfstring (lua_State *L, const char *fmt, ...) {
  const char *ret;
  va_list argp;
  lua_lock(L);
  luaC_checkGC(L);
  va_start(argp, fmt);
  ret = luaO_pushvfstring(L, fmt, argp);
  va_end(argp);
  lua_unlock(L);
  return ret;
}


LUA_API void lua_pushcclosure (lua_State *L, lua_CFunction fn, int n) {
  Closure *cl;
  lua_lock(L);
  luaC_checkGC(L);
  api_checknelems(L, n);
  cl = luaF_newCclosure(L, n, getcurrenv(L));
  cl->c.f = fn;
  L->top -= n;
  while (n--)
    setobj2n(L, &cl->c.upvalue[n], L->top+n);
  setclvalue(L, L->top, cl);
  lua_assert(iswhite(obj2gco(cl)));
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void lua_pushboolean (lua_State *L, int b) {
  lua_lock(L);
  setbvalue(L->top, (b != 0));  /* ensure that true is 1 */
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void agn_pushboolean (lua_State *L, int b) {  /* 16.01.2013 */
  lua_lock(L);
  setbvalue(L->top, (b == -1) ? 2 : (b != 0));  /* ensure that true is 1 */
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void lua_pushlightuserdata (lua_State *L, void *p) {
  lua_lock(L);
  setpvalue(L->top, p);
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API int lua_pushthread (lua_State *L) {
  lua_lock(L);
  setthvalue(L, L->top, L);
  api_incr_top(L);
  lua_unlock(L);
  return (G(L)->mainthread == L);
}



/*
** get functions (Lua -> stack)
*/


LUA_API void lua_gettable (lua_State *L, int idx) {
  StkId t;
  lua_lock(L);
  t = index2adr(L, idx);
  api_checkvalidindex(L, t);
  luaV_gettable(L, t, L->top - 1, L->top - 1);
  lua_unlock(L);
}


LUA_API void lua_getfield (lua_State *L, int idx, const char *k) {
  StkId t;
  TValue key;
  lua_lock(L);
  t = index2adr(L, idx);
  api_checkvalidindex(L, t);
  setsvalue(L, &key, luaS_new(L, k));
  luaV_gettable(L, t, &key, L->top);
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void lua_rawget (lua_State *L, int idx) {
  StkId t;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttistable(t));
  setobj2s(L, L->top - 1, luaH_get(hvalue(t), L->top - 1));
  lua_unlock(L);
}


/* check whether the value at stack index idx exists in a set */

LUA_API void lua_srawget (lua_State *L, int idx) {  /* 0.10.0 */
  StkId t;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttisuset(t));
  setbvalue(L->top - 1, agnUS_get(usvalue(t), L->top - 1));
  lua_unlock(L);
}


LUA_API void lua_seqrawget (lua_State *L, int idx) {
  StkId t;
  const TValue *r;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttisseq(t));
  r = agnSeq_rawgeti(seqvalue(t), L->top - 1);
  if (r != NULL) {
    setobj2s(L, L->top - 1, r);
  }
  else
    luaG_runerror(L, "index out of range");
  lua_unlock(L);
}


LUA_API void lua_seqrawget2 (lua_State *L, int idx) {
  StkId t;
  const TValue *r;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttisseq(t));
  r = agnSeq_rawgeti(seqvalue(t), L->top - 1);
  if (r != NULL) {
    setobj2s(L, L->top - 1, r);
  } else {
    setobj2s(L, L->top - 1, luaO_nilobject);
  }
  lua_unlock(L);
}


/* agn_pairrawget: idx should be a stack value k with k=1 for the left hand side, or
   k=2 for the right hand side */

LUA_API void agn_pairrawget (lua_State *L, int idx) {
  StkId t;
  const TValue *r;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttispair(t));
  r = agnPair_rawgeti(pairvalue(t), L->top - 1);
  if (r != NULL) {
    setobj2s(L, L->top - 1, r);
  }
  else
    luaG_runerror(L, "index out of range");
  lua_unlock(L);
}


/* get a value at index n from a sequence which resides at stack pos */

LUA_API void lua_seqgeti (lua_State *L, int idx, size_t n) {  /* patched 0.31.6 */
  StkId t;
  const TValue *r;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttisseq(t));
  r = agnSeq_geti(seqvalue(t), n);
  if (r != NULL) {
    setobj2s(L, L->top, r);
    api_incr_top(L);
  }
  else
    luaG_runerror(L, "index out of range");
  lua_unlock(L);
}


LUA_API lua_Number lua_seqgetinumber (lua_State *L, int idx, int n) {  /* 0.12.0 */
  StkId t;
  const TValue *o;
  TValue p;
  t = index2adr(L, idx);
  api_check(L, ttisseq(t));
  o = agnSeq_geti(seqvalue(t), n);
  if (tonumber(o, &p))
    return nvalue(o);
  else
    return HUGE_VAL;
}


LUA_API void agn_createpair (lua_State *L, int idxleft, int idxright) {  /* 0.11.1; tuned 0.12.2 */
  StkId l, r;
  lua_lock(L);
  luaC_checkGC(L);
  l = index2adr(L, idxleft);
  r = index2adr(L, idxright);
  setpairvalue(L, L->top, agnPair_create(L, l, r));
  api_incr_top(L);
  lua_unlock(L);
}


#ifndef PROPCMPLX
LUA_API void agn_createcomplex (lua_State *L, agn_Complex c) {  /* 0.12.0, changed 0.22.1 */
  setcvalue(L->top, c);
  api_incr_top(L);
}
#else
LUA_API void agn_createcomplex (lua_State *L, lua_Number a, lua_Number b) {  /* 0.12.0 */
  lua_Number z[2];
  z[0] = a;
  z[1] = b;
  setcvalue(L->top, z);
  api_incr_top(L);
}
#endif


/* get left (n=1) or right (n=2) value in a pair, FIXME: make this consistent */

LUA_API void agn_pairgeti (lua_State *L, int idx, int n) {  /* 0.11.1 */
  StkId t;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttispair(t));
  setobj2s(L, L->top, agnPair_geti(pairvalue(t), n));
  api_incr_top(L);
  lua_unlock(L);
}


/* get real part of complex number */

LUA_API void agn_complexgetreal (lua_State *L, int idx) {  /* 0.12.0 */
  StkId t;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttiscomplex(t));
  setnvalue(L->top, complexreal(t));
  api_incr_top(L);
  lua_unlock(L);
}

/* get imaginary part of complex number */

LUA_API void agn_complexgetimag (lua_State *L, int idx) {  /* 0.12.0 */
  StkId t;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttiscomplex(t));
  setnvalue(L->top, compleximag(t));
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void lua_rawgeti (lua_State *L, int idx, int n) {
  StkId o;
  lua_lock(L);
  o = index2adr(L, idx);
  api_check(L, ttistable(o));
  setobj2s(L, L->top, luaH_getnum(hvalue(o), n));
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void lua_createtable (lua_State *L, int narray, int nrec) {
  lua_lock(L);
  luaC_checkGC(L);
  sethvalue(L, L->top, luaH_new(L, narray, nrec));
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void agn_createtable (lua_State *L, int narray, int nrec) {
  Table *newtable;
  lua_lock(L);
  luaC_checkGC(L);
  newtable = luaH_new(L, narray, nrec);  /* Agena 1.7.10, avoid trouble with `size` */
  newtable->hasnil = 1;
  sethvalue(L, L->top, newtable);
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void agn_createset (lua_State *L, int nrec) {  /* 0.10.0 */
  lua_lock(L);
  luaC_checkGC(L);
  setusvalue(L, L->top, agnUS_new(L, nrec));
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void agn_createseq (lua_State *L, int nrec) {  /* 0.11.0 */
  lua_lock(L);
  luaC_checkGC(L);
  setseqvalue(L, L->top, agnSeq_new(L, nrec));
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API int lua_getmetatable (lua_State *L, int objindex) {
  const TValue *obj;
  Table *mt = NULL;
  int res;
  lua_lock(L);
  obj = index2adr(L, objindex);
  switch (ttype(obj)) {
    case LUA_TTABLE:
      mt = hvalue(obj)->metatable;
      break;
    case LUA_TUSERDATA:
      mt = uvalue(obj)->metatable;
      break;
    case LUA_TSET:  /* 0.11.0 */
      mt = usvalue(obj)->metatable;
      break;
    case LUA_TSEQ:  /* 0.11.0 */
      mt = seqvalue(obj)->metatable;
      break;
    case LUA_TPAIR:  /* 0.11.0 */
      mt = pairvalue(obj)->metatable;
      break;
    default:
      mt = G(L)->mt[ttype(obj)];
      break;
  }
  if (mt == NULL)
    res = 0;
  else {
    sethvalue(L, L->top, mt);
    api_incr_top(L);
    res = 1;
  }
  lua_unlock(L);
  return res;
}


LUA_API void lua_getfenv (lua_State *L, int idx) {
  StkId o;
  lua_lock(L);
  o = index2adr(L, idx);
  api_checkvalidindex(L, o);
  switch (ttype(o)) {
    case LUA_TFUNCTION:
      sethvalue(L, L->top, clvalue(o)->c.env);
      break;
    case LUA_TUSERDATA:
      sethvalue(L, L->top, uvalue(o)->env);
      break;
    case LUA_TTHREAD:
      setobj2s(L, L->top,  gt(thvalue(o)));
      break;
    default:
      setnilvalue(L->top);
      break;
  }
  api_incr_top(L);
  lua_unlock(L);
}


/*
** set functions (stack -> Lua)
*/


LUA_API void lua_settable (lua_State *L, int idx) {
  StkId t;
  lua_lock(L);
  api_checknelems(L, 2);
  t = index2adr(L, idx);
  api_checkvalidindex(L, t);
  luaV_settable(L, t, L->top - 2, L->top - 1);
  L->top -= 2;  /* pop index and value */
  lua_unlock(L);
}


LUA_API void lua_setfield (lua_State *L, int idx, const char *k) {
  StkId t;
  TValue key;
  lua_lock(L);
  api_checknelems(L, 1);
  t = index2adr(L, idx);
  api_checkvalidindex(L, t);
  setsvalue(L, &key, luaS_new(L, k));
  luaV_settable(L, t, &key, L->top - 1);
  L->top--;  /* pop value */
  lua_unlock(L);
}


LUA_API void lua_rawset (lua_State *L, int idx) {
  StkId t;
  lua_lock(L);
  api_checknelems(L, 2);
  t = index2adr(L, idx);
  api_check(L, ttistable(t));
  setobj2t(L, luaH_set(L, hvalue(t), L->top-2), L->top-1);
  luaC_barriert(L, hvalue(t), L->top-1);
  /* never remove the following line [sic !], otherwise `size' will return wrong results in case of deletions */
  if (ttisnil(L->top-1)) hvalue(t)->hasnil = 1;
  L->top -= 2;
  lua_unlock(L);
}


LUA_API void lua_rawseti (lua_State *L, int idx, int n) {
  StkId o;
  lua_lock(L);
  api_checknelems(L, 1);
  o = index2adr(L, idx);
  api_check(L, ttistable(o));
  setobj2t(L, luaH_setnum(L, hvalue(o), n), L->top-1);
  luaC_barriert(L, hvalue(o), L->top-1);
  /* 1.6.9, never remove the following line [sic !], otherwise `size' will return wrong results in case of deletions */
  if (ttisnil(L->top-1)) hvalue(o)->hasnil = 1;
  L->top--;
  lua_unlock(L);
}


LUA_API void lua_srawset (lua_State *L, int idx) {
  StkId t;
  lua_lock(L);
  api_checknelems(L, 1);
  t = index2adr(L, idx);
  api_check(L, ttisuset(t));
  agnUS_set(L, usvalue(t), L->top-1);
  luaC_barrierset(L, usvalue(t), L->top-1);
  L->top--;
  lua_unlock(L);
}


LUA_API void lua_sdelete (lua_State *L, int idx) {  /* 0.10.0, leave this proc here */
  StkId t;
  lua_lock(L);
  api_checknelems(L, 1);
  t = index2adr(L, idx);
  api_check(L, ttisuset(t));
  agnUS_delete(L, usvalue(t), L->top - 1);
  L->top--;  /* pop value */
  lua_unlock(L);
}


LUA_API void agn_pairrawset (lua_State *L, int idx) {
  StkId t;
  int r;
  lua_lock(L);
  api_checknelems(L, 2);
  t = index2adr(L, idx);
  api_check(L, ttispair(t));
  r = agnPair_rawseti(L, pairvalue(t), L->top-2, L->top-1);
  if (r != 1)
    luaG_runerror(L, "error in rawset (agn_pairrawset)");
  luaC_barrierpair(L, pairvalue(t), L->top-1);
  L->top -= 2;
  lua_unlock(L);
}


LUA_API void lua_seqinsert (lua_State *L, int idx) {  /* 0.11.0 */
  StkId t;
  lua_lock(L);
  api_checknelems(L, 1);
  t = index2adr(L, idx);
  api_check(L, ttisseq(t));
  agnSeq_set(L, seqvalue(t), L->top-1);
  luaC_barrierseq(L, seqvalue(t), L->top-1);  /* 1.6.11 */
  L->top--;  /* pop value */
  lua_unlock(L);
}


LUA_API void lua_seqset (lua_State *L, int idx) {
  StkId t;
  int r;
  lua_lock(L);
  api_checknelems(L, 2);
  t = index2adr(L, idx);
  api_check(L, ttisseq(t));
  r = agnSeq_seti(L, seqvalue(t), (int)nvalue(L->top-2), L->top-1);
  if (r != 1)
    luaG_runerror(L, "error in rawset (lua_seqrawset)");
  luaC_barrierseq(L, seqvalue(t), L->top-1);
  L->top -= 2;
  lua_unlock(L);
}


LUA_API void lua_seqseti (lua_State *L, int idx, int n) {  /* 0.11.0 */
  StkId t;
  lua_lock(L);
  api_checknelems(L, 1);
  t = index2adr(L, idx);
  api_check(L, ttisseq(t));
  agnSeq_seti(L, seqvalue(t), n, L->top-1);
  luaC_barrierseq(L, seqvalue(t), L->top-1);
  L->top--;  /* pop inserted value */
  lua_unlock(L);
}


LUA_API void agn_setutype (lua_State *L, int idxobj, int idxtype) {  /* 0.11.1; extended 0.14.0, patched 0.28.0,
  patched 1.2.1 */
  StkId t, ty;
  TString *str;
  lua_lock(L);
  t = index2adr(L, idxobj);
  ty = index2adr(L, idxtype);
  if (ttisstring(ty))
    str = luaS_new(L, svalue(ty));
  else if (ttisnil(ty))
    str = NULL;
  else {
    luaG_runerror(L, "string or null expected.");
    return;
  }
  switch(ttype(t)) {
    case LUA_TSEQ:
      seqvalue(t)->type = str;
      if (str != NULL) luaC_objbarrierseq(L, seqvalue(t), str);
      break;
    case LUA_TPAIR:
      pairvalue(t)->type = str;
      if (str != NULL) luaC_objbarrierpair(L, pairvalue(t), str);
      break;
    case LUA_TTABLE:
      hvalue(t)->type = str;
      if (str != NULL) luaC_objbarriert(L, hvalue(t), str);
      break;
    case LUA_TFUNCTION: {
      Closure *fcl = clvalue(t);
      if (fcl->c.isC)
        fcl->c.type = str;
      else
        fcl->l.type = str;
      if (str != NULL) luaC_objbarrier(L, clvalue(t), str);
      break;
    }
    case LUA_TSET:
      usvalue(t)->type = str;
      if (str != NULL) luaC_objbarrierset(L, usvalue(t), str);
      break;
    default:
      luaG_runerror(L, "procedure, table, sequence, set, or pair expected.");
  }
  lua_unlock(L);
}


LUA_API int agn_getutype (lua_State *L, int idx) {  /* 0.11.1; extended 0.14.0, modified 0.28.0 */
  StkId t;
  TString *r;
  lua_lock(L);
  t = index2adr(L, idx);
  if (ttisseq(t))
    r = seqvalue(t)->type;
  else if (ttispair(t))
    r = pairvalue(t)->type;
  else if (ttistable(t))
    r = hvalue(t)->type;
  else if (ttisuset(t))
    r = usvalue(t)->type;
  else {  /* assume idx is a procedure */
    Closure *fcl = clvalue(t);
    if (fcl->c.isC && fcl->c.type != NULL)
      r = fcl->c.type;
    else if (!fcl->c.isC && fcl->l.type != NULL)
      r = fcl->l.type;
    else
      r = NULL;
  }
  if (r == NULL) {
    lua_unlock(L);
    return 0;
  }
  else {
    setsvalue(L, L->top, r);
    api_incr_top(L);
  }
  lua_unlock(L);
  return 1;
}


LUA_API int agn_isutype (lua_State *L, int idx, const char *str) {  /* based on agn_getutype, 1.9.1 */
  int result;
  StkId t;
  TString *r;
  lua_lock(L);
  t = index2adr(L, idx);
  if (ttisseq(t))
    r = seqvalue(t)->type;
  else if (ttispair(t))
    r = pairvalue(t)->type;
  else if (ttistable(t))
    r = hvalue(t)->type;
  else if (ttisuset(t))
    r = usvalue(t)->type;
  else {  /* assume idx is a procedure */
    Closure *fcl = clvalue(t);
    if (fcl->c.isC && fcl->c.type != NULL)
      r = fcl->c.type;
    else if (!fcl->c.isC && fcl->l.type != NULL)
      r = fcl->l.type;
    else
      r = NULL;
  }
  if (r == NULL)
    result = 0;
  else
    result = (strcmp(getstr(r), str) == 0);
  lua_unlock(L);
  return result;
}


LUA_API int agn_isutypeset (lua_State *L, int idx) {  /* 0.12.2; extended 0.14.0, modified 0.28.0 */
  StkId t;
  TString *r;
  lua_lock(L);
  t = index2adr(L, idx);
  if (ttisseq(t))
    r = seqvalue(t)->type;
  else if (ttispair(t))
    r = pairvalue(t)->type;
  else if (ttistable(t))
    r = hvalue(t)->type;
  else if (ttisuset(t))  /* 1.9.1 */
    r = usvalue(t)->type;
  else if (ttisfunction(t)) {
    Closure *fcl = clvalue(t);
    if (fcl->c.isC && fcl->c.type != NULL)
      r = fcl->c.type;
    else if (!fcl->c.isC && fcl->l.type != NULL)
      r = fcl->l.type;
    else
      r = NULL;
  }
  else if (ttisuset(t))
    r = usvalue(t)->type;
  else
    r = NULL;
  lua_unlock(L);
  return (r != NULL);
}


LUA_API int agn_issequtype (lua_State *L, int idx, const char *str) {  /* 0.12.1, modified 0.28.0 */
  StkId t;
  TString *r;
  lua_lock(L);
  t = index2adr(L, idx);
  if (ttisseq(t))
    r = seqvalue(t)->type;
  else
    r = NULL;
  lua_unlock(L);
  if (r == NULL)
    return 0;
  else
    return (strcmp(getstr(r), str) == 0);
}


LUA_API int agn_istableutype (lua_State *L, int idx, const char *str) {  /* 0.12.2, modified 0.28.0 */
  StkId t;
  TString *r;
  lua_lock(L);
  t = index2adr(L, idx);
  if (ttistable(t))
    r = hvalue(t)->type;
  else
    r = NULL;
  lua_unlock(L);
  if (r == NULL)
    return 0;
  else
    return (strcmp(getstr(r), str) == 0);
}


LUA_API int agn_issetutype (lua_State *L, int idx, const char *str) {  /* 0.14.0, modified 0.28.0 */
  StkId t;
  TString *r;
  lua_lock(L);
  t = index2adr(L, idx);
  if (ttisuset(t))
    r = usvalue(t)->type;
  else
    r = NULL;
  lua_unlock(L);
  if (r == NULL)
    return 0;
  else
    return (strcmp(getstr(r), str) == 0);
}


LUA_API int lua_setmetatable (lua_State *L, int objindex) {
  TValue *obj;
  Table *mt;
  lua_lock(L);
  api_checknelems(L, 1);
  obj = index2adr(L, objindex);
  api_checkvalidindex(L, obj);
  if (ttisnil(L->top - 1))
    mt = NULL;
  else {
    api_check(L, ttistable(L->top - 1));
    mt = hvalue(L->top - 1);
  }
  switch (ttype(obj)) {
    case LUA_TTABLE: {
      hvalue(obj)->metatable = mt;
      if (mt)
        luaC_objbarriert(L, hvalue(obj), mt);
      break;
    }
    case LUA_TUSERDATA: {
      uvalue(obj)->metatable = mt;
      if (mt)
        luaC_objbarrier(L, rawuvalue(obj), mt);
      break;
    }
    case LUA_TSEQ: {  /* 0.11.0 */
      seqvalue(obj)->metatable = mt;
      if (mt)
        luaC_objbarrierseq(L, seqvalue(obj), mt);
      break;
    }
    case LUA_TPAIR: {  /* 0.11.1 */
      pairvalue(obj)->metatable = mt;
      if (mt)
        luaC_objbarrierpair(L, pairvalue(obj), mt);
      break;
    }
    case LUA_TSET: {  /* 0.11.0 */
      usvalue(obj)->metatable = mt;
      if (mt)
        luaC_objbarrierset(L, usvalue(obj), mt);
      break;
    }
    default: {
      G(L)->mt[ttype(obj)] = mt;
      break;
    }
  }
  L->top--;
  lua_unlock(L);
  return 1;
}


LUA_API int lua_setfenv (lua_State *L, int idx) {
  StkId o;
  int res = 1;
  lua_lock(L);
  api_checknelems(L, 1);
  o = index2adr(L, idx);
  api_checkvalidindex(L, o);
  api_check(L, ttistable(L->top - 1));
  switch (ttype(o)) {
    case LUA_TFUNCTION:
      clvalue(o)->c.env = hvalue(L->top - 1);
      break;
    case LUA_TUSERDATA:
      uvalue(o)->env = hvalue(L->top - 1);
      break;
    case LUA_TTHREAD:
      sethvalue(L, gt(thvalue(o)), hvalue(L->top - 1));
      break;
    default:
      res = 0;
      break;
  }
  /* luaC_objbarrier(L, gcvalue(o), hvalue(L->top - 1)); */
  if (res) luaC_objbarrier(L, gcvalue(o), hvalue(L->top - 1));  /* Lua 5.1.3 patch */
  L->top--;
  lua_unlock(L);
  return res;
}


/*
** `load' and `call' functions (run Lua code)
*/


#define adjustresults(L,nres) \
    { if (nres == LUA_MULTRET && L->top >= L->ci->top) L->ci->top = L->top; }


#define checkresults(L,na,nr) \
     api_check(L, (nr) == LUA_MULTRET || (L->ci->top - L->top >= (nr) - (na)))


LUA_API void lua_call (lua_State *L, int nargs, int nresults) {
  StkId func;
  lua_lock(L);
  api_checknelems(L, nargs+1);
  checkresults(L, nargs, nresults);
  func = L->top - (nargs+1);
  luaD_call(L, func, nresults);
  adjustresults(L, nresults);
  lua_unlock(L);
}


/* calls a function and returns a numeric result */

LUA_API lua_Number agn_ncall (lua_State *L, int nargs, int nresults) {
  StkId func;
  const TValue *o;
  lua_Number r;
  lua_lock(L);
  api_checknelems(L, nargs+1);
  checkresults(L, nargs, nresults);
  func = L->top - (nargs+1);
  luaD_call(L, func, nresults);
  adjustresults(L, nresults);
  o = L->top-1;
  if (!ttisnumber(o)) luaG_runerror(L, "return of function call is not a number.");  /* new in 0.27.1 */
  r = nvalue(o);
  L->top--;  /* delete numeric result from stack (since it is already stored to r) */
  lua_unlock(L);
  return r;
}


/* calls a function and returns a complex result; used by fractals package */

#ifndef PROPCMPLX
LUA_API agn_Complex agn_ccall (lua_State *L, int nargs, int nresults) {
  StkId func;
  const TValue *o;
  agn_Complex r;
  lua_lock(L);
  api_checknelems(L, nargs+1);
  checkresults(L, nargs, nresults);
  func = L->top - (nargs+1);
  luaD_call(L, func, nresults);
  adjustresults(L, nresults);
  o = L->top-1;
  if (!ttiscomplex(o))
    luaG_runerror(L, "return of function call is not complex, but a %s.", lua_typename(L, ttype(o)));  /* new in 0.27.1, 1.9.3 */
  r = cvalue(o);
  api_check(L, 1 <= (L->top - L->base));
  L->top--;  /* delete numeric result from stack (since it has already been stored to r) */
  lua_unlock(L);
  return r;
}
#else
LUA_API void agn_ccall (lua_State *L, int nargs, int nresults, lua_Number *real, lua_Number *imag) {  /* 0.27.1 */
  StkId func;
  const TValue *o;
  lua_lock(L);
  api_checknelems(L, nargs+1);
  checkresults(L, nargs, nresults);
  func = L->top - (nargs+1);
  luaD_call(L, func, nresults);
  adjustresults(L, nresults);
  o = L->top-1;
  if (!ttiscomplex(o))
    luaG_runerror(L, "return of function call is not complex, but a %s.", lua_typename(L, ttype(o)));  /* new in 0.27.1, 1.9.3 */
  *real = complexreal(o);
  *imag = compleximag(o);
  api_check(L, 1 <= (L->top - L->base));
  L->top--;  /* delete numeric result from stack (since it is already stored to real and imag) */
  lua_unlock(L);
}
#endif


/*
** Execute a protected call.
*/
struct CallS {  /* data to `f_call' */
  StkId func;
  int nresults;
};


static void f_call (lua_State *L, void *ud) {
  struct CallS *c = cast(struct CallS *, ud);
  luaD_call(L, c->func, c->nresults);
}



LUA_API int lua_pcall (lua_State *L, int nargs, int nresults, int errfunc) {
  struct CallS c;
  int status;
  ptrdiff_t func;
  lua_lock(L);
  api_checknelems(L, nargs+1);
  checkresults(L, nargs, nresults);
  if (errfunc == 0)
    func = 0;
  else {
    StkId o = index2adr(L, errfunc);
    api_checkvalidindex(L, o);
    func = savestack(L, o);
  }
  c.func = L->top - (nargs+1);  /* function to be called */
  c.nresults = nresults;
  status = luaD_pcall(L, f_call, &c, savestack(L, c.func), func);
  adjustresults(L, nresults);
  lua_unlock(L);
  return status;
}


/*
** Execute a protected C call.
*/
struct CCallS {  /* data to `f_Ccall' */
  lua_CFunction func;
  void *ud;
};


static void f_Ccall (lua_State *L, void *ud) {
  struct CCallS *c = cast(struct CCallS *, ud);
  Closure *cl;
  cl = luaF_newCclosure(L, 0, getcurrenv(L));
  cl->c.f = c->func;
  setclvalue(L, L->top, cl);  /* push function */
  api_incr_top(L);
  setpvalue(L->top, c->ud);  /* push only argument */
  api_incr_top(L);
  luaD_call(L, L->top - 2, 0);
}


LUA_API int lua_cpcall (lua_State *L, lua_CFunction func, void *ud) {
  struct CCallS c;
  int status;
  lua_lock(L);
  c.func = func;
  c.ud = ud;
  status = luaD_pcall(L, f_Ccall, &c, savestack(L, L->top), 0);
  lua_unlock(L);
  return status;
}


LUA_API int lua_dump (lua_State *L, lua_Writer writer, void *data) {  /* reintroduced 1.6.0 */
  int status;
  TValue *o;
  lua_lock(L);
  api_checknelems(L, 1);
  o = L->top - 1;
  if (isLfunction(o))
    status = luaU_dump(L, clvalue(o)->l.p, writer, data, 0);
  else
    status = 1;
  lua_unlock(L);
  return status;
}


LUA_API int lua_load (lua_State *L, lua_Reader reader, void *data,
                      const char *chunkname) {
  ZIO z;
  int status;
  lua_lock(L);
  if (!chunkname) chunkname = "?";
  luaZ_init(L, &z, reader, data);
  status = luaD_protectedparser(L, &z, chunkname);
  lua_unlock(L);
  return status;
}


LUA_API int  lua_status (lua_State *L) {
  return L->status;
}


/*
** Garbage-collection function
*/

LUA_API int lua_gc (lua_State *L, int what, int data) {
  int res = 0;
  global_State *g;
  lua_lock(L);
  g = G(L);
  switch (what) {
    case LUA_GCSTOP: {
      g->GCthreshold = MAX_LUMEM;
      break;
    }
    case LUA_GCRESTART: {
      g->GCthreshold = g->totalbytes;
      break;
    }
    case LUA_GCCOLLECT: {
      luaC_fullgc(L);
      break;
    }
    case LUA_GCCOUNT: {
      /* GC values are expressed in Kbytes: #bytes/2^10 */
      res = cast_int(g->totalbytes >> 10);
      break;
    }
    case LUA_GCCOUNTB: {
      res = cast_int(g->totalbytes & 0x3ff);
      break;
    }
    case LUA_GCSTEP: {
      lu_mem a = (cast(lu_mem, data) << 10);
      if (a <= g->totalbytes)
        g->GCthreshold = g->totalbytes - a;
      else
        g->GCthreshold = 0;
      while (g->GCthreshold <= g->totalbytes) {  /* Lua 5.1.3 patch 10 */
        luaC_step(L);
        if (g->gcstate == GCSpause) {  /* end of cycle? */
          res = 1;  /* signal it */
          break;
        }
      }
      break;
    }
    case LUA_GCSETPAUSE: {
      res = g->gcpause;
      g->gcpause = data;
      break;
    }
    case LUA_GCSETSTEPMUL: {
      res = g->gcstepmul;
      g->gcstepmul = data;
      break;
    }
    default: res = -1;  /* invalid option */
  }
  lua_unlock(L);
  return res;
}


LUA_API LUAI_UMEM agn_usedbytes (lua_State *L) {
  global_State *g;
  LUAI_UMEM res;
  lua_lock(L);
  g = G(L);
  res = (LUAI_UMEM)g->totalbytes;
  lua_unlock(L);
  return res;
}


/*
** miscellaneous functions
*/


LUA_API int lua_error (lua_State *L) {
  lua_lock(L);
  api_checknelems(L, 1);
  luaG_errormsg(L);
  lua_unlock(L);
  return 0;  /* to avoid warnings */
}


LUA_API int lua_next (lua_State *L, int idx) {
  StkId t;
  int more;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttistable(t));
  more = luaH_next(L, hvalue(t), L->top - 1);
  if (more) {
    api_incr_top(L);
  }
  else  /* no more elements */
    L->top--;  /* remove key */
  lua_unlock(L);
  return more;
}


LUA_API int lua_usnext (lua_State *L, int idx) {  /* 0.10.0 */
  StkId t;
  int more;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttisuset(t));
  more = agnUS_next(L, usvalue(t), L->top - 1);
  if (more) {
    api_incr_top(L);
  }
  else  /* no more elements */
    L->top--;  /* remove key */
  lua_unlock(L);
  return more;
}


LUA_API int lua_seqnext (lua_State *L, int idx) {  /* 0.11.2 */
  StkId t;
  int more;
  lua_lock(L);
  t = index2adr(L, idx);
  api_check(L, ttisseq(t));
  more = agnSeq_next(L, seqvalue(t), L->top - 1);
  if (more) {
    api_incr_top(L);
  }
  else  /* no more elements */
    L->top--;  /* remove key */
  lua_unlock(L);
  return more;
}


/* 0.12.0, used by for i in string loop; optimized 0.24.1 */
LUA_API int lua_strnext (lua_State *L, int idx) {
  StkId t, below;
  int i, more;
  const char *str;
  lua_lock(L);
  below = L->top-1;
  t = index2adr(L, idx);
  api_check(L, ttisstring(t));
  if (ttisnil(below))  /* first iteration */
    i = 0;
  else if (ttisnumber(below))
    i = nvalue(below);  /* get current string position */
  else {
    i = -1;
    luaG_runerror(L, "invalid key to " LUA_QL("next"));
  }
  str = svalue(t);
  if (i < strlen(str)) {
    more = 1;
    setnvalue(below, cast_num(i+1));  /* set new key */
    setsvalue(L, L->top, luaS_newchar(L, str+i));  /* set character */
    api_incr_top(L);
  }
  else {  /* no more elements */
    more = 0;
    L->top--;  /* remove key */
  }
  lua_unlock(L);
  return more;
}


LUA_API void lua_concat (lua_State *L, int n) {
  lua_lock(L);
  api_checknelems(L, n);
  if (n >= 2) {
    luaC_checkGC(L);
    luaV_concat(L, n, cast_int(L->top - L->base) - 1);
    L->top -= (n-1);
  }
  else if (n == 0) {  /* push empty string */
    setsvalue2s(L, L->top, luaS_newlstr(L, "", 0));
    api_incr_top(L);
  }
  /* else n == 1; nothing to do */
  lua_unlock(L);
}


LUA_API lua_Alloc lua_getallocf (lua_State *L, void **ud) {
  lua_Alloc f;
  lua_lock(L);
  if (ud) *ud = G(L)->ud;
  f = G(L)->frealloc;
  lua_unlock(L);
  return f;
}


LUA_API void lua_setallocf (lua_State *L, lua_Alloc f, void *ud) {
  lua_lock(L);
  G(L)->ud = ud;
  G(L)->frealloc = f;
  lua_unlock(L);
}


LUA_API void *lua_newuserdata (lua_State *L, size_t size) {
  Udata *u;
  lua_lock(L);
  luaC_checkGC(L);
  u = luaS_newudata(L, size, getcurrenv(L));
  setuvalue(L, L->top, u);
  api_incr_top(L);
  lua_unlock(L);
  return u + 1;
}


static const char *aux_upvalue (StkId fi, int n, TValue **val) {
  Closure *f;
  if (!ttisfunction(fi)) return NULL;
  f = clvalue(fi);
  if (f->c.isC) {
    if (!(1 <= n && n <= f->c.nupvalues)) return NULL;
    *val = &f->c.upvalue[n-1];
    return "";
  }
  else {
    Proto *p = f->l.p;
    if (!(1 <= n && n <= p->sizeupvalues)) return NULL;
    *val = f->l.upvals[n-1]->v;
    return getstr(p->upvalues[n-1]);
  }
}


LUA_API const char *lua_getupvalue (lua_State *L, int funcindex, int n) {
  const char *name;
  TValue *val;
  lua_lock(L);
  name = aux_upvalue(index2adr(L, funcindex), n, &val);
  if (name) {
    setobj2s(L, L->top, val);
    api_incr_top(L);
  }
  lua_unlock(L);
  return name;
}


LUA_API const char *lua_setupvalue (lua_State *L, int funcindex, int n) {
  const char *name;
  TValue *val;
  StkId fi;
  lua_lock(L);
  fi = index2adr(L, funcindex);
  api_checknelems(L, 1);
  name = aux_upvalue(fi, n, &val);
  if (name) {
    L->top--;
    setobj(L, val, L->top);
    luaC_barrier(L, clvalue(fi), L->top);
  }
  lua_unlock(L);
  return name;
}

/* changed/additions */


/* agn_poptop: pops the value at the top of the stack; written July 12, 2007 */

LUA_API void agn_poptop (lua_State *L) {
  lua_lock(L);
  api_check(L, 1 <= (L->top - L->base));
  L->top--;  /* decrease top */
  lua_unlock(L);
}


/* agn_poptop: pops the value at the top of the stack; written April 2, 2008 */

LUA_API void agn_poptoptwo (lua_State *L) {
  lua_lock(L);
  api_check(L, 1 <= (L->top - L->base));
  L->top -= 2;  /* decrease top */
  lua_unlock(L);
}


/* gets the i-th number from the table at the given index idx and returns it as a lua_Number;
   written July 12, 2007 */

LUA_API lua_Number agn_getinumber (lua_State *L, int idx, int n) {
  StkId o;
  const TValue *p;
  lua_lock(L);
  o = index2adr(L, idx);
  api_check(L, ttistable(o));
  p = luaH_getnum(hvalue(o), n);
  lua_unlock(L);
  if (ttype(p) == LUA_TNUMBER)
    return nvalue(p);
  else
    return 0;
}


/* gets the i-th string from the table at the given index idx and returns it;
   written July 21, 2007; used by baselib `concat` function for high-speed to table elements */

LUA_API const char *agn_getistring (lua_State *L, int idx, int n) {
  StkId o;
  const TValue *p;
  lua_lock(L);
  o = index2adr(L, idx);
  api_check(L, ttistable(o));
  p = luaH_getnum(hvalue(o), n);
  lua_unlock(L);
  if (ttype(p) == LUA_TSTRING)
    return svalue(p);
  else
    return NULL;
}


/* for lbaselib `concat` function; tuned. */
LUA_API const char *agn_getseqlstring (lua_State *L, int idx, int n, size_t *l) {
  StkId o;
  const TValue *p;
  lua_lock(L);
  o = index2adr(L, idx);
  api_check(L, ttisseq(o));
  p = agnSeq_geti(seqvalue(o), n);
  lua_unlock(L);
  if (ttype(p) == LUA_TSTRING) {
    *l = tsvalue(p)->len;
    return svalue(p);
  }
  else {
    l = NULL;
    return NULL;
  }
}


/* for fast determination of indices in tables.indices */
LUA_API void lua_rawsetikey (lua_State *L, int idx, int n) {
  StkId o;
  lua_lock(L);
  api_checknelems(L, 2);
  o = index2adr(L, idx);
  api_check(L, ttistable(o));
  setobj2t(L, luaH_setnum(L, hvalue(o), n), L->top-2);
  luaC_barriert(L, hvalue(o), L->top-2);
  if (ttisnil(L->top-2)) hvalue(o)->hasnil = 1;
  L->top--;
  lua_unlock(L);
}


/* same as lua_rawset, but only deleting the value, and not the key; August 04, 2007 */

LUA_API void lua_rawset2 (lua_State *L, int idx) {
  StkId t;
  lua_lock(L);
  api_checknelems(L, 2);
  t = index2adr(L, idx);
  api_check(L, ttistable(t));
  setobj2t(L, luaH_set(L, hvalue(t), L->top-2), L->top-1);
  luaC_barriert(L, hvalue(t), L->top-1);
  if (ttisnil(L->top-1)) hvalue(t)->hasnil = 1;
  L->top--;  /* pop value from the stack */
  lua_unlock(L);
}


/* fast number check, 0.6.0; October 25, 2007 */
LUALIB_API lua_Number agn_checknumber (lua_State *L, int idx) {
  StkId o;
  lua_lock(L);
  o = index2adr(L, idx);
  if (ttype(o) != LUA_TNUMBER)
    tag_error(L, idx, LUA_TNUMBER);
  lua_unlock(L);
  return nvalue(o);
}


LUALIB_API lua_Integer agn_checkinteger (lua_State *L, int idx) {  /* 1.12.9 */
  StkId o;
  lua_Number x;
  lua_lock(L);
  o = index2adr(L, idx);
  if (ttype(o) != LUA_TNUMBER)
    tag_error(L, idx, LUA_TNUMBER);
  x = nvalue(o);
  if (ISFLOAT(x))
    luaL_error(L, "Wrong argument #%d: expected an integer, got float.", idx);
  lua_unlock(L);
  return x;
}


#ifndef PROPCMPLX
LUA_API agn_Complex agn_checkcomplex (lua_State *L, int idx) {
  StkId o;
  lua_lock(L);
  o = index2adr(L, idx);
  if (ttype(o) != LUA_TCOMPLEX)
    tag_error(L, idx, LUA_TCOMPLEX);
  lua_unlock(L);
  return cvalue(o);
}


LUA_API agn_Complex agn_optcomplex (lua_State *L, int narg, agn_Complex def) {
  return luaL_opt(L, agn_checkcomplex, narg, def);
}
#endif


/* string check without conversion of numbers, 0.9.1, 20.01.2008 */
LUALIB_API const char *agn_checklstring (lua_State *L, int idx, size_t *len) {
  StkId o;
  lua_lock(L);
  o = index2adr(L, idx);
  if (!ttisstring(o)) {
    tag_error(L, idx, LUA_TSTRING);
    if (len != NULL) *len = 0;
    lua_unlock(L);
    return NULL;
  }
  if (len != NULL) *len = tsvalue(o)->len;
  lua_unlock(L);
  return svalue(o);
}


/* string check without conversion of numbers, 0.9.1, 20.01.2008 */
LUALIB_API const char *agn_checkstring (lua_State *L, int idx) {
  StkId o;
  lua_lock(L);
  o = index2adr(L, idx);
  if (!ttisstring(o)) {
    tag_error(L, idx, LUA_TSTRING);
    lua_unlock(L);
    return NULL;
  }
  lua_unlock(L);
  return svalue(o);
}


/* Checks for Booleans, 1.6.0; April 14, 2012 */
LUALIB_API lua_Integer agn_checkboolean (lua_State *L, int idx) {
  StkId o;
  lua_lock(L);
  o = index2adr(L, idx);
  if (ttype(o) != LUA_TBOOLEAN)
    tag_error(L, idx, LUA_TBOOLEAN);
  lua_unlock(L);
  return bvalue(o);  /* 1 is true, 2 is fail, 0 is false */
}


/* 0.5.2, June 01, 2007; tuned April 02, 2008; FIXME: define as macros ? */
LUA_API int agn_istrue (lua_State *L, int idx) {
  return bvalue(index2adr(L, idx)) == 1;  /* 1 is true, 2 is fail, 0 is false */
}


/* 0.10.0, April 02, 2008; FIXME: define as macro ? */
LUA_API int agn_isfalse (lua_State *L, int idx) {
  return bvalue(index2adr(L, idx)) == 0;  /* 1 is true, 2 is fail, 0 is false */
}


/* 0.10.0, April 02, 2008; FIXME: define as macro ? */
LUA_API int agn_isfail (lua_State *L, int idx) {
  return l_isfail(index2adr(L, idx));
}


/* 0.9.1, January 12, 2008; extended 0.10.0 to handle sets and sequences */
LUA_API size_t agn_nops (lua_State *L, int idx) {
  StkId o;
  o = index2adr(L, idx);
  switch (ttype(o)) {
    case LUA_TTABLE:
      return agenaV_nops(L, hvalue(o));
    case LUA_TSET:
      return usvalue(o)->size;
    case LUA_TSEQ:
      return seqvalue(o)->size;
    default:
      return 0;
  }
}


LUA_API int agn_fnext (lua_State *L, int idxt, int idxf, int mode) {
  StkId t, f;
  int more;
  lua_lock(L);
  t = index2adr(L, idxt);
  api_check(L, ttistable(t));
  f = index2adr(L, idxf);
  api_check(L, ttisfunction(f));
  more = luaHF_next(L, hvalue(t), f, L->top - 1, mode);
  if (more) {
    if (mode == 1) {
      api_incr_top_by_three(L);
    }
    else {
      api_incr_top_by_two(L);
    }
  }
  else  /* no more elements */
    L->top--;  /* remove key */
  lua_unlock(L);
  return more;
}


#define actnodesize(t)    ( ((t)->lsizenode) == 0 ? 0 : 1<<((t)->lsizenode))  /* patched Agena 1.0.3 */

/* extended 0.22.0 to report holes in an array */

LUA_API void agn_tablestate (lua_State *L, int idx, size_t a[], int mode) {
  StkId o;
  Table *t;
  lua_lock(L);
  o = index2adr(L, idx);
  api_check(L, ttistable(o));
  t = hvalue(o);
  if (mode)
    agnH_nops(L, t, a);  /* a[0] = assigned elements in array, a[1] = assigned elements in hash part,
                            a[2] = info whether array contains at least one hole */
  else {
    a[0] = a[1] = a[2] = 0;
  }
  a[3] = t->sizearray;  /* number of allocated elements in array */
  /* number of allocated elements in hash part: */
  a[4] = (a[1] == 1 && actnodesize(t) == 0) ? 1 : actnodesize(t);  /* Agena 1.0.5 patch */
  a[5] = t->hasnil;
  lua_unlock(L);
}


LUA_API void agn_sstate (lua_State *L, int idx, size_t a[]) {
  StkId o;
  UltraSet *t;
  lua_lock(L);
  o = index2adr(L, idx);
  api_check(L, ttisuset(o));
  t = usvalue(o);
  a[0] = t->size;
  a[1] = actnodesize(t);
  lua_unlock(L);
}


LUA_API void agn_seqstate (lua_State *L, int idx, size_t a[]) {
  StkId o;
  Seq *t;
  lua_lock(L);
  o = index2adr(L, idx);
  api_check(L, ttisseq(o));
  t = seqvalue(o);
  a[0] = t->size;
  a[1] = t->maxsize;
  lua_unlock(L);
}


LUA_API size_t agn_seqsize (lua_State *L, int idx) {
  StkId o;
  Seq *t;
  lua_lock(L);
  o = index2adr(L, idx);
  api_check(L, ttisseq(o));
  t = seqvalue(o);
  lua_unlock(L);
  return t->size;
}


LUA_API size_t agn_size (lua_State *L, int idx) {
  StkId o;
  Table *t;
  lua_lock(L);
  o = index2adr(L, idx);
  api_check(L, ttistable(o));
  t = hvalue(o);
  lua_unlock(L);
  return agenaV_nops(L, t);
}


LUA_API size_t agn_ssize (lua_State *L, int idx) {
  StkId o;
  UltraSet *t;
  lua_lock(L);
  o = index2adr(L, idx);
  api_check(L, ttisuset(o));
  t = usvalue(o);
  lua_unlock(L);
  return t->size;
}


/* 0.22.0: returns 1 if the function at idx is a C function, 0 if the function at idx is an Agena function,
   and -1 of it is no function at all. */
LUA_API int agn_getfunctiontype (lua_State *L, int idx) {
  const TValue *o;
  int fn;
  lua_lock(L);
  o = index2adr(L, idx);
  if (ttype(o) == LUA_TFUNCTION)
    fn = clvalue(o)->l.isC;
  else
    fn = -1;
  lua_unlock(L);
  return fn;
}


/* 0.22.1: returns a true copy of the structure at stack index idx. The copy is put on top of the stack, but the
   original structure is not removed. */
LUA_API void agn_copy (lua_State *L, int idx) {
  const TValue *o;
  lua_lock(L);
  o = index2adr(L, idx);
  agenaV_copy(L, o, L->top, NULL);
  api_incr_top(L);
  lua_unlock(L);
}


LUA_API void agn_creatertable (lua_State *L, int idx, int writemode) {
  TValue *o;
  Table *nullt;
  lua_lock(L);
  api_checknelems(L, 1);
  o = index2adr(L, idx);
  if (isLfunction(o)) {
    nullt = luaH_new(L, 0, 0);
    clvalue(o)->l.rtable = nullt;
    if (nullt) {
      luaC_objbarrier(L, clvalue(o), nullt);
    } else
      if (writemode)
        luaG_runerror(L, "rtable initialisation failure");
      else
        luaG_runerror(L, "rotable initialisation failure");
    if (!ttistable(clvalue(o)->l.rtable)) {
      clvalue(o)->l.rtable = NULL;
      if (writemode)
        luaG_runerror(L, "rtable initialisation failure");
      else
        luaG_runerror(L, "rotable initialisation failure");
    }
    if (writemode) clvalue(o)->l.updatertable = 1;  /* 0.22.0 */
  }
  else if (iscfunction(o) && !writemode) {
    nullt = luaH_new(L, 0, 0);
    clvalue(o)->c.rtable = nullt;
    if (nullt) {
      luaC_objbarrier(L, clvalue(o), nullt);
    }
    else
      luaG_runerror(L, "rotable initialisation failure");
    if (!ttistable(clvalue(o)->c.rtable)) {
      clvalue(o)->c.rtable = NULL;
      luaG_runerror(L, "rotable initialisation failure");
    }
  }
  else
    luaG_runerror(L, "rtables are available for Agena procedures only.");
  lua_unlock(L);
}


LUA_API int agn_getrtable (lua_State *L, int idx) {
  const TValue *o;
  Table *mt = NULL;
  int res;
  lua_lock(L);
  o = index2adr(L, idx);
  if (ttype(o) == LUA_TFUNCTION)
    mt = clvalue(o)->l.rtable;
  else
    luaG_runerror(L, "remember tables are available for procedures only.");
  if (mt == NULL) {
    res = 0; }
  else {
    sethvalue(L, L->top, mt);
    api_incr_top(L);
    res = 1;
  }
  lua_unlock(L);
  return res;
}


/* 0.22.0: returns 0 if rtable cannot be updated by the RETURN statement, 1 if it can,
   2 if `idx` has no remember table at all, and -1 if `idx` is not a function. */
LUA_API int agn_getrtablewritemode (lua_State *L, int idx) {
  const TValue *o;
  int mode = 0;
  lua_lock(L);
  o = index2adr(L, idx);
  if (ttype(o) == LUA_TFUNCTION)
    mode = (clvalue(o)->l.rtable == NULL) ? 2 : clvalue(o)->l.updatertable;
  else
    mode = -1;
  lua_unlock(L);
  return mode;
}


LUA_API void agn_setrtable (lua_State *L, int find, int kind, int vind) {
  TValue *o;
  Table *rt, *dict;
  int hashval, newone, flag;
  const TValue *entry;
  lua_lock(L);
  if (!lua_isfunction(L, find))
    luaG_runerror(L, "must get a function as first argument");
  if (!lua_istable(L, kind))
    luaG_runerror(L, "must get a table as second argument");
  if (!lua_istable(L, vind))
    luaG_runerror(L, "must get a table as third argument");
  o = index2adr(L, find);
  api_checkvalidindex(L, o);
  rt = clvalue(o)->l.rtable;
  if (rt == NULL)
    luaG_runerror(L, "rtable not initialised");
  if (!lua_checkstack(L, 2))
    luaG_runerror(L, "not enough memory for stack");
  lua_pushnil(L);
  /* compute the respective hash value for the argument list given */
  hashval = 0;
  while (lua_next(L, kind) != 0) {
    hashval += hash(L->top-1);  /* compute hash for all values in argument table */
    agn_poptop(L);
  }
  /* look whether the hash value already exists */
  entry = luaH_getnum(rt, hashval);
  if ((newone = (ttisnil(entry)))) {
    dict = luaH_new(L, 0, 1);  /* create a new dictionary */
  } else {
    dict = hvalue(entry);      /* use the existing dictionary */
  }
  flag = 0;
  if (!newone) {
    /* search for existing equal keys (arguments) in dictionary assigned to the respective hash value, 0.22.0 */
    Node *node;
    int i;
    Table *rk, *tk;
    tk = hvalue(index2adr(L, kind));  /* get argument table given by rset */
    for (i = 0; i < sizenode(dict); i++) {
      node = gnode(dict, i);
      if (ttisnotnil(gval(node))) {  /* key (the table of arguments) found */
        rk = hvalue(key2tval(node));  /* assign key (table of arguments) to Table object rk */
        if ((flag = agenaV_comptablesonebyone(L, rk, tk))) {  /* there is already in entry in the rtable */
          sethvalue(L, L->top, rk);
          setobjt2t(L, luaH_set(L, dict, L->top), index2adr(L, vind));  /* get results table given by rset */
          luaC_barriert(L, dict, index2adr(L, vind));
          break;
        }
      }
    }
  }
  if (!flag) {  /* arguments not yet included in rtable ? */
    /* first enter arguments~results pair into the dictionary */
    setobjt2t(L, luaH_set(L, dict, index2adr(L, kind)), index2adr(L, vind));
    luaC_barriert(L, dict, index2adr(L, vind));
  }
  /* set table to stack top */
  sethvalue(L, L->top, dict);
  /* then enter key~value pair (arguments~results) into rtable and assign it to key `hashvalue` */
  setobjt2t(L, luaH_setnum(L, rt, hashval), L->top);
  luaC_barriert(L, rt, L->top);
  lua_unlock(L);
}


LUA_API void agn_deletertable (lua_State *L, int objindex) {
  TValue *o;
  lua_lock(L);
  api_checknelems(L, 1);
  o = index2adr(L, objindex);
  api_checkvalidindex(L, o);  /* Agena 1.0.3 fix */
  if (isLfunction(o)) {
    clvalue(o)->l.rtable = NULL;
    clvalue(o)->l.updatertable = 0; }
  else if (iscfunction(o)) {
    clvalue(o)->c.rtable = NULL;
    clvalue(o)->c.updatertable = 0; }
  else
    luaG_runerror(L, "remember tables are available for procedures only.");
  lua_unlock(L);
}


LUA_API int agn_setreadlibbed (lua_State *L, const char *name) {
  int rc = 0;
  lua_lock(L);
  lua_getglobal(L, "package");
  if (lua_istable(L, -1)) {
    lua_getfield(L, -1, "readlibbed");
    if (lua_isset(L, -1)) {
      lua_pushstring(L, name);
      lua_srawset(L, -2);
      rc = 1;
    }
    agn_poptop(L);  /* pop field */
  }
  agn_poptop(L);  /* pop `package' */
  lua_unlock(L);
  return rc;
}


/* see: http://graphics.stanford.edu/~seander/bithacks.html on how to modify bits, 0.32.0 */

LUA_API void agn_setbitwise (lua_State *L, int value) {
  L->settings ^= (-(value > 0 ? (lu_byte)1 : (lu_byte)0) ^ L->settings) & 1;
}


LUA_API int agn_getbitwise (lua_State *L) {  /* 0.32.0 */
  return L->settings & 1;
}


LUA_API void agn_setemptyline (lua_State *L, int value) {  /* 0.32.0 */
  L->settings ^= (-(value > 0 ? (lu_byte)1 : (lu_byte)0) ^ L->settings) & 2;
}


LUA_API int agn_getemptyline (lua_State *L) {  /* 0.32.0 */
  return L->settings & 2;
}


LUA_API void agn_setlibnamereset (lua_State *L, int value) {  /* 0.32.0 */
  L->settings ^= (-(value > 0 ? (lu_byte)1 : (lu_byte)0) ^ L->settings) & 4;
}


LUA_API int agn_getlibnamereset (lua_State *L) {  /* 0.32.0 */
  return L->settings & 4;
}


LUA_API void agn_setlongtable (lua_State *L, int value) {  /* 0.32.0 */
  L->settings ^= (-(value > 0 ? (lu_byte)1 : (lu_byte)0) ^ L->settings) & 8;
}


LUA_API int agn_getlongtable (lua_State *L) {  /* 0.32.0 */
  return L->settings & 8;
}


LUA_API void agn_setdebug (lua_State *L, int value) {  /* 0.32.2a */
  L->settings ^= (-(value > 0 ? (lu_byte)1 : (lu_byte)0) ^ L->settings) & 16;
}


LUA_API int agn_getdebug (lua_State *L) {  /* 0.32.0 */
  return L->settings & 16;
}


LUA_API void agn_setgui (lua_State *L, int value) {  /* 0.33.3 */
  L->settings ^= (-(value > 0 ? (lu_byte)1 : (lu_byte)0) ^ L->settings) & 32;
}


LUA_API int agn_getgui (lua_State *L) {  /* 0.33.3 */
  return L->settings & 32;
}


LUA_API void agn_setzeroedcomplex (lua_State *L, int value) {  /* 1.7.6 */
  L->settings ^= (-(value > 0 ? (lu_byte)1 : (lu_byte)0) ^ L->settings) & 64;
}


LUA_API int agn_getzeroedcomplex (lua_State *L) {  /* 1.7.6 */
  return L->settings & 64;
}


LUA_API void agn_setpromptnewline (lua_State *L, int value) {  /* 1.7.6 */
  L->settings ^= (-(value > 0 ? (lu_byte)1 : (lu_byte)0) ^ L->settings) & 128;
}


LUA_API int agn_getpromptnewline (lua_State *L) {  /* 1.7.6 */
  return L->settings & 128;
}


LUA_API void agn_setdigits (lua_State *L, ptrdiff_t x) {  /* x must be a number in [1, 17] */
  const char *num;
  if (x < 1 || x > 17) luaG_runerror(L, LUA_QS " setting must be an integer in [1, 17].", "environ.kernel/digits");
  lua_pushinteger(L, x);
  num = lua_tolstring(L, -1, NULL);
  lua_pushstring(L, "%.");
  lua_pushstring(L, num);
  lua_pushstring(L, "g");
  lua_concat(L, 3);
  L->numberformat = strdup(lua_tolstring(L, -1, NULL));
  if (L->numberformat == NULL) luaG_runerror(L, "internal error in " LUA_QS ".", "environ.kernel/digits");
  agn_poptoptwo(L);  /* pop format string and x */
}


LUA_API lua_Number agn_getdigits (lua_State *L) {
  char *endptr, *fmt, *newpos;
  lua_Number n;
  fmt = strdup(L->numberformat);
  if (fmt == NULL) return 0;  /* conversion failed */
  while (!isdigit(*fmt)) fmt++;
  newpos = fmt;
  while (isdigit(*fmt)) fmt++;
  *fmt = '\0';
  n = lua_str2number(newpos, &endptr);
  if (endptr == newpos) return 0;  /* conversion failed */
  return n;
}


LUA_API lua_Number agn_getstructuresize (lua_State *L, int idx) {
  const TValue *o;
  o = index2adr(L, idx);
  api_checkvalidindex(L, o);  /* Agena 1.0.3 fix */
  switch (ttype(o)) {
    case LUA_TTABLE: {
      Table *h;
      h = hvalue(o);
      return (lua_Number)sizeof(Table) + sizeof(TValue) * h->sizearray + sizeof(Node) * actnodesize(h);
    }
    case LUA_TSET: {
      UltraSet *s;
      s = usvalue(o);
      return (lua_Number)(sizeof(UltraSet) + sizeof(LNode)*actnodesize(s));
    }
    case LUA_TSEQ: {
      Seq *s;
      s = seqvalue(o);
      return (lua_Number)(sizeof(Seq) + sizeof(TValue)*s->maxsize);
    }
    case LUA_TPAIR: {
      return (lua_Number)sizeof(Pair) + sizeof(TValue)*2;
    }
    case LUA_TFUNCTION: {
      Closure *cl = clvalue(o);
      return (lua_Number)((cl->c.isC) ? sizeCclosure(cl->c.nupvalues) :
                           sizeLclosure(cl->l.nupvalues));
    }
  }
  return 0;
}


/* agn_paircheckbooloption

For the given Agena procedure `procname`, checks whether the value at index `idx` is a pair, and whether its left operand is
equals to `option` (of type string), and whether the right operand is a boolean.

Returns -2 if the value at idx is not a pair, or the result of the call to the lua_toboolean C API function.

The function issues an error if the left operand of the pair is not equals to `option`, or if the right operand is not a
boolean.

The function does _not_ pop the pair at `idx`. */

LUA_API int agn_paircheckbooloption (lua_State *L, const char *procname, int idx, const char *option) {  /* 1.8.9, 10.11.2012 */
  int r = -2;
  if (idx > 0 && lua_ispair(L, idx)) {
    agn_pairgeti(L, idx, 1);  /* get left value, set to stack index -2 */
    agn_pairgeti(L, idx, 2);  /* get right value, set to stack index  -1 */
    if (strcmp(option, agn_checkstring(L, -2)) != 0)
      luaL_error(L, "Error in " LUA_QS ": unknown option %s, expected %s.", procname, agn_checkstring(L, -2), option);
    else if (lua_type(L, -1) != LUA_TBOOLEAN)
      luaL_error(L, "Error in " LUA_QS ": invalid type for %s option, got %s.", procname, option, lua_typename(L, lua_type(L, -1)));
    else
      r = lua_toboolean(L, -1);
    agn_poptoptwo(L);
  }
  return r;
}


/*
agn_pairgetnumbers

For the given Agena procedure `procname`, checks whether the value at stack index `idx` is a pair. It then checks whether the
left-hand and right-hand side are numbers and returns these numbers in x and y.

If the value at `idx` is not a pair, or if at least one of its operands is not a number, it issues an error.
*/

LUA_API void agn_pairgetnumbers (lua_State *L, const char *procname, int idx, lua_Number *x, lua_Number *y) {  /* 1.8.7 */
  int type;
  type = lua_type(L, idx);
  if (type != LUA_TPAIR) {
    luaL_error(L, "Error in " LUA_QS ": pair expected, got %s.", procname, lua_typename(L, type));
  }
  agn_pairgeti(L, idx, 1);  /* push left-hand side */
  if (lua_type(L, -1) == LUA_TNUMBER) {
    *x = agn_tonumber(L, -1);
    agn_poptop(L);  /* pop number */
  } else {
    int typeop = lua_type(L, -1);
    agn_poptop(L);  /* pop non-number */
    luaL_error(L, "Error in " LUA_QS ": number expected in left operand of the pair, got %s.", procname,
      lua_typename(L, typeop));
  }
  agn_pairgeti(L, idx, 2);  /* push right-hand side */
  if (lua_type(L, -1) == LUA_TNUMBER) {
    *y = agn_tonumber(L, -1);
    agn_poptop(L);  /* pop number */
  } else {
    int typeop = lua_type(L, -1);
    agn_poptop(L);  /* pop non-number */
    luaL_error(L, "Error in " LUA_QS ": number expected in right operand of the pair, got %s.", procname,
      lua_typename(L, typeop));
  }
  lua_remove(L, idx);  /* remove pair at idx */
}


/* Allocates size bytes of memory and returns a pointer to the newly allocated block. In case memory could not be
   allocated, it returns an error message including procname that called agn_malloc. The function optionally can
   free one or more objects referenced by their pointers in case memory allocation failed.

   In all cases, the last argument must be NULL. */

LUA_API void *agn_malloc (lua_State *L, size_t size, const char *procname, ...) {  /* 1.9.1 */
  void *ptr;
  ptr = malloc(size);
  if (ptr == NULL) {
    void *s;
    va_list args;
    va_start(args, procname);
    while ((s = va_arg(args, void *))) {
      xfree(s);
    }
    va_end(args);
    luaL_error(L, "Error in " LUA_QS ": failed to allocate memory.", procname);
  }
  return ptr;
}


/* Deallocates one or more blocks of memory pointed to by pointers of type void *. The last argument must be NULL. */

LUA_API void agn_free (lua_State *L, ...) {  /* 1.9.1 */
  void *ptr;
  va_list args;
  va_start(args, L);
  while ((ptr = va_arg(args, void *))) {
    xfree(ptr);
  }
  va_end(args);
}


LUA_API void agn_arraytoseq (lua_State *L, lua_Number *a, size_t n) {  /* converts an array to a sequence and pushes it on the top of the stack */
  size_t i;
  agn_createseq(L, n);
  for (i=0; i < n; i++)
    lua_seqsetinumber(L, -1, i+1, a[i]);
}

