/*
** $Id: lfractals.c, initiated June 15, 2008 $
** Fractals library
** See Copyright Notice in agena.h
*/

/* for C99 only */

#ifndef PROPCMPLX

#include <stdlib.h>
#include <math.h>

#define fractals_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "agncmpt.h"


#define AGENA_FRACTLIBNAME "fractals"
LUALIB_API int (luaopen_fractals) (lua_State *L);

static int fractals_mandel (lua_State *L) {
  lua_Number p, q, x, y, a, r, radius;
  int i, iter;
  p = agn_checknumber(L, 1);
  q = agn_checknumber(L, 2);
  x = p; y = q;
  iter = (int)agn_checknumber(L, 3);
  radius = agn_checknumber(L, 4);
  a = sqrt(p*p + q*q);
  for (i=0; i < iter && a < radius; i++) {
    r = p;
    p = p*p - q*q + x;
    q = 2*r*q + y;
    a = sqrt(p*p + q*q);
  }
  lua_pushnumber(L, i);
  return 1;
}


/* lambdasin */

static int fractals_lsin (lua_State *L) {
  lua_Number p, q, a, t3, t6, radius;  /* 1.12.9 */
  int i, iter;
  p = agn_checknumber(L, 1);
  q = agn_checknumber(L, 2);
  iter = (int)agn_checknumber(L, 3);
  radius = agn_checknumber(L, 4);
  a = sqrt(p*p+q*q);
  for (i=0; i < iter && a < radius; i++) {
    t3 = sin(p)*cosh(q);
    t6 = cos(p)*sinh(q);
    p = 1.0*t3-0.4*t6;
    q = 0.4*t3+1.0*t6;
    a = sqrt(p*p + q*q);
  }
  lua_pushnumber(L, i);
  return 1;
}


/* The fractal properties of Netwon's method to calculate complex roots were first
   discovered by John Hubbard of Orsay, France. */

static int fractals_newton (lua_State *L) {
  lua_Number p, oldp, q, a, t1, t2, t3, t5, t6, t7, t8, t11, t15, t17, radius;
  int i, iter;
  p = agn_checknumber(L, 1);
  q = agn_checknumber(L, 2);
  iter = (int)agn_checknumber(L, 3);
  radius = agn_checknumber(L, 4);
  t1 = p*p;
  t2 = t1*t1;
  t3 = q*q;
  t7 = t3*t3;
  a = sqrt(t2*t1+3.0*t2*t3-2.0*t1*p+3.0*t1*t7+6.0*p*t3+1.0+t7*t3);
  for (i=0; i < iter && a >= radius; i++) {
    t1 = p*p;
    t3 = q*q;
    t5 = -t1*p/3+p*t3+1.0/3.0;
    t6 = t1-t3;
    t8 = t6*t6;
    t11 = 1/(t8+4.0*t1*t3);
    t15 = -t1*q+t3*q/3;
    t17 = q*t11;
    oldp = p;
    p = p+t5*t6*t11+2.0*t15*p*t17;
    q = q+t15*t6*t11-2.0*t5*oldp*t17;
    t1 = p*p;
    t2 = t1*t1;
    t3 = q*q;
    t7 = t3*t3;
    a = sqrt(t2*t1+3.0*t2*t3-2.0*t1*p+3.0*t1*t7+6.0*p*t3+1.0+t7*t3);
  }
  lua_pushnumber(L, i);
  return 1;
}

/* Complex Mark's Mandelbrot set, using complex C arithmetic, July 21, 2008/June 06, 2009 */

static int fractals_markmandel (lua_State *L) {
  lua_Number a, b, r, m, n, radius, t1, t2, t4, t5, t7, t9, t11, t12, t13, t16;
  int i, iter;
  a = agn_checknumber(L, 1);
  b = agn_checknumber(L, 2);
  m = a; n = b;
  t4 = m*m;
  t5 = n*n;
  t7 = log(t4+t5);
  t9 = exp(0.5E-1*t7);
  t11 = atan2(n,m);
  t12 = 0.1*t11;
  t13 = cos(t12);
  t16 = sin(t12);
  iter = (int)agn_checknumber(L, 3);
  radius = agn_checknumber(L, 4);
  for (i=0; i < iter && sqrt(a*a+b*b) < radius; i++) {
    t1 = a*a;
    t2 = b*b;
    r = a;
    a = (t1-t2)*t9*t13-2.0*a*b*t9*t16+m;
    b = 2.0*r*b*t9*t13+(t1-t2)*t9*t16+n;
  }
  lua_pushnumber(L, i);
  return 1;
}


static int fractals_esctime (lua_State *L) {
  agn_Complex z, c;
  size_t i, iter;  /* 1.9.3 */
  lua_Number r;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  z = agn_checknumber(L, 2)+I*agn_checknumber(L, 3);
  c = z;
  iter = agn_checknumber(L, 4);  /* number of iterations, 1.9.3 */
  r = agnL_optnumber(L, 5, 2);   /* radius, default is 2 */
  for (i=0; i < iter && cabs(z) < r; i++) {
    lua_pushvalue(L, 1);
    agn_createcomplex(L, z);
    agn_createcomplex(L, c);
    z = agn_ccall(L, 2, 1);
  }
  lua_pushnumber(L, i);
  return 1;
}


static int fractals_lbea (lua_State *L) {
  lua_Number p, q, a, t3, t6, radius, pntre, pntim;  /* 1.12.9 */
  agn_Complex pnt;
  int i, iter;
  p = agn_checknumber(L, 1);
  q = agn_checknumber(L, 2);
  iter = (int)agn_checknumber(L, 3);
  radius = agn_checknumber(L, 4);
  pnt = agn_optcomplex(L, 5, 1+I*0.4);
  pntre = creal(pnt);
  pntim = cimag(pnt);
  a = sqrt(p*p+q*q);
  for (i=0; i < iter && a < radius; i++) {
    t3 = sin(p)*sinh(q);
    t6 = cos(p)*cosh(q);
    p = pntre*t3-pntim*t6;
    q = pntim*t3+pntre*t6;
    a = sqrt(p*p + q*q);
  }
  lua_pushnumber(L, i);
  return 1;
}

/* probably mathematically meaningless function, but creates beautiful fractals */

static int fractals_bea (lua_State *L) {
  agn_Complex z;
  lua_Number x, y;
  z = agn_checkcomplex(L, 1);
  x = creal(z);
  y = cimag(z);
  agn_createcomplex(L, sin(x)*sinh(y)+I*cos(x)*cosh(y));
  return 1;
}

/* FRACTINT's buggy cos function till v16, creates beautiful fractals */

static int fractals_cosxx (lua_State *L) {
  agn_Complex z;
  lua_Number x, y;
  z = agn_checkcomplex(L, 1);
  x = creal(z);
  y = cimag(z);
  agn_createcomplex(L, cos(x)*cosh(y)+I*sin(x)*sinh(y));
  return 1;
}


/* FRACTINT's flip function clone; swaps real and imaginary parts, 1.12.7 */
static int fractals_flip (lua_State *L) {
  agn_Complex z;
  lua_Number x, y;
  z = agn_checkcomplex(L, 1);
  x = creal(z);
  y = cimag(z);
  agn_createcomplex(L, y+I*x);
  return 1;
}


static const luaL_Reg fractlib[] = {
  {"bea", fractals_bea},                    /* added July 03, 2008 */
  {"cosxx", fractals_cosxx},                /* added on June 22, 2008 */
  {"esctime", fractals_esctime},            /* added June 20, 2008 */
  {"flip", fractals_flip},                  /* added August 25, 2013 */
  {"lbea", fractals_lbea},                  /* added July 03, 2008 */
  {"lsin", fractals_lsin},                  /* added June 15, 2008 */
  {"markmandel", fractals_markmandel},      /* added on July 21, 2008 */
  {"mandel", fractals_mandel},              /* added June 15, 2008 */
  {"newton", fractals_newton},              /* added June 15, 2008 */
  {NULL, NULL}
};

/*
** Open fractals library
*/
LUALIB_API int luaopen_fractals (lua_State *L) {
  luaL_register(L, AGENA_FRACTLIBNAME, fractlib);
  return 1;
}

#endif
