/* Cephes mathematical functions

   taken from various C files written by Stephen L. Moshier.

   Cephes Math Library, Copyright by Stephen L. Moshier
   Direct inquiries to 30 Frost Street, Cambridge, MA 02140 */


#include <math.h>

#ifndef PROPCMPLX
#include <complex.h>
#endif

#define cephes_c
#define AGENA_LIB

#include "agnconf.h"
#include "cephes.h"
#include "agncmpt.h"

static double LOGPI = 1.14472988584940017414;

/*							polevl.c
 *							p1evl.c
 *
 *	Evaluate polynomial
 *
 *
 *
 * SYNOPSIS:
 *
 * int N;
 * double x, y, coef[N+1], polevl[];
 *
 * y = polevl( x, coef, N );
 *
 *
 *
 * DESCRIPTION:
 *
 * Evaluates polynomial of degree N:
 *
 *                     2          N
 * y  =  C  + C x + C x  +...+ C x
 *        0    1     2          N
 *
 * Coefficients are stored in reverse order:
 *
 * coef[0] = C  , ..., coef[N] = C  .
 *            N                   0
 *
 *  The function p1evl() assumes that coef[N] = 1.0 and is
 * omitted from the array.  Its calling arguments are
 * otherwise the same as polevl().
 *
 *
 * SPEED:
 *
 * In the interest of speed, there are no checks for out
 * of bounds arithmetic.  This routine is used by most of
 * the functions in the library.  Depending on available
 * equipment features, the user may wish to rewrite the
 * program in microcode or assembly language.
 *
 */

/*
Cephes Math Library Release 2.1:  December, 1988
Copyright 1984, 1987, 1988 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/


double polevl (double x, double coef[], int N) {
  double ans;
  int i;
  double *p;
  p = coef;
  ans = *p++;
  i = N;
  do
    ans = ans*x + *p++;
  while(--i);
  return ans;
}

/*							p1evl()	*/
/*                                          N
 * Evaluate polynomial when coefficient of x  is 1.0.
 * Otherwise same as polevl.
 */

double p1evl (double x, double coef[], int N) {
  double ans;
  double *p;
  int i;
  p = coef;
  ans = x + *p++;
  i = N-1;
  do
    ans = ans * x + *p++;
  while(--i);
  return ans;
}


/*							ei.c
 *
 *	Exponential integral
 *
 *
 * SYNOPSIS:
 *
 * double x, y, ei();
 *
 * y = ei( x );
 *
 *
 *
 * DESCRIPTION:
 *
 *               x
 *                -     t
 *               | |   e
 *    Ei(x) =   -|-   ---  dt .
 *             | |     t
 *              -
 *             -inf
 *
 * Not defined for x <= 0.
 * See also expn.c.
 *
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE       0,100       50000      8.6e-16     1.3e-16
 *
 */

/*
Cephes Math Library Release 2.8:  May, 1999
Copyright 1999 by Stephen L. Moshier
*/

/* 0 < x <= 2
   Ei(x) - EUL - ln(x) = x A(x)/B(x)
   Theoretical peak relative error 9.73e-18  */
static double A[6] = {
  -5.350447357812542947283E0,
   2.185049168816613393830E2,
  -4.176572384826693777058E3,
   5.541176756393557601232E4,
  -3.313381331178144034309E5,
   1.592627163384945414220E6,
};

static double B[6] = {
  /*  1.000000000000000000000E0, */
  -5.250547959112862969197E1,
   1.259616186786790571525E3,
  -1.756549581973534652631E4,
   1.493062117002725991967E5,
  -7.294949239640527645655E5,
   1.592627163384945429726E6,
};


/* 8 <= x <= 20
   x exp(-x) Ei(x) - 1 = 1/x R(1/x)
   Theoretical peak absolute error = 1.07e-17  */
static double A2[10] = {
-2.106934601691916512584E0,
 1.732733869664688041885E0,
-2.423619178935841904839E-1,
 2.322724180937565842585E-2,
 2.372880440493179832059E-4,
-8.343219561192552752335E-5,
 1.363408795605250394881E-5,
-3.655412321999253963714E-7,
 1.464941733975961318456E-8,
 6.176407863710360207074E-10,
};
static double B2[9] = {
  /* 1.000000000000000000000E0, */
-2.298062239901678075778E-1,
 1.105077041474037862347E-1,
-1.566542966630792353556E-2,
 2.761106850817352773874E-3,
-2.089148012284048449115E-4,
 1.708528938807675304186E-5,
-4.459311796356686423199E-7,
 1.394634930353847498145E-8,
 6.150865933977338354138E-10,
};

/* x > 20
   x exp(-x) Ei(x) - 1  =  1/x A3(1/x)/B3(1/x)
   Theoretical absolute error = 6.15e-17  */
static double A3[9] = {
-7.657847078286127362028E-1,
 6.886192415566705051750E-1,
-2.132598113545206124553E-1,
 3.346107552384193813594E-2,
-3.076541477344756050249E-3,
 1.747119316454907477380E-4,
-6.103711682274170530369E-6,
 1.218032765428652199087E-7,
-1.086076102793290233007E-9,
};
static double B3[9] = {
  /* 1.000000000000000000000E0, */
-1.888802868662308731041E0,
 1.066691687211408896850E0,
-2.751915982306380647738E-1,
 3.930852688233823569726E-2,
-3.414684558602365085394E-3,
 1.866844370703555398195E-4,
-6.345146083130515357861E-6,
 1.239754287483206878024E-7,
-1.086076102793126632978E-9,
};

/* 16 <= x <= 32
   x exp(-x) Ei(x) - 1  =  1/x A4(1/x) / B4(1/x)
   Theoretical absolute error = 1.22e-17  */
static double A4[8] = {
-2.458119367674020323359E-1,
-1.483382253322077687183E-1,
 7.248291795735551591813E-2,
-1.348315687380940523823E-2,
 1.342775069788636972294E-3,
-7.942465637159712264564E-5,
 2.644179518984235952241E-6,
-4.239473659313765177195E-8,
};
static double B4[8] = {
  /* 1.000000000000000000000E0, */
-1.044225908443871106315E-1,
-2.676453128101402655055E-1,
 9.695000254621984627876E-2,
-1.601745692712991078208E-2,
 1.496414899205908021882E-3,
-8.462452563778485013756E-5,
 2.728938403476726394024E-6,
-4.239462431819542051337E-8,
};

/* 4 <= x <= 8
   x exp(-x) Ei(x) - 1  =  1/x A5(1/x) / B5(1/x)
   Theoretical absolute error = 2.20e-17  */
static double A5[8] = {
  -1.373215375871208729803E0,
  -7.084559133740838761406E-1,
   1.580806855547941010501E0,
  -2.601500427425622944234E-1,
   2.994674694113713763365E-2,
  -1.038086040188744005513E-3,
   4.371064420753005429514E-5,
   2.141783679522602903795E-6,
};

static double B5[8] = {
  /* 1.000000000000000000000E0, */
   8.585231423622028380768E-1,
   4.483285822873995129957E-1,
   7.687932158124475434091E-2,
   2.449868241021887685904E-2,
   8.832165941927796567926E-4,
   4.590952299511353531215E-4,
  -4.729848351866523044863E-6,
   2.665195537390710170105E-6,
};
/* 2 <= x <= 4
   x exp(-x) Ei(x) - 1  =  1/x A6(1/x) / B6(1/x)
   Theoretical absolute error = 4.89e-17  */
static double A6[8] = {
   1.981808503259689673238E-2,
  -1.271645625984917501326E0,
  -2.088160335681228318920E0,
   2.755544509187936721172E0,
  -4.409507048701600257171E-1,
   4.665623805935891391017E-2,
  -1.545042679673485262580E-3,
   7.059980605299617478514E-5,
};

static double B6[7] = {
  /* 1.000000000000000000000E0, */
   1.476498670914921440652E0,
   5.629177174822436244827E-1,
   1.699017897879307263248E-1,
   2.291647179034212017463E-2,
   4.450150439728752875043E-3,
   1.727439612206521482874E-4,
   3.953167195549672482304E-5,
};
/* 32 <= x <= 64
   x exp(-x) Ei(x) - 1  =  1/x A7(1/x) / B7(1/x)
   Theoretical absolute error = 7.71e-18  */
static double A7[6] = {
   1.212561118105456670844E-1,
  -5.823133179043894485122E-1,
   2.348887314557016779211E-1,
  -3.040034318113248237280E-2,
   1.510082146865190661777E-3,
  -2.523137095499571377122E-5,
};

static double B7[5] = {
  /* 1.000000000000000000000E0, */
  -1.002252150365854016662E0,
   2.928709694872224144953E-1,
  -3.337004338674007801307E-2,
   1.560544881127388842819E-3,
  -2.523137093603234562648E-5,
};

double ei (double x) {
  double f, w;
  if (x <= 0.0) return AGN_NAN;
  else if (x < 2.0) {
    /* Power series.
                            inf    n
                             -    x
     Ei(x) = EUL + ln x  +   >   ----
                             -   n n!
                            n=1
    */
    f = polevl(x, A, 5)/p1evl(x, B, 6);
    /*      f = polevl(x, A, 6)/p1evl(x, B, 7); */
    /*      f = polevl(x, A, 8)/p1evl(x, B, 9); */
    return EUL + log(x) + x * f;
  }
  else if (x < 4.0) {
    /* Asymptotic expansion.
                            1       2       6
    x exp(-x) Ei(x) =  1 + ---  +  ---  +  ---- + ...
                            x        2       3
                                    x       x
    */
    w = 1.0/x;
    f = polevl(w, A6, 7)/p1evl(w, B6, 7);
    return exp(x)*w*(1.0 + w*f);
  }
  else if (x < 8.0) {
    w = 1.0/x;
    f = polevl(w, A5, 7) / p1evl(w, B5, 8);
    return exp(x)*w*(1.0 + w*f);
  }
  else if (x < 16.0) {
    w = 1.0/x;
    f = polevl(w, A2, 9)/p1evl(w, B2, 9);
    return exp(x)*w*(1.0 + w*f);
  }
  else if (x < 32.0) {
    w = 1.0/x;
    f = polevl(w, A4, 7) / p1evl(w, B4, 8);
    return exp(x)*w*(1.0 + w*f);
  }
  else if (x < 64.0) {
    w = 1.0/x;
    f = polevl(w,A7,5) / p1evl(w,B7,5);
    return exp(x)*w*(1.0 + w*f);
  } else {
    w = 1.0/x;
    f = polevl(w,A3,8) / p1evl(w,B3,9);
    return exp(x)*w*(1.0 + w*f);
  }
}


/*							sici.c
 *
 *	Sine and cosine integrals
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, Ci, Si, sici();
 *
 * sici( x, &Si, &Ci );
 *
 *
 * DESCRIPTION:
 *
 * Evaluates the integrals
 *
 *                          x
 *                          -
 *                         |  cos t - 1
 *   Ci(x) = eul + ln x +  |  --------- dt,
 *                         |      t
 *                        -
 *                         0
 *             x
 *             -
 *            |  sin t
 *   Si(x) =  |  ----- dt
 *            |    t
 *           -
 *            0
 *
 * where eul = 0.57721566490153286061 is Euler's constant.
 * The integrals are approximated by rational functions.
 * For x > 8 auxiliary functions f(x) and g(x) are employed
 * such that
 *
 * Ci(x) = f(x) sin(x) - g(x) cos(x)
 * Si(x) = pi/2 - f(x) cos(x) - g(x) sin(x)
 *
 *
 * ACCURACY:
 *    Test interval = [0,50].
 * Absolute error, except relative when > 1:
 * arithmetic   function   # trials      peak         rms
 *    IEEE        Si        30000       4.4e-16     7.3e-17
 *    IEEE        Ci        30000       6.9e-16     5.1e-17
 *    DEC         Si         5000       4.4e-17     9.0e-18
 *    DEC         Ci         5300       7.9e-17     5.2e-18
 */

/*
Cephes Math Library Release 2.1:  January, 1989
Copyright 1984, 1987, 1989 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

static double SN[] = {
  -8.39167827910303881427E-11,
   4.62591714427012837309E-8,
  -9.75759303843632795789E-6,
   9.76945438170435310816E-4,
  -4.13470316229406538752E-2,
   1.00000000000000000302E0,
};

static double SD[] = {
   2.03269266195951942049E-12,
   1.27997891179943299903E-9,
   4.41827842801218905784E-7,
   9.96412122043875552487E-5,
   1.42085239326149893930E-2,
   9.99999999999999996984E-1,
};

static double CN[] = {
   2.02524002389102268789E-11,
  -1.35249504915790756375E-8,
   3.59325051419993077021E-6,
  -4.74007206873407909465E-4,
   2.89159652607555242092E-2,
  -1.00000000000000000080E0,
};

static double CD[] = {
   4.07746040061880559506E-12,
   3.06780997581887812692E-9,
   1.23210355685883423679E-6,
   3.17442024775032769882E-4,
   5.10028056236446052392E-2,
   4.00000000000000000080E0,
};

static double FN4[] = {
   4.23612862892216586994E0,
   5.45937717161812843388E0,
   1.62083287701538329132E0,
   1.67006611831323023771E-1,
   6.81020132472518137426E-3,
   1.08936580650328664411E-4,
   5.48900223421373614008E-7,
};

static double FD4[] = {
/*  1.00000000000000000000E0,*/
   8.16496634205391016773E0,
   7.30828822505564552187E0,
   1.86792257950184183883E0,
   1.78792052963149907262E-1,
   7.01710668322789753610E-3,
   1.10034357153915731354E-4,
   5.48900252756255700982E-7,
};

static double FN8[] = {
   4.55880873470465315206E-1,
   7.13715274100146711374E-1,
   1.60300158222319456320E-1,
   1.16064229408124407915E-2,
   3.49556442447859055605E-4,
   4.86215430826454749482E-6,
   3.20092790091004902806E-8,
   9.41779576128512936592E-11,
   9.70507110881952024631E-14,
};

static double FD8[] = {
/*  1.00000000000000000000E0,*/
   9.17463611873684053703E-1,
   1.78685545332074536321E-1,
   1.22253594771971293032E-2,
   3.58696481881851580297E-4,
   4.92435064317881464393E-6,
   3.21956939101046018377E-8,
   9.43720590350276732376E-11,
   9.70507110881952025725E-14,
};

static double GN4[] = {
   8.71001698973114191777E-2,
   6.11379109952219284151E-1,
   3.97180296392337498885E-1,
   7.48527737628469092119E-2,
   5.38868681462177273157E-3,
   1.61999794598934024525E-4,
   1.97963874140963632189E-6,
   7.82579040744090311069E-9,
};

static double GD4[] = {
/*  1.00000000000000000000E0,*/
   1.64402202413355338886E0,
   6.66296701268987968381E-1,
   9.88771761277688796203E-2,
   6.22396345441768420760E-3,
   1.73221081474177119497E-4,
   2.02659182086343991969E-6,
   7.82579218933534490868E-9,
};

static double GN8[] = {
   6.97359953443276214934E-1,
   3.30410979305632063225E-1,
   3.84878767649974295920E-2,
   1.71718239052347903558E-3,
   3.48941165502279436777E-5,
   3.47131167084116673800E-7,
   1.70404452782044526189E-9,
   3.85945925430276600453E-12,
   3.14040098946363334640E-15,
};

static double GD8[] = {
/*  1.00000000000000000000E0,*/
   1.68548898811011640017E0,
   4.87852258695304967486E-1,
   4.67913194259625806320E-2,
   1.90284426674399523638E-3,
   3.68475504442561108162E-5,
   3.57043223443740838771E-7,
   1.72693748966316146736E-9,
   3.87830166023954706752E-12,
   3.14040098946363335242E-15,
};

void sici (double x, double *si, double *ci) {
  double z, c, s, f, g;
  short sign;
  if (x < 0.0) {
    sign = -1;
    x = -x;
  } else
    sign = 0;
  if (x == 0.0) {
    *si = 0.0;
    *ci = AGN_NAN;
    return;
  }
  if (x > 1.0e9) {
	*si = PIO2 - cos(x)/x;
	*ci = sin(x)/x;
	return;
  }
  if (x > 4.0) goto asympt;
  z = x * x;
  s = x * polevl(z, SN, 5) / polevl(z, SD, 5);
  c = z * polevl(z, CN, 5) / polevl(z, CD, 5);
  if (sign) s = -s;
  *si = s;
  *ci = EUL + log(x) + c;	/* real part if x < 0 */
  return;

/* The auxiliary functions are:
 *
 *
 * *si = *si - PIO2;
 * c = cos(x);
 * s = sin(x);
 *
 * t = *ci * s - *si * c;
 * a = *ci * c + *si * s;
 *
 * *si = t;
 * *ci = -a;
 */

asympt:

  s = sin(x);
  c = cos(x);
  z = 1.0/(x*x);
  if (x < 8.0) {
    f = polevl(z, FN4, 6) / (x * p1evl(z, FD4, 7));
    g = z * polevl(z, GN4, 7) / p1evl(z, GD4, 7);
  } else {
    f = polevl(z, FN8, 8) / (x * p1evl(z, FD8, 8));
    g = z * polevl(z, GN8, 8) / p1evl(z, GD8, 9);
  }
  *si = PIO2 - f * c - g * s;
  if (sign) *si = -(*si);
  *ci = f * s - g * c;
  return;
}

/*							chbevl.c
 *
 *	Evaluate Chebyshev series
 *
 *
 *
 * SYNOPSIS:
 *
 * int N;
 * double x, y, coef[N], chebevl();
 *
 * y = chbevl( x, coef, N );
 *
 *
 *
 * DESCRIPTION:
 *
 * Evaluates the series
 *
 *        N-1
 *         - '
 *  y  =   >   coef[i] T (x/2)
 *         -            i
 *        i=0
 *
 * of Chebyshev polynomials Ti at argument x/2.
 *
 * Coefficients are stored in reverse order, i.e. the zero
 * order term is last in the array.  Note N is the number of
 * coefficients, not the order.
 *
 * If coefficients are for the interval a to b, x must
 * have been transformed to x -> 2(2x - b - a)/(b-a) before
 * entering the routine.  This maps x from (a, b) to (-1, 1),
 * over which the Chebyshev polynomials are defined.
 *
 * If the coefficients are for the inverted interval, in
 * which (a, b) is mapped to (1/b, 1/a), the transformation
 * required is x -> 2(2ab/x - b - a)/(b-a).  If b is infinity,
 * this becomes x -> 4a/x - 1.
 *
 *
 *
 * SPEED:
 *
 * Taking advantage of the recurrence properties of the
 * Chebyshev polynomials, the routine requires one more
 * addition per loop than evaluating a nested polynomial of
 * the same degree.
 *
 */

/*							chbevl.c	*/

/*
Cephes Math Library Release 2.0:  April, 1987
Copyright 1985, 1987 by Stephen L. Moshier
Direct inquiries to 30 Frost Street, Cambridge, MA 02140
*/

double chbevl (double x, double array[], int n) {
  double b0, b1, b2, *p;
  int i;
  p = array;
  b0 = *p++;
  b1 = 0.0;
  i = n - 1;
  do {
    b2 = b1;
    b1 = b0;
    b0 = x*b1 - b2 + *p++;
  } while( --i );
  return 0.5*(b0-b2);
}

/*							shichi.c
 *
 *	Hyperbolic sine and cosine integrals
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, Chi, Shi, shichi();
 *
 * shichi( x, &Chi, &Shi );
 *
 *
 * DESCRIPTION:
 *
 * Approximates the integrals
 *
 *                            x
 *                            -
 *                           | |   cosh t - 1
 *   Chi(x) = eul + ln x +   |    -----------  dt,
 *                         | |          t
 *                          -
 *                          0
 *
 *               x
 *               -
 *              | |  sinh t
 *   Shi(x) =   |    ------  dt
 *            | |       t
 *             -
 *             0
 *
 * where eul = 0.57721566490153286061 is Euler's constant.
 * The integrals are evaluated by power series for x < 8
 * and by Chebyshev expansions for x between 8 and 88.
 * For large x, both functions approach exp(x)/2x.
 * Arguments greater than 88 in magnitude return MAXNUM.
 *
 *
 * ACCURACY:
 *
 * Test interval 0 to 88.
 *                      Relative error:
 * arithmetic   function  # trials      peak         rms
 *    DEC          Shi       3000       9.1e-17
 *    IEEE         Shi      30000       6.9e-16     1.6e-16
 *        Absolute error, except relative when |Chi| > 1:
 *    DEC          Chi       2500       9.3e-17
 *    IEEE         Chi      30000       8.4e-16     1.4e-16
 */

/*
Cephes Math Library Release 2.8:  June, 2000
Copyright 1984, 1987, 2000 by Stephen L. Moshier
*/

/* x exp(-x) shi(x), inverted interval 8 to 18 */
static double S1[] = {
   1.83889230173399459482E-17,
  -9.55485532279655569575E-17,
   2.04326105980879882648E-16,
   1.09896949074905343022E-15,
  -1.31313534344092599234E-14,
   5.93976226264314278932E-14,
  -3.47197010497749154755E-14,
  -1.40059764613117131000E-12,
   9.49044626224223543299E-12,
  -1.61596181145435454033E-11,
  -1.77899784436430310321E-10,
   1.35455469767246947469E-9,
  -1.03257121792819495123E-9,
  -3.56699611114982536845E-8,
   1.44818877384267342057E-7,
   7.82018215184051295296E-7,
  -5.39919118403805073710E-6,
  -3.12458202168959833422E-5,
   8.90136741950727517826E-5,
   2.02558474743846862168E-3,
   2.96064440855633256972E-2,
   1.11847751047257036625E0
};

/* x exp(-x) shi(x), inverted interval 18 to 88 */
static double S2[] = {
  -1.05311574154850938805E-17,
   2.62446095596355225821E-17,
   8.82090135625368160657E-17,
  -3.38459811878103047136E-16,
  -8.30608026366935789136E-16,
   3.93397875437050071776E-15,
   1.01765565969729044505E-14,
  -4.21128170307640802703E-14,
  -1.60818204519802480035E-13,
   3.34714954175994481761E-13,
   2.72600352129153073807E-12,
   1.66894954752839083608E-12,
  -3.49278141024730899554E-11,
  -1.58580661666482709598E-10,
  -1.79289437183355633342E-10,
   1.76281629144264523277E-9,
   1.69050228879421288846E-8,
   1.25391771228487041649E-7,
   1.16229947068677338732E-6,
   1.61038260117376323993E-5,
   3.49810375601053973070E-4,
   1.28478065259647610779E-2,
   1.03665722588798326712E0
};

/* x exp(-x) chin(x), inverted interval 8 to 18 */
static double C1[] = {
  -8.12435385225864036372E-18,
   2.17586413290339214377E-17,
   5.22624394924072204667E-17,
  -9.48812110591690559363E-16,
   5.35546311647465209166E-15,
  -1.21009970113732918701E-14,
  -6.00865178553447437951E-14,
   7.16339649156028587775E-13,
  -2.93496072607599856104E-12,
  -1.40359438136491256904E-12,
   8.76302288609054966081E-11,
  -4.40092476213282340617E-10,
  -1.87992075640569295479E-10,
   1.31458150989474594064E-8,
  -4.75513930924765465590E-8,
  -2.21775018801848880741E-7,
   1.94635531373272490962E-6,
   4.33505889257316408893E-6,
  -6.13387001076494349496E-5,
  -3.13085477492997465138E-4,
   4.97164789823116062801E-4,
   2.64347496031374526641E-2,
   1.11446150876699213025E0
};

/* x exp(-x) chin(x), inverted interval 18 to 88 */
static double C2[] = {
   8.06913408255155572081E-18,
  -2.08074168180148170312E-17,
  -5.98111329658272336816E-17,
   2.68533951085945765591E-16,
   4.52313941698904694774E-16,
  -3.10734917335299464535E-15,
  -4.42823207332531972288E-15,
   3.49639695410806959872E-14,
   6.63406731718911586609E-14,
  -3.71902448093119218395E-13,
  -1.27135418132338309016E-12,
   2.74851141935315395333E-12,
   2.33781843985453438400E-11,
   2.71436006377612442764E-11,
  -2.56600180000355990529E-10,
  -1.61021375163803438552E-9,
  -4.72543064876271773512E-9,
  -3.00095178028681682282E-9,
   7.79387474390914922337E-8,
   1.06942765566401507066E-6,
   1.59503164802313196374E-5,
   3.49592575153777996871E-4,
   1.28475387530065247392E-2,
   1.03665693917934275131E0
  };

/* Sine and cosine integrals */

void shichi (double x, double *si, double *ci) {
  double k, z, c, s, a;
  short sign;
  if (x < 0.0) {
    sign = -1;
    x = -x;
  }
  else
    sign = 0;
  if (x == 0.0) {
    *si = 0.0;
    *ci = AGN_NAN;
	return;
  }
  if (x >= 8.0) goto chb;
  z = x * x;
  /* Direct power series expansion */
  a = 1.0;
  s = 1.0;
  c = 0.0;
  k = 2.0;
  do {
    a *= z/k;
    c += a/k;
    k += 1.0;
    a /= k;
    s += a/k;
    k += 1.0;
  } while(fabs(a/s) > MACHEP);
  s *= x;
  goto done;

chb:
  if (x < 18.0) {
    a = (576.0/x - 52.0)/10.0;
    k = exp(x) / x;
    s = k * chbevl(a, S1, 22);
    c = k * chbevl(a, C1, 23);
    goto done;
  }
  if (x <= 88.0) {
    a = (6336.0/x - 212.0)/70.0;
    k = exp(x) / x;
    s = k * chbevl(a, S2, 23);
    c = k * chbevl(a, C2, 24);
    goto done;
  } else {
    if (sign) *si = -HUGE_VAL;
	else *si = HUGE_VAL;
    *ci = HUGE_VAL;
    return;
  }

done:
  if (sign) s = -s;
  *si = s;
  *ci = EUL + log(x) + c;
}

/*							dawsn.c
 *
 *	Dawson's Integral
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, dawsn();
 *
 * y = dawsn( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Approximates the integral
 *
 *                             x
 *                             -
 *                      2     | |        2
 *  dawsn(x)  =  exp( -x  )   |    exp( t  ) dt
 *                          | |
 *                           -
 *                           0
 *
 * Three different rational approximations are employed, for
 * the intervals 0 to 3.25; 3.25 to 6.25; and 6.25 up.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      0,10        10000       6.9e-16     1.0e-16
 *    DEC       0,10         6000       7.4e-17     1.4e-17
 *
 *
 */

/*							dawsn.c */


/*
Cephes Math Library Release 2.8:  June, 2000
Copyright 1984, 1987, 1989, 2000 by Stephen L. Moshier
*/

/* Dawson's integral, interval 0 to 3.25 */
static double An[10] = {
   1.13681498971755972054E-11,
   8.49262267667473811108E-10,
   1.94434204175553054283E-8,
   9.53151741254484363489E-7,
   3.07828309874913200438E-6,
   3.52513368520288738649E-4,
  -8.50149846724410912031E-4,
   4.22618223005546594270E-2,
  -9.17480371773452345351E-2,
   9.99999999999999994612E-1,
};

static double Ad[11] = {
   2.40372073066762605484E-11,
   1.48864681368493396752E-9,
   5.21265281010541664570E-8,
   1.27258478273186970203E-6,
   2.32490249820789513991E-5,
   3.25524741826057911661E-4,
   3.48805814657162590916E-3,
   2.79448531198828973716E-2,
   1.58874241960120565368E-1,
   5.74918629489320327824E-1,
   1.00000000000000000539E0,
};

/* interval 3.25 to 6.25 */
static double Bn[11] = {
   5.08955156417900903354E-1,
  -2.44754418142697847934E-1,
   9.41512335303534411857E-2,
  -2.18711255142039025206E-2,
   3.66207612329569181322E-3,
  -4.23209114460388756528E-4,
   3.59641304793896631888E-5,
  -2.14640351719968974225E-6,
   9.10010780076391431042E-8,
  -2.40274520828250956942E-9,
   3.59233385440928410398E-11,
};

static double Bd[10] = {
  /*  1.00000000000000000000E0,*/
  -6.31839869873368190192E-1,
   2.36706788228248691528E-1,
  -5.31806367003223277662E-2,
   8.48041718586295374409E-3,
  -9.47996768486665330168E-4,
   7.81025592944552338085E-5,
  -4.55875153252442634831E-6,
   1.89100358111421846170E-7,
  -4.91324691331920606875E-9,
   7.18466403235734541950E-11,
};

/* 6.25 to infinity */
static double Cn[5] = {
  -5.90592860534773254987E-1,
   6.29235242724368800674E-1,
  -1.72858975380388136411E-1,
   1.64837047825189632310E-2,
  -4.86827613020462700845E-4,
};

static double Cd[5] = {
  /* 1.00000000000000000000E0,*/
  -2.69820057197544900361E0,
   1.73270799045947845857E0,
  -3.93708582281939493482E-1,
   3.44278924041233391079E-2,
  -9.73655226040941223894E-4,
};

double dawsn (double xx) {
  double x, y;
  int sign;
  sign = 1;
  if (xx < 0.0) {
    sign = -1;
    xx = -xx;
  }
  if (xx < 3.25) {
    x = xx*xx;
    y = xx*polevl(x, An, 9)/polevl(x, Ad, 10);
    return sign*y;
  }
  x = 1.0/(xx*xx);
  if (xx < 6.25) {
    y = 1.0/xx + x*polevl(x, Bn, 10)/(p1evl(x, Bd, 10)*xx);
    return sign*0.5*y;
  }
  if (xx > 1.0e9) return (sign * 0.5)/xx;
  /* 6.25 to infinity */
  y = 1.0/xx + x*polevl(x, Cn, 4)/(p1evl(x, Cd, 5)*xx);
  return sign*0.5*y;
}

/*							psi.c
 *
 *	Psi (digamma) function
 *
 *
 * SYNOPSIS:
 *
 * double x, y, psi();
 *
 * y = psi( x );
 *
 *
 * DESCRIPTION:
 *
 *              d      -
 *   psi(x)  =  -- ln | (x)
 *              dx
 *
 * is the logarithmic derivative of the gamma function.
 * For integer x,
 *                   n-1
 *                    -
 * psi(n) = -EUL  +   >  1/k.
 *                    -
 *                   k=1
 *
 * This formula is used for 0 < n <= 10.  If x is negative, it
 * is transformed to a positive argument by the reflection
 * formula  psi(1-x) = psi(x) + pi cot(pi x).
 * For general positive x, the argument is made greater than 10
 * using the recurrence  psi(x+1) = psi(x) + 1/x.
 * Then the following asymptotic expansion is applied:
 *
 *                           inf.   B
 *                            -      2k
 * psi(x) = log(x) - 1/2x -   >   -------
 *                            -        2k
 *                           k=1   2k x
 *
 * where the B2k are Bernoulli numbers.
 *
 * ACCURACY:
 *    Relative error (except absolute when |psi| < 1):
 * arithmetic   domain     # trials      peak         rms
 *    DEC       0,30         2500       1.7e-16     2.0e-17
 *    IEEE      0,30        30000       1.3e-15     1.4e-16
 *    IEEE      -30,0       40000       1.5e-15     2.2e-16
 *
 * ERROR MESSAGES:
 *     message         condition      value returned
 * psi singularity    x integer <=0      MAXNUM
 */

/*
Cephes Math Library Release 2.8:  June, 2000
Copyright 1984, 1987, 1992, 2000 by Stephen L. Moshier
*/

static double Apsi[] = {
   8.33333333333333333333E-2,
  -2.10927960927960927961E-2,
   7.57575757575757575758E-3,
  -4.16666666666666666667E-3,
   3.96825396825396825397E-3,
  -8.33333333333333333333E-3,
   8.33333333333333333333E-2
};

double psi (double x) {
  double p, q, nz, s, w, y, z;
  int i, n, negative;
  negative = 0;
  nz = 0.0;
  if (x <= 0.0) {
	 negative = 1;
    q = x;
    p = floor(q);
    if (p == q) {
      return AGN_NAN;
    }
    /* Remove the zeros of tan(PI x)
    * by subtracting the nearest integer from x
    */
    nz = q - p;
    if (nz != 0.5) {
		if (nz > 0.5)	{
        p += 1.0;
        nz = q - p;
      }
		nz = PI/tan(PI*nz);
    }
    else
      nz = 0.0;
    x = 1.0 - x;
  }
  /* check for positive integer up to 10 */
  if ((x <= 10.0) && (x == floor(x))) {
    y = 0.0;
    n = x;
    for (i=1; i < n; i++) {
      w = i;
      y += 1.0/w;
    }
    y -= EUL;
    goto done;
  }
  s = x;
  w = 0.0;
  while (s < 10.0) {
    w += 1.0/s;
    s += 1.0;
  }
  if (s < 1.0e17) {
    z = 1.0/(s*s);
    y = z*polevl(z, Apsi, 6);
  }
  else
    y = 0.0;
  y = log(s) - (0.5/s) - y - w;

done:
  if (negative) y -= nz;
  return y;
}

/*							spence.c
 *
 *	Dilogarithm
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, spence();
 *
 * y = spence( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Computes the integral
 *
 *                    x
 *                    -
 *                   | | log t
 * spence(x)  =  -   |   ----- dt
 *                 | |   t - 1
 *                  -
 *                  1
 *
 * for x >= 0.  A rational approximation gives the integral in
 * the interval (0.5, 1.5).  Transformation formulas for 1/x
 * and 1-x are employed outside the basic expansion range.
 *
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      0,4         30000       3.9e-15     5.4e-16
 *    DEC       0,4          3000       2.5e-16     4.5e-17
 *
 *
 */

/*							spence.c */


/*
Cephes Math Library Release 2.8:  June, 2000
Copyright 1985, 1987, 1989, 2000 by Stephen L. Moshier
*/

static double Asp[8] = {
  4.65128586073990045278E-5,
  7.31589045238094711071E-3,
  1.33847639578309018650E-1,
  8.79691311754530315341E-1,
  2.71149851196553469920E0,
  4.25697156008121755724E0,
  3.29771340985225106936E0,
  1.00000000000000000126E0,
};

static double Bsp[8] = {
  6.90990488912553276999E-4,
  2.54043763932544379113E-2,
  2.82974860602568089943E-1,
  1.41172597751831069617E0,
  3.63800533345137075418E0,
  5.03278880143316990390E0,
  3.54771340985225096217E0,
  9.99999999999999998740E-1,
};

double spence (double x) {
  double w, y, z;
  int flag;
  if (x < 0.0) {
    return(AGN_NAN);
  }
  if (x == 1.0)
	 return 0.0;
  if (x == 0.0)
	 return PI*PI/6.0;
  flag = 0;
  if (x > 2.0) {
    x = 1.0/x;
    flag |= 2;
  }
  if (x > 1.5) {
    w = (1.0/x) - 1.0;
    flag |= 2;
  }
  else if (x < 0.5) {
    w = -x;
    flag |= 1;
  }
  else
    w = x - 1.0;
  y = -w * polevl(w, Asp, 7) / polevl(w, Bsp, 7);
  if (flag & 1)
    y = (PI*PI)/6.0 - log(x)*log(1.0-x) - y;
  if (flag & 2) {
    z = log(x);
    y = -0.5*z*z - y;
  }
  return y;
}


/*							fresnl.c
 *
 *	Fresnel integral
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, S, C;
 * void fresnl();
 *
 * fresnl( x, _&S, _&C );
 *
 *
 * DESCRIPTION:
 *
 * Evaluates the Fresnel integrals
 *
 *           x
 *           -
 *          | |
 * C(x) =   |   cos(pi/2 t**2) dt,
 *        | |
 *         -
 *          0
 *
 *           x
 *           -
 *          | |
 * S(x) =   |   sin(pi/2 t**2) dt.
 *        | |
 *         -
 *          0
 *
 *
 * The integrals are evaluated by a power series for x < 1.
 * For x >= 1 auxiliary functions f(x) and g(x) are employed
 * such that
 *
 * C(x) = 0.5 + f(x) sin( pi/2 x**2 ) - g(x) cos( pi/2 x**2 )
 * S(x) = 0.5 - f(x) cos( pi/2 x**2 ) - g(x) sin( pi/2 x**2 )
 *
 *
 *
 * ACCURACY:
 *
 *  Relative error.
 *
 * Arithmetic  function   domain     # trials      peak         rms
 *   IEEE       S(x)      0, 10       10000       2.0e-15     3.2e-16
 *   IEEE       C(x)      0, 10       10000       1.8e-15     3.3e-16
 *   DEC        S(x)      0, 10        6000       2.2e-16     3.9e-17
 *   DEC        C(x)      0, 10        5000       2.3e-16     3.9e-17
 */

/*
Cephes Math Library Release 2.8:  June, 2000
Copyright 1984, 1987, 1989, 2000 by Stephen L. Moshier
*/

/* S(x) for small x */
static double sn[6] = {
  -2.99181919401019853726E3,
   7.08840045257738576863E5,
  -6.29741486205862506537E7,
   2.54890880573376359104E9,
  -4.42979518059697779103E10,
   3.18016297876567817986E11,
};

static double sd[6] = {
/* 1.00000000000000000000E0,*/
   2.81376268889994315696E2,
   4.55847810806532581675E4,
   5.17343888770096400730E6,
   4.19320245898111231129E8,
   2.24411795645340920940E10,
   6.07366389490084639049E11,
};

/* C(x) for small x */
static double cn[6] = {
  -4.98843114573573548651E-8,
   9.50428062829859605134E-6,
  -6.45191435683965050962E-4,
   1.88843319396703850064E-2,
  -2.05525900955013891793E-1,
   9.99999999999999998822E-1,
};

static double cd[7] = {
   3.99982968972495980367E-12,
   9.15439215774657478799E-10,
   1.25001862479598821474E-7,
   1.22262789024179030997E-5,
   8.68029542941784300606E-4,
   4.12142090722199792936E-2,
   1.00000000000000000118E0,
};

/* Auxiliary function f(x) */
  static double fn[10] = {
    4.21543555043677546506E-1,
    1.43407919780758885261E-1,
    1.15220955073585758835E-2,
    3.45017939782574027900E-4,
    4.63613749287867322088E-6,
    3.05568983790257605827E-8,
    1.02304514164907233465E-10,
    1.72010743268161828879E-13,
    1.34283276233062758925E-16,
    3.76329711269987889006E-20,
};
static double fd[10] = {
/*  1.00000000000000000000E0,*/
    7.51586398353378947175E-1,
    1.16888925859191382142E-1,
    6.44051526508858611005E-3,
    1.55934409164153020873E-4,
    1.84627567348930545870E-6,
    1.12699224763999035261E-8,
    3.60140029589371370404E-11,
    5.88754533621578410010E-14,
    4.52001434074129701496E-17,
    1.25443237090011264384E-20,
};

/* Auxiliary function g(x) */
static double gn[11] = {
    5.04442073643383265887E-1,
    1.97102833525523411709E-1,
    1.87648584092575249293E-2,
    6.84079380915393090172E-4,
    1.15138826111884280931E-5,
    9.82852443688422223854E-8,
    4.45344415861750144738E-10,
    1.08268041139020870318E-12,
    1.37555460633261799868E-15,
    8.36354435630677421531E-19,
    1.86958710162783235106E-22,
};

static double gd[11] = {
/*  1.00000000000000000000E0,*/
    1.47495759925128324529E0,
    3.37748989120019970451E-1,
    2.53603741420338795122E-2,
    8.14679107184306179049E-4,
    1.27545075667729118702E-5,
    1.04314589657571990585E-7,
    4.60680728146520428211E-10,
    1.10273215066240270757E-12,
    1.38796531259578871258E-15,
    8.39158816283118707363E-19,
    1.86958710162783236342E-22,
};


void fresnl (double xxa, double *ssa, double *cca) {
  double f, g, cc, ss, c, s, t, u, x, x2;
  x = fabs(xxa);
  x2 = x*x;
  if (x2 < 2.5625) {
    t = x2*x2;
    ss = x*x2*polevl(t, sn, 5)/p1evl(t, sd, 6);
    cc = x*polevl( t, cn, 5)/polevl(t, cd, 6 );
    goto done;
  }
  if (x > 36974.0) {
    cc = 0.5;
    ss = 0.5;
    goto done;
  }
  /*		Asymptotic power series auxiliary functions
   *		for large argument
   */
  x2 = x*x;
  t = PI*x2;
  u = 1.0/(t*t);
  t = 1.0/t;
  f = 1.0 - u*polevl(u, fn, 9)/p1evl(u, fd, 10);
  g = t*polevl( u, gn, 10)/p1evl(u, gd, 11);
  t = PIO2 * x2;
  c = cos(t);
  s = sin(t);
  t = PI*x;
  cc = 0.5 + (f*s - g*c)/t;
  ss = 0.5 - (f*c + g*s)/t;

done:
  if (xxa < 0.0) {
    cc = -cc;
    ss = -ss;
  }
  *cca = cc;
  *ssa = ss;
}

/*							cgamma
 *
 *	Complex gamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * #include <complex.h>
 * agn_Complex x, y, cgamma();
 *
 * y = cgamma( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns complex-valued gamma function of the complex argument.
 * This variable is also filled in by the logarithmic gamma
 * function clgam().
 *
 * Arguments |x| < 18 are increased by recurrence.
 * Large arguments are handled by Stirling's formula. Large negative
 * arguments are made positive using the reflection formula.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      -20,20      500000      2.0e-14     2.7e-15
 *    IEEE     -100,100     100000      1.4e-13     1.5e-14
 *
 * Error for arguments outside the test range will be larger
 * owing to error amplification by the exponential function.
 *
 */

/*							clgam
 *
 *	Natural logarithm of complex gamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * #include <complex.h>
 * agn_Complex x, y, clgam();
 *
 * y = clgam( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the base e (2.718...) logarithm of the complex gamma
 * function of the argument.
 *
 * The logarithm of the gamma function is approximated by the
 * logarithmic version of Stirling's asymptotic formula.
 * Arguments of real part less than 14 are increased by recurrence.
 * The cosecant reflection formula is employed for arguments
 * having real part less than -14.
 *
 * Arguments greater than MAXLGM return MAXNUM and an error
 * message.  MAXLGM = 2.556348e305 for IEEE arithmetic.
 *
 *
 *
 * ACCURACY:
 *
 *
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      -20,20      500000      1.4e-14     4.5e-16
 *    IEEE     -100,100     100000                  1.6e-16
 * The error criterion was relative when the function magnitude
 * was greater than one but absolute when it was less than one.
 */

/*
Cephes Math Library Release 2.7:  March, 1998
Copyright 1984, 1998 Stephen L. Moshier
*/

/* Asymptotic expansion of log gamma  */
static double Alg[] = {
-1.9175269175269175269175269175269175269175E-3,
 8.4175084175084175084175084175084175084175E-4,
-5.9523809523809523809523809523809523809524E-4,
 7.9365079365079365079365079365079365079365E-4,
-2.7777777777777777777777777777777777777778E-3,
 8.3333333333333333333333333333333333333333E-2
};

/* Logarithm of gamma function */

#ifndef PROPCMPLX
agn_Complex clgam (agn_Complex x) {
  double p, q, a;
  agn_Complex c, w, u, v;
  int i, cj;
  cj = 0;
  if (cimag(x) < 0) {
    cj = 1;
    x = conj(x);
  } else if (cimag(x) == 0.0) {  /* added A. Walz, Agena 0.32.4 */
    if (creal(x) == trunc(creal(x)) && creal(x) <= 0.0)
      return AGN_NAN;
    else if (creal(x) == 1 || creal(x) == 2)
      return 0;
  }
  /* Reflection formula -z gamma(-z) gamma(z) = pi / sin(pi z) */
  if ((creal(x) < -14.0) || (cimag(x) < -14.0)) {
    q = creal(x);
    p = floor(q);
    if (p == q) goto loverf;
    if (fabs(cimag(x)) > 36.7) {
      /* sin z grows exponentially with Im(z).  Find ln sin(pi z)
      from |sin z| = sqrt( sin^2 x + sinh^2 y),
      arg sin z = arctan(tanh y / tan x).  */
      c = PI * cimag(x) - 0.6931471805599453094 + I*PI*(0.5 - q);
      c = LOGPI - c - clgam(1.0 - x);
    } else {
      /* Reduce sine arg mod pi.  */
      u = csin(PI*(x - p));
      if (u == 0.0) goto loverf;
      w = clgam(1.0 - x);
      c = LOGPI - clog(u) - w;
      c = c + PI*p*I;
    }
    goto ldone;
  }
  w = 0.0;
  if (creal(x) < 14.0) {
    /* To satisfy Im {clgam(z)} = arg cgamma(z), accumulate
	   arg u during the recurrence.  */
    a = 0.0;
    w = 1.0;
    p = 0.0;
    u = x;
    while (creal(u) < 14.0) {
      if (u == 0.0) goto loverf;
      w *= u;
      a += carg(u);
      p += 1.0;
      u = x + p;
    }
    x = u;
    w = -log(cabs(w)) - I * a;
  }
  if (creal(x) > MAXLGM) {
loverf:
    c = MAXNUM + MAXNUM * I;
    goto ldone;
  }
  c = (x - 0.5)*clog(x) - x + 0.91893853320467274178 + w;  /* 0.9189... = log( sqrt( 2*pi ) ) */
  if (cabs(x) > 1.0e8) goto ldone;
  v = 1.0/(x*x);
  u = Alg[0];
  for (i=1; i < 6; i++) {
    u = u * v + Alg[i];
  }
  c = c + u / x;
ldone:
  if (cj) c = conj(c);
  return c;
}

#else

void clgam (double re, double im, double *rre, double *rim) {
  double p, q, a, t1, t2, t3, t4, t5, t7, old, rrre, rrim, c[2], w[2], u[2], v[2];
  int i, cj;
  cj = 0;
  if (im < 0) {
    cj = 1;
    im = -im;
  } else if (im == 0.0) {  /* added A. Walz, Agena 0.32.4 */
    if (re == trunc(re) && re <= 0.0) {
      *rre = AGN_NAN;
      *rim = AGN_NAN;
      return;
    }
    else if (re == 1 || re == 2) {
      *rre = 0;
      *rre = 0;
      return;
    }
  }
  /* Reflection formula -z gamma(-z) gamma(z) = pi / sin(pi z) */
  if ((re < -14.0) || (im < -14.0)) {
    q = re;
    p = floor(q);
    if (p == q) goto loverf;
    if (fabs(im) > 36.7) {
      /* sin z grows exponentially with Im(z).  Find ln sin(pi z)
      from |sin z| = sqrt( sin^2 x + sinh^2 y),
      arg sin z = arctan(tanh y / tan x).  */
      c[0] = PI * im - 0.6931471805599453094;
      c[1] = PI*(0.5 - q);
      clgam(1.0 - re, im, &rrre, &rrim);
      c[0] = LOGPI - c[0] - rrre;
      c[1] = -c[1] + rrim;
    } else {
      /* Reduce sine arg mod pi.  */
      double t3, t5;
      t3 = PI*re-PI*p;
      t5 = PI*im;
      u[0] = sin(t3)*cosh(t5);
      u[1] = cos(t3)*sinh(t5);
      if (u[0] == 0.0 && u[1] == 0) goto loverf;
      clgam(1-re, im, &w[0], &w[1]);
      t3 = pow(u[0],2.0);
      t5 = pow(u[1],2.0);
      c[0] = LOGPI - log(t3+t5)/2 - w[0];
      c[1] = -atan2(u[1], u[0]) - w[1];
      c[0] = c[0];
      c[1] = c[1] + PI*p;
    }
    goto ldone;
  }
  w[0] = 0.0; w[1] = 0.0;
  if (re < 14.0) {
    /* To satisfy Im {clgam(z)} = arg cgamma(z), accumulate
	   arg u during the recurrence.  */
    double t1, t2, t4;
    a = 0.0;
    w[0] = 1.0; w[1] = 0;
    p = 0.0;
    u[0] = re; u[1] = im;
    while (u[0] < 14.0) {
      if (u[0] == 0.0 && u[1] == 0.0) goto loverf;
      old = w[0];
      w[0] = w[0]*u[0]-w[1]*u[1];
      w[1] = old*u[1]+w[1]*u[0];
      a = a + atan2(u[1], u[0]);
      p += 1.0;
      u[0] = re + p;
      u[1] = im;
    }
    re = u[0]; im = u[1];
    t1 = w[0]*w[0];
    t2 = w[1]*w[1];
    t4 = sqrt(t1 + t2);
    w[0] = -log(t4);
    w[1] = -a;
  }
  if (re > MAXLGM) {
loverf:
    c[0] = HUGE_VAL;
    c[1] = HUGE_VAL;
    goto ldone;
  }
  t1 = re-0.5;
  t2 = re*re;
  t3 = im*im;
  t5 = log(t2 + t3);
  t7 = atan2(im, re);
  c[0] = t1*t5/2 - im*t7 - re + 0.91893853320467274178 + w[0];
  c[1] = im*t5/2 + t1*t7 - im + w[1];
  if (sqrt(re*re + im*im) > 1.0e8) goto ldone;
  t1 = re*re;
  t2 = im*im;
  t3 = t1 - t2;
  t4 = t3*t3;
  t7 = 1/(t4+4.0*t1*t2);
  v[0] = t3*t7;
  v[1] = -2.0*re*im*t7;
  u[0] = Alg[0]; u[1] = 0;
  for (i=1; i < 6; i++) {
    old = u[0];
    u[0] = u[0]*v[0] - u[1]*v[1] + Alg[i];
    u[1] = old*v[1] + u[1]*v[0];
  }
  t2 = re*re;
  t3 = im*im;
  t5 = 1/(t2+t3);
  c[0] = c[0]+u[0]*re*t5+u[1]*im*t5;
  c[1] = c[1]+u[1]*re*t5-u[0]*im*t5;
ldone:
  if (cj) c[1] = -c[1];
  *rre = c[0];
  *rim = c[1];
}
#endif


/*							catan()
 *
 *	Complex circular arc tangent
 *
 *
 *
 * SYNOPSIS:
 *
 * agn_Complex catan();
 * agn_Complex z, w;
 *
 * w = catan (z);
 *
 *
 *
 * DESCRIPTION:
 *
 * If
 *     z = x + iy,
 *
 * then
 *          1       (    2x     )
 * Re w  =  - arctan(-----------)  +  k PI
 *          2       (     2    2)
 *                  (1 - x  - y )
 *
 *               ( 2         2)
 *          1    (x  +  (y+1) )
 * Im w  =  - log(------------)
 *          4    ( 2         2)
 *               (x  +  (y-1) )
 *
 * Where k is an arbitrary integer.
 *
 * catan(z) = -i catanh(iz).
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC       -10,+10      5900       1.3e-16     7.8e-18
 *    IEEE      -10,+10     30000       2.3e-15     8.5e-17
 * The check catan( ctan(z) )  =  z, with |x| and |y| < PI/2,
 * had peak relative error 1.5e-16, rms relative error
 * 2.9e-17.  See also clog().
 */

double redupi (double x) {
  double t;
  long i;
  t = x/PI;
  if (t >= 0.0)
    t += 0.5;
  else
    t -= 0.5;
  i = t;	/* the multiple */
  t = i;
  t = ((x - t*DP1) - t*DP2) - t*DP3;
  return (t);
}


/* this function has been modified for Agena to comply with Maple V Release 4 returns, 0.33.1 */
#ifndef PROPCMPLX
agn_Complex carctan (agn_Complex z) {
  agn_Complex w;
  double a, t, x, x2, y;
  x = creal (z);
  y = cimag (z);
  if ((x == 0 || x == -0) && (y == -1 || y == 1)) return AGN_NAN;
  x2 = x*x;
  a = 1.0 - x2 - (y*y);
  t = 0.5 * atan2(2.0*x, a);
  w = redupi(t);
  t = y - 1.0;
  a = x2 + (t*t);
  if (a == 0.0) return AGN_NAN;
  t = y + 1.0;
  a = (x2 + (t*t))/a;
  w = w + (0.25*log(a))*I;
  if ((x == 0 || x == -0) && y > 1 && y == trunc(y) ) {
    return -creal(w) + I*cimag(w); }
  else
    return w;
}
#endif


/*							gamma.c
 *
 *	Gamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, gamma();
 * extern int sgngam;
 *
 * y = gamma( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns gamma function of the argument.  The result is
 * correctly signed, and the sign (+1 or -1) is also
 * returned in a global (extern) variable named sgngam.
 * This variable is also filled in by the logarithmic gamma
 * function lgam().
 *
 * Arguments |x| <= 34 are reduced by recurrence and the function
 * approximated by a rational function of degree 6/7 in the
 * interval (2,3).  Large arguments are handled by Stirling's
 * formula. Large negative arguments are made positive using
 * a reflection formula.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    DEC      -34, 34      10000       1.3e-16     2.5e-17
 *    IEEE    -170,-33      20000       2.3e-15     3.3e-16
 *    IEEE     -33,  33     20000       9.4e-16     2.2e-16
 *    IEEE      33, 171.6   20000       2.3e-15     3.2e-16
 *
 * Error for arguments outside the test range will be larger
 * owing to error amplification by the exponential function.
 *
 */

/*							lgam()
 *
 *	Natural logarithm of gamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * double x, y, lgam();
 * extern int sgngam;
 *
 * y = lgam( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns the base e (2.718...) logarithm of the absolute
 * value of the gamma function of the argument.
 * The sign (+1 or -1) of the gamma function is returned in a
 * global (extern) variable named sgngam.
 *
 * For arguments greater than 13, the logarithm of the gamma
 * function is approximated by the logarithmic version of
 * Stirling's formula using a polynomial approximation of
 * degree 4. Arguments between -33 and +33 are reduced by
 * recurrence to the interval [2,3] of a rational approximation.
 * The cosecant reflection formula is employed for arguments
 * less than -33.
 *
 * Arguments greater than MAXLGM return MAXNUM and an error
 * message.  MAXLGM = 2.035093e36 for DEC
 * arithmetic or 2.556348e305 for IEEE arithmetic.
 *
 *
 *
 * ACCURACY:
 *
 *
 * arithmetic      domain        # trials     peak         rms
 *    DEC     0, 3                  7000     5.2e-17     1.3e-17
 *    DEC     2.718, 2.035e36       5000     3.9e-17     9.9e-18
 *    IEEE    0, 3                 28000     5.4e-16     1.1e-16
 *    IEEE    2.718, 2.556e305     40000     3.5e-16     8.3e-17
 * The error criterion was relative when the function magnitude
 * was greater than one but absolute when it was less than one.
 *
 * The following test used the relative error criterion, though
 * at certain points the relative error could be much higher than
 * indicated.
 *    IEEE    -200, -4             10000     4.8e-16     1.3e-16
 *
 */

/*							gamma.c	*/
/*	gamma function	*/

/*
Cephes Math Library Release 2.8:  June, 2000
Copyright 1984, 1987, 1989, 1992, 2000 by Stephen L. Moshier
*/

static double P[] = {
  1.60119522476751861407E-4,
  1.19135147006586384913E-3,
  1.04213797561761569935E-2,
  4.76367800457137231464E-2,
  2.07448227648435975150E-1,
  4.94214826801497100753E-1,
  9.99999999999999996796E-1
};

static double Q[] = {
 -2.31581873324120129819E-5,
  5.39605580493303397842E-4,
 -4.45641913851797240494E-3,
  1.18139785222060435552E-2,
  3.58236398605498653373E-2,
 -2.34591795718243348568E-1,
  7.14304917030273074085E-2,
  1.00000000000000000320E0
};

/* Stirling's formula for the gamma function */
static double STIR[5] = {
  7.87311395793093628397E-4,
 -2.29549961613378126380E-4,
 -2.68132617805781232825E-3,
  3.47222221605458667310E-3,
  8.33333333333482257126E-2,
};


/* Gamma function computed by Stirling's formula.
 * The polynomial STIR is valid for 33 <= x <= 172.
 */
static double stirf (double x) {
  double y, w, v;
  w = 1.0/x;
  w = 1.0 + w*polevl(w, STIR, 4);
  y = exp(x);
  if (x > MAXSTIR) { /* Avoid overflow in pow() */
    v = pow(x, 0.5*x-0.25);
    y = v*(v/y);
  } else
    y = pow(x, x-0.5)/y;
  y = 2.50662827463100050242E0*y*w;  /* SQTPI = 2.50662827463100050242E0 */
  return y;
}

double cephes_gamma (double x) {
  double p, q, z;
  int i, sgngam;
  sgngam = 1;
  if (isnan(x)) return x;
  if (x == HUGE_VAL) return x;
  if (x == -HUGE_VAL) return AGN_NAN;
  q = fabs(x);
  if (q > 33.0) {
    if (x < 0.0) {
      p = floor(q);
      if (p == q) return AGN_NAN;
      i = p;
      if ((i & 1) == 0) sgngam = -1;
      z = q - p;
      if (z > 0.5) {
        p += 1.0;
        z = q - p;
      }
      z = q * sin(PI*z);
      if (z == 0.0) return sgngam*HUGE_VAL;
      z = fabs(z);
      z = PI/(z * stirf(q));
    } else {
      z = stirf(x);
      if (isnan(z)) z = HUGE_VAL;  /* overflow with arguments > 0: added 0.33.2 */
    }
    return sgngam*z;
  }
  z = 1.0;
  while (x >= 3.0) {
    x -= 1.0;
    z *= x;
  }
  while (x < 0.0) {
	 if (x > -1.E-9) goto small;
    z /= x;
    x += 1.0;
  }
  while (x < 2.0) {
    if (x < 1.e-9) goto small;
    z /= x;
    x += 1.0;
  }
  if (x == 2.0) return(z);
  x -= 2.0;
  p = polevl(x, P, 6);
  q = polevl(x, Q, 7);
  return z*p/q;

small:
  if (x == 0.0)
    return AGN_NAN;
  else
    return z/((1.0 + 0.5772156649015329*x)*x);
}


/*							cgamma
 *
 *	Complex gamma function
 *
 *
 *
 * SYNOPSIS:
 *
 * #include <complex.h>
 * agn_Complex x, y, cgamma();
 *
 * y = cgamma( x );
 *
 *
 *
 * DESCRIPTION:
 *
 * Returns complex-valued gamma function of the complex argument.
 * This variable is also filled in by the logarithmic gamma
 * function clgam().
 *
 * Arguments |x| < 18 are increased by recurrence.
 * Large arguments are handled by Stirling's formula. Large negative
 * arguments are made positive using the reflection formula.
 *
 *
 * ACCURACY:
 *
 *                      Relative error:
 * arithmetic   domain     # trials      peak         rms
 *    IEEE      -20,20      500000      2.0e-14     2.7e-15
 *    IEEE     -100,100     100000      1.4e-13     1.5e-14
 *
 * Error for arguments outside the test range will be larger
 * owing to error amplification by the exponential function.
 *
 */
/*
Cephes Math Library Release 2.7:  March, 1998
Copyright 1984, 1998 Stephen L. Moshier
*/

#ifndef PROPCMPLX

#define NSTIRCG   7

/* Stirling's formula for the gamma function */
static double STIRCG[NSTIRCG] = {
 -5.92166437353693882865E-4,
  6.97281375836585777429E-5,
  7.84039221720066627474E-4,
 -2.29472093621399176955E-4,
 -2.68132716049382716049E-3,
  3.47222222222222222222E-3,
  8.33333333333333333333E-2
};

/* Gamma function computed by Stirling's formula.  */

/* static agn_Complex cstirf(x) */
agn_Complex cstirf (agn_Complex x) {
  agn_Complex y, w;
  int i;
  w = 1.0/x;
  y = STIRCG[0];
  for (i=1; i < NSTIRCG; i++)
    y = y*w + STIRCG[i];
  w = 1.0 + w*y;
  y = cpow(x, x - 0.5)*cexp(-x);
  y = SQTPI*y*w;
  return y;
}

agn_Complex cephes_cgamma (agn_Complex x) {
  double p, q, re, im;
  agn_Complex c, u;
  int k;
  re = creal(x);
  im = cimag(x);
  if (im == 0 && re < 0 && TRUNC(re) == re) return AGN_NAN;  /* added 0.33.2 */
  if (fabs(re) > 18.0) {
    if (re < 0.0) {
      q = re;
      p = floor(q);
      if ((p == q) && (im == 0.0)) return HUGE_VAL;
      /*	c = csin( PI * x ); */
      /* Compute sin(pi x)  */
      k = q - 2.0*floor(0.5*q);
      q = PI*(q - p);
      p = PI*im;
      c = sin(q)*cosh(p) + cos(q)*sinh(p)*I;
      if (k & 1) c = -c;
		/* Reflection formula.  */
      c = PI/(c*cephes_cgamma(1.0 - x));
    } else {
      c = cstirf(x);
      if (isnan(c)) c = HUGE_VAL;  /* added 0.33.2 */
    }
    return c;
  }
  c = 1.0;
  p = 0.0;
  u = x;
  while (creal(u) < 18.0) {
    if ((fabs(creal(u)) < 1.0e-9) && (fabs(cimag(u)) < 1.0e-9)) goto small;
    c *= u;
    p += 1.0;
    u = x + p;
  }
  u = cstirf(u);
  return u/c;
small:
  if ((re == 0.0) && (im == 0.0))
    return AGN_NAN;
  else
    return 1.0/(((1.0 + 0.5772156649015329*u)*u)*c);
}

#endif


