/*
** $Id: calc.c, initiated June 20, 2008 $
** Calculus library
** See Copyright Notice in agena.h
*/


#include <stdlib.h>
#include <math.h>

#define calc_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "agncmpt.h"

#include "cephes.h"
#include "interp.h"

#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_CALCLIBNAME "calc"
LUALIB_API int (luaopen_calc) (lua_State *L);
#endif

static int calc_fsum (lua_State *L) {
  /* computes the sum of f(a), ..., f(b); 0.10.0 - April 20, 2008;
     12 % faster than an Agena implementation;
     extended 0.30.0, December 30, 2009: the procedure now uses a modified Kahan summation algorithm
     to avoid roundoff errors; the modified Kahan algorithm has been developed by Kazufumi Ozawa
     and has been published in his paper `Analysis and Improvement of Kahan's Summation Algorithm`,
     available at http://ci.nii.ac.jp/naid/110002673355/en */
  lua_Number a, b;
  volatile lua_Number q, s, sold, u, v, w, x, t;
  s = q = 0;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  a = agn_checknumber(L, 2);  /* start value */
  b = agn_checknumber(L, 3);  /* stop value */
  while (a <= b) {
    lua_pushvalue(L, 1);  /* push function */
    lua_pushnumber(L, a);
    x = agn_ncall(L, 1, 1);
    v = x - q;
    sold = s;
    s = s + v;
    if (fabs(x) < fabs(q)) {
      t = x;
      x = -q;
      q = t;
    }
    u = (v - x) + q;
    if (fabs(sold) < fabs(v)) {
      t = sold;
      sold = v;
      v = t;
    }
    w = (s - sold) - v;
    q = u + w;
    a++;
  }
  lua_pushnumber(L, s);
  return 1;
}


static int calc_fprod (lua_State *L) {
  /* computes the product of f(a), ..., f(b); 0.29.0 - Nov 23, 2009 */
  lua_Number a, b, r;
  r = 1;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  a = agn_checknumber(L, 2);
  b = agn_checknumber(L, 3);
  while (a <= b) {
    lua_pushvalue(L, 1);
    lua_pushnumber(L, a);
    r *= agn_ncall(L, 1, 1);
    a++;
  }
  lua_pushnumber(L, r);
  return 1;
}


/* DE-Quadrature, Numerical Automatic Integrator for Improper Integral
      method    : Double Exponential (DE) Transformation
      dimension : one
      table     : use
   functions
      intde  : integrator of f(x) over (a,b).

   written by Takuya Ooura in C and available at http://www.kurims.kyoto-u.ac.jp/~ooura/intde.html.
   See intde2.c file.
   For an explanation of the algorithm used, see: http://en.wikipedia.org/wiki/Tanh-sinh_quadrature

   Copyright(C) 1996 Takuya OOURA.
   You may use, copy, modify this code for any purpose and without fee.
   You may distribute this ORIGINAL package. */

void intdeini (int lenaw, double tiny, double eps, double *aw) {
  /* ---- adjustable parameter ---- */
  double efs = 0.1, hoff = 8.5;
  /* ------------------------------ */
  int noff, nk, k, j;
  double pi2, tinyln, epsln, h0, ehp, ehm, h, t, ep, em, xw, wg;

  pi2 = 2 * atan(1.0);
  tinyln = -log(tiny);
  epsln = 1 - log(efs * eps);
  h0 = hoff / epsln;
  ehp = exp(h0);
  ehm = 1 / ehp;
  aw[2] = eps;
  aw[3] = exp(-ehm * epsln);
  aw[4] = sqrt(efs * eps);
  noff = 5;
  aw[noff] = 0.5;
  aw[noff + 1] = h0;
  aw[noff + 2] = pi2 * h0 * 0.5;
  h = 2;
  nk = 0;
  k = noff + 3;
  do {
    t = h * 0.5;
    do {
      em = exp(h0 * t);
      ep = pi2 * em;
      em = pi2 / em;
      j = k;
      do {
        xw = 1 / (1 + exp(ep - em));
        wg = xw * (1 - xw) * h0;
        aw[j] = xw;
        aw[j + 1] = wg * 4;
        aw[j + 2] = wg * (ep + em);
        ep *= ehp;
        em *= ehm;
        j += 3;
      } while (ep < tinyln && j <= lenaw - 3);
      t += h;
      k += nk;
    } while (t < 1);
    h *= 0.5;
    if (nk == 0) {
      if (j > lenaw - 6) j -= 3;
      nk = j - noff;
      k += nk;
      aw[1] = nk;
    }
  } while (2 * k - noff - 3 <= lenaw);
  aw[0] = k - 3;
}


static int calc_intde (lua_State *L) {
  int noff, lenawm, nk, k, j, jtmp, jm, m, klim;
  double aw[8000], r, a, b, eps, err, epsh, ba, ir, xa, fa, fb, errt, errh, errd, h, iback, irback;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  a = agnL_checknumber(L, 2);
  b = agnL_checknumber(L, 3);
  eps = agnL_optnumber(L, 4, 1.0e-15);
  intdeini(8000, 1.0e-307, eps, aw);
  noff = 5;
  lenawm = (int) (aw[0] + 0.5);
  nk = (int) (aw[1] + 0.5);
  epsh = aw[4];
  ba = b - a;
  lua_pushvalue(L, 1);
  lua_pushnumber(L, (a + b) * aw[noff]);
  r = agn_ncall(L, 1, 1);
  ir = r * aw[noff + 1];
  r *= aw[noff + 2];
  err = fabs(r);
  k = nk + noff;
  j = noff;
  do {
    j += 3;
    xa = ba * aw[j];
    lua_pushvalue(L, 1);
    lua_pushnumber(L, a + xa);
    fa = agn_ncall(L, 1, 1);
    lua_pushvalue(L, 1);
    lua_pushnumber(L, b - xa);
    fb = agn_ncall(L, 1, 1);
    ir += (fa + fb) * aw[j + 1];
    fa *= aw[j + 2];
    fb *= aw[j + 2];
    r += fa + fb;
    err += fabs(fa) + fabs(fb);
  } while (aw[j] > epsh && j < k);
  errt = err * aw[3];
  errh = err * epsh;
  errd = 1 + 2 * errh;
  jtmp = j;
  while (fabs(fa) > errt && j < k) {
    j += 3;
    lua_pushvalue(L, 1);
    lua_pushnumber(L, a + ba * aw[j]);
    fa = agn_ncall(L, 1, 1);
    ir += fa * aw[j + 1];
    fa *= aw[j + 2];
    r += fa;
  }
  jm = j;
  j = jtmp;
  while (fabs(fb) > errt && j < k) {
    j += 3;
    lua_pushvalue(L, 1);
    lua_pushnumber(L, b - ba * aw[j]);
    fb = agn_ncall(L, 1, 1);
    ir += fb * aw[j + 1];
    fb *= aw[j + 2];
    r += fb;
  }
  if (j < jm) jm = j;
  jm -= noff + 3;
  h = 1;
  m = 1;
  klim = k + nk;
  while (errd > errh && klim <= lenawm) {
    iback = r;
    irback = ir;
    do {
      jtmp = k + jm;
      for (j = k + 3; j <= jtmp; j += 3) {
        xa = ba * aw[j];
        lua_pushvalue(L, 1);
        lua_pushnumber(L, a + xa);
        fa = agn_ncall(L, 1, 1);
        lua_pushvalue(L, 1);
        lua_pushnumber(L, b - xa);
        fb = agn_ncall(L, 1, 1);
        ir += (fa + fb) * aw[j + 1];
        r += (fa + fb) * aw[j + 2];
      }
      k += nk;
      j = jtmp;
      do {
        j += 3;
        lua_pushvalue(L, 1);
        lua_pushnumber(L, a + ba * aw[j]);
        fa = agn_ncall(L, 1, 1);
        ir += fa * aw[j + 1];
        fa *= aw[j + 2];
        r += fa;
      } while (fabs(fa) > errt && j < k);
      j = jtmp;
      do {
        j += 3;
        lua_pushvalue(L, 1);
        lua_pushnumber(L, b - ba * aw[j]);
        fb = agn_ncall(L, 1, 1);
        ir += fb * aw[j + 1];
        fb *= aw[j + 2];
        r += fb;
      } while (fabs(fb) > errt && j < k);
    } while (k < klim);
    errd = h * (fabs(r - 2 * iback) + fabs(ir - 2 * irback));
    h *= 0.5;
    m *= 2;
    klim = 2 * klim - noff;
  }
  r *= h * ba;
  if (errd > errh) {
    err = -errd * (m * fabs(ba));
  } else {
    err = err * aw[2] * (m * fabs(ba));
  }
  if (err < 0)
    lua_pushfail(L);
  else
    lua_pushnumber(L, r);
  lua_pushnumber(L, err);
  return 2;
}


void intdeiini (int lenaw, double tiny, double eps, double *aw) {
  /* ---- adjustable parameter ---- */
  double efs = 0.1, hoff = 11.0;
  /* ------------------------------ */
  int noff, nk, k, j;
  double pi4, tinyln, epsln, h0, ehp, ehm, h, t, ep, em, xp, xm,
    wp, wm;

  pi4 = atan(1.0);
  tinyln = -log(tiny);
  epsln = 1 - log(efs * eps);
  h0 = hoff / epsln;
  ehp = exp(h0);
  ehm = 1 / ehp;
  aw[2] = eps;
  aw[3] = exp(-ehm * epsln);
  aw[4] = sqrt(efs * eps);
  noff = 5;
  aw[noff] = 1;
  aw[noff + 1] = 4 * h0;
  aw[noff + 2] = 2 * pi4 * h0;
  h = 2;
  nk = 0;
  k = noff + 6;
  do {
    t = h * 0.5;
    do {
      em = exp(h0 * t);
      ep = pi4 * em;
      em = pi4 / em;
      j = k;
      do {
        xp = exp(ep - em);
        xm = 1 / xp;
        wp = xp * ((ep + em) * h0);
        wm = xm * ((ep + em) * h0);
        aw[j] = xm;
        aw[j + 1] = xp;
        aw[j + 2] = xm * (4 * h0);
        aw[j + 3] = xp * (4 * h0);
        aw[j + 4] = wm;
        aw[j + 5] = wp;
        ep *= ehp;
        em *= ehm;
        j += 6;
      } while (ep < tinyln && j <= lenaw - 6);
      t += h;
      k += nk;
    } while (t < 1);
    h *= 0.5;
    if (nk == 0) {
      if (j > lenaw - 12) j -= 6;
      nk = j - noff;
      k += nk;
      aw[1] = nk;
    }
  } while (2 * k - noff - 6 <= lenaw);
  aw[0] = k - 6;
}


static int calc_intdei (lua_State *L) {
  int noff, lenawm, nk, k, j, jtmp, jm, m, klim;
  double aw[8000], err, eps, r, a, epsh, ir, fp, fm, errt, errh, errd, h, iback, irback;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  a = agnL_checknumber(L, 2);
  eps = agnL_optnumber(L, 3, 1.0e-15);
  intdeiini(8000, 1.0e-307, eps, aw);
  noff = 5;
  lenawm = (int) (aw[0] + 0.5);
  nk = (int) (aw[1] + 0.5);
  epsh = aw[4];
  lua_pushvalue(L, 1);
  lua_pushnumber(L, a + aw[noff]);
  r = agn_ncall(L, 1, 1);
  ir = r * aw[noff + 1];
  r *= aw[noff + 2];
  err = fabs(r);
  k = nk + noff;
  j = noff;
  do {
    j += 6;
    lua_pushvalue(L, 1);
    lua_pushnumber(L, a + aw[j]);
    fm = agn_ncall(L, 1, 1);
    lua_pushvalue(L, 1);
    lua_pushnumber(L, a + aw[j + 1]);
    fp = agn_ncall(L, 1, 1);
    ir += fm * aw[j + 2] + fp * aw[j + 3];
    fm *= aw[j + 4];
    fp *= aw[j + 5];
    r += fm + fp;
    err += fabs(fm) + fabs(fp);
  } while (aw[j] > epsh && j < k);
  errt = err * aw[3];
  errh = err * epsh;
  errd = 1 + 2 * errh;
  jtmp = j;
  while (fabs(fm) > errt && j < k) {
    j += 6;
    lua_pushvalue(L, 1);
    lua_pushnumber(L, a + aw[j]);
    fm = agn_ncall(L, 1, 1);
    ir += fm * aw[j + 2];
    fm *= aw[j + 4];
    r += fm;
  }
  jm = j;
  j = jtmp;
  while (fabs(fp) > errt && j < k) {
    j += 6;
    lua_pushvalue(L, 1);
    lua_pushnumber(L, a + aw[j + 1]);
    fp = agn_ncall(L, 1, 1);
    ir += fp * aw[j + 3];
    fp *= aw[j + 5];
    r += fp;
  }
  if (j < jm) jm = j;
  jm -= noff + 6;
  h = 1;
  m = 1;
  klim = k + nk;
  while (errd > errh && klim <= lenawm) {
    iback = r;
    irback = ir;
    do {
      jtmp = k + jm;
      for (j = k + 6; j <= jtmp; j += 6) {
        lua_pushvalue(L, 1);
        lua_pushnumber(L, a + aw[j]);
        fm = agn_ncall(L, 1, 1);
        lua_pushvalue(L, 1);
        lua_pushnumber(L, a + aw[j + 1]);
        fp = agn_ncall(L, 1, 1);
        ir += fm * aw[j + 2] + fp * aw[j + 3];
        r += fm * aw[j + 4] + fp * aw[j + 5];
      }
      k += nk;
      j = jtmp;
      do {
        j += 6;
        lua_pushvalue(L, 1);
        lua_pushnumber(L, a + aw[j]);
        fm = agn_ncall(L, 1, 1);
        ir += fm * aw[j + 2];
        fm *= aw[j + 4];
        r += fm;
      } while (fabs(fm) > errt && j < k);
      j = jtmp;
      do {
        j += 6;
        lua_pushvalue(L, 1);
        lua_pushnumber(L, a + aw[j + 1]);
        fp = agn_ncall(L, 1, 1);
        ir += fp * aw[j + 3];
        fp *= aw[j + 5];
        r += fp;
      } while (fabs(fp) > errt && j < k);
    } while (k < klim);
    errd = h * (fabs(r - 2 * iback) + fabs(ir - 2 * irback));
    h *= 0.5;
    m *= 2;
    klim = 2 * klim - noff;
  }
  r *= h;
  if (errd > errh) {
    err = -errd * m;
  } else {
    err *= aw[2] * m;
  }
  if (err < 0)
    lua_pushfail(L);
  else
    lua_pushnumber(L, r);
  lua_pushnumber(L, err);
  return 2;
}


void intdeoini(int lenaw, double tiny, double eps, double *aw) {
  /* ---- adjustable parameter ---- */
  int lmax = 5;
  double efs = 0.1, enoff = 0.40, pqoff = 2.9, ppoff = -0.72;
  /* ------------------------------ */
  int noff0, nk0, noff, k, nk, j;
  double pi4, tinyln, epsln, frq4, per2, pp, pq, ehp, ehm, h, t,
    ep, em, tk, xw, wg, xa;

  pi4 = atan(1.0);
  tinyln = -log(tiny);
  epsln = 1 - log(efs * eps);
  frq4 = 1 / (2 * pi4);
  per2 = 4 * pi4;
  pq = pqoff / epsln;
  pp = ppoff - log(pq * pq * frq4);
  ehp = exp(2 * pq);
  ehm = 1 / ehp;
  aw[3] = lmax;
  aw[4] = eps;
  aw[5] = sqrt(efs * eps);
  noff0 = 6;
  nk0 = 1 + (int) (enoff * epsln);
  aw[1] = nk0;
  noff = 2 * nk0 + noff0;
  wg = 0;
  xw = 1;
  for (k = 1; k <= nk0; k++) {
    wg += xw;
    aw[noff - 2 * k] = wg;
    aw[noff - 2 * k + 1] = xw;
    xw = xw * (nk0 - k) / k;
  }
  wg = per2 / wg;
  for (k = noff0; k <= noff - 2; k += 2) {
    aw[k] *= wg;
    aw[k + 1] *= wg;
  }
  xw = exp(pp - 2 * pi4);
  aw[noff] = sqrt(xw * (per2 * 0.5));
  aw[noff + 1] = xw * pq;
  aw[noff + 2] = per2 * 0.5;
  h = 2;
  nk = 0;
  k = noff + 3;
  do {
    t = h * 0.5;
    do {
      em = exp(2 * pq * t);
      ep = pi4 * em;
      em = pi4 / em;
      tk = t;
      j = k;
      do {
        xw = exp(pp - ep - em);
        wg = sqrt(frq4 * xw + tk * tk);
        xa = xw / (tk + wg);
        wg = (pq * xw * (ep - em) + xa) / wg;
        aw[j] = xa;
        aw[j + 1] = xw * pq;
        aw[j + 2] = wg;
        ep *= ehp;
        em *= ehm;
        tk += 1;
        j += 3;
      } while (ep < tinyln && j <= lenaw - 3);
      t += h;
      k += nk;
    } while (t < 1);
    h *= 0.5;
    if (nk == 0) {
      if (j > lenaw - 6) j -= 3;
      nk = j - noff;
      k += nk;
      aw[2] = nk;
    }
  } while (2 * k - noff - 3 <= lenaw);
  aw[0] = k - 3;
}


static int calc_intdeo (lua_State *L) {
  int lenawm, nk0, noff0, nk, noff, lmax, m, k, j, jm, l;
  double aw[8000], r, a, omega, err, eps, per, perw, w02, ir, h, iback, irback, t, tk,
    xa, fm, fp, errh, s0, s1, s2, errd;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  a = agnL_checknumber(L, 2);
  omega = agnL_optnumber(L, 3, 1);
  eps = agnL_optnumber(L, 4, 1.0e-15);
  intdeoini(8000, 1.0e-307, eps, aw);
  lenawm = (int) (aw[0] + 0.5);
  nk0 = (int) (aw[1] + 0.5);
  noff0 = 6;
  nk = (int) (aw[2] + 0.5);
  noff = 2 * nk0 + noff0;
  lmax = (int) (aw[3] + 0.5);
  eps = aw[4];
  per = 1 / fabs(omega);
  w02 = 2 * aw[noff + 2];
  perw = per * w02;
  lua_pushvalue(L, 1);
  lua_pushnumber(L, a + aw[noff] * per);
  r = agn_ncall(L, 1, 1);
  ir = r * aw[noff + 1];
  r *= aw[noff + 2];
  err = fabs(r);
  h = 2;
  m = 1;
  k = noff;
  jm = fm = fp = errh = 0;  /* to prevent compiler warnings */
  do {
    iback = r;
    irback = ir;
    t = h * 0.5;
    do {
      if (k == noff) {
        tk = 1;
        k += nk;
        j = noff;
        do {
          j += 3;
          xa = per * aw[j];
          lua_pushvalue(L, 1);
          lua_pushnumber(L, a + xa);
          fm = agn_ncall(L, 1, 1);
          lua_pushvalue(L, 1);
          lua_pushnumber(L, a + xa + perw * tk);
          fp = agn_ncall(L, 1, 1);
          ir += (fm + fp) * aw[j + 1];
          fm *= aw[j + 2];
          fp *= w02 - aw[j + 2];
          r += fm + fp;
          err += fabs(fm) + fabs(fp);
          tk += 1;
        } while (aw[j] > eps && j < k);
        errh = err * aw[5];
        err *= eps;
        jm = j - noff;
      } else {
        tk = t;
        for (j = k + 3; j <= k + jm; j += 3) {
          xa = per * aw[j];
          lua_pushvalue(L, 1);
          lua_pushnumber(L, a + xa);
          fm = agn_ncall(L, 1, 1);
          lua_pushvalue(L, 1);
          lua_pushnumber(L, a + xa + perw * tk);
          fp = agn_ncall(L, 1, 1);
          ir += (fm + fp) * aw[j + 1];
          fm *= aw[j + 2];
          fp *= w02 - aw[j + 2];
          r += fm + fp;
          tk += 1;
        }
        j = k + jm;
        k += nk;
      }
      while (fabs(fm) > err && j < k) {
        j += 3;
        lua_pushvalue(L, 1);
        lua_pushnumber(L, a + per * aw[j]);
        fm = agn_ncall(L, 1, 1);
        ir += fm * aw[j + 1];
        fm *= aw[j + 2];
        r += fm;
      }
      lua_pushvalue(L, 1);
      lua_pushnumber(L, a + perw * tk);
      fm = agn_ncall(L, 1, 1);
      s2 = w02 * fm;
      r += s2;
      if (fabs(fp) > err || fabs(s2) > err) {
        l = 0;
        for (;;) {
          l++;
          s0 = 0;
          s1 = 0;
          s2 = fm * aw[noff0 + 1];
          for (j = noff0 + 2; j <= noff - 2; j += 2) {
            tk += 1;
            lua_pushvalue(L, 1);
            lua_pushnumber(L, a + perw * tk);
            fm = agn_ncall(L, 1, 1);
            s0 += fm;
            s1 += fm * aw[j];
            s2 += fm * aw[j + 1];
          }
          if (s2 <= err || l >= lmax) break;
          r += w02 * s0;
        }
        r += s1;
        if (s2 > err) err = s2;
      }
      t += h;
    } while (t < 1);
    if (m == 1) {
      errd = 1 + 2 * errh;
    } else {
      errd = h * (fabs(r - 2 * iback) + fabs(ir - 2 * irback));
    }
    h *= 0.5;
    m *= 2;
  } while (errd > errh && 2 * k - noff <= lenawm);
  r *= h * per;
  if (errd > errh) {
    err = -errd * per;
  } else {
    err *= per * m * 0.5;
  }
  if (err < 0)
    lua_pushfail(L);
  else
    lua_pushnumber(L, r);
  lua_pushnumber(L, err);
  return 2;
}


static int calc_Si (lua_State *L) {  /* 0.32.2 */
  lua_Number x, si, ci;
  x = agn_checknumber(L, 1);
  if (x == HUGE_VAL)
    lua_pushnumber(L, 0.5*PI);
  else if (-x == HUGE_VAL)
    lua_pushnumber(L, -0.5*PI);
  else {
    sici(x, &si, &ci);
    lua_pushnumber(L, si);
  }
  return 1;
}


static int calc_Ci (lua_State *L) {  /* 0.32.2 */
  lua_Number x, si, ci;
  x = agn_checknumber(L, 1);
  if (x == HUGE_VAL)
    lua_pushnumber(L, 0);
  else if (-x == HUGE_VAL)
    lua_pushfail(L);
  else if (x < 0)
    lua_pushundefined(L);
  else {
    sici(agn_checknumber(L, 1), &si, &ci);
    lua_pushnumber(L, ci);
  }
  return 1;
}


static int calc_Ssi (lua_State *L) {  /* 0.32.2 */
  lua_Number x, si, ci;
  x = agn_checknumber(L, 1);
  if (x == HUGE_VAL)
    lua_pushnumber(L, 0);
  else if (-x == HUGE_VAL)
    lua_pushnumber(L, -PI);
  else {
    sici(x, &si, &ci);
    lua_pushnumber(L, si - 0.5*PI);
  }
  return 1;
}


static int calc_Shi (lua_State *L) {  /* 0.32.2 */
  lua_Number x, shi, chi;
  x = agn_checknumber(L, 1);
  if (-x == HUGE_VAL)
    lua_pushnumber(L, x);
  else {
    shichi(x, &shi, &chi);
    lua_pushnumber(L, shi);
  }
  return 1;
}


static int calc_Chi (lua_State *L) {  /* 0.32.2 */
  lua_Number x, shi, chi;
  x = agn_checknumber(L, 1);
  if (-x == HUGE_VAL)
    lua_pushfail(L);
  else if (x < 0)
    lua_pushundefined(L);
  else {
    shichi(agn_checknumber(L, 1), &shi, &chi);
    lua_pushnumber(L, chi);
  }
  return 1;
}


static int calc_dawson (lua_State *L) {  /* 0.32.2 */
  lua_Number x;
  x = agn_checknumber(L, 1);
  lua_pushnumber(L, dawsn(x));
  return 1;
}


static int calc_Ei (lua_State *L) {  /* 0.32.2 */
  lua_Number x;
  x = agn_checknumber(L, 1);
  if (x == HUGE_VAL)
    lua_pushnumber(L, HUGE_VAL);
  else if (-x == HUGE_VAL)
    lua_pushnumber(L, 0);
  else
    lua_pushnumber(L, ei(x));
  return 1;
}


static int calc_Psi (lua_State *L) {  /* 0.32.2 */
  lua_Number x;
  x = agn_checknumber(L, 1);
  lua_pushnumber(L, psi(x));
  return 1;
}


static int calc_dilog (lua_State *L) {  /* 0.32.2 */
  lua_Number x;
  x = agn_checknumber(L, 1);
  if (x == HUGE_VAL)
    lua_pushnumber(L, -HUGE_VAL);
  else
    lua_pushnumber(L, spence(x));
  return 1;
}


static int calc_fresnelc (lua_State *L) {  /* 0.32.3 */
  lua_Number x, s, c;
  x = agn_checknumber(L, 1);
  fresnl(x, &s, &c);
  lua_pushnumber(L, c);
  return 1;
}


static int calc_fresnels (lua_State *L) {  /* 0.32.3 */
  lua_Number x, s, c;
  x = agn_checknumber(L, 1);
  fresnl(x, &s, &c);
  lua_pushnumber(L, s);
  return 1;
}

/*
Differentiation using Richardson`s extrapolation, taken from http://netlib.org/textbook/mathews/chap6.f

NUMERICAL METHODS: FORTRAN Programs, (c) John H. Mathews 1994, used by kind permission.

Author of the original FORTRAN routine:
Prof. John  H.  Mathews
Department of Mathematics
California State University Fullerton
Fullerton, CA  92634

The FORTRAN routine accompanies the book `NUMERICAL METHODS for Mathematics, Science and Engineering`,
2nd Ed, 1992, Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.

Algorithm 6.2 (Differentiation Using Extrapolation).
Section 6.1, Approximating the Derivative, Page 327
Agena port January 01, 2010, 0.30.1
*/

static int calc_xpdiff (lua_State *L) {  /* 0.32.3 */
  lua_Number fph, fmh, x, d[16][16], eps, delta, err, h, relerr, small;
  int j, k, n, max_n;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  x = agnL_checknumber(L, 2);
  eps = agnL_optnumber(L, 3, AGN_EPSILON);
  delta = agnL_optnumber(L, 4, AGN_EPSILON);
  lua_pushvalue(L, 1);
  lua_pushnumber(L, x);
  fph = agn_ncall(L, 1, 1);
  if (!isfinite(fph)) {
    lua_pushundefined(L);
    return 1;
  }
  max_n = 15;
  small = 1e-7;
  h = 1.0;
  n = 0;
  err = 1.0;
  relerr = 1.0;
  lua_pushvalue(L, 1);
  lua_pushnumber(L, x+h);
  fph = agn_ncall(L, 1, 1);
  lua_pushvalue(L, 1);
  lua_pushnumber(L, x-h);
  fmh = agn_ncall(L, 1, 1);
  d[0][0] = 0.5*(fph - fmh)/h;
  for (j=1; (j <= max_n) && (relerr > eps) && (err > delta); j++) {
    h = h/2;
    lua_pushvalue(L, 1);
    lua_pushnumber(L, x+h);
    fph = agn_ncall(L, 1, 1);
    lua_pushvalue(L, 1);
    lua_pushnumber(L, x-h);
    fmh = agn_ncall(L, 1, 1);
    d[j][0] = 0.5*(fph - fmh)/h;
    for (k=1; k <= j; k++) {
      d[j][k] = d[j][k-1] + (d[j][k-1] - d[j-1][k-1])/(pow(4, k)-1);
    }
    err = fabs(d[j][j] - d[j-1][j-1]);
    relerr = 2*err/(fabs(d[j][j]) + fabs(d[j-1][j-1]) + small);
    n = j;
  }
  lua_pushnumber(L, d[n][n]);
  lua_pushnumber(L, err);
  return 2;
}


/* returns all intervals where the function has a change in sign
   Maple versions: 1997, 1998; tuned Agena version January 21, 2008;
   C Version of Agena version 29.05.2010, 0.32.3 */

static int calc_sections (lua_State *L) {
  lua_Number xleft, xright, step, fl, fr, total;
  volatile lua_Number c, y, t, i;
  size_t seqcounter, counter;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  xleft = agn_checknumber(L, 2);
  xright = agn_checknumber(L, 3);
  step = agn_checknumber(L, 4);
  agn_createseq(L, 4);  /* result */
  lua_pushvalue(L, 1);
  lua_pushnumber(L, xleft);
  fl = agn_ncall(L, 1, 1);
  i = xleft;
  c = 0.0;
  seqcounter = counter = 0;
  total = trunc(fabs(xright-xleft)/step+1);  /* total number of iterations, do not use floor, (int), etc ! */
  while (i <= xright || total > counter) {
    lua_pushvalue(L, 1);
    lua_pushnumber(L, i + step);
    fr = agn_ncall(L, 1, 1);
    if (fl*fr <= 0) {
      agn_createseq(L, 2);
      lua_seqsetinumber(L, -1, 1, i);
      lua_seqsetinumber(L, -1, 2, i+step);
      lua_seqseti(L, -2, ++seqcounter);
    }
    counter++;
    /* since right border in the curent iteration is equal to the left border in the next
      iteration, avoid calling f twice. */
    fl = fr;
    /* apply Kahan summation algorithm to avoid roundoff errors, 0.30.0 */
    y = step - c;
    t = i + y;
    c = (t - i) - y;
    i = t;
  }
  return 1;  /* return result */
}


/*
  Modified Regula Falsi method; Maple versions: 1996, 1997; Agena port January 20, 2008/July 02, 2008
  This algorithm, taken from an unknown FORTRAN book in the 1980s, is the most accurate and the
  fastest one, applicable to a large range of function types. All other methods tested
  including checking for a change of sign are slower and more inaccurate.
*/

static int calc_regulafalsi (lua_State *L) {
  lua_Number x1, x2, u, v, z, a, b, eps;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  a = agn_checknumber(L, 2);
  b = agn_checknumber(L, 3);
  eps = agn_checknumber(L, 4);
  x1 = a;
  x2 = b;
  z = AGN_NAN;
  lua_pushvalue(L, 1);
  lua_pushnumber(L, x1);
  u = agn_ncall(L, 1, 1);
  lua_pushvalue(L, 1);
  lua_pushnumber(L, x2);
  v = agn_ncall(L, 1, 1);
  if (fabs(u) < eps) {
    lua_pushnumber(L, x1);
    return 1;
  }
  if (fabs(v) < eps) {
    lua_pushnumber(L, x2);
    return 1;
  }
  while (fabs(u-v) > eps) {
    z = x2-v*(x2-x1)/(v-u);
    u = v;
    x1 = x2;
    x2 = z;
    lua_pushvalue(L, 1);
    lua_pushnumber(L, x2);
    v = agn_ncall(L, 1, 1);
  }
  if (z >= a && z <= b)
    lua_pushnumber(L, z);
  else
    lua_pushnil(L);
  return 1;
}


/*
  Compute the value of the first differentiation of a function f at a point x.
  The algorithm is based on Conte and de Boor's `Coefficients of Newton form of
  polynomial of degree 3`; written June 09, 2008, 0.11.2
*/
static int calc_diff (lua_State *L) {
  lua_Number eps, h, q, x, a[4], b[4], fph, fmh;
  int i, j;
  luaL_checktype(L, 1, LUA_TFUNCTION);
  x = agn_checknumber(L, 2);
  eps = agnL_optnumber(L, 3, AGN_EPSILON);
  h = eps;
  q = 0;
  for (i=0; i < 4; i++) {
    a[i] = x+(i-1)*h;
    lua_pushvalue(L, 1);
    lua_pushnumber(L, a[i]);
    b[i] = agn_ncall(L, 1, 1);
  }
  for (j=0; j < 4; j++) {
    for (i=0; i < 3-j; i++) {
      b[i] = (b[i+1]-b[i])/(a[i+j+1]-a[i]);
    }
  }
  for (i=0; i < 4; i++) q += b[i];
  q = fabs(q);
  if (q < 100*eps) q = 100*eps;
  h = cbrt(eps/(2*q));
  if (h > 100*eps) h = 100*eps;
  lua_pushvalue(L, 1);
  lua_pushnumber(L, x+h);
  fph = agn_ncall(L, 1, 1);
  lua_pushvalue(L, 1);
  lua_pushnumber(L, x-h);
  fmh = agn_ncall(L, 1, 1);
  lua_pushnumber(L, (fph-fmh)/(2*h));
  lua_pushnumber(L, fabs(h*h*q*100));
  return 2;
}



/* An estimate to the min location
   f:	    function under investigation
   a:		left border of the range the min is seeked
   b:  	right border of the range the min is seeked
   tol:	(optional) Acceptable tolerance */

static int calc_fminbr (lua_State *L) {
  lua_Number x, v, w, fv, fx, fw, r, a, b, tol, Eps;  /* Abscissae, descr. see above, f(x), f(v), f(w), r */
  luaL_checktype(L, 1, LUA_TFUNCTION);
  a = agn_checknumber(L, 2);
  b = agn_checknumber(L, 3);
  tol = agnL_optnumber(L, 4, AGN_EPSILON);
  r = (3-sqrt(5))/2;	  /* Gold section ratio */
  if (!(tol > 0 && b > a))
    luaL_error(L, "Error in " LUA_QS ": tolerance <= 0 or a <= b.", "calc.fminbr");
  v = a + r*(b-a);
  lua_pushvalue(L, 1);
  lua_pushnumber(L, v);
  fv = agn_ncall(L, 1, 1);  /* First step - always gold section */
  x = v; w = v;
  fx = fv; fw = fv;
  while (1) {              /* Main iteration loop */
    lua_Number range, middle_range, tol_act, new_step, t, fa, fb, ft;
    range = b-a;           /* Range over which the minimum is seeked for */
    middle_range = (a+b)/2;
    lua_getglobal(L, "Eps");
    if (lua_type(L, -1) != LUA_TNUMBER)
      Eps = AGN_EPSILON;
    else
      Eps = agn_tonumber(L, -1);
    agn_poptop(L);
    tol_act = Eps*fabs(x) + tol/3;  /* Actual tolerance; new_step: Step at this iteration */
    if (fabs(x-middle_range) + range/2 <= 2*tol_act) {
      lua_pushvalue(L, 1);
      lua_pushnumber(L, a);
      fa = agn_ncall(L, 1, 1);
      lua_pushvalue(L, 1);
      lua_pushnumber(L, b);
      fb = agn_ncall(L, 1, 1);
      lua_pushvalue(L, 1);
      lua_pushnumber(L, x);
      fx = agn_ncall(L, 1, 1);
      if (fa < fx) {  /* the function is not that good if the minimum is exactly at the left or right border */
        lua_pushnumber(L, a);   /* 0.29.4 patch, 28.12.2009 */
      } else if (fb < fx) {
        lua_pushnumber(L, b);
      } else {
        lua_pushnumber(L, x);   /* Acceptable approx. is found */
      }
      return 1;
    }
    /* Obtain the gold section step */
    new_step = r * (x < middle_range ? b-x : a-x);
    /*Decide if the interpolation can be tried */
    if (fabs(x-w) >= tol_act) {	/* If x and w are distinct interpolatiom may be tried */
      lua_Number p, q, t;        /* Interpolation step is calculated as p/q; division operation is delayed */
                                 /*  until last moment */
      t = (x-w) * (fx-fv);
      q = (x-v) * (fx-fw);
      p = (x-v)*q - (x-w)*t;
      q = 2*(q-t);
      if (q > 0)        /*  q was calculated with the op-  */
        p = -p;         /*  posite sign; make q positive   */
      else              /*  and assign possible minus to   */
        q = -q;         /*  p                              */
      if (fabs(p) < fabs(new_step*q) &&  /*  If x+p/q falls in [a,b]  */
        p > q*(a-x+2*tol_act) &&     /*  not too close to a and   */
        p < q*(b-x-2*tol_act)) {      /*  b, and isn't too large   */
        new_step = p/q;              /*  it is accepted  */
      }
         /* If p/q is too large then the gold section procedure can reduce [a,b] range to more extent  */
    }
    if (fabs(new_step) < tol_act) {       /* Adjust the step to be not less */
      if (new_step > 0)                 /* than tolerance */
        new_step = tol_act;
      else
        new_step = -tol_act;
    }
    /* Obtain the next approximation to min and reduce the enveloping range */
    t = x + new_step;	/* Tentative point for the min */
    lua_pushvalue(L, 1);
    lua_pushnumber(L, t);
    ft = agn_ncall(L, 1, 1);
    if (ft <= fx) {    /* t is a better approximation */
      if (t < x)       /* Reduce the range so that */
        b = x;         /* t would fall within it */
      else
        a = x;
      v = w; w = x; x = t;  /* Assign the best approx to x */
      fv = fw; fw = fx; fx = ft;
    } else { /* x remains the better approx */
      if (t < x)     /* Reduce the range enclosing x */
        a = t;
      else
        b = t;
      if (ft <= fw || w == x) {
        v = w; w = t;
        fv = fw; fw = ft;
      } else if (ft <= fv || v == x || v == w) {
        v = t;
        fv = ft;
      }
    }
  }
}


/* creates an array a with size n, FREE it ! */
#define createarray(a, n, procname) { \
  if ((n) < 1) \
    luaL_error(L, "Error in " LUA_QS ": table or sequence with at least one pair expected.", (procname)); \
  (a) = malloc((n)*sizeof(lua_Number)); \
  if ((a) == NULL) \
    luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", (procname)); \
}

static void agn_toarray (lua_State *L, int idx, const char *procname, size_t n, lua_Number *x, lua_Number *y) {
  size_t i;
  int type;
  type = lua_type(L, idx);
  luaL_argcheck(L, type == LUA_TTABLE || type == LUA_TSEQ, 1, "table or sequence of pairs expected");
  if (type == LUA_TTABLE) {
    i = 0;
    lua_pushnil(L);
    while (lua_next(L, idx)) {
      agn_pairgetnumbers(L, procname, -1, &(x[i]), &(y[i])); i++;
    }
  }
  else {
    for (i=0; i < n; i++) {
      lua_seqgeti(L, idx, i+1);  /* push object */
      agn_pairgetnumbers(L, procname, -1, &(x[i]), &(y[i]));
    }
  }
}


static void agn_seqtoarray (lua_State *L, lua_Number *x, int idx, size_t n) {  /* converts sequence to an array */
  size_t i;
  for (i=0; i < n; i++) {
    x[i] = lua_seqgetinumber(L, lua_upvalueindex(idx), i+1);
  }
}


static int calc_neville (lua_State *L) {
  size_t n;
  lua_Number t, *x, *y;
  n = agn_nops(L, 1);  /* returns 0 if object at idx is not a structure */
  createarray(x, n, "calc.neville");
  createarray(y, n, "calc.neville");
  agn_toarray(L, 1, "calc.neville", n, x, y);
  t = agn_checknumber(L, 2);
  lua_pushnumber(L, neville(n, x, y, t));
  xfree(x); xfree(y);  /* 1.12.9 fix */
  return 1;
}


int newton (lua_State *L) {
  int i, n;
  lua_Number t, r, *x, *nf;
  t = agn_checknumber(L, 1);
  n = agn_tonumber(L, lua_upvalueindex(3));
  createarray(x, n, "calc.interp");
  createarray(nf, n, "calc.interp");
  agn_seqtoarray(L, x, 1, n);
  agn_seqtoarray(L, nf, 2, n);
  r = nf[n-1];
  for (i=n-2; i >= 0; i--)
    r = r * (t - x[i]) + nf[i];
  xfree(x); xfree(nf);
  lua_pushnumber(L, r);
  return 1;
}


static int calc_interp (lua_State *L) {
  size_t n, nargs;
  lua_Number *x, *y, *coeff;
  n = agn_nops(L, 1);  /* returns 0 if object at idx is not a structure */
  nargs = lua_gettop(L);
  createarray(x, n, "calc.interp");
  createarray(y, n, "calc.interp");
  agn_toarray(L, 1, "calc.interp", n, x, y);
  if (nargs == 3 && lua_type(L, 3) == LUA_TSEQ) {
    size_t i;
    coeff = NULL;
    if (agn_seqsize(L, 3) != n) luaL_error(L, "Error in " LUA_QS ": number of coefficients must be equal to number of points.", "calc.interp");
    createarray(coeff, n, "calc.interp");
    for (i=0; i < n; i++)
      coeff[i] = lua_seqgetinumber(L, 3, i+1);
  } else
    coeff = divdiff(n, x, y);
  if (coeff == NULL) luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "calc.interp");
  if (nargs > 1)
    lua_pushnumber(L, nf_eval(n, x, coeff, agn_checknumber(L, 2)));
  else {
    agn_arraytoseq(L, x, n);  /* push x as a sequence */
    agn_arraytoseq(L, coeff, n);  /* push coefficients as a sequence */
    lua_pushnumber(L, n);     /* push number of elements */
    lua_pushcclosure(L, &newton, 3);
  }
  xfree(x); xfree(y); xfree(coeff);  /* 1.12.9 fix */
  return 1;
}


static int calc_newtoncoeffs (lua_State *L) {
  size_t i, n;
  lua_Number *x, *y, *coeff;
  n = agn_nops(L, 1);  /* returns 0 if object at idx is not a structure */
  createarray(x, n, "calc.newtoncoeffs");
  createarray(y, n, "calc.newtoncoeffs");
  agn_toarray(L, 1, "calc.newtoncoeffs", n, x, y);
  coeff = divdiff(n, x, y);
  if (coeff == NULL) luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "calc.newtoncoeffs");
  agn_createseq(L, n);
  for (i=0; i < n; i++) {
    lua_seqsetinumber(L, -1, i+1, coeff[i]);
  }
  xfree(x); xfree(y); xfree(coeff);  /* 1.12.9 fix */
  return 1;
}


static void agn_nakcoeffs (lua_State *L, int mainidx, int idx, lua_Number *x, size_t n) {  /* copies numbers in a sequence to an array */
  size_t i;
  lua_seqgeti(L, mainidx, idx);
  if (lua_type(L, -1) != LUA_TSEQ)
    luaL_error(L, "Error in " LUA_QS ": expected a sequence, got %s.", "calc.nakspline", lua_typename(L, lua_type(L, -1)));
  if (agn_seqsize(L, -1) != n)
    luaL_error(L, "Error in " LUA_QS ": number of coefficients not equal to number of points, got .", "calc.nakspline");
  for (i=0; i < n; i++) {
    x[i] = lua_seqgetinumber(L, -1, i+1);
  }
  agn_poptop(L);
}


static int calc_naksplinecoeffs (lua_State *L) {
  size_t n;
  lua_Number *x, *y, *b, *c, *d;
  n = agn_nops(L, 1);  /* returns 0 if object at idx is not a structure */
  if (n < 4) {  /* 1.12.9 fix to prevent invalid array access in cubic_nak */
    lua_pushfail(L);
    return 1;
  }
  createarray(x, n, "calc.naksplinecoeffs");
  createarray(y, n, "calc.naksplinecoeffs");
  createarray(b, n, "calc.naksplinecoeffs");
  createarray(c, n, "calc.naksplinecoeffs");
  createarray(d, n, "calc.naksplinecoeffs");
  agn_toarray(L, 1, "calc.naksplinecoeffs", n, x, y);
  if (cubic_nak(n, x, y, b, c, d) == -1) {
    xfree(x); xfree(y);  /* 1.12.9 */
    luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "calc.naksplinecoeffs");
  }
  agn_createseq(L, 3);
  agn_arraytoseq(L, b, n);
  lua_seqseti(L, -2, 1);
  agn_arraytoseq(L, c, n);
  lua_seqseti(L, -2, 2);
  agn_arraytoseq(L, d, n);
  lua_seqseti(L, -2, 3);
  xfree(x); xfree(y); xfree(b); xfree(c); xfree(d);  /* 1.12.9 fix */
  return 1;
}


static int calc_clampedsplinecoeffs (lua_State *L) {
  size_t n;
  lua_Number *x, *y, *b, *c, *d, da, db;
  n = agn_nops(L, 1);  /* returns 0 if object at idx is not a structure */
  if (n < 2) {  /* 1.12.9 fix to prevent invalid array access in cubic_nak */
    lua_pushfail(L);
    return 1;
  }
  lua_pushvalue(L, 2);
  agn_pairgetnumbers(L, "calc.clampedsplinecoeffs", -1, &da, &db);  /* removes pair pushed on the stack */
  createarray(x, n, "calc.clampedsplinecoeffs");
  createarray(y, n, "calc.clampedsplinecoeffs");
  createarray(b, n, "calc.clampedsplinecoeffs");
  createarray(c, n, "calc.clampedsplinecoeffs");
  createarray(d, n, "calc.clampedsplinecoeffs");
  agn_toarray(L, 1, "calc.clampedsplinecoeffs", n, x, y);
  if (cubic_clamped(n, x, y, b, c, d, da, db) == -1) {
    xfree(x); xfree(y);  /* 1.12.9 fix */
    luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "calc.clampedsplinecoeffs");
  }
  agn_createseq(L, 3);
  agn_arraytoseq(L, b, n);
  lua_seqseti(L, -2, 1);
  agn_arraytoseq(L, c, n);
  lua_seqseti(L, -2, 2);
  agn_arraytoseq(L, d, n);
  lua_seqseti(L, -2, 3);
  xfree(x); xfree(y); xfree(b); xfree(c); xfree(d);  /* 1.12.9 fix */
  return 1;
}


int nakspline (lua_State *L) {
  size_t i, found, n;
  lua_Number t, *x, *y, *b, *c, *d;
  i = 1;
  found = 0;
  t = agn_checknumber(L, 1);
  n = agn_tonumber(L, lua_upvalueindex(6));
  createarray(x, n, "calc.nakspline");
  createarray(y, n, "calc.nakspline");
  createarray(b, n, "calc.nakspline");
  createarray(c, n, "calc.nakspline");
  createarray(d, n, "calc.nakspline");
  agn_seqtoarray(L, x, 1, n);
  agn_seqtoarray(L, y, 2, n);
  agn_seqtoarray(L, b, 3, n);
  agn_seqtoarray(L, c, 4, n);
  agn_seqtoarray(L, d, 5, n);
  while (!found && (i < n-1)) {
    if (t < x[i])
      found = 1;
    else
      i++;
  }
  t = y[i-1] + (t - x[i-1]) * (b[i-1] + (t - x[i-1]) * (c[i-1] + (t - x[i-1]) * d[i-1]));
  xfree(x); xfree(y); xfree(b); xfree(c); xfree(d);
  lua_pushnumber(L, t);
  return 1;
}


static int calc_nakspline (lua_State *L) {
  size_t n, nargs;
  lua_Number t, *x, *y, *b, *c, *d;
  n = agn_nops(L, 1);  /* returns 0 if object at idx is not a structure */
  if (n < 4) {  /* 1.12.9 fix to prevent invalid array access in cubic_nak */
    lua_pushfail(L);
    return 1;
  }
  nargs = lua_gettop(L);
  createarray(x, n, "calc.nakspline");
  createarray(y, n, "calc.nakspline");
  createarray(b, n, "calc.nakspline");
  createarray(c, n, "calc.nakspline");
  createarray(d, n, "calc.nakspline");
  agn_toarray(L, 1, "calc.nakspline", n, x, y);
  if (nargs == 3 && lua_type(L, 3) == LUA_TSEQ) {
    if (agn_seqsize(L, 3) != 3)
      luaL_error(L, "Error in " LUA_QS ": expected a sequence of three sequences.", "calc.nakspline");
    agn_nakcoeffs(L, 3, 1, b, n);
    agn_nakcoeffs(L, 3, 2, c, n);
    agn_nakcoeffs(L, 3, 3, d, n);
  } else {
    if (cubic_nak(n, x, y, b, c, d) == -1) {
      xfree(x); xfree(y);  /* 1.12.9 fix */
      luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "calc.nakspline");
    }
  }
  if (nargs > 1) {
    t = agn_checknumber(L, 2);
    lua_pushnumber(L, spline_eval(n, x, y, b, c, d, t));
  } else {
    agn_arraytoseq(L, x, n);  /* push x as a sequence */
    agn_arraytoseq(L, y, n);  /* push y as a sequence */
    agn_arraytoseq(L, b, n);  /* push b as a sequence */
    agn_arraytoseq(L, c, n);  /* push c as a sequence */
    agn_arraytoseq(L, d, n);  /* push d as a sequence */
    lua_pushnumber(L, n);     /* push number of elements */
    lua_pushcclosure(L, &nakspline, 6);
  }
  xfree(x); xfree(y); xfree(b); xfree(c); xfree(d);  /* 1.12.9 fix */
  return 1;
}


static int calc_clampedspline (lua_State *L) {
  size_t n, nargs;
  lua_Number t, *x, *y, *b, *c, *d, da, db;
  n = agn_nops(L, 1);  /* returns 0 if object at idx is not a structure */
  if (n < 2) {  /* 1.12.9 fix to prevent invalid array access in cubic_nak */
    lua_pushfail(L);
    return 1;
  }
  nargs = lua_gettop(L);
  if (nargs < 2) {
    luaL_error(L, "Error in " LUA_QS ": at least two arguments expected.", "calc.clampedspline");
  }
  lua_pushvalue(L, 2);
  agn_pairgetnumbers(L, "calc.clampedsplinecoeffs", -1, &da, &db);  /* removes pair pushed on the stack */
  createarray(x, n, "calc.clampedspline");
  createarray(y, n, "calc.clampedspline");
  createarray(b, n, "calc.clampedspline");
  createarray(c, n, "calc.clampedspline");
  createarray(d, n, "calc.clampedspline");
  agn_toarray(L, 1, "calc.clampedspline", n, x, y);
  if (nargs == 4 && lua_type(L, 4) == LUA_TSEQ) {
    if (agn_seqsize(L, 4) != 3)
      luaL_error(L, "Error in " LUA_QS ": expected a sequence of three sequences.", "calc.clampedspline");
    agn_nakcoeffs(L, 4, 1, b, n);
    agn_nakcoeffs(L, 4, 2, c, n);
    agn_nakcoeffs(L, 4, 3, d, n);
  } else {
    if (cubic_clamped(n, x, y, b, c, d, da, db) == -1) {
      xfree(x); xfree(y);
      luaL_error(L, "Error in " LUA_QS ": memory allocation failed.", "calc.clampedspline");
    }
  }
  if (nargs > 2) {
    t = agn_checknumber(L, 3);
    lua_pushnumber(L, spline_eval(n, x, y, b, c, d, t));
  } else {
    agn_arraytoseq(L, x, n);  /* push x as a sequence */
    agn_arraytoseq(L, y, n);  /* push y as a sequence */
    agn_arraytoseq(L, b, n);  /* push b as a sequence */
    agn_arraytoseq(L, c, n);  /* push c as a sequence */
    agn_arraytoseq(L, d, n);  /* push d as a sequence */
    lua_pushnumber(L, n);     /* push number of elements */
    lua_pushcclosure(L, &nakspline, 6);
  }
  xfree(x); xfree(y); xfree(b); xfree(c); xfree(d);  /* 1.12.9 fix */
  return 1;
}


static int polygen_generator (lua_State *L) {
  size_t nops, i;
  lua_Number r, x;
  x = agn_checknumber(L, 1);  /* get argument */
  nops = lua_tonumber(L, lua_upvalueindex(1));
  r = lua_tonumber(L, lua_upvalueindex(2));
  for (i=3; i <= nops + 1; i++)
    r = r*x + lua_tonumber(L, lua_upvalueindex(i));
  lua_pushnumber(L, r);
  return 1;
}

static int calc_polygen (lua_State *L) {
  size_t i, nops;
  nops = lua_gettop(L);
  luaL_checkstack(L, nops+1, "too many arguments");
  if (nops == 0)
    luaL_error(L, "Error in " LUA_QS ": expected at least one number.", "calc.gpoly");
  lua_pushinteger(L, nops);  /* number of coefficients */
  for (i=0; i < nops; i++) {
    if (lua_isnumber(L, i+1))
      lua_pushvalue(L, i+1); /* single coefficient */
    else
      luaL_error(L, "Error in " LUA_QS ": expected a number, got %s.", "calc.gpoly", lua_typename(L, lua_type(L, i+1)));
  }
  lua_pushcclosure(L, &polygen_generator, nops+1);
  return 1;
}


static const luaL_Reg calclib[] = {
  {"Chi", calc_Chi},                       /* added on May 24, 2010 */
  {"Ci", calc_Ci},                         /* added on May 24, 2010 */
  {"clampedspline", calc_clampedspline},   /* added on November 04, 2012 */
  {"clampedsplinecoeffs", calc_clampedsplinecoeffs},  /* added on November 04, 2012 */
  {"dawson", calc_dawson},                 /* added on May 25, 2010 */
  {"diff", calc_diff},                     /* added on May 29, 2010 */
  {"dilog", calc_dilog},                   /* added on May 25, 2010 */
  {"Ei", calc_Ei},                         /* added on May 25, 2010 */
  {"fminbr", calc_fminbr},                 /* added on May 29, 2010 */
  {"fprod", calc_fprod},                   /* added on Nov 23, 2009 */
  {"fresnelc", calc_fresnelc},             /* added on May 29, 2010 */
  {"fresnels", calc_fresnels},             /* added on May 29, 2010 */
  {"fsum", calc_fsum},                     /* added on April 20, 2008 */
  {"intde", calc_intde},                   /* added on May 18, 2010 */
  {"intdei", calc_intdei},                 /* added on May 22, 2010 */
  {"intdeo", calc_intdeo},                 /* added on May 22, 2010 */
  {"interp", calc_interp},                 /* added on October 28, 2012 */
  {"nakspline", calc_nakspline},           /* added on October 28, 2012 */
  {"naksplinecoeffs", calc_naksplinecoeffs}, /* added on October 28, 2012 */
  {"neville", calc_neville},               /* added on October 28, 2012 */
  {"newtoncoeffs", calc_newtoncoeffs},     /* added on October 28, 2012 */
  {"polygen", calc_polygen},               /* added on July 21, 2013 */
  {"Psi", calc_Psi},                       /* added on May 25, 2010 */
  {"regulafalsi", calc_regulafalsi},       /* added on May 29, 2010 */
  {"sections", calc_sections},             /* added on May 29, 2010 */
  {"Shi", calc_Shi},                       /* added on May 24, 2010 */
  {"Si", calc_Si},                         /* added on May 24, 2010 */
  {"Ssi", calc_Ssi},                       /* added on May 24, 2010 */
  {"xpdiff", calc_xpdiff},                 /* added on May 29, 2010 */
  {NULL, NULL}
};


/*
** Open calculus library
*/
LUALIB_API int luaopen_calc (lua_State *L) {
  luaL_register(L, AGENA_CALCLIBNAME, calclib);
  return 1;
}


