/*
** $Id: biniolib.c,v 0.1 13.07.2008 $
** Library for reading and writing binary files
** See Copyright Notice in agena.h
*/

#include <stdio.h>      /* also for BUFSIZ */
#include <stdlib.h>
#include <string.h>
#include <errno.h>      /* for strerror */
#include <unistd.h>     /* for ftruncate and close */
#include <fcntl.h>      /* for open */
#include <sys/stat.h>   /* S_I* constants */
#include <math.h>       /* for HUGE_VAL */
#include "agncfg.h"     /* for Endianness */

#define binio_c
#define LUA_LIB

#include "agena.h"
#include "agnxlib.h"
#include "agenalib.h"
#include "lobject.h"

#include "agncmpt.h"


#define AGN_BINIO_OPEN "binio.open"
static int binio_open (lua_State *L) {
  int hnd, readwrite, en;
  const char *file = agn_checkstring(L, 1);  /* 0.13.4 patch: does not accept numbers */
  /* no option given: open in readwrite mode, option given: open in readonly mode */
  readwrite = (lua_gettop(L) == 1);
  /* if the file already exists, check whether you have the proper file rights */
  if (access(file, F_OK) == 0) {  /* file already exists ? */
    if (access(file, (readwrite) ? R_OK|W_OK : R_OK) == -1)
      luaL_error(L, "Error in " LUA_QS ": missing permissions for " LUA_QS ".", AGN_BINIO_OPEN, file);
  }
  hnd = (readwrite) ? my_open(file) : my_roopen(file);
  en = errno;  /* 1.12.2 */
  if (hnd == -1) {  /* file does not yet exist ? */
    if (readwrite) {  /* a new file shall be created ? */
      hnd = my_create(file);
      if (hnd < 0)
        luaL_error(L, "Error in " LUA_QS " with " LUA_QS ": %s.", AGN_BINIO_OPEN, file, strerror(en));
    } else
      luaL_error(L, "Error in " LUA_QS " with " LUA_QS ": %s.", AGN_BINIO_OPEN, file, strerror(en));
  }
  /* enter new open file to global binio.openfiles table */
  if (agnL_gettablefield(L, "binio", "openfiles", AGN_BINIO_OPEN, 1) == LUA_TTABLE) {  /* 0.20.2 & 1.6.4, avoid segmentation faults */
    /* DO NOT use lua_istable since it checks a value on the stack and not the return value of agn_gettablefield (a nonneg int) */
    lua_pushinteger(L, hnd);
    lua_pushstring(L, file);
    lua_rawset(L, -3);
  }  else
    luaL_error(L, "Error in " LUA_QS ": 'binio.openfiles' table not found.", AGN_BINIO_OPEN);  /* Agena 1.6.0 */
  agn_poptop(L);  /* delete "openfiles" */
  lua_pushinteger(L, hnd);
  return 1;
}


#define AGN_BINIO_CLOSE "binio.close"
static int binio_close (lua_State *L) {
  int nargs, hnd, i, en, result;
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  result = 1;
  for (i=1; i <= nargs; i++) {
    hnd = agnL_checkinteger(L, i);
    if (hnd < 3) {  /* avoid standard streams to be closed */
      result = 0;
      continue;
    }
    if (!my_close(hnd) == 0) {  /* 1.12.2 */
      en = errno;
      luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_CLOSE, strerror(en));
    } else {
      /* delete file from global binio.openfiles table */
      if (agnL_gettablefield(L, "binio", "openfiles", AGN_BINIO_CLOSE, 1) == LUA_TTABLE) {  /* 0.20.2 & 1.6.4, avoid segmentation faults */
        lua_pushinteger(L, hnd);
        lua_pushnil(L);
        lua_rawset(L, -3);
      } else
        luaL_error(L, "Error in " LUA_QS ": 'binio.openfiles' table not found.", AGN_BINIO_CLOSE);  /* Agena 1.6.0 */
      agn_poptop(L);  /* delete "openfiles" */
    }
  }
  lua_pushboolean(L, result);
  return 1;
}


/* base_sync: flushes all unwritten content to the given file.
   Patched December 26, 2009, 0.30.0 */

#define AGN_BINIO_SYNC "binio.sync"
static int binio_sync (lua_State *L) {
  int hnd, res, en;
  hnd = agnL_checkinteger(L, 1);
  #ifdef _WIN32
  res = _commit(hnd);
  #else
  res = fsync(hnd);
  #endif
  en = errno;  /* 1.12.2 */
  if (!res)
    lua_pushtrue(L);
  else
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_SYNC, hnd, strerror(en));
  return 1;
}


#define AGN_BINIO_FILEPOS "binio.filepos"
static int binio_filepos (lua_State *L) {
  int hnd, en;
  off64_t fpos;
  hnd = agnL_checkinteger(L, 1);
  fpos = my_fpos(hnd);
  en = errno;
  if (fpos == -1)
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_FILEPOS, hnd, strerror(en));
  else
    lua_pushnumber(L, fpos);
  return 1;
}


#define AGN_BINIO_SEEK "binio.seek"
static int binio_seek (lua_State *L) {
  int hnd, en;
  off64_t pos;
  hnd = agnL_checkinteger(L, 1);
  pos = agnL_checkinteger(L, 2);
  if (lseek64(hnd, pos, SEEK_CUR) == -1) {
    en = errno;
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_SEEK, hnd, strerror(en));
  } else
    lua_pushtrue(L);
  return 1;
}


#define AGN_BINIO_LENGTH "binio.length"
static int binio_length (lua_State *L) {  /* faster than using fstat */
  int hnd, en, result;
  off64_t size, oldpos;
  hnd = agnL_checkinteger(L, 1);
  oldpos = lseek64(hnd, 0L, SEEK_CUR);
  en = errno;
  if (oldpos == -1)
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_LENGTH, hnd, strerror(en));
  size = lseek64(hnd, 0L, SEEK_END);
  en = errno;
  if (size == -1)
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_LENGTH, hnd, strerror(en));
  result = lseek64(hnd, oldpos, SEEK_SET);  /* reset cursor to original file position */
  en = errno;
  if (result == -1)
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s", AGN_BINIO_LENGTH, hnd, strerror(en));
  else
    lua_pushnumber(L, size);
  return 1;
}


#define AGN_BINIO_WRITECHAR "binio.writechar"
static int binio_writechar (lua_State *L) {
  int hnd, en;
  size_t size, nargs, i;
  unsigned char data;
  hnd = agnL_checkinteger(L, 1);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1)
    luaL_error(L, "Error in " LUA_QS ": expected at least a second argument.", AGN_BINIO_WRITECHAR);
  for (i=2; i <= nargs; i++) {
    data = agn_checknumber(L, i);
    size = sizeof(data);
    if (write(hnd, &data, size) != (ssize_t)size) {
      en = errno;
      luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_WRITECHAR, hnd, strerror(en));
    }
  }
  lua_pushtrue(L);
  return 1;
}


/* writes bytes stored in a sequence to a file, 0.25.0, 19.07.2009, extended 1.10.5, 05.04.2013 */
#define AGN_BINIO_WRITEBYTES "binio.writebytes"
static int binio_writebytes (lua_State *L) {
  int hnd, en;
  size_t size, i, nargs, type;
  hnd = agnL_checkinteger(L, 1);
  type = lua_type(L, 2);
  luaL_typecheck(L, type == LUA_TSEQ, 2, "sequence expected", type);
  nargs = agn_seqsize(L, 2);
  if (nargs == 0)
    lua_pushfail(L);  /* better sure than sorry */
  else {
    unsigned char buffer[nargs];
    size = sizeof(buffer);
    for (i=0; i < nargs; i++)
      buffer[i] = (unsigned char)lua_seqgetinumber(L, 2, i+1);
    if (write(hnd, &buffer, size) != (ssize_t)size) {
      en = errno;
      luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_WRITEBYTES, hnd, strerror(en));
    }
    else
      lua_pushtrue(L);
  }
  return 1;
}


#define AGN_BINIO_WRITENUMBER "binio.writenumber"
static int binio_writenumber (lua_State *L) {  /* extended 1.10.5, 05.04.2013 */
  int hnd, en;
  size_t size, nargs, i;
  lua_Number data;
  uint64_t ulong;
  hnd = agnL_checkinteger(L, 1);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1)
    luaL_error(L, "Error in " LUA_QS ": expected at least a second argument.", AGN_BINIO_WRITENUMBER);
  for (i=2; i <= nargs; i++) {
    data = agn_checknumber(L, i);
    ulong = tools_swapdouble(data);  /* only swaps Little Endian bytes */
    size = sizeof(ulong);
    if (write(hnd, &ulong, size) != (ssize_t)size) {
      en = errno;
      luaL_error(L, "Error in " LUA_QS " with #%d: %s.", AGN_BINIO_WRITENUMBER, hnd, strerror(en));
    }
  }
  lua_pushtrue(L);
  return 1;
}


#define AGN_BINIO_WRITELONG "binio.writelong"
static int binio_writelong (lua_State *L) {  /* extended 1.10.5, 05.04.2013 */
  int hnd, en;
  size_t size, nargs, i;
  int32_t data;
  hnd = agnL_checkinteger(L, 1);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1)
    luaL_error(L, "Error in " LUA_QS ": expected at least a second argument.", AGN_BINIO_WRITELONG);
  for (i=2; i <= nargs; i++) {
    data = agn_checknumber(L, i);
    #if BYTE_ORDER != BIG_ENDIAN
    tools_swapint32_t(&data);
    #endif
    size = sizeof(data);
    if (write(hnd, &data, size) != (ssize_t)size) {
      en = errno;
      luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_WRITELONG, hnd, strerror(en));
    }
  }
  lua_pushtrue(L);
  return 1;
}


#define AGN_BINIO_WRITESTRING "binio.writestring"
static int binio_writestring (lua_State *L) {  /* extended 1.10.5, 05.04.2013 */
  int hnd, en;
  size_t size, nargs, i;
  const char *value;
  hnd = agnL_checkinteger(L, 1);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1)
    luaL_error(L, "Error in " LUA_QS ": expected at least a second argument.", AGN_BINIO_WRITESTRING);
  for (i=2; i <= nargs; i++) {
    value = agn_checklstring(L, i, &size);
    char data[size];
    strcpy(data, value);
    my_writel(hnd, size);
    if (write(hnd, &data, size) != (ssize_t)size) {
      en = errno;
      luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_WRITESTRING, hnd, strerror(en));
    }
  }
  lua_pushtrue(L);
  return 1;
}


#define AGN_BINIO_WRITESHORTSTRING "binio.writeshortstring"
static int binio_writeshortstring (lua_State *L) {  /* 0.32.0 */
  int hnd, en;
  size_t size, nargs, i;
  const char *value;
  hnd = agnL_checkinteger(L, 1);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1)
    luaL_error(L, "Error in " LUA_QS ": expected at least a second argument.", AGN_BINIO_WRITESHORTSTRING);
  for (i=2; i <= nargs; i++) {
    value = agn_checklstring(L, i, &size);
    char data[size];  /* 1.11.2 fix */
    if (size > 255)
      luaL_error(L, "Error in " LUA_QS " with writing data to file #%d: string too long", AGN_BINIO_WRITESHORTSTRING, hnd);
    strcpy(data, value);
    my_writec(hnd, (unsigned char)size);
    if (write(hnd, &data, size) != (ssize_t)size) {
      en = errno;
      luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_WRITESHORTSTRING, hnd, strerror(en));
    }
  }
  lua_pushtrue(L);
  return 1;
}


#define AGN_BINIO_READCHAR "binio.readchar"
static int binio_readchar (lua_State *L) {
  off64_t pos;  /* 1.5.1 */
  int hnd, res, en;
  unsigned char data;
  hnd = agnL_checkinteger(L, 1);
  pos = agnL_optinteger(L, 2, 0);
  if (pos != 0 && lseek64(hnd, pos, SEEK_CUR) == -1) {
    lua_pushfail(L);
    return 1;
  }
  res = read(hnd, &data, sizeof(char));
  en = errno;
  if (res > 0)
    lua_pushnumber(L, data);
  else if (res == 0)
    lua_pushnil(L);
  else
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_READCHAR, hnd, strerror(en));
  return 1;
}


/* 0.25.0, 18.07.2009; reads n bytes from a file and returns a sequence of numbers, i.e. the bytes read;
   patched 20.07.2009, 0.25.1 */
#define AGN_BINIO_READBYTES "binio.readbytes"
static int binio_readbytes (lua_State *L) {
  int32_t n, i;
  int hnd, res, en;
  hnd = agnL_checkinteger(L, 1);
  if (lua_gettop(L) == 1) {
    agnL_gettablefield(L, "environ", "buffersize", AGN_BINIO_READBYTES, 1);  /* 1.6.4 */
    n = ( lua_type(L, -1) == LUA_TNUMBER ) ? agn_tonumber(L, -1) : LUAL_BUFFERSIZE;
    if (n < 1) n = LUAL_BUFFERSIZE;  /* 1.6.4 */
    agn_poptop(L);  /* pop "buffersize", 0.32.0 */
  } else
    n = agn_checknumber(L, 2);
  if (n < 1) n = LUAL_BUFFERSIZE;
  unsigned char buffer[n];
  res = read(hnd, buffer, n);
  en = errno;
  if (res > 0) {
    agn_createseq(L, res);
    for (i=0; i < res; i++)
      lua_seqsetinumber(L, -1, i+1, buffer[i]);
  }
  else if (res == 0)
    lua_pushnil(L);
  else
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_READBYTES, hnd, strerror(en));
  return 1;
}


#define AGN_BINIO_READSTRING "binio.readstring"
static int binio_readstring (lua_State *L) {
  int hnd, en;
  ssize_t size, success, result;
  result = 0;
  hnd = agnL_checkinteger(L, 1);
  size = sec_readul(hnd, &success);  /* first, read length of the string, patched 1.12.2 */
  if (size == -1)  /* check errno does not help here for it returns `invalid file handle` with existing handles and files */
    luaL_error(L, "Error in " LUA_QS " with file #%d: invalid file handle or data.", AGN_BINIO_READSTRING, hnd);
  if (success) {
    char data[size+1];
    if ((result = read(hnd, &data, size)) == (ssize_t)size) {  /* now read string itself */
      data[size] = '\0';
      lua_pushlstring(L, data, size);
      return 1;
    }
  }
  en = errno;
  luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_READSTRING, hnd, strerror(en));
  return 1;
}


#define AGN_BINIO_READSHORTSTRING "binio.readshortstring"
static int binio_readshortstring (lua_State *L) {
  int hnd, en;
  size_t size;
  hnd = agnL_checkinteger(L, 1);
  if (sec_read(hnd, &size, sizeof(unsigned char))) {  /* first, read length of the string */
    char data[size+1];
    if (read(hnd, &data, size) == (ssize_t)size) {  /* now read string itself */
      data[size] = '\0';
      lua_pushlstring(L, data, size);
      return 1;
    }
  }
  en = errno;
  luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_READSHORTSTRING, hnd, strerror(en));
  return 1;
}


#define AGN_BINIO_READNUMBER "binio.readnumber"
static int binio_readnumber (lua_State *L) {
  int hnd;
  size_t success;
  uint64_t data;
  lua_Number num;
  hnd = agnL_checkinteger(L, 1);
  success = sec_read(hnd, &data, sizeof(uint64_t));
  if (success) {
    num = tools_unswapdouble(data);
    lua_pushnumber(L, num);
  } else {
    int en = errno;
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_READNUMBER, hnd, strerror(en));
  }
  return 1;
}


#define AGN_BINIO_READLONG "binio.readlong"
static int binio_readlong (lua_State *L) {
  int hnd;
  size_t success;
  int32_t data;
  hnd = agnL_checkinteger(L, 1);
  success = sec_read(hnd, &data, sizeof(int32_t));
  if (success) {
    #if BYTE_ORDER != BIG_ENDIAN
    tools_swapint32_t(&data);
    #endif
    lua_pushnumber(L, data);
  } else {
    int en = errno;
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_READLONG, hnd, strerror(en));
  }
  return 1;
}


#define AGN_BINIO_TOEND "binio.toend"
static int binio_toend (lua_State *L) {
  int hnd;
  off64_t size;
  hnd = agnL_checkinteger(L, 1);
  size = lseek64(hnd, 0L, SEEK_END);
  if (size == -1) {
    int en = errno;
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_TOEND, hnd, strerror(en));
  } else
    lua_pushnumber(L, size);
  return 1;
}


#define AGN_BINIO_REWIND "binio.rewind"
static int binio_rewind (lua_State *L) {
  int hnd;
  off64_t size;
  hnd = agnL_checkinteger(L, 1);
  size = lseek64(hnd, 0L, SEEK_SET);
  if (size == -1) {
    int en = errno;
    luaL_error(L, "Error in " LUA_QS " with file #%d: %s.", AGN_BINIO_REWIND, hnd, strerror(en));
  } else
    lua_pushnumber(L, size);
  return 1;
}


#define AGN_BINIO_READOBJECT "binio.readobject"
static void binio_readobjecterror (lua_State *L, int hnd, int message) {  /* May 03, 2010 */
  /* close the file and delete file name from global binio.openfiles table */
  if (my_close(hnd) == 0) {
    if (agnL_gettablefield(L, "binio", "openfiles", AGN_BINIO_READOBJECT, 1) == LUA_TTABLE) {  /* 0.20.2 & 1.6.4, avoid segmentation faults */
      lua_pushinteger(L, hnd);
      lua_pushnil(L);
      lua_rawset(L, -3);
    } else
      luaL_error(L, "Error in " LUA_QS ": 'binio.openfiles' table not found.", AGN_BINIO_READOBJECT);  /* Agena 1.6.0 */
    agn_poptop(L);  /* delete "openfiles" */
  }
  if (message)
    luaL_error(L, "Error in " LUA_QS " with reading file #%d: I/O error.", AGN_BINIO_READOBJECT, hnd);
  else
    luaL_error(L, "Error in " LUA_QS ": wrong workspace file version.", AGN_BINIO_READOBJECT);
}


static void binio_readobjectutypename (lua_State *L, int hnd, int idx) {
  size_t size;
  ssize_t success;
  size = sec_readl(hnd, &success);  /* first read length of the string */
  if (success) {
    char data[size+1];
    if (read(hnd, &data, size) == (ssize_t)size) {  /* now read string itself */
      data[size] = '\0';
      lua_pushlstring(L, data, size);
      agn_setutype(L, idx-1, -1);  /* set user-defined type to set */
      agn_poptop(L);            /* delete utype string */
      return;
    }  /* else issue an error */
  }
  binio_readobjecterror(L, hnd, 1);
}


static void binio_readobjectaux (lua_State *L, int hnd) {  /* May 03, 2010 */
  unsigned char data;
  int res;
  if (lua_checkstack(L, 2) == 0)  /* Agena 1.8.5 fix to prevent stack overflow = segmentation fault */
    luaL_error(L, "Error in " LUA_QS ": stack cannot be extended, structure too nested.", "binio.readobject");
  res = read(hnd, &data, sizeof(char));
  if (res > 0) {
    switch (data - (data & 32)) {  /* get data type ID */
      case 0: {  /* read char */
        unsigned char num;
        if (sec_read(hnd, &num, sizeof(unsigned char)))
          lua_pushnumber(L, num);
        else
          binio_readobjecterror(L, hnd, 1);
        break;
      }
      case 1: {  /* read long */
        int32_t data;
        if (sec_read(hnd, &data, sizeof(int32_t))) {
#if BYTE_ORDER != BIG_ENDIAN
          tools_swapint32_t(&data);
#endif
          lua_pushnumber(L, data);
        } else
          binio_readobjecterror(L, hnd, 1);
        break;
      }
      case 2: {  /* read number */
        lua_Number n;
        uint64_t num;
        if (sec_read(hnd, &num, sizeof(uint64_t))) {
          n = tools_unswapdouble(num);
          lua_pushnumber(L, n);
        } else
          binio_readobjecterror(L, hnd, 1);
        break;
      }
      case 3: {  /* read string */
        size_t size;
        ssize_t success;
        size = sec_readl(hnd, &success);  /* first read length of the string */
        if (success) {
          char data[size+1];
          if (read(hnd, &data, size) == (ssize_t)size) {  /* now read string itself */
            data[size] = '\0';
            lua_pushlstring(L, data, size);
          }
        } else
          binio_readobjecterror(L, hnd, 1);
        break;
      }
      case 4: {  /* read short string < 256 chars */
        unsigned char size;
        if (sec_read(hnd, &size, sizeof(unsigned char))) {
          char data[size+1];
          if (read(hnd, &data, size) == size) {  /* now read string itself */
            data[size] = '\0';
            lua_pushlstring(L, data, size);
          }
        } else
          binio_readobjecterror(L, hnd, 1);
        break;
      }
      case 5: {  /* read table */
        lua_Number i, size;
        uint64_t num;
        if (sec_read(hnd, &num, sizeof(uint64_t))) {
          size = tools_unswapdouble(num);
          lua_newtable(L);
          if ((data & 32) == 32)  /* has a user-defined type been set ? */
            binio_readobjectutypename(L, hnd, -1);
          for (i=1; i <= size; i++) {
            binio_readobjectaux(L, hnd);
            binio_readobjectaux(L, hnd);
            lua_rawset(L, -3);
          }
        } else
          binio_readobjecterror(L, hnd, 1);
        break;
      }
      case 6: {  /* read set */
        lua_Number i, size;
        uint64_t num;
        if (sec_read(hnd, &num, sizeof(uint64_t))) {
          size = tools_unswapdouble(num);
          agn_createset(L, size);
          for (i=1; i <= size; i++) {
            binio_readobjectaux(L, hnd);
            lua_srawset(L, -2);
          }
        } else
          binio_readobjecterror(L, hnd, 1);
        break;
      }
      case 7: {  /* read sequence */
        lua_Number i, size;
        uint64_t num;
        if (sec_read(hnd, &num, sizeof(uint64_t))) {
          size = tools_unswapdouble(num);
          agn_createseq(L, size);
          if ((data & 32) == 32)  /* has a user-defined type been set ? */
            binio_readobjectutypename(L, hnd, -1);
          /* read contents of sequence */
          for (i=1; i <= size; i++) {
            binio_readobjectaux(L, hnd);
            lua_seqinsert(L, -2);
          }
        } else
          binio_readobjecterror(L, hnd, 1);
        break;
      }
      case 8: {  /* read pair */
        int typeset = ((data & 32) == 32);
        if (typeset) {  /* read user defined type */
          size_t size;
          ssize_t success;
          size = sec_readl(hnd, &success);  /* first read length of the string */
          if (success) {
            char data[size+1];
            if (read(hnd, &data, size) == size) {  /* now read string itself */
              data[size] = '\0';
              lua_pushlstring(L, data, size);
            } else
              binio_readobjecterror(L, hnd, 1);
          } else
            binio_readobjecterror(L, hnd, 1);
        }
        /* read pair */
        binio_readobjectaux(L, hnd);
        binio_readobjectaux(L, hnd);
        agn_createpair(L, -2, -1);  /* assign it */
        lua_remove(L, -2);  /* remove left .. */
        lua_remove(L, -2);  /* .. and right hand-side */
        if (typeset) {  /* now assign user-defined type */
          agn_setutype(L, -1, -2);
          lua_remove(L, -2);  /* and remove utype string */
        }
        break;
      }
      case 9: {  /* read complex */
        lua_Number r, i;
        binio_readobjectaux(L, hnd);  /* read real part (long or double) */
        binio_readobjectaux(L, hnd);  /* read imaginary part (long or double) */
        r = agn_checknumber(L, -2);   /* Agena 1.4.3/1.5.0 */
        i = agn_checknumber(L, -1);   /* Agena 1.4.3/1.5.0 */
        agn_poptoptwo(L);
#ifndef PROPCMPLX
        agn_createcomplex(L, r+I*i);
#else
        agn_createcomplex(L, r, i);
#endif
        break;
      }
      case 10: {  /* read boolean */
        unsigned char type;
        if (read(hnd, &type, sizeof(char)) < 1)
          binio_readobjecterror(L, hnd, 1);
        else {
          if (type == 2)
            lua_pushfail(L);
          else
            lua_pushboolean(L, type);
        }
        break;
      }
      case 11: {  /* read null */
        lua_pushnil(L);
        break;
      }
      case 12: {  /* read procedure, Agena 1.6.1 */
        size_t size;
        ssize_t success;
        size = sec_readl(hnd, &success);  /* first read length of the string */
        if (success) {
          char data[size+1];
          if (read(hnd, &data, size) == (ssize_t)size) {  /* now read `strings.dump` string */
            data[size] = '\0';
            if (luaL_loadbuffer(L, data, size, data) != 0) {
              luaL_error(L, "Error in " LUA_QS ": %s.", "binio.readobject", agn_tostring(L, -1));
            }
          }
        } else
          binio_readobjecterror(L, hnd, 1);
        break;
      }
      default:
        luaL_error(L, "Error in " LUA_QS " with reading file #%d: unknown type.", "binio.readobject", hnd);
    }
  }
  else if (res == 0)
    lua_pushnil(L);
  else
    binio_readobjecterror(L, hnd, 1);
}


#define FILESPEC "AGENAWORKSPACE"
#define FILESPECLEN 14
#define FILESPECVER 1
#define FILESPECSUB 0

static int binio_readobject (lua_State *L) {
  int hnd;
  hnd = agnL_checkinteger(L, 1);
  if (lseek64(hnd, 0L, SEEK_CUR) == -1)
    binio_readobjecterror(L, hnd, 1);
  else {
    char data[FILESPECLEN+1];
    if (read(hnd, &data, FILESPECLEN) == FILESPECLEN) {
      unsigned char version, subversion;
      data[FILESPECLEN] = '\0';
      /* read file header */
      if (strcmp(data, FILESPEC) != 0)
        binio_readobjecterror(L, hnd, 0);
      /* read file version and subversion */
      if (sec_read(hnd, &version, sizeof(unsigned char)) && (version != FILESPECVER))
        binio_readobjecterror(L, hnd, 0);
      if (sec_read(hnd, &subversion, sizeof(unsigned char)) && (subversion < FILESPECSUB))
        binio_readobjecterror(L, hnd, 0);
    } else
      binio_readobjecterror(L, hnd, 1);
    binio_readobjectaux(L, hnd);
  }
  return 1;
}


#define AGN_BINIO_LOCK "binio.lock"
static int binio_lock (lua_State *L) {  /* 0.32.5 */
  int hnd;
  size_t nargs;
  off64_t start, size;
  hnd = agnL_checkinteger(L, 1);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1) {  /* lock entire file (in Windows lock 2^63 bytes only) */
    start = 0;
    size = 0;
  } else {
    /* lock from current file position */
    start = my_fpos(hnd);
    size = agnL_checknumber(L, 2);
    if (size < 0) luaL_error(L, "Error in " LUA_QS ": must lock at least zero or more bytes.", AGN_BINIO_LOCK);  /* 1.12.2 */
  }
  lua_pushboolean(L, my_lock(hnd, start, size) == 0);
  return 1;
}


#define AGN_BINIO_UNLOCK "binio.unlock"
static int binio_unlock (lua_State *L) {  /* 0.32.5 */
  int hnd;
  size_t nargs;
  off64_t start, size;
  hnd = agnL_checkinteger(L, 1);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1) {  /* lock entire file */
    start = 0;
    size = 0;
  } else {
    /* lock from current file position */
    start = my_fpos(hnd);
    size = agnL_checknumber(L, 2);
    if (size < 0) luaL_error(L, "Error in " LUA_QS ": must unlock at least one byte.", AGN_BINIO_UNLOCK);
  }
  lua_pushboolean(L, my_unlock(hnd, start, size) == 0);
  return 1;
}


static const luaL_Reg binio[] = {
  {"close", binio_close},                  /* July 13, 2008 */
  {"filepos", binio_filepos},              /* July 13, 2008 */
  {"length", binio_length},                /* July 13, 2008 */
  {"lock", binio_lock},                    /* June 12, 2010 */
  {"open", binio_open},                    /* July 13, 2008 */
  {"readbytes", binio_readbytes},          /* July 18, 2009 */
  {"readchar", binio_readchar},            /* July 13, 2008 */
  {"readlong", binio_readlong},            /* October 26, 2008 */
  {"readnumber", binio_readnumber},        /* October 25, 2008 */
  {"readstring", binio_readstring},        /* July 20, 2008 */
  {"readshortstring", binio_readshortstring},  /* May 08, 2010 */
  {"rewind", binio_rewind},                /* July 20, 2008 */
  {"seek", binio_seek},                    /* July 13, 2008 */
  {"sync", binio_sync},                    /* July 13, 2008 */
  {"toend", binio_toend},                  /* July 20, 2008 */
  {"unlock", binio_unlock},                /* June 12, 2010 */
  {"writebytes", binio_writebytes},        /* July 19, 2009 */
  {"writechar", binio_writechar},          /* July 19, 2008 */
  {"writelong", binio_writelong},          /* October 26, 2008 */
  {"writenumber", binio_writenumber},      /* October 25, 2008 */
  {"writestring", binio_writestring},      /* July 20, 2008 */
  {"writeshortstring", binio_writeshortstring},  /* May 08, 2010 */
  {"readobject", binio_readobject},        /* May 02, 2010 */
  {NULL, NULL}
};


static void createmetatable (lua_State *L) {
  lua_createtable(L, 0, 1);  /* create metatable for strings */
  lua_pushliteral(L, "");  /* dummy string */
  lua_pushvalue(L, -2);
  lua_setmetatable(L, -2);  /* set string metatable */
  agn_poptop(L);  /* pop dummy string */
  lua_pushvalue(L, -2);  /* string library... */
  lua_setfield(L, -2, "__index");  /* ...is the __index metamethod */
  agn_poptop(L);  /* pop metatable */
}


/*
** Open binio library
*/
LUALIB_API int luaopen_binio (lua_State *L) {
  luaL_register(L, AGENA_BINIOLIBNAME, binio);
  lua_newtable(L);
  lua_setfield(L, -2, "openfiles");  /* table for information on all open files */
  createmetatable(L);
  return 1;
}

