/*
** $Id: bags.c,v 0.1 26.07.2012 $
** Multiset Library
** See Copyright Notice in agena.h
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define bags_c
#define LUA_LIB

#include "agena.h"
#include "agnxlib.h"
#include "agenalib.h"

#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_BAGSLIBNAME "bags"
LUALIB_API int (luaopen_bags) (lua_State *L);
#endif


#define AGN_BAGS_BAG "bags.bag"
static int bags_bag (lua_State *L) {
  int nargs, i;
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  agn_createtable(L, 0, nargs);  /* 1.7.10 */
  lua_pushstring(L, "bag");
  agn_setutype(L, -2, -1);
  agn_poptop(L);
  if (agnL_gettablefield(L, "bags", "aux", AGN_BAGS_BAG, 1) == LUA_TTABLE) {
    lua_getfield(L, -1, "mt");
    if (lua_istable(L, -1)) {
      lua_setmetatable(L, -3);
      agn_poptop(L);  /* pop base.aux */
    } else {
      lua_pop(L, 3);  /* pop anything, base.aux, and `bag` */
      luaL_error(L, "Error in " LUA_QS ": 'bags.aux.mt' metatable table not found.", AGN_BAGS_BAG);
    }
  }  else {
    agn_poptop(L);
    luaL_error(L, "Error in " LUA_QS ": 'bags.aux' table not found.", AGN_BAGS_BAG);
  }
  for (i=0; i < nargs; i++) {
    lua_pushvalue(L, i+1);
    lua_gettable(L, -2);
    if (lua_isnil(L, -1)) {  /* element not yet in bag ? */
      agn_poptop(L);  /* pop null */
      lua_pushvalue(L, i+1);
      lua_pushnumber(L, 1);
      lua_settable(L, -3);
    } else {
      if (!lua_isnumber(L, -1)) {
        agn_poptop(L);  /* pop anything */
        luaL_error(L, "Error in " LUA_QS ": invalid bag.", AGN_BAGS_BAG);
      } else {
        lua_pushvalue(L, i+1);
        lua_pushnumber(L, agn_tonumber(L, -2) + 1);
        lua_settable(L, -4);
        agn_poptop(L);  /* drop number */
      }
    }
  }
  return 1;
}


#define AGN_BAGS_INCLUDE "bags.include"
static int bags_include (lua_State *L) {
  int nargs, i;
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs < 2)
    luaL_error(L, "Error in " LUA_QS ": expected at least two arguments (a bag and an element).", AGN_BAGS_INCLUDE);
  else if (!(lua_istable(L, 1)) || agn_getutype(L, 1) == 0 || strcmp(agn_tostring(L, -1), "bag") != 0) {
    luaL_error(L, "Error in " LUA_QS ": structure is not a bag.", AGN_BAGS_INCLUDE);
  } else
    agn_poptop(L);  /* pop utype string */
  for (i=2; i <= nargs; i++) {
    lua_pushvalue(L, i);
    lua_gettable(L, 1);
    if (lua_isnil(L, -1)) {  /* element not yet in bag ? */
      agn_poptop(L);  /* pop null */
      lua_pushvalue(L, i);
      lua_pushnumber(L, 1);
      lua_settable(L, 1);
    } else {
      if (lua_isnumber(L, -1)) {
        lua_pushvalue(L, i);
        lua_pushnumber(L, agn_tonumber(L, -2) + 1);
        lua_settable(L, 1);
        agn_poptop(L);  /* drop number */
      } else {
        agn_poptop(L);  /* pop anything */
        luaL_error(L, "Error in " LUA_QS ": invalid bag.", AGN_BAGS_INCLUDE);
      }
    }
  }
  return 0;
}


#define AGN_BAGS_MINCLUDE "bags.minclude"
static int bags_minclude (lua_State *L) {  /* based on bags.include */
  int nargs, i;
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs != 2)
    luaL_error(L, "Error in " LUA_QS ": expected at least two arguments (a bag and a sequence).", AGN_BAGS_MINCLUDE);
  else if (!(lua_istable(L, 1)) || agn_getutype(L, 1) == 0 || strcmp(agn_tostring(L, -1), "bag") != 0)
    luaL_error(L, "Error in " LUA_QS ": first argument is not a bag.", AGN_BAGS_MINCLUDE);
  else if (!(lua_isseq(L, 2)))
    luaL_error(L, "Error in " LUA_QS ": second argument is not a sequence.", AGN_BAGS_MINCLUDE);
  else
    agn_poptop(L);  /* pop utype string */
  for (i=1; i <= agn_seqsize(L, 2); i++) {
    lua_seqgeti(L, 2, i);
    lua_gettable(L, 1);
    if (lua_isnil(L, -1)) {  /* element not yet in bag ? */
      agn_poptop(L);  /* pop null */
      lua_seqgeti(L, 2, i);
      lua_pushnumber(L, 1);
      lua_settable(L, 1);
    } else {
      if (lua_isnumber(L, -1)) {
        lua_seqgeti(L, 2, i);
        lua_pushnumber(L, agn_tonumber(L, -2) + 1);
        lua_settable(L, 1);
        agn_poptop(L);  /* drop number */
      } else {
        agn_poptop(L);  /* pop anything */
        luaL_error(L, "Error in " LUA_QS ": invalid bag.", AGN_BAGS_MINCLUDE);
      }
    }
  }
  return 0;
}


#define AGN_BAGS_REMOVE "bags.remove"
static int bags_remove (lua_State *L) {
  int nargs, i;
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs < 2)
    luaL_error(L, "Error in " LUA_QS ": expected at least two arguments (a bag and an element).", AGN_BAGS_REMOVE);
  else if (!(lua_istable(L, 1)) || agn_getutype(L, 1) == 0 || strcmp(agn_tostring(L, -1), "bag") != 0) {
    luaL_error(L, "Error in " LUA_QS ": structure is not a bag.", AGN_BAGS_REMOVE);
  } else
    agn_poptop(L);  /* pop utype string */
  for (i=1; i <= agn_seqsize(L, 2); i++) {
    lua_seqgeti(L, 2, i);
    lua_gettable(L, 1);
    if (lua_isnil(L, -1)) {  /* element not in bag ? */
      agn_poptop(L);  /* pop null and ignore it */
    } else {
      if (lua_isnumber(L, -1)) {
        lua_Number c = agn_tonumber(L, -1);
        lua_seqgeti(L, 2, i);
        if (c == 1)
          lua_pushnil(L);
        else
          lua_pushnumber(L, c - 1);
        lua_settable(L, 1);
        agn_poptop(L);  /* drop number */
      } else {
        agn_poptop(L);  /* pop anything */
        luaL_error(L, "Error in " LUA_QS ": invalid bag.", AGN_BAGS_REMOVE);
      }
    }
  }
  return 0;
}


#define AGN_BAGS_BAGTOSET "bags.bagtoset"
static int bags_bagtoset (lua_State *L) {
  if (!(lua_istable(L, 1)) || agn_getutype(L, 1) == 0 || strcmp(agn_tostring(L, -1), "bag") != 0) {
    luaL_error(L, "Error in " LUA_QS ": structure is not a bag.", AGN_BAGS_BAGTOSET);
  } else
    agn_poptop(L);  /* pop utype string */
  agn_createset(L, agn_size(L, 1));
  lua_pushnil(L);
  while (lua_next(L, 1)) {
    lua_pushvalue(L, -2);  /* put table key at top of the stack */
    lua_sinsert(L, -4);    /* put key into set and pop it */
    agn_poptop(L);         /* remove value */
  }
  return 1;
}


#define AGN_BAGS_ATTRIBS "bags.attribs"
static int bags_attribs (lua_State *L) {
  size_t c = 0;
  if (!(lua_istable(L, 1)) || agn_getutype(L, 1) == 0 || strcmp(agn_tostring(L, -1), "bag") != 0) {
    luaL_error(L, "Error in " LUA_QS ": structure is not a bag.", AGN_BAGS_ATTRIBS);
  } else
    agn_poptop(L);  /* pop utype string */
  lua_pushinteger(L, agn_size(L, 1));  /* number of unique elements */
  lua_pushnil(L);
  while (lua_next(L, 1)) {
    c += luaL_checknumber(L, -1);
    agn_poptop(L);  /* remove value */
  }
  lua_pushinteger(L, c);  /* accumulated number of all occurences of the elements in a bag */
  return 2;
}


static const luaL_Reg bagslib[] = {
  {"bag", bags_bag},                  /* July 26, 2012 */
  {"bagtoset", bags_bagtoset},        /* July 27, 2012 */
  {"include", bags_include},          /* July 26, 2012 */
  {"minclude", bags_minclude},        /* september 03, 2013 */
  {"remove", bags_remove},            /* July 27, 2012 */
  {"attribs", bags_attribs},          /* August 14, 2012 */
  {NULL, NULL}
};


/*
** Open bags library
*/
LUALIB_API int luaopen_bags (lua_State *L) {
  luaL_register(L, AGENA_BAGSLIBNAME, bagslib);
  return 1;
}

