/*
** $Id: astro.c, initiated January 06, 2013 $
** Astronomy library
** See Copyright Notice in agena.h
*/


#include <stdlib.h>
#include <math.h>
#include <string.h>

#define astro_c
#define LUA_LIB

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"
#include "agnhlps.h"
#include "agncmpt.h"

#include "sofa.h"
#include "sunriset.h"
#include "moon.h"


#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_ASTROLIBNAME "astro"
LUALIB_API int (luaopen_astro) (lua_State *L);
#endif


static int ndays[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

/* Returns the sunrise/sunset times in UTC for years starting with 1800 A.D. to 2099 A.D. It is a workhorse function,
   maybe you would like to use `astro.sunriset` for a more convenient interface.

   year, month and day, all integers, are the values of the day to evaluate. lon is the longitude (west/east),
   and lat the latitude (west/east), both in decimal degrees of type float of the location that is of interest.
   Use astro.todec to convert coordinates containing degrees (integer), minutes (integer), and seconds (integer
   or float), and the orientation to decimal degrees.

   Example: astro.sunriseset(2013, 1, 7, astro.dmstodec(6, 46, 58, 'E'), astro.dmstodec(51, 13, 32, 'N'))

   The first and second returns are the sunrise/sunset times which are considered to occur when the Sun's upper
   limb is 35 arc minutes below the horizon (this accounts for the refraction of the Earth's atmosphere).

   The third return is 0, if the rises and sun sets in a day; +1 if the Sun is above the specified "horizon" 24 hours,
   -1 if the Sun is below the specified "horizon" 24 hours.

   The fourth and fifth returns are start and end times of civil twilight. Civil twilight starts/ends when the
   Sun's centre is 6 degrees below the horizon.

   The sixth return is 0, if the rises and sun sets in a day; +1 if the Sun is above the specified "civil twilight
   horizon" 24 hours, -1 if the Sun is below the specified "horizon" 24 hours.

   The seventh and eighth returns are the start and end times of nautical twilight. Nautical twilight starts/ends
   when the Sun's centre is 12 degrees below the horizon.

   The ninth return is 0, if the rises and sun sets in a day; +1 if the Sun is above the specified "nautical twilight
   horizon" 24 hours, -1 if the Sun is below the specified "horizon" 24 hours.

   The tenth and eleventh returns are the start and end times of astronomical twilight. Astronomical twilight starts/ends
   when the Sun's centre is 18 degrees below the horizon.

   The twelfth return is 0, if the rises and sun sets in a day; +1 if the Sun is above the specified "nautical twilight
   horizon" 24 hours, -1 if the Sun is below the specified "astronomical twilight horizon" 24 hours.

   The thirteenth return is the time when the Sun is at south (in decimal UTC).

   All times returned are given in decimal hours of type number. Use `astro.dectotm` to convert them into `tm` notation.
*/

static void checkdateloc (lua_State *L, int year, int month, int day, lua_Number lon, lua_Number lat, const char *procname) {
  if (year < 1801 || year > 2099)
    luaL_error(L, "in `%s`: year must be in the range [1801, 2099].", procname);
  if (month < 1 || month > 12)
    luaL_error(L, "in `%s`: month must be in the range [1, 12].", procname);
  if (day < 1 || day > ndays[month - 1] + (month == 2) * isleapyear(year))  /* 1.9.1 */
    luaL_error(L, "in `%s`: day must be in the range [1, %d].", ndays[month - 1], procname);
  if (lon < -180 || lon > 180)  /* longitude = Lngengrad (E/W) */
    luaL_error(L, "in `%s`: longitude must be in the range [-180, 180].", procname);
  if (lat < -90 || lat > 90)  /* latitude = Breitengrad (N/S) */
    luaL_error(L, "in `%s`: latitude must be in the range [-90, 90].", procname);
}

static int astro_sunriseset (lua_State *L) {
  int year, month, day, r[4];
  lua_Number lon, lat, rise, set, civilrise, civilset, nauticalrise, nauticalset, astrorise, astroset, south;
  year = agn_checknumber(L, 1);
  month = agn_checknumber(L, 2);
  day = agn_checknumber(L, 3);
  lon = agn_checknumber(L, 4);
  lat = agn_checknumber(L, 5);
  checkdateloc(L, year, month, day, lon, lat, "astro.sunriseset");
  r[0] = sun_rise_set(year, month, day, lon, lat, &rise, &set, &south);
  r[1] = civil_twilight(year, month, day, lon, lat, &civilrise, &civilset, &south);
  r[2] = nautical_twilight(year, month, day, lon, lat, &nauticalrise, &nauticalset, &south);
  r[3] = astronomical_twilight(year, month, day, lon, lat, &astrorise, &astroset, &south);
  lua_pushnumber(L, rise);
  lua_pushnumber(L, set);
  lua_pushnumber(L, r[0]);
    /* 0 if Sun is not above/below the limb, +1 if sun is above the limb for 24 hours, -1 if it is below the limb for 24 hours. */
  lua_pushnumber(L, civilrise);
  lua_pushnumber(L, civilset);
  lua_pushnumber(L, r[1]);
  lua_pushnumber(L, nauticalrise);
  lua_pushnumber(L, nauticalset);
  lua_pushnumber(L, r[2]);
  lua_pushnumber(L, astrorise);
  lua_pushnumber(L, astroset);
  lua_pushnumber(L, r[3]);
  lua_pushnumber(L, south);
  return 13;
}


static int checkcoords (lua_State *L, lua_Number x, lua_Number y, lua_Number z, const char *d, const char *procname) {
  /* throw an exception with a negative x */
  if ((strcmp(d, "N") != 0) && (strcmp(d, "S") != 0) && (strcmp(d, "W") != 0 ) && (strcmp(d, "E") != 0))
    luaL_error(L, "in `%s`: fourth argument, unknown orientation `%s`.", procname, d);
  if ((strcmp(d, "N") == 0 || strcmp(d, "S") == 0) && (x < 0 || x >= 90))
    luaL_error(L, "in `%s`: latitude must be in the range [0, 90[.", procname);
  else if ((strcmp(d, "E") == 0 || strcmp(d, "W") == 0) && (x < 0 || x >= 180))
    luaL_error(L, "in `%s`: longitude must be in the range [0, 180[.", procname);
  if (y < 0 || y >= 60)
    luaL_error(L, "in `%s`: second argument must be in the range [0, 60[.", procname);
  if (z < 0 || z >= 60)
    luaL_error(L, "in `%s`: third argument must be in the range [0, 60[.", procname);
  if (strcmp(d, "S") == 0 || strcmp(d, "W") == 0)
    return -x;
  else
    return x;
}

/* Converts coordinates in the form degree, minute, second, and their orientation 'N', 'S', 'W', or 'E' (DMS format)
   to their corresponding decimal degree representation (DegDec format). The return is a number. */
static int astro_dmstodec (lua_State *L) {
  lua_Number x, y, z, fraction;
  const char *d;
  x = agn_checknumber(L, 1);
  y = agn_checknumber(L, 2);
  z = agn_checknumber(L, 3);
  d = agn_checkstring(L, 4);
  x = checkcoords(L, x, y, z, d, "astro.dmstodec");
  /* Calculate the total number of seconds of the fraction */
  fraction = (y * 60 + z)/3600;
  lua_pushnumber(L, x + (x < 0 ? -fraction : fraction));
  return 1;
}


/* Converts coordinates x in decimal degrees (a number) to the form degree, minute, second, and their orientation
   'N', 'S', 'W', or 'E' (DMS format). You must also specify whether to compute latitude or longitude values,
   by passing the strings "lat" or "lon", respectively. The return are three numbers and the orientation, a string. */
static int astro_dectodms (lua_State *L) {
  lua_Number x, tx, minute, second;
  const char *o;
  x = agn_checknumber(L, 1);
  o = agn_checkstring(L, 2);
  if (strcmp(o, "lat") != 0 && strcmp(o, "lon") != 0)
    luaL_error(L, "in `%s`: second argument must be either 'lon' or 'lat'.", "astro.dectogeo");
  tx = trunc(x);
  minute = fabs(tx-x) * 60;
  second = fabs(trunc(minute)-minute) * 60;
  lua_pushnumber(L, fabs(tx));
  lua_pushnumber(L, trunc(minute));
  lua_pushnumber(L, second);
  lua_pushstring(L, (strcmp(o, "lat") == 0) ? (x < 0 ? "S" : "N") : (x < 0 ? "W" : "E"));
  return 4;
}


/* Converts a Julian date (a number) to its Gregorian representation: the year, the month, and the day (all integers). */
/* Test case: cdate(2456299) -> 2013    1       6 */

/* # taken from an Excel file import library written for Modula-2. C port written on January 07, 2013.
   The original Modula-2 procedure had presumably been published at modula2.org, and was at least available
   around March 01, 2004. The author unfortunately is unknown, but the Modula-2 code mentions `Numerical Recipes`
   (the book ?) as the possible original FORTRAN or C algorithm. As of 2013, the Modula-2 code is no longer available. */

static void gregorian_day (double julian, int *y, int *m, int *d) {
  /* From Numerical Recipies */
  int ja, jalpha, jb, jc, jd, je, iyyy, mm, id, greg;
  greg = 2299161;
  if (julian >= greg) {
    jalpha = (int)((((double)julian - 1867216.0) - 0.25)/36524.25);
    ja = julian + 1 + jalpha - (int)(0.25 * (double)jalpha);
  }
  else if (julian < 0)
    ja = julian + 36525*(1 - intdiv(julian, 36525));
  else
    ja = julian;
  jb = ja + 1524;
  jc = (int)(6680.0 + (((double)jb - 2439870.0) - 122.1)/365.25);
  jd = 365 * jc + (int)(0.25 * (double)jc);
  je = trunc((double)(jb - jd)/30.6001);
  id = jb - jd - (int)(30.6001 * (double)je);
  mm = je - 1;
  if (mm > 12) mm -= 12;
  iyyy = jc - 4715;
  if (mm > 2) iyyy--;
  if (iyyy <= 0) iyyy--;
  if (julian < 0) iyyy = iyyy - 100*(1 - intdiv(julian, 36525));
  *y = iyyy;
  *m = mm;
  *d = id;
  return;
}

static int astro_cdate (lua_State *L) {
  int yy, mm, dd;
  gregorian_day(agn_checknumber(L, 1), &yy, &mm, &dd);
  lua_pushnumber(L, yy);
  lua_pushnumber(L, mm);
  lua_pushnumber(L, dd);
  return 3;
}


/* Converts a Gregorian date represented by year, month, day and optionally hour, minute, and second (all numbers) to the
   corresponding Julian date. The return is a number, or `fail` if the date or time is of a wrong format. */
/* Test case: jdate(2013, 1, 6) -> 2456298.5 */

static int astro_jdate (lua_State *L) {
  int yy, mm, dd, h, m, s;
  lua_Number r;
  yy = agn_checknumber(L, 1);
  mm = agn_checknumber(L, 2);
  dd = agn_checknumber(L, 3);
  h = luaL_optint(L, 4, 0);
  m = luaL_optint(L, 5, 0);
  s = luaL_optint(L, 6, 0);
  r = juliandate(yy, mm, dd, h, m, s);
  if (r == HUGE_VAL)
    lua_pushfail(L);
  else
    lua_pushnumber(L, r);
  return 1;
}


static int astro_isleapyear (lua_State *L) {
  int y;
  y = agn_checknumber(L, 1);
  lua_pushboolean(L, isleapyear(y));
  return 1;
}


/* Takes a year, a month, a day, and an hour (all numbers) and returns the moon phase as a real number in the
   range [0, 1], where 0 is new moon and 1 is full Moon; and an integer in the range [0, 7], where 0 indicates
   new moon and 4 indicates full moon. */
static int astro_moonphase (lua_State *L) {
  int y, m, d, ip;
  lua_Number h, r;
  y = agn_checknumber(L, 1);
  m = agn_checknumber(L, 2);
  d = agn_checknumber(L, 3);
  h = luaL_optnumber(L, 4, 0);
  r = moon_phase(y, m, d, h, &ip);
  lua_pushnumber(L, r);
  lua_pushnumber(L, ip);
  return 2;
}

/* Returns the times of Lunar rise and set in GMT. Receives the year, month day, and the longitude (all of type number)
   and returns two numbers: the GMT rise time in a decimal, and the GMT set time also in a decimal. Use `clock.dectotm`
   to convert the rise and set times to sexagesimal format, or try `astro.moon`.

   Test case: astro.moonriseset(2013, 1, 8, astro.dmstodec(7, 6, 0, 'E'), astro.dmstodec(50, 43, 48, 'N')) ->
   3.7666666666667 12.566666666667 */
static int astro_moonriseset (lua_State *L) {
  double year, month, day, lon, lat, rise, set;
  year = agn_checknumber(L, 1);
  month = agn_checknumber(L, 2);
  day = agn_checknumber(L, 3);
  lon = agn_checknumber(L, 4);
  lat = agn_checknumber(L, 5);
  checkdateloc(L, year, month, day, lon, lat, "astro.moonriseset");
  riseset(lat, lon, day, month, year, 0, &rise, &set);
  lua_pushnumber(L, rise);
  lua_pushnumber(L, set);
  return 2;
}


static const luaL_Reg astrolib[] = {
  {"cdate", astro_cdate},                    /* added on January 07, 2013 */
  {"dectodms", astro_dectodms},              /* added on January 08, 2013 */
  {"dmstodec", astro_dmstodec},              /* added on January 06, 2013 */
  {"isleapyear", astro_isleapyear},          /* added on January 07, 2013 */
  {"jdate", astro_jdate},                    /* added on January 07, 2013 */
  {"moonphase", astro_moonphase},            /* added on January 08, 2013 */
  {"moonriseset", astro_moonriseset},        /* added on January 08, 2013 */
  {"sunriseset", astro_sunriseset},          /* added on January 06, 2013 */
  {NULL, NULL}
};


/*
** Open astro library
*/
LUALIB_API int luaopen_astro (lua_State *L) {
  luaL_register(L, AGENA_ASTROLIBNAME, astrolib);
  return 1;
}

