/*
** $Id: agnxlib.c, 15.03.2009 12:37:08 $
** Auxiliary functions for building Agena libraries
** Revision 08.08.2009 22:14:11
** See Copyright Notice in agena.h
*/


#include <ctype.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined(__unix__) || defined(__OS2__) || defined(__APPLE__) || defined(__HAIKU__)
#include <unistd.h>   /* for access */
#endif

#if _WIN32
#include <io.h>       /* for access */
#endif

/* This file uses only the official API of Lua.
** Any function declared here could be written as an application function.
*/

#define agnxlib_c
#define LUA_LIB

#include "agena.h"
#include "agnxlib.h"
#include "agnhlps.h"

#define FREELIST_REF	0	/* free list of references */

/* convert a stack index to positive */
#define abs_index(L, i)		((i) > 0 || (i) <= LUA_REGISTRYINDEX ? (i) : \
					lua_gettop(L) + (i) + 1)


/*
** {======================================================
** Error-report functions
** =======================================================
*/


LUALIB_API int luaL_argerror (lua_State *L, int narg, const char *extramsg) {
  lua_Debug ar;
  if (!lua_getstack(L, 0, &ar))  /* no stack frame? */
    return luaL_error(L, "Wrong argument #%d: %s.", narg, extramsg);
  lua_getinfo(L, "n", &ar);
  if (strcmp(ar.namewhat, "method") == 0) {
    narg--;  /* do not count `self' */
    if (narg == 0)  /* error is in the self argument itself? */
      return luaL_error(L, "Calling " LUA_QS " on bad self (%s).",
                           ar.name, extramsg);
  }
  if (ar.name == NULL)
    ar.name = "?";
  return luaL_error(L, "Wrong argument #%d to " LUA_QS ": %s.",
                        narg, ar.name, extramsg);
}


LUALIB_API int luaL_typeerror (lua_State *L, int narg, const char *extramsg, int type) {
  lua_Debug ar;
  if (!lua_getstack(L, 0, &ar))  /* no stack frame? */
    return luaL_error(L, "Wrong argument #%d: %s.", narg, extramsg);
  lua_getinfo(L, "n", &ar);
  if (strcmp(ar.namewhat, "method") == 0) {
    narg--;  /* do not count `self' */
    if (narg == 0)  /* error is in the self argument itself? */
      return luaL_error(L, "Calling " LUA_QS " on bad self (%s).",
                           ar.name, extramsg);
  }
  if (ar.name == NULL)
    ar.name = "?";
  return luaL_error(L, "Wrong argument #%d to " LUA_QS ": %s, got %s.",
                        narg, ar.name, extramsg, lua_typename(L, type));
}


LUALIB_API int luaL_typerror (lua_State *L, int narg, const char *tname) {
  const char *msg = lua_pushfstring(L, "%s expected, got %s",
                                    tname, luaL_typename(L, narg));
  return luaL_argerror(L, narg, msg);
}


LUALIB_API void tag_error (lua_State *L, int narg, int tag) {
  luaL_typerror(L, narg, lua_typename(L, tag));
}


LUALIB_API void luaL_where (lua_State *L, int level) {
  lua_Debug ar;
  if (lua_getstack(L, level, &ar)) {  /* check function at level */
    lua_getinfo(L, "Sl", &ar);  /* get info about it */
    if (ar.currentline > 0) {  /* is there info? */
      /* lua_pushfstring(L, "%s:%d: ", ar.short_src, ar.currentline); */
      lua_pushfstring(L, "", ar.currentline);
      return;
    }
  }
  lua_pushliteral(L, "");  /* else, no information available... */
}


LUALIB_API int luaL_error (lua_State *L, const char *fmt, ...) {
  va_list argp;
  va_start(argp, fmt);
  luaL_where(L, 1);
  lua_pushvfstring(L, fmt, argp);
  va_end(argp);
  lua_concat(L, 2);
  return lua_error(L);
}

/* }====================================================== */


LUALIB_API int luaL_checkoption (lua_State *L, int narg, const char *def,
                                 const char *const lst[]) {
  const char *name = (def) ? luaL_optstring(L, narg, def) :
                             luaL_checkstring(L, narg);
  int i;
  for (i=0; lst[i]; i++)
    if (strcmp(lst[i], name) == 0)
      return i;
  return luaL_argerror(L, narg,
                       lua_pushfstring(L, "invalid option " LUA_QS, name));
}


LUALIB_API int luaL_newmetatable (lua_State *L, const char *tname) {
  lua_getfield(L, LUA_REGISTRYINDEX, tname);  /* get registry.name */
  if (!lua_isnil(L, -1))  /* name already in use? */
    return 0;  /* leave previous value on top, but return 0 */
  agn_poptop(L);
  lua_newtable(L);  /* create metatable */
  lua_pushvalue(L, -1);
  lua_setfield(L, LUA_REGISTRYINDEX, tname);  /* registry.name = metatable */
  return 1;
}


LUALIB_API void *luaL_checkudata (lua_State *L, int ud, const char *tname) {
  void *p = lua_touserdata(L, ud);
  if (p != NULL) {  /* value is a userdata? */
    if (lua_getmetatable(L, ud)) {  /* does it have a metatable? */
      lua_getfield(L, LUA_REGISTRYINDEX, tname);  /* get correct metatable */
      if (lua_rawequal(L, -1, -2)) {  /* does it have the correct mt? */
        agn_poptoptwo(L);  /* remove both metatables */
        return p;
      }
    }
  }
  luaL_typerror(L, ud, tname);  /* else error */
  return NULL;  /* to avoid warnings */
}


LUALIB_API void *luaL_getudata (lua_State *L, int ud, const char *tname, int *result) {  /* 0.12.2 */
  void *p = lua_touserdata(L, ud);
  if (p != NULL) {  /* value is a userdata? */
    if (lua_getmetatable(L, ud)) {  /* does it have a metatable? */
      lua_getfield(L, LUA_REGISTRYINDEX, tname);  /* get correct metatable */
      if (lua_rawequal(L, -1, -2)) {  /* does it have the correct mt? */
        agn_poptoptwo(L);  /* remove both metatables */
        *result = 1;
        return p;
      }
    }
  }
  *result = 0;
  return NULL;
}


LUALIB_API void luaL_checkstack (lua_State *L, int space, const char *mes) {
  if (!lua_checkstack(L, space))
    luaL_error(L, "stack overflow (%s).", mes);
}


LUALIB_API void luaL_checktype (lua_State *L, int narg, int t) {
  if (lua_type(L, narg) != t)
    tag_error(L, narg, t);
}


LUALIB_API void luaL_checkany (lua_State *L, int narg) {
  if (lua_type(L, narg) == LUA_TNONE)
    luaL_argerror(L, narg, "value expected");
}


LUALIB_API const char *luaL_checklstring (lua_State *L, int narg, size_t *len) {
  const char *s = lua_tolstring(L, narg, len);
  if (!s) tag_error(L, narg, LUA_TSTRING);
  return s;
}


LUALIB_API const char *luaL_optlstring (lua_State *L, int narg,
                                        const char *def, size_t *len) {
  if (lua_isnoneornil(L, narg)) {
    if (len)
      *len = (def ? strlen(def) : 0);
    return def;
  }
  else return luaL_checklstring(L, narg, len);
}


LUALIB_API lua_Number luaL_checknumber (lua_State *L, int narg) {
  lua_Number d = lua_tonumber(L, narg);
  if (d == 0 && !lua_isnumber(L, narg))  /* avoid extra test when d is not 0 */
    tag_error(L, narg, LUA_TNUMBER);
  return d;
}


LUALIB_API lua_Number agnL_checknumber (lua_State *L, int narg) {  /* Agena 1.4.3/1.5.0 */
  if (!agn_isnumber(L, narg))  /* 0.5.3 */
    tag_error(L, narg, LUA_TNUMBER);
  return agn_tonumber(L, narg);
}


LUALIB_API int agnL_checkboolean (lua_State *L, int narg) {  /* Agena 1.6.0 */
  if (!lua_isboolean(L, narg))
    tag_error(L, narg, LUA_TBOOLEAN);
  return lua_toboolean(L, narg);  /* -1 for fail, 0 for false, 1 for true */
}


LUALIB_API lua_Number luaL_optnumber (lua_State *L, int narg, lua_Number def) {
  return luaL_opt(L, luaL_checknumber, narg, def);
}


/* If the value at stack index narg is a number, returns this number. If this argument is absent or is NULL,
   returns def. Otherwise, raises an error. The function does not try to convert a string into a number,
   see luaL_optnumber instead. */
LUALIB_API lua_Number agnL_optnumber (lua_State *L, int narg, lua_Number def) {  /* Agena 1.4.3/1.5.0 */
  return luaL_opt(L, agnL_checknumber, narg, def);
}


/* If the value at stack index narg is a Boolean, returns this Booelan as an integer: -1 for fail, 0 for false, and 1 for true.
   If there is no value at index narg or is NULL, returns def. Otherwise, raises an error. */
LUALIB_API int agnL_optboolean (lua_State *L, int narg, int def) {  /* Agena 1.6.0 */
  return luaL_opt(L, agnL_checkboolean, narg, def);
}


LUALIB_API lua_Integer luaL_checkinteger (lua_State *L, int narg) {
  lua_Integer d = lua_tointeger(L, narg);
  if (d == 0 && !lua_isnumber(L, narg))  /* avoid extra test when d is not 0 */
    tag_error(L, narg, LUA_TNUMBER);
  return d;
}


LUALIB_API lua_Integer agnL_checkinteger (lua_State *L, int narg) {  /* Agena 1.4.3/1.5.0 */
  if (!agn_isnumber(L, narg))  /* 0.5.3 */
    tag_error(L, narg, LUA_TNUMBER);
  return lua_tointeger(L, narg);
}


LUALIB_API int32_t luaL_checkint32_t (lua_State *L, int narg) {
  if (!agn_isnumber(L, narg))  /* avoid extra test when d is not 0 */  /* 0.5.3 */
    tag_error(L, narg, LUA_TNUMBER);
  return lua_toint32_t(L, narg);
}


LUALIB_API off64_t luaL_checkoff64_t (lua_State *L, int narg) {
  if (!agn_isnumber(L, narg))  /* avoid extra test when d is not 0 */  /* 0.5.3 */
    tag_error(L, narg, LUA_TNUMBER);
  return lua_tooff64_t(L, narg);
}


LUALIB_API lua_Integer luaL_optinteger (lua_State *L, int narg,
                                                      lua_Integer def) {
  return luaL_opt(L, luaL_checkinteger, narg, def);
}


LUALIB_API lua_Integer agnL_optinteger (lua_State *L, int narg,  /* Agena 1.4.3/1.5.0 */
                                                      lua_Integer def) {
  return luaL_opt(L, agnL_checkinteger, narg, def);
}


LUALIB_API int32_t luaL_optint32_t (lua_State *L, int narg, int32_t def) {
  return luaL_opt(L, luaL_checkint32_t, narg, def);
}


LUALIB_API off64_t luaL_optoff64_t (lua_State *L, int narg, off64_t def) {
  return luaL_opt(L, luaL_checkoff64_t, narg, def);
}


LUALIB_API int luaL_getmetafield (lua_State *L, int obj, const char *event) {
  if (!lua_getmetatable(L, obj))  /* no metatable? */
    return 0;
  lua_pushstring(L, event);
  lua_rawget(L, -2);
  if (lua_isnil(L, -1)) {
    agn_poptoptwo(L);  /* remove metatable and metafield */
    return 0;
  }
  else {
    lua_remove(L, -2);  /* remove only metatable */
    return 1;
  }
}


LUALIB_API int luaL_callmeta (lua_State *L, int obj, const char *event) {
  obj = abs_index(L, obj);
  if (!luaL_getmetafield(L, obj, event))  /* no metafield? */
    return 0;
  lua_pushvalue(L, obj);
  lua_call(L, 1, 1);
  return 1;
}


LUALIB_API void (luaL_register) (lua_State *L, const char *libname,
                                const luaL_Reg *l) {
  luaI_openlib(L, libname, l, 0);
}


static int libsize (const luaL_Reg *l) {
  int size = 0;
  for (; l->name; l++) size++;
  return size;
}


LUALIB_API void luaI_openlib (lua_State *L, const char *libname,
                              const luaL_Reg *l, int nup) {
  if (libname) {
    int size = libsize(l);
    /* check whether lib already exists */
    luaL_findtable(L, LUA_REGISTRYINDEX, "_LOADED", 1);  /* Lua 5.1.4 version */
    lua_getfield(L, -1, libname);  /* get _LOADED[libname] */
    if (!lua_istable(L, -1)) {  /* not found? */
      agn_poptop(L);  /* remove previous result */
      /* try global variable (and create one if it does not exist) */
      if (luaL_findtable(L, LUA_GLOBALSINDEX, libname, size) != NULL)
        luaL_error(L, "name conflict for module " LUA_QS ".", libname);
      lua_pushvalue(L, -1);
      lua_setfield(L, -3, libname);  /* _LOADED[libname] = new table */
    }
    lua_remove(L, -2);  /* remove _LOADED table */
    lua_insert(L, -(nup+1));  /* move library table to below upvalues */
  }
  for (; l->name; l++) {
    int i;
    for (i=0; i<nup; i++)  /* copy upvalues to the top */
      lua_pushvalue(L, -nup);
    lua_pushcclosure(L, l->func, nup);
    lua_setfield(L, -(nup+2), l->name);
  }
  lua_pop(L, nup);  /* remove upvalues */
}

/* }====================================================== */

LUALIB_API const char *luaL_gsub (lua_State *L, const char *s, const char *p,
                                                               const char *r) {
  const char *wild;
  size_t l = strlen(p);
  luaL_Buffer b;
  luaL_buffinit(L, &b);
  while ((wild = strstr(s, p)) != NULL) {
    luaL_addlstring(&b, s, wild - s);  /* push prefix */
    luaL_addstring(&b, r);  /* push replacement in place of pattern */
    s = wild + l;  /* continue after `p' */
  }
  luaL_addstring(&b, s);  /* push last suffix */
  luaL_pushresult(&b);
  return lua_tostring(L, -1);
}


LUALIB_API const char *luaL_findtable (lua_State *L, int idx,
                                       const char *fname, int szhint) {
  const char *e;
  lua_pushvalue(L, idx);
  do {
    e = strchr(fname, '.');
    if (e == NULL) e = fname + strlen(fname);
    lua_pushlstring(L, fname, e - fname);
    lua_rawget(L, -2);
    if (lua_isnil(L, -1)) {  /* no such field? */
      agn_poptop(L);  /* remove this nil */
      lua_createtable(L, 0, (*e == '.' ? 1 : szhint)); /* new table for field */
      lua_pushlstring(L, fname, e - fname);
      lua_pushvalue(L, -2);
      lua_settable(L, -4);  /* set new table into field */
    }
    else if (!lua_istable(L, -1)) {  /* field has a non-table value? */
      agn_poptoptwo(L);  /* remove table and value */
      return fname;  /* return problematic part of the name */
    }
    lua_remove(L, -2);  /* remove previous table */
    fname = e + 1;
  } while (*e == '.');
  return NULL;
}



/*
** {======================================================
** Generic Buffer manipulation
** =======================================================
*/


#define bufffree(B)	((size_t)(LUAL_BUFFERSIZE - bufflen(B)))

#define LIMIT	(LUA_MINSTACK/2)


int emptybuffer (luaL_Buffer *B) {
  size_t l = bufflen(B);
  if (l == 0) return 0;  /* put nothing on stack */
  else {
    lua_pushlstring(B->L, B->buffer, l);
    B->p = B->buffer;
    B->lvl++;
    return 1;
  }
}


static void adjuststack (luaL_Buffer *B) {
  if (B->lvl > 1) {
    lua_State *L = B->L;
    int toget = 1;  /* number of levels to concat */
    size_t toplen = lua_strlen(L, -1);
    do {
      size_t l = lua_strlen(L, -(toget+1));
      if (B->lvl - toget + 1 >= LIMIT || toplen > l) {
        toplen += l;
        toget++;
      }
      else break;
    } while (toget < B->lvl);
    lua_concat(L, toget);
    B->lvl = B->lvl - toget + 1;
  }
}


LUALIB_API char *luaL_prepbuffer (luaL_Buffer *B) {
  if (emptybuffer(B))
    adjuststack(B);
  return B->buffer;
}


LUALIB_API void luaL_addlstring (luaL_Buffer *B, const char *s, size_t l) {
  while (l--)
    luaL_addchar(B, *s++);
}


LUALIB_API void luaL_addstring (luaL_Buffer *B, const char *s) {
  luaL_addlstring(B, s, strlen(s));
}


LUALIB_API void luaL_pushresult (luaL_Buffer *B) {
  emptybuffer(B);
  lua_concat(B->L, B->lvl);
  B->lvl = 1;
}


LUALIB_API void luaL_addvalue (luaL_Buffer *B) {
  lua_State *L = B->L;
  size_t vl;
  const char *s = lua_tolstring(L, -1, &vl);
  if (vl <= bufffree(B)) {  /* fit into buffer? */
    memcpy(B->p, s, vl);  /* put it there */
    B->p += vl;
    agn_poptop(L);  /* remove from stack */
  }
  else {
    if (emptybuffer(B))
      lua_insert(L, -2);  /* put buffer before new value */
    B->lvl++;  /* add new value into B stack */
    adjuststack(B);
  }
}


LUALIB_API void luaL_buffinit (lua_State *L, luaL_Buffer *B) {
  B->L = L;
  B->p = B->buffer;
  B->lvl = 0;
}

/* }====================================================== */


LUALIB_API int luaL_ref (lua_State *L, int t) {
  int ref;
  t = abs_index(L, t);
  if (lua_isnil(L, -1)) {
    agn_poptop(L);  /* remove from stack */
    return LUA_REFNIL;  /* `nil' has a unique fixed reference */
  }
  lua_rawgeti(L, t, FREELIST_REF);  /* get first free element */
  ref = (int)lua_tointeger(L, -1);  /* ref = t[FREELIST_REF] */
  agn_poptop(L);  /* remove it from stack */
  if (ref != 0) {  /* any free element? */
    lua_rawgeti(L, t, ref);  /* remove it from list */
    lua_rawseti(L, t, FREELIST_REF);  /* (t[FREELIST_REF] = t[ref]) */
  }
  else {  /* no free elements */
    ref = (int)lua_objlen(L, t);
    ref++;  /* create new reference */
  }
  lua_rawseti(L, t, ref);
  return ref;
}


LUALIB_API void luaL_unref (lua_State *L, int t, int ref) {
  if (ref >= 0) {
    t = abs_index(L, t);
    lua_rawgeti(L, t, FREELIST_REF);
    lua_rawseti(L, t, ref);  /* t[ref] = t[FREELIST_REF] */
    lua_pushinteger(L, ref);
    lua_rawseti(L, t, FREELIST_REF);  /* t[FREELIST_REF] = ref */
  }
}



/*
** {======================================================
** Load functions
** =======================================================
*/

typedef struct LoadF {
  int extraline;
  FILE *f;
  char buff[LUAL_BUFFERSIZE];
} LoadF;


/* 0.13.3 */
static const char *getF (lua_State *L, void *ud, size_t *size) {
  LoadF *lf = (LoadF *)ud;
  (void)L;
  if (lf->extraline) {
    lf->extraline = 0;
    *size = 1;
    return "\n";
  }
  if (feof(lf->f)) return NULL;
  *size = fread(lf->buff, 1, sizeof(lf->buff), lf->f);
  return (*size > 0) ? lf->buff : NULL;
}


static int errfile (lua_State *L, const char *what, int fnameindex) {
  const char *serr = strerror(errno);
  const char *filename = lua_tostring(L, fnameindex) + 1;
  lua_pushfstring(L, "cannot %s %s: %s", what, filename, serr);
  lua_remove(L, fnameindex);
  return LUA_ERRFILE;
}


/* 0.13.3 */
LUALIB_API int luaL_loadfile (lua_State *L, const char *filename) {
  LoadF lf;
  int status, readstatus;
  int c;
  int fnameindex = lua_gettop(L) + 1;  /* index of filename on the stack */
  lf.extraline = 0;
  if (filename == NULL) {
    lua_pushliteral(L, "=stdin");
    lf.f = stdin;
  }
  else {
    lua_pushfstring(L, "@%s", filename);
    lf.f = fopen(filename, "r");
    if (lf.f == NULL) return errfile(L, "open", fnameindex);
  }
  c = getc(lf.f);
  if (c == '#') {  /* Unix exec. file? */
    lf.extraline = 1;
    while ((c = getc(lf.f)) != EOF && c != '\n') ;  /* skip first line */
    if (c == '\n') c = getc(lf.f);
  }
  if (c == LUA_SIGNATURE[0] && filename) {  /* binary file? */
    lf.f = freopen(filename, "rb", lf.f);  /* reopen in binary mode */
    if (lf.f == NULL) return errfile(L, "reopen", fnameindex);
    /* skip eventual `#!...' */
   while ((c = getc(lf.f)) != EOF && c != LUA_SIGNATURE[0]) ;
    lf.extraline = 0;
  }
  ungetc(c, lf.f);
  status = lua_load(L, getF, &lf, lua_tostring(L, -1));
  readstatus = ferror(lf.f);
  if (filename) fclose(lf.f);  /* close file (even in case of errors) */
  if (readstatus) {
    lua_settop(L, fnameindex);  /* ignore results from `lua_load' */
    return errfile(L, "read", fnameindex);
  }
  lua_remove(L, fnameindex);
  return status;
}


typedef struct LoadS {
  const char *s;
  size_t size;
} LoadS;


static const char *getS (lua_State *L, void *ud, size_t *size) {
  LoadS *ls = (LoadS *)ud;
  (void)L;
  if (ls->size == 0) return NULL;
  *size = ls->size;
  ls->size = 0;
  return ls->s;
}


LUALIB_API int luaL_loadbuffer (lua_State *L, const char *buff, size_t size,
                                const char *name) {
  LoadS ls;
  ls.s = buff;
  ls.size = size;
  return lua_load(L, getS, &ls, name);
}


LUALIB_API int (luaL_loadstring) (lua_State *L, const char *s) {
  return luaL_loadbuffer(L, s, strlen(s), s);
}



/* }====================================================== */


static void *l_alloc (void *ud, void *ptr, size_t osize, size_t nsize) {
  (void)ud;
  (void)osize;
  if (nsize == 0) {
    free(ptr);
    return NULL;
  }
  else
    return realloc(ptr, nsize);
}


static int panic (lua_State *L) {
  (void)L;  /* to avoid warnings */
  fprintf(stderr, "PANIC: unprotected error in call to Agena API (%s)\n",
                   lua_tostring(L, -1));
  return 0;
}


LUALIB_API lua_State *luaL_newstate (void) {
  lua_State *L = lua_newstate(l_alloc, NULL);
  if (L) lua_atpanic(L, &panic);
  return L;
}

/* Agena initialisation (loading library.agn & agena.ini), 0.24.0 */

void printwarning (int i, const char *path) {
  if (i)
    fprintf(stderr, "\nWarning, main Agena library not found in:\n\n   %s\n\n"
                    "Please set or reset libname manually.\n\n", path);
  else {  /* Agena 1.3.1 fix */
    fprintf(stderr, "Warning, unsuccessful initialisation. Could not determine the path to the main\n"
                    "Agena library directory");
    if (path != NULL)
       fprintf(stderr, " in\n\n   %s.\n\n", path);
    else
       fprintf(stderr, ".\n\n");
  }
  fprintf(stderr, "Some functions will not work. Please set or reset libname manually.\n\n");
  fprintf(stderr, "You should permanently set the operating system variable AGENAPATH to the path\n"
                  "of the main Agena library folder by adding the following line to your profile\n"
                  "file:\n\n"
#if defined(__unix__) || defined(__APPLE__) || defined(__HAIKU__)
                  "   export AGENAPATH=/usr/agena/lib\n\n"
                  "in UNIX.\n\n"
#else
                  "   set AGENAPATH=<drive letter>:/agena/lib\n\n"
                  "in OS/2, Windows, or DOS.\n\n"
#endif
                  "Alternatively, you may change into the Agena folder and run Agena from there.\n");
#ifdef __OS2__
  fprintf(stderr, "\nIf you run Agena the first time in OS/2, just reboot your computer for the\n"
                  "addition of AGENAPATH to PATH in CONFIG.SYS to take effect.\n");
#endif
  if (!i) fprintf(stderr, "\n");
  fflush(stderr);
}

#define debuginfo(debug, string) { \
  if (debug) { \
    fprintf(stderr, string); fflush(stderr); \
  } \
}

#define debuginfo2(debug, string1, string2) { \
  if (debug) { \
    fprintf(stderr, string1, string2); fflush(stderr); \
  } \
}

LUALIB_API const char *luaL_pushnexttemplate (lua_State *L, const char *path) {
  const char *l;
  while (*path == *LUA_PATHSEP) path++;  /* skip separators */
  if (*path == '\0') return NULL;  /* no more templates */
  l = strchr(path, *LUA_PATHSEP);  /* find next separator */
  if (l == NULL) l = path + strlen(path);
  lua_pushlstring(L, path, l - path);  /* template */
  return l;
}

/* changed Agena 1.0.2 */
static void agnL_initialise_auxloop (lua_State *L, const char *path, char *what, int warning, int *ret, int debug) {
  const char *fn;
  int result1, result2;
  result1 = result2 = 0;
  while ((path = luaL_pushnexttemplate(L, path)) != NULL) {  /* pushes a path onto the stack */
    debuginfo2(debug,  "      Searching %s", luaL_checkstring(L, -1));
    debuginfo2(debug,  "%s: ", what);
    lua_pushstring(L, what);
    lua_concat(L, 2);
    fn = luaL_checkstring(L, -1);
    result1 = (access(fn, 00|04) == -1);  /* 00 = exists, 04 = read permissions */
    debuginfo(debug, (result1) ? "failed.\n" : "succeeded.\n");
    result2 = 0;
    if (!result1) {  /* file found and readable */
      debuginfo2(debug, "      Executing %s: ", fn);
      result2 = luaL_dofile(L, fn);  /* execute it; 0 = no errros, 1 = errors;
      a string is pushed on the stack in case of errors */
      debuginfo(debug, (result2) ? "failed.\n" : "succeeded.\n");
    }
    if (result1 || result2) {  /* any error ? */
      if (result2) {  /* print syntax error */
        warning = 1;
        fprintf(stderr, "%s\n", lua_tostring(L, -1));
        agn_poptop(L);  /* drop error message generated by `docall` */
      }
      agn_poptop(L);  /* drop file name */
    } else {
      agn_poptop(L);  /* drop file name */
      break;  /* file successfully read, skip remaining paths */
    }
  }
  if (warning && (result1 || result2)) {  /* any error ? */
    what++;
    fprintf(stderr, LUA_QS " not found or executable, ignoring.\n", what);
    fflush(stderr);
  }
  if (warning == 0)  /* changed Agena 1.0.2 */
    *ret = 0;
  else if (result2 == 1)
    *ret = 2;  /* syntax error in file */
  else
    *ret = result1;  /* file found (0) or file not found (1) */
}


char *getuserhome () {
  char *userhome = NULL;
#ifdef _WIN32
  userhome = getenv("UserProfile");
#elif defined(__unix__) || defined(__APPLE__)
  userhome = getenv("HOME");
#elif defined(__HAIKU__)  /* 0.33.1 */
  userhome = getenv("HOME");
  if (userhome == NULL)
    userhome = "/boot/home";
#elif defined(__OS2__) || defined(LUA_DOS)
  userhome = getenv("HOME");
  if (userhome == NULL)
    userhome = getenv("USER");
#endif
  return userhome;
}

/* IMPORTANT NOTE: Since values popped may be gc'ed, the strings put on the stack will only be popped
   by this procedure when not needed any longer. Do not put an agn_poptop() statement right after the
   str = agn_tostring(L, n) assignment !!! */
LUALIB_API void agnL_initialise (lua_State *L, int skipinis, int debug) {
  const char *path;
  char *userhome;
  int result1, result2, result3, userhomenotnull;
  result1 = result2 = result3 = userhomenotnull = 0;
  debuginfo(debug, "\nCalling agnL_initialise:\n");
  lua_getglobal(L, "libname");
  if (!agn_isstring(L, -1)) {  /* 0.20.2 */
    agn_poptop(L);  /* remove `something` */
    fprintf(stderr, "Error: " LUA_QS
      " is not a string, must be a path to the main Agena library\nfolder.\n", "libname");
    fflush(stderr);
    return;
  }
  path = agn_tostring(L, -1);  /* path must not be dropped immediately because it might be gc'ed */
  debuginfo(debug, "   Getting libname: "); debuginfo2(debug, "%s.\n", path);
  result1 = 0;
  /* load library.agn file */
  debuginfo(debug, "   Trying to read library.agn:\n");
  agnL_initialise_auxloop(L, path, "/library.agn", 1, &result1, debug);
  if (result1 == 1) printwarning(1, path);  /* library.agn not found, extended in 0.31.5 */
  /* run primary agena.ini file */
  if (!skipinis) {
    debuginfo(debug, "   Trying to read global initialisation file:\n");
#if (defined(__unix__) || defined(__APPLE__) || defined(__HAIKU__))
    agnL_initialise_auxloop(L, path, "/.agenainit", 0, &result2, debug);
#else
    agnL_initialise_auxloop(L, path, "/agena.ini", 0, &result2, debug);
#endif
  }
  debuginfo(debug, "   Getting user home: ");
  userhome = getuserhome();  /* 0.31.6 */
  if (userhome != NULL) {  /* 0.31.6, 1.0.4 patch */
    userhome = strdup(userhome);
    userhomenotnull = 1;
    debuginfo2(debug, "%s.\n", userhome);
  }
  if (userhomenotnull && !skipinis) {
    /* run user's agena.file, 0.29.4 */
    debuginfo(debug, "   Trying to read personal initialisation file:\n");
#if defined(__unix__) || defined(__APPLE__) || defined(__HAIKU__)
    agnL_initialise_auxloop(L, userhome, "/.agenainit", 0, &result3, debug);
#else
    agnL_initialise_auxloop(L, userhome, "/agena.ini", 0, &result3, debug);
#endif
  }
  if (result1 || result2 || result3) {  /* error message printed ? */
    fprintf(stderr, "\n");
    fflush(stderr);
  }
  agn_poptop(L);  /* drop "libname" */
  /* assign `environ.homedir` environment variable */
  if (userhomenotnull) {
    debuginfo(debug, "   environ.homedir set to ");
    lua_getglobal(L, "environ");
    if (!lua_istable(L, -1)) {
      agn_poptop(L);  /* pop null */
      luaL_error(L, "Error during initialisation: table `environ` not found.");
    }
    charreplace(userhome, '\\', '/', 1);
    lua_pushstring(L, userhome);
    lua_setfield(L, -2, "homedir");
    agn_poptop(L);  /* pop environ table */
    debuginfo2(debug, "%s.\n\n", userhome);
    xfree(userhome);
  }
  agn_setdebug(L, debug);  /* 1.9.4 */
}

/* end of Agena initialisation */


LUALIB_API void agnL_setLibname (lua_State *L, int warning, int debug) {  /* changed Agena 1.0.2, 1.0.3*/
  char *found, *oldapath, *path, *apath;
  debuginfo(debug, "Calling agnL_setLibname:\n");
  lua_getglobal(L, "libname");
  if (!lua_isnil(L, -1)) {  /* libname already set via command line option ? */
    agn_poptop(L);
    debuginfo(debug, "   libname already set, exiting agnL__setLibname.\n");
    return;
  }
  agn_poptop(L);  /* pop null */
  /* try to assign libname from AGENAPATH variable set in the operating system */
  debuginfo(debug, "   Trying to get AGENAPATH: ");
  path = getenv("AGENAPATH");
  debuginfo(debug,  (path == NULL) ? "unassigned.\n" : "found.\n");
  if (path != NULL) {
    path = strdup(path);  /* Agena 1.0.4 patch */
    debuginfo(debug, "   Setting libname to AGENAPATH: ");
    charreplace(path, '\\', '/', 1);
    lua_pushstring(L, path);  /* 0.28.2 patch */
    lua_setglobal(L, "libname");
    debuginfo2(debug, "%s.\n", path);
    xfree(path);
    return;
  }
  /* try to build path from current working directory */
  debuginfo(debug, "   Trying path to current working directory.\n");
  apath = (char *)malloc((PATH_MAX+1)*sizeof(char));
  /* Agena 1.0.4 */
  if (tools_cwd(apath) != 0)
    luaL_error(L, "memory failure during initialisation, in " LUA_QL("agnL__setLibname") ".");
  found = strstr(apath, "/agena");  /* in UNIX */
  if (found == NULL) {  /* if not found, try DOS path */
    found = strstr(apath, "\\agena");
    if (found != NULL) charreplace(apath, '\\', '/', 1);
    xfree(apath);
  } else {
    debuginfo(debug, "   Setting libname to ");
    oldapath = apath;
    apath = found+6;
    *apath = '\0';
    lua_pushstring(L, oldapath);
    lua_pushstring(L, "/lib");
    lua_concat(L, 2);
    lua_setglobal(L, "libname");
    /* reset pointer to original beginning, otherwise the free command would crash Agena,
       especially on Sparcs, Agena 1.0.3 */
    apath = oldapath;
    debuginfo2(debug, "current path %s/lib.\n", oldapath);
    xfree(apath);
    return;
  }
#if defined(__unix__) || defined(__APPLE__)
  /* set default path if path could not be determined before */
  if (access("/usr/agena/lib", 00|04) != -1) {
    lua_pushstring(L, "/usr/agena/lib");
    lua_setglobal(L, "libname");
    debuginfo(debug, "   libname set to default path /usr/agena/lib.\n");
    return;
  }
#elif defined(__HAIKU__)
  /* set default path if path could not be determined before */
  if (access("/boot/common/share/agena/lib", 00|04) != -1) {
    lua_pushstring(L, "/boot/common/share/agena/lib");
    lua_setglobal(L, "libname");
    debuginfo(debug, "   libname set to default path /boot/common/share/agena/lib.\n");
    return;
  }
#elif defined(_WIN32)
  /* try to find Agena installation in the standard Windows programme subdirectory such as `C:\Program Files`
     and set libname accordingly; 0.24.0 - June 21, 2009 */
  path = getenv("ProgramFiles");
  if (path != NULL) {  /* "ProgramFiles" is set ? */
    path = strdup(path);  /* Agena 1.0.4 patch */
    charreplace(path, '\\', '/', 1);
    lua_pushstring(L, path);
    lua_pushstring(L, "/agena/lib");
    lua_concat(L, 2);  /* string `/c/programmes/agena/lib` is now on stack */
    if (access(agn_tostring(L, -1), 00|04) != -1) {  /* Agena programme folder found and readable */
      debuginfo2(debug, "   libname set to '%s/agena/lib'.\n", agn_tostring(L, -1));
      lua_setglobal(L, "libname");  /* pops string */
      return;
    }
    agn_poptop(L);  /* pop string `/c/programmes/agena/lib` */
    xfree(path);
  }
#endif
  if (warning) {
    printwarning(0, path);
  }
}


/* assumes that i is negative ! */
LUALIB_API void agnL_printnonstruct (lua_State *L, int i) {
  const char *r;
  lua_getglobal(L, "tostring");
  lua_pushvalue(L, i-1);
  lua_call(L, 1, 1);  /* this will also call the __string metamethod if one exists; if tostring is
                         unassigned, lua_call issues an error */
  r = lua_tostring(L, -1);
  if (r != NULL)  /* 1.10.5 patch */
    fprintf(stdout, "%s", r);
  else
    luaL_error(L, "value is not a number or string");
  agn_poptop(L);  /* pops string printed by fprintf */
}


LUALIB_API int agnL_gettablefield (lua_State *L, const char *table, const char *field, const char *procname, int issueerror) {  /* 1.6.4 */
  lua_getglobal(L, table);
  if (!lua_istable(L, -1)) {
    if (issueerror) {
      agn_poptop(L);  /* in case of an error, remove object on the top */
      luaL_error(L, "Error in " LUA_QS ": table " LUA_QS " does not exist.", procname, table);
    }
    return LUA_TNONE-1;  /* = -3 */
  }
  lua_getfield(L, -1, field);
  lua_remove(L, -2);  /* remove table */
  return (lua_type(L, -1));
}


