/*
** $Id: agnhlps.c $
** C Helper routines
** See Copyright notices in this file, and - where not indicated - in agena.h
** initiated Alexander Walz, July 20, 2007
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdarg.h>
#include <limits.h>    /* for PATH_MAX length */
#include <sys/stat.h>
#include <math.h>      /* for pow, floor, fabs, etc. */
#include <locale.h>    /* for localeconv */

#ifdef _WIN32
#include <sys/locking.h>
#include <windows.h>   /* free memory */
#endif

#if defined(__unix__) || defined(__APPLE__)  /* for getch */
#include <termios.h>
#include <unistd.h>
#endif

#if defined(__OS2__)
#include <unistd.h>
#endif

#define agnhlps_c
#define LUA_LIB

#include "llimits.h"  /* for MAX_SIZET */

#include "agena.h"
#include "agncfg.h"   /* read in endianness */
#include "agncmpt.h"  /* for O_BINARY */
#include "agnconf.h"

#define BASE_MOVELEN  16384L     /* number of bytes in index section to be moved in a write or delete operation */
#define BASE_OFFSET     256L     /* first position of an index to a key */
#define MAXNRECORDS     104L     /* position of Maximum Records allowed */
#define COMMENT         129L     /* position of comment */

#define uchar(c) ((unsigned char)(c))

/* swap() and quicksort() are used in ADS package */
LUALIB_API void swap (off64_t v[], int i, int j) {
  int temp;
  temp = v[i];
  v[i] = v[j];
  v[j] = temp;
}


LUALIB_API void tools_quicksort (off64_t v[], int left, int right) {
  int i, last;
  if (left >= right) return;
  swap(v, left, (left+right)/2);
  last = left;
  for (i=left+1; i <= right; i++)
    if (v[i] < v[left]) {
      swap(v, ++last, i);
    }
  swap(v, left, last);
  tools_quicksort(v, left, last-1);
  tools_quicksort(v, last+1, right);
}


#define SWAP(a,b,t) { (t) = (a); (a) = (b); (b) = (t); }

/* left: left border, right: right border of array */
LUALIB_API void tools_dquicksort (lua_Number v[], long int left, long int right) {
  register long int i, last;
  register lua_Number temp;
  if (left >= right) return;
  SWAP(v[left], v[(left+right)/2], temp);
  last = left;
  for (i=left+1; i <= right; i++)
    if (v[i] < v[left]) {
      last++;
      SWAP(v[last], v[i], temp);
    }
  SWAP(v[left], v[last], temp);
  tools_dquicksort(v, left, last-1);
  tools_dquicksort(v, last+1, right);
}


/*

Function: pixel_qsort

Old and supposedly optimised quicksort algorithm.

Function : pixel_qsort()

    In : pixel array, size of the array
    Out : int
    Job : sort out the array of pixels
    Note : optimized implementation.

References PIX_STACK_SIZE, and PIX_SWAP.

Taken from https://github.com/wme7/aero-shock/blob/master/mycpp/test/benchmed.c,
written by Nicolas Devillard <ndevilla@free.fr> August 1998. This code in public domain.

The function _can_ be 15 % faster on older systems than the recursive quicksort implementation for
lists of numbers in random order. Only if the list is already sorted in descending order, pixel sort
is much slower.

See also: http://www.eso.org/projects/dfs/papers/jitter99/node27.html

Modified by Alexander Walz to return a success code: 0 - failure, 1 - success, and to dynamically
increase the internal stack instead of throwing an error if the stack size has become too small.

*/

#define PIX_SWAP(a,b) { double temp=(a);(a)=(b);(b)=temp; }
#define PIX_STACK_SIZE 50


LUALIB_API int pixel_qsort (double *pix_arr, int npix) {  /* changed from void to int by Alexander Walz */
  int i, ir, j, k, l, j_stack;
  int *i_stack;
  double  a;
  ir = npix;
  l = 1;
  j_stack = 0;
  i_stack = malloc(PIX_STACK_SIZE * sizeof(double));
  for (;;) {
    if (ir-l < 7) {
      for (j=l+1; j<=ir; j++) {
        a = pix_arr[j-1];
        for (i=j-1; i>=1; i--) {
          if (pix_arr[i-1] <= a) break;
          pix_arr[i] = pix_arr[i-1];
        }
        pix_arr[i] = a;
      }
      if (j_stack == 0) break;
      ir = i_stack[j_stack-- -1];
      l  = i_stack[j_stack-- -1];
    } else {
      k = (l+ir) >> 1;
      PIX_SWAP(pix_arr[k-1], pix_arr[l])
      if (pix_arr[l] > pix_arr[ir-1]) {
        PIX_SWAP(pix_arr[l], pix_arr[ir-1])
      }
      if (pix_arr[l-1] > pix_arr[ir-1]) {
        PIX_SWAP(pix_arr[l-1], pix_arr[ir-1])
      }
      if (pix_arr[l] > pix_arr[l-1]) {
        PIX_SWAP(pix_arr[l], pix_arr[l-1])
      }
      i = l+1;
      j = ir;
      a = pix_arr[l-1];
      for (;;) {
        do i++; while (pix_arr[i-1] < a);
        do j--; while (pix_arr[j-1] > a);
        if (j < i) break;
        PIX_SWAP(pix_arr[i-1], pix_arr[j-1]);
      }
      pix_arr[l-1] = pix_arr[j-1];
      pix_arr[j-1] = a;
      j_stack += 2;
      if (j_stack > PIX_STACK_SIZE) {
         i_stack = realloc(i_stack, (j_stack+PIX_STACK_SIZE) * sizeof(double));  /* changed by Alexander Walz */
         if (i_stack == NULL) return 0;  /* changed by Alexander Walz */
      }
      if (ir-i+1 >= j-l) {
        i_stack[j_stack-1] = ir;
        i_stack[j_stack-2] = i;
        ir = j-1;
      } else {
        i_stack[j_stack-1] = j-1;
        i_stack[j_stack-2] = l;
        l = i;
      }
    }
  }
  xfree(i_stack);
  return 1;
}


/*---------------------------------------------------------------------------

   Algorithm from N. Wirth's book, implementation by N. Devillard.
   This code in public domain.

   Function :   kth_smallest()
   In       :   array of elements, # of elements in the array, rank k
   Out      :   one element
   Job      :   find the kth smallest element in the array
   [Notice  :   use the median() macro defined below to get the median.]
   Notice   :   works on both sorted as well as unsorted data

                Reference:

                  Author: Wirth, Niklaus
                   Title: Algorithms + data structures = programs
               Publisher: Englewood Cliffs: Prentice-Hall, 1976
    Physical description: 366 p.
                  Series: Prentice-Hall Series in Automatic Computation

 ---------------------------------------------------------------------------*/

/* #define ELEM_SWAP(a,b) { register lua_Number t = (a); (a) = (b); (b) = t; } */

LUALIB_API lua_Number tools_kth_smallest (lua_Number a[], long int n, long int k) {
  register long int i, j, l, m;
  register lua_Number x, temp;
  l = 0; m = n - 1;
  while (l < m) {
    x = a[k];
    i = l;
    j = m;
    do {
      while (a[i] < x) i++;
      while (x < a[j]) j--;
      if (i <= j) {
        SWAP(a[i], a[j], temp);
        i++; j--;
      }
    } while (i <= j);
    if (j < k) l = i;
    if (k < i) m = j;
  }
  return a[k];
}


/* Dr. F.H.Toor's database C functions */

LUALIB_API int my_open (const char *file) {
  int hnd;
  hnd = open(file, O_BINARY | O_RDWR);
  if (hnd < 3)
    return -1;
  else
    return(hnd);
}


/* read-only open */

LUALIB_API int my_roopen (const char *file) {
  int hnd;
  hnd = open(file, O_BINARY | O_RDONLY); /*|_S_IWRITE); */
  if (hnd < 3) {
    return -1;
  } else
    return(hnd);
}


LUALIB_API int my_create (const char *file) {
  int hnd;
  hnd = open(file, O_BINARY | O_RDWR | O_CREAT | O_TRUNC, S_IREAD | S_IWRITE);  /* GCC 4.3.2 needs a 3rd arg */
  if (hnd < 3) {
    /* fprintf(stderr, "create error\n"); */
    return -1;
  } else
    return(hnd);
}


LUALIB_API int my_close (int hnd) {
  return close(hnd);
}


LUALIB_API int my_seek (int hnd, int32_t pos) {
  if (lseek64(hnd, pos, SEEK_SET) == -1) {
    return(-1);
  } else
    return(1);
}


LUALIB_API off_t my_lof (int hnd) {
  off_t size;
  size = lseek64(hnd, 0, SEEK_END);
  if (size == -1)
    fprintf(stderr, "Agena IO subsystem: LOF error\n");
  return(size);
}


LUALIB_API void my_read (int hnd, void *data, size_t size) {
  if (read(hnd, data, size) != (ssize_t)size) {
    fprintf(stderr, "Agena IO subsystem: read error\n");
  }
}


LUALIB_API size_t sec_read (int hnd, void *data, size_t size) {
  return (read(hnd, data, size) == (ssize_t)size);
}


LUALIB_API int32_t sec_readl (int hnd, ssize_t *success) {
  int32_t data;
  *success = (read(hnd, &data, sizeof(int32_t)) > 0);
  if (*success == 0) return -1;
#if BYTE_ORDER != BIG_ENDIAN
  tools_swapint32_t(&data);
#endif
  return(data);
}


LUALIB_API int32_t sec_readul (int hnd, ssize_t *success) {
  int32_t data;
  *success = (read(hnd, &data, sizeof(int32_t)) > 0);
  if (*success == 0 || data < 0) return -1;
#if BYTE_ORDER != BIG_ENDIAN
  tools_swapint32_t(&data);
#endif
  return(data);
}


/* longs are always stored in Big Endian notation */

LUALIB_API int32_t my_readl (int hnd) {
  int32_t data;
  my_read(hnd, &data, sizeof(int32_t));
#if BYTE_ORDER != BIG_ENDIAN
  tools_swapint32_t(&data);
#endif
  return(data);
}


LUALIB_API char my_readc (int hnd) {
  char data;
  my_read(hnd, &data, sizeof(char));
  return(data);
}


LUALIB_API void my_write (int hnd, void *data, size_t size) {
  if (write(hnd, data, size) != (ssize_t)size)
    fprintf(stderr, "Agena IO subsystem: write error\n");
}


/* longs are always stored in Big Endian notation */

LUALIB_API void my_writel (int hnd, int32_t data) {
#if BYTE_ORDER != BIG_ENDIAN
  tools_swapint32_t(&data);
#endif
  my_write(hnd, &data, sizeof(int32_t));
}


LUALIB_API void my_writec (int hnd, char data) {
  my_write(hnd, &data, sizeof(char));
}


/* in UNIX and Windows, if size == 0 then lock entire file */
LUALIB_API int my_lock (int hnd, off64_t start, off64_t size) {
  #ifdef _WIN32
  /* see: http://msdn.microsoft.com/en-us/library/8054ew2f.aspx, topic `_locking` */
  int r;
  off64_t oldpos, cursor;
  if (size == 0) size = (off64_t)pow(2, 63);  /* = lock 2^63 bytes (cannot be more in Windows) */
  /* reset file cursor to `start' */
  oldpos = my_fpos(hnd);
  if (oldpos == -1) return -1;
  cursor = lseek64(hnd, start, SEEK_SET);
  if (cursor == -1) return -1;
  /* lock the file */
  r = _locking(hnd, _LK_NBLCK, size);
  /* set file cursor to original file position */
  cursor = lseek64(hnd, oldpos, SEEK_SET);
  if (cursor == -1 || r != 0) return -1;
  #elif defined(__unix__) || defined(__APPLE__)
  struct flock file;
  file.l_type = F_WRLCK | F_RDLCK;   /* exclusive read and write lock */
  file.l_start = (off_t)start;   /* locking from file position `start` */
  file.l_whence = SEEK_SET;      /* offset is relative to the beginning of the file */
  file.l_len = (size == 0) ? 0L : (off_t)size;  /* lock size bytes (size == 0 query: better sure than sorry) */
  if (fcntl(hnd, F_SETLK, &file) == -1)
    return -1;
  #endif
  return 0;
}


LUALIB_API int my_unlock (int hnd, off64_t start, off64_t size) {
  #ifdef _WIN32
  /* see: http://msdn.microsoft.com/en-us/library/8054ew2f.aspx, topic `_locking` */
  int r;
  off64_t oldpos, cursor;
  if (size == 0) size = (off64_t)pow(2, 63);  /* = lock 2^63 bytes (cannot be more in Windows) */
  /* reset file cursor to `start' */
  oldpos = my_fpos(hnd);
  if (oldpos == -1) return -1;
  cursor = lseek64(hnd, start, SEEK_SET);
  if (cursor == -1) return -1;
  /* lock the file */
  r = _locking(hnd, LK_UNLCK, size);
  /* set file cursor to original file position */
  cursor = lseek64(hnd, oldpos, SEEK_SET);
  if (cursor == -1 || r != 0) return -1;
  #elif defined(__unix__) || defined(__APPLE__)
  struct flock file;
  file.l_type = F_WRLCK | F_RDLCK;
  file.l_whence = SEEK_SET;
  file.l_start = (off_t)start;
  file.l_len = (size == 0) ? 0L : (off_t)size;  /* unlock size bytes (size == 0 query: better sure than sorry) */
  if (fcntl(hnd, F_UNLCK, &file) == -1)
    return -1;
  #endif
  return 0;
}


/* my_move(hnd, low*4L+24L, low*4L+28L, cnt*4L+24L); */
/* fpos: from position, tpos: to position, size */
LUALIB_API void my_move (int hnd, int32_t fpos, int32_t tpos, int32_t size) {
  int32_t segment;
  char buff[BASE_MOVELEN];
  segment = size-fpos;
  if (segment < 1)
    return;
  if (fpos < 0 || tpos < 0)
    fprintf(stderr, "Agena IO subsystem: move error\n");
  if (fpos > tpos) {
    while (segment >= BASE_MOVELEN) {
      my_seek(hnd, fpos);
      my_read(hnd, buff, BASE_MOVELEN);
      my_seek(hnd, tpos);
      my_write(hnd, buff, BASE_MOVELEN);
      fpos+=BASE_MOVELEN;
      tpos+=BASE_MOVELEN;
      segment-=BASE_MOVELEN;
    }
    if (segment > 0) {
      my_seek(hnd, fpos);
      my_read(hnd, buff, segment);
      my_seek(hnd, tpos);
      my_write(hnd, buff, segment);
    }
  }
  if (fpos < tpos) {
    while (segment >= BASE_MOVELEN) {
      my_seek(hnd, fpos+segment-BASE_MOVELEN);
      my_read(hnd, buff, BASE_MOVELEN);
      my_seek(hnd, tpos+segment-BASE_MOVELEN);
      my_write(hnd, buff, BASE_MOVELEN);
      segment-=BASE_MOVELEN;
    }
    if (segment > 0) {
      my_seek(hnd, fpos);
      my_read(hnd, buff, segment);
      my_seek(hnd, tpos);
      my_write(hnd, buff, segment);
    }
  }
}


/* return current file position, written by Alexander Walz on June 28, 2007; extended
   0.32.5 */

LUALIB_API off64_t my_fpos (int hnd) {
  return lseek64(hnd, 0, SEEK_CUR);
}


/* my_expand: core functionality for expanding a database by a) reshifting all records, thus
   b) adding more index entries.
   Assumes that the file is open and locked when envoked.
   Restores the file cursor to the position before it was called.

   written by Alexander Walz on June 28, 2007

   Arguments:
      hnd   - the file handler
      mrc   - maximum number of records currently allowed
      cnt   - current number of actual records
      count - number of records to be added */

LUALIB_API void my_expand (int hnd, int mrc, int cnt, int count) {
  int32_t dsbegin, dsend, offset, j, index, bufsize, cfpos, commentpos, commentlen, newindex;
  int i;
  cfpos = my_fpos(hnd);           /* save the current file position for later restoration */
  my_seek(hnd, COMMENT);
  commentpos = my_readl(hnd);
  commentlen = 0;
  if (commentpos != 0) {
    my_seek(hnd, commentpos);
    commentlen = my_readl(hnd)+4L;
  }
  dsbegin = mrc*4L+BASE_OFFSET;   /* start of current dataset section */
  dsend = my_lof(hnd)-1;          /* end of current dataset section */
  /* write 0L at end of file count times */
  my_seek(hnd, dsend+1);
  for (i=1; i<=count; i++) {
    my_writel(hnd, 0L);          /* write zeros */
  }
  offset = count * 4L;
  char buffer[offset];
  /* beginning with the last data set move data sets to the new file end */
  bufsize = offset;
  j = dsend+1;
  while (j > dsbegin) {
    if ((j-dsbegin) >= offset) {
      bufsize = offset;
      my_seek(hnd, j-bufsize);
      my_read(hnd, buffer, bufsize);
      my_seek(hnd, j);
      my_write(hnd, buffer, bufsize); }
    else {  /* last few bytes need special treatment */
      my_seek(hnd, dsbegin);
      my_read(hnd, buffer, j-dsbegin);
      my_seek(hnd, j+bufsize-(j-dsbegin));
      my_write(hnd, buffer, j-dsbegin);
    }
    j -= bufsize;
  }
  /* update indices */
  for (j=0; j<cnt; j++) {
    my_seek(hnd, j*4L+BASE_OFFSET);
    index = my_readl(hnd);
    my_seek(hnd, j*4L+BASE_OFFSET);
    newindex = (index > commentpos) ? index+offset+commentlen : index+offset;
    my_writel(hnd, newindex);
  }
  /* add new indices to end of index section */
  for (j=0; j<count; j++) {
    my_writel(hnd, 0L);
  }
  /* update maximum number of entries field */
  my_seek(hnd, MAXNRECORDS);
  my_writel(hnd, mrc+count);
  /* update position of comment if present */
  if (commentpos != 0) {
    my_seek(hnd, COMMENT);
    my_writel(hnd, commentpos+offset);
  }
  my_seek(hnd, cfpos);  /* reset cursor to position before the function was called */
}


#if defined(__unix__) || defined(__APPLE__)
static struct termios new_io, old_io;

/* Taken from `C von A bis Z` by Jrgen Wolf, 2nd Ed., pp. 452 */
/* Funktion schaltet das Terminal in den cbreak-Modus:         */
/* Kontrollflag ECHO und ICANON auf 0 setzen                   */
/* Steuerzeichen: Leseoperation liefert 1 Byte VMIN=1 VTIME=1  */
LUALIB_API int cbreak (int fd) {
  /*Sichern unseres Terminals*/
  if((tcgetattr(fd, &old_io)) == -1)
    return -1;
  new_io = old_io;
  /* Wir verndern jetzt die Flags fr den cbreak-Modus */
  new_io.c_lflag &= ~(ECHO | ICANON);
  new_io.c_cc[VMIN] = 1;
  new_io.c_cc[VTIME]= 0;
  /* Jetzt setzen wir den cbreak-Modus */
  if((tcsetattr(fd, TCSAFLUSH, &new_io)) == -1)
    return -1;
  return 1;
}


LUALIB_API int getch (void) {
  int c;
  if(cbreak(STDIN_FILENO) == -1) {
    printf("Agena IO subsystem: error in function cbreak (called by getch).\n");
    exit(EXIT_FAILURE);
  }
  c = getchar();
  /* Alten Terminal-Modus wiederherstellen */
  tcsetattr(STDIN_FILENO, TCSANOW, &old_io);
  return c;
}

#endif

/* Copyright (c) 2008    Johnathon

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.

Source: http://code.google.com/p/linux-kbhit/linux_kbhit.tar.gz

*/

#if defined(__unix__)

LUALIB_API int kbhit (void) {  /* does not work on Mac */
  struct termios oldt, newt;
  int ch;
  if (tcgetattr(STDIN_FILENO, &oldt) == -1) return -1;
  newt = oldt;
  newt.c_cc[VMIN]  = 0;
  newt.c_cc[VTIME] = 1;
  newt.c_lflag &= ~(ICANON | ECHO);
  if (tcsetattr(STDIN_FILENO, TCSANOW, &newt) == -1) return -1;
  ch = getchar();  /* ch == -1 means key not pressed */
  if (tcsetattr(STDIN_FILENO, TCSANOW, &oldt) == -1) return -1;
  return (ch != -1);
}

#endif


#if defined(__APPLE__)
#include <sys/select.h>

/* found on: http://stackoverflow.com/questions/448944/c-non-blocking-keyboard-input, originally written by Alnitak;
   XXX this implementation most of the times still echoes the key being pressed at stdin */

LUALIB_API int kbhit (void) {
  int r, i;
  fd_set fds;
  struct termios newt, oldt;
  struct timeval tv = {0L, 0L};
  if (tcgetattr(STDIN_FILENO, &oldt) == -1) return -1;  /* save current terminal mode */
  newt = oldt;
  /* cfmakeraw(&newt); */  /* see the GNU C manual for the more info on this function */
  newt.c_lflag &= ~(ICANON | ECHO);
  newt.c_cc[VMIN]  = 1;  /* default: 1 */
  newt.c_cc[VTIME] = 0;  /* default: 0 */
  if (tcsetattr(STDIN_FILENO, TCSANOW, &newt) == -1) return -1;  /* change terminal mode */
  FD_ZERO(&fds);
  FD_SET(STDIN_FILENO, &fds);
  r = select(FD_SETSIZE, &fds, NULL, NULL, &tv);  /* listen for signals */
  i = tcsetattr(STDIN_FILENO, TCSANOW, &oldt);  /* reset terminal mode */
  FD_CLR(STDIN_FILENO, &fds);
  return (i == -1) ? -1 : r;  /* -1: an error has occurred, 0: no input, 1: key pressed */
}
#endif


/* stores the current working directory in the already allocated buffer variable buffer,
   and returns 0 on success or -1 otherwise. In case of failure, the function frees buffer,
   so you do not have to free it in the function calling tools_cwd. */
LUALIB_API int tools_cwd (char *buffer) {  /* Agena 1.0.4, Agena 1.5.0, Agena 1.6.0 Valgrind */
  if (buffer == NULL) return -1;
  if (getcwd(buffer, PATH_MAX) == buffer) return 0;
  xfree(buffer);
  return -1;
}

/* Determination of endianness; taken from Jrgen Wolf, `C von A bis Z`, 2. Auflage, Galileo Computing */

LUALIB_API char tools_endian (void) {
  unsigned int Word = 0x22CCDDEE;
  unsigned char *Byte;
  Byte = (unsigned char *) &Word;
  if (Byte[0] == ((Word >> 0) & 0xFF))
    return 0;  /* little endian */
  else if (Byte[0] == ((Word >> 24) & 0xFF))
    return 1;  /* big endian */
  else
    return -1;
}


LUALIB_API double tools_intpow (double x, int n) {
  double r = 1.0;
  if (n < 0) {
    if (x == 0) return AGN_NAN;  /* 0.32.4 patch */
    x = 1.0 / x;
    n = -n;
  }
  do {
    if (n & 1) r *= x;
    n >>= 1;
    x *= x;
  } while (n);
  return r;
}


/* is number finite ? */
LUALIB_API int tools_isfinite (const double a) {
  const double b = a - a;
  int r = (b == b);
  return r;
}


/* is number NaN ? */
LUALIB_API int tools_isnan (const double x) {
  int r = (x != x);
  return r;
}


/* define NaN */
LUALIB_API double tools_nan (void) {
  return (0.0/0.0);
}


/* code taken from http://www.dmh2000.com/cpp/dswap.shtml */

#if BYTE_ORDER != BIG_ENDIAN
LUALIB_API uint64_t tools_swapdouble (double d) {
  uint64_t a;
  unsigned char *dst = (unsigned char *)&a;
  unsigned char *src = (unsigned char *)&d;
  dst[0] = src[7];
  dst[1] = src[6];
  dst[2] = src[5];
  dst[3] = src[4];
  dst[4] = src[3];
  dst[5] = src[2];
  dst[6] = src[1];
  dst[7] = src[0];
  return a;
}

/* unswap using char pointers */
LUALIB_API double tools_unswapdouble (uint64_t a) {
  double d;
  unsigned char *src = (unsigned char *)&a;
  unsigned char *dst = (unsigned char *)&d;
  dst[0] = src[7];
  dst[1] = src[6];
  dst[2] = src[5];
  dst[3] = src[4];
  dst[4] = src[3];
  dst[5] = src[2];
  dst[6] = src[1];
  dst[7] = src[0];
  return d;
}
#else
/* BIG ENDIAN */
LUALIB_API uint64_t tools_swapdouble (double d) {
  uint64_t a;
  unsigned char *dst = (unsigned char *)&a;
  unsigned char *src = (unsigned char *)&d;
  dst[0] = src[0];
  dst[1] = src[1];
  dst[2] = src[2];
  dst[3] = src[3];
  dst[4] = src[4];
  dst[5] = src[5];
  dst[6] = src[6];
  dst[7] = src[7];
  return a;
}

/* unswap using char pointers */
LUALIB_API double tools_unswapdouble (uint64_t a) {
  double d;
  unsigned char *src = (unsigned char *)&a;
  unsigned char *dst = (unsigned char *)&d;
  dst[0] = src[0];
  dst[1] = src[1];
  dst[2] = src[2];
  dst[3] = src[3];
  dst[4] = src[4];
  dst[5] = src[5];
  dst[6] = src[6];
  dst[7] = src[7];
  return d;
}
#endif


#if (ACTUAL_SIZE_OF_C_LONG == 4 && NECESSARY_SIZE_OF_C_LONG == 4) || defined(GCCROBUG)
LUALIB_API void tools_swaplong (long *n) {
  union {
    long d;
    unsigned char b[4];
  } p, q;
  p.d = *n;
  q.b[0] = p.b[3];
  q.b[1] = p.b[2];
  q.b[2] = p.b[1];
  q.b[3] = p.b[0];
  *n = q.d;
}
#elif (ACTUAL_SIZE_OF_C_LONG == 8 && NECESSARY_SIZE_OF_C_LONG == 8)  /* 0.31.7 patch for 64 bits architectures */
LUALIB_API void tools_swaplong (long *n) {
  union {
    long d;
    unsigned char b[8];
  } p, q;
  p.d = *n;
  q.b[0] = p.b[7];
  q.b[1] = p.b[6];
  q.b[2] = p.b[5];
  q.b[3] = p.b[4];
  q.b[4] = p.b[3];
  q.b[5] = p.b[2];
  q.b[6] = p.b[1];
  q.b[7] = p.b[0];
  *n = q.d;
}
#else
LUALIB_API void tools_swaplong (long *n) {
  int i;
  union {
    long d;
    unsigned char b[sizeof(long)];
  } p, q;
  p.d = *n;
  for (i=0; i < sizeof(long); i++) {
    q.b[i] = p.b[sizeof(long)-i-1];
  }
  *n = q.d;
}
#endif


LUALIB_API void tools_swapint32_t (int32_t *n) {
  union {
    int32_t d;
    unsigned char b[4];
  } p, q;
  p.d = *n;
  q.b[0] = p.b[3];
  q.b[1] = p.b[2];
  q.b[2] = p.b[1];
  q.b[3] = p.b[0];
  *n = q.d;
}


/* complex sign */
LUALIB_API double csgn (double a, double b) {
  if (a > 0 || (a == 0 && b > 0))
    return 1.0;
  else if (a < 0 || (a == 0 && b < 0))
    return -1.0;
  else
    return 0.0;
}


LUALIB_API char *charreplace (char *s, char from, char to, int flag) {
  char *olds, *e;
  olds = s;
  while (NULL != (e = strchr(s, from))) {  /* traverses the entire string */
    s = e;
    *s = to;
  }
  s = s + strlen(s) - 1;  /* set pointer to end of string */
  /* delete trailing slash if present, only if str has more than one char, 0.31.4 */
  if (flag && strlen(olds) > 1 && (*s == '/' || *s == '\\')) *s = '\0';
  return olds;
}


/*
 * concat() - allocate memory and safely concatenate strings in portable C
 * (and C++ if you like).
 *
 * This code deals gracefully with potential integer overflows (perhaps when
 * input strings are maliciously long), as well as with input strings changing
 * from under it (perhaps because of misbehavior of another thread).  It does
 * not depend on non-portable functions such as snprintf() and asprintf().
 *
 * Written by Solar Designer <solar at openwall.com> and placed in the
 * public domain.
 *
 * Originally written for and currently maintained as a part of popa3d,
 * a POP3 server:
 *
 *   http://www.openwall.com/popa3d/
 */

/* Concatenates strings and returns a null-terminated string, or NULL if something failed.

   Always pass NULL as the last argument to denote the end of the strings to be concatenated, e.g.
   concat(argv[0], argv[1], argv[2], argv[3], NULL).

   You must FREE the memory allocated by concat in case of success. ONLY assign the result to a variable,
   use it in the VM or API, and then finally FREE it !

   Please note that pushing strings onto the stack and then calling `lua_concat` may be 30 % faster in
   standard situations. */

LUALIB_API char *concat (const char *s1, ...) {
  va_list args;
  const char *s;
  char *p, *result;
  unsigned long l, m, n;
  m = n = strlen(s1);
  va_start(args, s1);
  while ((s = va_arg(args, char *))) {
    l = strlen(s);
    if ((m += l) < l) break;
  }
  va_end(args);
  if (s || m >= INT_MAX) return NULL;
  result = (char *)malloc((m + 1)*sizeof(char));
  if (!result) return NULL;
  memcpy(p = result, s1, n);
  p += n;
  va_start(args, s1);
  while ((s = va_arg(args, char *))) {
    l = strlen(s);
    if ((n += l) < l || n > m) break;
    memcpy(p, s, l);
    p += l;
  }
  va_end(args);
  if (s || m != n || p != result + n) {
    xfree(result);
    return NULL;
  }
  *p = 0;
  return result;
}


/* Taken from: http://c-faq.com/lib/regex.html, maintained by Steve Summit

   Quick little wildcard matcher by Arjan Kenter (Copyright 1995, Arjan Kenter). Processes ? and *
   wildcards and return 0 (false) or 1 (true).

   With this definition, the call match("a*b.c", "aplomb.c") would return 1. */

LUALIB_API int glob (const char *pat, const char *str) {
  switch(*pat) {
    case '\0':  return !*str;
    case '*':   return glob(pat+1, str) ||
                  (*str && glob(pat, str+1));
    case '?':   return *str && glob(pat+1, str+1);
    default:    return *pat == *str && glob(pat+1, str+1);
  }
}


/* Copyright (C) 1991-2002, The Numerical Algorithms Group Ltd.
   All rights reserved.

   Copyright (C) 2007-2011, Gabriel Dos Reis.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:

       - Redistributions of source code must retain the above copyright
         notice, this list of conditions and the following disclaimer.

       - Redistributions in binary form must reproduce the above copyright
         notice, this list of conditions and the following disclaimer in
         the documentation and/or other materials provided with the
         distribution.

       - Neither the name of The Numerical Algorithms Group Ltd. nor the
         names of its contributors may be used to endorse or promote products
         derived from this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
   IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
   TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
   PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
   OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. */


/* based on oa_substr released with the sources of OpenAxiom 1.4.1, see file open-axiom-1.4.1/src/lib/cfuns-c.c;
   modified by Alexander Walz, 1.10.0.

   Returns a null-erminated substring. argument begin must be in the range 0 .. strlen()-1. error will be set to 1
   in case of an index-out-of-range error, to -1 in case of a failed memory allocation, and 0 in case of success.

   FREE the return after use !

   A light edition of this function without the error parameter and index out-of-range checks is not faster, although
   this may sound strangely. */

LUALIB_API char *substr (const char *str, const size_t begin, const size_t end, int *error) {
  char *substring;
  size_t len, s;    /* modified by Alex Walz */
  s = strlen(str);  /* modified by Alex Walz */
  if (str == NULL || s == 0 || s < begin || end >= s || begin > end || begin < 0 || end < 0) {
    *error = 1;     /* index out-of-range; modified by Alex Walz */
    return NULL;
  }
  len = (end - begin) + 2;  /* including terminating \0 */
  substring = (char*)malloc(len * sizeof(char));
  if (substring == NULL) {  /* memory allocation failed ?  Modified by Alex Walz */
    *error = -1;
    return NULL;
  }
  memset(substring, '\0', len);
  memcpy(substring, str + begin, len - 1);
  *error = 0;
  return substring;
}


/* ISO 8859/1 Latin-1 alphabetic and upper and lower case bit vector tables.

   Taken from the entropy utility ENT written by John Walker, January 28th, 2008,
   Fourmilab, http://www.fourmilab.ch.

   This software is in the public domain. Permission to use, copy, modify, and distribute this software
   and its documentation for any purpose and without fee is hereby granted, without any conditions or
   restrictions. This software is provided as is without express or implied warranty. */

unsigned char isoalpha[32] = {
    0,   0,   0,   0,   0,   0,   0,   0,
  127, 255, 255, 224, 127, 255, 255, 224,
    0,   0,   0,   0,   0,   0,   0,   0,
  255, 255, 254, 255, 255, 255, 254, 255
};

unsigned char isoupper[32] = {
    0,   0,   0,   0,   0,   0,   0,   0,
  127, 255, 255, 224,   0,   0,   0,   0,
    0,   0,   0,   0,   0,   0,   0,   0,
  255, 255, 254, 254,   0,   0,   0,   0
};

unsigned char isolower[32] = {
    0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   0, 127, 255, 255, 224,
    0,   0,   0,   0,   0,   0,   0,   0,
    0,   0,   0,   1, 255, 255, 254, 255
};


/* taken from: http://stackoverflow.com/questions/11258019/conversion-from-iso-8859-15-latin9-to-utf-8
   ISO 8859-15 is ISO8859/1 plus the EUR symbol;
   written by Nominal Animal, http://stackoverflow.com/users/1475978/nominal-animal.

   Creates a dynamically allocated copy of string, changing the encoding from ISO-8859-15 to UTF-8. */

LUALIB_API char *latin9_to_utf8 (const char *string) {
  char *result;
  size_t  n = 0;
  if (string) {
    const unsigned char *s = (const unsigned char *)string;
    while (*s) {
      if (*s < 128) {
        s++;
        n += 1;
      } else
      if (*s == 164) {
        s++;
        n += 3;
      } else {
        s++;
        n += 2;
      }
    }
  }
  /* Allocate n+1 (to n+7) bytes for the converted string. */
  result = malloc(sizeof(char)*((n | 7) + 1));
  if (!result) return NULL;
  /* Clear the tail of the string, setting the trailing NUL. */
  memset(result + (n | 7) - 7, 0, 8);
  if (n) {
    const unsigned char *s = (const unsigned char *)string;
    unsigned char *d = (unsigned char *)result;
    while (*s) {
      if (*s < 128) {
        *(d++) = *(s++);
      } else
      if (*s < 192) switch (*s) {
        case 164: *(d++) = 226; *(d++) = 130; *(d++) = 172; s++; break;
        case 166: *(d++) = 197; *(d++) = 160; s++; break;
        case 168: *(d++) = 197; *(d++) = 161; s++; break;
        case 180: *(d++) = 197; *(d++) = 189; s++; break;
        case 184: *(d++) = 197; *(d++) = 190; s++; break;
        case 188: *(d++) = 197; *(d++) = 146; s++; break;
        case 189: *(d++) = 197; *(d++) = 147; s++; break;
        case 190: *(d++) = 197; *(d++) = 184; s++; break;
        default:  *(d++) = 194; *(d++) = *(s++); break;
      } else {
        *(d++) = 195;
        *(d++) = *(s++) - 64;
      }
    }
  }
  /* Done. Remember to free() the resulting string when no longer needed. */
  return result;
}


/* taken from: http://stackoverflow.com/questions/11258019/conversion-from-iso-8859-15-latin9-to-utf-8
   ISO 8859-15 is ISO8859/1 plus the EUR symbol;
   written by Nominal Animal, http://stackoverflow.com/users/1475978/nominal-animal.

   Creates a dynamically allocated copy of string, changing the encoding from UTF-8 to ISO-8859-1/15.
   Unsupported code points are ignored. */

LUALIB_API char *utf8_to_latin9 (const char *string) {
  size_t size = 0;
  size_t used = 0;
  unsigned char *result = NULL;
  if (string) {
    const unsigned char *s = (const unsigned char *)string;
    while (*s) {
      if (used >= size) {
        void *const old = result;
        size = (used | 255) + 257;
        result = realloc(result, size*sizeof(char));
        if (!result) {
          if (old) free(old);
          return NULL;
        }
      }
      if (*s < 128) {
        result[used++] = *(s++);
        continue;
      } else
      if (s[0] == 226 && s[1] == 130 && s[2] == 172) {
        result[used++] = 164;
        s += 3;
        continue;
      } else
      if (s[0] == 194 && s[1] >= 128 && s[1] <= 191) {
        result[used++] = s[1];
        s += 2;
        continue;
      } else
      if (s[0] == 195 && s[1] >= 128 && s[1] <= 191) {
        result[used++] = s[1] + 64;
        s += 2;
        continue;
      } else
      if (s[0] == 197 && s[1] == 160) {
        result[used++] = 166;
        s += 2;
        continue;
      } else
      if (s[0] == 197 && s[1] == 161) {
        result[used++] = 168;
        s += 2;
        continue;
      } else
      if (s[0] == 197 && s[1] == 189) {
        result[used++] = 180;
        s += 2;
        continue;
      } else
      if (s[0] == 197 && s[1] == 190) {
        result[used++] = 184;
        s += 2;
        continue;
      } else
      if (s[0] == 197 && s[1] == 146) {
        result[used++] = 188;
        s += 2;
        continue;
      } else
      if (s[0] == 197 && s[1] == 147) {
        result[used++] = 189;
        s += 2;
        continue;
      } else
      if (s[0] == 197 && s[1] == 184) {
        result[used++] = 190;
        s += 2;
        continue;
      }
      if (s[0] >= 192 && s[0] < 224 &&
        s[1] >= 128 && s[1] < 192) {
        s += 2;
        continue;
      } else
      if (s[0] >= 224 && s[0] < 240 &&
        s[1] >= 128 && s[1] < 192 &&
        s[2] >= 128 && s[2] < 192) {
        s += 3;
        continue;
      } else
      if (s[0] >= 240 && s[0] < 248 &&
        s[1] >= 128 && s[1] < 192 &&
        s[2] >= 128 && s[2] < 192 &&
        s[3] >= 128 && s[3] < 192) {
        s += 4;
        continue;
      } else
      if (s[0] >= 248 && s[0] < 252 &&
        s[1] >= 128 && s[1] < 192 &&
        s[2] >= 128 && s[2] < 192 &&
        s[3] >= 128 && s[3] < 192 &&
        s[4] >= 128 && s[4] < 192) {
        s += 5;
        continue;
      } else
      if (s[0] >= 252 && s[0] < 254 &&
        s[1] >= 128 && s[1] < 192 &&
        s[2] >= 128 && s[2] < 192 &&
        s[3] >= 128 && s[3] < 192 &&
        s[4] >= 128 && s[4] < 192 &&
        s[5] >= 128 && s[5] < 192) {
        s += 6;
        continue;
      }
      s++;
    }
  }
  {
    void *const old = result;
    size = (used | 7) + 1;
    result = realloc(result, size*sizeof(char));
    if (!result) {
      if (old) free(old);
      return NULL;
    }
    memset(result + used, 0, (size - used)*sizeof(char));
  }
  /* Done. Remember to free() the resulting string when no longer needed. */
  return (char *)result;
}


/* detects that the given string is in UTF-8 encoding;
   written by Christoph, http://stackoverflow.com/users/48015/christoph,
   see: http://stackoverflow.com/questions/1031645/how-to-detect-utf-8-in-plain-c */

LUALIB_API int is_utf8 (const char *string) {
  if (!string) return 0;
  const unsigned char *bytes = (const unsigned char *)string;
  while (*bytes) {
    if ( (  /* ASCII
              use bytes[0] <= 0x7F to allow ASCII control characters */
      bytes[0] == 0x09 ||
      bytes[0] == 0x0A ||
      bytes[0] == 0x0D ||
      (0x20 <= bytes[0] && bytes[0] <= 0x7E)
      )
    ) {
      bytes += 1;
      continue;
    }
    if( (  /* non-overlong 2-byte */
      (0xC2 <= bytes[0] && bytes[0] <= 0xDF) &&
      (0x80 <= bytes[1] && bytes[1] <= 0xBF)
    )
  ) {
    bytes += 2;
    continue;
  }
  if ( (  /* excluding overlongs */
        bytes[0] == 0xE0 &&
        (0xA0 <= bytes[1] && bytes[1] <= 0xBF) &&
        (0x80 <= bytes[2] && bytes[2] <= 0xBF)
      ) ||
      (  /* straight 3-byte */
        ((0xE1 <= bytes[0] && bytes[0] <= 0xEC) ||
          bytes[0] == 0xEE ||
          bytes[0] == 0xEF) &&
        (0x80 <= bytes[1] && bytes[1] <= 0xBF) &&
        (0x80 <= bytes[2] && bytes[2] <= 0xBF)
      ) ||
      (  /* excluding surrogates */
        bytes[0] == 0xED &&
        (0x80 <= bytes[1] && bytes[1] <= 0x9F) &&
        (0x80 <= bytes[2] && bytes[2] <= 0xBF)
      )
    ) {
      bytes += 3;
      continue;
    }
    if ( (  /* planes 1-3 */
        bytes[0] == 0xF0 &&
        (0x90 <= bytes[1] && bytes[1] <= 0xBF) &&
        (0x80 <= bytes[2] && bytes[2] <= 0xBF) &&
        (0x80 <= bytes[3] && bytes[3] <= 0xBF)
      ) ||
      (  /* planes 4-15 */
        (0xF1 <= bytes[0] && bytes[0] <= 0xF3) &&
        (0x80 <= bytes[1] && bytes[1] <= 0xBF) &&
        (0x80 <= bytes[2] && bytes[2] <= 0xBF) &&
        (0x80 <= bytes[3] && bytes[3] <= 0xBF)
      ) ||
      (  /* plane 16 */
        bytes[0] == 0xF4 &&
        (0x80 <= bytes[1] && bytes[1] <= 0x8F) &&
        (0x80 <= bytes[2] && bytes[2] <= 0xBF) &&
        (0x80 <= bytes[3] && bytes[3] <= 0xBF)
      )
    ) {
      bytes += 4;
      continue;
    }
    return 0;
  }
  return 1;
}

/* determines the size of an UTF-8 string; written by mpez0, http://stackoverflow.com/users/27898/mpez0 */

LUALIB_API size_t size_utf8 (const char *str) {
  size_t i = 0, j = 0;
  while (str[i]) {
    if ((str[i] & 0xc0) != 0x80) j++;
    i++;
  }
  return j;
}


static int ndays[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

LUALIB_API int tools_checkdatetime (int year, int month, int day, int hour, int minute, int second) {
  if (year < 1900 || year > 2099 || month < 1 || month > 12 || day < 1 ||
    hour < 0 || hour > 23 || minute < 0 || minute > 59 || second < 0 || second > 59) return 0;
  /* now check for February 29 */
  return day <= ndays[month - 1] + (month == 2) * isleapyear(year);
}


LUALIB_API int tools_isnumericstring (const char *s) {  /* 26.08.2012, Agena 1.7.7/1.9.1 */
  struct lconv *cv = localeconv();
  char decpoint = (cv ? cv->decimal_point[0] : '.');
  int flag = 1;
  size_t l = strlen(s);
  if (*s == '\0') return 0;
  for ( ; *s != '\0'; s++) {
    if (uchar(*s) < '0' || uchar(*s) > '9') {
      if (uchar(*s) == decpoint && flag && !(l == 1))
        flag = 0;
      else return 0;
    }
  }
  return 1;
}


/* taken from: http://stackoverflow.com/questions/779875/what-is-the-function-to-replace-string-in-c,
   written by rampion. */
LUALIB_API char *strreplace (const char *original, const char *pattern, const char *replacement) {
  const char *oriptr, *patloc;
  char *retptr;
  size_t const replen = strlen(replacement);
  size_t const patlen = strlen(pattern);
  size_t const orilen = strlen(original);
  size_t patcnt = 0;
  /* find how many times the pattern occurs in the original string */
  for (oriptr=original; (patloc = strstr(oriptr, pattern)); oriptr = patloc + patlen) {
    patcnt++;
  }
  /* allocate memory for the new string */
  size_t const retlen = orilen + patcnt * (replen - patlen);
  char * const returned = (char *)malloc(sizeof(char) * (retlen + 1));
  if (returned != NULL) {
    /* copy the original string, replacing all the instances of the pattern */
    retptr = returned;
    for (oriptr=original; (patloc = strstr(oriptr, pattern)); oriptr = patloc + patlen) {
      size_t const skplen = patloc - oriptr;
      /* copy the section until the occurence of the pattern */
      strncpy(retptr, oriptr, skplen);
      retptr += skplen;
      /* copy the replacement */
      strncpy(retptr, replacement, replen);
      retptr += replen;
    }
    /* copy the rest of the string. */
    strcpy(retptr, oriptr);
  }
  return returned;
}


/* Convert double (or int, etc.) into a string, taken from agnconf.h, safe for doubles

  ! FREE the string after usage ! */

#define MAXNUMBER2STR 32  /* 16 digits, sign, point, and \0 */

LUALIB_API char *my_dtoa (double x) {
  char *buf = malloc(MAXNUMBER2STR * sizeof(char));
  sprintf(buf, "%.14g", x);
  return buf;
}

