/*
** $Id: agena.h,v 1.218 2009/03/01 15:38:27 $
** Agena - A procedural programming language based on Lua 5.1
** http://agena.sourceforge.net
** Lua.org, PUC-Rio, Brazil (http://www.lua.org)
** based on file $Id: lua.h,v 1.218 2006/06/02 15:34:00 roberto Exp $
** See Copyright Notice at the end of this file
*/

#ifndef agena_h
#define agena_h

#include <stdarg.h>
#include <stddef.h>

#ifndef LUA_DOS
  #include <stdint.h>
#endif

#include "agnconf.h"
#include "agncfg.h"
#include "agncmpt.h"

#define AGENA_VERSION     "AGENA 2.1"     /* Added */
#define AGENA_RELEASE     "AGENA >> 2.1.0"  /* Added */
#define AGENA_COPYRIGHT   "Interpreter as of " AGENA_BUILDDATE

#define AGENA_LICENCE     "(C) 2006-2014 http://agena.sourceforge.net.\n"

#define LUA_AUTHORS       "Lua: R. Ierusalimschy, L. H. de Figueiredo & W. Celes; derived Agena: A. Walz"


/* mark for precompiled code (`<esc>Agena') */
#define   LUA_SIGNATURE   "\033Agena"

/* option for multiple returns in `lua_pcall' and `lua_call' */
#define LUA_MULTRET   (-1)


/*
** pseudo-indices
*/
#define LUA_REGISTRYINDEX     (-10000)
#define LUA_ENVIRONINDEX      (-10001)
#define LUA_GLOBALSINDEX      (-10002)
#define lua_upvalueindex(i)   (LUA_GLOBALSINDEX-(i))


/* thread status; 0 is OK */
#define LUA_YIELD       1
#define LUA_ERRRUN      2
#define LUA_ERRSYNTAX   3
#define LUA_ERRMEM      4
#define LUA_ERRERR      5


typedef struct lua_State lua_State;

typedef int (*lua_CFunction) (lua_State *L);


/*
** functions that read/write blocks when loading/dumping Lua chunks
*/
typedef const char * (*lua_Reader) (lua_State *L, void *ud, size_t *sz);

typedef int (*lua_Writer) (lua_State *L, const void* p, size_t sz, void* ud);


/*
** prototype for memory-allocation functions
*/
typedef void * (*lua_Alloc) (void *ud, void *ptr, size_t osize, size_t nsize);


/*
** basic types; grep "(GREP_POINT) types" if you want to change their order or add new ones
*/
#define LUA_TNONE            (-2)

#define LUA_TFAIL            (-1)
#define LUA_TNIL             0
#define LUA_TBOOLEAN         1
#define LUA_TNUMBER          2
#define LUA_TCOMPLEX         3
#define LUA_TSTRING          4
#define LUA_TFUNCTION        5
#define LUA_TUSERDATA        6
#define LUA_TLIGHTUSERDATA   7
#define LUA_TTHREAD          8
#define LUA_TTABLE           9
#define LUA_TSEQ             10
#define LUA_TPAIR            11
#define LUA_TSET             12


/* minimum Lua stack available to a C function */
#define LUA_MINSTACK   20


/*
** generic extra include file
*/
#if defined(LUA_USER_H)
#include LUA_USER_H
#endif


/* type of numbers in Lua */
typedef LUA_NUMBER lua_Number;


/* type for integer functions */
typedef LUA_INTEGER lua_Integer;



/*
** state manipulation
*/
LUA_API lua_State *(lua_newstate) (lua_Alloc f, void *ud);
LUA_API void       (lua_close) (lua_State *L);
LUA_API lua_State *(lua_newthread) (lua_State *L);

LUA_API lua_CFunction (lua_atpanic) (lua_State *L, lua_CFunction panicf);


/*
** basic stack manipulation
*/
LUA_API int   (lua_gettop) (lua_State *L);
LUA_API void  (lua_settop) (lua_State *L, int idx);
LUA_API void  (lua_pushvalue) (lua_State *L, int idx);
LUA_API void  (lua_remove) (lua_State *L, int idx);
LUA_API void  (lua_insert) (lua_State *L, int idx);
LUA_API void  (lua_replace) (lua_State *L, int idx);
LUA_API int   (lua_checkstack) (lua_State *L, int sz);

LUA_API void  (lua_xmove) (lua_State *from, lua_State *to, int n);

LUA_API void  (agn_poptop) (lua_State *L);  /* added 0.5.3 */
LUA_API void  (agn_poptoptwo) (lua_State *L);  /* added 0.10.0 */
LUA_API lua_Number (agn_getinumber) (lua_State *L, int idx, int n);  /* changed 0.5.3 */
LUA_API const char *(agn_getistring) (lua_State *L, int idx, int n);  /* changed 0.5.3 */
LUA_API lua_Number  (agn_checknumber) (lua_State *L, int idx);
LUA_API lua_Integer (agn_checkinteger) (lua_State *L, int idx);
#ifndef PROPCMPLX
LUA_API agn_Complex (agn_checkcomplex) (lua_State *L, int idx);
LUA_API agn_Complex (agn_optcomplex) (lua_State *L, int narg, agn_Complex def);  /* 0.12.0 */
#else
LUA_API lua_Number (agn_complexreal) (lua_State *L, int idx);
LUA_API lua_Number (agn_compleximag) (lua_State *L, int idx);
#endif
/*
** access functions (stack -> C)
*/

LUA_API int             (lua_isnumber) (lua_State *L, int idx);
LUA_API int             (lua_isstring) (lua_State *L, int idx);
LUA_API int             (agn_isnumber) (lua_State *L, int idx);  /* Agena 1.4.3/1.5.0 */
LUA_API int             (agn_isstring) (lua_State *L, int idx);  /* Agena 1.4.3/1.5.0 */
LUA_API int             (lua_iscfunction) (lua_State *L, int idx);
LUA_API int             (lua_isuserdata) (lua_State *L, int idx);
LUA_API int             (lua_type) (lua_State *L, int idx);
LUA_API const char     *(lua_typename) (lua_State *L, int tp);

LUA_API int             (lua_equal) (lua_State *L, int idx1, int idx2);
LUA_API int             (lua_rawequal) (lua_State *L, int idx1, int idx2);
LUA_API int             (lua_lessthan) (lua_State *L, int idx1, int idx2);

LUA_API lua_Number      (lua_tonumber) (lua_State *L, int idx);
LUA_API lua_Number      (agn_tonumber) (lua_State *L, int idx);  /* 0.12.0 */
LUA_API lua_Number      (agn_tonumberx) (lua_State *L, int idx, int *exception);
LUA_API const char     *(agn_tostring) (lua_State *L, int idx);  /* 0.24.0 */
#ifndef PROPCMPLX
LUA_API agn_Complex     (agn_tocomplex) (lua_State *L, int idx);   /* 0.12.0 */
LUA_API agn_Complex     (agn_tocomplexx) (lua_State *L, int idx, int *exception);  /* added 10.08.2009, 0.26.1 */
#else
LUA_API void            (agn_tocomplexx) (lua_State *L,
                            int idx, int *exception, lua_Number *a);  /* added 10.08.2009, 0.26.1 */
#endif
LUA_API lua_Integer     (lua_tointeger) (lua_State *L, int idx);
LUA_API int32_t         (lua_toint32_t) (lua_State *L, int idx);
LUA_API off64_t         (lua_tooff64_t) (lua_State *L, int idx);
LUA_API int             (lua_toboolean) (lua_State *L, int idx);
LUA_API int             (agn_istrue) (lua_State *L, int idx);   /* 0.5.2, June 01, 2007 */
LUA_API int             (agn_isfalse) (lua_State *L, int idx);  /* 0.10.0, April 02, 2008 */
LUA_API int             (agn_isfail) (lua_State *L, int idx);   /* 0.10.0, April 02, 2008 */
LUA_API const char     *(lua_tolstring) (lua_State *L, int idx, size_t *len);
LUA_API size_t          (lua_objlen) (lua_State *L, int idx);
LUA_API lua_CFunction   (lua_tocfunction) (lua_State *L, int idx);
LUA_API void           *(lua_touserdata) (lua_State *L, int idx);
LUA_API lua_State      *(lua_tothread) (lua_State *L, int idx);
LUA_API const void     *(lua_topointer) (lua_State *L, int idx);

LUA_API size_t          (agn_nops) (lua_State *L, int idx);  /* 0.9.1 */

LUA_API void            (agn_tablestate) (lua_State *L, int idx, size_t a[], int mode);  /* 0.12.3 */
LUA_API void            (agn_sstate) (lua_State *L, int idx, size_t a[]); /* 0.10.0 */
LUA_API void            (agn_seqstate) (lua_State *L, int idx, size_t a[]);  /* 0.11.0 */

/*
** push functions (C -> stack)
*/
LUA_API void  (lua_pushnil) (lua_State *L);
LUA_API void  (lua_pushnumber) (lua_State *L, lua_Number n);
LUA_API void  (lua_pushundefined) (lua_State *L);  /* added 0.10.0 */
LUA_API void  (lua_pushinteger) (lua_State *L, lua_Integer n);
LUA_API void  (lua_pushlstring) (lua_State *L, const char *s, size_t l);
LUA_API void  (lua_pushstring) (lua_State *L, const char *s);
LUA_API const char *(lua_pushvfstring) (lua_State *L, const char *fmt,
                                                      va_list argp);
LUA_API const char *(lua_pushfstring) (lua_State *L, const char *fmt, ...);
LUA_API void  (lua_pushcclosure) (lua_State *L, lua_CFunction fn, int n);
LUA_API void  (lua_pushboolean) (lua_State *L, int b);
LUA_API void  (agn_pushboolean) (lua_State *L, int b);  /* 1.8.16 */
LUA_API void  (lua_pushlightuserdata) (lua_State *L, void *p);
LUA_API int   (lua_pushthread) (lua_State *L);

#define lua_pushtrue(L)     lua_pushboolean(L, 1)
#define lua_pushfalse(L)    lua_pushboolean(L, 0)
#define lua_pushfail(L)     agn_pushboolean(L, -1)

/*
** get functions (Lua -> stack)
*/
LUA_API void  (lua_gettable) (lua_State *L, int idx);
LUA_API void  (lua_getfield) (lua_State *L, int idx, const char *k);
LUA_API void  (lua_rawget) (lua_State *L, int idx);
LUA_API void  (lua_rawgeti) (lua_State *L, int idx, int n);
LUA_API void  (lua_createtable) (lua_State *L, int narr, int nrec);
LUA_API void  (agn_createtable) (lua_State *L, int narray, int nrec);  /* 1.7.10 */
LUA_API void *(lua_newuserdata) (lua_State *L, size_t sz);
LUA_API int   (lua_getmetatable) (lua_State *L, int objindex);
LUA_API void  (lua_getfenv) (lua_State *L, int idx);
LUA_API size_t (agn_size) (lua_State *L, int idx);  /* 0.12.3 */

LUA_API void  (agn_createset) (lua_State *L, int nrec);  /* 0.10.0 */
LUA_API void  (lua_srawget) (lua_State *L, int idx);  /* 0.11.2 */
LUA_API size_t (agn_ssize) (lua_State *L, int idx);  /* 0.12.3 */

LUA_API void  (agn_createseq) (lua_State *L, int nrec);  /* 0.11.0 */
/* #ifndef __cplusplus */
#ifndef PROPCMPLX
LUA_API void  (agn_createcomplex) (lua_State *L, agn_Complex c);  /* 0.12.0 */
#else
LUA_API void  (agn_createcomplex) (lua_State *L, lua_Number a, lua_Number b);  /* 0.12.2 */
#endif
/* #endif */
LUA_API void  (lua_seqgeti) (lua_State *L, int idx, size_t n);  /* 0.11.0 */
LUA_API lua_Number (lua_seqgetinumber) (lua_State *L, int idx, int n);  /* 0.12.0 */
LUA_API void  (lua_seqinsert) (lua_State *L, int idx);  /* 0.11.0 */
LUA_API void  (lua_seqseti) (lua_State *L, int idx, int n);  /* 0.11.0 */
LUA_API const char *(agn_getseqlstring) (lua_State *L, int idx, int n, size_t *l);  /* 0.11.0 */
LUA_API size_t   (agn_seqsize) (lua_State *L, int idx);  /* 0.11.0 */
LUA_API void  (lua_seqset) (lua_State *L, int idx);  /* 0.11.2 */
LUA_API void  (lua_seqrawget) (lua_State *L, int idx);  /* 0.11.2 */
LUA_API void  (lua_seqrawget2) (lua_State *L, int idx);  /* 0.31.5 */

LUA_API void  (agn_pairrawset) (lua_State *L, int idx);  /* 0.11.2 */
LUA_API void  (agn_pairrawget) (lua_State *L, int idx);  /* 0.11.2 */


LUA_API void  (agn_createpair) (lua_State *L, int idxleft, int idxright);  /* 0.11.1 */
LUA_API void  (agn_pairgeti) (lua_State *L, int idx, int n);  /* 0.11.1 */

LUA_API void  (agn_setutype) (lua_State *L, int idxobj, int idxtype);  /* 0.11.1 */
LUA_API int   (agn_getutype) (lua_State *L, int idx);  /* 0.11.1 */
LUA_API int   (agn_isutype) (lua_State *L, int idx, const char *str);  /* 1.9.1 */
LUA_API int   (agn_isutypeset) (lua_State *L, int idx);  /* 0.12.2 */
LUA_API int   (agn_issequtype) (lua_State *L, int idx, const char *str);  /* 0.12.1 */
LUA_API int   (agn_istableutype) (lua_State *L, int idx, const char *str);  /* 0.12.1 */
LUA_API int   (agn_issetutype) (lua_State *L, int idx, const char *str);  /* 0.14.0 */

LUA_API void  (agn_complexgetreal) (lua_State *L, int idx);  /* 0.12.0 */
LUA_API void  (agn_complexgetimag) (lua_State *L, int idx);  /* 0.12.0 */
/*
** set functions (stack -> Lua)
*/
LUA_API void  (lua_settable) (lua_State *L, int idx);
LUA_API void  (lua_setfield) (lua_State *L, int idx, const char *k);
LUA_API void  (lua_rawset) (lua_State *L, int idx);
LUA_API void  (lua_rawseti) (lua_State *L, int idx, int n);
LUA_API int   (lua_setmetatable) (lua_State *L, int objindex);
LUA_API int   (lua_setfenv) (lua_State *L, int idx);

LUA_API void  (lua_srawset) (lua_State *L, int idx);  /* 0.11.2 */
LUA_API void  (lua_sdelete) (lua_State *L, int idx);  /* 0.10.0 */

/* remember tables functions, initiated 0.9.2 */
LUA_API void  (agn_creatertable) (lua_State *L, int idx, int writemode);
LUA_API int   (agn_getrtable) (lua_State *L, int objindex);
LUA_API void  (agn_setrtable) (lua_State *L, int find, int kind, int vind);
LUA_API void  (agn_deletertable) (lua_State *L, int objindex);
LUA_API int   (agn_getrtablewritemode) (lua_State *L, int idx);  /* 0.22.0 */

LUA_API int   (agn_getfunctiontype) (lua_State *L, int idx);  /* 0.22.1 */

LUA_API void  (agn_copy) (lua_State *L, int idx);  /* 0.22.1 */

LUA_API void  (agn_toname) (lua_State *L, int idx);  /* 0.90.0 */

LUA_API int   (agn_setreadlibbed) (lua_State *L, const char *name);  /* 0.26.0 */

LUA_API void  (lua_rawsetikey) (lua_State *L, int idx, int n);  /* Added 0.5.3 */
LUA_API void  (lua_rawset2) (lua_State *L, int idx);  /* Added 0.5.3 */
LUALIB_API lua_Number (agn_checknumber) (lua_State *L, int idx);
LUALIB_API const char *(agn_checklstring) (lua_State *L, int idx, size_t *len);
LUALIB_API const char *(agn_checkstring) (lua_State *L, int idx);

/*
** `load' and `call' functions (load and run Lua code)
*/
LUA_API void  (lua_call) (lua_State *L, int nargs, int nresults);
LUA_API int   (lua_pcall) (lua_State *L, int nargs, int nresults, int errfunc);
LUA_API int   (lua_cpcall) (lua_State *L, lua_CFunction func, void *ud);
LUA_API int   (lua_load) (lua_State *L, lua_Reader reader, void *dt,
                                        const char *chunkname);

LUA_API int   (lua_dump) (lua_State *L, lua_Writer writer, void *data);

LUA_API lua_Number (agn_ncall) (lua_State *L, int nargs, int nresults);  /* 0.9.2 */
#ifndef PROPCMPLX
LUA_API agn_Complex (agn_ccall) (lua_State *L, int nargs, int nresults);  /* 0.12.0 */
#else
LUA_API void (agn_ccall) (lua_State *L, int nargs, int nresults, lua_Number *real, lua_Number *imag);  /* 0.27.1 */
#endif
LUA_API void  (agn_setbitwise) (lua_State *L, int value);  /* 0.27.0 */
LUA_API int   (agn_getbitwise) (lua_State *L);  /* 0.27.0 */
LUA_API void  (agn_setemptyline) (lua_State *L, int value);  /* 0.30.4 */
LUA_API int   (agn_getemptyline) (lua_State *L);  /* 0.30.4 */
LUA_API void  (agn_setlibnamereset) (lua_State *L, int value);  /* 0.32.0 */
LUA_API int   (agn_getlibnamereset) (lua_State *L);  /* 0.32.0 */
LUA_API void  (agn_setlongtable) (lua_State *L, int value);  /* 0.32.0 */
LUA_API int   (agn_getlongtable) (lua_State *L);  /* 0.32.0 */
LUA_API void  (agn_setdebug) (lua_State *L, int value);  /* 0.32.2a */
LUA_API int   (agn_getdebug) (lua_State *L);  /* 0.32.2a */
LUA_API void  (agn_setgui) (lua_State *L, int value);  /* 0.33.3 */
LUA_API int   (agn_getgui) (lua_State *L);  /* 0.33.3 */
LUA_API void  (agn_setzeroedcomplex) (lua_State *L, int value);  /* 1.7.6 */
LUA_API int   (agn_getzeroedcomplex) (lua_State *L);  /* 1.7.6 */
LUA_API void  (agn_setpromptnewline) (lua_State *L, int value);  /* 1.7.6 */
LUA_API int   (agn_getpromptnewline) (lua_State *L);  /* 1.7.6 */
LUA_API void  (agn_setdigits) (lua_State *L, ptrdiff_t x);  /* 0.27.0 */
LUA_API lua_Number (agn_getdigits) (lua_State *L);  /* 0.27.0 */

LUA_API lua_Number (agn_getstructuresize) (lua_State *L, int idx);

LUA_API int   (agn_paircheckbooloption) (lua_State *L, const char *procname, int idx, const char *option);  /* 1.8.9 */
LUA_API void  (agn_pairgetnumbers) (lua_State *L, const char *procname, int idx, lua_Number *x, lua_Number *y);  /* internal: 1.8.7/C API: 1.8.9 */

LUA_API void  *agn_malloc (lua_State *L, size_t size, const char *procname, ...);  /* 1.9.1 */
LUA_API void  (agn_free) (lua_State *L, ...);  /* 1.9.1 */
LUA_API void  (agn_arraytoseq) (lua_State *L, lua_Number *a, size_t n);  /* 2.1 RC 1 */

/*
** coroutine functions
*/
LUA_API int   (lua_yield) (lua_State *L, int nresults);
LUA_API int   (lua_resume) (lua_State *L, int narg);
LUA_API int   (lua_status) (lua_State *L);

/*
** garbage-collection function and options
*/

#define LUA_GCSTOP         0
#define LUA_GCRESTART      1
#define LUA_GCCOLLECT      2
#define LUA_GCCOUNT        3
#define LUA_GCCOUNTB       4
#define LUA_GCSTEP         5
#define LUA_GCSETPAUSE     6
#define LUA_GCSETSTEPMUL   7

LUA_API int (lua_gc) (lua_State *L, int what, int data);
LUA_API LUAI_UMEM agn_usedbytes (lua_State *L);

/*
** miscellaneous functions
*/

LUA_API int   (lua_error) (lua_State *L);

LUA_API int   (lua_next) (lua_State *L, int idx);
LUA_API int   (lua_usnext) (lua_State *L, int idx);
LUA_API int   (lua_seqnext) (lua_State *L, int idx);  /* 0.11.2 */
LUA_API int   (lua_strnext) (lua_State *L, int idx);  /* 0.12.0 */
LUA_API int   (agn_fnext) (lua_State *L, int idxtable, int idxfunction, int mode);  /* added 0.9.1 */

LUA_API void  (lua_concat) (lua_State *L, int n);

LUA_API lua_Alloc (lua_getallocf) (lua_State *L, void **ud);
LUA_API void  (lua_setallocf) (lua_State *L, lua_Alloc f, void *ud);

/*
** ===============================================================
** some useful macros
** ===============================================================
*/

#define lua_pop(L,n)      lua_settop(L, -(n)-1)

#define lua_newtable(L)      lua_createtable(L, 0, 0)

#define lua_register(L,n,f) (lua_pushcfunction(L, (f)), lua_setglobal(L, (n)))

#define lua_pushcfunction(L,f)   lua_pushcclosure(L, (f), 0)

#define lua_strlen(L,i)      lua_objlen(L, (i))

#define lua_isfunction(L,n)   (lua_type(L, (n)) == LUA_TFUNCTION)
#define lua_istable(L,n)   (lua_type(L, (n)) == LUA_TTABLE)
#define lua_islightuserdata(L,n)   (lua_type(L, (n)) == LUA_TLIGHTUSERDATA)
#define lua_isnil(L,n)      (lua_type(L, (n)) == LUA_TNIL)
#define lua_isboolean(L,n)   (lua_type(L, (n)) == LUA_TBOOLEAN)
#define lua_isthread(L,n)   (lua_type(L, (n)) == LUA_TTHREAD)
#define lua_isnone(L,n)      (lua_type(L, (n)) == LUA_TNONE)
#define lua_isnoneornil(L, n)   (lua_type(L, (n)) <= 0)
#define lua_isset(L,n)   (lua_type(L, (n)) == LUA_TSET)
#define lua_isseq(L,n)   (lua_type(L, (n)) == LUA_TSEQ)
#define lua_ispair(L,n)   (lua_type(L, (n)) == LUA_TPAIR)
#define lua_iscomplex(L,n)   (lua_type(L, (n)) == LUA_TCOMPLEX)


#define lua_pushliteral(L, s)   \
   lua_pushlstring(L, "" s, (sizeof(s)/sizeof(char))-1)

#define lua_setglobal(L,s)   lua_setfield(L, LUA_GLOBALSINDEX, (s))
#define lua_getglobal(L,s)   lua_getfield(L, LUA_GLOBALSINDEX, (s))

#define lua_tostring(L,i)   lua_tolstring(L, (i), NULL)

/* Agena 1.6.7 and later: macros for former C API functions */

#define lua_rawsetinumber(L, idx, n, num) { \
  lua_pushnumber(L, num); \
  lua_rawseti(L, idx-1, n); }

#define lua_rawsetistring(L,idx,n,str) { \
  lua_pushstring(L, str); \
  lua_rawseti(L, idx-1, n); }

#define lua_rawsetilstring(L,idx,n,str,l) { \
  lua_pushlstring(L, str, l); \
  lua_rawseti(L, idx-1, n); }

#define lua_sinsertnumber(L,idx,n) { \
  lua_pushnumber(L, n); \
  lua_srawset(L, idx-1); }

#define lua_sinsertstring(L,idx,str) { \
  lua_pushstring(L, str); \
  lua_srawset(L, idx-1); }

#define lua_sinsertlstring(L,idx,str,l) { \
  lua_pushlstring(L, str, l); \
  lua_srawset(L, idx-1); }

#define lua_seqsetinumber(L, idx, n, num) { \
  lua_pushnumber(L, num); \
  lua_seqseti(L, idx-1, n); }

#define lua_seqsetistring(L, idx, n, str) { \
  lua_pushstring(L, str); \
  lua_seqseti(L, idx-1, n); }

#define lua_seqrawsetilstring(L, idx, n, str, len) { \
  lua_pushlstring(L, str, len); \
  lua_seqseti(L, idx-1, n); }

#define lua_rawsetstringnumber(L, idx, str, n) { \
  lua_pushstring(L, str); \
  lua_pushnumber(L, n); \
  lua_rawset(L, idx-2); }

#define lua_rawsetstringstring(L, idx, str, text) { \
  lua_pushstring(L, str); \
  lua_pushstring(L, text); \
  lua_rawset(L, idx-2); }

#define lua_rawsetstringboolean(L, idx, str, boolean) { \
  lua_pushstring(L, str); \
  lua_pushboolean(L, boolean); \
  lua_rawset(L, idx-2); }

#define lua_sinsert	lua_srawset
#define lua_seqrawset   lua_seqset
#define lua_seqrawgeti  lua_seqgeti

/*
** compatibility macros and functions
*/

#define lua_open()   luaL_newstate()

#define lua_getregistry(L)   lua_pushvalue(L, LUA_REGISTRYINDEX)

#define lua_Chunkreader      lua_Reader
#define lua_Chunkwriter      lua_Writer

#define ISFLOAT(x)   (TRUNC((x)) != (x))
#define ISINT(x)     (TRUNC((x)) == (x))

/*
** {======================================================================
** Debug API
** =======================================================================
*/


/*
** Event codes
*/
#define LUA_HOOKCALL      0
#define LUA_HOOKRET       1
#define LUA_HOOKLINE      2
#define LUA_HOOKCOUNT     3
#define LUA_HOOKTAILRET   4


/*
** Event masks
*/
#define LUA_MASKCALL    (1 << LUA_HOOKCALL)
#define LUA_MASKRET     (1 << LUA_HOOKRET)
#define LUA_MASKLINE    (1 << LUA_HOOKLINE)
#define LUA_MASKCOUNT   (1 << LUA_HOOKCOUNT)

typedef struct lua_Debug lua_Debug;  /* activation record */


/* Functions to be called by the debuger in specific events */
typedef void (*lua_Hook) (lua_State *L, lua_Debug *ar);


LUA_API int (lua_getstack) (lua_State *L, int level, lua_Debug *ar);
LUA_API int (lua_getinfo) (lua_State *L, const char *what, lua_Debug *ar);
LUA_API const char *(lua_getlocal) (lua_State *L, const lua_Debug *ar, int n);
LUA_API const char *(lua_setlocal) (lua_State *L, const lua_Debug *ar, int n);
LUA_API const char *(lua_getupvalue) (lua_State *L, int funcindex, int n);
LUA_API const char *(lua_setupvalue) (lua_State *L, int funcindex, int n);

LUA_API int (lua_sethook) (lua_State *L, lua_Hook func, int mask, int count);
LUA_API lua_Hook (lua_gethook) (lua_State *L);
LUA_API int (lua_gethookmask) (lua_State *L);
LUA_API int (lua_gethookcount) (lua_State *L);


struct lua_Debug {
  int event;
  const char *name;   /* (n) */
  const char *namewhat;   /* (n) `global', `local', `field', `method' */
  const char *what;   /* (S) `Lua', `C', `main', `tail' */
  const char *source;   /* (S) */
  int currentline;   /* (l) */
  int nups;      /* (u) number of upvalues */
  int linedefined;   /* (S) */
  int lastlinedefined;   /* (S) */
  char short_src[LUA_IDSIZE]; /* (S) */
  /* private part */
  int i_ci;  /* active function */
};

/* }====================================================================== */


/******************************************************************************

* Copyright (C) 1994-2007 Lua.org, PUC-Rio.  All rights reserved.
* Copyright Agena: (C) 2006-2013 Alexander Walz. All rights reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining
* a copy of this software and associated documentation files (the
* "Software"), to deal in the Software without restriction, including
* without limitation the rights to use, copy, modify, merge, publish,
* distribute, sublicense, and/or sell copies of the Software, and to
* permit persons to whom the Software is furnished to do so, subject to
* the following conditions:
*
* The above copyright notice and this permission notice shall be
* included in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
* CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
* TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
* SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
******************************************************************************/

#endif

