/*
** $Id: lua.c,v 1.160 2006/06/02 15:34:00 roberto Exp $
** Lua stand-alone interpreter
** See Copyright Notice in agena.h
*/


#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined(_WIN32)
#include <windows.h>   /* for memory status */
#endif

#if defined(__unix__) || defined (__HAIKU__)
#include <unistd.h>   /* for memory status */
#endif

#ifdef __OS2__
#define INCL_DOS
#include <os2.h>     /* for memory status query */
#endif

#ifdef __APPLE__
#include <mach/mach.h>
#endif

#include <limits.h>    /* for PATH_MAX length */

#define lua_c

#include "agena.h"

#include "agnxlib.h"
#include "agenalib.h"

#ifdef LUA_DOS
#include "agnhlps.h"
#endif


static lua_State *globalL = NULL;

static const char *progname = LUA_PROGNAME;



static void lstop (lua_State *L, lua_Debug *ar) {
  (void)ar;  /* unused arg. */
  lua_sethook(L, NULL, 0, 0);
  luaL_error(L, "Execution interrupted.");  /* 2.0.0 RC 2 */
}


static void laction (int i) {
  signal(i, SIG_DFL); /* if another SIGINT happens before lstop,
                              terminate process (default action) */
  lua_sethook(globalL, lstop, LUA_MASKCALL | LUA_MASKRET | LUA_MASKCOUNT, 1);
}


static void print_usage (void) {
  fprintf(stderr,
  "Usage: %s [options] [script [arguments]].\n\n"
  "Available options are:\n"
  "  -d         print debugging information during startup and within a session\n"
  "  -e \"stat\"  execute statement \"stat\" (double quotes needed)\n"
  "  -h, -?     display this help\n"
  "  -i         enter interactive mode after executing " LUA_QL("script") " or other options\n"
  "  -l         print licence\n"
  "  -m         also print the amount of free RAM\n"
#if !defined(__unix__) && !defined(__APPLE__) && !defined(__HAIKU__)
  "  -n         do not run initialisation file(s) `agena.ini`\n", progname);
#else
  "  -n         do not run initialisation file(s) `.agenainit`\n", progname);
#endif
  fprintf(stderr,
  "  -p path    sets <path> to libname, overriding the standard initialisation\n"
  "             procedure for libname\n"
  "  -r name    readlib library <name> (no quotes needed)\n"
  "  -v         show version and compilation time information\n"
  "  --         stop handling options\n"
  "  -          execute stdin and stop handling options\n"
  );
  fflush(stderr);
}


static void l_message (const char *pname, const char *msg) {
  if (pname) fprintf(stderr, "%s: ", pname);
  fprintf(stderr, "%s\n", msg);
  fflush(stderr);
}


static int report (lua_State *L, int status) {
  if (status && !lua_isnil(L, -1)) {
    const char *msg = lua_tostring(L, -1);
    if (msg == NULL) msg = "(error object is not a string)";
    l_message(progname, msg);
    agn_poptop(L);
  }
  return status;
}


static int traceback (lua_State *L) {
  lua_getfield(L, LUA_GLOBALSINDEX, "debug");
  if (!lua_istable(L, -1)) {
    agn_poptop(L);
    return 1;
  }
  lua_getfield(L, -1, "traceback");
  if (!lua_isfunction(L, -1)) {
    agn_poptoptwo(L);
    return 1;
  }
  lua_pushvalue(L, 1);  /* pass error message */
  lua_pushinteger(L, 2);  /* skip this function and traceback */
  lua_call(L, 2, 1);  /* call debug.traceback */
  return 1;
}


static int docall (lua_State *L, int narg, int clear) {
  int status;
  int base = lua_gettop(L) - narg;  /* function index */
  lua_pushcfunction(L, traceback);  /* push traceback function */
  lua_insert(L, base);  /* put it under chunk and args */
  signal(SIGINT, laction);
  status = lua_pcall(L, narg, (clear ? 0 : LUA_MULTRET), base);
  signal(SIGINT, SIG_DFL);
  lua_remove(L, base);  /* remove traceback function */
  /* force a complete garbage collection in case of errors */
  if (status != 0) lua_gc(L, LUA_GCCOLLECT, 0);
  return status;
}

/* added 0.5.4 */

#define MAX_ITEM   512

/* print an `integer` without decimals with thousand separators; for positive numbers only */

const char *nformat (double dv, char thousands) {
  int i, c;
  unsigned long v;
  char digit, *result, *fmt;
  fmt = (char *)malloc(MAX_ITEM*sizeof(char));
  if (fmt == NULL) return NULL;
  v = (unsigned long) ((dv * 100.0) + .5);
  if (dv < 0) return NULL;
  i = -2; c = 0;
  do {
    if ((i > 0) && (!(i % 3))) {
      fmt[c] = thousands;
      c++;
    }
    digit = (v % 10) + '0';
    if (i > -1) {  /* skip decimals */
      fmt[c] = digit;
      c++;
    }
    v /= 10;
    i++;
  } while((v) || (i < 1));
  fmt[c] = '\0';
  result = (char *)malloc((c+1)*sizeof(char));  /* Agena 1.6.0, Valgrind fix */
  if (result == NULL) return NULL;  /* Agena 1.0.4 */
  i = 0;
  /* the string is `vice versa`, so reverse the string */
  while (i < c) {
    result[i] = fmt[c-i-1];
    i++;
  }
  result[i] = '\0';
  free(fmt);
  fmt = NULL;
  return result;
}


static void print_memorystatus (void) {
#if defined(_WIN32)
  const char *s;
  MEMORYSTATUS memstat;
  GlobalMemoryStatus(&memstat);
  s = nformat(memstat.dwAvailPhys/1024, '\'');
  if (s == NULL) return;
  fprintf(stderr, "%s KBytes of physical RAM free.\n\n", s);
#elif defined(__unix__) && !defined(LUA_DOS) || defined(__HAIKU__)
  const char *s;
  s = nformat(sysconf(_SC_AVPHYS_PAGES)*sysconf(_SC_PAGESIZE)/1024, '\'');
  if (s == NULL) return;
  fprintf(stderr, "%s KBytes of physical RAM free.\n\n", s);
#elif defined(__OS2__)
  const char *s;
  APIRET result;
  ULONG memInfo[3];
  result = DosQuerySysInfo(QSV_TOTPHYSMEM, QSV_TOTAVAILMEM,
    &memInfo[0], sizeof(memInfo));
  if (result == 0) {
    s = nformat((lua_Number)memInfo[2]/1024, '\'');
    if (s == NULL) return;
    fprintf(stderr, "%s KBytes of virtual memory free.\n\n", s);
  }
#elif defined(__APPLE__)
  const char *s;
  vm_size_t pagesize;
  vm_statistics_data_t vminfo;
  mach_port_t system;
  unsigned int c = HOST_VM_INFO_COUNT;
  system = mach_host_self();
  if (host_page_size(mach_host_self(), &pagesize) != KERN_SUCCESS) pagesize = 4096;
  if (host_statistics(system, HOST_VM_INFO, (host_info_t)(&vminfo), &c) == KERN_SUCCESS) {
    s = nformat((uint64_t)(vminfo.free_count - vminfo.speculative_count) * pagesize/1024, '\'');
    if (s == NULL) return;
    fprintf(stderr, "%s KBytes of physical RAM free.\n\n", s);
  }
#endif
}


static void print_version (lua_State *L, int mode) {  /* Agena 1.6.12 */
  const char *pl = "";
  agnL_gettablefield(L, "environ", "libpatchlevel", "<print_version>", 0);
  if (lua_isnumber(L, -1)) {
    lua_pushstring(L, " Update ");
    lua_pushstring(L, lua_tostring(L, -2));
    lua_concat(L, 2);
    pl = agn_checkstring(L, -1);
    lua_remove(L, -2);  /* remove patchlevel number */
  }
  if (mode == 1) {
    l_message(NULL, lua_pushfstring(L, "\n" AGENA_RELEASE "%s. " AGENA_LICENCE, pl));
  }
  else if (mode == 2) {
    l_message(NULL,
      lua_pushfstring(L, "\n" AGENA_RELEASE "%s " AGENA_COPYRIGHT " at " AGENA_BUILDTIME "\n", pl));
  }
  else {
    l_message(NULL,
      lua_pushfstring(L, AGENA_RELEASE "%s " AGENA_COPYRIGHT " at " AGENA_BUILDTIME, pl));
  }
  agn_poptoptwo(L);  /* drop fstring and (patchlevel string or anything else) */
}


static int getargs (lua_State *L, char **argv, int n) {
  int narg;
  int i;
  int argc = 0;
  while (argv[argc]) argc++;  /* count total number of arguments */
  narg = argc - (n + 1);  /* number of arguments to the script */
  luaL_checkstack(L, narg + 3, "too many arguments to script");
  for (i=n+1; i < argc; i++)
    lua_pushstring(L, argv[i]);
  lua_createtable(L, narg, n + 1);
  for (i=0; i < argc; i++) {
    lua_pushstring(L, argv[i]);
    lua_rawseti(L, -2, i - n);
  }
  return narg;
}


static int dofile (lua_State *L, const char *name) {
  int status = luaL_loadfile(L, name) || docall(L, 0, 1);
  return report(L, status);
}


static int dostring (lua_State *L, const char *s, const char *name) {
  int status = luaL_loadbuffer(L, s, strlen(s), name) || docall(L, 0, 1);
  return report(L, status);
}


static int dolibrary (lua_State *L, const char *name) {
  lua_getglobal(L, "readlib");
  lua_pushstring(L, name);
  return report(L, docall(L, 1, 1));  /* 5.1.3 patch */
}


static int setpath (lua_State *L, char *name) {
  lua_pushstring(L, charreplace(name, '\\', '/', 1));  /* Agena 1.6.7 */
  lua_setglobal(L, "libname");
  return 0;
}


/* ------------------------------------------------------------------------ */


static const char *get_prompt (lua_State *L, int firstline) {
  const char *p;
  lua_getfield(L, LUA_GLOBALSINDEX, firstline ? "_PROMPT" : "_PROMPT2");
  p = lua_tostring(L, -1);
  if (p == NULL) p = (firstline ? LUA_PROMPT : LUA_PROMPT2);
  agn_poptop(L);  /* remove global */
  return p;
}


static int incomplete (lua_State *L, int status) {
  if (status == LUA_ERRSYNTAX) {
    size_t lmsg;
    const char *msg = lua_tolstring(L, -1, &lmsg);
    const char *tp = msg + lmsg - (sizeof(LUA_QL("<eof>")) - 1);
    if (strstr(msg, LUA_QL("<eof>")) == tp) {
      agn_poptop(L);
      return 1;
    }
  }
  return 0;  /* else... */
}


static int pushline (lua_State *L, int firstline) {
  char buffer[LUA_MAXINPUT];
  char *b = buffer;
  size_t l, newl;
  char *posstr;
  const char *prmt = get_prompt(L, firstline);
  if (lua_readline(L, b, prmt) == 0)
    return 0;  /* no input */
  l = strlen(b);
  newl = l;
  if (l > 0 && b[l-1] == '\n') newl--;  /* 1.6.0 Valgrind */
  while (newl > 0 && b[newl-1] == ' ') newl--;  /* 1.6.0 Valgrind */
  if (firstline && newl > 0 && b[newl-1] == ':') {  /* 0.26.2 patch */
    b[newl-1] = '\0';  /* remove colon */
    posstr = strstr(b, ":=");
    if (posstr == NULL) {  /* no assignment statement */
      lua_pushfstring(L, "return %s", b);
    } else {  /* assignment statement */
      int i, pos;
      char *bcopy;
      pos = posstr - b + 1;  /* changed 0.5.2; changed 0.12.0; patched 0.12.1 */
      bcopy = (char *)malloc((l+1)*sizeof(char));
      if (bcopy == NULL) luaL_error(L, "memory failure during output.");  /* Agena 1.0.4 */
      for (i=0; i < l; i++) bcopy[i] = b[i];
      bcopy[l] = '\0';  /* better sure than sorry */
      bcopy[pos-1] = '\0';  /* cut off := and rhs */
      lua_pushfstring(L, "%s; return %s", b, bcopy);
      free(bcopy);
      bcopy = NULL;  /* Agena 1.0.4 */
    }
    lua_freeline(L, b);  /* 0.26.2 patch */
    return 1;
  }
  if (l > 0 && b[l-1] == '\n')  /* line ends with newline? */
    b[l-1] = '\0';  /* remove it */
  lua_pushstring(L, b);
  lua_freeline(L, b);
  return 1;
}


static int loadline (lua_State *L) {
  int status;
  lua_settop(L, 0);
  if (!pushline(L, 1))
    return -1;  /* no input */
  for (;;) {  /* repeat until gets a complete line */
    status = luaL_loadbuffer(L, lua_tostring(L, 1), lua_strlen(L, 1), "=stdin");
    if (!incomplete(L, status)) break;  /* cannot try to add lines? */
    if (!pushline(L, 0))  /* no more input? */
      return -1;
    lua_pushliteral(L, "\n");  /* add a new line... */
    lua_insert(L, -2);  /* ...between the two lines */
    lua_concat(L, 3);  /* join them */
  }
  lua_saveline(L, 1);
  lua_remove(L, 1);  /* remove line */
  return status;
}


static void dotty (lua_State *L) {
  int status;
  const char *oldprogname = progname;
  progname = NULL;
  while ((status = loadline(L)) != -1) {
    if (status == 0) status = docall(L, 0, 0);
    report(L, status);
    if (status == 0 && lua_gettop(L) > 0) {  /* any result to print? */
      if (agn_getpromptnewline(L) != 0) {  /* 1.7.6 */
        /* DO NOT use lua_isboolean since it checks a value on the stack and not the return value of agn_gettablefield (a nonneg int) */
        /* a newline is printed at the console after entering a statement */
        fputs("\n", stdout);
      }
      /* save last value printed to `ans` */
      if (!lua_isnil(L, 1)) {  /* Agena 1.6.6, via but not due to Valgrind */
        lua_pushvalue(L, 1);
        lua_setglobal(L, "ans");  /* pops the value from the stack */
      }
      lua_getglobal(L, "print");
      lua_insert(L, 1);
      if (lua_pcall(L, lua_gettop(L)-1, 0, 0) != 0)
        l_message(progname, lua_pushfstring(L,
                            "error calling " LUA_QL("print") " (%s)",
                            lua_tostring(L, -1)));
    }
    /* a newline is printed at the console after entering a statement */
    if (agn_getemptyline(L)) fputs("\n", stdout);
  }
  lua_settop(L, 0);  /* clear stack */
  fputs("\n", stdout);
  fflush(stdout);
  progname = oldprogname;
}


static int handle_script (lua_State *L, char **argv, int n) {
  int status;
  const char *fname;
  int narg = getargs(L, argv, n);  /* collect arguments */
  lua_setglobal(L, "args");
  fname = argv[n];
  if (strcmp(fname, "-") == 0 && strcmp(argv[n-1], "--") != 0)
    fname = NULL;  /* stdin */
  status = luaL_loadfile(L, fname);
  lua_insert(L, -(narg+1));
  if (status == 0)
    status = docall(L, narg, 0);
  else
    lua_pop(L, narg);
  return report(L, status);
}


/* check that argument has no extra characters at the end */
#define notail(x)   {if ((x)[2] != '\0') return -1;}


static int collectargs (lua_State *L, char **argv, int *pi, int *pv, int *pe, int *pnoini, int *pl, int *pb, int *pp, int *pd, int *pm) {
  int i;
  for (i = 1; argv[i] != NULL; i++) {
    if (argv[i][0] != '-')  /* not an option? */
      return i;
    switch (argv[i][1]) {  /* option */
      case '-':
        notail(argv[i]);
        return (argv[i+1] != NULL ? i+1 : 0);
      case '\0':
        return i;
      case 'i':
        notail(argv[i]);
        *pi = 1;  /* go through */
        break;
      case 'n':
        *pnoini = 1;
        break;
      case 'd':
        *pd = 1;
        break;
      case 'm':
        *pm = 1;
        break;
      case 'v':
        notail(argv[i]);
        *pv = 1;
        break;
      case 'l':
        notail(argv[i]);
        *pl = 1;
        break;
      case '?':  /* 1.12.9 */
        return -1;
      case 'e':
        *pe = 1;  /* go through */
      case 'r':
        if (argv[i][2] == '\0') {
          i++;
          if (argv[i] == NULL) return -1;
        }
        break;
      case 'p': {
        char *path;
        if (argv[i][2] == '\0') {
          i++;
          if (argv[i] == NULL) return -1;
        }
        *pp = 1;
        path = argv[i];
        if (*path == '\0') path = argv[++i];
        lua_assert(path != NULL);
        setpath(L, path);
        break;
      }
      default: return -1;  /* invalid option */
    }
  }
  return 0;
}


static int runargs (lua_State *L, char **argv, int n) {
  int i;
  for (i = 1; i < n; i++) {
    if (argv[i] == NULL || argv[i][0] != '-') continue;
    switch (argv[i][1]) {  /* option */
      case 'e': {
        const char *chunk = argv[i] + 2;
        if (*chunk == '\0') chunk = argv[++i];
        lua_assert(chunk != NULL);
        if (dostring(L, chunk, "=(command line)") != 0)
          return 1;
        break;
      }
      case 'r': {
        const char *filename = argv[i] + 2;
        if (*filename == '\0') filename = argv[++i];
        lua_assert(filename != NULL);
        if (dolibrary(L, filename))
          return 1;  /* stop if file fails */
        break;
      }
      default: break;
    }
  }
  return 0;
}


struct Smain {
  int argc;
  char **argv;
  int status;
};

/* 0.22.1: sets a true copy of the structure at stack index idx. Changed Agena 1.0.3 */

static int savestate (lua_State *L, const char *a, const char *b) {
  lua_newtable(L);
  lua_getfield(L, LUA_GLOBALSINDEX, a);  /* get _G on stack */
  lua_pushnil(L);
  while (lua_next(L, -2) != 0) {
    lua_pushvalue(L, -2);  /* key */
    lua_pushvalue(L, -2);  /* value */
    if (lua_istable(L, -1)) {
      lua_newtable(L);
      lua_pushnil(L);
      while (lua_next(L, -3) != 0) {
        lua_pushvalue(L, -2);  /* key */
        lua_pushvalue(L, -2);  /* value */
        lua_settable(L, -5);
        agn_poptop(L);  /* remove value */
      }  /* newtable is at -1 */
      lua_pushvalue(L, -3);  /* push key */
      lua_pushvalue(L, -2);  /* push newtable */
      lua_settable(L, -9);
      lua_pop(L, 4); }  /* pop newtable, value, key value */
    else {
      lua_settable(L, -6);
      agn_poptop(L);
    }
  }
  agn_poptop(L);  /* pop _G */
  lua_setfield(L, LUA_GLOBALSINDEX, b);
  return 0;  /* Agena 1.0.3 */
}


static void print_licence (void) {  /* extended 1.12.9 */
  fprintf(stderr,
    "AGENA Copyright 2006-2013 by Alexander Walz. All rights reserved.\n\n"
    "Portions Copyright 2006 Lua.org, PUC-Rio. All rights reserved.\n\n"
    "Please see the Agena manual (doc/agena.pdf) and the `licence` file for all\n"  /* 1.12.9 */
    "credits and licences.\n\n"
    "--------------------------------------------------------------------------\n\n"
    "The Agena source code is licenced under the terms of the MIT licence\n"  /* 1.12.9 */
    "reproduced below. This means that Agena is free software and can be used\n"
    "for both private, academic, and commercial purposes at absolutely no cost.\n\n"
    "Permission is hereby granted, free of charge, to any person obtaining a\n"
    "copy of this software and associated documentation files (the \"Software\"),\n"
    "to deal in the Software without restriction, including without limitation\n"
    "the rights to use, copy, modify, merge, publish, distribute, sublicence,\n"
    "and/or sell copies of the Software, and to permit persons to whom the\n"
    "Software is furnished to do so, subject to the following conditions:\n\n"
    "The above copyright notices and this permission notice shall be included\n"
    "in all copies or portions of the Software.\n\n"
    "THE SOFTWARE IS PROVIDED \"AS IS\' WITHOUT WARRANTY OF ANY KIND, EXPRESS\n"
    "OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF\n"
    "MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.\n\n"
    "IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY\n"
    "CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,\n"
    "TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE\n"
    "SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.\n\n"
    "--------------------------------------------------------------------------\n\n"
    "The Agena binaries are licenced under the terms of the GPL v2 licence:\n\n"
    "This programme is free software; you can redistribute it and/or modify\n"
    "it under the terms of the GNU General Public Licence as published by\n"
    "the Free Software Foundation; either version 2 of the Licence, or\n"
    "(at your option) any later version.\n\n"
    "This programme is distributed in the hope that it will be useful,\n"
    "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
    "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
    "GNU General Public Licence for more details.\n\n"
    "You should have received a copy of the GNU General Public Licence along\n"
    "with this program; if not, write to the Free Software Foundation, Inc.,\n"
    "51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.\n");
  exit(0);
}


static int pmain (lua_State *L) {  /* extended 1.6.12 */
  struct Smain *s = (struct Smain *)lua_touserdata(L, 1);
  char **argv = s->argv;
  int script, put_version;
  int has_i, has_v, has_e, has_skipinis, has_l, has_b, has_p, has_d, has_m;
  has_i = 0; has_v = 0; has_e = 0; has_skipinis = 0; has_l = 0; has_b = 0; has_p = 0; has_d = 0; has_m = 0;
  globalL = L;
  put_version = 1;
  if (argv[0] && argv[0][0]) progname = argv[0];
  lua_gc(L, LUA_GCSTOP, 0);  /* stop collector during initialisation phase 1 */
  luaL_openlibs(L);  /* open libraries */
  lua_gc(L, LUA_GCRESTART, 0);  /* restart collector. savestate must be called with activated collector ! */
  savestate(L, "_G", "_origG");  /* save all global names assigned so far for restart command */
  script = collectargs(L, argv, &has_i, &has_v, &has_e, &has_skipinis, &has_l, &has_b, &has_p, &has_d, &has_m);
  if (script < 0) {  /* invalid args? */
    print_usage();
    s->status = 1;
    return 0;
  }
  lua_gc(L, LUA_GCSTOP, 0);  /* stop collector again during initialisation phase 2, Agena 1.0.4 */
  if (!has_p) {
    agnL_setLibname(L, 1, has_d);  /* determine value of libname */
  }
  agnL_initialise(L, has_skipinis, has_d);  /* load and run library.agn and optionally initialisation files */
  lua_gc(L, LUA_GCRESTART, 0);  /* restart collector */
  if (has_v) {
    print_version(L, 0);
    put_version = 0;
  }
  if (has_l) {
    print_licence();
    put_version = 0;
  }
  if (script == 0 && !has_e && !has_v && !has_l && lua_stdin_is_tty() && put_version) {
    if (has_b)
      print_version(L, 2);
    else
      print_version(L, 1);
    if (has_m) print_memorystatus();
  }
  /* new start-up sequence 0.25.5, 05.08.2009 */
  s->status = runargs(L, argv, (script > 0) ? script : s->argc);
  if (s->status != 0) return 0;
  if (script)
    s->status = handle_script(L, argv, script);
  if (s->status != 0) return 0;
  if (has_i)
    dotty(L);
  else if (script == 0 && !has_e && !has_v && !has_l) {
    if (lua_stdin_is_tty())
      dotty(L);
    else
      dofile(L, NULL);  /* executes stdin as a file */
  }
  return 0;
}


int main (int argc, char **argv) {
  int status;
  struct Smain s;
  lua_State *L = lua_open();  /* create state */
  if (L == NULL) {
    l_message(argv[0], "Error, cannot create state: not enough memory.");
    return EXIT_FAILURE;
  }
  s.argc = argc;
  s.argv = argv;
  status = lua_cpcall(L, &pmain, &s);
  report(L, status);
  lua_close(L);
  return (status || s.status) ? EXIT_FAILURE : EXIT_SUCCESS;
}

