/* ////////////////////////////////////////////////////////////////////////
//                             DB VERSION 1.0                            //
//                          Author: Dr. F.H.Toor                         //
//                           Dated: 08/07/2003                           //
//                             Licence: FREE                             //
//                 modified and extended by Alexander Walz               //
/////////////////////////////////////////////////////////////////////////*/

/* optimization attempts:
   1) stricmp is 2 % slower than strcmp
   2) integrating the code of the my* procedures into the db* functions speed up the functions.

   This version works correctly also with deleted key-value pairs. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>    /* added */
#include <unistd.h>  /* for ftruncate and close */
#include <fcntl.h>   /* for open */

#include <limits.h>    /* for PATH_MAX length */

#define ads_c
#define LUA_LIB

#include "agena.h"
#include "agnxlib.h"
#include "agenalib.h"

#include "agnhlps.h"

#if !(defined(LUA_DOS) || defined(__OS2__) || defined(LUA_ANSI))
#define AGENA_ADSLIBNAME "ads"
LUALIB_API int (luaopen_ads) (lua_State *L);
#endif


#define STAMP                    0
#define STAMP_TEXT               "AgenaBASE DATA SYSTEM  \0"   /* DB stamp */
#define STAMP_LEN                25              /* length of STAMP */
#define VERSION                  25L
#define VERSION_NUM              300L            /* DB version, 230L used for phonetiQs AGB version ! */
#define DESCRIPTION              29L
#define DESCRIPTION_LEN          75
#define DESCRIPTION_TEXT         "                                                                         \0"
#define MAXNRECORDS              104L
#define MAXNRECORDS_NUM          20000           /* DB Maximum Records allowed */
#define ACTNRECORDS              108L
#define KEYLENGTH                112L
#define KEYLEN_NUM               30              /* DB Maximum Key Length */
#define COLUMNS                  116L
#define COLUMNS_NUM              2               /* Number of columns */
#define TYPE                     120L
#define DATEUPDATE               121L
#define DATECREATION             125L
#define COMMENT                  129L

#define ADS_OFFSET 256L             /* first position of an entry in the index section */
#define ADS_EXPANSION_SIZE 10L      /* number of records to be added in write op when db is full */


static int base_openbase (lua_State *L) {
  off64_t ver;
  int hnd, readwrite, i;
  const char *stamp = STAMP_TEXT;
  const char *file = luaL_checkstring(L, 1);
  /* no option given: open in readwrite mode, option given: open in readonly mode */
  readwrite = (lua_gettop(L) == 1);
  if (access(file, F_OK) == 0) {  /* file already exists ? */
    if (access(file, (readwrite) ? R_OK|W_OK : R_OK) == -1) {
      luaL_error(L, "Error in " LUA_QS ": lacking permissions for " LUA_QS ".", "ads.openbase", file);
    }
  }
  hnd = (readwrite) ? my_open(file) : my_roopen(file);
  if (hnd == -1)
    luaL_error(L, "Error in " LUA_QS ": cannot open file " LUA_QS ".", "ads.openbase", file);
  /* 1.6.7 fix */
  my_seek(hnd, STAMP);
  for (i=0; i < STAMP_LEN; i++) {
    if (my_readc(hnd) != *stamp++) {
      my_close(hnd);
      luaL_error(L, "Error in " LUA_QS ": %s is not an ADS base.", "ads.openbase", file);
    }
  }
  my_seek(hnd, VERSION);
  ver = my_readl(hnd);
  if (ver != VERSION_NUM) {
    my_close(hnd);  /* 1.6.7 */
    luaL_error(L, "Error, invalid base version %ld of " LUA_QS ", need version %ld.",
      ver, file, VERSION_NUM);
  }
  /* enter new open file to global base.openfiles table */
  if (agnL_gettablefield(L, "ads", "openfiles", "ads.openbase", 1) == LUA_TTABLE) {
    /* 0.20.2 & 1.6.4, avoid segmentation faults if openfiles does not exist */
    lua_pushinteger(L, hnd);
    lua_pushstring(L, file);
    lua_rawset(L, -3);
  }
  agn_poptop(L);  /* delete "openfiles" */
  lua_pushinteger(L, hnd);
  return 1;
}


/* usage: base.create(filename::string,
                    max_num_records::long,
                    max_key_legth::long,
                    max_value_length::integer,
                    type::string,
                    description::string[max 24 chars]) */

static int base_createbase (lua_State *L) {
  off64_t i, columns, maxrec, maxkeylen;
  size_t l;
  int hnd, type, next;
  char Desc[DESCRIPTION_LEN];
  const char *file = agn_checkstring(L, 1);  /* 0.13.4 patch */
  maxrec = luaL_optoff64_t(L, 2, MAXNRECORDS_NUM);
  if (maxrec < 1)
    luaL_error(L, "Error in " LUA_QS ": must get at least one record in file " LUA_QS ".", "ads.createbase", file);
  const char *list = luaL_optstring(L, 3, "database");
  if (strcmp(list, "database") != 0 && strcmp(list, "list") != 0 && strcmp(list, "seq") != 0 &&
    strcmp(list, "seq2") != 0) {
    luaL_error(L, "Error in "LUA_QS ": unknown base type `%s` in file " LUA_QS ".", "ads.createbase", list, file);
  }
  if (strcmp(list, "database") == 0) {
    columns = luaL_optoff64_t(L, 4, COLUMNS_NUM);  /* including terminal \0 */
    next = 5; }
  else {
    columns = 1;
    next = 4;
  }
  maxkeylen = luaL_optoff64_t(L, next, KEYLEN_NUM)+1;  /* including terminal \0 */
  if (columns < 1)
    luaL_error(L, "Error in " LUA_QS ": invalid number of columns in " LUA_QS ".", "ads.createbase", file);
  next++;
  const char *desc = luaL_optlstring(L, next, DESCRIPTION_TEXT, &l);
  if (l > DESCRIPTION_LEN-1)
    luaL_error(L, "Error in " LUA_QS ": description too long, must be %d chars.", "ads.createbase", DESCRIPTION_LEN-1);
  strcpy(Desc, desc);
  Desc[l] = '\0';
  time_t seconds = time(NULL);
  hnd = my_create(file);
  my_write(hnd, STAMP_TEXT, STAMP_LEN);      /* stamp */
  my_writel(hnd, VERSION_NUM);               /* database version (4 bytes) */
  my_write(hnd, Desc, DESCRIPTION_LEN);      /* description */
  my_writel(hnd, maxrec);                    /* maximum number of entries allowed in base */
  my_writel(hnd, 0L);                        /* 0L, number of actual entries */
  my_writel(hnd, maxkeylen);                 /* maximum length of keys */
  my_writel(hnd, columns);                   /* number of columns */
  if (strcmp(list, "list") == 0) {           /* sequence (2), list (1), or database (0) */
    my_writec(hnd, 1);
    type = 0;
  }
  else if (strcmp(list, "seq") == 0) {
    my_writec(hnd, 2);
    type = 2;
  }
  else {
    my_writec(hnd, 0);
    type = 0;
  }
  my_writel(hnd, 0L);                        /* date of update */
  my_writel(hnd, (off64_t)seconds);          /* date of creation */
  my_writel(hnd, 0L);                        /* position of comment */
  my_writec(hnd, 1);                         /* endianness of file, always Big Endian, my_writel treats endianness automatically */
  for (i = 0; i < 122; i++)
    my_writec(hnd, 0);                       /* reserved space: 17 fields, each 4 bytes long */
  if (type < 2)
    for (i = 0; i < maxrec; i++) {
      my_writel(hnd, 0L);                    /* write zeros for key positions */
    }
  my_close(hnd);  /* otherwise the file would remain open */
  lua_pushinteger(L, hnd);
  return 1;
}


static int base_closebase (lua_State *L) {
  int nargs, hnd, i, result;
  result = 1;
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments for " LUA_QS "ads.closebase");
  for (i=1; i <= nargs; i++) {
    hnd = agnL_checkinteger(L, i);
    if (hnd < 3) {  /* avoid standard streams to be closed */
      result = 0;
      continue;
    }
    if (!my_close(hnd) == 0)
      result = 0;
    else {
      /* delete file from global base.openfiles table */
      if (agnL_gettablefield(L, "ads", "openfiles", "ads.closebase", 1) == LUA_TTABLE) {
        /* 0.20.2 & 1.6.4, avoid segmentation faults if openfiles does not exist */
        lua_pushinteger(L, hnd);
        lua_pushnil(L);
        lua_rawset(L, -3);
      }
      agn_poptop(L);  /* delete "openfiles" */
    }
  }
  lua_pushboolean(L, result);
  return 1;
}


#define STACKMAX   LUA_MINSTACK+10

static int base_readbase (lua_State *L) {
  off64_t mid, low, high, pos, rcln, keylen, columns;
  int res, try, i;
  char listflag;
  int hnd = agnL_checkinteger(L, 1);
  const char *key = luaL_checkstring(L, 2);
  low = 0;
  try = my_seek(hnd, ACTNRECORDS);
  if (try == -1)  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.readbase", hnd);
  high = my_readl(hnd)-1L;
  if (high < 0) {
    lua_pushnil(L);
    return 1;
  }
  keylen = my_readl(hnd);
  char tkey[keylen];
  columns = my_readl(hnd);
  listflag = my_readc(hnd);
  if (listflag == 2) {  /* read sequence; this is the fastest way known; storing the length for each item
    and reading it during file traversal - thus avoiding checking each char for \0 - is around 30 % slower */
    int i, j, k, oldj;
    char buffer[LUAL_BUFFERSIZE];
    char data[keylen];
    my_seek(hnd, ADS_OFFSET);
    while (1) {
      res = read(hnd, buffer, LUAL_BUFFERSIZE);
      if (res == -1)  /* 1.6.11 */
        luaL_error(L, "Error in " LUA_QS ": could not read base.", "ads.readbase");
      i = res - 1;  /* get number of characters read, adjust to index counting mode (indices start from 0 */
      /* now set pointer to end of the last item that is stored completely in current chunk */
      while (buffer[i] != '\0') i--;
      j = oldj = 0;
      while (j < i) {  /* from the beginning of the chunk read in item by item */
        k = 0;
        while (buffer[j] != '\0') {  /* assign item to data; memcpy is much slower  */
          data[k] = buffer[j];
          j++; k++;
        }
        data[k] = '\0';
        if (strcmp(data, key) == 0) {  /* compare item with key searched */
          lua_pushtrue(L);
          return 1;
        }
        j++;  /* skip '\0' */
      }
      if (res != LUAL_BUFFERSIZE) break;  /* EOF reached */
      lseek(hnd, i-LUAL_BUFFERSIZE+1, SEEK_CUR);
    }  /* if end of buffer is reached, read in new chunk and parse it to the first \0 */
    lua_pushfalse(L);
    return 1;
  }
  while (low <= high) {
    mid = (low+high)/2;
    my_seek(hnd, mid*4L+ADS_OFFSET);
    pos = my_readl(hnd);
    my_seek(hnd, pos);
    my_read(hnd, tkey, my_readl(hnd));
    res = strcmp(key, tkey);
    if (res == 0) {
      if (listflag) {
        lua_pushtrue(L);
        return 1; }
      else {
        if (columns > STACKMAX) {
          if (lua_checkstack(L, columns-STACKMAX) == 0) { /* increase stack size */
            luaL_error(L, "Error in " LUA_QS ": could not increase stack size, too many items: %ld.", "ads.readbase", columns);
          }
        }
        for (i=1; i<columns; i++) {
          rcln = my_readl(hnd);
          char data[rcln];
          my_read(hnd, data, rcln);
          lua_pushstring(L, data);  /* pushlstring is not faster */
        }
        return columns-1;
      }
    }
    if (res > 0)
      low = mid+1;
    else
      high = mid-1;
  }
  if (listflag)
    lua_pushfalse(L);
  else
    lua_pushnil(L);
  return 1;
}


static int base_fastseek (lua_State *L) {
  off64_t mid, low, high, rcln, keylen;
  int res;
  char listflag;
  int hnd = agnL_checkinteger(L, 1);
  const char *key = luaL_checkstring(L, 2);
  luaL_checktype(L, 3, LUA_TTABLE);     /* a table containing all valid indices, see ads.indices */
  low = 0;
  high = agnL_checkinteger(L, 4) - 1L;  /* actual number of records, must be > 0 */
  keylen = agnL_checkinteger(L, 5);     /* standard length of the key */
  listflag = agnL_checkinteger(L, 6);   /* 0 for database, 1 for list, 2 for sequence */
  char tkey[keylen];
  if (listflag > 1) {  /* got sequence */
    luaL_error(L, "Error in " LUA_QS ": function is not applicable to sequences.", "ads.fastseek");
  }
  while (low <= high) {
    mid = (low+high)/2;
    my_seek(hnd, (off64_t)agn_getinumber(L, 3, mid));  /* 2nd arg: get value from table at idx 3 at given index mid */
    my_read(hnd, tkey, my_readl(hnd));
    res = strcmp(key, tkey);
    if (res == 0) {
      if (!listflag) {
        rcln = my_readl(hnd);
        char data[rcln];
        my_read(hnd, data, rcln);
        lua_pushstring(L, data);
      } else
        lua_pushtrue(L);
      return 1;
    }
    if (res > 0)
      low = mid+1;
    else
      high = mid-1;
  }
  if (listflag)
    lua_pushfalse(L);
  else
    lua_pushnil(L);
  return 1;
}


/* searches for the given key (a string) and returns the index as a number */

static int base_index (lua_State *L) {
  off64_t mid, low, high, pos, keylen;
  int res, try;
  char listflag;
  int hnd = agnL_checkinteger(L, 1);
  const char *key = luaL_checkstring(L, 2);
  low = 0;
  try = my_seek(hnd, ACTNRECORDS);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.index", hnd);
  }
  high = my_readl(hnd)-1L;
  if (high < 0) {
    lua_pushnil(L);
    return 1;
  }
  keylen = my_readl(hnd);
  char tkey[keylen];
  my_seek(hnd, TYPE);
  listflag = my_readc(hnd);
  if (listflag == 2) {  /* read sequence */
    int i, j, k;
    char buffer[LUAL_BUFFERSIZE];
    char data[keylen];
    my_seek(hnd, ADS_OFFSET);
    while (1) {
      res = read(hnd, buffer, LUAL_BUFFERSIZE);
      if (res == -1)
        luaL_error(L, "Error in " LUA_QS ": could not read base.", "ads.index");
      i = res - 1;
      while (buffer[i] != '\0') i--;
      j = 0;
      while (j < i) {
        k = 0;
        while (buffer[j] != '\0') {
          data[k] = buffer[j];
          j++; k++;
        }
        data[k] = '\0';
        if (strcmp(data, key) == 0) {
          lua_pushnumber(L, j-k);
          return 1;
        }
        j++;  /* skip '\0' */
      }
      if (res != LUAL_BUFFERSIZE) break;  /* EOF reached */
      lseek(hnd, i-LUAL_BUFFERSIZE+1, SEEK_CUR);
    }
    lua_pushfalse(L);
    return 1;
  }
  while (low <= high) {
    mid = (low+high)/2;
    my_seek(hnd, mid*4L+ADS_OFFSET);
    pos = my_readl(hnd);
    my_seek(hnd, pos);
    my_read(hnd, tkey, my_readl(hnd));
    res = strcmp(key, tkey);
    if (res == 0) {
      lua_pushnumber(L, pos);
      return 1;
    }
    if (res > 0)
      low = mid+1;
    else
      high = mid-1;
  }
  if (listflag)
    lua_pushfalse(L);
  else
    lua_pushnil(L);
  return 1;
}


static int base_iterate (lua_State *L) {
  off64_t mid, low, high, pos, rcln, cnt, keylen, columns;
  size_t l;
  int res, try, i, hnd = agnL_checkinteger(L, 1);
  const char *key1 = luaL_checklstring(L, 2, &l);
  char listflag, key[l+1];
  strcpy(key, key1);
  key[l+1] = '\0';
  low = 0;
  try = my_seek(hnd, ACTNRECORDS);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.iterate", hnd);
  }
  cnt = my_readl(hnd);
  if (cnt < 1) {  /* base is empty ? */
    lua_pushnil(L);
    return(1);
  }
  high = cnt-1;
  keylen = my_readl(hnd);
  char tkey[keylen];
  columns = my_readl(hnd);
  listflag = my_readc(hnd);
  if (listflag > 1)
    luaL_error(L, "Error in " LUA_QS ": function not applicable to sequences.", "ads.iterate");
  if (l == 0) {  /* '' has been passed to get first record */
    my_seek(hnd, ADS_OFFSET);
    pos = my_readl(hnd);
    my_seek(hnd, pos);
    my_read(hnd, tkey, my_readl(hnd));
    lua_pushstring(L, tkey);
    if (listflag) return 1;
    if (columns > STACKMAX) {
      if (lua_checkstack(L, columns-STACKMAX) == 0) {  /* increase stack size */
        luaL_error(L, "Error in " LUA_QS ": could not increase stack size, too many items: %ld.", "ads.iterate", columns) ;
        return 1;
      }
    }
    for (i=1; i<columns; i++) {
      rcln = my_readl(hnd);
      char data[rcln];
      my_read(hnd, data, rcln);
      lua_pushstring(L, data);
    }
    return columns;
  }
  while (low <= high) {
    mid = (low+high)/2;
    my_seek(hnd, mid*4L+ADS_OFFSET);
    pos = my_readl(hnd);
    my_seek(hnd, pos);
    my_read(hnd, tkey, my_readl(hnd));
    res = strcmp(key, tkey);
    if (res == 0) {
      if (mid == cnt-1) {
        lua_pushnil(L);
        return 1;
      }
      mid++;
      my_seek(hnd, mid*4L+ADS_OFFSET);
      pos = my_readl(hnd);
      my_seek(hnd, pos);
      my_read(hnd, tkey, my_readl(hnd));
      if (listflag) {
        lua_pushstring(L, tkey);
        return 1;
      }
      if (columns > STACKMAX) {
        if (lua_checkstack(L, columns-STACKMAX) == 0) {  /* increase stack size */
          luaL_error(L, "Error in " LUA_QS ": could not increase stack size, too many items: %ld.", "ads.iterate", columns) ;
          return 1;
        }
      }
      lua_pushstring(L, tkey);
      for (i=1; i<columns; i++) {
        rcln = my_readl(hnd);
        char data[rcln];
        my_read(hnd, data, rcln);
        lua_pushstring(L, data);
      }
      return columns;
    }
    if (res > 0)
      low = mid+1;
    else
      high = mid-1;
  }
  lua_pushnil(L);
  return 1;
}


static int base_writebase (lua_State *L) {
  off64_t mid, low, high, pos, cnt, mrc, keylen, columns;
  size_t l, k;
  int res, flag, i;
  int hnd = agnL_checkinteger(L, 1);
  const char *key1 = luaL_checklstring(L, 2, &l);
  char key[l+1];
  strcpy(key, key1);
  key[l] = '\0';
  if (my_seek(hnd, 0L) == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.writebase", hnd);
  }
  low = 0;
  my_seek(hnd, MAXNRECORDS);
  mrc = my_readl(hnd);
  cnt = my_readl(hnd);
  high = cnt-1;
  keylen = my_readl(hnd);
  char tkey[keylen];
  columns = my_readl(hnd);
  char listflag = my_readc(hnd);
  if (listflag == 2) {  /* sequence */
    my_seek(hnd, my_lof(hnd));  /* write to end */
    if (l+1 >= keylen) {     /* word plus \0 longer than maxkeylen ? */
      key[keylen-1] = 0;     /* strings count from 0 */
      l = keylen-1;          /* adjust length (i.e. cut string) */
    }
    my_write(hnd, key, l+1); /* l+1: write string plus \0 */
    cnt++;
  } else {  /* base or list */
    flag = 0;
    if (mrc-cnt == 0) {  /* 0.32.0 */
      my_expand(hnd, mrc, cnt, ADS_EXPANSION_SIZE);
      mrc += ADS_EXPANSION_SIZE;
    }
    mid = 0;  /* to avoid compiler warnings */
    while (low <= high) {
      mid = (low+high)/2;
      my_seek(hnd, mid*4L+ADS_OFFSET);   /* seek index */
      pos = my_readl(hnd);               /* read index */
      my_seek(hnd, pos);                 /* set cursor to key position in record area */
      my_read(hnd, tkey, my_readl(hnd)); /* read key */
      res = strcmp(key, tkey);           /* compare found key with search key */
      if (res == 0) {
        cnt--;                           /* decrement number of actual entries */
        flag = 1;
        break;
      }
      if (res > 0)
        low = mid+1;
      else
        high = mid-1;
    }
    pos = my_lof(hnd);
    if (flag && listflag) {  /* list and key already stored ? -> do nothing */
      my_seek(hnd, 0L);
      lua_pushtrue(L);
      return 1;
    }
    if (flag) {  /* does key already exist ? */
      /* prepare storing its new position at existing index position */
      my_seek(hnd, mid*4L+ADS_OFFSET);
    } else {
      if (low < cnt)  /* if new key */
        my_move(hnd, low*4L+ADS_OFFSET, low*4L+ADS_OFFSET+4L, cnt*4L+ADS_OFFSET);  /* move keys upwards */
      my_seek(hnd, low*4L+ADS_OFFSET);
    }
    my_writel(hnd, pos);  /* insert new index position */
    my_seek(hnd, pos);    /* write key (and value) to end of file */
    if (l+1 >= keylen) {  /* word plus \0 longer than maxkeylen ? */
      key[keylen-1] = 0;
      l = keylen-1;       /* adjust length (i.e. cut string) */
    }
    my_writel(hnd, l+1);
    my_write(hnd, key, l+1);
    if (!listflag) {
      /* write values */
      for (i=2; i<=columns; i++) {
        const char *value = luaL_optlstring(L, i+1, "", &k);
        char data[k+1];
        strcpy(data, value);
        data[k] = '\0';
        my_writel(hnd, k+1);
        my_write(hnd, data, k+1);
      }
    }
    cnt++;
  }
  /* set number of entries */
  my_seek(hnd, ACTNRECORDS);
  my_writel(hnd, cnt);
  my_seek(hnd, 0L);
  lua_pushtrue(L);
  return 1;
}


static int base_expand (lua_State *L) {
  off64_t mrc, cnt;
  int hnd, try, count;
  hnd = agnL_checkinteger(L, 1);
  try = my_seek(hnd, 0L);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.expand", hnd);
  }
  count = agnL_optinteger(L, 2, 10);
  my_seek(hnd, MAXNRECORDS);
  mrc = my_readl(hnd);
  cnt = my_readl(hnd);
  my_seek(hnd, TYPE);
  char listflag = my_readc(hnd);
  if (listflag > 1)
    luaL_error(L, "Error in " LUA_QS ": function not applicable to sequences.", "ads.expand");
  /* call expansion procedure;
     mrc: maximum number of records currently allowed,
     cnt: current number of actual records,
     count: number of records to be added */
  my_expand(hnd, mrc, cnt, count);
  /* unlock file */
  my_seek(hnd, 0L);
  lua_pushtrue(L);
  return 1;
}


static int base_remove (lua_State *L) {
  off64_t mid, low, high, pos, cnt, keylen;
  int res, result, try, hnd = agnL_checkinteger(L, 1);
  const char *key = luaL_checkstring(L, 2);
  result = 0;
  try = my_seek(hnd, 0L);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.remove", hnd);
  }
  low = 0;
  my_seek(hnd, ACTNRECORDS);
  cnt = my_readl(hnd);
  high = cnt-1;
  keylen = my_readl(hnd);
  my_seek(hnd, TYPE);
  char listflag = my_readc(hnd);
  if (listflag > 1)
    luaL_error(L, "Error in " LUA_QS ": function not applicable to sequences.", "ads.remove");
  char tkey[keylen];
  while (low <= high) {
    mid = (low+high)/2;
    my_seek(hnd, mid*4L+ADS_OFFSET);
    pos = my_readl(hnd);
    my_seek(hnd, pos);
    my_read(hnd, tkey, my_readl(hnd));
    res = strcmp(key, tkey);
    if (res == 0) {
      if (mid < cnt)
        my_move(hnd, mid*4L+ADS_OFFSET+4L, mid*4L+ADS_OFFSET, cnt*4L+ADS_OFFSET);
      cnt--;
      my_seek(hnd, ACTNRECORDS);
      my_writel(hnd, cnt);
      result = 1;
      break;
    }
    if (res > 0)
      low = mid+1;
    else
      high = mid-1;
  }
  my_seek(hnd, 0L);
  lua_pushboolean(L, result);
  return 1;
}


/* assumes that table is on the top of the stack */

static void setintegerfield (lua_State *L, const char *key, int value) {
  lua_pushinteger(L, value);
  lua_setfield(L, -2, key);
}

/* assumes that table is on the top of the stack */

static void setstringfield (lua_State *L, const char *key, const char *value) {
  lua_pushstring(L, value);
  lua_setfield(L, -2, key);
}


static int base_attrib (lua_State *L) {
  off64_t columns, cnt, mrc, ver, keylen, cpos;
  time_t creationdate;
  char list;
  struct tm *stm;  /* FIXME with agnt64.h */
  int try, hnd = agnL_checkinteger(L, 1);
  char desc[DESCRIPTION_LEN];
  char stamp[STAMP_LEN];
  try = my_seek(hnd, 0L);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.attrib", hnd);
  }
  my_read(hnd, stamp, STAMP_LEN);
  ver = my_readl(hnd);
  my_read(hnd, desc, DESCRIPTION_LEN);
  mrc = my_readl(hnd);
  cnt = my_readl(hnd);
  keylen = my_readl(hnd);
  columns = my_readl(hnd);
  list = my_readc(hnd);
  my_seek(hnd, DATECREATION);
  creationdate = my_readl(hnd);
  stm = localtime(&creationdate);
  cpos = my_readl(hnd);
  lua_newtable(L);
  setstringfield(L, "stamp", stamp);
  setintegerfield(L, "version", ver);
  setintegerfield(L, "maxsize", mrc);
  setintegerfield(L, "size", cnt);
  setintegerfield(L, "keylength", keylen);
  setintegerfield(L, "columns", columns);
  /* setstringfield(L, "desc", desc); */
  setintegerfield(L, "type", list);
  setintegerfield(L, "indexstart", ADS_OFFSET);
  setintegerfield(L, "indexend", mrc*4L+ADS_OFFSET-1);
  setintegerfield(L, "commentpos", cpos);
  char datestring[20];  /* `2007/01/01-01:01:01` = 19 chars */
  if (stm != NULL) {  /* 0.31.3 patch */
    sprintf(datestring, "%d/%02d/%02d-%02d:%02d:%02d",
      stm->tm_year+1900, stm->tm_mon+1, stm->tm_mday, stm->tm_hour, stm->tm_min, stm->tm_sec);
  } else {
    sprintf(datestring, "%d/%02d/%02d-%02d:%02d:%02d", 0, 0, 0, 0, 0, 0);
  }
  setstringfield(L, "creation", datestring);
  my_seek(hnd, 0L);
  return 1;
}


/* return the number of free entries */

static int base_free (lua_State *L) {
  off64_t cnt, mrc;
  int try, hnd = agnL_checkinteger(L, 1);
  try = my_seek(hnd, 0L);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.free", hnd);
  }
  my_seek(hnd, MAXNRECORDS);
  mrc = my_readl(hnd);
  cnt = my_readl(hnd);
  lua_pushinteger(L, mrc-cnt);
  my_seek(hnd, 0L);
  return 1;
}


/* base_getall: get all valid keys and entries of a database, list or sequence and return them in a new set.
   Argument: file handle (integer).

   Moving the cursor to the beginning of the dataset section and then sequentially reading all key-value pairs
   does not work properly since deleted sets are also returned. */

static int base_getall (lua_State *L) {
  off64_t i;
  char listflag;
  int hnd, try, n;  /* 1.12.9 */
  hnd = agnL_checkinteger(L, 1);
  try = my_seek(hnd, COLUMNS);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.getall", hnd);
  }
  my_readl(hnd);  /* skip long, 1.12.9 */
  listflag = my_readc(hnd);
  my_seek(hnd, ACTNRECORDS);
  n = (int)my_readl(hnd);
  if (listflag > 1) {  /* sequence ? */
    int j, k, result;
    char *buffer, *data;
    buffer = (char *)agn_malloc(L, (LUAL_BUFFERSIZE+1)*sizeof(char), "ads.getall", NULL);
    my_seek(hnd, KEYLENGTH);
    int keylen = my_readl(hnd);
    data = (char *)agn_malloc(L, (keylen+1)*sizeof(char), "ads.getall", buffer, NULL);  /* 1.9.1 */
    my_seek(hnd, ADS_OFFSET);
    agn_createset(L, n);
    do {
      result = read(hnd, buffer, LUAL_BUFFERSIZE);
      if (result == -1)  /* 1.6.11 */
        luaL_error(L, "Error in " LUA_QS ": error while reading sequence.", "ads.getall");
      i = result - 1;
      while (buffer[i]) i--;  /* short for (buffer[i] != '\0') */
      j = 0;
      while (j < i) {
        k = 0;
        while (buffer[j]) {   /* short for (buffer[j] != '\0') */
          data[k] = buffer[j];
          j++; k++;
        }
        data[k] = '\0';  /* Agena 1.6.0 Valgrind */
        /* set values to set */
        lua_sinsertlstring(L, -1, data, k);
        j++;  /* skip '\0' */
      }
      lseek(hnd, i-LUAL_BUFFERSIZE+1, SEEK_CUR);
    } while (result == LUAL_BUFFERSIZE);  /* EOF not reached ? */
    agn_free(L, buffer, data, NULL);
  } else {
    luaL_error(L, "Error in " LUA_QS ": cannot process lists or bases.", "ads.getall");
    my_seek(hnd, 0L);
    return 0;
  }
  my_seek(hnd, 0L);
  return 1;
}


static int base_getkeys (lua_State *L) {
  off64_t cnt, keylen, i, j, columns;
  char listflag;
  int hnd = agnL_checkinteger(L, 1);
  int try = my_seek(hnd, ACTNRECORDS);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.getkeys", hnd);
  }
  cnt = my_readl(hnd);
  if (cnt < 1) {
    lua_pushnil(L);
    return 1;
  }
  keylen = my_readl(hnd);
  char tkey[keylen];
  columns = my_readl(hnd);
  listflag = my_readc(hnd);
  if (listflag > 1)
    luaL_error(L, "Error in " LUA_QS ": function not applicable to sequences.", "ads.getkeys");
  my_seek(hnd, ADS_OFFSET);
  off64_t pos[cnt];
  /* read all indices */
  for (i = 0; i < cnt; i++) {
    pos[i] = my_readl(hnd);
  }
  /* read keys from record section */
  lua_newtable(L);
  for (i = 0; i < cnt; i++) {
    my_seek(hnd, pos[i]);
    my_read(hnd, tkey, my_readl(hnd));
    lua_rawsetistring(L, -1, i+1, tkey);
    if (!listflag)
      for (j=1; j<columns; j++)
        lseek(hnd, my_readl(hnd), SEEK_CUR);  /* skip values */
  }
  my_seek(hnd, 0L);
  return 1;
}

/* base_values: get all values from a table; May 12, 2007; changed July 20, 2007 */

static int base_getvalues (lua_State *L) {
  off64_t cnt, rcln, i, columns;
  int hnd = agnL_checkinteger(L, 1);
  int colnum = agnL_optinteger(L, 2, 2);
  int try, j;
  char listflag;
  try = my_seek(hnd, ACTNRECORDS);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L,  "Error in " LUA_QS ": file #%d is not open.", "ads.getvalues", hnd);
  }
  cnt = my_readl(hnd);
  if (cnt < 1) {
    lua_pushnil(L);
    return 1;
  }
  my_seek(hnd, COLUMNS);
  columns = my_readl(hnd);
  listflag = my_readc(hnd);
  if (listflag != 0) {
    lua_pushnil(L);
    return 1;
  }
  if (columns < colnum) {
    luaL_error(L, "Error in " LUA_QS ": column %d does not exist.", "ads.getvalues", colnum);
  }
  my_seek(hnd, ADS_OFFSET);
  off64_t pos[cnt];
  for (i = 0; i < cnt; i++) {
    pos[i] = my_readl(hnd);
  }
  lua_newtable(L);
  for (i = 0; i < cnt; i++) {
    my_seek(hnd, pos[i]);
    for (j = 1; j < colnum; j++)
      lseek(hnd, my_readl(hnd), SEEK_CUR);  /* skip first n columns */
    rcln = my_readl(hnd);  /* read column */
    char data[rcln];
    my_read(hnd, data, rcln);
    lua_rawsetistring(L, -1, i+1, data);
    for (j=colnum+1; j<=columns; j++)
      lseek(hnd, my_readl(hnd), SEEK_CUR);
  }
  my_seek(hnd, 0L);
  return 1;
}


/* base_rawsearch: search a given column for a substring; July 21, 2007 */

static int base_rawsearch (lua_State *L) {
  off64_t cnt, rcln, i, columns, c, keylen;
  int hnd = agnL_checkinteger(L, 1);
  size_t l;
  const char *key = luaL_checklstring(L, 2, &l);
  int colnum = agnL_optinteger(L, 3, 2);
  int try, j;
  char listflag;
  try = my_seek(hnd, ACTNRECORDS);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.rawsearch", hnd);
    lua_pushfail(L);
    return 1;
  }
  cnt = my_readl(hnd);
  if (cnt < 1) {
    lua_pushnil(L);
    return 1;
  }
  my_seek(hnd, COLUMNS);
  columns = my_readl(hnd);
  listflag = my_readc(hnd);
  if (listflag != 0) {
    lua_pushnil(L);
    return 1;
  }
  if (columns < colnum) {
    luaL_error(L, "Error in " LUA_QS ": column %d does not exist.", "ads.rawsearch", colnum);
  }
  my_seek(hnd, ADS_OFFSET);
  /* read entire index section */
  off64_t pos[cnt];
  for (i = 0; i < cnt; i++) {
    pos[i] = my_readl(hnd);
  }
  c = 0;
  lua_newtable(L);
  /* for each key search specified column */
  for (i = 0; i < cnt; i++) {
    my_seek(hnd, pos[i]);
    keylen = my_readl(hnd);
    char indexkey[keylen];
    my_read(hnd, indexkey, keylen);
    if (colnum == 1)
      lseek(hnd, -keylen-4L, SEEK_CUR);  /* reset cursor to key length field */
    for (j = 2; j < colnum; j++)
      lseek(hnd, my_readl(hnd), SEEK_CUR);  /* skip first n columns */
    rcln = my_readl(hnd);  /* read column */
    char data[rcln];
    my_read(hnd, data, rcln);
    if (strstr(data, key) != 0) {
      c++;
      lua_pushstring(L, indexkey);
      lua_pushstring(L, data);
      lua_rawset(L, -3);
    }
    for (j=colnum+1; j<=columns; j++)
      lseek(hnd, my_readl(hnd), SEEK_CUR);
  }
  my_seek(hnd, 0L);
  return 1;
}


/* base_indices: return the file positions of all datasets as a table; May 12, 2007 */

static int base_indices (lua_State *L) {
  off64_t i, cnt;
  int hnd = agnL_checkinteger(L, 1);
  int try = my_seek(hnd, ACTNRECORDS);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.indices", hnd);
  }
  cnt = my_readl(hnd);
  if (cnt < 0) {  /* FIXME: can this even happen ? */
    lua_pushinteger(L, 0);
    return 1;
  }
  my_seek(hnd, TYPE);
  char listflag = my_readc(hnd);
  if (listflag > 1)
    luaL_error(L, "Error in " LUA_QS ": function not applicable to sequences.", "ads.indices");
  my_seek(hnd, ADS_OFFSET);  /* set file handler to beginning of index section */
  lua_newtable(L);
  for (i = 0; i < cnt; i++) {
    lua_rawsetinumber(L, -1, i+1, my_readl(hnd));
  }
  my_seek(hnd, 0L);
  return 1;
}


/* puts a set with all valid and invalid indices on top of the stack */

int base_getallindices (lua_State *L, int hnd, int mrc, int listflag) {
  off64_t dsbegin, dsend, cnt, i, j, k, c, x, z, cpos, columns;
  int flag;
  size_t lenkey, lenval;
  cnt = my_readl(hnd);
  my_seek(hnd, COLUMNS);
  columns = my_readl(hnd);
  my_seek(hnd, COMMENT);
  cpos = my_readl(hnd);
  off64_t indextbl[cnt];
  agn_createset(L, 0);
  c = 0;
  /* read all indices */
  my_seek(hnd, ADS_OFFSET);
  for (i=0; i<cnt; i++) {
    indextbl[i] = my_readl(hnd);
    c++;
    lua_sinsertnumber(L, -1, indextbl[i]);
  }
  /* sort them */
  tools_quicksort(indextbl, 0, cnt-1);
  dsbegin = mrc*4L+ADS_OFFSET;  /* start of current dataset section */
  dsend = my_lof(hnd)-1;        /* end of current dataset section */
  my_seek(hnd, dsbegin);
  z = dsbegin;
  while (z < dsend+1) {
    /* binsearch indextbl for an entry of the current position */
    i = 0; j = cnt-1; flag = 1;
    while (i <= j) {
      x = (i+j)/2;
      if (indextbl[x] == z) {
        flag = 0;
        break;
      }
      if (indextbl[x] < z)
        i = x + 1;
      else
        j = x - 1;
    }
    if (flag && z != cpos) {  /* invalid key found that is not the comment ? */
      c++;
      lua_sinsertnumber(L, -1, z);
    }
    lenkey = my_readl(hnd)+4L;
    /* set cursor to next entry */
    z += lenkey;
    my_seek(hnd, z);
    if (!listflag && z-lenkey != cpos) {
      lenval = 0;
      for (k=2; k<=columns; k++) {
        lenval = my_readl(hnd)+4L;
        z += lenval;
        lseek(hnd, lenval-4L, SEEK_CUR);
      }
    }
  }
  /* move back to beginning of file */
  my_seek(hnd, 0L);
  /* increase stack top */
  return 1;
}


/* base_retrieve: get key and value from a database (file handler, 1st arg) at the given file
   position (integer, 2nd arg). Returns are the respective key and its value. */

static int base_retrieve (lua_State *L) {
  off64_t high, pos, rcln, keylen, columns, mrc;
  int i;
  int hnd = agnL_checkinteger(L, 1);
  pos = agnL_checkinteger(L, 2);
  if (pos < ADS_OFFSET) {  /* otherwise there will be a crash with certain non-index file positions */
    luaL_error(L, "Error in " LUA_QS ": invalid position given for file with handle #%d.", "ads.retrieve", hnd);
  }
  int try = my_seek(hnd, MAXNRECORDS);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.retrieve", hnd);
  }
  mrc = my_readl(hnd);
  high = my_readl(hnd)-1L;
  if (high < 0) {
    lua_pushnil(L);
    return 1;
  }
  keylen = my_readl(hnd);
  char tkey[keylen];
  columns = my_readl(hnd);
  char listflag = my_readc(hnd);
  if (listflag > 1)
    luaL_error(L, "Error in " LUA_QS ": function not applicable to sequences.", "ads.retrieve");
  base_getallindices(L, hnd, mrc, listflag);  /* get all valid and invalid indices as a set */
  lua_pushnumber(L, pos);
  lua_srawget(L, -2);
  if (agn_isfalse(L, -1)) {
    agn_poptoptwo(L);  /* remove false and set */
    luaL_error(L, "Error in " LUA_QS ": invalid position given for file #%d.", "ads.retrieve", hnd);
  }
  agn_poptoptwo(L);  /* remove false and set */
  my_seek(hnd, pos);                      /* set cursor to key */
  my_read(hnd, tkey, my_readl(hnd));      /* read key */
  lua_newtable(L);
  lua_pushstring(L, tkey);
  lua_rawseti(L, -2, 1);
  if (!listflag) {
    for (i=1; i<columns; i++) {
      rcln = my_readl(hnd);
      char data[rcln];
      my_read(hnd, data, rcln);
      lua_pushstring(L, data);
      lua_rawseti(L, -2, i+1);
    }
    my_seek(hnd, 0L);
    return 1;
  }
  my_seek(hnd, 0L);
  return 1;
}


/* base.peek: get length and entry at given position */

static int base_peek (lua_State *L) {
  off64_t mrc, pos, keylen, len, try;
  int hnd;
  hnd = agnL_checkinteger(L, 1);
  pos = luaL_checkoff64_t(L, 2);
  try = my_seek(hnd, MAXNRECORDS);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L,  "Error in " LUA_QS ": file #%d is not open.", "ads.peek", hnd);
  }
  mrc = my_readl(hnd);
  /* peeking of the header or the index section is not allowed */
  if (pos < mrc*4L+ADS_OFFSET || pos >= my_lof(hnd)) {
    luaL_error(L, "Error in " LUA_QS ": invalid position given for file #%d.", "ads.peek", hnd);
  }
  my_seek(hnd, KEYLENGTH);
  keylen = my_readl(hnd);
  char tkey[keylen];
  my_seek(hnd, TYPE);
  char listflag = my_readc(hnd);
  if (listflag > 1)
    luaL_error(L, "Error in " LUA_QS ": function not applicable to sequences.", "ads.peek");
  /* check whether pos is a valid index position */
  base_getallindices(L, hnd, mrc, listflag);  /* get all valid and invalid indices as a set */
  lua_pushnumber(L, pos);
  lua_srawget(L, -2);
  if (agn_isfalse(L, -1)) {
    agn_poptoptwo(L);  /* remove false and set */
    luaL_error(L, "Error in " LUA_QS ": invalid position given for file #%d.", "ads.peek", hnd);
  }
  agn_poptoptwo(L);  /* remove false and set */
  my_seek(hnd, pos);           /* set cursor to key */
  len = my_readl(hnd);
  my_read(hnd, tkey, len);     /* read key */
  lua_pushnumber(L, len);
  lua_pushstring(L, tkey);
  my_seek(hnd, 0L);
  return 2;
}


/* base_size: number of valid entries; July 06, 2007 */

static int base_sizeof (lua_State *L) {
  int hnd = agnL_checkinteger(L, 1);
  off64_t pos = my_seek(hnd, ACTNRECORDS);
  if (pos == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.sizeof", hnd);
  }
  lua_pushnumber(L, my_readl(hnd));
  my_seek(hnd, 0L);
  return 1;
}


/* set desciption field in database; May 12, 2007; extended July 21, 2007 */

static int base_desc (lua_State *L) {
  int try;
  size_t l;
  int hnd = agnL_checkinteger(L, 1);
  char Desc[DESCRIPTION_LEN];
  try = my_seek(hnd, DESCRIPTION);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.desc", hnd);
  }
  if (lua_gettop(L) == 1) {
    my_read(hnd, Desc, DESCRIPTION_LEN);
    lua_pushstring(L, Desc);
    return 1;
  }
  const char *desc = luaL_checklstring(L, 2, &l);
  if (l > DESCRIPTION_LEN-1) {
    luaL_error(L, "Error in " LUA_QS ": description too long; must be %d chars.", "ads.desc", DESCRIPTION_LEN-1);
  }
  strcpy(Desc, desc);
  Desc[l] = '\0';
  my_write(hnd, Desc, l);
  lua_pushtrue(L);
  my_seek(hnd, 0L);
  return 1;
}


static int base_comment (lua_State *L) {
  off64_t pos, cpos, length, clen, cur, cnt, i, ind;
  size_t l, len;
  int try, hnd;
  hnd = agnL_checkinteger(L, 1);
  try = my_seek(hnd, 0L);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.comment", hnd);
  }
  char listflag = my_readc(hnd);
  if (listflag == 2)
    luaL_error(L, "Error in " LUA_QS ": function not applicable to sequences.", "ads.comment");
  my_seek(hnd, COMMENT);  
  cpos = my_readl(hnd);
  if (lua_gettop(L) == 1) {
    if (cpos == 0) {
      lua_pushnil(L);
    } else {
      my_seek(hnd, cpos);
      len = my_readl(hnd);
      char comment[len];
      my_read(hnd, comment, len);
      lua_pushstring(L, comment);
    }
    return 1;
  }
  my_seek(hnd, 0L);
  my_seek(hnd, TYPE);
  const char *comment = luaL_checklstring(L, 2, &l);
  char value[l+1], buffer[LUAL_BUFFERSIZE];
  strcpy(value, comment);
  value[l] = '\0';
  if (cpos != 0) {
    /* delete comment and shift all records to the left*/
    length = my_lof(hnd);
    my_seek(hnd, cpos);                      /* read it */
    clen = my_readl(hnd)+4L;                 /* length of comment including length info */
    cur = cpos+clen;
    while (length-cur+1 >= LUAL_BUFFERSIZE) {
      my_seek(hnd, cur);                     /* set cursor to record following invalid one */
      my_read(hnd, buffer, LUAL_BUFFERSIZE);      /* read bufsize chars */
      my_seek(hnd, cur-clen);                /* set cursor to begin of invalid record */
      my_write(hnd, buffer, LUAL_BUFFERSIZE);     /* shift valid sets to the left */
      cur += LUAL_BUFFERSIZE;
    }
    if (cur < length) {
      my_seek(hnd, cur);
      my_read(hnd, buffer, length-cur);      /* read rest */
      my_seek(hnd, cur-clen);
      my_write(hnd, buffer, length-cur);
    }
    /* truncate base */
    try = ftruncate(hnd, length-clen);
    if (try == -1) {
      /* unlock file */
      my_seek(hnd, 0L);
      luaL_error(L, "Error in " LUA_QS " while truncating file %d.", "ads.comment", hnd);
    }
    /* read indices */
    my_seek(hnd, MAXNRECORDS);
    my_readl(hnd);  /* skip long, 1.12.9 */
    cnt = my_readl(hnd);
    my_seek(hnd, ADS_OFFSET);
    lua_newtable(L);
    for (i=0; i < cnt; i++) {
      lua_rawsetinumber(L, -1, i+1, my_readl(hnd));
    }
    /* update indices */
    my_seek(hnd, ADS_OFFSET);
    for (i=0; i < cnt; i++) {
      ind = agn_getinumber(L, -1, i+1);
      if (ind > cpos)
        my_writel(hnd, ind-clen);
      else
        lseek(hnd, 4L, SEEK_CUR);
    }
  }
  pos = (strcmp(comment, "") == 0) ? 0L : my_lof(hnd);
  /* prepare storing its new position at existing COMMENT position */
  my_seek(hnd, COMMENT);
  my_writel(hnd, pos);  /* insert comment position */
  /* write comment to end of file */
  if (pos != 0) {  /* new comment to be added ? */
    my_seek(hnd, pos);
    my_writel(hnd, l+1);
    my_write(hnd, value, l+1);
  }
  /* unlock file */
  my_seek(hnd, 0L);
  lua_pushtrue(L);
  return 1;
}


/* base_sync: flushes all unwritten content to the database file. */

static int base_sync (lua_State *L) {
  int try;
  int hnd = agnL_checkinteger(L, 1);
  /* file is not open ? -> leave this here since there will be lock errors later with closed files*/
  try = my_seek(hnd, 0L);
  if (try == -1) {
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.sync", hnd);
  }
  /* lock the file, seeker is at position 0 already */
  #ifdef _WIN32
  int res = _commit(hnd);
  #else
  int res = fsync(hnd);
  #endif
  /* unlock file */
  my_seek(hnd, 0L);
  if (!res)
    lua_pushboolean(L, 1);
  else
    lua_pushfail(L);
  return 1;
}


/* get all invalid (old) entries in a database, June 25, 2007 */

static int base_invalids (lua_State *L) {
  off64_t dsbegin, dsend, mrc, cnt, i, j, k, c, x, z, cpos, columns;
  int hnd, try, flag;
  size_t lenkey, lenval;
  hnd = agnL_checkinteger(L, 1);
  try = my_seek(hnd, 0L);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.invalids", hnd);
  }
  my_seek(hnd, MAXNRECORDS);
  mrc = my_readl(hnd);
  cnt = my_readl(hnd);
  my_seek(hnd, COLUMNS);
  columns = my_readl(hnd);
  char listflag = my_readc(hnd);
  if (listflag > 1)
    luaL_error(L, "Error in " LUA_QS ": function not applicable to sequences.", "ads.invalids");
  my_seek(hnd, COMMENT);
  cpos = my_readl(hnd);
  off64_t indextbl[cnt];
  /* read all indices */
  my_seek(hnd, ADS_OFFSET);
  for (i=0; i<cnt; i++) {
    indextbl[i] = my_readl(hnd);
  }
  /* sort them */
  tools_quicksort(indextbl, 0, cnt-1);
  dsbegin = mrc*4L+ADS_OFFSET;  /* start of current dataset section */
  dsend = my_lof(hnd)-1;        /* end of current dataset section */
  my_seek(hnd, dsbegin);
  lua_newtable(L);
  z = dsbegin;
  c = 0;
  while (z < dsend+1) {
    /* binsearch indextbl for an entry of the current position */
    i = 0; j = cnt-1; flag = 1;
    while (i <= j) {
      x = (i+j)/2;
      if (indextbl[x] == z) {
        flag = 0;
        break;
      }
      if (indextbl[x] < z)
        i = x + 1;
      else
        j = x - 1;
    }
    if (flag && z != cpos) {  /* invalid key found that is not the comment ? */
      c++;
      lua_rawsetinumber(L, -1, c, z);
    }
    lenkey = my_readl(hnd)+4L;
    /* set cursor to next entry */
    z += lenkey;
    my_seek(hnd, z);
    if (!listflag && z-lenkey != cpos) {
      lenval = 0;
      for (k=2; k<=columns; k++) {
        lenval = my_readl(hnd)+4L;
        z += lenval;
        lseek(hnd, lenval-4L, SEEK_CUR);
      }
    }
  }
  /* unlock file */
  my_seek(hnd, 0L);
  return 1;
}


static int base_clean (lua_State *L) {
  off64_t dsend, mrc, cnt, i, j, k, c, x, z, ind, length, cur, num, cpos, columns, lenvalues;
  int hnd, try, flag, cposflag, verbose;
  size_t lenkey, lenval, totlen;
  double time;
  hnd = agnL_checkinteger(L, 1);
  verbose = (agnL_optinteger(L, 2, 0));
  try = my_seek(hnd, 0L);
  if (try == -1) {  /* file is not open ? */
    luaL_error(L, "Error in " LUA_QS ": file #%d is not open.", "ads.clean", hnd);
  }
  time = clock();
  length = my_lof(hnd);
  my_seek(hnd, MAXNRECORDS);
  mrc = my_readl(hnd);
  cnt = my_readl(hnd);
  my_seek(hnd, COLUMNS);
  columns = my_readl(hnd);
  char listflag = my_readc(hnd);
  if (listflag > 1) {
    my_seek(hnd, 0L);
    luaL_error(L, "Error in " LUA_QS ": function not applicable to sequences.", "ads.clean");
  }
  my_seek(hnd, COMMENT);
  cpos = my_readl(hnd);
  char buffer[LUAL_BUFFERSIZE];
  off64_t indextbl[cnt];
  /* read all indices */
  if (verbose) fprintf(stderr, "Reading index section ... ");
  my_seek(hnd, ADS_OFFSET);
  /* store valid index in a C array and in a Lua table. The C array gets sorted afterwards.
     Using a Lua table is necessary here since too large C arrays crash the system. */
  lua_newtable(L);
  for (i=0; i<cnt; i++) {
    ind = my_readl(hnd);
    indextbl[i] = ind;
    lua_rawsetinumber(L, -1, i+1, ind);
  }
  /* sort them */
  tools_quicksort(indextbl, 0, cnt-1);
  if (verbose) {
    const char *str = (cnt == 1) ? "position" : "positions";
    fprintf(stderr, "got %lf %s.\n", (lua_Number)cnt, str);
  }
  dsend = my_lof(hnd)-1;       /* end of current dataset section */
  z = mrc*4L+ADS_OFFSET;       /* start of current dataset section */
  my_seek(hnd, z);
  c = 0;
  lua_newtable(L);
  if (verbose) fprintf(stderr, "Scanning invalid records ... ");
  /* z is cursor */
  while (z < dsend+1) {
    /* binsearch indextbl for an entry of the current position */
    i = 0; j = cnt-1; flag = 1;
    while (i <= j) {
      x = (i+j)/2;
      if (z == indextbl[x] || z == cpos) {  /* position is a valid index or comment */
        flag = 0;
        break;
      }
      if (indextbl[x] < z)
        i = x + 1;
      else
        j = x - 1;
    }
    if (flag) {  /* file position is not in index table */
      c++;
      lua_rawsetinumber(L, -1, c, z);
    }
    lenkey = my_readl(hnd)+4L;
    /* set cursor to next entry */
    z += lenkey;
    my_seek(hnd, z);
    if (!listflag && z-lenkey != cpos) {
      lenval = 0;
      for (k=2; k<=columns; k++) {
        lenval = my_readl(hnd)+4L;
        z += lenval;
        lseek(hnd, lenval-4L, SEEK_CUR);
      }
    }
  }
  int top = luaL_getn(L, -1);  /* number of entries in invalids Lua table */
  if (top == 0) {
    if (verbose) fprintf(stderr, "Nothing to be cleaned.\n");
    /* unlock file */
    my_seek(hnd, 0L);
    agn_poptop(L);  /* delete Lua table */
    lua_pushfalse(L);
    return 1;
  }
  if (verbose) {
    const char *str = (top == 1) ? "position" : "positions";
    fprintf(stderr, "got %d %s.\n", top, str);
  }
  /* store contents of Lua table in C array */
  off64_t invalids[top];
  for (i=0; i<top; i++)
    invalids[i] = (off64_t)agn_getinumber(L, -1, i+1);
  agn_poptop(L);  /* drop invalids Lua table */
  off64_t tobeshortened = 0;
  if (verbose) fprintf(stderr, "Reshifting valid records ... ");
  cposflag = 1;  /* change position of comment only once */
  /* for each invalid record do */
  for (i=0; i < top; i++) {
    ind = invalids[i];                       /* position of invalid item in record section */
    my_seek(hnd, ind);                       /* read it */
    lenkey = my_readl(hnd)+4L;               /*    length of key including length info */
    my_seek(hnd, ind+lenkey);                /*    set cursor to value */
    lenval = 0;
    if (!listflag) {
      for (j=2; j <= columns; j++) {           /* determine entire length of second to last column */
        lenvalues = my_readl(hnd)+4L;
        lenval += lenvalues;                 /*    length of value including length info */
        lseek(hnd, lenvalues-4L, SEEK_CUR);
      }
    }
    totlen = lenkey+lenval;                  /*    lengths of key and value plus length info */
    cur = ind+totlen;
    while (length-cur+1 >= LUAL_BUFFERSIZE) {
      my_seek(hnd, cur);                     /* set cursor to record following invalid one */
      my_read(hnd, buffer, LUAL_BUFFERSIZE);      /* read bufsize chars */
      my_seek(hnd, cur-totlen);              /* set cursor to begin of invalid record */
      my_write(hnd, buffer, LUAL_BUFFERSIZE);     /* shift valid sets to the left */
      cur += LUAL_BUFFERSIZE;
    }
    if (cur < length) {
      my_seek(hnd, cur);
      my_read(hnd, buffer, length-cur);     /* read rest */
      my_seek(hnd, cur-totlen);
      my_write(hnd, buffer, length-cur);
    }
    /* update valid indices Lua table and also position of comment */
    for (j=0; j < cnt; j++) {
      num = (off64_t)agn_getinumber(L, -1, j+1);
      if (num > ind) {
        lua_rawsetinumber(L, -1, j+1, num-totlen);
      }
      if (cpos != 0 && num > cpos && cposflag) {  /* if a comment is present, change cpos only once */
        cpos -= totlen;
        cposflag = 0;
      }
    }
    /* update invalid indices */
    for (j=i+1; j < top; j++) {
      invalids[j] = invalids[j]-totlen;
    }
    tobeshortened += totlen;
  }
  if (cposflag && cpos != 0) {  /* comment exists and is at end of file */
    cpos -= tobeshortened;
  }
  if (verbose) {
    fprintf(stderr, "Done.\n");
#ifndef LUA_DOS
    fprintf(stderr, "Truncating file by %lf bytes ... ", (lua_Number)tobeshortened);
#else
    fprintf(stderr, "Truncating file by %lld bytes ... ", tobeshortened);
#endif
  }
  try = ftruncate(hnd, length-tobeshortened);
  if (try == -1) {
    /* unlock file */
    my_seek(hnd, 0L);
    luaL_error(L, "Error while truncating file.\n");
  }
  if (verbose) {
    fprintf(stderr, "Done.\n");
    fprintf(stderr, "Rebuilding index ... ");
  }
  /* rewriting index section */
  my_seek(hnd, ADS_OFFSET);
  for (i=0; i < cnt; i++) {
    my_writel(hnd, (off64_t)agn_getinumber(L, -1, i+1));
  }
  agn_poptop(L);  /* drop Lua valid index table */
  if (verbose) {
    fprintf(stderr, "Done.\n");
    time = (clock()-time)/CLOCKS_PER_SEC;
    fprintf(stderr, "All done in %1.2f seconds.\n", (float)time);
  }
  /* reset comment position */
  my_seek(hnd, COMMENT);
  my_writel(hnd, cpos);
  /* unlock file */
  my_seek(hnd, 0L);
  lua_pushtrue(L);
  return 1;
}


static int base_filepos (lua_State *L) {
  int hnd;
  off64_t fpos;
  hnd = agnL_checkinteger(L, 1);
  fpos = my_fpos(hnd);
  if (fpos == -1)
    lua_pushfalse(L);
  else
    lua_pushnumber(L, fpos);
  return 1;
}


static int base_lock (lua_State *L) {  /* 0.32.5 */
  int hnd;
  size_t nargs;
  off64_t start, size;
  hnd = agnL_checkinteger(L, 1);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1) {  /* lock entire file (in Windows lock 2^63 bytes only) */
    start = 0;
    size = 0;
  } else {
    /* lock from current file position */
    start = my_fpos(hnd);
    size = agnL_checknumber(L, 2);
    if (size < 0) luaL_error(L, "Error in " LUA_QS ": must lock at least one byte.", "ads.lock");
  }
  lua_pushboolean(L, my_lock(hnd, start, size) == 0);
  return 1;
}


static int base_unlock (lua_State *L) {  /* 0.32.5 */
  int hnd;
  size_t nargs;
  off64_t start, size;
  hnd = agnL_checkinteger(L, 1);
  nargs = lua_gettop(L);
  luaL_checkstack(L, nargs, "too many arguments");
  if (nargs == 1) {  /* lock entire file */
    start = 0;
    size = 0;
  } else {
    /* lock from current file position */
    start = my_fpos(hnd);
    size = agnL_checknumber(L, 2);
    if (size < 0) luaL_error(L, "Error in " LUA_QS ": must unlock at least one byte.", "ads.unlock");
  }
  lua_pushboolean(L, my_unlock(hnd, start, size) == 0);
  return 1;
}


static const luaL_Reg ads[] = {
  {"attrib", base_attrib},                /* May 06, 2007 */
  {"clean", base_clean},                  /* July 06, 2007 */
  {"closebase", base_closebase},          /* May 05, 2007 */
  {"comment", base_comment},              /* July 11, 2007 */
  {"createbase", base_createbase},        /* May 05, 2007 */
  {"desc", base_desc},                    /* May 12, 2007 */
  {"expand", base_expand},                /* June 25, 2007 */
  {"fastseek", base_fastseek},            /* July 12, 2007 */
  {"filepos", base_filepos},              /* June 28, 2007 */
  {"free", base_free},                    /* July 06, 2007 */
  {"getall", base_getall},                /* April 08, 2008; tweaked December 24, 2008 */
  {"getkeys", base_getkeys},              /* May 06, 2007 */
  {"getvalues", base_getvalues},          /* May 12, 2007 */
  {"index", base_index},                  /* July 12, 2007 */
  {"indices", base_indices},              /* May 12, 2007 */
  {"invalids", base_invalids},            /* June 25, 2007 */
  {"iterate", base_iterate},              /* May 06, 2007 */
  {"lock", base_lock},                    /* June 12, 2010 */
  {"openbase", base_openbase},            /* May 05, 2007 */
  {"peek", base_peek},                    /* July 12, 2007 */
  {"rawsearch", base_rawsearch},          /* July 21, 2007 */
  {"readbase", base_readbase},            /* May 06, 2007 */
  {"remove", base_remove},                /* May 06, 2007 */
  {"retrieve", base_retrieve},            /* May 06, 2007 */
  {"sizeof", base_sizeof},                /* May 06, 2007 */
  {"sync", base_sync},                    /* May 12, 2007 */
  {"unlock", base_unlock},                /* June 12, 2010 */
  {"writebase", base_writebase},          /* May 06, 2007 */
  {NULL, NULL}
};


static void createmetatable (lua_State *L) {
  lua_createtable(L, 0, 1);  /* create metatable for strings */
  lua_pushliteral(L, "");  /* dummy string */
  lua_pushvalue(L, -2);
  lua_setmetatable(L, -2);  /* set string metatable */
  agn_poptop(L);  /* pop dummy string */
  lua_pushvalue(L, -2);  /* string library... */
  lua_setfield(L, -2, "__index");  /* ...is the __index metamethod */
  agn_poptop(L);  /* pop metatable */
}


/*
** Open base library
*/
LUALIB_API int luaopen_ads (lua_State *L) {
  luaL_register(L, AGENA_ADSLIBNAME, ads);
  lua_newtable(L);
  lua_setfield(L, -2, "openfiles");  /* table for information on all open files */
  createmetatable(L);
  return 1;
}

