/*
** $Id: agnconf.h, 2009/03/01 15:40:26 $
** Configuration file for Agena
** See Copyright Notice in agena.h
*/

#ifndef agnconf_h
#define agnconf_h


#include <limits.h>
#include <stddef.h>

#ifdef __cplusplus
extern "C"
{
#endif


/*
** ==================================================================
** Search for "@@" to find all configurable definitions.
** ===================================================================
*/


/*
@@ LUA_ANSI controls the use of non-ansi features.
** CHANGE it (define it) if you want Lua to avoid the use of any
** non-ansi feature or library.
*/
#if defined(__STRICT_ANSI__)
#define LUA_ANSI
#endif


#if !defined(LUA_ANSI) && defined(_WIN32)
#define LUA_WIN
#endif

#if defined(LUA_USE_LINUX)
#define LUA_USE_POSIX
#define LUA_USE_DLOPEN      /* needs an extra library: -ldl */
#define LUA_USE_READLINE   /* needs some extra libraries */
#endif

#if defined(LUA_USE_LINUXPLAIN)
#define LUA_USE_POSIX
#define LUA_USE_DLOPEN      /* needs an extra library: -ldl */
#endif

#if defined(LUA_USE_MACOSX)
#define LUA_USE_POSIX
#define LUA_DL_DYLD      /* does not need extra library */
#endif



/*
@@ LUA_USE_POSIX includes all functionallity listed as X/Open System
@* Interfaces Extension (XSI).
** CHANGE it (define it) if your system is XSI compatible.
*/
#if defined(LUA_USE_POSIX)
#define LUA_USE_MKSTEMP
#define LUA_USE_ISATTY
#define LUA_USE_POPEN
#ifndef __HAIKU__
#define LUA_USE_ULONGJMP
#endif
#endif


/*
@@ LUA_PATH and LUA_CPATH are the names of the environment variables that
@* Lua check to set its paths.
@* checks for initialisation code.
** CHANGE them if you want different names.
*/
#define LUA_PATH        "LUA_PATH"
#define LUA_CPATH       "LUA_CPATH"


/*
@@ LUA_PATH_DEFAULT is the default path that Lua uses to look for
@* Lua libraries.
@@ LUA_CPATH_DEFAULT is the default path that Lua uses to look for
@* C libraries.
** CHANGE them if your machine has a non-conventional directory
** hierarchy or if you want to install your libraries in
** non-conventional directories.
*/
#if defined(_WIN32)
#define AGN_CLIB  ".dll"
#else
#define AGN_CLIB  ".so"
#endif


/*
@@ LUA_DIRSEP is the directory separator (for submodules).
** CHANGE it if your machine does not use "/" as the directory separator
** and is not Windows. (On Windows Lua automatically uses "\".)
*/
#if defined(_WIN32)
#define LUA_DIRSEP   "\\"
#else
#define LUA_DIRSEP   "/"
#endif


/*
@@ LUA_PATHSEP is the character that separates templates in a path.
@@ LUA_PATH_MARK is the string that marks the substitution points in a
@* template.
@@ LUA_EXECDIR in a Windows path is replaced by the executable's
@* directory.
@@ LUA_IGMARK is a mark to ignore all before it when bulding the
@* luaopen_ function name.
** CHANGE them if for some reason your system cannot use those
** characters. (E.g., if one of those characters is a common character
** in file/directory names.) Probably you do not need to change them.
*/
#define LUA_PATHSEP   ";"
#define LUA_IGMARK   "-"


/*
@@ LUA_INTEGER is the integral type used by lua_pushinteger/lua_tointeger.
** CHANGE that if ptrdiff_t is not adequate on your machine. (On most
** machines, ptrdiff_t gives a good choice between int or long.)
*/
#define LUA_INTEGER   ptrdiff_t


/*
@@ LUA_API is a mark for all core API functions.
@@ LUALIB_API is a mark for all standard library functions.
** CHANGE them if you need to define those functions in some special way.
** For instance, if you want to create one Windows DLL with the core and
** the libraries, you may want to use the following definition (define
** LUA_BUILD_AS_DLL to get it).
*/
#if defined(LUA_BUILD_AS_DLL)

#if defined(LUA_CORE) || defined(LUA_LIB)
#define LUA_API __declspec(dllexport)
#else
#define LUA_API __declspec(dllimport)
#endif

#else

#define LUA_API      extern

#endif

/* more often than not the libs go together with the core */
#define LUALIB_API   LUA_API


/*
@@ LUAI_FUNC is a mark for all extern functions that are not to be
@* exported to outside modules.
@@ LUAI_DATA is a mark for all extern (const) variables that are not to
@* be exported to outside modules.
** CHANGE them if you need to mark them in some special way. Elf/gcc
** (versions 3.2 and later) mark them as "hidden" to optimize access
** when Lua is compiled as a shared library.
*/
#if defined(luaall_c)
#define LUAI_FUNC   static
#define LUAI_DATA   /* empty */

#elif defined(__GNUC__) && ((__GNUC__*100 + __GNUC_MINOR__) >= 302) && \
      defined(__ELF__)

/* avoid useless compiler warnings in Solaris, taken from:
   http://lua-users.org/lists/lua-l/2008-07/msg00165.html */
#if !(defined (__SVR4) && defined (__sun))
#define LUAI_FUNC   __attribute__((visibility("hidden"))) extern
#else
#define LUAI_FUNC	extern
#endif

#define LUAI_DATA   LUAI_FUNC

#else
#define LUAI_FUNC   extern
#define LUAI_DATA   extern
#endif



/*
@@ LUA_QL describes how error messages quote program elements.
** CHANGE it if you want a different appearance.
*/
#define LUA_QL(x)   "`" x "`"
#define LUA_QS      LUA_QL("%s")


/*
@@ LUA_IDSIZE gives the maximum size for the description of the source
@* of a function in debug information.
** CHANGE it if you want a different size.
*/
#define LUA_IDSIZE   60


/*
** {==================================================================
** Stand-alone configuration
** ===================================================================
*/

#if defined(lua_c) || defined(luaall_c)

/*
@@ lua_stdin_is_tty detects whether the standard input is a 'tty' (that
@* is, whether we're running lua interactively).
** CHANGE it if you have a better definition for non-POSIX/non-Windows
** systems.
*/
#if defined(LUA_USE_ISATTY)
#include <unistd.h>
#define lua_stdin_is_tty()   isatty(0)
#elif defined(LUA_WIN)
#include <io.h>
#include <stdio.h>
#define lua_stdin_is_tty()   _isatty(_fileno(stdin))
#else
#define lua_stdin_is_tty()   1  /* assume stdin is a tty */
#endif


/*
@@ LUA_PROMPT is the default prompt used by stand-alone Lua.
@@ LUA_PROMPT2 is the default continuation prompt used by stand-alone Lua.
** CHANGE them if you want different prompts. (You can also change the
** prompts dynamically, assigning to globals _PROMPT/_PROMPT2.)
*/
#define LUA_PROMPT      "> "
#define LUA_PROMPT2      "> "

/*
@@ LUA_PROGNAME is the default name for the stand-alone Lua program.
** CHANGE it if your stand-alone interpreter has a different name and
** your system is not able to detect that name automatically.
*/
#define LUA_PROGNAME      "agena"


/*
@@ LUA_MAXINPUT is the maximum length for an input line in the
@* stand-alone interpreter.
** CHANGE it if you need longer lines.
*/
#define LUA_MAXINPUT   2048  /* Agena 1.6.0 */


/*
@@ lua_readline defines how to show a prompt and then read a line from
@* the standard input.
@@ lua_saveline defines how to "save" a read line in a "history".
@@ lua_freeline defines how to free a line read by lua_readline.
** CHANGE them if you want to improve this functionality (e.g., by using
** GNU readline and history facilities).
*/
#if defined(LUA_USE_READLINE)
#include <stdio.h>
#include <readline/readline.h>
#include <readline/history.h>
#define lua_readline(L,b,p)   ((void)L, ((b)=readline(p)) != NULL)
#define lua_saveline(L,idx) \
   if (lua_strlen(L,idx) > 0)  /* non-empty line? */ \
     add_history(lua_tostring(L, idx));  /* add it to history */
#define lua_freeline(L,b)   ((void)L, free(b))
#else
#define lua_readline(L,b,p) \
   ((void)L, fputs(p, stdout), fflush(stdout),  /* show prompt */ \
   fgets(b, LUA_MAXINPUT, stdin) != NULL)  /* get line */
#define lua_saveline(L,idx)   { (void)L; (void)idx; }
#define lua_freeline(L,b)   { (void)L; (void)b; }
#endif

#endif

/* }================================================================== */


/*
@@ LUAI_GCPAUSE defines the default pause between garbage-collector cycles
@* as a percentage.
** CHANGE it if you want the GC to run faster or slower (higher values
** mean larger pauses which mean slower collection.) You can also change
** this value dynamically.
*/
#define LUAI_GCPAUSE   200  /* 200% (wait memory to double before next GC) */


/*
@@ LUAI_GCMUL defines the default speed of garbage collection relative to
@* memory allocation as a percentage.
** CHANGE it if you want to change the granularity of the garbage
** collection. (Higher values mean coarser collections. 0 represents
** infinity, where each step performs a full collection.) You can also
** change this value dynamically.
*/
#define LUAI_GCMUL   200 /* GC runs 'twice the speed' of memory allocation */

/*
@@ LUA_COMPAT_LOADLIB controls compatibility about global loadlib.
** CHANGE it to undefined as soon as you do not need a global 'loadlib'
** function (the function is still available as 'package.loadlib').
*/
#undef LUA_COMPAT_LOADLIB

/*
@@ LUA_COMPAT_VARARG controls compatibility with old vararg feature.
** CHANGE it to undefined as soon as your programs use only '...' to
** access vararg parameters (instead of the old 'arg' table).

#define LUA_COMPAT_VARARG
*/

/*
@@ LUA_COMPAT_OPENLIB controls compatibility with old 'luaL_openlib'
@* behavior.
** CHANGE it to undefined as soon as you replace to 'luaL_register'
** your uses of 'luaL_openlib'
*/
#define LUA_COMPAT_OPENLIB


/*
@@ luai_apicheck is the assert macro used by the Lua-C API.
** CHANGE luai_apicheck if you want Lua to perform some checks in the
** parameters it gets from API calls. This may slow down the interpreter
** a bit, but may be quite useful when debugging C code that interfaces
** with Lua. A useful redefinition is to use assert.h.
*/
#if defined(LUA_USE_APICHECK)
#include <assert.h>
#define luai_apicheck(L,o)   { (void)L; assert(o); }
#else
#define luai_apicheck(L,o)   { (void)L; }
#endif


/*
@@ LUAI_BITSINT defines the number of bits in an int.
** CHANGE here if Lua cannot automatically detect the number of bits of
** your machine. Probably you do not need to change this.
*/
/* avoid overflows in comparison */
#if INT_MAX-20 < 32760
#define LUAI_BITSINT   16
#elif INT_MAX > 2147483640L
/* int has at least 32 bits */
#define LUAI_BITSINT   32
#else
#error "you must define LUA_BITSINT with number of bits in an integer"
#endif


/*
@@ LUAI_UINT32 is an unsigned integer with at least 32 bits.
@@ LUAI_INT32 is an signed integer with at least 32 bits.
@@ LUAI_UMEM is an unsigned integer big enough to count the total
@* memory used by Lua.
@@ LUAI_MEM is a signed integer big enough to count the total memory
@* used by Lua.
** CHANGE here if for some weird reason the default definitions are not
** good enough for your machine. (The definitions in the 'else'
** part always works, but may waste space on machines with 64-bit
** longs.) Probably you do not need to change this.
*/
#if LUAI_BITSINT >= 32
#define LUAI_UINT32   unsigned int
#define LUAI_INT32   int
#define LUAI_MAXINT32   INT_MAX
#define LUAI_MININT32	INT_MIN
#define LUAI_UMEM   size_t
#define LUAI_MEM   ptrdiff_t
#else
/* 16-bit ints */
#define LUAI_UINT32   unsigned long
#define LUAI_INT32   long
#define LUAI_MAXINT32   LONG_MAX
#define LUAI_MININT32	LONG_MIN
#define LUAI_UMEM   unsigned long
#define LUAI_MEM   long
#endif


/*
@@ LUAI_MAXCALLS limits the number of nested calls.
** CHANGE it if you need really deep recursive calls. This limit is
** arbitrary; its only purpose is to stop infinite recursion before
** exhausting memory.
*/
#define LUAI_MAXCALLS   20000


/*
@@ LUAI_MAXCSTACK limits the number of Lua stack slots that a C function
@* can use.
** CHANGE it if you need lots of (Lua) stack space for your C
** functions. This limit is arbitrary; its only purpose is to stop C
** functions to consume unlimited stack space.
*/
#define LUAI_MAXCSTACK   2048



/*
** {==================================================================
** CHANGE (to smaller values) the following definitions if your system
** has a small C stack. (Or you may want to change them to larger
** values if your system has a large C stack and these limits are
** too rigid for you.) Some of these constants control the size of
** stack-allocated arrays used by the compiler or the interpreter, while
** others limit the maximum number of recursive calls that the compiler
** or the interpreter can perform. Values too large may cause a C stack
** overflow for some forms of deep constructs.
** ===================================================================
*/


/*
@@ LUAI_MAXCCALLS is the maximum depth for nested C calls (short) and
@* syntactical nested non-terminals in a program.
*/
#define LUAI_MAXCCALLS      200


/*
@@ LUAI_MAXVARS is the maximum number of local variables per function
@* (must be smaller than 250).
*/
#define LUAI_MAXVARS      200


/*
@@ LUAI_MAXUPVALUES is the maximum number of upvalues per function
@* (must be smaller than 250).
*/
#define LUAI_MAXUPVALUES   60


/*
@@ LUAL_BUFFERSIZE is the buffer size used by the lauxlib buffer system.
*/
#define LUAL_BUFFERSIZE      BUFSIZ

/* }================================================================== */




/*
** {==================================================================
@@ LUA_NUMBER is the type of numbers in Lua.
** CHANGE the following definitions only if you want to build Lua
** with a number type different from double. You may also need to
** change lua_number2int & lua_number2integer.
** ===================================================================
*/

#define LUA_NUMBER_DOUBLE
#define LUA_NUMBER   double

#ifndef __cplusplus

#ifndef PROPCMPLX
#include <complex.h>
#define agn_Complex complex double
#endif

#else

/* we are in C++ */
#ifndef PROPCMPLX

#ifdef __linux__
#include <complex.h>
#else
#include <complex>
using std::complex;
#endif

#define agn_Complex complex double
#endif

#endif  /* __cplusplus */

/*
@@ LUAI_UACNUMBER is the result of an 'usual argument conversion'
@* over a number.
*/
#define LUAI_UACNUMBER   double


/*
@@ LUA_NUMBER_SCAN is the format for reading numbers.
@@ LUA_NUMBER_FMT is the format for writing numbers.
@@ lua_number2str converts a number to a string.
@@ LUAI_MAXNUMBER2STR is maximum size of previous conversion.
@@ lua_str2number converts a string to a number.
*/
#define LUA_NUMBER_SCAN      "%lf"
#define LUA_NUMBER_FMT       "%.14g"
#define lua_number2str(s,n)  sprintf((s), L->numberformat, (n))
#define LUAI_MAXNUMBER2STR   32 /* 16 digits, sign, point, and \0 */
#define lua_str2number(s,p)  strtod((s), (p))

/*
@@ The luai_num* macros define the primitive operations over numbers.
*/
#if defined(LUA_CORE)

#include <math.h>

#include "agncmpt.h"

#define luai_numsign(x)     ( ((x) == 0) ? 0 : (((x) < 0) ? -1 : ((x) > 0 ? 1 : AGN_NAN)))
#define luai_numadd(a,b)    ((a)+(b))
#define luai_numsub(a,b)    ((a)-(b))
#define luai_nummul(a,b)    ((a)*(b))
#define luai_numdiv(a,b)    ( ((b) != 0) ? ((a)/(b)) : (AGN_NAN) )          /* changed 0.10.0 */
#define luai_numrecip(a)    ( ((a) != 0) ? (1/(a)) : (AGN_NAN) )            /* added 1.13.0 */
#define luai_numpercent(a,b)  ( ((b) != 0) ? ((a)*(b)/100) : (AGN_NAN) )    /* added 1.11.3 */
#define luai_numpercentratio(a,b)    ( ((b) != 0) ? ((a)/(b))*100 : (AGN_NAN) )   /* added 1.10.6 */
#define luai_numpercentadd(a,b) ( ((b) != 0) ? (a) + ((a)*(b)/100) : (AGN_NAN) )  /* added 1.11.3 */
#define luai_numpercentsub(a,b) ( ((b) != 0) ? (a) - ((a)*(b)/100) : (AGN_NAN) )  /* added 1.11.3 */
#define luai_numintdiv(a,b) (((b) != 0) ? (luai_numsign((a))*luai_numsign((b))*floor(fabs((a)/(b)))) : (AGN_NAN) )  /*0.32.2b fix */
   /* more exact for neg values; added 0.5.4; changed 0.10.0 */
#define luai_nummod(a,b)    ((a) - floor((a)/(b))*(b))
#define luai_numpow(a,b)    (((a) == 0 && (b) < 0 && trunc((b)) == (b)) ? AGN_NAN : pow((a),(b)) )
#define luai_numipow(a,b)   (tools_intpow((a),(b))) /* added 0.9.2; integer exponentiation is 40 % faster than pow */
#define luai_numabs(a)      (fabs((a)))        /* added 0.6.0; fabs is faster than a macro */
#define luai_numasin(a)     ( (fabs((a)) > 1) ? AGN_NAN : asin((a)))       /* added 0.27.0 */
#define luai_numacos(a)     ( (fabs((a)) > 1) ? AGN_NAN : acos((a)))       /* added 0.27.0 */
#define luai_numatan(a)     (atan((a)))        /* added 0.7.1 */
#define luai_numexp(a)      (exp((a)))         /* added 0.6.0 */
#define luai_numsqrt(a)     ( ((a) >= 0) ? (sqrt((a))) : (AGN_NAN) )        /* changed 0.10.0 */
#define luai_numln(a)       ( ((a) > 0) ? (log((a))) : (AGN_NAN) )          /* changed 0.10.0 */
#define luai_numsin(a)      (sin((a)))         /* added 0.6.0 */
#define luai_numcos(a)      (cos((a)))         /* added 0.6.0 */
#define luai_numtan(a)      (tan((a)))         /* added 0.7.1 */
#define luai_numsinh(a)     (sinh((a)))        /* added 0.23.0 */
#define luai_numcosh(a)     (cosh((a)))        /* added 0.23.0 */
#define luai_numtanh(a)     (tanh((a)))        /* added 0.23.0 */
#define luai_numint(a)      (trunc((a)))       /* added 0.6.0 */
#define luai_numentier(a)   (floor((a)))       /* added 0.7.1 */
#define luai_numiseven(a)   (fmod((a), 2) == 0)  /* added 0.7.1 */
#define luai_numlngamma(a)  ( ((a) > 0) ? (lgamma((a))) : (AGN_NAN) )        /* changed 0.10.0 */
#define luai_numunm(a)      (-(a))
#define luai_numeq(a,b)     (((a)==(b)) || ((isnan((a)) && isnan((b)))))  /* changed 0.10.0 */
#define luai_numlt(a,b)     ((a)<(b))
#define luai_numle(a,b)     ((a)<=(b))
#define luai_numisnan(a)    (isnan((a)))
#define luai_numisfinite(a) (isfinite((a)))  /* added 0.9.2 */
#define luai_bnot(a)        (~(lua_Integer)(a))  /* added 0.27.0 */

/*
@@ The agnc_* macros define the primitive operations over complex numbers.
*/

#ifndef PROPCMPLX
#define agnc_add(a,b)       ((a) + (b))
#define agnc_sub(a,b)       ((a) - (b))
#define agnc_mul(a,b)       ((a) * (b))
#define agnc_div(a,b)       (((b) == 0) ? AGN_NAN : (a) / (b))

/* do not use _SUNSOLARIS here. It won't work. But why not ? 1.12.7, XXX */
#if !(defined (__SVR4) && defined (__sun))
#define agnc_pow(a,b) { \
    agn_Complex r; \
    if (creal((a)) == 0 && cimag((a)) == 0 && creal((b)) <= 0) { \
      if (cimag((b)) == 0) \
        r = AGN_NAN; \
      else \
        r = 0; \
    } else \
      r = cpow((a), (b)); \
    (r); \
}
#else
/* Solaris 10, patched 1.12.9 */
#define agnc_pow(a,b) { \
    agn_Complex r; \
    lua_Number ar; \
    ar = creal((a)); \
    if (ar != 0) { \
      r = cpow((a), (b)); \
    } else { \
      lua_Number ai, br, bi, t1, t2, t4, t6, t9, t12; \
      ai = cimag((a)); br = creal((b)); bi = cimag((b)); \
      if (ai == 0) { \
        if (br <= 0) \
          r = (bi == 0) ? AGN_NAN : 0; \
        else \
          r = 0; \
      } else { \
        t1 = ar*ar; t2 = ai*ai; t4 = log(t1+t2); t6 = atan2(ai,ar); \
        t9 = exp(br*t4/2-bi*t6); t12 = bi*t4/2+br*t6; \
        r = t9*cos(t12) + I*t9*sin(t12); \
      } \
    } \
    (r); \
}
#endif

#define agnc_unm(a)         ((-1)*(a))
#define agnc_eq(a,b)        (((a)==(b)) || ((isnan((a)) && isnan((b)))))

/* 0.31.2 patch: if imag(r) == -0, then other complex functions return the imaginary unit with the wrong sign */
#define agnc_sin(a) { \
  agn_Complex r; \
  r = csin((a)); \
  if (cimag(r) == -0) r = creal(r) + 0*I; \
  (r); \
}

/* 0.31.2 patch: if imag(r) == -0, then other complex functions return the imaginary unit with the wrong sign */
#define agnc_cos(a) { \
  agn_Complex r; \
  r = ccos((a)); \
  if (cimag(r) == -0) r = creal(r) + 0*I; \
  (r); \
}

#define agnc_tan(a)         (ctan((a)))
#define agnc_sinh(a)        (csinh((a)))
#define agnc_cosh(a)        (ccosh((a)))
#define agnc_tanh(a)        (ctanh((a)))

/* patched 0.31.2, ANSI C casin returns wrong results */

#if (!(defined (__SVR4) && defined (__sun) && defined(__sparc)))
#define agnc_asin(z) { \
    agn_Complex (r); \
    if (cimag((z)) != 0.0 || fabs(creal((z))) < 1) { \
      (r) = casin((z)); \
      if (cimag(r) == -0) (r) = creal(r) + 0*I; \
    } \
    else {  /* asin(x+I*y) with |x| > 1 and y = 0 returns undefined, so circumvent this */ \
      lua_Number a, b, t1, t2, t4, t6, t12, t14; \
      a = creal((z)); b = cimag((z)); \
      t1 = a*a; \
      t2 = b*b; \
      t4 = sqrt(t1+2.0*a+1.0+t2); \
      t6 = sqrt(t1-2.0*a+1.0+t2); \
      t12 = pow((t4+t6)/2,2.0); \
      t14 = sqrt(t12-1.0); \
      (r) = asin((t4-t6)/2)+I*csgn(b, -a)*log((t4+t6)/2+t14); \
    } \
    (r); \
  }

/* patched 0.31.2, ANSI C cacos returns wrong results, e.g. arccos(-1) = undefined, and arccos(1) = undefined */

#define agnc_acos(z) { \
    agn_Complex (r); \
    if (cimag((z)) != 0.0 || fabs(creal((z))) < 1) { \
      r = cacos((z)); \
      if (cimag((r)) == -0) (r) = creal(r) + 0*I; \
    } \
    else {  /* acos(x+I*y) with |x| > 1 and y = 0 returns undefined, so circumvent this */ \
      lua_Number a, b, t1, t2, t4, t6, t12, t14, t15; \
      a = creal((z)); b = cimag((z)); \
      t1 = a*a; \
      t2 = b*b; \
      t4 = sqrt(t1+2.0*a+1.0+t2); \
      t6 = sqrt(t1-2.0*a+1.0+t2); \
      t12 = pow((t4+t6)/2,2.0); \
      t14 = sqrt(t12-1.0); \
      t15 = t4/2-t6/2; \
      if (t15 == 1) \
        r = I*csgn(-b, a)*log((t4+t6)/2+t14); \
      else if (t15 == -1) \
        r = PI+I*csgn(-b, a)*log((t4+t6)/2+t14); \
      else \
        r = acos(t15)+I*csgn(-b, a)*log((t4+t6)/2+t14); \
    } \
    (r); \
  }

#else

#define agnc_asin(z) { \
    agn_Complex (r); \
    (r) = casin((z)); \
    if ((cimag((z)) == 0 || cimag((z)) == -0) && creal((z)) > 0) (r) = conj((r)); \
    (r); \
  }

#define agnc_acos(z) { \
    agn_Complex (r); \
    (r) = cacos((z)); \
    if ((cimag((z)) == 0 || cimag((z)) == -0) && creal((z)) > 0) (r) = conj((r)); \
    (r); \
  }


#endif

/* patched 0.31.2, ANSI C atan returns wrong results */

/* LEAVE THIS HERE, OTHERWISE GCC WON'T COMPILE */
#include "cephes.h"  /* for complex lngamma */

#define agnc_atan(a) { \
  agn_Complex r; \
  r = carctan((a)); \
  if (cimag(r) == -0) r = creal(r) + 0*I; \
  (r); \
}

#define agnc_exp(a)        (cexp((a)))
#define agnc_ln(a)         (((a) == 0) ? (AGN_NAN) : clog((a)))  /* 0.26.1 fix */
#define agnc_sqrt(a)       (csqrt((a)))
#define agnc_abs(a)        (cabs((a)))
#define agnc_entier(a) { \
    lua_Number aa, bb, re, im; \
    agn_Complex (r), (X); \
    re = (creal(a)); \
    im = (cimag(a)); \
    aa = (re)-floor((re)); \
    bb = (im)-floor((im)); \
    if ((aa+bb) < 1) { \
      X = 0+0*I; } \
    else if ((aa+bb) >= 1 && (aa >= bb)) { \
      X = 1+0*I; } \
    else { \
      X = 1*I; } \
    r = floor((re)) + I*floor((im)) + X; \
    (r); \
  }

/* 0.33.2 */
#define agnc_int(a) { \
    lua_Number re, im; \
    agn_Complex (r); \
    re = (creal(a)); \
    im = (cimag(a)); \
    r = trunc((re)) + I*trunc((im)); \
    (r); \
  }

/* 0.22.2: compute the integer power of (z::complex)^(e::number). If n is a posint and n < 30 then
   use faster iteration formula. If n >= 30, then the iteration formula is slower on an Atom 330 CPU
   than the general formula. Checking for n = 2 and n = 3 does not slow down the procedure. */
#define agnc_ipow(z,e) {  \
    lua_Number (a), (b), (n);  \
    agn_Complex (r); \
    (a) = creal((z)); (b) = cimag((z));  \
    if (cimag((e)) != 0) luaG_runerror(L, "exponent must be of type number");  \
    n = creal((e));  \
    if ((n) == 2.0) { \
      lua_Number t1, t4;  \
      t1 = a*a; t4 = b*b;  \
      r = t1+(2.0*I)*a*b-t4;  \
    }  \
    else if ((n) == 3.0) { \
      lua_Number t1, t5; \
      t1 = a*a; \
      t5 = b*b; \
      r = t1*a+(3.0*I)*t1*b-3.0*a*t5+(-I)*t5*b; \
    }  \
    else if (((n) > 3.0) && ((n) < 30.0) && ((n) == floor(n))) {  \
      lua_Number (ra), (rb), (t);  \
      int i;  \
      ra = (a); rb = (b);  \
      for (i=1; i < (n); i++) {  \
        t = (a);  \
        (a) = (a)*ra-(b)*rb;  \
        (b) = t*rb+(b)*ra;  \
      }  \
      r = (a) + (b)*I;  \
    }  \
    else { \
      lua_Number t1, t2, t5, t6, t7, t8, t9; \
      t1 = (a)*(a);  \
      t2 = (b)*(b);  \
      t5 = pow(t1+t2,0.5*(n));  \
      t6 = atan2((b),(a));  \
      t7 = (n)*t6;  \
      t8 = cos(t7);  \
      t9 = sin(t7);  \
      r = t5*(t8+I*t9);  \
    }; \
    (r); \
}


#define agnc_lngamma(z) { \
  (clgam(z)); \
}

#else

#define agnc_add(z,a,b,c,d)    agnCmplx_create((z), ((a)+(c)), ((b)+(d)))
#define agnc_sub(z,a,b,c,d)    agnCmplx_create((z), ((a)-(c)), ((b)-(d)))
#define agnc_mul(z,a,b,c,d)    agnCmplx_create((z), ((a)*(c)-(b)*(d)), ((a)*(d)+(b)*(c)))
#define agnc_div(z,a,b,c,d)    \
    {lua_Number (e) = ((c)*(c)+(d)*(d)); agnCmplx_create((z), ((a)*(c)+(b)*(d))/((e)), ((b)*(c)-(a)*(d))/(e)); }
/* patched 1.12.7, 1.12.9 */
#define agnc_pow(z,a,b,c,d) { \
    lua_Number t1, t2, t4, t6, t9, t12; \
    if ( \
        (a) == 0 && (b) == 0 && \
        ( ( (c) == 0 && (d) != 0 ) || ( (c) > 0 && (d) == 0 ) ) \
      ) { \
      agnCmplx_create((z), 0, 0); \
    } else { \
      t1 = (a)*(a); t2 = (b)*(b); t4 = log(t1+t2); t6 = atan2((b),(a)); \
      t9 = exp((c)*t4/2-(d)*t6); t12 = d*t4/2+(c)*t6; \
      agnCmplx_create((z), t9*cos(t12), t9*sin(t12)); \
    } \
}
#define agnc_unm(z,a,b)        agnCmplx_create((z), (-(a)), (-(b)))
#define agnc_eq(a,b,c,d)       (((a)==(c) && (b)==(d)) || ((isnan((a)) && isnan((c)))))
#define agnc_sin(z,a,b) { \
    lua_Number i; \
    (i) = cos((a))*sinh((b)); \
    if ((i) == -0) (i) = 0; \
    agnCmplx_create((z), sin((a))*cosh((b)), (i)); \
}

#define agnc_cos(z,a,b) { \
    lua_Number i; \
    (i) = -sin((a))*sinh((b)); \
    if ((i) == -0) (i) = 0; \
    agnCmplx_create((z), cos((a))*cosh((b)), (i)); \
}

#define agnc_tan(z,a,b) { \
    lua_Number t2, t4, t5, t6, t8; \
    t2 = cos((a)); t4 = t2*t2; t5 = sinh((b)); \
    t6 = t5*t5; t8 = 1/(t4+t6); \
    agnCmplx_create((z), sin((a))*t2*t8, t5*cosh((b))*t8); \
  }

#define agnc_asin(z,a,b) { \
    lua_Number t1, t2, t4, t6, t12, t13, t14; \
    t1 = (a)*(a); \
    t2 = (b)*(b); \
    t4 = sqrt((t1)+2.0*(a)+1.0+t2); \
    t6 = sqrt((t1)-2.0*(a)+1.0+t2); \
    t12 = pow(t4/2+t6/2, 2.0); \
    t13 = t12-1.0; \
    if (fabs(t13) < 1.2e-019) (t13) = 0.0; \
    t14 = sqrt((t13)); \
    agnCmplx_create((z), asin((t4)/2-(t6)/2), csgn((b), -(a)) * log((t4)/2+(t6)/2+(t14))); \
  }

#define agnc_acos(z,a,b) { \
    lua_Number t1, t2, t4, t6, t12, t13, t14, t15, r, i; \
    t1 = a*a; \
    t2 = b*b; \
    t4 = sqrt(t1+2.0*a+1.0+t2); \
    t6 = sqrt(t1-2.0*a+1.0+t2); \
    t12 = pow(t4/2+t6/2, 2.0); \
    (t13) = (t12)-1.0; \
    if (fabs(t13) < 1.2e-019) (t13) = 0.0; \
    t14 = sqrt(t13); \
    t15 = t4/2-t6/2; \
    i = csgn(-b, a)*log(t4/2+t6/2+t14); \
    if (t15 == 1) r = 0; \
    else if (t15 == -1) r = PI; \
    else r = acos(t15); \
    agnCmplx_create((z), (r), (i)); \
  }

#define agnc_sinh(z,a,b)       agnCmplx_create((z), sinh((a))*cos((b)), cosh((a))*sin((b)))
#define agnc_cosh(z,a,b)       agnCmplx_create((z), cosh((a))*cos((b)), sinh((a))*sin((b)))

#define agnc_tanh(z,a,b) {  \
    lua_Number t1, t4, t5, t6, t8;  \
    t1 = sinh((a));  \
    t4 = t1*t1;  \
    t5 = cos((b));  \
    t6 = t5*t5;  \
    t8 = 1/(t4+t6);  \
    agnCmplx_create((z), t1*cosh((a))*t8, sin((b))*t5*t8);  \
  }

#define agnc_atan(z,a,b) { \
    lua_Number t3, t5, t6, t9, r, i; \
    t3 = (b)+1.0; t5 = (a)*(a); t6 = t3*t3; t9 = pow((b)-1.0,2.0); \
    if (a == 0 || a == -0) { \
      if ((b == 1 || b == -1)) { \
        r = AGN_NAN; \
        i = AGN_NAN; \
        goto lcatan2; \
      } \
      else if (b < -1 && b == trunc(b)) { \
        r = -PI/2; \
        goto lcatan1; \
      } \
    } \
    r = atan2((a), 1.0-(b))/2-atan2(-(a),t3)/2; \
  lcatan1: \
    i = log((t5+t6)/(t5+t9))/4; \
    if (i == -0) i = 0; \
  lcatan2: \
    agnCmplx_create((z), r, i); \
  }

#define agnc_exp(z,a,b) { \
    lua_Number t1; \
    t1 = exp((a)); \
    agnCmplx_create((z), t1*cos((b)), t1*sin((b))); \
  }

#define agnc_ln(z,a,b) { \
    lua_Number t1, t2; \
    t1 = (a)*(a); t2 = (b)*(b); \
    if (t1 == 0 && t2 == 0) { \
      agnCmplx_create((z), AGN_NAN, 0); } \
    else { \
      agnCmplx_create((z), log(t1+t2)/2, atan2((b),(a))); } \
  }

#define agnc_sqrt(z,a,b) { \
    lua_Number t1, t2, t4, t6, t10; \
    t1 = (a)*(a); t2 = (b)*(b); t4 = sqrt(t1+t2); \
    t6 = sqrt(2.0*t4+2.0*(a)); t10 = sqrt(2.0*t4-2.0*(a)); \
    agnCmplx_create((z), t6/2, csgn((b), -(a))*t10/2); \
  }

#define agnc_abs(a,b)    (sqrt((a)*(a)+(b)*(b)))

#define agnc_entier(z,a,b) { \
    lua_Number aa, bb, X, Y; \
    aa = (a)-floor((a)); \
    bb = (b)-floor((b)); \
    if ((aa+bb) < 1) { \
      X = 0; Y = 0; } \
    else if ((aa+bb) >= 1 && (aa >= bb)) { \
      X = 1; Y = 0; } \
    else { \
      X = 0; Y = 1; } \
    agnCmplx_create((z), floor((a))+X, floor((b))+Y); \
  }

/* 0.33.2 */
#define agnc_int(z,a,b) { \
    lua_Number aa, bb; \
    aa = (a)-trunc((a)); \
    bb = (b)-trunc((b)); \
    agnCmplx_create((z), trunc((a)), trunc((b))); \
  }

/* 0.22.2: with n = 3, A and B need to be defined in order to compile correctly
   (a and b cannot be assigned values) */
#define agnc_ipow(z,a,b,n,p) {  \
    if ((p) != 0) luaG_runerror(L, "exponent must be of type number");  \
    if ((n) == 2.0) {  \
      lua_Number t1, t4;  \
      t1 = a*a; t4 = b*b;  \
      agnCmplx_create((z), t1-t4, 2.0*a*b);  \
    }  \
    else if ((n) == 3.0) { \
      lua_Number t1, t5; \
      t1 = a*a; \
      t5 = b*b; \
      agnCmplx_create((z), t1*a-3.0*a*t5, 3.0*t1*b-t5*b);  \
    }  \
    else if (((n) > 3.0) && ((n) < 30.0) && ((n) == floor(n))) {  \
      lua_Number (ra), (rb), (t), (A), (B);  \
      int i;  \
      A = (a); B = (b);  \
      ra = (a); rb = (b);  \
      for (i=1; i < (n); i++) {  \
        t = (A);  \
        (A) = (A)*ra-(B)*rb;  \
        (B) = t*rb+(B)*ra;  \
      }  \
      agnCmplx_create((z), (A), (B));  \
    }  \
    else {  \
      lua_Number t1, t2, t5, t6, t7, t8, t9; \
      t1 = (a)*(a);  \
      t2 = (b)*(b);  \
      t5 = pow(t1+t2,0.5*(n));  \
      t6 = atan2((b),(a));  \
      t7 = (n)*t6;  \
      t8 = cos(t7);  \
      t9 = sin(t7);  \
      agnCmplx_create((z), t5*t8, t5*t9);  \
    };  \
}

/* LEAVE THIS HERE, OTHERWISE GCC WON'T COMPILE */
#include "cephes.h"  /* for complex lngamma */

#define agnc_lngamma(z,a,b) { \
  double re, im; \
  clgam((a), (b), &(re), &(im)); \
  agnCmplx_create((z), (re), (im)); \
}

#endif  /* PROPCMPLX */

#endif

/*
@@ lua_number2int is a macro to convert lua_Number to int.
@@ lua_number2integer is a macro to convert lua_Number to lua_Integer.
** CHANGE them if you know a faster way to convert a lua_Number to
** int (with any rounding method and without throwing errors) in your
** system. In Pentium machines, a naive typecast from double to int
** in C is extremely slow, so any alternative is worth trying.
*/

/* On a Pentium, resort to a trick */
#if defined(LUA_NUMBER_DOUBLE) && !defined(LUA_ANSI) && !defined(__SSE2__) && \
    (defined(__i386) || defined (_M_IX86) || defined(__i386__))

/* On a Microsoft compiler, use assembler */
#if defined(_MSC_VER)

#define lua_number2int(i,d)   __asm fld d   __asm fistp i
#define lua_number2integer(i,n)      lua_number2int(i, n)

/* the next trick should work on any Pentium, but sometimes clashes
   with a DirectX idiosyncrasy */
#else

union luai_Cast { double l_d; long l_l; };
#define lua_number2int(i,d) \
  { volatile union luai_Cast u; u.l_d = (d) + 6755399441055744.0; (i) = u.l_l; }
#define lua_number2integer(i,n)      lua_number2int(i, n)

#endif


/* this option always works, but may be slow */
#else
#define lua_number2int(i,d)   ((i)=(int)(d))
#define lua_number2integer(i,d)   ((i)=(lua_Integer)(d))

#endif

/* }================================================================== */


/*
@@ LUAI_USER_ALIGNMENT_T is a type that requires maximum alignment.
** CHANGE it if your system requires alignments larger than double. (For
** instance, if your system supports long doubles and they must be
** aligned in 16-byte boundaries, then you should add long double in the
** union.) Probably you do not need to change this.
*/
#define LUAI_USER_ALIGNMENT_T   union { double u; void *s; long l; }


/*
@@ LUAI_THROW/LUAI_TRY define how Lua does exception handling.
** CHANGE them if you prefer to use longjmp/setjmp even with C++
** or if want/don't to use _longjmp/_setjmp instead of regular
** longjmp/setjmp. By default, Lua handles errors with exceptions when
** compiling as C++ code, with _longjmp/_setjmp when asked to use them,
** and with longjmp/setjmp otherwise.
*/
#if defined(__cplusplus)
/* C++ exceptions */
#define LUAI_THROW(L,c)   throw(c)
#define LUAI_TRY(L,c,a)   try { a } catch(...) \
   { if ((c)->status == 0) (c)->status = -1; }
#define luai_jmpbuf   int  /* dummy variable */

#elif defined(LUA_USE_ULONGJMP)
/* in Unix, try _longjmp/_setjmp (more efficient) */
#define LUAI_THROW(L,c)   _longjmp((c)->b, 1)
#define LUAI_TRY(L,c,a)   if (_setjmp((c)->b) == 0) { a }
#define luai_jmpbuf   jmp_buf

#else
/* default handling with long jumps */
#define LUAI_THROW(L,c)   longjmp((c)->b, 1)
#define LUAI_TRY(L,c,a)   if (setjmp((c)->b) == 0) { a }
#define luai_jmpbuf   jmp_buf

#endif


/*
@@ LUA_MAXCAPTURES is the maximum number of captures that a pattern
@* can do during pattern-matching.
** CHANGE it if you need more captures. This limit is arbitrary.
*/
#define LUA_MAXCAPTURES      32


/*
@@ lua_tmpnam is the function that the OS library uses to create a
@* temporary name.
@@ LUA_TMPNAMBUFSIZE is the maximum size of a name created by lua_tmpnam.
** CHANGE them if you have an alternative to tmpnam (which is considered
** insecure) or if you want the original tmpnam anyway.  By default, Lua
** uses tmpnam except when POSIX is available, where it uses mkstemp.
*/
#if defined(loslib_c) || defined(luaall_c)

#if defined(LUA_USE_MKSTEMP)
#include <unistd.h>
#define LUA_TMPNAMBUFSIZE   32
#define lua_tmpnam(b,e)   { \
   strcpy(b, "/tmp/lua_XXXXXX"); \
   e = mkstemp(b); \
   if (e != -1) close(e); \
   e = (e == -1); }

#else
#define LUA_TMPNAMBUFSIZE   1000
/* L_tmpnam */
#define lua_tmpnam(b,e)      { e = (tmpnam(b) == NULL); }
#endif

#endif


/*
@@ lua_popen spawns a new process connected to the current one through
@* the file streams.
** CHANGE it if you have a way to implement it in your system.
*/
#if defined(LUA_USE_POPEN)

#define lua_popen(L,c,m)   ((void)L, popen(c,m))
#define lua_pclose(L,file)   ((void)L, (pclose(file) != -1))

#elif defined(LUA_WIN)

#define lua_popen(L,c,m)   ((void)L, _popen(c,m))
#define lua_pclose(L,file)   ((void)L, (_pclose(file) != -1))

#else

#define lua_popen(L,c,m)   ((void)((void)c, m),  \
      luaL_error(L, LUA_QL("popen") " not supported"), (FILE*)0)
#define lua_pclose(L,file)      ((void)((void)L, file), 0)

#endif

/*
@@ LUA_DL_* define which dynamic-library system Lua should use.
** CHANGE here if Lua has problems choosing the appropriate
** dynamic-library system for your platform (either Windows' DLL, Mac's
** dyld, or Unix's dlopen). If your system is some kind of Unix, there
** is a good chance that it has dlopen, so LUA_DL_DLOPEN will work for
** it.  To use dlopen you also need to adapt the src/Makefile (probably
** adding -ldl to the linker options), so Lua does not select it
** automatically.  (When you change the makefile to add -ldl, you must
** also add -DLUA_USE_DLOPEN.)
** If you do not want any kind of dynamic library, undefine all these
** options.
** By default, _WIN32 gets LUA_DL_DLL and MAC OS X gets LUA_DL_DYLD.
*/
#if defined(LUA_USE_DLOPEN)
#define LUA_DL_DLOPEN
#endif

#if defined(LUA_WIN)
#define LUA_DL_DLL
#endif


/*
@@ LUAI_EXTRASPACE allows you to add user-specific data in a lua_State
@* (the data goes just *before* the lua_State pointer).
** CHANGE (define) this if you really need that. This value must be
** a multiple of the maximum alignment required for your machine.
*/
#define LUAI_EXTRASPACE      0


/*
@@ luai_userstate* allow user-specific actions on threads.
** CHANGE them if you defined LUAI_EXTRASPACE and need to do something
** extra when a thread is created/deleted/resumed/yielded.
*/
#define luai_userstateopen(L)      ((void)L)
#define luai_userstateclose(L)      ((void)L)
#define luai_userstatethread(L,L1)   ((void)L)
#define luai_userstatefree(L)      ((void)L)
#define luai_userstateresume(L,n)   ((void)L)
#define luai_userstateyield(L,n)   ((void)L)


/*
@@ LUA_INTFRMLEN is the length modifier for integer conversions
@* in 'string.format'.
@@ LUA_INTFRM_T is the integer type correspoding to the previous length
@* modifier.
** CHANGE them if your system supports long long or does not support long.
*/

#if defined(LUA_USELONGLONG)

#define LUA_INTFRMLEN      "ll"
#define LUA_INTFRM_T      long long

#else

#define LUA_INTFRMLEN      "l"
#define LUA_INTFRM_T      long

#endif



/* =================================================================== */

/*
** Local configuration. You can use this space to add your redefinitions
** without modifying the main part of the file.
*/

/* used in llex.c to denote end of short strings (`$'<string>)
   -> CHANGE THE MANUAL IF ALTERED */

#define AGN_SHORTSTRINGDELIM " ,~[]{}();:#'=?&%$\"!^@<>|\r\n\t"

/* BUFSIZ in Linux is 8k, so define a smaller buffer for io.readlines */
#define AGN_IOREADLINES_MAXLINESIZE  512

#ifdef __cplusplus
}
#endif


#endif

