# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

from oslo_config import cfg

from keystone.conf import constants
from keystone.conf import utils


assertion_expiration_time = cfg.IntOpt(
    'assertion_expiration_time',
    default=3600,
    help=utils.fmt("""
Determines the lifetime for any SAML assertions generated by keystone, using
`NotOnOrAfter` attributes.
"""))

xmlsec1_binary = cfg.StrOpt(
    'xmlsec1_binary',
    default='xmlsec1',
    help=utils.fmt("""
Name of, or absolute path to, the binary to be used for XML signing. Although
only the XML Security Library (`xmlsec1`) is supported, it may have a
non-standard name or path on your system. If keystone cannot find the binary
itself, you may need to install the appropriate package, use this option to
specify an absolute path, or adjust keystone's PATH environment variable.
"""))

certfile = cfg.StrOpt(
    'certfile',
    default=constants._CERTFILE,
    help=utils.fmt("""
Absolute path to the public certificate file to use for SAML signing. The value
cannot contain a comma (`,`).
"""))

keyfile = cfg.StrOpt(
    'keyfile',
    default=constants._KEYFILE,
    help=utils.fmt("""
Absolute path to the private key file to use for SAML signing. The value cannot
contain a comma (`,`).
"""))

idp_entity_id = cfg.URIOpt(
    'idp_entity_id',
    max_length=1024,
    help=utils.fmt("""
This is the unique entity identifier of the identity provider (keystone) to use
when generating SAML assertions. This value is required to generate identity
provider metadata and must be a URI (a URL is recommended). For example:
`https://keystone.example.com/v3/OS-FEDERATION/saml2/idp`.
"""))

idp_sso_endpoint = cfg.URIOpt(
    'idp_sso_endpoint',
    help=utils.fmt("""
This is the single sign-on (SSO) service location of the identity provider
which accepts HTTP POST requests. A value is required to generate identity
provider metadata. For example:
`https://keystone.example.com/v3/OS-FEDERATION/saml2/sso`.
"""))

idp_lang = cfg.StrOpt(
    'idp_lang',
    default='en',
    help=utils.fmt("""
This is the language used by the identity provider's organization.
"""))

idp_organization_name = cfg.StrOpt(
    'idp_organization_name',
    default='SAML Identity Provider',
    help=utils.fmt("""
This is the name of the identity provider's organization.
"""))

idp_organization_display_name = cfg.StrOpt(
    'idp_organization_display_name',
    default='OpenStack SAML Identity Provider',
    help=utils.fmt("""
This is the name of the identity provider's organization to be displayed.
"""))

idp_organization_url = cfg.URIOpt(
    'idp_organization_url',
    default='https://example.com/',
    help=utils.fmt("""
This is the URL of the identity provider's organization. The URL referenced
here should be useful to humans.
"""))

idp_contact_company = cfg.StrOpt(
    'idp_contact_company',
    default='Example, Inc.',
    help=utils.fmt("""
This is the company name of the identity provider's contact person.
"""))

idp_contact_name = cfg.StrOpt(
    'idp_contact_name',
    default='SAML Identity Provider Support',
    help=utils.fmt("""
This is the given name of the identity provider's contact person.
"""))

idp_contact_surname = cfg.StrOpt(
    'idp_contact_surname',
    default='Support',
    help=utils.fmt("""
This is the surname of the identity provider's contact person.
"""))

idp_contact_email = cfg.StrOpt(
    'idp_contact_email',
    default='support@example.com',
    help=utils.fmt("""
This is the email address of the identity provider's contact person.
"""))

idp_contact_telephone = cfg.StrOpt(
    'idp_contact_telephone',
    default='+1 800 555 0100',
    help=utils.fmt("""
This is the telephone number of the identity provider's contact person.
"""))

idp_contact_type = cfg.StrOpt(
    'idp_contact_type',
    default='other',
    choices=['technical', 'support', 'administrative', 'billing', 'other'],
    help=utils.fmt("""
This is the type of contact that best describes the identity provider's contact
person.
"""))

idp_metadata_path = cfg.StrOpt(
    'idp_metadata_path',
    default='/etc/keystone/saml2_idp_metadata.xml',
    help=utils.fmt("""
Absolute path to the identity provider metadata file. This file should be
generated with the `keystone-manage saml_idp_metadata` command. There is
typically no reason to change this value.
"""))

relay_state_prefix = cfg.StrOpt(
    'relay_state_prefix',
    default='ss:mem:',
    help=utils.fmt("""
The prefix of the RelayState SAML attribute to use when generating enhanced
client and proxy (ECP) assertions. In a typical deployment, there is no reason
to change this value.
"""))


GROUP_NAME = __name__.split('.')[-1]
ALL_OPTS = [
    assertion_expiration_time,
    xmlsec1_binary,
    certfile,
    keyfile,
    idp_entity_id,
    idp_sso_endpoint,
    idp_lang,
    idp_organization_name,
    idp_organization_display_name,
    idp_organization_url,
    idp_contact_company,
    idp_contact_name,
    idp_contact_surname,
    idp_contact_email,
    idp_contact_telephone,
    idp_contact_type,
    idp_metadata_path,
    relay_state_prefix,
]


def register_opts(conf):
    conf.register_opts(ALL_OPTS, group=GROUP_NAME)


def list_opts():
    return {GROUP_NAME: ALL_OPTS}
