/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.controller;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.IntPredicate;
import java.util.stream.Collectors;
import org.apache.kafka.common.DirectoryId;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.message.AlterPartitionRequestData;
import org.apache.kafka.common.metadata.PartitionChangeRecord;
import org.apache.kafka.common.protocol.ApiMessage;
import org.apache.kafka.controller.PartitionReassignmentReplicas;
import org.apache.kafka.metadata.LeaderRecoveryState;
import org.apache.kafka.metadata.PartitionRegistration;
import org.apache.kafka.metadata.Replicas;
import org.apache.kafka.metadata.placement.DefaultDirProvider;
import org.apache.kafka.server.common.ApiMessageAndVersion;
import org.apache.kafka.server.common.MetadataVersion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PartitionChangeBuilder {
    private static final Logger log = LoggerFactory.getLogger(PartitionChangeBuilder.class);
    private final PartitionRegistration partition;
    private final Uuid topicId;
    private final int partitionId;
    private final IntPredicate isAcceptableLeader;
    private final MetadataVersion metadataVersion;
    private final int minISR;
    private final Map<Integer, Uuid> targetDirectories;
    private List<Integer> targetIsr;
    private List<Integer> targetReplicas;
    private List<Integer> targetRemoving;
    private List<Integer> targetAdding;
    private List<Integer> targetElr;
    private List<Integer> targetLastKnownElr;
    private List<Integer> uncleanShutdownReplicas;
    private Election election = Election.ONLINE;
    private LeaderRecoveryState targetLeaderRecoveryState;
    private boolean zkMigrationEnabled;
    private boolean eligibleLeaderReplicasEnabled;
    private DefaultDirProvider defaultDirProvider;
    private boolean useLastKnownLeaderInBalancedRecovery = true;

    public static boolean changeRecordIsNoOp(PartitionChangeRecord record) {
        if (record.isr() != null) {
            return false;
        }
        if (record.eligibleLeaderReplicas() != null) {
            return false;
        }
        if (record.lastKnownElr() != null) {
            return false;
        }
        if (record.leader() != -2) {
            return false;
        }
        if (record.replicas() != null) {
            return false;
        }
        if (record.removingReplicas() != null) {
            return false;
        }
        if (record.addingReplicas() != null) {
            return false;
        }
        if (record.leaderRecoveryState() != -1) {
            return false;
        }
        return record.directories() == null;
    }

    public PartitionChangeBuilder(PartitionRegistration partition, Uuid topicId, int partitionId, IntPredicate isAcceptableLeader, MetadataVersion metadataVersion, int minISR) {
        this.partition = partition;
        this.topicId = topicId;
        this.partitionId = partitionId;
        this.isAcceptableLeader = isAcceptableLeader;
        this.metadataVersion = metadataVersion;
        this.zkMigrationEnabled = false;
        this.eligibleLeaderReplicasEnabled = false;
        this.minISR = minISR;
        this.targetIsr = Replicas.toList(partition.isr);
        this.targetReplicas = Replicas.toList(partition.replicas);
        this.targetRemoving = Replicas.toList(partition.removingReplicas);
        this.targetAdding = Replicas.toList(partition.addingReplicas);
        this.targetElr = Replicas.toList(partition.elr);
        this.targetLastKnownElr = Replicas.toList(partition.lastKnownElr);
        this.targetLeaderRecoveryState = partition.leaderRecoveryState;
        this.targetDirectories = DirectoryId.createAssignmentMap((int[])partition.replicas, (Uuid[])partition.directories);
        this.defaultDirProvider = uuid -> {
            throw new IllegalStateException("DefaultDirProvider is not set");
        };
    }

    public PartitionChangeBuilder setTargetIsr(List<Integer> targetIsr) {
        this.targetIsr = targetIsr;
        return this;
    }

    public PartitionChangeBuilder setTargetIsrWithBrokerStates(List<AlterPartitionRequestData.BrokerState> targetIsrWithEpoch) {
        return this.setTargetIsr(targetIsrWithEpoch.stream().map(AlterPartitionRequestData.BrokerState::brokerId).collect(Collectors.toList()));
    }

    public PartitionChangeBuilder setTargetReplicas(List<Integer> targetReplicas) {
        this.targetReplicas = targetReplicas;
        return this;
    }

    public PartitionChangeBuilder setUncleanShutdownReplicas(List<Integer> uncleanShutdownReplicas) {
        this.uncleanShutdownReplicas = uncleanShutdownReplicas;
        return this;
    }

    public PartitionChangeBuilder setElection(Election election) {
        this.election = election;
        return this;
    }

    public PartitionChangeBuilder setTargetRemoving(List<Integer> targetRemoving) {
        this.targetRemoving = targetRemoving;
        return this;
    }

    public PartitionChangeBuilder setTargetAdding(List<Integer> targetAdding) {
        this.targetAdding = targetAdding;
        return this;
    }

    public PartitionChangeBuilder setTargetLeaderRecoveryState(LeaderRecoveryState targetLeaderRecoveryState) {
        this.targetLeaderRecoveryState = targetLeaderRecoveryState;
        return this;
    }

    public PartitionChangeBuilder setZkMigrationEnabled(boolean zkMigrationEnabled) {
        this.zkMigrationEnabled = zkMigrationEnabled;
        return this;
    }

    public PartitionChangeBuilder setEligibleLeaderReplicasEnabled(boolean eligibleLeaderReplicasEnabled) {
        this.eligibleLeaderReplicasEnabled = eligibleLeaderReplicasEnabled;
        return this;
    }

    public PartitionChangeBuilder setUseLastKnownLeaderInBalancedRecovery(boolean useLastKnownLeaderInBalancedRecovery) {
        this.useLastKnownLeaderInBalancedRecovery = useLastKnownLeaderInBalancedRecovery;
        return this;
    }

    public PartitionChangeBuilder setDirectory(int brokerId, Uuid dir) {
        this.targetDirectories.put(brokerId, dir);
        return this;
    }

    public PartitionChangeBuilder setDefaultDirProvider(DefaultDirProvider defaultDirProvider) {
        this.defaultDirProvider = defaultDirProvider;
        return this;
    }

    public List<Integer> targetIsr() {
        return this.targetIsr;
    }

    ElectionResult electLeader() {
        if (this.election == Election.PREFERRED) {
            return this.electPreferredLeader();
        }
        return this.electAnyLeader();
    }

    private ElectionResult electPreferredLeader() {
        int preferredReplica = this.targetReplicas.get(0);
        if (this.isValidNewLeader(preferredReplica)) {
            return new ElectionResult(preferredReplica, false);
        }
        if (this.isValidNewLeader(this.partition.leader)) {
            return new ElectionResult(this.partition.leader, false);
        }
        Optional<Integer> onlineLeader = this.targetReplicas.stream().skip(1L).filter(this::isValidNewLeader).findFirst();
        if (onlineLeader.isPresent()) {
            return new ElectionResult(onlineLeader.get(), false);
        }
        if (this.canElectLastKnownLeader()) {
            return new ElectionResult(this.partition.lastKnownElr[0], true);
        }
        return new ElectionResult(-1, false);
    }

    private ElectionResult electAnyLeader() {
        if (this.isValidNewLeader(this.partition.leader)) {
            return new ElectionResult(this.partition.leader, false);
        }
        Optional<Integer> onlineLeader = this.targetReplicas.stream().filter(this::isValidNewLeader).findFirst();
        if (onlineLeader.isPresent()) {
            return new ElectionResult(onlineLeader.get(), false);
        }
        if (this.canElectLastKnownLeader()) {
            return new ElectionResult(this.partition.lastKnownElr[0], true);
        }
        if (this.election == Election.UNCLEAN) {
            Optional<Integer> uncleanLeader = this.targetReplicas.stream().filter(this.isAcceptableLeader::test).findFirst();
            if (uncleanLeader.isPresent()) {
                return new ElectionResult(uncleanLeader.get(), true);
            }
        }
        return new ElectionResult(-1, false);
    }

    private boolean canElectLastKnownLeader() {
        if (!this.eligibleLeaderReplicasEnabled || !this.useLastKnownLeaderInBalancedRecovery) {
            log.trace("Try to elect last known leader for " + this.topicId + "-" + this.partitionId + " but elrEnabled=" + this.eligibleLeaderReplicasEnabled + ", useLastKnownLeaderInBalancedRecovery=" + this.useLastKnownLeaderInBalancedRecovery);
            return false;
        }
        if (!this.targetElr.isEmpty() || !this.targetIsr.isEmpty()) {
            log.trace("Try to elect last known leader for " + this.topicId + "-" + this.partitionId + " but ELR/ISR is not empty. ISR=" + this.targetIsr + ", ELR=" + this.targetElr);
            return false;
        }
        if (this.partition.lastKnownElr.length != 1) {
            log.trace("Try to elect last known leader for " + this.topicId + "-" + this.partitionId + " but lastKnownElr does not only have 1 member. lastKnownElr=" + Arrays.toString(this.partition.lastKnownElr));
            return false;
        }
        if (this.isAcceptableLeader.test(this.partition.lastKnownElr[0])) {
            log.trace("Try to elect last known leader for " + this.topicId + "-" + this.partitionId + " but last known leader is not alive. last known leader=" + this.partition.lastKnownElr[0]);
        }
        return true;
    }

    private boolean isValidNewLeader(int replica) {
        return (this.targetIsr.contains(replica) || this.targetIsr.isEmpty() && this.targetElr.contains(replica)) && this.isAcceptableLeader.test(replica);
    }

    private void tryElection(PartitionChangeRecord record) {
        ElectionResult electionResult = this.electLeader();
        if (electionResult.node != this.partition.leader) {
            if (this.targetElr.contains(electionResult.node)) {
                this.targetIsr = Collections.singletonList(electionResult.node);
                this.targetElr = this.targetElr.stream().filter(replica -> replica != electionResult.node).collect(Collectors.toList());
                log.trace("Setting new leader for topicId {}, partition {} to {} using ELR", new Object[]{this.topicId, this.partitionId, electionResult.node});
            } else if (electionResult.unclean) {
                log.info("Setting new leader for topicId {}, partition {} to {} using an unclean election", new Object[]{this.topicId, this.partitionId, electionResult.node});
            } else {
                log.trace("Setting new leader for topicId {}, partition {} to {} using a clean election", new Object[]{this.topicId, this.partitionId, electionResult.node});
            }
            record.setLeader(electionResult.node);
            if (electionResult.unclean) {
                record.setIsr(Collections.singletonList(electionResult.node));
                if (this.partition.leaderRecoveryState != LeaderRecoveryState.RECOVERING && this.metadataVersion.isLeaderRecoverySupported()) {
                    record.setLeaderRecoveryState(LeaderRecoveryState.RECOVERING.value());
                }
            }
        } else {
            log.trace("Failed to find a new leader with current state: {}", (Object)this);
        }
    }

    void triggerLeaderEpochBumpForReplicaReassignmentIfNeeded(PartitionChangeRecord record) {
        if (record.leader() != -2) {
            return;
        }
        if (!Replicas.contains(this.targetReplicas, this.partition.replicas)) {
            record.setLeader(this.partition.leader);
        }
    }

    void triggerLeaderEpochBumpForIsrShrinkIfNeeded(PartitionChangeRecord record) {
        if (!this.metadataVersion.isLeaderEpochBumpRequiredOnIsrShrink() && !this.zkMigrationEnabled) {
            return;
        }
        if (record.leader() != -2) {
            return;
        }
        if (record.isr() == null) {
            return;
        }
        if (!Replicas.contains(record.isr(), this.partition.isr)) {
            record.setLeader(this.partition.leader);
        }
    }

    private void completeReassignmentIfNeeded() {
        PartitionReassignmentReplicas reassignmentReplicas = new PartitionReassignmentReplicas(this.targetRemoving, this.targetAdding, this.targetReplicas);
        Optional<PartitionReassignmentReplicas.CompletedReassignment> completedReassignmentOpt = reassignmentReplicas.maybeCompleteReassignment(this.targetIsr);
        if (!completedReassignmentOpt.isPresent()) {
            return;
        }
        PartitionReassignmentReplicas.CompletedReassignment completedReassignment = completedReassignmentOpt.get();
        this.targetIsr = completedReassignment.isr;
        this.targetReplicas = completedReassignment.replicas;
        this.targetRemoving = Collections.emptyList();
        this.targetAdding = Collections.emptyList();
    }

    public Optional<ApiMessageAndVersion> build() {
        PartitionChangeRecord record = new PartitionChangeRecord().setTopicId(this.topicId).setPartitionId(this.partitionId);
        this.completeReassignmentIfNeeded();
        this.maybePopulateTargetElr();
        this.tryElection(record);
        this.triggerLeaderEpochBumpForReplicaReassignmentIfNeeded(record);
        this.maybeUpdateRecordElr(record);
        if (!(record.isr() != null || this.targetIsr.isEmpty() && !this.eligibleLeaderReplicasEnabled || this.targetIsr.equals(Replicas.toList(this.partition.isr)))) {
            if (this.targetIsr.isEmpty()) {
                log.debug("A partition will have an empty ISR. " + this);
            }
            record.setIsr(this.targetIsr);
        }
        this.triggerLeaderEpochBumpForIsrShrinkIfNeeded(record);
        this.maybeUpdateLastKnownLeader(record);
        this.setAssignmentChanges(record);
        if (this.targetLeaderRecoveryState != this.partition.leaderRecoveryState) {
            record.setLeaderRecoveryState(this.targetLeaderRecoveryState.value());
        }
        if (PartitionChangeBuilder.changeRecordIsNoOp(record)) {
            return Optional.empty();
        }
        return Optional.of(new ApiMessageAndVersion((ApiMessage)record, this.metadataVersion.partitionChangeRecordVersion()));
    }

    private void setAssignmentChanges(PartitionChangeRecord record) {
        if (!this.targetReplicas.isEmpty()) {
            if (this.metadataVersion.isDirectoryAssignmentSupported()) {
                ArrayList<Uuid> directories = new ArrayList<Uuid>(this.targetReplicas.size());
                for (int replica : this.targetReplicas) {
                    directories.add(this.targetDirectories.getOrDefault(replica, this.defaultDirProvider.defaultDir(replica)));
                }
                if (!directories.equals(Arrays.asList(this.partition.directories))) {
                    record.setDirectories(directories);
                }
            }
            if (!this.targetReplicas.equals(Replicas.toList(this.partition.replicas))) {
                record.setReplicas(this.targetReplicas);
            }
        }
        if (!this.targetRemoving.equals(Replicas.toList(this.partition.removingReplicas))) {
            record.setRemovingReplicas(this.targetRemoving);
        }
        if (!this.targetAdding.equals(Replicas.toList(this.partition.addingReplicas))) {
            record.setAddingReplicas(this.targetAdding);
        }
    }

    private void maybeUpdateLastKnownLeader(PartitionChangeRecord record) {
        if (!this.useLastKnownLeaderInBalancedRecovery || !this.eligibleLeaderReplicasEnabled) {
            return;
        }
        if (record.isr() != null && record.isr().isEmpty() && (this.partition.lastKnownElr.length != 1 || this.partition.lastKnownElr[0] != this.partition.leader)) {
            record.setLastKnownElr(Collections.singletonList(this.partition.leader));
        } else if ((record.leader() >= 0 || this.partition.leader != -1 && record.leader() != -1) && this.partition.lastKnownElr.length > 0) {
            record.setLastKnownElr(Collections.emptyList());
        }
    }

    private void maybeUpdateRecordElr(PartitionChangeRecord record) {
        boolean isCleanLeaderElection;
        boolean bl = isCleanLeaderElection = record.isr() == null;
        if (!isCleanLeaderElection || !this.eligibleLeaderReplicasEnabled) {
            this.targetElr = Collections.emptyList();
            this.targetLastKnownElr = Collections.emptyList();
        }
        if (!this.targetElr.equals(Replicas.toList(this.partition.elr))) {
            record.setEligibleLeaderReplicas(this.targetElr);
        }
        if (this.useLastKnownLeaderInBalancedRecovery && this.partition.lastKnownElr.length == 1 && (record.leader() == -1 || record.leader() == -2 && this.partition.leader == -1)) {
            this.targetLastKnownElr = Replicas.toList(this.partition.lastKnownElr);
        }
        if (this.useLastKnownLeaderInBalancedRecovery) {
            return;
        }
        if (!this.targetLastKnownElr.equals(Replicas.toList(this.partition.lastKnownElr))) {
            record.setLastKnownElr(this.targetLastKnownElr);
        }
    }

    private void maybePopulateTargetElr() {
        if (!this.eligibleLeaderReplicasEnabled) {
            return;
        }
        if (this.targetIsr.size() >= this.minISR) {
            this.targetElr = Collections.emptyList();
            this.targetLastKnownElr = Collections.emptyList();
            return;
        }
        HashSet<Integer> targetIsrSet = new HashSet<Integer>(this.targetIsr);
        HashSet<Integer> candidateSet = new HashSet<Integer>(this.targetElr);
        Arrays.stream(this.partition.isr).forEach(candidateSet::add);
        this.targetElr = candidateSet.stream().filter(replica -> !targetIsrSet.contains(replica)).filter(replica -> this.uncleanShutdownReplicas == null || !this.uncleanShutdownReplicas.contains(replica)).collect(Collectors.toList());
        candidateSet.addAll(this.targetLastKnownElr);
        this.targetLastKnownElr = candidateSet.stream().filter(replica -> !targetIsrSet.contains(replica)).filter(replica -> !this.targetElr.contains(replica)).collect(Collectors.toList());
    }

    public String toString() {
        return "PartitionChangeBuilder(partition=" + this.partition + ", topicId=" + this.topicId + ", partitionId=" + this.partitionId + ", isAcceptableLeader=" + this.isAcceptableLeader + ", targetIsr=" + this.targetIsr + ", targetReplicas=" + this.targetReplicas + ", targetRemoving=" + this.targetRemoving + ", targetAdding=" + this.targetAdding + ", targetElr=" + this.targetElr + ", targetLastKnownElr=" + this.targetLastKnownElr + ", uncleanShutdownReplicas=" + this.uncleanShutdownReplicas + ", election=" + (Object)((Object)this.election) + ", targetLeaderRecoveryState=" + (Object)((Object)this.targetLeaderRecoveryState) + ')';
    }

    public static enum Election {
        PREFERRED,
        ONLINE,
        UNCLEAN;

    }

    static class ElectionResult {
        final int node;
        final boolean unclean;

        private ElectionResult(int node, boolean unclean) {
            this.node = node;
            this.unclean = unclean;
        }
    }
}

