/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.exec.repl;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import jodd.exception.UncheckedException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ContentSummary;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.OpenFilesIterator;
import org.apache.hadoop.util.ExitUtil;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplicationMigrationTool
implements Tool {
    protected static Logger LOG = LoggerFactory.getLogger(ReplicationMigrationTool.class);
    private Configuration conf;
    private String help = "\tSample Usage: \nhive --replMigration -dumpFilePath <path to external table info file> [-dirLevelCheck] [-fileLevelCheck] [-verifyOpenFiles] [-verifyChecksum] [-filters] [-conf] [-queueSize] [-numThreads] [-checksumQueueSize] [-checksumNumThreads] [-help]\n-dumpFilePath: The fully qualified path to the external table info file. \n-dirLevelCheck: Validate at directory level.-fileLevelCheck: Validate at file level.-verifyOpenFiles: Validate there is no open files on the source path. \n-verifyChecksum: Whether the checksum needs to be validated for each file. Can not be used with -dirLevelCheck. Will fail in case the source & target are in different encryption zones or uses different checksum algorithm.\n-filters: Comma separated list of filters, Can not be used along with -dirLevelCheck. \n-conf: Semi-Colon separated list of additional configurations in key1=value1;key2=value2 format. \n-queueSize: Queue size for the thread pool executor for table level validation. Default:200-numThreads: Number of threads for thread pool executor for table level validation. Default:10-checksumQueueSize: Queue size for the thread pool executor for checksum computation. Default:200-checksumNumThreads: Number of threads for thread pool executor for checksum computation. Default:5-help: Prints the help message.\nNote: The dumpFilePath for a scheduled query can be fetched using the beeline query: \nselect * from sys.replication_metrics where policy_name=\u2018<policy name>\u2019 order by scheduled_execution_id desc limit 1;";

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int run(String[] args) throws Exception {
        ArrayList<String> argsList = new ArrayList<String>(Arrays.asList(args));
        boolean isHelp = StringUtils.popOption((String)"-help", argsList);
        if (isHelp) {
            System.out.println(this.help);
            return 0;
        }
        String dumpPath = StringUtils.popOptionWithArgument((String)"-dumpFilePath", argsList);
        boolean isValidateChecksum = StringUtils.popOption((String)"-verifyChecksum", argsList);
        boolean checkAtDirLevel = StringUtils.popOption((String)"-dirLevelCheck", argsList);
        boolean checkAtFileLevel = StringUtils.popOption((String)"-fileLevelCheck", argsList);
        boolean checkOpenFiles = StringUtils.popOption((String)"-verifyOpenFiles", argsList);
        List<Pattern> filtersPattern = this.getFilterPatterns(argsList);
        this.extractAndSetConfigs(argsList);
        int queueSize = this.getParamValue(argsList, "-queueSize", 200);
        int numThreads = this.getParamValue(argsList, "-numThreads", 10);
        ThreadPoolExecutor threadPool = this.getThreadPoolExecutor(queueSize, numThreads);
        LOG.info("Using a ThreadPoolExecutor with {} threads and {} as Queue size.", (Object)numThreads, (Object)queueSize);
        ThreadPoolExecutor threadPoolChecksum = null;
        if (isValidateChecksum) {
            int checksumQueueSize = this.getParamValue(argsList, "-checksumQueueSize", 200);
            int checksumNumThreads = this.getParamValue(argsList, "-checksumNumThreads", 5);
            threadPoolChecksum = this.getThreadPoolExecutor(checksumQueueSize, checksumNumThreads);
            LOG.info("Using a ThreadPoolExecutor with {} threads and {} as Queue size for checksum computation.", (Object)checksumNumThreads, (Object)checksumQueueSize);
        }
        this.validateArguments(argsList, dumpPath, isValidateChecksum, checkAtFileLevel);
        Path dump = this.getExternalTableFileListPath(dumpPath);
        FileSystem fs = dump.getFileSystem(this.conf);
        BufferedReader br = new BufferedReader(new InputStreamReader((InputStream)fs.open(dump)));
        ArrayList<Future<Boolean>> futures = new ArrayList<Future<Boolean>>();
        long startTime = System.currentTimeMillis();
        try {
            String line = br.readLine();
            while (line != null) {
                LOG.debug("Line read from {} is {} :", (Object)dumpPath, (Object)line);
                DirectoryProcessor dProcessor = new DirectoryProcessor(line, checkAtDirLevel, checkAtFileLevel, checkOpenFiles, isValidateChecksum, filtersPattern, threadPoolChecksum);
                Future<Boolean> future = threadPool.submit(dProcessor);
                futures.add(future);
                line = br.readLine();
            }
            int failed = 0;
            for (Future future : futures) {
                if (((Boolean)future.get()).booleanValue()) continue;
                ++failed;
            }
            LOG.error("Total {} paths failed", (Object)failed);
            threadPool.shutdown();
            System.out.println("Completed verification. Source & Target are " + (failed == 0 ? "in Sync." : "not in Sync."));
            System.out.println("Time Taken: " + (System.currentTimeMillis() - startTime) + " ms");
            if (failed != 0) {
                int n = -1;
                return n;
            }
        }
        catch (UnsupportedOperationException e) {
            System.err.println(e.getMessage());
            System.err.println(this.help);
            int n = -1;
            return n;
        }
        finally {
            br.close();
        }
        return 0;
    }

    private void validateArguments(List<String> argsList, String dumpPath, boolean isValidateChecksum, boolean checkAtFileLevel) {
        if (!checkAtFileLevel && isValidateChecksum) {
            throw new UnsupportedOperationException("-verifyChecksum can not be used without fileLevelCheck");
        }
        if (dumpPath == null || dumpPath.isEmpty()) {
            throw new UnsupportedOperationException("-dumpFilePath is not specified");
        }
        if (!argsList.isEmpty()) {
            throw new UnsupportedOperationException("Invalid Arguments: " + argsList.toString());
        }
    }

    @NotNull
    private ThreadPoolExecutor getThreadPoolExecutor(int queueSize, int numThreads) {
        ArrayBlockingQueue<Runnable> queue = new ArrayBlockingQueue<Runnable>(queueSize);
        ThreadPoolExecutor threadPoolExecutor = new ThreadPoolExecutor(numThreads, numThreads, 0L, TimeUnit.MILLISECONDS, queue);
        threadPoolExecutor.setRejectedExecutionHandler(new RejectedExecutionHandler(this){

            @Override
            public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
                try {
                    executor.getQueue().put(r);
                }
                catch (InterruptedException e) {
                    throw new UncheckedException((Throwable)e);
                }
            }
        });
        return threadPoolExecutor;
    }

    private int getParamValue(List<String> argsList, String parameterName, int defaultValue) {
        String queueSizeString = StringUtils.popOptionWithArgument((String)parameterName, argsList);
        int queueSize = defaultValue;
        if (queueSizeString != null && (queueSize = Integer.parseInt(queueSizeString)) < 0) {
            queueSize = defaultValue;
        }
        return queueSize;
    }

    @NotNull
    private List<Pattern> getFilterPatterns(List<String> argsList) {
        ArrayList<Pattern> filtersPattern = new ArrayList<Pattern>();
        String filter = StringUtils.popOptionWithArgument((String)"-filters", argsList);
        if (filter != null && !filter.isEmpty()) {
            String[] filters;
            for (String filterEntry : filters = filter.split(",")) {
                Pattern pattern = Pattern.compile(filterEntry);
                filtersPattern.add(pattern);
            }
        }
        return filtersPattern;
    }

    private void extractAndSetConfigs(List<String> argsList) throws IOException {
        String configs = StringUtils.popOptionWithArgument((String)"-conf", argsList);
        if (configs != null && !configs.isEmpty()) {
            String[] configList;
            for (String confPair : configList = configs.split(";")) {
                String[] confArr = confPair.split("=");
                if (confArr.length != 2) {
                    throw new IOException("Invalid Configuration " + confPair);
                }
                this.conf.set(confArr[0], confArr[1]);
            }
        }
    }

    private boolean validateOpenFilesAtSource(Path srcPath, FileSystem srcFileSystem) throws IOException {
        if (srcFileSystem instanceof DistributedFileSystem) {
            DistributedFileSystem srcDFS = (DistributedFileSystem)srcPath.getFileSystem(this.conf);
            if (srcDFS.listOpenFiles(EnumSet.of(OpenFilesIterator.OpenFilesType.ALL_OPEN_FILES), Path.getPathWithoutSchemeAndAuthority((Path)srcPath).toString()).hasNext()) {
                System.err.println("There are open files in " + String.valueOf(srcPath));
                return false;
            }
            LOG.error("Open file check is ignored since the source filesystem is not of type of DistributedFileSystem. The source file system is of " + String.valueOf(srcFileSystem.getClass()) + " type.");
        }
        return true;
    }

    private boolean validateAtDirectoryLevel(Path srcPath, Path trgPath, FileSystem srcFileSystem, FileSystem tgtFileSystem) throws IOException {
        ContentSummary srcContentSummary = srcFileSystem.getContentSummary(srcPath);
        ContentSummary tgtContentSummary = tgtFileSystem.getContentSummary(trgPath);
        if (srcContentSummary.getLength() != tgtContentSummary.getLength()) {
            System.err.println("Directory Size mismatch in source directory " + String.valueOf(srcPath) + " and target directory " + String.valueOf(trgPath));
            return false;
        }
        if (srcContentSummary.getDirectoryCount() != tgtContentSummary.getDirectoryCount()) {
            System.err.println("Directory Count mismatch in source directory " + String.valueOf(srcPath) + " and target directory " + String.valueOf(trgPath));
            return false;
        }
        LOG.debug("Directory count matched for {} and {}", (Object)srcPath, (Object)trgPath);
        if (srcContentSummary.getFileCount() != tgtContentSummary.getFileCount()) {
            System.err.println("File Count mismatch in source directory " + String.valueOf(srcPath) + " and target directory " + String.valueOf(trgPath));
            return false;
        }
        LOG.debug("File count matched for {} and {}", (Object)srcPath, (Object)trgPath);
        return true;
    }

    public boolean validateAtFileLevel(Path srcDir, Path tgtDir, FileSystem srcFs, FileSystem tgtFs, List<Pattern> filtersPattern, boolean verifyChecksum, ThreadPoolExecutor threadPoolChecksum) throws Exception {
        LocatedFileStatus sourceFile;
        RemoteIterator srcListing = srcFs.listFiles(srcDir, true);
        RemoteIterator tgtListing = tgtFs.listFiles(tgtDir, true);
        ArrayList<Future<Boolean>> futures = new ArrayList<Future<Boolean>>();
        boolean response = true;
        while (srcListing.hasNext() && tgtListing.hasNext()) {
            sourceFile = (LocatedFileStatus)srcListing.next();
            if (filtersPattern != null && !this.isCopied(sourceFile.getPath(), filtersPattern)) {
                LOG.info("Entry: {} is filtered.", (Object)sourceFile.getPath());
                continue;
            }
            LocatedFileStatus tgtFile = (LocatedFileStatus)tgtListing.next();
            LOG.info("Comparing {} and {}", (Object)sourceFile.getPath(), (Object)tgtFile.getPath());
            String sourcePath = Path.getPathWithoutSchemeAndAuthority((Path)sourceFile.getPath()).toString().replaceFirst(Path.getPathWithoutSchemeAndAuthority((Path)srcDir).toString(), "/");
            String targetPath = Path.getPathWithoutSchemeAndAuthority((Path)tgtFile.getPath()).toString().replaceFirst(Path.getPathWithoutSchemeAndAuthority((Path)tgtDir).toString(), "/");
            if (!sourcePath.equals(targetPath)) {
                System.err.println("Entries mismatch between source: " + String.valueOf(srcDir) + " and target: " + String.valueOf(tgtDir) + " for sourceFile: " + String.valueOf(sourceFile.getPath()) + " at target with " + String.valueOf(tgtFile.getPath()) + " Mismatched subPaths: source: " + sourcePath + " and target: " + targetPath + "Either Source Or Target has an extra/less files.");
                this.validateChecksumStatus(srcDir, tgtDir, futures);
                return false;
            }
            if (sourceFile.getLen() != tgtFile.getLen()) {
                System.err.println("File Size mismatch in source directory " + String.valueOf(srcDir) + " and target directory " + String.valueOf(tgtDir) + " for source: " + String.valueOf(sourceFile.getPath()) + " and target " + String.valueOf(tgtFile.getPath()));
                response = false;
            }
            LOG.debug("Length matched for {} and {}", (Object)sourceFile.getPath(), (Object)tgtFile.getPath());
            if (!verifyChecksum) continue;
            ChecksumVerifier checksumVerifier = new ChecksumVerifier(this, sourceFile.getPath(), tgtFile.getPath(), srcFs, tgtFs, srcDir, tgtDir);
            Future<Boolean> future = threadPoolChecksum.submit(checksumVerifier);
            futures.add(future);
        }
        if (tgtListing.hasNext()) {
            this.validateChecksumStatus(srcDir, tgtDir, futures);
            System.err.println("Extra entry at target: " + String.valueOf(((LocatedFileStatus)tgtListing.next()).getPath()));
            return false;
        }
        LOG.debug("No target entries remaining for {} and {}", (Object)srcDir, (Object)tgtDir);
        if (this.validateChecksumStatus(srcDir, tgtDir, futures)) {
            return false;
        }
        while (srcListing.hasNext()) {
            sourceFile = (LocatedFileStatus)srcListing.next();
            if (filtersPattern != null && !this.isCopied(sourceFile.getPath(), filtersPattern)) {
                LOG.info("Entry: {} is filtered.", (Object)sourceFile.getPath());
                continue;
            }
            System.err.println("Extra entry at source: " + String.valueOf(sourceFile.getPath()));
            return false;
        }
        LOG.debug("No source entries remaining for {} and {}", (Object)srcDir, (Object)tgtDir);
        return response;
    }

    private boolean validateChecksumStatus(Path srcDir, Path tgtDir, ArrayList<Future<Boolean>> futures) throws InterruptedException, ExecutionException {
        int numFailed = 0;
        int numSuccess = 0;
        for (Future<Boolean> f : futures) {
            if (!f.get().booleanValue()) {
                ++numFailed;
                continue;
            }
            ++numSuccess;
        }
        if (numFailed > 0) {
            LOG.warn("{} files failed checksum validation for source: {} and target: {}, numSuccess: {} numFailed: {}", new Object[]{srcDir, tgtDir, numSuccess, numFailed});
            return true;
        }
        return false;
    }

    private boolean isCopied(Path path, List<Pattern> filters) {
        for (Pattern filter : filters) {
            if (!filter.matcher(path.toString()).matches()) continue;
            return false;
        }
        return true;
    }

    private Path getExternalTableFileListPath(String dumpPath) {
        if (dumpPath.endsWith("/_file_list_external")) {
            return new Path(dumpPath);
        }
        if (dumpPath.endsWith("/hive")) {
            return new Path(dumpPath, "_file_list_external");
        }
        return new Path(dumpPath, "hive/_file_list_external");
    }

    public void setConf(Configuration configuration) {
        this.conf = configuration;
    }

    public Configuration getConf() {
        return this.conf;
    }

    public static void main(String[] args) throws Exception {
        Configuration conf = new Configuration();
        ReplicationMigrationTool inst = new ReplicationMigrationTool();
        inst.setConf(conf);
        int result = ToolRunner.run((Tool)inst, (String[])args);
        ExitUtil.terminate((int)result);
    }

    private class DirectoryProcessor
    implements Callable<Boolean> {
        String line;
        boolean checkOpenFiles;
        boolean isValidateChecksum;
        boolean checkAtDirLevel;
        boolean checkAtFileLevel;
        List<Pattern> filtersPattern;
        ThreadPoolExecutor threadPoolChecksum;

        DirectoryProcessor(String line, boolean checkAtDirLevel, boolean checkAtFileLevel, boolean checkOpenFiles, boolean isValidateChecksum, List<Pattern> filtersPattern, ThreadPoolExecutor threadPoolChecksum) {
            this.line = line;
            this.checkAtDirLevel = checkAtDirLevel;
            this.checkAtFileLevel = checkAtFileLevel;
            this.checkOpenFiles = checkOpenFiles;
            this.isValidateChecksum = isValidateChecksum;
            this.filtersPattern = filtersPattern;
            this.threadPoolChecksum = threadPoolChecksum;
        }

        @Override
        public Boolean call() throws Exception {
            String[] lineComponents = this.line.split("#");
            Path srcPath = new Path(lineComponents[0]);
            Path trgPath = new Path(lineComponents[1]);
            try {
                FileSystem srcFileSystem = srcPath.getFileSystem(ReplicationMigrationTool.this.conf);
                FileSystem tgtFileSystem = trgPath.getFileSystem(ReplicationMigrationTool.this.conf);
                if (this.checkOpenFiles) {
                    LOG.debug("Validating {} and {} for open files at source", (Object)srcPath, (Object)trgPath);
                    if (!ReplicationMigrationTool.this.validateOpenFilesAtSource(srcPath, srcFileSystem)) {
                        return false;
                    }
                }
                if (this.checkAtDirLevel) {
                    LOG.debug("Validating {} and {} at directory level", (Object)srcPath, (Object)trgPath);
                    if (!ReplicationMigrationTool.this.validateAtDirectoryLevel(srcPath, trgPath, srcFileSystem, tgtFileSystem)) {
                        return false;
                    }
                }
                if (this.checkAtFileLevel) {
                    LOG.debug("Validating {} and {} at file level", (Object)srcPath, (Object)trgPath);
                    if (!ReplicationMigrationTool.this.validateAtFileLevel(srcPath, trgPath, srcFileSystem, tgtFileSystem, this.filtersPattern, this.isValidateChecksum, this.threadPoolChecksum)) {
                        return false;
                    }
                }
                LOG.info(String.valueOf(srcPath) + " and " + String.valueOf(trgPath) + " are in Sync");
            }
            catch (Exception e) {
                LOG.error("Failed to verify source: {} with target: {}", (Throwable)e);
                System.err.println("Failed to verify source: " + String.valueOf(srcPath) + " with target: " + String.valueOf(trgPath) + " error:" + e.getMessage());
                return false;
            }
            return true;
        }
    }

    private class ChecksumVerifier
    implements Callable<Boolean> {
        private final Path srcDir;
        private final Path tgtDir;
        private Path sourcePath;
        private Path targetPath;
        private FileSystem sourceFs;
        private FileSystem targetFs;

        ChecksumVerifier(ReplicationMigrationTool replicationMigrationTool, Path sourcePath, Path targetPath, FileSystem sourceFs, FileSystem targetFs, Path srcDir, Path tgtDir) {
            this.sourcePath = sourcePath;
            this.targetPath = targetPath;
            this.sourceFs = sourceFs;
            this.targetFs = targetFs;
            this.srcDir = srcDir;
            this.tgtDir = tgtDir;
        }

        @Override
        public Boolean call() throws Exception {
            LOG.debug("Verifying checksum for source: {} and target: {}", (Object)this.sourcePath, (Object)this.targetPath);
            try {
                boolean response = this.sourceFs.getFileChecksum(this.sourcePath).equals((Object)this.targetFs.getFileChecksum(this.targetPath));
                if (!response) {
                    System.err.println("File Checksum mismatch in source directory " + String.valueOf(this.srcDir) + " and target directory " + String.valueOf(this.tgtDir) + " for source: " + String.valueOf(this.sourcePath) + " and target " + String.valueOf(this.targetPath));
                }
                return response;
            }
            catch (Exception e) {
                System.out.println("Failed Checksum for: " + String.valueOf(this.sourcePath) + " and target: " + String.valueOf(this.targetPath) + " reason:" + e.getMessage());
                LOG.error("Failed Checksum for: {} and target: {}", new Object[]{this.sourcePath, this.targetPath, e});
                return false;
            }
        }
    }
}

